// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/dumpjit.cpp,v 1.3 2001/09/18 09:06:59 xhshi Exp $
//


#include "defines.h"

#if defined(PRINTABLE_O3) && defined(ORP_NT)

#include <iostream.h>
#include <strstrea.h>
#include <ios.h>
#include <stdio.h>
#include <string.h>
#include "dumpjit.h"
#include "Flow_Graph.h"
#include "disass_str.h"
#include "IR.h"
#include "Bit_Vector_Group.h"
#include "..\dump\disasm_intf.h"
#include "dump_jit.h"
#include "dotfiles.h"
#include "jit_common.h"
#include <fstream.h>

#define WIDTH_LABEL 3        // for printing bytecode array index
#define WIDTH_POST_LABEL 2   // separate bc array index and bc disassembly
#define WIDTH_COL_1 20       // width of disassembly
#define WIDTH_POST_COL_1 1   // separate disassembly and IR
#define WIDTH_COL_2 20
#define WIDTH_POST_COL_2 1   // separate disassembly and IR

extern bool extended_dumpjit;
extern bool In_O3_statistics ;

static void dumpjit_dot_eh_nodes(ostream &cout, Flow_Graph *fg);
static void dumpjit_dot_eh_edges(ostream &cout, Flow_Graph *fg);

static void print_x86_instruction(ostream &cout, char *x86_code)
{
    char buf[100];
    x86_disasm(x86_code, buf, true, true);
    cout << buf;
}

static void print_x86_machine_inst(ostream &cout, char *x86_code)
{
    char buf[100];
    x86_disasm(x86_code, buf, true, false);
    cout << buf;
}

static void print_x86_assembly_inst(ostream &cout, char *x86_code)
{
    char buf[100];
    x86_disasm(x86_code, buf, false, true);
    cout << buf;
}

// Prints the bytecodes for the method (no IR instructions)
// to cout.
void dumpjit_bytecodes_only(ostream &cout, const unsigned char *bytecodes, unsigned bc_length,
                            void *m_handle, Class_Handle c_handle)
{
    unsigned bc_idx;
  for (bc_idx = 0;
       bc_idx < bc_length;
       /*bc_idx += instruction_length(bytecodes,bc_idx)*/)
    {
      unsigned old_bc_idx = bc_idx;
      cout.setf(ios::right, ios::adjustfield);
      cout.width(WIDTH_LABEL);
      cout << bc_idx << ": ";
      disass_str(bytecodes, bc_idx, cout, c_handle, 0);
      cout << endl;
    }
}

// Dumps the entire method to cout.  This dumps in DFS order.
// Includes the IR instructions, and basic block boundaries.
void dumpjit(ostream &cout, Flow_Graph *fg)
{
    fg->dumpjit_dfs(cout);
}

// Dumps the entire method to cout, like dumpjit(), except
// in "dot" format.
void dumpjit_dot(ostream &cout, Flow_Graph *fg)
{
    extern bool one_dot_file;
    cout << "digraph flowgraph {" << endl
        << "center=TRUE;" << endl
        << "page=\"8.5,11\";" << endl;
    if (one_dot_file)
        cout << "ratio=compress;" << endl << "size=\"7.5,10\";" << endl;
    else
        cout << "ratio=auto;" << endl;
    cout << "fontpath=\"c:\\winnt\\fonts\";" << endl
        << "node [shape=record,fontname=\"Courier\",fontsize=9];" << endl;
    dumpjit_dot_eh_nodes(cout, fg);
    fg->dumpjit_dot_hierarchical(cout);
    fg->dumpjit_dot_dfs_bounds_nodes(cout);
    dumpjit_dot_eh_edges(cout, fg);
    cout << "}" << endl;
}

static void print_bc_inst(ostream &cout, Flow_Graph *fg, unsigned &bc_idx)
{
    char str[1024];
    sprintf(str, "%s%u", fg->prefix_str, bc_idx);
    cout.setf(ios::right, ios::adjustfield);
    cout.width(WIDTH_LABEL);
    cout << str << ": ";
    disass_str(fg->bytecodes(), bc_idx, cout, fg->c_handle(), 0);
}

static void print_dot_string_safely(ostream &cout, int &last_len, const char *str)
{
    int len = strlen(str);
    last_len += len;
    if (last_len > 500)
    {
        cout << "|";
        last_len = len;
    }
    cout << str;
}

static void print_bc_inst_dot(ostream &cout, Flow_Graph *fg, unsigned &bc_idx, int &last_len,
                              const char *suffix)
{
    ostrstream buf;
    buf << bc_idx << ": ";
    disass_str(fg->bytecodes(), bc_idx, buf, fg->c_handle(), 1);
    buf << suffix << ends;
    print_dot_string_safely(cout, last_len, buf.str());
}

// Dumps a single basic block to cout.
void dump_jit_bb(ostream &cout, Cfg_Node *node, char *code_block, char *endl_str)
{
    ostrstream bc_out;
    Flow_Graph *fg = node->flowgraph;
    unsigned last_bc_index_seen = ~0u;
    Inst *cur_inst = node->IR_instruction_list()->next();
    Inst *last_inst = node->IR_instruction_list();
    while (cur_inst != last_inst)
    {
        unsigned cur_bc_idx = cur_inst->bc_index;
        if (cur_bc_idx == last_bc_index_seen || cur_bc_idx == (~0u))
        {
			if(!In_O3_statistics){
				// print the IR instruction without a BC instruction
				cout.width(WIDTH_LABEL+WIDTH_POST_LABEL+WIDTH_COL_1);
				cout << " ";
				cur_inst->print(cout);
			}
            if (code_block != NULL)
            {
                cout << " ";
                print_x86_instruction(cout, code_block + cur_inst->emitter_offset);
            }
            cout << endl_str;
        }
        else
        {
            // print all intervening bytecodes
            if (last_bc_index_seen != (~0u))
            {
                last_bc_index_seen += instruction_length(fg->bytecodes(),last_bc_index_seen);
				if(!In_O3_statistics){
					while (last_bc_index_seen < cur_bc_idx)
					{
						print_bc_inst(cout, fg, last_bc_index_seen);
						cout << endl_str;
					}
					assert(last_bc_index_seen == cur_bc_idx);
				}
            }
            else
                last_bc_index_seen = cur_bc_idx;
            
            // print current BC and current IR on one line
			if(!In_O3_statistics){
				print_bc_inst(bc_out, fg, last_bc_index_seen);
				bc_out << ends;
				int bc_strlen = bc_out.pcount();
				cout.width(WIDTH_LABEL+WIDTH_POST_LABEL+WIDTH_COL_1);
				cout.setf(ios::left, ios::adjustfield);
				cout << bc_out.str();
				bc_out.rdbuf()->freeze(0);
				bc_out.seekp(0);
				// If the line is too long, be sure to emit a newline
				if (bc_strlen > WIDTH_LABEL+WIDTH_POST_LABEL+WIDTH_COL_1)
				{
					cout << endl_str;
					cout.width(WIDTH_LABEL+WIDTH_POST_LABEL+WIDTH_COL_1);
					cout << " ";
				}
				cur_inst->print(cout);
			}
            if (code_block != NULL)
            {
                cout << " ";
                print_x86_instruction(cout, code_block + cur_inst->emitter_offset);
            }
            cout << endl_str;
        }
        if (cur_bc_idx != (~0u))
            last_bc_index_seen = cur_bc_idx;
        cur_inst = cur_inst->next();
    }
    cout << endl;
	
	//::Something lost after last_inst
	if(last_bc_index_seen != (~0u)){
		last_bc_index_seen += instruction_length(fg->bytecodes(),
											last_bc_index_seen);
		while (last_bc_index_seen < node->first_bc_idx() + node->bc_length()){
			print_bc_inst(cout, fg, last_bc_index_seen);
			last_bc_index_seen += instruction_length(fg->bytecodes(),
				last_bc_index_seen);
            cout << endl_str;
		}
	}
}

// Given the instruction and a bit position in inst->live_ranges_ended,
// determine whether the live range is ended, and if it is, whether
// the live range was a reference or a non-reference.
static const char *reference_change_str(Inst *inst, unsigned bit)
{
    Operand *opnd;
    if (!(inst->live_ranges_ended & (1u << bit)))
        return ".";
    if (inst->exp != NULL && inst->type() == JIT_TYPE_UNKNOWN)
    {
        assert(inst->is_assignment());
        Assign_Inst *assn = (Assign_Inst *)inst;
        switch (assn->real_type)
        {
        case JIT_TYPE_CLASS:
            return "L";
            break;
        case JIT_TYPE_UNKNOWN:
            return "?";
            break;
        default:
            return "x";
            break;
        }
    }
    // An odd-numbered bit position can only be an offset, not a reference.
    if (bit % 2 == 1)
        return "x";
    bit /= 2;
    if (bit == 0) // destination operand
       return "L"; // OPERAND must be inst->dst()->base()
    opnd = inst->src(bit-1);
    assert(opnd != NULL);
    if (opnd->is_status_flags())
        return "x";
    if (!opnd->is_reg())
    {
        if (opnd->base()->type == JIT_TYPE_ADDR) // vtable reference
            return "x";
        else
            return "L"; // OPERAND must be opnd->base()
    }
    switch (opnd->type)
    {
    case JIT_TYPE_CLASS:
    case JIT_TYPE_ARRAY:
        return "L"; // OPERAND must be opnd
    default:
        return "x";
    }
}

// makes sure the segment of text is < 512 characters or so
// to work around a "dot" bug
static void print_ir_inst_dot(ostream &cout, Inst *inst, int &last_len,
                              const char *suffix)
{
    ostrstream buf;
    buf << "[" << inst->global_id << "] ";
    if (inst->is_dead())
        buf << "XXX";
    else if (inst->is_spill == -1)
        buf << "vv ";
    else if (inst->is_spill == 1)
        buf << "^^ ";
    inst->print(buf);
    buf << "[";
    unsigned i;
    for (i=0; i<2*(MAX_SRCS+1); i++)
        buf << reference_change_str(inst, i);
    if (inst->changes_ret_addr())
        buf << " x";
    else
        buf << " .";
    buf << "]";
    buf << suffix << ends;
    print_dot_string_safely(cout, last_len, buf.str());
}

static void print_asm_inst_dot_1(ostream &cout, Inst *inst, char *code_block, int &last_len,
                                 const char *suffix)
{
    ostrstream buf;
    print_x86_machine_inst(buf, code_block + inst->emitter_offset);
    buf << suffix << ends;
    print_dot_string_safely(cout, last_len, buf.str());
}

static void print_asm_inst_dot_2(ostream &cout, Inst *inst, char *code_block, int &last_len,
                                 const char *suffix)
{
    ostrstream buf;
    print_x86_assembly_inst(buf, code_block + inst->emitter_offset);
    buf << suffix << ends;
    print_dot_string_safely(cout, last_len, buf.str());
}

// Returns 1 if it actually printed something.
static int dump_portion_dot(ostream &cout, Cfg_Node *node, char *endl_str, char *code_block,
                            int portion  // 0=bytecodes, 1=IR instructions, 2=asm1, 3=asm2
                            )
{
    Flow_Graph *fg = node->flowgraph;
    int last_len = 0;
    int result = 0;
    
    unsigned last_bc_index_seen = ~0u;
    Inst *cur_inst = node->IR_instruction_list()->next();
    Inst *last_inst = node->IR_instruction_list();
    if (cur_inst->bc_index != ~0u)
    {
        unsigned cur_bc_idx = node->first_bc_idx();
        while (cur_bc_idx < cur_inst->bc_index)
        {
            switch (portion)
            {
            case 0:
                print_bc_inst_dot(cout, fg, cur_bc_idx, last_len, endl_str);
                result = 1;
                break;
            case 1:
            case 2:
            case 3:
                cur_bc_idx += instruction_length(fg->bytecodes(), cur_bc_idx);
                print_dot_string_safely(cout, last_len, endl_str);
                break;
            default:
                assert(0);
                break;
            }
        }
    }
    while (cur_inst != last_inst)
    {
        unsigned cur_bc_idx = cur_inst->bc_index;
        if (cur_bc_idx == last_bc_index_seen || cur_bc_idx == (~0u))
        {
            // print the IR instruction without a BC instruction
            switch (portion)
            {
            case 0:
                print_dot_string_safely(cout, last_len, endl_str);
                break;
            case 1:
                print_ir_inst_dot(cout, cur_inst, last_len, endl_str);
                result = 1;
                break;
            case 2:
                print_asm_inst_dot_1(cout, cur_inst, code_block, last_len, endl_str);
                break;
            case 3:
                print_asm_inst_dot_2(cout, cur_inst, code_block, last_len, endl_str);
                break;
            default:
                assert(0);
                break;
            }
        }
        else
        {
            // print all intervening bytecodes
            if (last_bc_index_seen != (~0u))
            {
                last_bc_index_seen += instruction_length(fg->bytecodes(),last_bc_index_seen);
                while (last_bc_index_seen < cur_bc_idx)
                {
                    switch (portion)
                    {
                    case 0:
                        print_bc_inst_dot(cout, fg, last_bc_index_seen, last_len, endl_str);
                        result = 1;
                        break;
                    case 1:
                    case 2:
                    case 3:
                        last_bc_index_seen += instruction_length(fg->bytecodes(),last_bc_index_seen);
                        print_dot_string_safely(cout, last_len, endl_str);
                        break;
                    default:
                        assert(0);
                        break;
                    }
                }
                assert(last_bc_index_seen == cur_bc_idx);
            }
            else
                last_bc_index_seen = cur_bc_idx;
            
            // print current BC and current IR on one line
            switch (portion)
            {
            case 0:
                print_bc_inst_dot(cout, fg, last_bc_index_seen, last_len, endl_str);
                result = 1;
                break;
            case 1:
                last_bc_index_seen += instruction_length(fg->bytecodes(),last_bc_index_seen);
                print_ir_inst_dot(cout, cur_inst, last_len, endl_str);
                result = 1;
                break;
            case 2:
                last_bc_index_seen += instruction_length(fg->bytecodes(),last_bc_index_seen);
                print_asm_inst_dot_1(cout, cur_inst, code_block, last_len, endl_str);
                result = 1;
                break;
            case 3:
                last_bc_index_seen += instruction_length(fg->bytecodes(),last_bc_index_seen);
                print_asm_inst_dot_2(cout, cur_inst, code_block, last_len, endl_str);
                result = 1;
                break;
            default:
                assert(0);
                break;
            }
        }
        if (cur_bc_idx != (~0u))
            last_bc_index_seen = cur_bc_idx;
        cur_inst = cur_inst->next();
    }
    
    if (last_bc_index_seen != (~0u))
    {
        last_bc_index_seen += instruction_length(fg->bytecodes(),last_bc_index_seen);
        while (last_bc_index_seen < node->first_bc_idx() + node->bc_length())
        {
            switch (portion)
            {
            case 0:
                print_bc_inst_dot(cout, fg, last_bc_index_seen, last_len, endl_str);
                result = 1;
                break;
            case 1:
            case 2:
            case 3:
                last_bc_index_seen += instruction_length(fg->bytecodes(), last_bc_index_seen);
                print_dot_string_safely(cout, last_len, endl_str);
                break;
            default:
                assert(0);
                break;
            }
        }
    }
    
    return result;
}

// Dumps a single basic block to cout.
void dump_jit_bb_dot(ostream &cout, Cfg_Node *node, char *code_block, char *endl_str)
{
    cout << "{{";
    if (node->bc_length() > 0 && dump_portion_dot(cout, node, endl_str, code_block, 0))
        cout << "}|{";
    dump_portion_dot(cout, node, endl_str, code_block, 1);
    if (code_block != NULL)
    {
        cout << "}|{";
        dump_portion_dot(cout, node, endl_str, code_block, 2);
        cout << "}|{";
        dump_portion_dot(cout, node, endl_str, code_block, 3);
    }
    cout << "}}";
}

static void dumpjit_dot_eh_nodes(ostream &cout, Flow_Graph *fg)
{
    Eh_Node *eh = fg->handlers()->next();
    Cfg_Int i;
    while (eh != fg->handlers())
    {
        // Generate the node itself (no edges).
        cout << "EH" << eh->unique_label << " [color=red,fontcolor=red,label=\"{";
        for (i=0; i<eh->out_edge_size(); i++)
        {
            if (i > 0)
                cout << "|";
            cout << "<h" << i << ">";
            if (eh->out_edges(i)->class_handle == NULL)
                cout << "*";
            else
                cout << class_get_name(eh->out_edges(i)->class_handle);
        }
        cout << "}\"];" << endl;
        eh = eh->next();
    }
}

static void dumpjit_dot_eh_edges(ostream &cout, Flow_Graph *fg)
{
    Eh_Node *eh = fg->handlers()->next();
    Cfg_Int i;
    while (eh != fg->handlers())
    {
        // Generate the out edges.
        for (i=0; i<eh->out_edge_size(); i++)
        {
            cout << "EH" << eh->unique_label << ":h" << i << " -> BB"
                 << eh->out_edges(i)->handler->unique_label << "[color=red];" << endl;
        }

        // Generate the in edges.
        for (i=0; i<eh->in_edge_size(); i++)
        {
            cout << "BB" << eh->in_edges(i)->unique_label << " -> EH"
                 << eh->unique_label << "[color=red];" << endl;
        }
        eh = eh->next();
    }
}

static void print_reg(unsigned reg_no, bool &is_first, FILE *f)
{
    if (!is_first)
        fprintf(f, ",");
    if (reg_no < n_reg)
        fprintf(f, "%s", X86_Reg_Str[reg_no]);
    else
        fprintf(f, "%d", reg_no);
    is_first = false;
}

static void print_live(Cfg_Node *node, FILE *f)
{
    fprintf(f, " live=(");
    bool first = true;
    unsigned i;
    if (node->live->is_live_var(esp_reg))
    {
        fprintf(f, "ERROR:esp");
    }
    else
    {
        for (i=0; i<node->live->numbits(); i++)
        {
            if (node->live->is_live_var(i))
                print_reg(i, first, f);
        }
    }
    fprintf(f, ")");
}

static void do_dump_jit_for_profile(Flow_Graph *fg, FILE *f)
{
    Cfg_Node_List *cur;
    Cfg_Node_List *last = &fg->linear_node_ordering;
    for (cur=last->next(); cur!=last; cur=cur->next())
    {
        Cfg_Node *node = cur->node();
        fprintf(f, "// BB#%d:", node->label);
        fprintf(f, " (%d)", node->unique_label);
/*
        if (node->eh_in_edge() != NULL)
            fprintf(f, " (* exception handler *)");
        if (node == fg->prolog())
            fprintf(f, " (* prolog *)");
        if (node == fg->epilog())
            fprintf(f, " (* epilog *)");
*/
//        print_live(node, f);
        fprintf(f, "\n");
        
        Flow_Graph *bcode_fg = node->flowgraph;
        unsigned last_bc_index_seen = ~0u;
        Inst *cur_inst = node->IR_instruction_list()->next();
        Inst *last_inst = node->IR_instruction_list();
        char strbuf[100];
        // Print the bytecodes in the basic block that don't come between
        // two instructions' bc_index values.
        if (cur_inst->bc_index != ~0u)
        {
            unsigned cur_bc_idx = node->first_bc_idx();
            while (cur_bc_idx < cur_inst->bc_index)
            {
                ostrstream buf;
                print_bc_inst(buf, bcode_fg, cur_bc_idx);
                buf << ends;
                fprintf(f, "%s\n", buf.str());
            }
        }
        while (cur_inst != last_inst)
        {
            unsigned cur_bc_idx = cur_inst->bc_index;
            if (cur_bc_idx == last_bc_index_seen || cur_bc_idx == (~0u))
            {
                x86_disasm(fg->code_block + cur_inst->emitter_offset, strbuf, true, true);
				strbuf[54] = 0;
                fprintf(f, "\t%s", strbuf);
                if (extended_dumpjit)
                {
                    ostrstream irbuf;
                    int last_len = 0;
                    print_ir_inst_dot(irbuf, cur_inst, last_len, "");
                    irbuf << ends;
                    unsigned i;
                    for (i=strlen(strbuf); i<73; i++)
                        fprintf(f, " ");
                    fprintf(f, "%s", irbuf.str());
                }
                fprintf(f, "\n");
            }
            else
            {
                // print all intervening bytecodes
                if (last_bc_index_seen != (~0u))
                {
                    last_bc_index_seen += instruction_length(bcode_fg->bytecodes(),last_bc_index_seen);
                    while (last_bc_index_seen < cur_bc_idx)
                    {
                        ostrstream buf;
                        print_bc_inst(buf, bcode_fg, last_bc_index_seen);
                        buf << ends;
//                        fprintf(f, "%s\n", buf.str());
                    }
                    assert(last_bc_index_seen == cur_bc_idx);
                }
                else
                    last_bc_index_seen = cur_bc_idx;
                
                // print current BC and current IR on one line
                {
                    ostrstream buf;
                    print_bc_inst(buf, bcode_fg, last_bc_index_seen);
                    buf << ends;
//                    fprintf(f, "%s\n", buf.str());
                    last_bc_index_seen += instruction_length(bcode_fg->bytecodes(),last_bc_index_seen);
                    x86_disasm(fg->code_block + cur_inst->emitter_offset, strbuf, true, true);
					strbuf[54] = 0;
                    fprintf(f, "\t%s", strbuf);
                    if (extended_dumpjit)
                    {
                        ostrstream irbuf;
                        int last_len = 0;
                        print_ir_inst_dot(irbuf, cur_inst, last_len, "");
                        irbuf << ends;
                        unsigned i;
                        for (i=strlen(strbuf); i<73; i++)
                            fprintf(f, " ");
                        fprintf(f, "%s", irbuf.str());
                    }
                    fprintf(f, "\n");
                }
            }
            if (cur_bc_idx != (~0u))
                last_bc_index_seen = cur_bc_idx;
            cur_inst = cur_inst->next();
        }
        if (last_bc_index_seen != (~0u))
        {
            last_bc_index_seen += instruction_length(bcode_fg->bytecodes(),last_bc_index_seen);
            while (last_bc_index_seen < node->first_bc_idx() + node->bc_length())
            {
                ostrstream buf;
                print_bc_inst(buf, bcode_fg, last_bc_index_seen);
                last_bc_index_seen += instruction_length(bcode_fg->bytecodes(),last_bc_index_seen);
                buf << ends;
//                fprintf(f, "%s\n", buf.str());
            }
        }
    }
}

static void do_dump_jit(Flow_Graph *fg, FILE *f)
{
    Cfg_Node_List *cur;
    Cfg_Node_List *last = &fg->linear_node_ordering;
    for (cur=last->next(); cur!=last; cur=cur->next())
    {
        Cfg_Node *node = cur->node();
        fprintf(f, "// BB#%d:", node->label);
        fprintf(f, " (%d)", node->unique_label);
        if (node->eh_in_edge() != NULL)
            fprintf(f, " (* exception handler *)");
        if (node == fg->prolog())
            fprintf(f, " (* prolog *)");
        if (node == fg->epilog())
            fprintf(f, " (* epilog *)");
        print_live(node, f);
        fprintf(f, "\n");
        
        Flow_Graph *bcode_fg = node->flowgraph;
        unsigned last_bc_index_seen = ~0u;
        Inst *cur_inst = node->IR_instruction_list()->next();
        Inst *last_inst = node->IR_instruction_list();
        char strbuf[100];
        // Print the bytecodes in the basic block that don't come between
        // two instructions' bc_index values.
        if (cur_inst->bc_index != ~0u)
        {
            unsigned cur_bc_idx = node->first_bc_idx();
            while (cur_bc_idx < cur_inst->bc_index)
            {
                ostrstream buf;
                print_bc_inst(buf, bcode_fg, cur_bc_idx);
                buf << ends;
                fprintf(f, "%s\n", buf.str());
            }
        }
        while (cur_inst != last_inst)
        {
            unsigned cur_bc_idx = cur_inst->bc_index;
            if (cur_bc_idx == last_bc_index_seen || cur_bc_idx == (~0u))
            {
                x86_disasm(fg->code_block + cur_inst->emitter_offset, strbuf, true, true);
                fprintf(f, "\t%s", strbuf);
                if (extended_dumpjit)
                {
                    ostrstream irbuf;
                    int last_len = 0;
                    print_ir_inst_dot(irbuf, cur_inst, last_len, "");
                    irbuf << ends;
                    unsigned i;
                    for (i=strlen(strbuf); i<73; i++)
                        fprintf(f, " ");
                    fprintf(f, "%s", irbuf.str());
                }
                fprintf(f, "\n");
            }
            else
            {
                // print all intervening bytecodes
                if (last_bc_index_seen != (~0u))
                {
                    last_bc_index_seen += instruction_length(bcode_fg->bytecodes(),last_bc_index_seen);
                    while (last_bc_index_seen < cur_bc_idx)
                    {
                        ostrstream buf;
                        print_bc_inst(buf, bcode_fg, last_bc_index_seen);
                        buf << ends;
                        fprintf(f, "%s\n", buf.str());
                    }
                    assert(last_bc_index_seen == cur_bc_idx);
                }
                else
                    last_bc_index_seen = cur_bc_idx;
                
                // print current BC and current IR on one line
                {
                    ostrstream buf;
                    print_bc_inst(buf, bcode_fg, last_bc_index_seen);
                    buf << ends;
                    fprintf(f, "%s\n", buf.str());
                    last_bc_index_seen += instruction_length(bcode_fg->bytecodes(),last_bc_index_seen);
                    x86_disasm(fg->code_block + cur_inst->emitter_offset, strbuf, true, true);
                    fprintf(f, "\t%s", strbuf);
                    if (extended_dumpjit)
                    {
                        ostrstream irbuf;
                        int last_len = 0;
                        print_ir_inst_dot(irbuf, cur_inst, last_len, "");
                        irbuf << ends;
                        unsigned i;
                        for (i=strlen(strbuf); i<73; i++)
                            fprintf(f, " ");
                        fprintf(f, "%s", irbuf.str());
                    }
                    fprintf(f, "\n");
                }
            }
            if (cur_bc_idx != (~0u))
                last_bc_index_seen = cur_bc_idx;
            cur_inst = cur_inst->next();
        }
        if (last_bc_index_seen != (~0u))
        {
            last_bc_index_seen += instruction_length(bcode_fg->bytecodes(),last_bc_index_seen);
            while (last_bc_index_seen < node->first_bc_idx() + node->bc_length())
            {
                ostrstream buf;
                print_bc_inst(buf, bcode_fg, last_bc_index_seen);
                last_bc_index_seen += instruction_length(bcode_fg->bytecodes(),last_bc_index_seen);
                buf << ends;
                fprintf(f, "%s\n", buf.str());
            }
        }
    }
}

#ifdef DUMP_JIT

void O3_dump_jit_for_profile(Flow_Graph *fg)
{
    extern bool O3_no_dumpjit;
	extern FILE* fp_O3_dumpjit_profile;
    if (O3_no_dumpjit)
        return;

//    FILE *_f = acquire_dump_jit_file(fg->c_handle());
	if ( fp_O3_dumpjit_profile == NULL)
		fp_O3_dumpjit_profile = fopen( "o3_dumpjit.txt", "w+");

    Method_Handle mh = fg->m_handle();
    Class_Handle ch = method_get_class(mh);

    fprintf( fp_O3_dumpjit_profile, "Method %s.%s%s\n",
        class_get_name(ch), method_get_name(mh), method_get_descriptor(mh));

    do_dump_jit_for_profile(fg, fp_O3_dumpjit_profile);

//    fprintf( fp_O3_dumpjit_profile, "End of Method %s.%s%s\n\n",
//        class_get_name(ch), method_get_name(mh), method_get_descriptor(mh));

	fflush( fp_O3_dumpjit_profile);
//    release_dump_jit_file();
}

void O3_dump_jit(Flow_Graph *fg)
{
    extern bool O3_no_dumpjit;
    if (O3_no_dumpjit)
        return;

    FILE *_f = acquire_dump_jit_file(fg->c_handle());

    Method_Handle mh = fg->m_handle();
    Class_Handle ch = method_get_class(mh);

    fprintf(_f, "Method %s.%s%s\n",
        class_get_name(ch), method_get_name(mh), method_get_descriptor(mh));

    do_dump_jit(fg, _f);

    fprintf(_f, "End of Method %s.%s%s\n\n",
        class_get_name(ch), method_get_name(mh), method_get_descriptor(mh));

    release_dump_jit_file();
}
#endif

//
// write BB 1 (2 insts) -> BB 2, BB 3, EH 1
//
static void dump_bb_info(Cfg_Node *node, 
                         Closure *c) 
{
    ostream *cout = (ostream*)c;
    int n_inst = 0;
    Inst *head = node->IR_instruction_list();
    Inst * i;
    for (i = head->next(); i != head; i = i->next())
        n_inst++;
    *cout << "BB " << node->label << " ( " << n_inst << " insts ) : ";
    Cfg_Int j;
    for (j = 0; j < node->out_edge_size(); j++) 
    {
        *cout << "BB " << node->out_edges(j)->label << " ";
    }
    Eh_Node *eh = node->eh_out_edge();
    if (eh != NULL) 
        *cout << "EH " << eh->label;
    *cout << endl;
}

//
// dump out control flow graph
//      number of bb 12
//      number of eh  2
//      BB1 (2 insts) -> BB2, BB3, EH1
//      BB2 (5 insts) -> BB3
//      EH1 -> BB4, BB5
//
void dump_control_flow(Flow_Graph *fg) 
{
    char filename[MAX_FILENAME];

    // Create a filename based on the class name, method name, and
    // signature. e.g, foo.dot
    create_filename(fg, filename, ""); 
    //
    // change file name from foo.dot to foo.cfg
    //
    size_t len = strlen(filename);
    filename[len-3] = 'c'; 
    filename[len-2] = 'f'; 
    filename[len-1] = 'g';
    // Try to open the file for output.
    ofstream fout(filename);
    if (fout) 
    {
        //
        // write number of bb 12
        //
        int max_label = fg->reassign_label();
        fout << "MAXBBS " << max_label << endl;
        //
        // write number of eh  3
        //
        int n_eh = 0;
        Eh_Node *head = fg->handlers();
        Eh_Node *eh;
        for (eh =head->next(); eh!=head; eh=eh->next())
            n_eh++;
        fout << "MAXEHS " << n_eh << endl;
        //
        // write BB 1 (2 insts) -> BB 2, BB 3, EH 1
        //
        fg->apply(dump_bb_info,(Closure*)&fout);
        //
        // write EH 1 -> BB 4, BB 5
        //
        for (eh =head->next(); eh!=head; eh=eh->next()) 
        {
            Cfg_Int out_size = eh->out_edge_size();
            fout << "EH " << eh->label << " : ";
            Cfg_Int i;
            for (i=0; i < out_size; i++)
            {
                fout << "BB " << eh->out_edges(i)->handler->label << " ";
            }
            fout << endl;
        }
    }

}
#endif // PRINTABLE_O3
