// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/debugger/jvmdi_common.cpp,v 1.2 2001/09/19 01:10:12 ywang13 Exp $
//


#include "platform.h"
#include <assert.h>
#include <iostream.h>

#ifdef ORP_NT
#include "orp_process.h"
#endif

#include "object_layout.h"
#include "environment.h"
#include "orp_types.h"
#include "jit_runtime_support.h"
#include "method_lookup.h"
#include "stack_manipulation.h"
#include "orp_threads.h"
#include "orp_utils.h"
#include "orp_synch.h"
#include "jit_intf.h"
#include "jit_intf_cpp.h"
#include "nogc.h"
#include "compile.h"
#include "orp_stats.h"
#include "finalize.h"
#include "Class_Loader.h"

#include "root_set_enum.h"
#include "enum_trampoline.h"
#include "exceptions.h"

#include "jni.h"

#include "jvmdi_clean.h"
#include "debugger_jvmdi_ia32.h"

bool report_jvmdi_event_class_prepare = false;
bool report_jvmdi_event_thread_start = false;

#ifdef USE_IA64_JIT
class Level_1a_JIT {
public:
    void *
        get_address_of_var(Frame_Context    *context,
                           Boolean          is_first,
                           unsigned         var_no)
    { return 0; };

}; //Level_1a_JIT
#else
#include "level_1a_jit_intf.h"
#include "disasm_intf.h"
#endif

#ifdef USE_IA64_JIT
#else
extern 
#endif
JIT *o1_jit;

void orp_delete_global_object_handle(Object_Handle handle);   // bugbug -- where does this go?


JVMDI_EventHook jvmdi_hook_function = 0;

extern const struct JNIEnv_ *jni_native_intf;

extern jframeID global_frame;

extern jvmdi_bp breakpoints[NBREAKPOINTS];

unsigned instruction_length(const unsigned char *bc, const unsigned idx);

#define MATCH(slot1, slot2, ty1, ty2)  \
{   \
    if ((slot1) == (slot2) && (ty1) == (ty2)) \
       return false;   \
}   \

static bool is_type_mismatch(Method_Handle mh,
                             int           slot,
                             Java_Type     ty)
{
    const unsigned char *first_bc = method_get_byte_code_addr(mh);
    const unsigned char *last_bc  = first_bc + method_get_byte_code_size(mh);
    const unsigned char *bc = first_bc;
    while (bc < last_bc)
    {
        switch(*bc)
        {
        case 0x15:	// iload
        case 0x16:	// lload
        case 0x17:	// fload
        case 0x18:	// dload
        case 0x19:	// aload
            MATCH(slot, bc[1], ty, "IJFDL"[*bc - 0x15]);
            break;
        case 0x1a: case 0x1b: case 0x1c: case 0x1d:	// iload_{0,1,2,3}
            MATCH(slot, *bc-0x1a, ty, JAVA_TYPE_INT);
            break;
        case 0x22: case 0x23: case 0x24: case 0x25:	// fload_{0,1,2,3}
            MATCH(slot, *bc-0x22, ty, JAVA_TYPE_FLOAT);
            break;
        case 0x2a: case 0x2b: case 0x2c: case 0x2d:	// aload_{0,1,2,3}
            MATCH(slot, *bc-0x2a, ty, JAVA_TYPE_CLASS);
            break;
        case 0x1e: case 0x1f: case 0x20: case 0x21:	// lload_{0,1,2,3}
            MATCH(slot, *bc-0x1e, ty, JAVA_TYPE_LONG);
            break;
        case 0x26: case 0x27: case 0x28: case 0x29:	// dload_{0,1,2,3}
            MATCH(slot, *bc-0x26, ty, JAVA_TYPE_DOUBLE);
            break;        
        case 0x36:	// istore
        case 0x37:	// lstore
        case 0x38:	// fstore
        case 0x39:	// dstore
        case 0x3a:	// astore
            MATCH(slot, bc[1], ty, "IJFDL"[*bc - 0x36]);
            break;
        case 0x4b: case 0x4c: case 0x4d: case 0x4e:	// astore_{0,1,2,3}
            MATCH(slot, *bc-0x4b, ty, JAVA_TYPE_CLASS);
            break;
        case 0x3b: case 0x3c: case 0x3d: case 0x3e:	// istore_{0,1,2,3}
            MATCH(slot, *bc-0x3b, ty, JAVA_TYPE_INT);
            break;
        case 0x43: case 0x44: case 0x45: case 0x46: // fstore_{0,1,2,3}
            MATCH(slot, *bc-0x43, ty, JAVA_TYPE_FLOAT);
            break;
        case 0x3f: case 0x40: case 0x41: case 0x42: // lstore_{0,1,2,3}
            MATCH(slot, *bc-0x3f, ty, JAVA_TYPE_LONG);
            break;
        case 0x47: case 0x48: case 0x49: case 0x4a:	// dstore_{0,1,2,3}
            MATCH(slot, *bc-0x47, ty, JAVA_TYPE_DOUBLE);
            break;        
        case 0xc4:	// wide
            if (bc[1] != 0x84) { 
                int index = (bc[2] << 8) + bc[3];
                switch (bc[1]) {
                case 0x15:	// iload
                case 0x16:	// lload
                case 0x17:	// fload
                case 0x18:	// dload
                case 0x19:	// aload
                    MATCH(slot, index, ty, "IJFDL"[bc[1] - 0x15]);
                    break;
                case 0x36:	// istore
                case 0x37:	// lstore
                case 0x38:	// fstore
                case 0x39:	// dstore
                case 0x3a:	// astore
                    MATCH(slot, index, ty, "IJFDL"[bc[1] - 0x36]);
                    break;
                }
            }
            break;
        }
        bc += instruction_length(first_bc, bc-first_bc);
    }
    return true;
}


JVMDI_AllocHook hook_a = 0;
JVMDI_DeallocHook hook_d = 0;

jvmdiError JNICALL
SetAllocationHooks(JVMDI_AllocHook ahook, JVMDI_DeallocHook dhook)
{
   assert(0);

   hook_a = ahook;
   hook_d = dhook;
   return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
Allocate(jlong size, jbyte** memPtr)
{
    if (hook_a) {
        jvmdiError je = hook_a(size, memPtr);
        return je;
    }
    else {
        jbyte *p_byte = (jbyte *)malloc( (uint32)size);
        *memPtr = p_byte;
        if (*p_byte == 0) return JVMDI_ERROR_NULL_POINTER;
        return JVMDI_ERROR_NONE;
    }
}


jvmdiError JNICALL
Deallocate(jbyte* mem)
{
    assert(0);

    if (hook_d) {
        jvmdiError je = hook_d(mem);
        return je;
    }
    else {
        free( (void *)mem);
        return JVMDI_ERROR_NONE;
    }
}


jvmdiError JNICALL
GetThreadStatus(jthread thread, jint *threadStatusPtr, jint *suspendStatusPtr)
{
    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    Object_Handle oh = (Object_Handle)thread;
    Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
    ORP_thread *p_thr = (ORP_thread *)p_jlt->PrivateInfo;

    if (p_thr == 0)
        return JVMDI_ERROR_INVALID_THREAD;

#ifdef _DEBUG
    if (p_thr->p_latest_mon_enter_lock_block)
        *threadStatusPtr = JVMDI_THREAD_STATUS_WAIT;
#else
    assert(0);
#endif
    switch (p_thr->app_status) {
        case zip: 
            {
            *threadStatusPtr = JVMDI_THREAD_STATUS_UNKNOWN;
            break;
            }
        case thread_is_dying:
            {
            *threadStatusPtr = JVMDI_THREAD_STATUS_ZOMBIE;
            break;
            }
        case thread_is_running:
            {
            *threadStatusPtr = JVMDI_THREAD_STATUS_RUNNING;
            break;
            }
        case thread_is_sleeping:
            {
            *threadStatusPtr = JVMDI_THREAD_STATUS_SLEEPING;
            break;
            }
        case thread_is_waiting:
            {
            *threadStatusPtr = JVMDI_THREAD_STATUS_WAIT;
            break;
            }
        default:
            {
            assert(0);
            }
    }

    if(p_thr->thread_is_java_suspended)
        *suspendStatusPtr = JVMDI_SUSPEND_STATUS_SUSPENDED;
    
    assert(0);
    // if (p_thr->is_thread_at_a_jvmdi_breakpoint)
    //    *suspendStatusPtr = JVMDI_SUSPEND_STATUS_BREAK;
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetAllThreads(jint *threadsCountPtr, jthread **threadsPtr)
{
    assert(0);

    ORP_thread *p_cursor = p_active_threads_list;
    uint32 number_of_java_threads = 0;

    while(p_cursor) {
        number_of_java_threads++;
        p_cursor = p_cursor->p_active;
    }

    jlong size_needed = number_of_java_threads * sizeof(Object_Handle);

    Object_Handle handle_array = 0;

    jvmdiError stat = Allocate(size_needed, (jbyte **)&handle_array);
    if (stat != JVMDI_ERROR_NONE)
        return JVMDI_ERROR_OUT_OF_MEMORY;

    p_cursor = p_active_threads_list;

    while(p_cursor) {
        Object_Handle oh = orp_create_global_object_handle();
        oh->java_reference = (struct Java_java_lang_Object *)
                                p_cursor->p_java_lang_thread;
        handle_array = oh;
        handle_array++;
        p_cursor = p_cursor->p_active;
    }
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
SuspendThreadxx(jthread thread)  //bugbug temporary hack, SuspendThread() collides w/ NT API
{

    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    orp_disable_gc();
    Object_Handle oh = (Object_Handle)thread;
    Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
    ORP_thread *p_thr = (ORP_thread *)p_jlt->PrivateInfo;
    
    if (p_thr == 0) {
        orp_enable_gc();
        return JVMDI_ERROR_INVALID_THREAD;
    }
    
    Java_java_lang_Thread *p_cjlt = (Java_java_lang_Thread *)(p_thr->p_java_lang_thread);

    java_lang_Thread_suspend0 (p_cjlt);

    orp_enable_gc();
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
ResumeThreadxx(jthread thread)   //bugbug temporary hack, SuspendThread() collides w/ NT API
{
    assert(0);
#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    orp_disable_gc();

    Object_Handle oh = (Object_Handle)thread;
    Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
    ORP_thread *p_thr = (ORP_thread *)p_jlt->PrivateInfo;
    
    if (p_thr == 0) {
        orp_enable_gc();
        return JVMDI_ERROR_INVALID_THREAD;
    }

    java_lang_Thread_resume0 ( (Java_java_lang_Thread *)p_thr->p_java_lang_thread);

    orp_enable_gc();
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
StopThread(jthread thread, jobject exception)
{
    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    Object_Handle oh = (Object_Handle)thread;
    Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
    ORP_thread *p_thr = (ORP_thread *)p_jlt->PrivateInfo;

    if (p_thr == 0)
        return JVMDI_ERROR_INVALID_THREAD;

    Object_Handle_Struct *p_ohs = (Object_Handle_Struct *)exception;

    java_lang_Thread_stop0 ( 
        (Java_java_lang_Thread *)p_thr->p_java_lang_thread,
        p_ohs->java_reference );
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
InterruptThread(jthread thread)
{
    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    Object_Handle oh = (Object_Handle)thread;
    Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
    ORP_thread *p_thr = (ORP_thread *)p_jlt->PrivateInfo;
    
    if (p_thr == 0)
        return JVMDI_ERROR_INVALID_THREAD;

    java_lang_Thread_interrupt0( (Java_java_lang_Thread *)p_thr->p_java_lang_thread);
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetThreadInfo(jthread thread, JVMDI_thread_info *infoPtr)
{
    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    Object_Handle oh = (Object_Handle)thread;
    Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
    ORP_thread *p_thr = (ORP_thread *)p_jlt->PrivateInfo;
    
    if (p_thr == 0)
        return JVMDI_ERROR_INVALID_THREAD;
    
    jlong len = p_thr->p_java_lang_thread->name->length;

    jbyte *p_mem = 0;

    jvmdiError stat = Allocate(len, &p_mem);
    if (stat != JVMDI_ERROR_NONE)
        assert(0);

    for (int xx = 0; xx < len; xx++) {
        unsigned short ss = p_thr->p_java_lang_thread->name->body[xx];
        p_mem[xx] = (signed char)ss;
    }

    infoPtr->name = (char *)p_mem;
    infoPtr->priority = p_thr->p_java_lang_thread->priority;
    infoPtr->is_daemon = (unsigned char)p_thr->p_java_lang_thread->daemon;

    Object_Handle tg = orp_create_global_object_handle();

    tg->java_reference = (struct Java_java_lang_Object*)
                            p_thr->p_java_lang_thread->group;

    infoPtr->thread_group = (jthreadGroup)tg;

    Object_Handle ccl = orp_create_global_object_handle();

    ccl->java_reference = 0;  // bugbug what should this be??

    infoPtr->context_class_loader = ccl;
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetOwnedMonitorInfo(jthread thread, JVMDI_owned_monitor_info *infoPtr)
{
    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    Object_Handle oh = (Object_Handle)thread;
    Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
    ORP_thread *p_thr = (ORP_thread *)p_jlt->PrivateInfo;
    
    if (p_thr == 0)
        return JVMDI_ERROR_INVALID_THREAD;

    assert(0);  // optional -- not supported
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetCurrentContendedMonitor(jthread thread, jobject *monitorPtr)
{
    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    Object_Handle oh = (Object_Handle)thread;
    Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
    ORP_thread *p_thr = (ORP_thread *)p_jlt->PrivateInfo;
    
    if (p_thr == 0)
        return JVMDI_ERROR_INVALID_THREAD;

    assert(0);  // optional -- not supported
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
RunDebugThread(jthread thread, JVMDI_StartFunction proc, void *arg, int priority)
{
    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    Object_Handle oh = (Object_Handle)thread;
    Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
    ORP_thread *p_thr = (ORP_thread *)p_jlt->PrivateInfo;
    
    if (p_thr == 0)
        return JVMDI_ERROR_INVALID_THREAD;

	unsigned thread_id = 0;

	unsigned long stat = _beginthreadex( 0, 
                                         (64*1024), 
                                         ( unsigned int (__stdcall *)(void *) )proc,
                                         arg,
                                         0, // running
                                         &thread_id);
#endif
    return JVMDI_ERROR_NONE;
}


#define MAX_GROUPS 64
Java_java_lang_ThreadGroup *tga[MAX_GROUPS];


jvmdiError JNICALL
GetTopThreadGroups(jint *groupCountPtr, jthreadGroup **groupsPtr)
{
    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    ORP_thread *p_cursor = p_active_threads_list;

    for (int xx = 0; xx < MAX_GROUPS; xx++)
        tga[xx] = 0;

    while(p_cursor) {

        int index = 0;
        Hjava_lang_ThreadGroup *p_group = p_cursor->p_java_lang_thread->group;

        for (; index < MAX_GROUPS; index++) {

            if ( tga[index] == p_group)
                break;
        }
        assert(index < MAX_GROUPS);

        if (tga[index] == 0) {
            tga[index] = p_group;
        }
        p_cursor = p_cursor->p_active;
    }

    int count = 0;
    for (xx = 0; xx < MAX_GROUPS; xx++) {
        if (tga[xx])
            count++;
    }
    
    jlong size_needed = count * sizeof(Object_Handle);

    Object_Handle handle_array = 0;

    jvmdiError stat = Allocate(size_needed, (jbyte **)&handle_array);
    if (stat != JVMDI_ERROR_NONE)
        return JVMDI_ERROR_OUT_OF_MEMORY;

    for(xx = 0; xx < count; xx++) {

        Object_Handle oh = orp_create_global_object_handle();
        oh->java_reference = (struct Java_java_lang_Object *) tga[xx];
        handle_array = oh;
        handle_array++;
    }
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetThreadGroupInfo(jthreadGroup group, JVMDI_thread_group_info *infoPtr)
{
    assert(0);
#if 0
    // (mjc 20000519)
    // All this stuff is not JNI-compliant.  Should we re-write it?

    Object_Handle oh = (Object_Handle)group;
    Hjava_lang_ThreadGroup *p_jltg = (Hjava_lang_ThreadGroup *)oh->java_reference;

    if (p_jltg == 0)
        return JVMDI_ERROR_INVALID_THREAD;

    Object_Handle ph = orp_create_global_object_handle();
    ph->java_reference = (struct Java_java_lang_Object *) p_jltg->parent;

    infoPtr->parent = (jthreadGroup)ph;

    jlong len = p_jltg->name->count;

    jbyte *p_mem = 0;

    jvmdiError stat = Allocate(len, &p_mem);
    if (stat != JVMDI_ERROR_NONE)
        assert(0);

    for (int xx = 0; xx < len; xx++) {
        uint16 ss = p_jltg->name->value->body[xx + p_jltg->name->offset];
        p_mem[xx] = (signed char)ss;
    }

    infoPtr->name = (char *)p_mem;

    infoPtr->max_priority = p_jltg->maxPriority;
    infoPtr->is_daemon = (unsigned char)p_jltg->daemon;
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetThreadGroupChildren(jthreadGroup group,
                 jint *threadCountPtr, jthread **threadsPtr,
                 jint *groupCountPtr, jthreadGroup **groupsPtr)
{
    assert(0);

#if 0
    // (mjc 20000519)
    // All this stuff is not JNI-compliant.  Should we re-write it?

    Object_Handle oh = (Object_Handle)group;
    Hjava_lang_ThreadGroup *p_jltg = (Hjava_lang_ThreadGroup *)oh->java_reference;

    struct JavaArrayOfObject *p_threads = p_jltg->threads;

    Object_Handle p_thread_mem = 0;
    int size = p_jltg->nthreads * (sizeof(void *));
    jvmdiError stat = Allocate( size, (jbyte **)&p_thread_mem);
    if (stat != JVMDI_ERROR_NONE)
        assert(0);

    *threadCountPtr = p_jltg->nthreads;
    *threadsPtr = (jthread *)p_thread_mem;

    for (int xx = 0; xx < p_jltg->nthreads; xx++) {

        Object_Handle p_handle = orp_create_global_object_handle();
        p_handle->java_reference = (struct Java_java_lang_Object *) p_threads->body[xx];
        p_thread_mem = p_handle;
        p_thread_mem++;
    }

    struct JavaArrayOfObject *p_groups = p_jltg->groups;

    Object_Handle p_group_mem = 0;
    size = p_jltg->ngroups * (sizeof(void *));
    stat = Allocate( size, (jbyte **)&p_group_mem);
    if (stat != JVMDI_ERROR_NONE)
        assert(0);

    *groupCountPtr = p_jltg->ngroups;
    *groupsPtr = (jthread *)p_group_mem;

    for (int zz = 0; zz < p_jltg->ngroups; zz++) {

        Object_Handle p_handle = orp_create_global_object_handle();
        p_handle->java_reference = (struct Java_java_lang_Object *) p_groups->body[xx];
        p_group_mem = p_handle;
        p_group_mem++;        
    }
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetCallerFrame(jframeID called, jframeID *framePtr)
{
    assert(0);

    Frame_Context *p_fc = (Frame_Context *)called;

    Boolean yn = unwind_to_next_java_frame(p_fc,     // in out
                                      p_fc->is_first,    // in
                                      FALSE  // in
                                      );
    if (yn == FALSE)
        return JVMDI_ERROR_NO_MORE_FRAMES;

    *framePtr = (jframeID)p_fc;

    return JVMDI_ERROR_NONE;
}


//
// assume all frames are valid for now
//

static bool is_native_frame(jframeID frame)
{
    return false;
}


static bool orp_is_valid_frame(jframeID frame)
{
    return true;
}


static jvmdiError JNICALL
check_errors(jframeID frame, jint slot, Java_Type ty)
{
    if (frame == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    else if (is_native_frame(frame))
        return JVMDI_ERROR_OPAQUE_FRAME;
    else if (!orp_is_valid_frame(frame))
        return JVMDI_ERROR_INVALID_FRAMEID;

    Method_Handle mh = frame_get_method(frame);
    if (slot >= (int)method_get_max_locals(mh))
        return JVMDI_ERROR_INVALID_SLOT;
    else if (is_type_mismatch(mh, slot, ty))
        return JVMDI_ERROR_TYPE_MISMATCH;

    return JVMDI_ERROR_NONE;
}


///////////////////////////////////////////////////////////////////
// 
// retrieve the value of a local variable
//
///////////////////////////////////////////////////////////////////
#ifdef USE_IA64_JIT

#define GET_VALUE(frame,slot,val_ptr,val_ty,ty) return JVMDI_ERROR_NONE;

#else

#define GET_VALUE(frame,slot,val_ptr,val_ty,ty) \
{ \
    jvmdiError err = check_errors(frame, slot, ty); \
    if (err != JVMDI_ERROR_NONE) return err;  \
    if ((ty == JAVA_TYPE_LONG)||(ty == JAVA_TYPE_DOUBLE)) { \
        *valuePtr = *((val_ty*)o1_jit->get_address_of_var(frame, frame->is_first, slot+1)); \
    }\
    else \
        *valuePtr = *((val_ty*)o1_jit->get_address_of_var(frame, frame->is_first, slot)); \
    return JVMDI_ERROR_NONE; \
} \

#endif

/*

        
    else \ 
*/

///////////////////////////////////////////////////////////////////
// 
// retrieve the value of a local variable
//
///////////////////////////////////////////////////////////////////

jvmdiError JNICALL
GetLocalObject(jframeID frame, jint slot, jobject *valuePtr)
{
//    GET_VALUE(frame, slot, valuePtr, jobject, JAVA_TYPE_CLASS);
//    jvmdiError err = check_errors(frame, slot, JAVA_TYPE_CLASS); 
//    if (err != JVMDI_ERROR_NONE) return err;  
    Object_Handle oh = orp_create_global_object_handle();
#ifdef _USE_IA64_JIT
    assert(0);
    return JVMDI_ERROR_NONE;
#else
    oh->java_reference = *((Java_java_lang_Object **)o1_jit->get_address_of_var(frame, frame->is_first, slot));
	*valuePtr = oh;
    return JVMDI_ERROR_NONE; 
#endif
}

jvmdiError JNICALL
GetLocalInt(jframeID frame, jint slot, jint *valuePtr)
{
    GET_VALUE(frame, slot, valuePtr, jint, JAVA_TYPE_INT);
}


jvmdiError JNICALL
GetLocalLong(jframeID frame, jint slot, jlong *valuePtr)
{
    GET_VALUE(frame, slot, valuePtr, jlong, JAVA_TYPE_LONG);
}


jvmdiError JNICALL
GetLocalFloat(jframeID frame, jint slot, jfloat *valuePtr)
{
    GET_VALUE(frame, slot, valuePtr, jfloat, JAVA_TYPE_FLOAT);
}


jvmdiError JNICALL
GetLocalDouble(jframeID frame, jint slot, jdouble *valuePtr)
{
    GET_VALUE(frame, slot, valuePtr, jdouble, JAVA_TYPE_DOUBLE);
}


#ifdef USE_IA64_JIT

#define SET_VALUE(frame,slot,val,val_ty,ty)  return JVMDI_ERROR_NONE;

#else

#define SET_VALUE(frame,slot,val,val_ty,ty) \
{ \
    jvmdiError err = check_errors(frame, slot, ty); \
    if (err != JVMDI_ERROR_NONE) return err;  \
    *((val_ty*)o1_jit->get_address_of_var(frame, frame->is_first, slot)) = val; \
    return JVMDI_ERROR_NONE; \
} \

#endif


//
// set local variable with new value
//


jvmdiError JNICALL
SetLocalObject(jframeID frame, jint slot, jobject value)
{
    SET_VALUE(frame, slot, value, jobject, JAVA_TYPE_CLASS);
}


jvmdiError JNICALL
SetLocalInt(jframeID frame, jint slot, jint value)
{
    SET_VALUE(frame, slot, value, jint, JAVA_TYPE_INT);
}


jvmdiError JNICALL
SetLocalLong(jframeID frame, jint slot, jlong value)
{
    SET_VALUE(frame, slot, value, jlong, JAVA_TYPE_LONG);
}


jvmdiError JNICALL
SetLocalFloat(jframeID frame, jint slot, jfloat value)
{
    SET_VALUE(frame, slot, value, jfloat, JAVA_TYPE_FLOAT);
}


jvmdiError JNICALL
SetLocalDouble(jframeID frame, jint slot, jdouble value)
{
    SET_VALUE(frame, slot, value, jdouble, JAVA_TYPE_DOUBLE);
}


jvmdiError JNICALL
SetFieldAccessWatch(jclass clazz, jfieldID field)
{
    assert(0);  // optional -- not implemented

    return (jvmdiError) 0;
}


jvmdiError JNICALL
ClearFieldAccessWatch(jclass clazz, jfieldID field)
{
    assert(0);  // optional -- not implemented

    return (jvmdiError) 0;
}


jvmdiError JNICALL
SetFieldModificationWatch(jclass clazz, jfieldID field)
{
    assert(0);  // optional -- not implemented

    return (jvmdiError) 0;
}


jvmdiError JNICALL
ClearFieldModificationWatch(jclass clazz, jfieldID field)
{
    assert(0);  // optional -- not implemented

    return (jvmdiError) 0;
}


jvmdiError JNICALL
GetClassSignature(jclass clazz, char **sigPtr)
{
    assert(0);
    Class *p_cls = (Class *)((Object_Handle)clazz)->java_reference;

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetClassStatus(jclass clazz, jint *statusPtr)
{
    assert(0);

    *statusPtr = 0;

    Object_Handle oh = (Object_Handle)clazz;
    Class *p_cls = (Class *)oh->java_reference;
    
    switch (p_cls->state) {
    
    case ST_Prepared:
        {
            *statusPtr |= JVMDI_CLASS_STATUS_PREPARED;
            break;
        }
    case ST_Initialized:
        {
            *statusPtr |= ( JVMDI_CLASS_STATUS_INITIALIZED |
                            JVMDI_CLASS_STATUS_PREPARED        );
            break;
        }

    case ST_Error:
        {
            *statusPtr |= JVMDI_CLASS_STATUS_ERROR;
            break;
        }
    default:
        {
            break;
        }
    }
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetSourceFileName(jclass clazz, char **sourceNamePtr)
{
    assert(0);

    if (clazz == NULL || sourceNamePtr == NULL) 
        return JVMDI_ERROR_NULL_POINTER;
    Class *p_cls = (Class *)((Object_Handle)clazz)->java_reference;
    const char *file_name = class_get_source_file_name(p_cls);
    if (file_name == NULL)
    {
        *sourceNamePtr = NULL;
        return JVMDI_ERROR_ABSENT_INFORMATION;
    }
    size_t len = strlen(file_name);
    //
    // allocate space and copy field name
    //
    jvmdiError err = Allocate((jlong)(len + 1), (jbyte**)sourceNamePtr);
    if (err != JVMDI_ERROR_NONE) return err;
    strcpy(*sourceNamePtr, file_name);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetClassModifiers(jclass clazz, jint *modifiersPtr)
{
    assert(0);

    if (clazz == NULL || modifiersPtr == NULL) 
        return JVMDI_ERROR_NULL_POINTER;
    Class *p_cls = (Class *)((Object_Handle)clazz)->java_reference;
    *modifiersPtr = class_get_flags(p_cls);
    //
    // KEN! may want to do something for array class ?
    //

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetClassMethods(jclass    clazz, 
                           jint      *methodCountPtr, 
                           jmethodID **methodsPtr)
{
    assert(0);

    if (clazz == NULL || methodCountPtr == NULL || methodsPtr == NULL) 
        return JVMDI_ERROR_NULL_POINTER;
    //
    // get method count
    //
    Class *c = (Class *)((Object_Handle)clazz)->java_reference;
    unsigned n_methods = c->n_methods;
    *methodCountPtr = n_methods;
    if (n_methods == 0) 
    {
        *methodsPtr = NULL;
        return JVMDI_ERROR_NONE;
    }
    //
    // create methodID array
    //
    jlong sz = (jlong)sizeof(jmethodID)*n_methods;
    jvmdiError err = Allocate(sz, (jbyte**)methodsPtr);
    if (err != JVMDI_ERROR_NONE) return err;
    //
    // fill in methodID array
    //
    for (unsigned i = 0; i < n_methods; i++)
    {
        (*methodsPtr)[i] = &c->methods[i];
    }

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetClassFields(jclass   clazz, 
                          jint     *fieldCountPtr, 
                          jfieldID **fieldsPtr)
{
    assert(0);

    if (clazz == NULL || fieldCountPtr == NULL || fieldsPtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    Class *c = (Class *)((Object_Handle)clazz)->java_reference;
    //
    // get field count
    //
    unsigned n_fields = c->n_fields;
    *fieldCountPtr = n_fields;
    if (n_fields == 0)
    {
        *fieldsPtr = NULL;
        return JVMDI_ERROR_NONE;
    }
    //
    // create field array
    //
    jlong sz = (jlong)sizeof(jmethodID)*n_fields;
    jvmdiError err = Allocate(sz, (jbyte**)fieldsPtr);
    if (err != JVMDI_ERROR_NONE) return err;
    //
    // fill in field array
    //
    for (unsigned i = 0; i < n_fields; i++)
    {
        (*fieldsPtr)[i] = &c->fields[i];
    }
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetImplementedInterfaces(jclass clazz,
                                    jint   *interfaceCountPtr, 
                                    jclass **interfacesPtr)
{
    assert(0);

    if (clazz == NULL || interfaceCountPtr == NULL || interfacesPtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    Class *c = (Class *)((Object_Handle)clazz)->java_reference;
    //
    // get interface count
    //
    unsigned n_intfs = c->n_superinterfaces;
    *interfaceCountPtr = n_intfs;
    if (n_intfs == 0)
    {
        *interfacesPtr = NULL;
        return JVMDI_ERROR_NONE;
    }
    //
    // create interface array
    //
    jlong sz = (jlong)sizeof(jclass)*n_intfs;
    jvmdiError err = Allocate(sz, (jbyte**)interfacesPtr);
    if (err != JVMDI_ERROR_NONE) return err;
    //
    // fill in interface array
    //
    for (unsigned i = 0; i < n_intfs; i++)
    {
        Object_Handle oh = orp_create_global_object_handle();
        oh->java_reference = (Java_java_lang_Object *)c->superinterfaces[i].clss;
        (*interfacesPtr)[i] = oh;
    }
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
IsInterface(jclass clazz, jboolean *isInterfacePtr)
{
    assert(0);

    if (clazz == NULL || isInterfacePtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    Class *c = (Class *)((Object_Handle)clazz)->java_reference;
    *isInterfacePtr = class_is_interface(c);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
IsArrayClass(jclass clazz, jboolean *isArrayClassPtr)
{
    assert(0);
    return (jvmdiError) 0;
}


jvmdiError JNICALL
GetClassLoader(jclass clazz, jobject *classloaderPtr)
{
    assert(0);

    if (clazz == NULL || classloaderPtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    Class *c = (Class *)((Object_Handle)clazz)->java_reference;
    *classloaderPtr = class_get_class_loader(c);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetObjectHashCode(jobject object, jint *hashCodePtr)
{
    assert(0);
    return (jvmdiError) 0;
}


jvmdiError JNICALL
GetMonitorInfo(jobject object, JVMDI_monitor_info *infoPtr)
{
    assert(0);
    return (jvmdiError) 0;
}


///////////////////////////////////////////////////////////////////
// 
// KEN! Field information
//
///////////////////////////////////////////////////////////////////
jvmdiError JNICALL
GetFieldName(jclass   clazz, 
                        jfieldID field, 
                        char     **namePtr, 
                        char     **signaturePtr)
{
    assert(0);

    if (namePtr == NULL || signaturePtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;

    const char *field_name = field_get_name(field);
    size_t len = strlen(field_name);
    //
    // allocate space and copy field name
    //
    jvmdiError err = Allocate((jlong)(len + 1), (jbyte**)namePtr);
    if (err != JVMDI_ERROR_NONE) return err;
    strcpy(*namePtr, field_name);
    //
    // allocate space and copy field signature
    //
    const char *field_descriptor = field_get_descriptor(field);
    len = strlen(field_descriptor);
    err = Allocate((jlong)len, (jbyte**)signaturePtr);
    if (err != JVMDI_ERROR_NONE) return err;
    strncpy(*signaturePtr, field_descriptor, len);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetFieldDeclaringClass(jclass   clazz, 
                                  jfieldID field, 
                                  jclass   *declaringClassPtr)
{
    assert(0);

    if (declaringClassPtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;

    Object_Handle new_handle = orp_create_global_object_handle();
    new_handle->java_reference = (Java_java_lang_Object *)field_get_class(field);
    *declaringClassPtr = new_handle;

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetFieldModifiers(jclass   clazz, 
                             jfieldID field, 
                             jint     *modifiersPtr)
{
    assert(0);

    *modifiersPtr = field_get_flags(field);

    return JVMDI_ERROR_NONE;
}

jvmdiError JNICALL
IsFieldSynthetic(jclass clazz, jfieldID field, jboolean *isSyntheticPtr)
{
    assert(0); // optional -- not implemented
    return (jvmdiError) 0;
}


///////////////////////////////////////////////////////////////////
//
// method information
//
///////////////////////////////////////////////////////////////////
jvmdiError JNICALL
GetMethodName(jclass    clazz, 
                         jmethodID method, 
                         char      **namePtr,
                         char      **signaturePtr)
{
    assert(0);

    if (namePtr == NULL || signaturePtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    //
    // we assume that method and clazz both are valid
    // we may need to check if they are valid and return
    // appropriate errors, JVMDI_ERROR_INVALID_METHODID, JVMDI_ERROR_INVALID_CLASS
    //
    const char *method_name = method_get_name(method);
    size_t len = strlen(method_name);
    //
    // allocate space for method name
    //
    jvmdiError err = Allocate((jlong)(len + 1), (jbyte**)namePtr);
    if (err != JVMDI_ERROR_NONE) return err;
    strcpy(*namePtr, method_name);
    //
    // allocate space for method signature
    //
    const char *method_descriptor = method_get_descriptor(method);
    len = strlen(method_descriptor);
    err = Allocate((jlong)(len + 1), (jbyte**)signaturePtr);
    if (err != JVMDI_ERROR_NONE) return err;
    strcpy(*signaturePtr, method_descriptor);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetMethodDeclaringClass(jclass    clazz, 
                                   jmethodID method,
                                   jclass    *declaringClassPtr)
{
    assert(0);

    Object_Handle new_handle = orp_create_global_object_handle();
    new_handle->java_reference = (Java_java_lang_Object *)method_get_class(method);
    *declaringClassPtr = new_handle;

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetMethodModifiers(jclass    clazz, 
                              jmethodID method,    
                              jint      *modifiersPtr)
{
    assert(0);

    *modifiersPtr = method_get_flags(method);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetMaxStack(jclass clazz, jmethodID method, jint *maxPtr)
{
    assert(0);

    *maxPtr = method_get_max_stack(method);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetMaxLocals(jclass clazz, jmethodID method, jint *maxPtr)
{
    assert(0);

    *maxPtr = method_get_max_locals(method);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetLineNumberTable(jclass                  clazz, 
                   jmethodID               method,
                   jint                    *entryCountPtr, 
                   JVMDI_line_number_entry **tablePtr)
{
    if (entryCountPtr == NULL || tablePtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    //
    // get the number of entries
    //
    unsigned n_entry = method_get_line_number_table_size(method);
    *entryCountPtr = n_entry;
    //
    // return if n_entry == 0
    //
    if (n_entry == 0)
    {
        *tablePtr = NULL;
        return JVMDI_ERROR_NONE;
    }
    //
    // create line number table
    //
    jlong sz = (jlong)sizeof(JVMDI_line_number_entry)*n_entry;
    jvmdiError err = Allocate(sz, (jbyte**)tablePtr);
    memset(*tablePtr, 0, (uint32)sz);

    if (err != JVMDI_ERROR_NONE) return err;
    //
    // fill in line number information
    //
    for (unsigned i = 0; i < n_entry; i++) {
        method_get_line_number(method, 
                               i, 
                               (unsigned*)&(*tablePtr)[i].start_location,
                               (unsigned*)&(*tablePtr)[i].line_number);
        assert( ( (*tablePtr)[i].start_location & __INT64_C(0xffFFffFF00000000)) == 0);
    }

    return JVMDI_ERROR_NONE;
}

jvmdiError JNICALL
GetMethodLocation(jclass    clazz, 
                             jmethodID method,
                             jlocation *startLocationPtr, 
                             jlocation *endLocationPtr)
{
    assert(0);

    if (startLocationPtr == NULL || endLocationPtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    Byte *start = method_get_code_block_addr(method);
    *startLocationPtr = (jlocation) start;
    unsigned code_sz = method_get_code_block_size(method);
    *endLocationPtr = (jlocation)(start+code_sz);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetLocalVariableTable(jclass     clazz, 
                      jmethodID  method,
                      jint       *entryCountPtr, 
                      JVMDI_local_variable_entry **tablePtr)
{

    if (entryCountPtr == NULL || tablePtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    //
    // compute entry count
    //
    unsigned n_entry = method_get_local_vars_table_size(method);
    *entryCountPtr = n_entry;
    if (n_entry == 0)
    {
        *tablePtr = NULL;
        return JVMDI_ERROR_NONE;
    }
    //
    // create local vars table
    //
    jlong sz = (jlong)sizeof(JVMDI_local_variable_entry)*n_entry;
    jvmdiError err = Allocate(sz, (jbyte**)tablePtr);
    if (err != JVMDI_ERROR_NONE) return err;
    //
    // fill in local var entries
    //
    unsigned n_locals = method_get_max_locals(method);
    unsigned nxt_entry = 0;

    // zhu feng hua
    unsigned slot = 0;

    for (unsigned i = 0; i < n_locals; i++) 
    {
        Local_Var_Entry *lve = ((Method*)method)->_local_vars_table[i]; 
        bool is_long_or_double = false;
        for (;lve != NULL; lve = lve->next) 
        {
            assert(nxt_entry < n_entry);
            JVMDI_local_variable_entry *entry = &(*tablePtr)[nxt_entry++];
            entry->start_location = lve->start_pc;
            entry->length         = lve->length;
            entry->name           = lve->sig->name->bytes;
            entry->signature      = lve->sig->descriptor->bytes;
            entry->slot           = i;
        }
    }
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetExceptionHandlerTable(jclass     clazz, 
                                    jmethodID  method,
                                    jint       *entryCountPtr, 
                                    JVMDI_exception_handler_entry **tablePtr)
{
    assert(0);

    if (entryCountPtr == NULL || tablePtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;

    unsigned n_entry = method_get_num_handlers(method);
    *entryCountPtr = n_entry;
    //
    // return if n_entry == 0
    //
    if (n_entry == 0)
    {
        *tablePtr = NULL;
        return JVMDI_ERROR_NONE;
    }
    //
    // create handler table
    //
    jlong sz = (jlong)sizeof(JVMDI_exception_handler_entry)*n_entry;
    jvmdiError err = Allocate(sz, (jbyte**)tablePtr);
    if (err != JVMDI_ERROR_NONE) return err;
    //
    // fill in handler entries
    //
    for (unsigned i = 0; i < n_entry; i++)
    {
        unsigned handler_cpidx;
        method_get_handler_info(method,
                                i,
                                (unsigned *)&(*tablePtr)[i].start_location,
                                (unsigned *)&(*tablePtr)[i].end_location,
                                (unsigned *)&(*tablePtr)[i].handler_location,
                                &handler_cpidx);
        //
        // create jclass for table[i].exception
        //
        if (handler_cpidx == 0)  // all exception
            (*tablePtr)[i].exception = NULL;
        else 
        {
            Loader_Exception exc;
            Class_Handle c = resolve_class(jit_get_comp_handle(o1_jit),
                                           method_get_class(method),
                                           handler_cpidx,
                                           &exc);
            Object_Handle new_handle = orp_create_global_object_handle();
            new_handle->java_reference = (Java_java_lang_Object *)c;
            (*tablePtr)[i].exception = new_handle;
        }
    }
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetThrownExceptions(jclass    clazz, 
                               jmethodID method,
                               jint      *exceptionCountPtr, 
                               jclass    **exceptionsPtr)
{
    assert(0);

    if (exceptionCountPtr == NULL || exceptionsPtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;

    Method *m = (Method*)method;
    unsigned n_excp = m->num_exceptions_method_can_throw();
    *exceptionCountPtr = n_excp;
    if (n_excp == 0) 
    {
        *exceptionsPtr = NULL;
        return JVMDI_ERROR_NONE;
    }
    //
    // create exception table
    //
    jlong sz = (jlong)sizeof(jclass)*n_excp;
    jvmdiError err = Allocate(sz, (jbyte**)exceptionsPtr);
    if (err != JVMDI_ERROR_NONE) return err;
    //
    // fill in exceptions
    //
    for (unsigned i = 0; i < n_excp; i++)
    {
        String *excp_str = m->get_exception_name(i);
        Loader_Exception ld_exception;
        Class *c = load_class(ORP_Global_State::loader_env,
                              excp_str,
                              &ld_exception);
        Object_Handle new_handle = orp_create_global_object_handle();
        new_handle->java_reference = (Java_java_lang_Object *)c;
        (*exceptionsPtr)[i] = new_handle;
    }
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetBytecodes(jclass    clazz, 
                        jmethodID method, 
                        jint      *bytecodeCountPtr,
                        jbyte     **bytecodesPtr)
{
    assert(0);

    if (bytecodeCountPtr == NULL || bytecodesPtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    *bytecodeCountPtr = method_get_byte_code_size(method);
    *bytecodesPtr = (jbyte*)method_get_byte_code_addr(method);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
IsMethodNative(jclass clazz, jmethodID method, jboolean *isNativePtr)
{
    assert(0);

    if (isNativePtr == NULL)
        return JVMDI_ERROR_NULL_POINTER;
    *isNativePtr = method_is_native(method);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
IsMethodSynthetic(jclass clazz, jmethodID method, jboolean *isSyntheticPtr)
{
    assert(0);  // optional -- not supported
    return (jvmdiError) 0;
}


jvmdiError JNICALL
CreateRawMonitor(char *name, JVMDI_RawMonitor *monitorPtr)
{
    assert(0);

    assert(p_TLS_orpthread->gc_enabled_status == disabled);

    String *s = ORP_Global_State::loader_env->string_pool.lookup(name);

    Java_java_lang_String *jls = orp_instantiate_cp_string_resolved(s);

    Object_Handle new_handle = orp_create_global_object_handle();

    new_handle->java_reference = (Java_java_lang_Object *)jls;

    assert(new_handle->java_reference->vt->clss->name->bytes);

    * monitorPtr = new_handle;

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
DestroyRawMonitor(JVMDI_RawMonitor monitor)
{
    assert(0);

    orp_delete_global_object_handle(monitor);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
RawMonitorEnter(JVMDI_RawMonitor monitor)
{
    assert(0);

    assert(monitor->java_reference);

    orp_monitor_enter(monitor->java_reference);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
RawMonitorExit(JVMDI_RawMonitor monitor)
{
    assert(0);

    orp_monitor_exit(monitor->java_reference);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
RawMonitorWait(JVMDI_RawMonitor monitor, jlong millis)
{
    assert(0);

#if 0
    // (mjc 20000519)
    // All this stuff is not JNI-compliant.  Should we re-write it?

    java_lang_Object_wait(monitor->java_reference, millis);

#endif

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
RawMonitorNotify(JVMDI_RawMonitor monitor)
{
    assert(0);

#if 0
    // (mjc 20000519)
    // All this stuff is not JNI-compliant.  Should we re-write it?

    java_lang_Object_notify(monitor->java_reference);

#endif

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
RawMonitorNotifyAll(JVMDI_RawMonitor monitor)
{
    assert(0);

#if 0
    // (mjc 20000519)
    // All this stuff is not JNI-compliant.  Should we re-write it?
    java_lang_Object_notifyAll(monitor->java_reference);

#endif

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
SetEventHook(JVMDI_EventHook hook)
{
    assert(0);

    jvmdi_hook_function = hook;

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
SetEventNotificationMode(jint mode, jint eventType, jthread thread, ...)
{ 
    assert(0);

#if 1
    // (mjc 20000519)
    // This tries to access java.lang.Thread's fields directly
    assert(0);
#else
    jvmdiError event = (jvmdiError)eventType;

    Object_Handle oh = (Object_Handle)thread;

    ORP_thread *p_thr = 0;
   
    if (oh) {
        Java_java_lang_Thread *p_jlt = (Java_java_lang_Thread *)oh->java_reference;
        p_thr = (ORP_thread *)p_jlt->PrivateInfo;
    }

    bool on_off = false;
    if (mode) on_off = true;

    switch (event) {

    // the following are global events (not thread events)

    case JVMDI_EVENT_VM_INIT:
        {
            report_jvmdi_event_vm_init = on_off;
            break;
        }

    case JVMDI_EVENT_VM_DEATH:
        {
            report_jvmdi_event_vm_death = on_off;
            break;
        }

    case JVMDI_EVENT_THREAD_START:
        {
            report_jvmdi_event_thread_start = on_off;
            break;
        }

    case JVMDI_EVENT_CLASS_UNLOAD:
        {
            report_jvmdi_event_class_unload = on_off;
            break;
        }
    
    // the following are thread specific events (not global events)

    case JVMDI_EVENT_SINGLE_STEP:
        {
            if (!p_thr)
                return JVMDI_ERROR_INVALID_THREAD;
#ifdef _DEBUG
            p_thr->report_jvmdi_event_single_step = on_off;
#else
            assert(0);
#endif
            break;
        }

    case JVMDI_EVENT_EXCEPTION_CATCH:
        {
            if (!p_thr)
                return JVMDI_ERROR_INVALID_THREAD;
#ifdef _DEBUG
            p_thr->report_jvmdi_event_exception_catch = on_off;
#else
            assert(0);
#endif
            break;
        }

    case JVMDI_EVENT_METHOD_ENTRY:
        {
            if (!p_thr)
                return JVMDI_ERROR_INVALID_THREAD;
#ifdef _DEBUG
            p_thr->report_jvmdi_event_method_entry = on_off;
#else
            assert(0);
#endif
           break;
        }

    case JVMDI_EVENT_METHOD_EXIT:
        {
            if (!p_thr)
                return JVMDI_ERROR_INVALID_THREAD;
#ifdef _DEBUG
            p_thr->report_jvmdi_event_method_exit = on_off;
#else
            assert(0);
#endif
            break;
        }

    default:
        assert(0);
    }
#endif
    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetLoadedClasses(jint *classCountPtr, jclass **classesPtr)
{
    assert(0);

    jint size_needed = CLASS_TABLE_SIZE * sizeof(Object_Handle);

    Object_Handle *handle_array = 0;

    jvmdiError stat = Allocate(size_needed, (jbyte **)&handle_array);
    if (stat != JVMDI_ERROR_NONE)
        return JVMDI_ERROR_OUT_OF_MEMORY;
    memset( (void *)handle_array, 0, size_needed );

    *classesPtr = (jclass *)handle_array;

    Class **ptrC = ORP_Global_State::loader_env->class_table.get_first_class();

    int xx = 0;
    for (; xx < CLASS_TABLE_SIZE; xx++) {

        Object_Handle oh = orp_create_global_object_handle();
        Class *c = *ptrC;
        oh->java_reference = (struct Java_java_lang_Object *)c;
                            
        *handle_array = oh;
        handle_array++;

        ptrC = ORP_Global_State::loader_env->class_table.get_next_class(c);
        if (ptrC == NULL)
            break;
    }
    *classCountPtr = xx;

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetClassLoaderClasses(jobject initiatingLoader, jint *classCountPtr, jclass **classesPtr)
{
    assert(0);

    // BUGBUG -- the following will not work on Java2.  See JVMDI ref doc for details
    jvmdiError je = GetLoadedClasses(classCountPtr, classesPtr);

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetVersionNumber(jint *versionPtr)
{
    assert(0);

    *versionPtr = 0;  // BUGBUG what should the version be??

    return JVMDI_ERROR_NONE;
}


jvmdiError JNICALL
GetCapabilities(JVMDI_capabilities *capabilitiesPtr)
{
    assert(0);

    capabilitiesPtr->can_watch_field_modification = 0;
    capabilitiesPtr->can_watch_field_access = 0;
    capabilitiesPtr->can_get_bytecodes = 0;
    capabilitiesPtr->can_get_synthetic_attribute = 0;
    capabilitiesPtr->can_get_owned_monitor_info = 0;
    capabilitiesPtr->can_get_current_contended_monitor = 0;
    capabilitiesPtr->can_get_monitor_info = 0;

    return JVMDI_ERROR_NONE;
}


//
//
////////////////// SUPPORT ROUTINES
//
//

#if 0 /////%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%TOSS THISSSSSS NOW
void jvmdi_method_exit()
{
    assert(0);

    void method_exit(jint kind);

    if (global_frame) {

        Frame_Context frcn;
        ro_unwind_native_stack_frame(&frcn);

        if ( *(frcn.p_eip) == *(global_frame->p_eip) ) {

            memset(global_frame, 0, sizeof(Frame_Context) );

            method_exit(JVMDI_EVENT_FRAME_POP);
        }
    }
#ifdef _DEBUG
    if (p_TLS_orpthread->report_jvmdi_event_method_exit)
        method_exit(JVMDI_EVENT_METHOD_EXIT);
#else
    assert(0);
#endif
}


void  jvmdi_exception(Frame_Context *original_context,
                   volatile Java_java_lang_Object **p_obj,
                   Frame_Context *new_context)                            
{
#ifdef _DEBUG
        if (p_TLS_orpthread->report_jvmdi_event_exception_catch == false)
            return;
#else
        assert(0);
#endif
        if (jvmdi_hook_function == 0) {  // temporary support

            Global_Env *env = ORP_Global_State::loader_env;

            if ( (*p_obj)->vt->clss == env->JavaLangClass_Class) {

                Class *p_c = (Class *)(*p_obj);
                orp_cout << "exception: " << p_c->name->bytes << endl;
            }
                else
                orp_cout << "exception: " << (*p_obj)->vt->clss->name->bytes <<endl;
            return;
        }

        Object_Handle jt = orp_create_global_object_handle();
        jt->java_reference = (struct Java_java_lang_Object *)
                                p_TLS_orpthread->p_java_lang_thread;
        
        JIT_Specific_Info *jit_info = methods.find((void *)*(original_context->p_eip));
        Method  *p_meth = jit_info->get_method();
        Class *p_clss = p_meth->get_class();

        Object_Handle jc = orp_create_global_object_handle();
        jc->java_reference = (struct Java_java_lang_Object *)p_clss;

        jmethodID jmid = (jmethodID)p_meth;

        jlocation jloc = (jlocation)*(original_context->p_eip);

        Object_Handle je = orp_create_global_object_handle();
        je->java_reference = (struct Java_java_lang_Object *)(*p_obj);

        jit_info = methods.find((void *)*(new_context->p_eip));
        Method *p_catch_meth = jit_info->get_method();
        Class *p_catch_clss = p_catch_meth->get_class();

        Object_Handle jc_catch = orp_create_global_object_handle();
        jc_catch->java_reference = (struct Java_java_lang_Object *)p_catch_clss;

        jmethodID j_catch_mid = (jmethodID)p_catch_meth;

        jlocation j_catch_loc = (jlocation)*(new_context->p_eip);

        JVMDI_Event *p_event_record = 0;
        Allocate(sizeof(JVMDI_Event), (jbyte**)&p_event_record );
        memset(p_event_record, 0, sizeof(JVMDI_Event) );

        p_event_record->kind = JVMDI_EVENT_EXCEPTION_CATCH;

        p_event_record->u.exception.thread = jt;
        p_event_record->u.exception.clazz = jc;
        p_event_record->u.exception.method = jmid;
        p_event_record->u.exception.location = jloc;
        p_event_record->u.exception.exception = je;
        p_event_record->u.exception.catch_clazz = jc_catch;
        p_event_record->u.exception.catch_method = j_catch_mid;
        p_event_record->u.exception.catch_location = j_catch_loc;

        jvmdi_hook_function( (struct JNIEnv_ *)jni_native_intf, p_event_record);
}

#endif
