/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


#ifndef PLATFORM_UNIX
#include <windows.h>
#include <gl\glu.h>
#pragma hdrstop
#include <condefs.h>
#else
#include <GL/glu.h>
#include <math.h>
#include "SDL.h"
#include "linux_inc.h"
#endif

#include "OpenGLRender.h"
#include "console.h"

static GLfloat projMatrix[]={1, 0, 0, 0,
            			     0, 1, 0, 0,
                             0, 0, 1, 1,
                             0, 0,-1, 0};

extern char errorStr[256];
extern int error;

static Poly p;

#define SPRITE 0
#define WINDOW 1

extern int software;

#ifndef PLATFORM_UNIX
extern HMODULE ddrawLib;
extern HINSTANCE hInst;
#endif

extern "C" void DisableDiv0Exception();
extern "C" void EnableDiv0Exception();

void OpenGLRender::EnumModes(ModeList *modes16,ModeList *modes32)
{
    modes16->count=modes32->count=5;
    modes16->mode[0].w=modes32->mode[0].w=640;
    modes16->mode[0].h=modes32->mode[0].h=480;
    modes16->mode[1].w=modes32->mode[1].w=800;
    modes16->mode[1].h=modes32->mode[1].h=600;
    modes16->mode[2].w=modes32->mode[2].w=1024;
    modes16->mode[2].h=modes32->mode[2].h=768;
    modes16->mode[3].w=modes32->mode[3].w=1280;
    modes16->mode[3].h=modes32->mode[3].h=1024;
    modes16->mode[4].w=modes32->mode[4].w=1600;
    modes16->mode[4].h=modes32->mode[4].h=1200;
}

#ifdef PLATFORM_UNIX
OpenGLRender::OpenGLRender(int _scrnX,int _scrnY, int depth)
#else
OpenGLRender::OpenGLRender(HINSTANCE _hInst,HWND _hWnd,int _scrnX,int _scrnY,int depth)
#endif
{
    init=0;
#ifdef PLATFORM_UNIX
    stencilValue=1; dc=0;
    Uint8 flags = SDL_OPENGL;
#else
    stencilValue=1; dc=NULL;
#endif
    extraPolyList=NULL;
    firstFont=lastFont=curFont=NULL;
    fontDisplayList=1000;
    translateMode=TM_VIEW;
    renderEnabled=sceneEnabled=false;
    scrnX=_scrnX; scrnY=_scrnY;
#ifndef PLATFORM_UNIX
    hInst=_hInst; hWnd=_hWnd;
    dc=GetDC(hWnd);
    PIXELFORMATDESCRIPTOR pfd={sizeof(PIXELFORMATDESCRIPTOR),
		1,PFD_DRAW_TO_WINDOW|PFD_SUPPORT_OPENGL|PFD_DOUBLEBUFFER,
		PFD_TYPE_RGBA,16,0,0,0,0,0,0,0,0,0,0,0,0,0,24,8,0,
        PFD_MAIN_PLANE,0,0,0,0};
    int pf=ChoosePixelFormat(dc,&pfd);
    SetPixelFormat(dc,pf,&pfd);
    cprintf("Attempting to initialize OpenGL...\n");
    hContext=wglCreateContext(dc);
    if (!hContext)
    {
        cprintf("OpenGL initialization failed\n");
        return;
    }
    cprintf("Initialization OK, using OpenGL\n");
    wglMakeCurrent(dc,hContext);
#else
    if (depth)
	    flags |= SDL_FULLSCREEN;
    if (SDL_SetVideoMode (scrnX, scrnY, 16, flags) == NULL)
    {
	    cprintf("OpenGL initialization failed\n");
	    return;
    }
#endif
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
    glDisable(GL_LIGHTING);
    glPixelStorei(GL_UNPACK_ALIGNMENT,1);
    glPolygonMode(GL_FRONT,GL_FILL);
    glClearColor(0,0,0,0);
    glMatrixMode(GL_PROJECTION);
#ifndef PLATFORM_UNIX
    RECT cr;
    GetClientRect(hWnd,&cr);
    projMatrix[5]=1*(float)cr.right/(float)cr.bottom;
#else
    projMatrix[5]=1*(float)(scrnX-1)/(float)(scrnY-1);
#endif
    glLoadMatrixf(projMatrix);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    x=0; y=0; z=0;
    xa=0; ya=0; za=0;
    cprintf("OpenGL setup complete\n");
    init=1;
}

OpenGLRender::~OpenGLRender()
{
    cprintf("Deinitializing OpenGL...\n");
#ifndef PLATFORM_UNIX
	wglMakeCurrent(NULL,NULL);
	wglDeleteContext(hContext);
    ReleaseDC(hWnd,dc);
#endif
    cprintf("OpenGL deinitialized successfully\n");
    init=0;
}

int OpenGLRender::Initialized()
{
    return init;
}

void OpenGLRender::ChangeResolution(int width,int height,int depth)
{
    scrnX=width; scrnY=height;
    cprintf("Attempting OpenGL resolution change...\n");
    Vector p=Vector(GetXPos(),GetYPos(),GetZPos());
    Vector r=Vector(GetXRotate(),GetYRotate(),GetZRotate());
    glViewport(0,0,width,height);
    glMatrixMode(GL_PROJECTION);
#ifndef PLATFORM_UNIX
    RECT cr;
    GetClientRect(hWnd,&cr);
    projMatrix[5]=1*(float)cr.right/(float)cr.bottom;
#else
    projMatrix[5]=1*(float)(scrnX-1)/(float)(scrnY-1);
#endif
    glLoadMatrixf(projMatrix);
    glMatrixMode(GL_MODELVIEW);
    SetXPos(p.x); SetYPos(p.y); SetZPos(p.z);
    SetXRotate(r.x); SetYRotate(r.y); SetZRotate(r.z);
    cprintf("Resolution change successful\n");
}

void OpenGLRender::SetRenderWnd(HWND _hWnd)
{
    hWnd=_hWnd;
}

void OpenGLRender::BeginRender()
{
#ifndef PLATFORM_UNIX
    DisableDiv0Exception();
#endif
    glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT);
    polyDrawn=polyTotal=polyReflect=0;
    renderEnabled=true;
}

void OpenGLRender::BeginScene()
{
    sceneEnabled=true;
}

void OpenGLRender::EndRender()
{
    glFlush();
#ifndef PLATFORM_UNIX
    SwapBuffers(dc);
    EnableDiv0Exception();
#else
    SDL_GL_SwapBuffers();
#endif
    renderEnabled=false;
}

void OpenGLRender::EndScene()
{
    sceneEnabled=false;
}

/*void _export RenderReflectPoly(Poly &poly,Plane &p,float n,float r,int noCheck=0)
{
    if (poly.nVert<3) return;
    if (poly.flags&PF_INVISIBLE)
        return;
    if (!noCheck)
    {
        Poly posPoly,negPoly;
        switch(ClassifyPoly(poly,p))
        {
            case CT_COINCIDENT:
                return;
            case CT_NEGATIVE:
                if (r>=0) return;
                break;
            case CT_POSITIVE:
                if (r<=0) return;
                break;
            case CT_SPLIT:
                posPoly.Clear(); negPoly.Clear();
                SplitPoly(poly,p,posPoly,negPoly);
//                if (r>0) RenderReflectPoly(posPoly,p,n,r);
//                else RenderReflectPoly(negPoly,p,n,r);
                return;
        }
    }
    Poly np=poly;
    for (int i=0;i<np.nVert;i++)
    {
        float nr=p.Evaluate(np.v[i].x,np.v[i].y,np.v[i].z);
        np.v[i].x+=p.A*nr*n;
        np.v[i].y+=p.B*nr*n;
        np.v[i].z+=p.C*nr*n;
    }
    if (!IsBackFace(np,viewVect))
        return;
    if ((poly.texture!=NULL)&&(!(poly.flags&PF_NONTEXTURED)))
    {
        d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_MODULATE2X);
        d3d->SetTextureStageState(0,D3DTSS_ALPHAOP,D3DTOP_MODULATE2X);
        d3d->SetTexture(0,(CTexture*)(poly.texture->data));
    }
    else
        d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_DISABLE);
    if (poly.flags&PF_TWOSIDED)
    	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_NONE);
    else if (poly.flags&PF_REVERSE)
    	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CCW);
    else
    	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CW);
    polyReflect++;
    d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,np.v,np.nVert,D3DDP_WAIT);
}

void RenderReflectPolyList(PolyList &list,Plane &p,float n,float r)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        RenderReflectPoly(ptr->poly,p,n,r);
        ptr=ptr->next;
    }
}

void RenderReflectPolyListNonBSP(PolyList &list,Plane &p,float n,float r)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        if (!(ptr->poly.flags&PF_INTREE))
            RenderReflectPoly(ptr->poly,p,n,r);
        ptr=ptr->next;
    }
}

void RenderReflectBSPNode(BSPNode *node,Plane &p,float n,float r,Vector eye)
{
    if (node==NULL) return;
    float result=node->part.Evaluate(eye.x,eye.y,eye.z);
    if (result>0)
    {
        RenderReflectBSPNode(node->neg,p,n,r,eye);
        RenderReflectPolyList(node->list,p,n,r);
        RenderReflectBSPNode(node->pos,p,n,r,eye);
    }
    else if (result<0)
    {
        RenderReflectBSPNode(node->pos,p,n,r,eye);
        RenderReflectPolyList(node->list,p,n,r);
        RenderReflectBSPNode(node->neg,p,n,r,eye);
    }
    else
    {
        RenderReflectBSPNode(node->pos,p,n,r,eye);
        RenderReflectBSPNode(node->neg,p,n,r,eye);
    }
}*/

void OpenGLRender::RenderPoly(Poly &poly)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    if (poly.flags&PF_INVISIBLE)
        return;
/*    if (poly.flags&PF_REFLECT)
    {
        Plane p=PlaneFromPoly(poly);
        float r=p.Evaluate(GetXPos(),GetYPos(),GetZPos());
        float n=-2/(p.A*p.A+p.B*p.B+p.C*p.C);
        Vector eye=Vector(GetXPos()+p.A*r*n,GetYPos()+
            p.B*r*n,GetZPos()+p.C*r*n);
        r=-p.Evaluate(eye.x,eye.y,eye.z);
        stencilValue++;
        if (stencilValue>255)
        {
            d3d->ClearStencilBuffer();
            stencilValue=1;
        }
        d3d->SetRenderState(D3DRENDERSTATE_STENCILENABLE,TRUE);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILPASS,D3DSTENCILOP_REPLACE);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILZFAIL,D3DSTENCILOP_KEEP);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILREF,stencilValue);
        d3d->SetRenderState(D3DRENDERSTATE_ZWRITEENABLE,FALSE);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILFUNC,D3DCMP_ALWAYS);
        d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_DISABLE);
        if (poly.flags&PF_TWOSIDED)
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_NONE);
        else if (poly.flags&PF_REVERSE)
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CW);
        else
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CCW);
        Poly np=poly;
        for (int i=0;i<np.nVert;i++)
            np.v[i].color=RGBA_MAKE(128,128,128,255);
        d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,np.v,np.nVert,D3DDP_WAIT);
        d3d->SetRenderState(D3DRENDERSTATE_ZWRITEENABLE,TRUE);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILFAIL,D3DSTENCILOP_KEEP);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILZFAIL,D3DSTENCILOP_KEEP);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILPASS,D3DSTENCILOP_KEEP);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILFUNC,D3DCMP_EQUAL);
        d3d->SetRenderState(D3DRENDERSTATE_ZFUNC,D3DCMP_ALWAYS);
      	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_NONE);
        d3d->SetView(lpZeroView);
        d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,farPoly.v,farPoly.nVert,D3DDP_WAIT);
        d3d->SetView(lpView);
        d3d->SetRenderState(D3DRENDERSTATE_ZFUNC,D3DCMP_LESSEQUAL);
        if (extraPolyList) RenderReflectPolyListNonBSP(*extraPolyList,p,n,r);
        RenderReflectBSPNode(rootNode,p,n,r,eye);
        if (np.texture!=NULL)
        {
            d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_MODULATE2X);
            d3d->SetTextureStageState(0,D3DTSS_ALPHAOP,D3DTOP_MODULATE2X);
            d3d->SetTexture(0,(CTexture*)(poly.texture->data));
        }
        else
            d3d->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_DISABLE);
        if (poly.flags&PF_TWOSIDED)
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_NONE);
        else if (poly.flags&PF_REVERSE)
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CW);
        else
        	d3d->SetRenderState(D3DRENDERSTATE_CULLMODE,D3DCULL_CCW);
        d3d->SetRenderState(D3DRENDERSTATE_ZFUNC,D3DCMP_ALWAYS);
        d3d->DrawPrimitive(D3DPT_TRIANGLEFAN,poly.v,poly.nVert,D3DDP_WAIT);
        d3d->SetRenderState(D3DRENDERSTATE_ZFUNC,D3DCMP_LESSEQUAL);
        d3d->SetRenderState(D3DRENDERSTATE_STENCILENABLE,FALSE);
        return;
    }*/
    if (poly.flags&PF_PRIORITY)
        glDisable(GL_DEPTH_TEST);
    else
        glEnable(GL_DEPTH_TEST);
    int normalTex=1;
    if ((poly.texture!=NULL)&&(!(poly.flags&PF_NONTEXTURED)))
    {
        glEnable(GL_TEXTURE_2D);
        for (int i=0;i<poly.nVert;i++)
        {
            if (((poly.v[i].color&0xff0000)>0x800000)||
                ((poly.v[i].color&0xff00)>0x8000)||
                ((poly.v[i].color&0xff)>0x80))
            {
                normalTex=0;
                break;
            }
        }
        if (normalTex)
            glBindTexture(GL_TEXTURE_2D,(int)(poly.texture->data));
        else
            glBindTexture(GL_TEXTURE_2D,(int)(poly.texture->data)+1);
    }
    else
        glDisable(GL_TEXTURE_2D);
    if (poly.flags&PF_TWOSIDED)
        glDisable(GL_CULL_FACE);
    else if (poly.flags&PF_REVERSE)
    {
        glEnable(GL_CULL_FACE);
        glCullFace(GL_BACK);
    }
    else
    {
        glEnable(GL_CULL_FACE);
        glCullFace(GL_FRONT);
    }
    polyDrawn++;
    if (poly.flags&PF_ZBIAS)
    {
        glBegin(GL_TRIANGLE_FAN);
        for (int i=0;i<poly.nVert;i++)
        {
            int r,g,b;
            if (normalTex)
            {
                r=((poly.v[i].color>>16)&0xff)<<1; if (r>255) r=255;
                g=((poly.v[i].color>>8)&0xff)<<1; if (g>255) g=255;
                b=(poly.v[i].color&0xff)<<1; if (b>255) b=255;
            }
            else
            {
                r=(poly.v[i].color>>16)&0xff;
                g=(poly.v[i].color>>8)&0xff;
                b=poly.v[i].color&0xff;
            }
            glColor4ub(r,g,b,(poly.v[i].color>>24)&0xff);
            glTexCoord2f(poly.v[i].u,poly.v[i].v);
            glVertex3f(GetXPos()+(poly.v[i].x-GetXPos())*0.99,
                GetYPos()+(poly.v[i].y-GetYPos())*0.99,
                GetZPos()+(poly.v[i].z-GetZPos())*0.99);
        }
        glEnd();
    }
    else
    {
        glBegin(GL_TRIANGLE_FAN);
        for (int i=0;i<poly.nVert;i++)
        {
            int r,g,b;
            if (normalTex)
            {
                r=((poly.v[i].color>>16)&0xff)<<1; if (r>255) r=255;
                g=((poly.v[i].color>>8)&0xff)<<1; if (g>255) g=255;
                b=(poly.v[i].color&0xff)<<1; if (b>255) b=255;
            }
            else
            {
                r=(poly.v[i].color>>16)&0xff;
                g=(poly.v[i].color>>8)&0xff;
                b=poly.v[i].color&0xff;
            }
            glColor4ub(r,g,b,(poly.v[i].color>>24)&0xff);
            glTexCoord2f(poly.v[i].u,poly.v[i].v);
            glVertex3f(poly.v[i].x,poly.v[i].y,poly.v[i].z);
        }
        glEnd();
    }
    glEnable(GL_DEPTH_TEST);
}

void OpenGLRender::RenderPolyList(PolyList &list)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        RenderPoly(ptr->poly);
        ptr=ptr->next;
        polyTotal++;
    }
}

void OpenGLRender::RenderPolyListNonBSP(PolyList &list)
{
    PolyListElem *ptr=list.first;
    while (ptr!=NULL)
    {
        if (!(ptr->poly.flags&PF_INTREE))
        {
            RenderPoly(ptr->poly);
            polyTotal++;
        }
        ptr=ptr->next;
    }
}

void OpenGLRender::UpdateViewMatrix()
{
	glLoadIdentity();
    if (translateMode==TM_VIEW)
    {
        glRotatef(xa,1,0,0);
        glRotatef(ya,0,1,0);
        glRotatef(za,0,0,1);
        glTranslatef(-x,-y,-z);
    }
    else
    {
        glTranslatef(-x,-y,-z);
        glRotatef(xa,1,0,0);
        glRotatef(ya,0,1,0);
        glRotatef(za,0,0,1);
    }
}

void OpenGLRender::SetXPos(float _x)
{
    x=_x;
    UpdateViewMatrix();
}

void OpenGLRender::SetYPos(float _y)
{
    y=_y;
    UpdateViewMatrix();
}

void OpenGLRender::SetZPos(float _z)
{
    z=_z;
    UpdateViewMatrix();
}

void OpenGLRender::AddXPos(float _x)
{
    x+=_x;
    UpdateViewMatrix();
}

void OpenGLRender::AddYPos(float _y)
{
    y+=_y;
    UpdateViewMatrix();
}

void OpenGLRender::AddZPos(float _z)
{
    z+=_z;
    UpdateViewMatrix();
}

void OpenGLRender::SetXRotate(float x)
{
    xa=x*(180/M_PI);
    UpdateViewMatrix();
}

void OpenGLRender::SetYRotate(float y)
{
    ya=y*(180/M_PI);
    UpdateViewMatrix();
}

void OpenGLRender::SetZRotate(float z)
{
    za=z*(180/M_PI);
    UpdateViewMatrix();
}

void OpenGLRender::AddXRotate(float x)
{
    xa+=x*(180/M_PI);
    UpdateViewMatrix();
}

void OpenGLRender::AddYRotate(float y)
{
    ya+=y*(180/M_PI);
    UpdateViewMatrix();
}

void OpenGLRender::AddZRotate(float z)
{
    za+=z*(180/M_PI);
    UpdateViewMatrix();
}

static MATRIX initViewMatrix={1, 0, 0, 0,
                              0, 1, 0, 0,
			        		  0, 0, 1, 0,
					          0, 0, 0, 1};

static void MatrixRotateX(MATRIX& mat,float a)
{
	MATRIX temp;
	float s=sin(a);
	float c=cos(a);

	temp._12=mat._12*c+mat._13*s;
	temp._22=mat._22*c+mat._23*s;
	temp._32=mat._32*c+mat._33*s;
	temp._42=mat._42*c+mat._43*s;
	temp._13=mat._12*-s+mat._13*c;
	temp._23=mat._22*-s+mat._23*c;
	temp._33=mat._32*-s+mat._33*c;
	temp._43=mat._42*-s+mat._43*c;

	mat._12=temp._12; mat._22=temp._22; mat._32=temp._32;
	mat._42=temp._42; mat._13=temp._13; mat._23=temp._23;
	mat._33=temp._33; mat._43=temp._43;
}

static void MatrixRotateY(MATRIX& mat,float a)
{
	MATRIX temp;
	float s=sin(a);
	float c=cos(a);
	
	temp._11=mat._11*c+mat._13*-s;
	temp._21=mat._21*c+mat._23*-s;
	temp._31=mat._31*c+mat._33*-s;
	temp._41=mat._41*c+mat._43*-s;
	temp._13=mat._11*s+mat._13*c;
	temp._23=mat._21*s+mat._23*c;
	temp._33=mat._31*s+mat._33*c;
	temp._43=mat._41*s+mat._43*c;
	
	mat._11=temp._11; mat._21=temp._21; mat._31=temp._31;
	mat._41=temp._41; mat._13=temp._13; mat._23=temp._23;
	mat._33=temp._33; mat._43=temp._43;
}

static void MatrixRotateZ(MATRIX& mat,float a)
{
	MATRIX temp;
	float s=sin(a);
	float c=cos(a);

	temp._11=mat._11*c+mat._12*s;
	temp._21=mat._21*c+mat._22*s;
	temp._31=mat._31*c+mat._32*s;
	temp._41=mat._41*c+mat._42*s;
	temp._12=mat._11*-s+mat._12*c;
	temp._22=mat._21*-s+mat._22*c;
	temp._32=mat._31*-s+mat._32*c;
	temp._42=mat._41*-s+mat._42*c;
	
	mat._11=temp._11; mat._21=temp._21; mat._31=temp._31;
	mat._41=temp._41; mat._12=temp._12; mat._22=temp._22;
	mat._32=temp._32; mat._42=temp._42;
}

static void TransformVector(Vector& v,MATRIX& mat)
{
	v=Vector(v.x*mat._11+v.y*mat._12+v.z*mat._13+mat._41,
			 v.x*mat._21+v.y*mat._22+v.z*mat._23+mat._42,
			 v.x*mat._31+v.y*mat._32+v.z*mat._33+mat._43);
}

void OpenGLRender::GoForward(float d)
{
	Vector forward(0,0,d);
	MATRIX transform;

	memcpy(&transform,&initViewMatrix,sizeof(D3DMATRIX));
	MatrixRotateZ(transform,-za*(M_PI/180));
	MatrixRotateY(transform,-ya*(M_PI/180));
//	MatrixRotateX(transform,-xa*(M_PI/180));
	TransformVector(forward,transform);
	x+=forward.x;
	y+=forward.y;
	z+=forward.z;
    UpdateViewMatrix();
}

void OpenGLRender::GoSide(float d)
{
	Vector forward(d,0,0);
	D3DMATRIX transform;

	memcpy(&transform,&initViewMatrix,sizeof(D3DMATRIX));
	MatrixRotateZ(transform,-za*(M_PI/180));
	MatrixRotateY(transform,-ya*(M_PI/180));
//	MatrixRotateX(transform,-xa*(M_PI/180));
	TransformVector(forward,transform);
	x+=forward.x;
	y+=forward.y;
	z+=forward.z;
    UpdateViewMatrix();
}

float OpenGLRender::GetXPos()
{
    return x;
}

float OpenGLRender::GetYPos()
{
    return y;
}

float OpenGLRender::GetZPos()
{
    return z;
}

float OpenGLRender::GetXRotate()
{
    return xa*(M_PI/180);
}

float OpenGLRender::GetYRotate()
{
    return ya*(M_PI/180);
}

float OpenGLRender::GetZRotate()
{
    return za*(M_PI/180);
}

Texture* OpenGLRender::LoadTextureBMP(char *filename)
{
    Texture *tex=new Texture;
    tex->location=TLOC_BMP;
    tex->file=new char[strlen(filename)+1]; strcpy(tex->file,filename);
    tex->alphaFile=NULL;
    FILE *fp=fopen(filename,"rb");
    fseek(fp,sizeof(BITMAPFILEHEADER),SEEK_SET);
    BITMAPINFOHEADER info;
    fread(&info,sizeof(BITMAPINFOHEADER),1,fp);
    GLuint nRet[1];
    glGenTextures(1,nRet);
    tex->data=(void*)nRet[0];
    glBindTexture(GL_TEXTURE_2D,(int)(tex->data));
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    unsigned char *texBits=new unsigned char[info.biHeight*info.biWidth*4];
    unsigned char *fileBits=new unsigned char[info.biHeight*
        info.biWidth*3];
    unsigned char *texPtr=texBits;
    fread(fileBits,info.biHeight*info.biWidth*3,1,fp);
    int i=0;
    for (int y=0;y<info.biHeight;y++)
    {
        for (int x=0;x<info.biWidth;x++)
        {
            *(texPtr++)=fileBits[i+2];
            *(texPtr++)=fileBits[i+1];
            *(texPtr++)=fileBits[i];
            i+=3;
            *(texPtr++)=255;
        }
    }
    delete[] fileBits;
    gluBuild2DMipmaps(GL_TEXTURE_2D,4,info.biWidth,info.biHeight,GL_RGBA,
        GL_UNSIGNED_BYTE,texBits);
    delete[] texBits;
    fclose(fp);
    return tex;
}

Texture* OpenGLRender::LoadAlphaTextureBMP(char *filename,char *alphaname)
{
    Texture *tex=new Texture;
    tex->location=TLOC_BMP;
    tex->file=new char[strlen(filename)+1]; strcpy(tex->file,filename);
    tex->alphaFile=new char[strlen(alphaname)+1]; strcpy(tex->alphaFile,alphaname);
    FILE *fp=fopen(filename,"rb");
    FILE *alpha=fopen(alphaname,"rb");
    fseek(fp,sizeof(BITMAPFILEHEADER),SEEK_SET);
    BITMAPINFOHEADER info;
    fread(&info,sizeof(BITMAPINFOHEADER),1,fp);
    fseek(alpha,10,SEEK_SET);
    DWORD ofs;
    fread(&ofs,4,1,alpha);
    fseek(alpha,ofs,SEEK_SET);
    GLuint nRet[1];
    glGenTextures(1,nRet);
    tex->data=(void*)nRet[0];
    glBindTexture(GL_TEXTURE_2D,(int)(tex->data));
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    unsigned char *texBits=new unsigned char[info.biHeight*info.biWidth*4];
    unsigned char *fileBits=new unsigned char[info.biHeight*
        info.biWidth*3];
    unsigned char *alphaBits=new unsigned char[info.biHeight*info.biWidth];
    unsigned char *texPtr=texBits;
    fread(fileBits,info.biHeight*info.biWidth*3,1,fp);
    fread(alphaBits,info.biHeight*info.biWidth,1,alpha);
    int i=0,j=0;
    for (int y=0;y<info.biHeight;y++)
    {
        for (int x=0;x<info.biWidth;x++)
        {
            *(texPtr++)=fileBits[i+2];
            *(texPtr++)=fileBits[i+1];
            *(texPtr++)=fileBits[i];
            i+=3;
            *(texPtr++)=alphaBits[j++];
        }
    }
    delete[] fileBits;
    delete[] alphaBits;
    gluBuild2DMipmaps(GL_TEXTURE_2D,4,info.biWidth,info.biHeight,GL_RGBA,
        GL_UNSIGNED_BYTE,texBits);
    delete[] texBits;
    fclose(fp);
    fclose(alpha);
    return tex;
}

void OpenGLRender::CreateMemoryTexture(Texture *tex)
{
    GLuint nRet[2];
    glGenTextures(2,nRet);
    tex->data=(void*)nRet[0];
    glBindTexture(GL_TEXTURE_2D,(int)(tex->data));
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    unsigned long *buf=new unsigned long[tex->memTex.width*tex->memTex.height];
    for (int y=0,i=0;y<tex->memTex.height;y++)
    {
        for (int x=0;x<tex->memTex.width;x++,i++)
        {
            int color=tex->memTex.data[i];
            buf[i]=((color&0x1f)<<19)|((color&0x3e0)<<6)|
                ((color&0x7c00)>>7)|((color&0x8000)?0xff000000:0);
        }
    }
    gluBuild2DMipmaps(GL_TEXTURE_2D,4,tex->memTex.width,tex->memTex.height,
        GL_RGBA,GL_UNSIGNED_BYTE,buf);
    glBindTexture(GL_TEXTURE_2D,(int)(tex->data)+1);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    glPixelTransferi(GL_RED_SCALE,2);
    glPixelTransferi(GL_GREEN_SCALE,2);
    glPixelTransferi(GL_BLUE_SCALE,2);
    gluBuild2DMipmaps(GL_TEXTURE_2D,4,tex->memTex.width,tex->memTex.height,
        GL_RGBA,GL_UNSIGNED_BYTE,buf);
    glPixelTransferi(GL_RED_SCALE,1);
    glPixelTransferi(GL_GREEN_SCALE,1);
    glPixelTransferi(GL_BLUE_SCALE,1);
    delete[] buf;
}

void OpenGLRender::UpdateMemoryTexture(Texture *tex)
{
    if (!tex->data) return;
    glBindTexture(GL_TEXTURE_2D,(int)(tex->data));
    unsigned long *buf=new unsigned long[tex->memTex.width*tex->memTex.height];
    for (int y=0,i=0;y<tex->memTex.height;y++)
    {
        for (int x=0;x<tex->memTex.width;x++,i++)
        {
            int color=tex->memTex.data[i];
            buf[i]=((color&0x1f)<<19)|((color&0x3e0)<<6)|
                ((color&0x7c00)>>7)|((color&0x8000)?0xff000000:0);
        }
    }
    gluBuild2DMipmaps(GL_TEXTURE_2D,4,tex->memTex.width,tex->memTex.height,
        GL_RGBA,GL_UNSIGNED_BYTE,buf);
    delete[] buf;
}

void OpenGLRender::ReloadTexture(Texture *tex)
{
}

void OpenGLRender::FreeTexture(Texture *tex)
{
    GLuint n[2];
    n[0]=(int)(tex->data);
    n[1]=(int)(tex->data)+1;
    glDeleteTextures(2,n);
}

void OpenGLRender::UseOriginView()
{
    glPushMatrix();
    glLoadIdentity();
}

void OpenGLRender::UseNormalView()
{
    glPopMatrix();
}

void OpenGLRender::StartText()
{
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_DEPTH_TEST);
}

void OpenGLRender::EndText()
{
    glEnable(GL_DEPTH_TEST);
}

void OpenGLRender::RenderText(int x,int y,char *text,unsigned long color)
{
	// DDOI - This will be a FIXME for a while :/
	/*
#ifndef PLATFORM_UNIX
    RECT cr;
    GetClientRect(hWnd,&cr);
    glColor4ub(color&0xff,(color>>8)&0xff,(color>>16)&0xff,0xff);
    glRasterPos3f(2.0*((float)x/(float)cr.right-0.5),2.0*(-((float)(y+curFont->height)/(float)cr.bottom-0.5)*((float)cr.bottom/(float)cr.right)),1);
#else
    glColor4ub(color&0xff,(color>>8)&0xff,(color>>16)&0xff,0xff);
    glRasterPos3f(2.0*((float)x/(float)(scrnX-1)-0.5),2.0*(-((float)(y+curFont->height)/(float)(scrnY-1)-0.5)*((float)(scrnY-1)/(float)(scrnX-1))),1);
#endif
    int i;
    for (i=strlen(text)-1;text[i];i--)
    {
        if (text[i]!=' ')
            break;
    }
    glListBase(curFont->id);
    glCallLists(i+1,GL_UNSIGNED_BYTE,text);
    */
}

void* OpenGLRender::AddFont(HFONT font)
{
#ifndef PLATFORM_UNIX
    SetTextColor(dc,0xffffff);
    SetBkMode(dc,TRANSPARENT);
    SelectObject(dc,font);
    if (firstFont==NULL)
    {
        firstFont=lastFont=new FontListElem;
        lastFont->next=NULL;
        TEXTMETRIC m;
        GetTextMetrics(dc,&m);
        lastFont->height=m.tmAscent;
        SIZE s;
        GetTextExtentPoint32(dc," ",1,&s);
        lastFont->width=s.cx;
        lastFont->id=fontDisplayList;
        wglUseFontBitmaps(dc,0,256,fontDisplayList);
        fontDisplayList+=256;
        return lastFont;
    }
    lastFont->next=new FontListElem;
    lastFont=lastFont->next;
    lastFont->next=NULL;
    lastFont->id=fontDisplayList;
    TEXTMETRIC m;
    GetTextMetrics(dc,&m);
    lastFont->height=m.tmAscent;
    SIZE s;
    GetTextExtentPoint32(dc," ",1,&s);
    lastFont->width=s.cx;
    wglUseFontBitmaps(dc,0,256,fontDisplayList);
    fontDisplayList+=256;
    return lastFont;
#else
// FIXME FONT STUFF!!!
#endif
}

void OpenGLRender::UseFont(void* font)
{
    curFont=(FontListElem*)font;
}

void OpenGLRender::GetStats(RenderStats *stat)
{
    stat->polyDrawn=polyDrawn;
    stat->polyTotal=polyTotal;
    stat->polyReflect=polyReflect;
}

void OpenGLRender::BeginBitmaps()
{
    UseOriginView();
    BeginScene();
}

void OpenGLRender::RenderBitmap(int x,int y,Art *art,int texIndex,unsigned long color)
{
    RenderBitmap(x,y,art->tex[1][texIndex],art->origSizeX[texIndex],art->origSizeY[texIndex],color);
}

void OpenGLRender::RenderBitmap(int x,int y,Art *art,int texIndex,int palNum,unsigned long color)
{
    RenderBitmap(x,y,art->tex[palNum][texIndex],art->origSizeX[texIndex],art->origSizeY[texIndex],color);
}

void OpenGLRender::RenderBitmap(int x,int y,Texture *tex,int w,int h,unsigned long color)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    float ofs=0.3125;
    if (tex->type!=TEXTYPE_REPEAT) ofs=1.3125;
    // Generate a polygon for the bitmap
    p.Clear();
    float xs=w;
    float ys=h;
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)tex->memTex.width,ofs/(float)tex->memTex.height,0,0,color);
    p.AddVert(20.0*(x+xs-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)tex->memTex.width,ofs/(float)tex->memTex.height,0,0,color);
    p.AddVert(20.0*(x+xs-160.0)/160.0,20.0*(100.0-(y+ys))/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)tex->memTex.width,(ys+ofs)/(float)tex->memTex.height,0,0,color);
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-(y+ys))/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)tex->memTex.width,(ys+ofs)/(float)tex->memTex.height,0,0,color);
    p.flags=PF_TWOSIDED|PF_PRIORITY;
    p.texture=tex;
    // Draw the polygon
    RenderPoly(p);
}

void OpenGLRender::RenderBitmapYFlipped(int x,int y,Art *art,int texIndex,unsigned long color)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    float ofs=0.3125;
    if (art->tex[1][texIndex]->type!=TEXTYPE_REPEAT) ofs=1.3125;
    // Generate a polygon for the bitmap
    p.Clear();
    float xs=art->origSizeX[texIndex];
    float ys=art->origSizeY[texIndex];
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)art->sizeX[texIndex],(ys+ofs)/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x+xs-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)art->sizeX[texIndex],(ys+ofs)/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x+xs-160.0)/160.0,20.0*(100.0-(y+ys))/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)art->sizeX[texIndex],ofs/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-(y+ys))/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)art->sizeX[texIndex],ofs/(float)art->sizeY[texIndex],0,0,color);
    p.flags=PF_TWOSIDED|PF_PRIORITY;
    p.texture=art->tex[1][texIndex];
    // Draw the polygon
    RenderPoly(p);
}

void OpenGLRender::RenderScaledBitmap(int x,int y,int w,int h,Art *art,int texIndex,unsigned long color)
{
    RenderScaledBitmap(x,y,w,h,art,texIndex,1,color);
}

void OpenGLRender::RenderScaledBitmap(int x,int y,int w,int h,Art *art,int texIndex,int palNum,unsigned long color)
{
    if ((!renderEnabled)||(!sceneEnabled))
        return;
    float ofs=0.3125;
    if (art->tex[palNum][texIndex]->type!=TEXTYPE_REPEAT) ofs=1.3125;
    // Generate a polygon for the bitmap
    p.Clear();
    float xs=art->origSizeX[texIndex];
    float ys=art->origSizeY[texIndex];
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)art->sizeX[texIndex],ofs/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x+w-160.0)/160.0,20.0*(100.0-y)/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)art->sizeX[texIndex],ofs/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x+w-160.0)/160.0,20.0*(100.0-(y+h))/100.0*((float)scrnY/(float)scrnX),20,(xs+ofs)/(float)art->sizeX[texIndex],(ys+ofs)/(float)art->sizeY[texIndex],0,0,color);
    p.AddVert(20.0*(x-160.0)/160.0,20.0*(100.0-(y+h))/100.0*((float)scrnY/(float)scrnX),20,ofs/(float)art->sizeX[texIndex],(ys+ofs)/(float)art->sizeY[texIndex],0,0,color);
    p.flags=PF_TWOSIDED|PF_PRIORITY;
    p.texture=art->tex[palNum][texIndex];
    // Draw the polygon
    RenderPoly(p);
}

void OpenGLRender::EndBitmaps()
{
    EndScene();
    UseNormalView();
}

void OpenGLRender::ScreenShot()
{
  FILE *f;
  int i, j;
  int sz = scrnX*scrnY;
  unsigned char *image = new unsigned char[sz*3];
  char buffer[32];
  unsigned char *swap_row;
  char filename[64];
  static int count = 0;
  bool done = false;


  while (!done)
  {
    sprintf(filename, "openduke%i.ppm", count++);

    f = fopen(filename, "rb");

    if (f)
      fclose(f);
    else
      done = true;
  }


  f = fopen(filename, "wb");

  swap_row = new unsigned char[scrnX*3];

  if (!f || !image || !swap_row)
  {
    cprintf ("Could not write screenshot.\n");
    return;
  }

  // Capture frame buffer
  glReadPixels(0, 0, scrnX, scrnY,
               GL_RGB, GL_UNSIGNED_BYTE, image);

  // Flip vertical
  for (i = 0, j = scrnY-1; i < scrnY/2; i++, j--)
  {
    memcpy(swap_row, &image[i*scrnX*3], scrnX*3);
    memcpy(&image[i*scrnX*3], &image[j*scrnX*3], scrnX*3);
    memcpy(&image[j*scrnX*3], swap_row, scrnX*3);
  }

  delete [] swap_row;

  sprintf(buffer, "P6\n# CREATOR: openduke\n");
  fwrite(&buffer, 1, strlen(buffer), f);
  sprintf(buffer, "%i %i\n", scrnX, scrnY);
  fwrite(&buffer, 1, strlen(buffer), f);
  sprintf(buffer, "255\n");
  fwrite(&buffer, 1, strlen(buffer), f);

  fwrite(image, sizeof(unsigned char), sz*3, f);

  fclose(f);

  delete [] image;

  cprintf("Took screenshot '%s'.\n", filename);
}
