/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.php.editor.completion;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.netbeans.api.gsf.CompletionProposal;
import org.netbeans.api.gsf.HtmlFormatter;
import org.netbeans.modules.php.doc.DocumentationRegistry;
import org.netbeans.modules.php.doc.FunctionDoc;
import org.netbeans.modules.php.model.Constant;
import org.netbeans.modules.php.model.DoStatement;
import org.netbeans.modules.php.model.Expression;
import org.netbeans.modules.php.model.ExpressionStatement;
import org.netbeans.modules.php.model.ForEachStatement;
import org.netbeans.modules.php.model.ForStatement;
import org.netbeans.modules.php.model.FunctionDeclaration;
import org.netbeans.modules.php.model.FunctionDefinition;
import org.netbeans.modules.php.model.IfStatement;
import org.netbeans.modules.php.model.PhpModel;
import org.netbeans.modules.php.model.ReturnStatement;
import org.netbeans.modules.php.model.SourceElement;
import org.netbeans.modules.php.model.Statement;
import org.netbeans.modules.php.model.SwitchStatement;
import org.netbeans.modules.php.model.WhileStatement;


/**
 * This implementation cares about any non special cases within ExpressionStatement
 * context. ( F.e. variable , that is part of expression statment, handled
 * in different provider ).
 *  
 * @author ads
 *
 */
public class ExpressionContext implements CompletionResultProvider
{

    /* (non-Javadoc)
     * @see org.netbeans.modules.php.editor.completion.CompletionResultProvider#isApplicable(org.netbeans.modules.php.editor.completion.CodeCompletionContext)
     */
    public boolean isApplicable(CodeCompletionContext context) {
        SourceElement e = context.getSourceElement();
        if(e == null) {
            return false;
        }
        return Expression.class.isAssignableFrom( e.getElementType() );
    }

    /* (non-Javadoc)
     * @see org.netbeans.modules.php.editor.completion.CompletionResultProvider#getProposals(org.netbeans.modules.php.editor.completion.CodeCompletionContext)
     */
    public List<CompletionProposal> getProposals(CodeCompletionContext context) {
//            SourceElement currentElement,
//            SourceElement context, CompilationInfo info, int caretOffset, 
//            String prefix, NameKind kind, QueryType queryType, 
//            boolean caseSensitive, HtmlFormatter formatter )
        String prefix = context.getPrefix();
        if ( prefix == null || prefix.length() == 0 ){
            return null;
        }
        SourceElement currentElement = context.getSourceElement();
        assert currentElement != null : "Current SourceElement is null";
        if ( currentElement.getElementType().equals( Constant.class )){
            int caretOffset = context.getCaretOffset();
            HtmlFormatter formatter = context.getFormatter();
            return createFunctionProposals(currentElement, caretOffset, prefix, 
                    formatter);
        }
        return null;
    }

    /**
     * Creates function proposals based on the specified <code>context</code>.
     * @param context a source element that defines the proposal context. 
     * @param caretOffset
     * @param prefix
     * @param formatter
     * @return
     * @see <a href="http://www.php.net/manual/en/language.functions.php">
     * Chapter 17. Functions</a>
     */
    private List<CompletionProposal> createFunctionProposals(
            SourceElement context, int caretOffset, String prefix, 
            HtmlFormatter formatter  ) 
    {
        List<CompletionProposal> list = new LinkedList<CompletionProposal>();
        if ( isMethodContext(context)){
            addBuiltinFunctionProposals(list, caretOffset, prefix, formatter);
            addUserDefinedFunctionProposals(list, context, caretOffset, 
                    prefix, formatter);
        }
        return list;
    }
 
    /**
     * Adds user defined function proposals to the specified <code>list</code>. 
     * @param list a list of the proposals.
     * @param context a source element that defines the proposal context.
     * @param caretOffset
     * @param prefix
     * @param formatter
     * @see <a href=
     * "http://www.php.net/manual/en/language.functions.php#functions.user-defined"
     * >User-defined functions</a> section of the Chapter 17. Functions of 
     * the PHP Manual.
     * @todo Example 17.2. Conditional functions
     * @todo Example 17.3. Functions within functions
     * @todo Example 17.4. Recursive functions
     */
    private void addUserDefinedFunctionProposals(List<CompletionProposal>  list,
            SourceElement context, int caretOffset, String prefix, 
            HtmlFormatter formatter  )
    {       
        PhpModel model = context.getModel();
        model.writeLock();
        try {
            model.sync();
            List<FunctionDefinition> fds = 
                    model.getStatements(FunctionDefinition.class);
            // TODO: Example 17.2. Conditional functions
            // TODO: Example 17.3. Functions within functions
            for (FunctionDefinition fd : fds) {
                FunctionDeclaration decl = fd.getDeclaration();
                if (isMatchedFunction(decl.getName(), prefix)) 
                {
                    list.add(new UserDefinedMethodItem(decl, 
                             caretOffset-prefix.length(), formatter ));
                }
            }
        } 
        finally {
            model.writeUnlock();
        }
    }
    
    private void addBuiltinFunctionProposals(List<CompletionProposal>  list,
            int caretOffset, String prefix, HtmlFormatter formatter ) 
    {
        char firstLetter = prefix.charAt( 0 );
        List<FunctionDoc> docs = DocumentationRegistry.getInstance().
            getFunctionByName( firstLetter );
        if (docs.isEmpty()) {
            return;
        }
        for (FunctionDoc doc : docs) {
            if ( doc.getOwnerName() == null && 
                    isMatchedFunction(doc.getName(), prefix ))
            {
                list.add(new BuiltinMethodItem(doc, caretOffset-prefix.length(),
                         formatter ));
            }
        }
    }
    
    /**
     * Matches given PHP function name with the specified prefix accordig to the
     * PHP language rules that are established by a note in the 
     * <a href=
     * "http://www.php.net/manual/en/language.functions.php#functions.user-defined">
     * User-defined functions</a> section of the <i>Chapter 17. Functions</i> of
     * the PHP Manual:
     * <p>
     * "Note:  Function names are <u>case-insensitive</u>, though it is usually 
     * good form to call functions as they appear in their declaration."
     * </p>
     * 
     * @param fName a PHP function name.
     * @param prefix a prefix.
     * @return <code>true</code> if <code>fName</code> is matched with 
     * <code>prefix</code>, otherwise <code>true</code>.
     */
    private static boolean isMatchedFunction(String fName, String prefix) {
        // TODO: declare this method as public and move it into the Php class.
        return fName.toLowerCase().startsWith(prefix.toLowerCase());
    }
    
    private boolean isMethodContext( SourceElement element ) {
        SourceElement parent = element.getParent();
        while( parent != null ) {
            if ( METHOD_CONTEXT.contains( parent.getElementType() ) ) {
                return true;
            }
            parent = parent.getParent();
        }
        return false;
    }
    
    private static Set<Class<? extends Statement>> METHOD_CONTEXT;
    
    static {
        METHOD_CONTEXT = new HashSet<Class<? extends Statement>>();
        METHOD_CONTEXT.add( ExpressionStatement.class );
        METHOD_CONTEXT.add( ReturnStatement.class );
        METHOD_CONTEXT.add( DoStatement.class );
        METHOD_CONTEXT.add( ForEachStatement.class );
        METHOD_CONTEXT.add( ForStatement.class );
        METHOD_CONTEXT.add( IfStatement.class );
        METHOD_CONTEXT.add( SwitchStatement.class );
        METHOD_CONTEXT.add( WhileStatement.class );
    }

}
