 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /*************************************************************************/
 /*                                                                       */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)             */
 /*                   Date: Fri May  9 1997                               */
 /* -------------------------------------------------------------------   */
 /* Example of declaration and use of tracks.                             */
 /*                                                                       */
 /*************************************************************************/


#include <iostream.h>
#include <stdlib.h>
#include "EST_Track.h"
#include "EST_error.h"

void dump_track(EST_Track &tr, EST_String comment)
{
  printf("[ Track %s\n", (const char *)comment);
  for (int f=0; f<tr.num_frames(); f++)
    if (tr.val(f))
      {
	printf("  %3d:\t%3.3f", f, tr.t(f));
	for(int c=0; c<tr.num_channels(); c++)
	  printf("\t%3.3f", tr.a(f,c));
	printf("\n");
      }
    else
      printf("  BREAK\n");
  printf("]\n");
}

static EST_TrackMap::ChannelMapping track_mapping =
{
  { channel_power, 0 },
  { channel_unknown},		// just to mark the end
};

EST_TrackMap TestTrackMap(track_mapping);		// decl 

/**@name EST_Track:example
  * 
  * Some examples of track manipulations.
  *
  * @see EST_Track
  * @see EST_TrackMap:example
  * @see EST_TMatrix
  * @see EST_TMatrix:example
  */
//@{

int main(void)

{
  /**@name Simple Access
    * 
    * Values in the track can be accessed and set by frame
    * number and channel number.
    */

  //@{ code
  EST_Track tr(10,1); // decl

  tr.assign_map(TestTrackMap);

  tr.set_equal_space(FALSE); // decl
  
    for(int i=0; i<tr.num_frames(); i++)
    {
	tr.t(i) = i*i;
	tr.a(i,0) = i+i;
    }

    dump_track(tr, "");
    //@} code

    /**@name File I/O
      */
    //@{ code

    if (tr.save("tmp/track.htk", "htk") != write_ok)
	EST_error("can't save htk\n");

    EST_Track tr2;

    if (tr2.load("tmp/track.htk") != read_ok)
	EST_error("can't reload htk\n");

    dump_track(tr2, "saved and loaded as htk");

    if (tr.save("tmp/track.esps", "esps") != write_ok)
	EST_error("can't save esps\n");

    if (tr2.load("tmp/track.esps") != read_ok)
	EST_error("can't reload esps\n");

    dump_track(tr2, "saved and loaded as esps");

    if (tr.save("tmp/track.est", "est") != write_ok)
	EST_error("can't save est\n");

    if (tr2.load("tmp/track.est") != read_ok)
	EST_error("can't reload est\n");

    dump_track(tr2, "saved and loaded as est");

    //@} code

    /**@name Resizing A Track
      */
    //@{ code

    tr.resize(-1,tr.num_channels()+3);

    dump_track(tr, "with extra channel added");

    //@} code

    /**@name Direct Access To Entire Frames.
      */
    //@{ code
    // This is the kind of evil thing some people want to do
    float *p = &(tr.a(3,0));

    for(int i2=0; i2<tr.num_channels(); i2++)
      p[i2] += 100+i2;

    // Here is a better way
    float *buf = new float[tr.num_channels()];

    tr.copy_frame(4,buf);
    
    for(int i3=0; i3<tr.num_channels(); i3++)
      buf[i3] += 200+i3;

    tr.set_frame(4, buf);

    // This is the right way to do it.
    EST_FVector frame_5;

    tr.frame(frame_5, 5);

    for(int i4=0; i4<tr.num_channels(); i4++)
      frame_5[i4] += 300+i4;

    dump_track(tr, "frames changed.");

    EST_Track subtrack;

    tr.sub_track(subtrack, 3, 5, 1, 2);

    subtrack.fill(12.34);

    dump_track(tr, "sub track changed.");

    subtrack.save("tmp/subtrack.est", "est");

    //@} code


    exit(0);
}

//@}
