 /*************************************************************************/
 /*                                                                       */
 /*                Centre for Speech Technology Research                  */
 /*                     University of Edinburgh, UK                       */
 /*                       Copyright (c) 1996,1997                         */
 /*                        All Rights Reserved.                           */
 /*  Permission to use, copy, modify, distribute this software and its    */
 /*  documentation for research, educational and individual use only, is  */
 /*  hereby granted without fee, subject to the following conditions:     */
 /*   1. The code must retain the above copyright notice, this list of    */
 /*      conditions and the following disclaimer.                         */
 /*   2. Any modifications must be clearly marked as such.                */
 /*   3. Original authors' names are not deleted.                         */
 /*  This software may not be used for commercial purposes without        */
 /*  specific prior written permission from the authors.                  */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
 /*  THIS SOFTWARE.                                                       */
 /*                                                                       */
 /*************************************************************************/
 /*                                                                       */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)             */
 /*                   Date: Tue Jul 22 1997                               */
 /* --------------------------------------------------------------------- */
 /* Example of matrix class use.                                          */
 /*                                                                       */
 /*************************************************************************/

#include <stdlib.h>
#include <iostream.h>
#include "EST_TMatrix.h"

static inline int max(int a, int b) { return a>b?a:b; }


/**@name EST_TMatrix:example
  * 
  * Example of matrix class use.
  *
  * @see EST_TMatrix
  * @see EST_TVector
  */
//@{

int main(void)
{
  /**@name Declaration
    * 
    * Matrixes are declared by giving the type and the number of 
    * rows and columns.
    */
  //@{ code

  // 10 rows by 5 columns
  EST_TMatrix<float> m(10, 5);

  //@} code


  /**@name Access
    *
    * Access to values in the matrix is via the a() member function.
    * This returns a reference, so you can assign values to matrix cells.
    */
  //@{ code

  // Fill it with something easy to recognise.
  for(int i=0; i<m.num_rows(); i++)
    for(int j=0; j<m.num_columns(); j++)
      m.a(i,j) = i+j/100.0;

  //@} code

  /**@name Output
    * A simple output method is supplied, it just outputs a row at a time,
    * tab separated.
    */
  //@{ code
  // Write to standard output in an ascii format.
  cout << "Initial Matrix\n";
  m.save("-");
  cout << "\n";
  //@} code

  /**@name Resizing
    * Resize to 20 rows by 10 columns This fills the new
    * area with {\tt def_val}, which is 0.0 for floats.
    */
  //@{ code
  m.resize(20,10);
  //@} code

  cout << "Resized Matrix\n";
  m.save("-");
  cout << "\n";

  // Fill it with something easy to recognise.
  for(int i0=0; i0<m.num_rows(); i0++)
    for(int j=0; j<m.num_columns(); j++)
      m.a(i0,j) = i0+j/100.0;

  // Write to standard output in an ascii format.
  cout << "Full Matrix\n";
  m.save("-");
  cout << "\n";

  /**@name Copying Data to/from a buffer
    * 
    * Whole rows or columns can be extracted into a buffer, or can be
    * filled with data from a buffer. The buffer must be pre-declared.
    */
  //@{ code
  float *buf = new float[max(m.num_rows(),m.num_columns())];

  // Extract data
  m.copy_row(5, buf);

  cout << "Row 5\n";
  for(int j1=0; j1<m.num_columns(); j1++)
    cout << buf[j1] << "\t";
  cout << "\n\n";

  // And put data in
  for(int i1=0; i1<m.num_rows(); i1++)
    buf[i1] = i1+100;

  m.set_column(5,buf);

  delete [] buf;
  //@} code

  cout << "Updated Matrix (column 5 replaced with 100s from buffer)\n";
  m.save("-");
  cout << "\n";

  /**@name Sub-Matrices and Sub-Vectors
    *
    * A sub-vector or sub-matrix is a window onto a matrix. If you obtain a
    * sub vector representing a row, for instance, you can treat it
    * a normal vector, any changes you make affecting the underlying
    * matrix.
    */

  //@{ code
  EST_TVector<float> row;
  EST_TVector<float> column;
  EST_TMatrix<float> rectangle;

  m.row(row, 10);
  m.column(column, 3);
  m.sub_matrix(rectangle, 
	       8, 5, 
	       2, 3);

  cout <<"Row 10 extracted as sub vector\n";
  row.save("-");
  cout << "\n";

  cout <<"Column 3 extracted as sub vector\n";
  column.save("-");
  cout << "\n";

  cout <<"Rectangle extracted as sub vector\n";
  rectangle.save("-");
  cout << "\n";

  // If we update the sub-vector, the main matrix changes.

  // Row becomes squares of the index
  for(int i2=0; i2<row.n(); i2++)
    row[i2] = i2*i2;

  // Column becomes cubes of the index
  for(int i3=0; i3<column.n(); i3++)
    column[i3] = i3*i3*i3;

  // Rectangle filled with -1
  for(int i4=0; i4<rectangle.num_rows(); i4++)
    for(int j4=0; j4<rectangle.num_columns(); j4++)
      rectangle.a(i4, j4) = -1;

  // Now we see if it works recursively

  EST_TVector<float> rrow;
  EST_TVector<float> rcolumn;

  rectangle.row(rrow, 2);
  rectangle.column(rcolumn, 1);

  for(int i6=0; i6<rcolumn.n(); i6++)
    rcolumn[i6] = -3;
  for(int i5=0; i5<rrow.n(); i5++)
    rrow[i5] = -2;

  //@} code

  cout << "Updated Matrix (row 10 becomes squares, column 3 becomes cubes, center becomes negative)\n";
  m.save("-");
  cout << "\n";

  exit(0);
}

//@}

