/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Paul Taylor and Alan Black                      */
/*             Date   :  June 1996                                       */
/*-----------------------------------------------------------------------*/
/*             more EST_Wave class methods involving streams             */
/*                                                                       */
/*=======================================================================*/

#include <string.h>
#include "EST_unix.h"
#include <stdlib.h>
#include "EST_cutils.h"
#include "EST_string_aux.h"
#include "EST_Wave.h"
#include "EST_wave_aux.h"
#include "ling_class/EST_Stream.h"
#include "ling_class/EST_stream_aux.h"

static int wave_subwave(EST_Wave &subsig,EST_Wave &sig, 
			float offset, float length);

int wave_divide(EST_WaveList &wl, EST_Wave &sig, EST_Stream &keylab,
		const EST_String &ext)
{
    wl.clear();
    EST_Wave a;
    EST_Stream_Item *k;
    EST_String filename;
    
    for (k = keylab.head(); k; k = next(k))
    {
	a.clear();
	if (k->dur() < 0)
	    continue;
	wave_subwave(a, sig, k->start(), k->dur());
	filename = k->f("file");
	a.set_name(filename + ext);
	wl.append(a);
    }

    return 0;
}

int wave_extract(EST_Wave &part, EST_Wave &sig, EST_Stream &keylab, 
		 const EST_String &file)
{
    EST_Wave a;
    EST_Stream_Item *k;
    EST_String key_file_name;
    
    for (k = keylab.head(); k; k = next(k))
    {
	key_file_name = k->f("file");
	if (key_file_name == file)
	{
	    wave_subwave(part, sig, k->start(), k->dur());
	    return (0);
	}
    }
    cerr << "Couldn't locate file fragment " << file << " in keylab file\n";
    return -1;
}


static int wave_subwave(EST_Wave &subsig,EST_Wave &sig, 
			float offset, float length)
{
    return wave_subwave(subsig, sig, (int)(offset *(float)sig.sample_rate()),
			(int)(length *(float)sig.sample_rate()));
}

int wave_subwave(EST_Wave &subsig,EST_Wave &sig,int offset,int length)
{
    // take out a subpart of sig and put it in subsig
    int ns;

    if (length == -1)
	ns = sig.num_samples() - offset;
    else
	ns = length;
    
    if ((offset+ns) > sig.num_samples())
    {
	cerr << "Subset past end of signal\n";
	return -1;
    }

    EST_Wave subwave;

    sig.sub_wave(subwave, offset, ns, 0, EST_ALL);

    subsig.copy(subwave);

    return 0;
}

int track_divide(EST_TList<EST_Track> &mtfr, EST_Track &fv, EST_Stream &key)
{
    EST_Track a;
    EST_Stream_Item  *k, t;
    float kstart, length;
    int i, j, l, n;
    
    mtfr.clear();
    
    if ((key.tail())->end() < (fv.t(fv.num_frames() - 1)))
    {
	cerr << "Key file must extend beyond end of EST_Track\n";
	cerr << "key end: " << key.tail()->end() << " EST_Track end: " 
	    << fv.t(fv.num_frames() - 1) << endl;
	return -1;
    }
    
    k = key.head();
    a.set_name(k->name());
    kstart = 0.0;
    
    length = k->end() - kstart;
    n = (int)(length / (float) fv.shift()) + 2;
    a.resize(n, fv.num_channels());
    
    for (i = 0, l = 0; i < fv.num_frames(); ++i, ++l)
    {
	for (j = 0; j < fv.num_channels(); ++j)
	    a(l, j) = fv(i, j);
	
	if (fv.t(i) > k->end())
	{
	    a.set_num_frames(l + 1);
	    mtfr.append(a);
	    
	    kstart = k->end();
	    k = next(k);
	    a.set_name(k->name());
	    length = k->end() - kstart;
	    n = (int)(length / (float) fv.shift()) + 2;
	    //	    cout << "n frames: " << n << endl;
	    a.resize(n, fv.num_channels());
	    a.fill_time(fv.shift());
	    
	    //	for (j = 0; j < fv.order(); ++j)
	    //	    a(0, j) = fv(i, j);
	    l = -1;
	}
    }
    a.set_num_frames(l);
    mtfr.append(a);
    return 0;
}

