/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1996-1998                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  February 1998                           */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* Functions to add Speech Tools basic objects to the SIOD LISP obj      */
/*                                                                       */
/* This offers non-intrusive support for arbitrary objects in LISP,      */
/* however because the deletion method are called this needs to access   */
/* Thus if you include siod_est_init(), you'll get Utterances, Nodes     */
/* Stream_Items, Waves and Tracks in your binary                         */
/*                                                                       */
/*=======================================================================*/
#include <iostream.h>
#include "siod.h"
#include "ling_class/EST_Utterance.h"
#include "ling_class/EST_Item.h"
#include "EST_THash.h"
#include "EST_Wave.h"
#include "EST_Track.h"

#if defined(INSTANTIATE_TEMPLATES)
#include "../base_class/EST_THash.cc"
template class EST_TStringHash<LISP>;
template class EST_THash<EST_String,LISP>;
template class EST_Hash_Pair<EST_String,LISP>;
#endif
LISP EST_THash<EST_String,LISP>::Dummy_Value = NIL;


// To make garbage collection easy the following functions offer an index
// of arbitrary objects to LISP cells.  You can use this to return the
// same LISP cell for the same object.  This is used for utterance
// objects otherwise I'd need to add reference counts to the utterance
// itself
//
// This is implemented as a hash table of printed address
// This if fine for hundreds of things, but probably not
// for thousands of things
static EST_TStringHash<LISP> estobjs(100);

static void void_to_addrname(void *v,EST_String &saddr)
{
    char addr[128];

    sprintf(addr,"%p",v);
    saddr = addr;
}

// The following are the types for EST objects in LISP, they are set when
// the objects are registered.  I don't think they should be required 
// out side this file so they are static functions like siod_utterance_p
// should be used elsewhere
static int tc_utt = -1;
static int tc_wave = -1;
static int tc_item = -1;
static int tc_track = -1;

class EST_Utterance *get_c_utt(LISP x)
{
    if (TYPEP(x,tc_utt))
	return (class EST_Utterance *)USERVAL(x);
    else
	err("wta to get_c_utt",x);

    return NULL;  // err doesn't return but compilers don't know that
}

int siod_utterance_p(LISP x)
{
    if (TYPEP(x,tc_utt))
	return TRUE;
    else
	return FALSE;
}

LISP siod_make_utt(class EST_Utterance *u)
{
    LISP utt;
    EST_String saddr;
    LISP cell;

    void_to_addrname(u,saddr);

    if ((cell = estobjs.val(saddr)) != NIL)
	return cell;

    // A new one 
    utt = siod_make_typed_cell(tc_utt,u);

    // Add to list
    estobjs.add_item(saddr,utt);

    return utt;
}

static void utt_free(LISP utt)
{
    class EST_Utterance *u = get_c_utt(utt);
    EST_String saddr;

    void_to_addrname(u,saddr);

    // Mark it unused, this doesn't gc the extra data in the hash
    // table to hold the index, this might be a problem over very
    // long runs of the system (i.e. this should be fixed).
    estobjs.remove_item(saddr);
    delete u;

    USERVAL(utt) = NULL;
}

LISP utt_mark(LISP utt)
{
    // Should mark all the LISP cells in it 
    // but at present we use the gc_(un)protect mechanism 
    return utt;
}

// Item functions
class EST_Item *get_c_item(LISP x)
{
    if (TYPEP(x,tc_item))
	return (class EST_Item *)USERVAL(x);
    else
	err("wta to get_c_item",x);

    return NULL;  // err doesn't return but compilers don't know that
}

int siod_item_p(LISP x)
{
    if (TYPEP(x,tc_item))
	return TRUE;
    else
	return FALSE;
}

LISP siod_make_item(class EST_Item *s)
{
    if (s==0)
	return NIL;
    else
	return siod_make_typed_cell(tc_item,s);
}

// Wave
class EST_Wave *get_c_wave(LISP x)
{
    if (TYPEP(x,tc_wave))
	return (class EST_Wave *)USERVAL(x);
    else
	err("wta to get_c_wave",x);

    return NULL;  // err doesn't return but compilers don't know that
}

int siod_wave_p(LISP x)
{
    if (TYPEP(x,tc_wave))
	return TRUE;
    else
	return FALSE;
}

LISP siod_make_wave(class EST_Wave *s)
{
    if (s==0)
	return NIL;
    else
	return siod_make_typed_cell(tc_wave,s);
}

static void wave_free(LISP wave)
{
    class EST_Wave *w = get_c_wave(wave);  // will throw error if not wave
    delete w;
    USERVAL(wave) = NULL;
}

// Track
class EST_Track *get_c_track(LISP x)
{
    if (TYPEP(x,tc_track))
	return (class EST_Track *)USERVAL(x);
    else
	err("wta to get_c_track",x);

    return NULL;  // err doesn't return but compilers don't know that
}

int siod_track_p(LISP x)
{
    if (TYPEP(x,tc_track))
	return TRUE;
    else
	return FALSE;
}

LISP siod_make_track(EST_Track *s)
{
    if (s==0)
	return NIL;
    else
	return siod_make_typed_cell(tc_track,s);
}

static void track_free(LISP track)
{
    class EST_Track *t = get_c_track(track);  // will throw error if not track
    delete t;
    USERVAL(track) = NULL;
}

void siod_est_init()
{
    // add EST specific objects as user types to LISP obj
    long kind;

    // In general to add a type
    // tc_TYPENAME = siod_register_user_type("TYPENAME");
    // define above 
    //    EST_TYPENAME *get_c_TYPENAME(LISP x) and
    //    int siod_TYPENAME_p(LISP x)
    //    LISP siod_make_utt(EST_TYPENAME *x)
    // you will often also need to define 
    //    TYPENAME_free(LISP x) too if you want the contents gc'd
    // other options to the set_*_hooks functions allow you to customize
    // the objects behavior more

    tc_utt = siod_register_user_type("Utterance");
    set_gc_hooks(tc_utt, 0, NULL,utt_mark,NULL,utt_free,NULL,&kind);

    // EST_Items are always part of Utterances so
    // don't gc them when their cell is gc'd
    tc_item = siod_register_user_type("Item");

    // Waves and Tracks are (usually) not elsewhere so they do get gc'd
    tc_wave = siod_register_user_type("Wave");
    set_gc_hooks(tc_wave, 0, NULL,NULL,NULL,wave_free,NULL,&kind);
    tc_track = siod_register_user_type("Track");
    set_gc_hooks(tc_track, 0, NULL,NULL,NULL,track_free,NULL,&kind);

}

