/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                 Author: Paul Taylor                                   */
/*                 Date   :  March 1998                                  */
/*-----------------------------------------------------------------------*/
/*    Signal processing functions which operate on entire utterances     */
/*                                                                       */
/*=======================================================================*/

#include "EST_error.h"
#include "EST_track_aux.h"
#include "sigpr/EST_sig2fv.h"
#include "sigpr/EST_sigpr_utt.h"
#include "EST_inline_utils.h"

void make_fixed_times(EST_Track &pm, float end, float shift)
{
    int num_frames = (int)ceil(end / shift);
    pm.resize(num_frames, EST_CURRENT);
//    cout << "nf: " << pm.num_frames() << " nc:" << pm.num_channels() << endl;
    pm.fill_time(shift);
}

void make_fixed_times(EST_Track &pm, EST_TrackMap &map, float end, 
		      float shift)
{
    int num_frames = (int)ceil(end / shift);
    pm.resize(num_frames, map);
    pm.fill_time(shift);
}

EST_String sigpr_options_supported(void)
{
    EST_String s("Signal processing operations supported:\n");

    s += "lpc      linear predictive coding\n";
    s += "cep      cepstrum coding from lpc coefficients\n";
    s += "melcep   Mel scale cepstrum coding via fbank\n";
    s += "fbank    Mel scale log filterbank analysis\n";
    s += "lsf      line spectral frequencies\n";
    s += "ref      Linear prediction reflection coefficients\n";
    s += "power\n";
    s += "f0\n";
    s += "energy: root mean square energy\n";

    return s;
}

void print_map(EST_TrackMap &map)
{
    cout << "map itself\n";
    for (int i = 0; i <= map.last_channel(); ++i)
	cout << "channel:" << i << "  " << ((int)map.channel_type(i)) 
	    << " " << 
	    EST_default_channel_names.value(map.channel_type(i)) << endl;
}

void print_track_map(EST_Track &t)
{
    cout << "track names\n";
    for (int i = 0; i < t.num_channels(); ++i)
	cout << "channel:" << i << ":" << t.channel_name(i) << endl;
}

static bool single_channel(const EST_String &name)
{
    if (name.contains("power") || name.contains("energy") 
	|| name.contains("f0"))
	return 1;
    return 0;
}

EST_String delta_name(const EST_String &name, int order = 1)
{
    EST_String d;

    if (order > 1)
	delta_name(name, order -1);

    if (single_channel(name))
	d = name + "_d";
    else
    {
	d = name;
	d.gsub("_0", "_d_0");
    }
    return d;
}

EST_String coef0_name(const EST_String &name, int order = 0)
{
    EST_String os;
    if (order == 2)
	os = "_a";
    else if (order == 1)
	os = "_d";

    return (single_channel(name)) ? (name + os) : (name + os + "_0");
}

EST_String coefN_name(const EST_String &name, int order = 0)
{
    EST_String os;
    if (order == 2)
	os = "_a";
    else if (order == 1)
	os = "_d";

    return (single_channel(name)) ? (name + os) : (name + os + "_N");
}

static int get_coef_order(const EST_Track &fv, EST_ChannelType start)
{
    EST_String b_name = EST_default_channel_names.value(start);

//    cout << endl << endl;

    if (single_channel(b_name))
	return 1;
    else
    {
//	cout << "start var: " << start << endl;
//	cout << "start channel: " << fv.channel_position(start) << endl;
//	cout << "end channel: " << fv.channel_position(EST_ChannelType(start + 1)) << endl;

	return fv.channel_position(EST_ChannelType(start + 1))
	- fv.channel_position(start) + 1;
    }
}

static int get_coef_order(EST_Track &fv, const EST_String &name)
{
//    cout << "name thang: " << name << endl;
    EST_ChannelType k = EST_default_channel_names.token(name);
//    cout << "name thang num: " << k << endl;
//    cout << "double check is: " << EST_default_channel_names.value(k) << endl;
    return get_coef_order(fv, k);
}

static void enum_sub_track(const EST_Track &main, EST_Track &sub, 
			   EST_ChannelType coef0)
{
    int start = main.channel_position(coef0);
    int num = get_coef_order(main, coef0);
//    cout << "coef0: " << coef0 << " start " << start << " num " << num << endl;
    main.sub_track(sub, 0, EST_ALL, start, num);
}

static void enum_sub_track(const EST_Track &main, EST_Track &sub, 
			   const EST_String &name)
{
    EST_ChannelType k = EST_default_channel_names.token(name);
    enum_sub_track(main, sub, k);
}

/*static void copy_times(EST_Track &main, EST_Track &fv, int channels=0)
{
    fv.resize(main.num_frames(), channels);
    for (int i = 0; i < main.num_frames(); ++i)
	fv.t(i) = main.t(i);
}
*/


void make_timed_track(const EST_Track &main, EST_Track &fv, 
		       const EST_String &name, int coef_order, int del_order)
{

    int c = 0; // must be variable because of reference in fn call

    EST_TrackMap::P map = new EST_TrackMap();
    map->start_refcounting(1);
    add_to_track_map(name, *map, coef_order, del_order, c);


    fv.resize(main.num_frames(), *map);

//    cout << "FV:" << coef_order << endl;
    print_track_map(fv);

    for (int i = 0; i < main.num_frames(); ++i)
	fv.t(i) = main.t(i);
}

void make_track(EST_Track &fv, const EST_String &name, 
		int coef_order, int del_order)
{
    int c = 0; // must be variable because of reference in fn call

    EST_TrackMap::P map = new EST_TrackMap();
    map->start_refcounting(1);
    add_to_track_map(name, *map, coef_order, del_order, c);

    fv.resize(EST_CURRENT, *map);
}

void make_fixed_timed_track(EST_Track &fv, const EST_String &name, 
			    float end, int coef_order, float shift,
			    int del_order)
{
    int c = 0; // must be variable because of reference in fn call

    EST_TrackMap::P map = new EST_TrackMap();
    map->start_refcounting(1);
    add_to_track_map(name, *map, coef_order, del_order, c);

    make_fixed_times(fv, *map, end, shift);
}

void add_to_track_map(const EST_String &name, EST_TrackMap &map, 
			   int coef_order, int delta_order, int &c)
{
    if (name == "cep")
    {
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_cepstrum_0, 
						     delta_order)), c);
	c += coef_order;
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_cepstrum_N, 
						     delta_order)), c -1);
    }
    else if (name == "melcep")
    {
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_melcepstrum_0, 
						     delta_order)), c);
	c += coef_order;
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_melcepstrum_N, 
						     delta_order)), c -1);
    }
    else if (name == "fbank")
    {
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_fbank_0, 
						     delta_order)), c);
	c += coef_order;
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_fbank_N, 
						     delta_order)), c -1);
    }
    else if (name == "filter")
    {
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_filter_0, 
						     delta_order)), c);
	c += coef_order;
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_filter_N, 
						     delta_order)), c -1);
    }
    else if (name == "lpc")
    {
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_lpc_0, 
						     delta_order)), c);
	c += coef_order;
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_lpc_N, 
						     delta_order)), c - 1);
    }
    else if (name == "lsf")
    {
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_lsf_0, 
						     delta_order)), c);
	c += coef_order;
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_lsf_N, 
						     delta_order)), c - 1);
    }
    else if (name == "power")
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_power, 
						     delta_order)), c++);
    else if (name == "energy")
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_energy, 
						     delta_order)), c++);
    else if (name == "f0")
	map.set(EST_ChannelType(EST_ChannelTypeDelta(channel_f0, 
						     delta_order)), c++);
    else
	EST_error("Error: Unnknown type of processing requested: %s\n", 
		  ((const char*)name));
}


void add_to_track_map(const EST_StrList &slist, EST_TrackMap &map, 
			   EST_Option &op, int delta_num, int &c)
{
    int length;
    for (EST_Litem *s = slist.head(); s; s = next(s))
    {
	if (slist(s) == "cep")
	    length = op.ival("cep_order");
	else if (slist(s) == "fbank")
	    length = op.ival("fbank_order");
	else if (slist(s) == "melcep")
	    length = op.ival("melcep_order");
	else if (slist(s) == "lpc")
	    length = op.ival("lpc_order") + 1;
	else if (slist(s) == "lsf")
	    length = op.ival("lpc_order") + 1;
	else
	    length = 1;
	add_to_track_map(slist(s), map, length, delta_num, c);
    }
}

void sig2fv_fixed(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
		  float shift, float length,
		  EST_StrList &base_list, const EST_StrList &delta_list, 
		  const EST_StrList &acc_list)
{
    (void) length;
    make_fixed_times(fv, sig.end(), shift);
    sig2fv(sig, fv, op, base_list, delta_list, acc_list);
}

void sig2fv(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
	    const EST_StrList &base_list, const EST_StrList &delta_list, 
	    const EST_StrList &acc_list)
{
    int c = 0;
    EST_TrackMap::P map = new EST_TrackMap();
    map->start_refcounting(1);

//    cout << "base list\n" << base_list << endl;
//    cout << "delta list\n" << delta_list << endl;

    add_to_track_map(base_list, *map, op, 0, c);
    add_to_track_map(delta_list, *map, op, 1, c);
    add_to_track_map(acc_list,* map, op, 2, c);

    fv.resize(EST_CURRENT, *map);

//    cout << "map:\n";
//    print_track_map(fv);

//    cout << "size " << fv.num_frames() << " " << fv.num_channels() << endl;

    sigpr_base(sig, fv, op, base_list);
    sigpr_delta(sig, fv, op, delta_list);
    sigpr_acc(sig, fv, op, acc_list);
}


static void parse_op_settings(EST_Option &op, EST_WindowFunc *&wf, float &f)
{
    EST_String w_name;

    if (op.present("window_type"))
	w_name = op.val("window_type");
    else
	w_name = DEFAULT_WINDOW_NAME;
    wf = EST_Window::creator(w_name);

    f = op.present("frame_factor") ? op.fval("frame_factor")
	: DEFAULT_FRAME_FACTOR;
}

void sigpr_base(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
		const EST_StrList &slist)
{
    EST_Track fill, tmp;
    EST_String b_name;
    EST_ChannelType k;
    float frame_factor;
    EST_WindowFunc *wf;

    int fbank_order;
    float liftering_parameter=0;
    bool use_power_rather_than_energy=false, take_logs=true, include_c0=false;

    parse_op_settings(op, wf, frame_factor);

    for (EST_Litem *s = slist.head(); s; s = next(s))
    {
	k = EST_default_channel_names.token(slist(s));
	if (k == channel_unknown)
	    k = EST_default_channel_names.token(slist(s) + "_0");
	if (!fv.has_channel(k))
	    EST_error("Unable to create coefficients for %s:"
		      "no space allocated in track\n", (const char *)slist(s));

//	cout << "making " << slist(s) << endl;
	enum_sub_track(fv, fill, k);

	if(op.present("usepower"))
	    cerr << "USING POWER" << endl;

	if ((slist(s) == "lpc") || (slist(s) == "cep") 
	    ||(slist(s) == "ref") || (slist(s) == "lsf"))
	    sig2coef(sig, fill, slist(s), frame_factor, wf);
	else if (slist(s) == "power")
	    power(sig, fill, frame_factor);
	else if (slist(s) == "energy")
	    energy(sig, fill, frame_factor);
	else if (slist(s) == "f0")
	    pda(sig, fill, op, "srpd");

 	else if (slist(s) == "fbank")
	{
	    use_power_rather_than_energy = op.present("usepower");
	    fbank(sig, fill, frame_factor, wf, use_power_rather_than_energy, take_logs);
	}
	
        else if (slist(s) == "melcep")
	{
	    fbank_order=op.ival("fbank_order");
	    use_power_rather_than_energy = op.present("usepower");
	    include_c0=op.present("include_c0");
	    
	    if(op.present("lifter"))
		liftering_parameter=op.fval("lifter");

	    melcep(sig, fill, frame_factor, fbank_order,
		   liftering_parameter, wf, include_c0, use_power_rather_than_energy);
	}

	else
	    EST_error("Error: Unnknown type of processing requested: %s\n", 
		      ((const char*) slist(s)));
    }
}

void sigpr_delta_track(EST_Wave &sig, EST_Track &fill, const EST_String &type,
		       EST_Option &op, int order, const EST_Track &fv)
{
    EST_Track base;

    EST_String cur_coef0 = coef0_name(type, order);
    EST_String cur_coefN = coefN_name(type, order);
    EST_String prev_coef0 = coef0_name(type, order - 1);
    EST_String prev_coefN = coefN_name(type, order - 1);
    EST_WindowFunc *wf;
    float frame_factor;

//    cout << "type = " << type << " coef name:" << cur_coef0 << endl;
//    cout << "type = " << type << " prev name:" << prev_coef0 << endl;
    /*    cout << "current map\n";
	  print_track_map(fv);
	  cout << "current fill map\n";
	  print_track_map(fill);
	  */
    parse_op_settings(op, wf, frame_factor);

    if (fv.has_channel(prev_coef0))
	enum_sub_track(fv, base, prev_coef0);
    else
    {
	int coef_order = get_coef_order(fill, cur_coef0);
	make_timed_track(fill, base, type, coef_order, order - 1);

	print_track_map(base);
	
	if (order > 1)
	    sigpr_delta_track(sig, base, type, op, order - 1, fill);
	else
	{
	    if ((type == "lpc") || (type == "cep") 
		||(type == "ref") || (type == "lsf"))
		sig2coef(sig, base, type, frame_factor, wf);
	    else if (type == "power")
		power(sig, base, frame_factor);
	    else if (type == "energy")
		energy(sig, base, frame_factor);
	    else if (type == "f0")
		pda(sig, base, op, "srpd");
	    else
		EST_error("Error: Unknown type of processing requested: %s\n",
			  ((const char*) type));
	}
    }
//    cout << "\n\n base: " << base.num_frames() 
//	<< " " << base.num_channels() << endl;
//    cout << "\n\n fill: " << fill.num_frames() 
//	<< " " << fill.num_channels() << endl;
    delta(base, fill);
}

void sigpr_delta(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
		 const EST_StrList &slist)
{
    EST_Track del;
    EST_ChannelType d;
    
    for (EST_Litem *s = slist.head(); s; s = next(s))
    {
	d = EST_default_channel_names.token(slist(s) + "_d");
	if (d == channel_unknown)
	    d = EST_default_channel_names.token(slist(s) + "_d_0");
	if (!fv.has_channel(d))
	    EST_error("Unable to create delta coefficients for %s:"
		      "no space allocated in track\n", (const char *)slist(s));
	
	enum_sub_track(fv, del, d);
	sigpr_delta_track(sig, del, slist(s), op, 1, fv);
    }
}

void sigpr_acc(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
	       const EST_StrList &slist)
{
    EST_Track del;
    EST_ChannelType a;
    EST_String d;
    
    for (EST_Litem *s = slist.head(); s; s = next(s))
    {
	a = EST_default_channel_names.token(slist(s) + "_a");
	if (a == channel_unknown)
	    a = EST_default_channel_names.token(slist(s) + "_a_0");
	if (!fv.has_channel(a))
	    EST_error("Unable to create acceleration coefficients for %s:"
		      "no space allocated in track\n", (const char *)slist(s));
	
	enum_sub_track(fv, del, a);
	d = delta_name(slist(s));
//	cout << "ac delta name : " << slist(s) << " " << d << endl;
	sigpr_delta_track(sig, del, d, op, 2, fv);
    }
}


