/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1995,1996                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Authors: Paul Taylor and Simon King              */
/*                       Date  : April 1995                              */
/*-----------------------------------------------------------------------*/
/*                     Generate feature vectors                          */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include "EST_speech_class.h"
#include "EST_string_aux.h"
#include "EST_cmd_line.h"
#include "EST_cmd_line_options.h"
#include "sigpr/EST_sig2fv.h"
#include "sigpr/EST_filter.h"

void calculate_orders(EST_StrList &clist, EST_IList &olist,
		      EST_Option &op);

EST_String sigpr_options_supported(void);

int main(int argc, char *argv[])
{
    EST_String out_file("-");
    EST_StrList files;
    EST_Option al, op;
    EST_Wave sig;
    EST_Track full;
    EST_StrList coef_list, delta_list, acc_list, tlist;
    EST_IList olist;

    parse_command_line(argc, argv, 
       EST_String("Usage:   ")+
       "fvgen <input file> -o <output file> <options>\n"+
       "Summary: generate acoustic feature vectors for a waveform file\n"+
       "use \"-\" to make input and output files stdin/out\n"+
       "-h               Options help\n"+
       options_wave_input() + "\n"+
       options_track_output() + "\n"+
       "-L <float>            output frame length\n"+
       "-factor <float>       Frames lengths will be <factor> times the\n"+
       "                       local pitch period \n" +
       "-pm <ifile>           Pitch mark file name\n\n"+
       "-coefs <string>       list of basic coefficients required\n"+
       "-delta <string>       list of delta coefficients required\n"+
       "-acc <string>         list of acceleration (delta delta)\n"+
       "                       coefficients required\n"+
       sigpr_options_supported()+"\n"+
       "-window_type <string> Type of window\n"+
       "-lpc_order <int>      Order of lpc analysis\n"+
       "-cep_order <int>      Order of lpc cepstral analysis\n"+
       "-melcep_order <int>   Order of Mel cepstral analysis\n"+
       "-fbank_order <int>    Order of filter bank analysis\n"+
       "-preemph <float>      Perform pre-emphasis with this factor\n"+
       "-lifter <float>       lifter coefficient\n"+
       "-usepower             use power rather than energy in filter bank analysis\n"+
       "-include_c0           include cepstral coefficient 0\n"+
       "-order <string>       order of analyses\n", files, al);

    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";
    init_lib_ops(al, op);
//    override_lib_ops(al, op);

    StringtoStrList(al.val("-coefs"), coef_list);
    StringtoStrList(al.val("-delta"), delta_list);
    StringtoStrList(al.val("-acc"), acc_list);

    StringtoStrList(al.val("-order"), tlist);
    StrListtoIList(tlist, olist);
    
    if (read_wave(sig, files.first(), al) != read_ok)
	exit(-1);
    
    if (al.present("-pm"))
    {
	if (read_track(full, al.val("-pm"), op))
	    exit(1);
    }
    else
	make_fixed_times(full, sig.end(), op.fval("frame_shift"));

    if (al.present("-preemph"))
	pre_emphasis(sig, al.fval("-preemph"));

    if(al.present("-usepower"))
	cerr << "main : USING POWER" << endl;

    sig2fv(sig, full, op, coef_list, delta_list, acc_list);

    if (full.save(out_file, al.val("-otype", 0)) != write_ok)
    {
	cerr << "sig2fv: failed to write output to \"" << out_file 
	    << "\"" << endl;
	exit(-1);
    }
    return 0;
}

#define EPSILON (0.0001)

#define DEFAULT_FRAME_SIZE 0.01
#define DEFAULT_FRAME_FACTOR 2.0
#define DEFAULT_LPC_ORDER 16
#define DEFAULT_CEP_ORDER 12
#define DEFAULT_FBANK_ORDER 20
#define DEFAULT_MELCEP_ORDER 12
#define DEFAULT_WINDOW "hamming"
#define DEFAULT_PREEMPH 0
#define DEFAULT_LIFTER 0

// sane values for pitchmarks (in seconds)

#define MINIMUM_PITCH_PERIOD (0.0033) // 300 hz
#define MAXIMUM_PITCH_PERIOD (0.02)   // 50 Hz
#define DEFAULT_PITCH_PERIOD (0.01)   // 100 Hz


void calculate_orders(EST_StrList &clist, EST_IList &olist,
		      EST_Option &op)
{
    EST_Litem *c, *o;
    EST_String k;
    int v;

    for (c = clist.head(), o = olist.head(); c && o; c= next(c), o = next(o))
    {
	k = clist(c) + "_order";
	v = olist(o);
	op.override_ival(k, v);
    }
}

void override_lib_ops(EST_Option &op, EST_Option &al)
{ 
    op.override_fval("frame_shift", DEFAULT_FRAME_SIZE);
    op.override_fval("frame_factor", DEFAULT_FRAME_FACTOR);
    op.override_val("window_type", DEFAULT_WINDOW); 

    op.override_fval("preemph", DEFAULT_PREEMPH);
    op.override_fval("lifter", DEFAULT_LIFTER);

    op.override_ival("lpc_order", DEFAULT_LPC_ORDER);
    op.override_ival("cep_order", DEFAULT_CEP_ORDER);
    op.override_ival("fbank_order", DEFAULT_FBANK_ORDER);
    op.override_ival("melcep_order", DEFAULT_MELCEP_ORDER);

    op.override_fval("max_period", MAXIMUM_PITCH_PERIOD);
    op.override_fval("min_period", MINIMUM_PITCH_PERIOD);
    op.override_fval("def_period", DEFAULT_PITCH_PERIOD);
    
    if (al.present("-max_period"))
	op.override_fval("max_period", al.fval("-max_period", 0));
    if (al.present("-min_period"))
	op.override_fval("min_period", al.fval("-min_period", 0));
    if (al.present("-def_period"))
	op.override_fval("def_period", al.fval("-def_period", 0));

    if (al.present("-window_type"))
	op.override_val("window_type", al.sval("-window_type", 1));
    
    if (al.present("-shift"))
	op.override_fval("frame_shift", al.fval("-shift", 1));
    if (al.present("-factor"))
	op.override_fval("frame_factor", al.fval("-factor", 1));
    if (al.present("-length"))
	op.override_fval("frame_factor", 
			 al.fval("-length", 1)/op.fval("frame_shift",1));
    
    if (al.present("-preemph"))
	op.override_fval("preemph", al.fval("-preemph", 1));
    if (al.present("-lifter"))
	op.override_fval("lifter", al.fval("-lifter", 1));

    if (al.present("-lpc_order"))
	op.override_ival("lpc_order", al.ival("-lpc_order", 1));
    if (al.present("-cep_order"))
	op.override_ival("cep_order", al.ival("-cep_order", 1));
    if (al.present("-fbank_order"))
	op.override_ival("fbank_order", al.ival("-fbank_order", 1));
    if (al.present("-melcep_order"))
	op.override_ival("melcep_order", al.ival("-melcep_order", 1));

    if (al.present("-usepower"))
	op.override_val("usepower", al.val("-usepower", 1));

}
