/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1998                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Alan W Black                              */
/*                   Date   :  February 1998                             */
/*-----------------------------------------------------------------------*/
/*             Generalised relations in utterances                       */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include "ling_class/EST_Relation.h"
#include "ling_class/EST_Item.h"

#if defined(INSTANTIATE_TEMPLATES)

#include "../base_class/EST_TList.cc"
#include "../base_class/EST_KV.cc"

template class KVL<EST_String, EST_Relation *>;
template class KVI<EST_String, EST_Relation *>;
template class EST_TList<KVI<EST_String, EST_Relation *> >;
template class EST_TItem<KVI<EST_String, EST_Relation *> >;

template class KVL<EST_String, EST_Item *>;
template class KVI<EST_String, EST_Item *>;
template class EST_TList<KVI<EST_String, EST_Item *> >;
template class EST_TItem<KVI<EST_String, EST_Item *> >;

// There are *far* too many of these around

template class KVL<int, EST_Item *>;
template class KVI<int, EST_Item *>;
template class EST_TList<KVI<int, EST_Item *> >;
template class EST_TItem<KVI<int, EST_Item *> >;

template class KVL<int, EST_Item_Content *>;
template class KVI<int, EST_Item_Content *>;
template class EST_TList<KVI<int, EST_Item_Content *> >;
template class EST_TItem<KVI<int, EST_Item_Content *> >;

template class KVL<void *, int>;
template class KVI<void *, int>;
template class EST_TList<KVI<void *, int> >;
template class EST_TItem<KVI<void *, int> >;

#endif

static EST_read_status load_esps_labels(EST_TokenStream &ts,EST_Relation &rel);

EST_Relation::EST_Relation(const EST_String &name)
{
    p_name = name;
    p_head = 0;
    p_tail = 0;
    p_utt = 0;
}

EST_Relation::EST_Relation()
{
    p_head = 0;
    p_tail = 0;
    p_utt = 0;
}

void EST_Relation::copy(const EST_Relation &r)
{
    // Do a *full* copy include the contents of all the items
    p_name = r.p_name;
    p_utt = 0;  // can't be in the same utterance as r
    clear();
    f = r.f;
    if (r.root() != 0)
    {
	EST_Item i = *r.root();
	EST_Item *to_root = append(&i);
	copy_node_tree_contents(r.root(),to_root);
    }
}

EST_Item *EST_Relation::append(EST_Item *si)
{
    EST_Item *nn;

    if (this == 0)
    {
	cerr << "EST_Relation: no relation to append to" << endl;
	return 0;
    }
    else if (p_tail == 0)
    {
	nn = new EST_Item(this,si);
	p_head = nn;
    }
    else
	nn = p_tail->insert_after(si);
    p_tail = nn;

    return nn;
}

EST_Item *EST_Relation::append()
{
    return append(0);
}

EST_Item *EST_Relation::prepend()
{
    return prepend(0);
}

EST_Item *EST_Relation::prepend(EST_Item *si)
{
    EST_Item *nn;

    if (this == 0)
    {
	cerr << "EST_Relation: no relation to prepend to" << endl;
	return 0;
    }
    else if (p_head == 0)
    {
	nn = new EST_Item(this,si);
	p_tail = nn;
    }
    else
	nn = p_head->insert_before(si);
    p_head = nn;

    return nn;
}

EST_Relation::~EST_Relation()
{
    clear();
}

int EST_Relation::length() const
{
    EST_Item *node;
    int i;

    if (this == 0)
	return 0;
    for (i=0,node=p_head; node; node=next(node))
	i++;
    return i;
}

void EST_Relation::clear()
{
    EST_Item *nn,*nnn;

    for (nn = p_head; nn != 0; nn = nnn)
    {
	nnn = next(nn);
	delete nn;
    }
    p_head = p_tail = 0;
}

void EST_Relation::remove_item(EST_Item *node)
{
    if (p_head == node)
	p_head = next(node);
    if (p_tail == node)
	p_tail = prev(node);
    delete node;
}

void EST_Relation::remove_item_feature(const EST_String &name)
{
    for (EST_Item *s = p_head; s; s = next(s))
	s->f_remove(name);
}


void copy_relation(const EST_Relation &from, EST_Relation &to)
{
    // clone the relation structure from into to, deleting any existing
    // nodes in to.

    to.clear();

    if (from.root() != 0)
    {
	EST_Item *to_root = to.append(from.root());
	copy_node_tree(from.root(),to_root);
    }
}

EST_write_status EST_Relation::save(const EST_String &filename)
{
    EST_Item *ptr;
    EST_Litem *p;
    
    ostream *outf;
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (!(*outf))  
    {
	cerr << "save_label_epsp: can't open label output file \"" << 
	    filename << "\"" << endl;
	return write_fail;
    }
    
    *outf << "separator ;\n";
    *outf << "nfields 1\n";
    for (p = f.head(); p; p = next(p))
	*outf << f.fname(p) << " " << f.fval(p) << endl;

    *outf << "#\n";
/*    if (f("timing_style") == "event")
        *outf << "timing_style event\n";
    else if (f("timing_style") == "unit")
        *outf << "timing_style unit\n";
*/


    
    for (ptr = head(); ptr != 0; ptr = next(ptr))
    {
	*outf << "\t";
	outf->precision(5);
	outf->setf(ios::fixed, ios::floatfield);
	outf->width(8);
	//	outf->fill('0');
	if (f("timing_style") == "event")
	    *outf << ptr->fF("position");
	else
	    *outf << ptr->fF("end");
	
	*outf << " 26 \t" << ptr->fS("name") << " ; ";

	EST_Features f2;
	f2 = ptr->features();
	f2.remove("name");
	f2.remove("end");
	f2.save(*outf);
	*outf << endl;
    }
    
    if (outf != &cout)
	delete outf;
    return write_ok;
}

EST_write_status EST_Relation::save(ostream &outf, 
				    KVL<void *,int> contents) const
{
    KVL<void *,int> nodenames;
    int node_count = 1;
    outf << "Relation " << name() << " ; ";
    f.save(outf);
    outf << endl;
    save_items(p_head,outf,contents,nodenames,node_count);
    outf << "End_of_Relation" << endl;
    return write_ok;
}

EST_write_status EST_Relation::save_items(EST_Item *node, 
					  ostream &outf,
					  KVL<void *,int> &cnames,
					  KVL<void *,int> &nodenames,
					  int &node_count) const
{
    if (node != 0)
    {
	int myname = node_count++;
	// Name it before dealing with the next nodes
	nodenames.add_item(node,myname);
	// This will need to be expanded if the we make Relations
	// have more complex structures
	save_items(next(node),outf,cnames,nodenames,node_count);
	save_items(down(node),outf,cnames,nodenames,node_count);
	outf << myname << " " <<
	   (node->contents() == 0 ? 0 : cnames.val(node->contents())) << " " <<
	   (node->up() == 0 ? 0 : nodenames.val(node->up())) << " " <<
	   (node->down() == 0 ? 0 : nodenames.val(node->down())) << " " <<
	   (node->next() == 0 ? 0 : nodenames.val(node->next())) << " " <<
	   (node->prev() == 0 ? 0 : nodenames.val(node->prev())) << endl;
    }
    return write_ok;
}

EST_read_status EST_Relation::load(EST_TokenStream &ts,
				   KVL<int,EST_Item_Content *> contents)
{
    if (ts.get() != "Relation")
    {
	cerr << "load_relation: " << ts.pos_description() << 
	    " no new Relation" << endl;
	return misc_read_error;
    }
    p_name = ts.get().string();
    if (ts.get() != ";")
    {
	cerr << "load_relation: " << ts.pos_description() << 
	    " semicolon missing after Relation name \"" <<
		p_name << "\"" << endl;
	return misc_read_error;
    }
    if (f.load(ts) != format_ok)
	return misc_read_error;
    if (load_items(ts,contents) != format_ok)
	return misc_read_error;

    return format_ok;
}

EST_read_status EST_Relation::load_items(EST_TokenStream &ts,
					 KVL<int,EST_Item_Content *> contents)
{
    // Load a set of nodes from a TokenStream, the file contains node
    // descriptions one per line as 5 ints, this nodes name, the
    // stream item it is to be related to, then the name of the 
    // nodes above, below, before and after it.
    KVL<int,EST_Item *> nodenames;
    EST_read_status r = format_ok;
    EST_Item_Content *SINil = 0;

    while (ts.peek() != "End_of_Relation")
    {
	EST_Item *node;
	EST_Item_Content *c;
	int name = atoi(ts.get().string());
	int siname;

	node = get_item_from_name(nodenames,name);
	siname = atoi(ts.get().string());
	if (siname != 0) 
	{
	    if ((c = contents.val_def(siname,SINil)) == 0)
	    {
		cerr << "load_nodes: " << ts.pos_description() << 
		    " node's stream item " << siname << " doesn't exist\n";
		r = misc_read_error;
		break;
	    }
	    else
		node->set_contents(c);
	}
	// up down next previous
	node->u = get_item_from_name(nodenames,atoi(ts.get().string()));
	node->d = get_item_from_name(nodenames,atoi(ts.get().string()));
	node->n = get_item_from_name(nodenames,atoi(ts.get().string()));
	node->p = get_item_from_name(nodenames,atoi(ts.get().string()));
    }

    ts.get(); // skip End_of_Relation

    if (r == format_ok)
    {
	if (nodenames.length() > 0)
	    p_head = get_item_from_name(nodenames,1);
	p_tail = p_head->last();
	if (!p_head->verify())
	{
	    cerr << "load_nodes: " << ts.pos_description() <<
		" nodes do not form consistent graph" << endl;
	    r = misc_read_error;
	}
    }

    if (r != format_ok)
    {
	// failed to read this relation so clear the created nodes
	// before returning, no idea what state the links are in so
	// explicitly unlink them before deleting them
	for (EST_Litem *p=nodenames.list.head(); p != 0; p=next(p))
	{
	    EST_Item *node = nodenames.list(p).v;
	    node->u = 0;
	    node->d = 0;
	    node->n = 0;
	    node->p = 0;
	    delete node;
	}
    }
    return r;
}    

EST_Item *EST_Relation::get_item_from_name(KVL<int,EST_Item *> &nodenames,
					   int name)
{
    // Return node named by name or create a new one if it doesn't
    // already exist
    EST_Item *NodeNil = 0;  // cause apparently I *need* a variable
    EST_Item *node;

    if (name == 0)
	return 0;
    else if ((node = nodenames.val_def(name,NodeNil)) == 0)
    {
	node = new EST_Item(this,0);
	nodenames.add_item(name,node);
    }
    return node;
}

EST_read_status EST_Relation::load(const EST_String &filename,
				   const EST_String &type)
{
    // Load an isolated relation from a file, assuming Xlabel format
    EST_TokenStream ts;

    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "load_relation: can't open relation input file " 
	    << filename << endl;
	return misc_read_error;
    }
    f.set("filename",filename);

    if (type == "esps")
	return load_esps_labels(ts,*this);
    else   // current esps is the default
	return load_esps_labels(ts,*this);
}

static EST_read_status load_esps_labels(EST_TokenStream &ts,EST_Relation &rel)
{
    ts.set_SingleCharSymbols(";");
    ts.set_quotes('"','\\');
    EST_String key, val;

    // Skip the header
    while (!ts.eof())
    {
	key = ts.get().string();
        if (key == "#")
            break;

	val = ts.get_upto_eoln().string();
	val.gsub(RXwhite, "");
//	cout << "key: " << key << endl;
//	cout << "val: " << val << endl;
    
	rel.f.set(key, val);
    }
//    cout << "rel f: " << rel.f << endl;
//    cout << "rel f end" << endl;
	    
    if (ts.eof()) return misc_read_error;
//    ts.get();  // skip "#"

    while (!ts.eof())
    {
	EST_Item *si = rel.append();
	EST_String name;
	
	si->fset("end",(float)atof(ts.get().string()));
	ts.get();  // skip the color;

	for (name = ""; (!ts.eoln()) && (ts.peek() != ";"); )
	{
	    EST_Token &t = ts.get();
	    if (name.length() > 0)	// preserve internal whitespace
		name += t.whitespace();  
	    name += t.string();
	}
	si->set_name(name);
//	cout << "name: " << *si << endl;
	
	if (ts.peek().string() == ";") // absorb separator
	{
	    ts.get();
	    si->features().load(ts);
	}
    }
    return format_ok;
}

int EST_Relation::num_leafs() const
{
    int count = 0;
    EST_Item *n;

    for (n=head()->first_leaf(); n != 0; n=n->next_leaf())
	count++;
    return count;
}

EST_Item *EST_Relation::first_leaf() const 
{ 
    return head()->first_leaf();
}

EST_Item *EST_Relation::last_leaf() const 
{ 
    return head()->last_leaf();
}

EST_Utterance *get_utt(EST_Item *s)
{
    // Occasionally you need to get the utterance from a stream_item
    // This finds any relations in s and follows them to the utterance
    // If there aren't any Relations the this streamitem isn't in an
    // utterance
    if (s == 0)
	return 0;
    else if (s->relation() == 0)
	return 0;
    else
	return s->relation()->utt();
}

EST_Relation &EST_Relation::operator=(const EST_Relation &s)
{
    copy(s);
    return *this;
}

ostream& operator << (ostream &s, const EST_Relation &a)
{
    s << a.f << endl;

    for (EST_Item *p = a.head(); p; p = next(p))
	s << *p << endl;

    return s;
}


