/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1995,1996                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                                                                       */
/*                   Author :  Paul Taylor                               */
/* -------------------------------------------------------------------   */
/*                  EST_Date Class header file                           */
/*                                                                       */
/*************************************************************************/

#ifndef __EST_FeatureData_H__
#define __EST_FeatureData_H__

#include "EST_Val.h"
#include "EST_TVector.h"
#include "EST_TList.h"
#include "EST_rw_status.h"
#include "EST_types.h"

typedef EST_TMatrix<EST_Val> EST_ValMatrix;
typedef EST_TVector<EST_Val> EST_ValVector;

/** A class for containing arbitrary multi-dimensional feature data.

A number of fields are defined in the EST_FeatureData class, each of
which represents a measurable quantity, such as height, age or
gender. Any number of fields can be defined and each field can take a
float, integer or string value. The class holds mutiple instances of
field values, representing samples taken from a population. 

Several statistical training, testing and analysis programs use
EST_FeatureData as input and output. Member functions exist for
comparing fields, extracting given named or numbered fields, can 
converting appropriate fields to a EST_FMatrix.

*/

class EST_FeatureInfo {
private:
    int p_num_fields;
    void default_vals();

    EST_StrVector p_field_name;
    EST_StrVector p_field_type;
    EST_StrListVector p_field_values;

    EST_FeatureInfo &copy(const EST_FeatureInfo &a);
public:
    EST_FeatureInfo();
    EST_FeatureInfo(const EST_FeatureInfo &a);

    EST_StrI_KVL group_start;
    EST_StrI_KVL group_end;

    void clear();
    /**@name Setting field information */
    //@{
    /** set field names, given as a space separated string, and
    set number of fields to length of list. */
    void set_field_names(const EST_String &names);

    /** set field names, given as a EST_StrList, and
    set number of fields to length of list. */
    void set_field_names(const EST_StrList &names);

    /** set field type, given as a space separated string, using
    strings "int", "float" and "string" */
    void set_field_types(const EST_String &names);

    /** set field type, given as a EST_StrList, using
    strings "int", "float" and "string" */
    void set_field_types(const EST_StrList &names);

    void set_field_values(int n, const EST_String &values);

    /** define a list of possible field values for fields which take
    a string. field {\bf name} can take values give in {\bf values}. */
    void set_field_values(const EST_String &name, const EST_String &values);

    /** define a list of possible field values for fields which take
    a string. field number {\bf n} can take values give in {\bf values}. */
    void set_field_values(int n, const EST_StrList &values);
    
    /** define a list of possible field values for fields which take
    a string. field number {\bf n} can take values give in {\bf values}. */
    void set_field_values(const EST_String &name, const EST_StrList &values);
    
    //@}
    void extract_named_fields(EST_FeatureInfo &f, const EST_StrList &n_fields) const;
    /**@name Information functions */
    //@{

    int num_fields() const {return p_num_fields;}

    int field_index(const EST_String &name) const;

    EST_String field_name(int i) const;
    const EST_StrVector &field_names() const;

    EST_String field_type(int i) const;
    EST_String field_type(const EST_String &name) const;
    const EST_StrVector &field_types() const;

    const EST_StrList &field_values(int i) const;
    const EST_StrList &field_values(const EST_String &name) const;

    //@}


    EST_FeatureInfo &operator=(const EST_FeatureInfo &f);

    friend ostream& operator << (ostream &st,const EST_FeatureInfo &a);
};


class EST_FeatureSample : public EST_ValVector {
private:
    bool p_sub_fd;

    void default_vals();
    void free_internals();
    void alloc_internals();

    EST_FeatureInfo *p_info;
    EST_FeatureSample &copy(const EST_FeatureSample &a);
public:
    EST_FeatureSample();
    EST_FeatureSample(const EST_FeatureSample &a);

    EST_FeatureInfo &info() {return *p_info;}
    const EST_FeatureInfo &info() const {return *p_info;}

    /**@name Information functions */
    //@{

    /** set number of samples to be held in object and allocate
    space for storing them */

    int num_fields() const {return info().num_fields();}

    //@}

    /**@name Copy Conversion functions */
    //@{

    /// convert to matrix of floats

/*    void extract_named_fields(EST_FeatureSample &f, const EST_String &fields)
	const;
    void extract_named_fields(EST_FeatureSample &f, const EST_StrList &fields) const;

    void extract_numbered_fields(EST_FeatureSample &f, 
				 const EST_String &fields) const;
    void extract_numbered_fields(EST_FeatureSample &f,
				 const EST_IList &fields) const;

    void extract_named_fields(const EST_String &fields);
    void extract_named_fields(const EST_StrList &fields);

    void extract_numbered_fields(const EST_String &fields);
    void extract_numbered_fields(const EST_IList &fields);
*/

    //@}

    EST_Val &a(int field)
    {return EST_ValVector::a(field);}

    const EST_Val &a(int field) const
    {return EST_ValVector::a(field);}

    EST_Val &a(const EST_String &name)
    {return EST_ValVector::a(info().field_index(name));}

    const EST_Val &a(const EST_String &name) const
    {return EST_ValVector::a(info().field_index(name));}

    /// const element access operator
//      const EST_Val &operator () (int sample, const EST_String &field);
    /// non-const element access operator
      EST_Val &operator () (const EST_String &field);
      EST_Val &operator () (int field);

    EST_FeatureSample &EST_FeatureSample::operator=
	(const EST_FeatureSample &f);

  friend ostream& operator << (ostream &st, const EST_FeatureSample &a);
  //@}


};

class EST_FeatureData : public EST_ValMatrix {
private:
    bool p_sub_fd;

    void default_vals();
    void free_internals();
    void alloc_internals();

    EST_FeatureInfo *p_info;
    EST_FeatureData &copy(const EST_FeatureData &a);
public:
    EST_FeatureData();
    EST_FeatureData(const EST_FeatureData &a);

    EST_FeatureInfo &info() {return *p_info;}
    const EST_FeatureInfo &info() const {return *p_info;}

    /**@name Information functions */
    //@{

    /** set number of samples to be held in object and allocate
    space for storing them */
    void set_num_samples(int n);
    int num_samples() const {return num_rows();}

    int num_fields() const {return info().num_fields();}

    //@}

    /**@name Copy Conversion functions */
    //@{

    /// convert to matrix of floats
    void float_matrix(EST_FMatrix &a, int inv = 0) const;



    void extract_named_fields(EST_FeatureData &f, const EST_String &fields)
	const;
    void extract_named_fields(EST_FeatureData &f, const EST_StrList &fields) const;

    void extract_numbered_fields(EST_FeatureData &f, 
				 const EST_String &fields) const;
    void extract_numbered_fields(EST_FeatureData &f,
				 const EST_IList &fields) const;

    void extract_named_fields(const EST_String &fields);
    void extract_named_fields(const EST_StrList &fields);

    void extract_numbered_fields(const EST_String &fields);
    void extract_numbered_fields(const EST_IList &fields);

    //@}

    /**@name Sub FeatureData functions */
    //@{

    void sub_samples(EST_FeatureData &f, int start, int num=1);
//    void sample(EST_FeatureSample &f, int start);

    //@}

    EST_read_status load(const EST_String &name);

    EST_write_status save(const EST_String &name, 
			const EST_String &EST_filetype = "") const;

    EST_Val &a(int sample, int field)
    {return EST_ValMatrix::a(sample, field);}

    const EST_Val &a(int sample, int field) const
    {return EST_ValMatrix::a(sample, field);}

    EST_Val &a(int sample, const EST_String &name);
    const EST_Val &a(int sample, const EST_String &name) const;


    /// const element access operator
//      const EST_Val &operator () (int sample, const EST_String &field);
    /// non-const element access operator
      EST_Val &operator () (int sample, const EST_String &field);
      EST_Val &operator () (int sample, int field);

    EST_FeatureData &EST_FeatureData::operator=(const EST_String &f);

  friend ostream& operator << (ostream &st,const EST_FeatureData &a);
  //@}


};


#endif /* __EST_FeatureData_H__ */
