/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1998                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Alan W Black                           */
/*                      Date   :  March 1998                             */
/*-----------------------------------------------------------------------*/
/*                  Feature value pairs in a KVL                         */
/*=======================================================================*/

#include <stdlib.h>
#include "EST_Features.h"
#include "EST_error.h"

EST_Val EST_Features::feature_default_value("0");
// Sometime we need a set of features even though there aren't any
static KVL<EST_String, EST_Val> default_features;

EST_Features::EST_Features()
{
    features = new KVL<EST_String, EST_Val>;
}

EST_Features::EST_Features(const EST_Features &f)
{
    features = new KVL<EST_String, EST_Val>;
    *features = *f.features;
}

EST_Features::~EST_Features()
{
    delete features;
}

const EST_Val &EST_Features::val(const char *name, bool m) const
{
    // Because so many access are from char* literals we all access
    // directly rather than requiring the creation of an EST_String
    EST_Litem *p;

    for (p=features->list.head(); p; p=next(p))
    {
	if (features->list(p).k == name)
	    return features->list(p).v;
    }
    if (m)
	EST_error("Feature %s not defined\n", name);

    return feature_default_value;
}

EST_write_status EST_Features::save(ostream &outf) const
{
    // Save features
    for (EST_Litem *p=features->list.head(); p != 0; p=next(p) )
    {
	outf << features->list(p).k << " ";
	if (features->list(p).v == ";")
	    outf << "\";\"";
	else if ((features->list(p).v.type() == val_string) &&
		 ((features->list(p).v.string().matches(RXint)) ||
		  (features->list(p).v.string().matches(RXdouble))))
	    // force quoting, cause it looks like a number but isn't
	    outf << quote_string(features->list(p).v,"\"","\\",1);
	else if (features->list(p).v.type() == val_float)
	{
	    char b[20];
	    sprintf(b,"%g",features->list(p).v.Float());
	    outf << b;
	}
	else
	    outf << quote_string(features->list(p).v);
	outf << " ; ";
    }

    return write_ok;
}

EST_read_status EST_Features::load(EST_TokenStream &ts)
{
    // load features from here to end of line separated by semicolons
    EST_String f;
    EST_Token v;
    
    while (!ts.eoln())
    {
	if (ts.eof())
	{
	    cerr << "load_features: unexpected end of file\n";
	    return misc_read_error;
	}
	f = ts.get().string();
	v = ts.get();
	while ((ts.peek() != ";") && (!ts.eof()) && (!ts.eoln()))
	    v = v + " " + ts.get().string();
	if (ts.get() != ";")
	{
	    cerr << "load_features: " << ts.pos_description() <<
		" missing semicolon in feature list\n";
	    return misc_read_error;
	}
	if (v.quoted())
	    features->add_item(f,EST_Val(v.string()));
	else if (v == "<contents>")  // unsupported type
	    continue;
	else if (v.string().matches(RXint))
	    features->add_item(f,EST_Val(atoi(v.string())));
	else if (v.string().matches(RXdouble))
	    features->add_item(f,EST_Val(atof(v.string())));
	else
	    features->add_item(f,EST_Val(v.string()));
    }
    return format_ok;
}

EST_Features &EST_Features::operator=(const EST_Features &x)
{
    *features = *x.features;
    return *this;
}

void merge_features(EST_Features &to,EST_Features &from)
{
    for (EST_Litem *p=from.head(); p != 0; p=next(p) )
	to.set_val(from.fname(p),from.fval(p));
}



