/***********************************************************          
 *   	              FINAL PROJECT     		   *
 *							   *   
 *           course: 216-605A				   *     
 *	     Digital Sound Synthesis & Audio Processing    *
 *                                                         *
 *                                                         *
 *			hrtferxk.c			   *
 * 			----------                         *
 *                                                         *
 *	Eli Breder (9111216) and David McIntyre (9005614)  *
 *                                                         *   
 * 	Dec. 17, 1995                                      *
 *                                                         *
 ***********************************************************/


/***************************************************************
 * This version of hrtfer loads the file HRTFcompact into memory.
 * Offsets into the file are calculated in the a-rate code to 
 * get the requested HRTF measurements. We've implemented
 * a linear crossfade to deal with the clicks which occur
 * when the input audio is convolved with new HRTFs. Although this
 * the clicking, it does not eliminate them. 
 * A better solution would be to implement interpolation between
 * the old and new HRTFs (probably a project in itself). 
 ***************************************************************/


#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include "cs.h"
#include "hrtferx.h"

/* This array transferred here so as to be declared once.  Belongs to
   the structure of the HRTF data really in 3Dug.h */

int elevation_data[N_ELEV] = {56, 60, 72, 72, 72, 72, 72, 60, 56, 45, 36, 24,
12, 1 };

#define ROUND(x) ((int)floor((x)+0.5))
#define GET_NFAZ(el_index)	((elevation_data[el_index] / 2) + 1)

void cfft (float [], int, int);
void bitreverse (float [], int);
void rfft (float [], int, int);
void cmult(float [], float [], float [], int);
extern int bytrevhost(void);

void hrtferxkSet(HRTFER *p)
{		
    int	   i; /* standard loop counter */
    /*	   input,      output left, output right, overlap left, overlap right */
    float  x[BUF_LEN], yl[BUF_LEN], yr[BUF_LEN], bl[FILT_LENm1], br[FILT_LENm1];
    /*     output left,   output right */
    float  outl[BUF_LEN], outr[BUF_LEN];
    float  rampup[128], rampdown[128];	  
    char   filename[MAXNAME];
    MEMFIL *mfp;
	
	/* first check if orchestra's sampling rate is compatible with HRTF
	   measurement's */
    if (esr != SAMP_RATE) {
	  printf ("Orchestra sampling rate is not compatible with HRTF.\n");
	  printf ("Should be %d...exiting.\n", SAMP_RATE);
	  exit(0);
    }  
		
    if (*p->ifilno == sstrcod)
      strcpy(filename, p->STRARG);
    else {
      printf("\nLast argument must be the string 'HRTFcompact' "
	     "...correcting.\n");
      strcpy(filename, "HRTFcompact");
      exit(0);
    }

    if ((mfp = p->mfp) == NULL)
      mfp = ldmemfile(filename);
    p->mfp = mfp;
    p->fpbegin = (short *) mfp->beginp;	
    if (bytrevhost()) {		/* Byte reverse on data set in necessary */
      short *x = p->fpbegin;
      long len = (mfp->length)/sizeof(short);
      while (len!=0) {
	short v = *x;
	v = ((v&0xFF)<<8) + ((v>>8)&0xFF); /* Swap bytes */
	*x = v;
	x++; len--;
      }
    }	
	/* initialize ramp arrays for crossfading */
    for (i=0; i<128; i++) {
        rampup[i] = (float)i / (float) FILT_LEN;
	rampdown[i] = 1.0f - rampup[i];
	p->rampup[i] = rampup[i];
	p->rampdown[i] = rampdown[i];
    }
	
	/* initialize counters and indices */
    p->outcount = 0;   
    p->incount = 0;
    p->outfront = p->outend = 0; /* working indices for circ output buffer */	
	
	/* initialize oldhrtf_data with zeros */
    for (i=0; i<BUF_LEN; i++) 	{
      p->oldhrtf_data.left[i] = 0.0f;
      p->oldhrtf_data.right[i] = 0.0f;
    }
	
	/* initialize input buffer */
    for (i=0; i<BUF_LEN; i++) {
      x[i] = 0.0f;	
      p->x[i] = x[i];
    }
	
	/* initialize left result buffer */
    for (i=0; i<BUF_LEN; i++) {
      yl[i] = 0.0f;	
      p->yl[i] = yl[i];
    }
	
	/* initialize right result buffer */
    for (i=0; i<BUF_LEN; i++) {
      yr[i] = 0.0f;	
      p->yr[i] = yr[i];
    }
	
	/* initialize left output buffer */
    for (i=0; i<BUF_LEN; i++) {
      outl[i] = 0.0f;	
      p->outl[i] = outl[i];
    }	
	
	/* initialize right output buffer */
    for (i=0; i<BUF_LEN; i++) {
      outr[i] = 0.0f;	
      p->outr[i] = outr[i];
    }
	
	/* initialize left overlap buffer */
    for (i=0; i<FILT_LENm1; i++) {
      bl[i] = 0.0f;	
      p->bl[i] = bl[i];
    }
	
	/* initialize right overlap buffer */
    for (i=0; i<FILT_LENm1; i++) {
      br[i] = 0.0f;	
      p->br[i] = br[i];
    }

    return;	
}

/********************** a-rate code ***********************************/

void hrtferxk(HRTFER *p)
{
    float      *aLeft, *aRight; /* audio output streams */
    float      *aIn, *kAz, *kElev; /* audio and control input streams */
    int	       azim, elev, el_index, az_index,oldel_index, oldaz_index;
    int	       nsmpsi, nsmpso; /* number of samples in/out */
    /*	       input,      out-left,    out-right */
    float      x[BUF_LEN], yl[BUF_LEN], yr[BUF_LEN];\
    /*         overlap left,   overlap right */
    float      bl[FILT_LENm1], br[BUF_LEN];
			/* copy of current input convolved with old HRTFs */
    float      yl2[BUF_LEN], yr2[BUF_LEN];
    float      outl[BUF_LEN], outr[BUF_LEN]; /* output left/right */ 
    int	       outfront, outend; /* circular output indices */
    int	       incount, outcount; /* number of samples in/out */
    int	       toread; /* number of samples to read */
    int	       i; /* standard loop counter */
    HRTF_DATUM hrtf_data,  oldhrtf_data; /* local hrtf instances */
    int	       flip; /* flag - true if we need to flip the channels */
    int	       crossfadeflag; /* flag - true if crossfading old and new HRTFs */
    short      *fpindex; /* pointer into HRTF file */
    short      numskip; /* number of shorts to skip in HRTF file */
			/* short arrays into which HRTFs are stored locally */
    short      sl[FILT_LEN], sr[FILT_LEN];
			/* float versions of above to be sent to FFT routines */
    float      xl[BUF_LEN], xr[BUF_LEN];
	
	
    if (p->mfp==NULL) {         /* RWD fix */
      initerror("hrtfer: not initialized");
      return;
    }
	/* update local variables */
    kElev = p->kElev;
    kAz = p->kAz;	
    elev = (int) *kElev;	
    azim = (int) *kAz;
    oldel_index = p->oldel_index;
    oldaz_index = p->oldaz_index;
    fpindex = (short *) p->fpbegin;
    flip = 0;	
    crossfadeflag = 0;
	
		
	/* Convert elevation in degrees to elevation array index. */
    el_index = ROUND((elev - MIN_ELEV) / ELEV_INC);
    if (el_index < 0)
      el_index = 0;
    else if (el_index >= N_ELEV) 
      el_index = N_ELEV-1;
	
	/* Convert azimuth in degrees to azimuth array index. */	
    azim = (int)fmod(azim, 360.0);
    if (azim < 0) 
      azim += 360;
    if (azim > 180) {
      azim = 360 - azim;
      flip = 1; /* set to true */
    }
    else
      flip = 0;  /* it is still false */
	
    	/* azim should be 0<=azim<=180 so calculate az_index and clip
	   to legal range
	   note - accesses global array elevation_data */
    az_index = ROUND(azim / (360.0 / elevation_data[el_index]));
    if (az_index < 0)
      az_index = 0;
    else if (az_index >= elevation_data[el_index])
      az_index = elevation_data[el_index] - 1;

	/* calculate offset into HRTFcompact file */
	/* first get to the first value of the requested elevation */
    if (el_index == 0)
      fpindex = (short *) p->fpbegin; 
    else
      for (i=0; i<=el_index; i++) { 
	numskip = 0;
	numskip += (short)(GET_NFAZ(i) * BUF_LEN);
	fpindex += (short) numskip;	
      }	
	/* fpindex should now point to first azimuth at requested el_index */
	/* now get to first value of requested azimuth */
    if (az_index == 0) {
				/*printf("in az_index == 0\n")*/
      numskip = 0;
    }
    else {	
      for (i=0, numskip=0; i<az_index; i++)
	numskip += BUF_LEN;
      fpindex += (short) (numskip); 
    }
		
    if ((oldel_index != el_index) || (oldaz_index != az_index))
      crossfadeflag = 1;
	
	/* read in (short) data from stereo interleave HRTF file.
	   Split into left and right channel data. */
    for (i=0; i<FILT_LEN; i++) {
      sl[i] = *fpindex++; 
      sr[i] = *fpindex++;
/*       printf("sl[%d] = %d, sr[%d] = %d\n", i, sl[i], i, sr[i]); */
    }
    for (i=0; i<FILT_LEN; i++) {
      xl[i] = (float)(sl[i]/32768.); /* copy short buffers into float buffers */
      xr[i] = (float)(sr[i]/32768.);	   
    }
    for (i=FILT_LEN; i<BUF_LEN; i++) {
      xl[i] = 0.0f;     /* pad buffers with zeros to BUF_LEN */
      xr[i] = 0.0f;
    }
    
	/**************
	FFT xl and xr here 
	***************/	
    rfft (xl, FILT_LEN, 1); 		   		 
    rfft (xr, FILT_LEN, 1);

	/* If azimuth called for right side of head, use left side
	   measurements and flip output channels.
	   This is due to the fact that the HRTFs we are using only
	   include left side placements. */
    if (flip) {
      for (i=0; i<BUF_LEN; i++) {
	hrtf_data.left[i] = xr[i];
	hrtf_data.right[i] = xl[i];
      }
    }
    else {
      for (i=0; i<BUF_LEN; i++) {
	hrtf_data.left[i] = xl[i];
	hrtf_data.right[i] = xr[i];
      }
    }	    	
	
	/* update local counters and indices */	
    incount = p->incount;
    outcount = p->outcount;
    outfront = p->outfront;
    outend = p->outend;
	
    for (i=0; i<FILT_LEN; i++) {	 
      oldhrtf_data.left[i] = p->oldhrtf_data.left[i];
      oldhrtf_data.right[i] = p->oldhrtf_data.right[i];
    }
		
    for (i=0; i<BUF_LEN; i++) {
      outl[i] = p->outl[i];
      outr[i] = p->outr[i];
    }
	
    for (i=0; i<FILT_LENm1; i++) {
      bl[i] = p->bl[i];
      br[i] = p->br[i];
    }
	
    for (i=0; i<BUF_LEN; i++)
	  x[i] = p->x[i];
	
    for (i=0; i<BUF_LEN; i++) {
      yl[i] = p->yl[i];
      yr[i] = p->yr[i];
    }
    
    aIn = p->aIn;
    aLeft = p->aLeft;
    aRight = p->aRight;
	
    nsmpsi = ksmps;
    nsmpso = ksmps;
	
	/* main loop for a-rate code.  Audio read in, processed,
	   and output in this loop.  Loop exits when control period
	   (ksmps) is finished.  */
    while (nsmpsi > 0) {
		/* determine how much audio may be read in */
      if ((incount + nsmpsi) <= FILT_LEN)
	toread = nsmpsi;
      else
	toread = FILT_LEN - incount;
      
		/* reading in audio into x */
      if (incount == 0) {
	for (i = 0; i < toread; i++)
	  x[i] = *aIn++;
      }
      else {
	for (i = incount; i<(incount + toread); i++)
	  x[i] = *aIn++;
      }

	  /* update counters for amount of audio read */
      nsmpsi -= toread;
      incount += toread;	
		
	  /* loop for audio processing */
      if (incount == FILT_LEN) { 
	      /* enough audio for convolution - so do it! */
	incount = 0;
			/* pad x to BUF_LEN with zeros for Moore FFT */
	for (i = FILT_LEN; i <  BUF_LEN; i++)
	  x[i] = 0.0f;
	rfft(x, FILT_LEN, 1);

	      /* complex multiplication, y = hrtf_data * x */
	cmult(yl, hrtf_data.left, x, BUF_LEN);
	cmult(yr, hrtf_data.right, x, BUF_LEN);
			
	      /* convolution is the inverse FFT of above result (yl,yr) */
	rfft(yl, FILT_LEN, 0);
	rfft(yr, FILT_LEN, 0);		
			
	if (crossfadeflag) {	/* convolve current input with old HRTFs */
		  /* complex multiplication, y2 = oldhrtf_data * x */
	  cmult(yl2, oldhrtf_data.left, x, BUF_LEN);
	  cmult(yr2, oldhrtf_data.right, x, BUF_LEN);
				
		  /* convolution is the inverse FFT of above result (y) */
	  rfft(yl2, FILT_LEN, 0);
	  rfft(yr2, FILT_LEN, 0);
				
		  /* linear crossfade */
	  for (i=0; i<FILT_LEN; i++) {
	    yl[i] = yl[i]*p->rampup[i] + yl2[i]*p->rampdown[i]; 
	    yr[i] = yr[i]*p->rampup[i] + yr2[i]*p->rampdown[i];
	  }					
	}
			
	      /* overlap-add the results */
	for (i = 0; i < FILT_LENm1; i++) {
	  yl[i] += bl[i];
	  yr[i] += br[i]; 
	  bl[i] = yl[FILT_LEN+i];
	  br[i] = yr[FILT_LEN+i];
	}
						
	      /* put convolution ouput into circular output buffer */
	if (outend <= FILT_LEN) {	
		  /* output will fit in buffer boundaries, therefore 
		     no circular reference required */	
	  for (i = outend; i < (outend + FILT_LEN); i++) {
	    outl[i] = yl[i-outend];
	    outr[i] = yr[i-outend];
	  }
	  outcount += FILT_LEN;
	  outend += FILT_LEN;
	}
	else {
		  /* circular reference required due to buffer boundaries */
	  for (i = outend; i < BUF_LEN; i++) {
	    outl[i] = yl[i-outend];
	    outr[i] = yr[i-outend];
	  }
	  for (i = 0; i < (-FILT_LEN + outend); i++) {
	    outl[i] = yl[(BUF_LEN-outend) + i];
	    outr[i] = yr[(BUF_LEN-outend) + i];
	  }
	  outcount += FILT_LEN;
	  outend += -FILT_LEN;
	}				
      }
		
	  /* output audio to audio stream.
		  Can only output one control period (ksmps) worth of samples */
      if (nsmpso < outcount) {
	if ((outfront+nsmpso) < BUF_LEN) {
	  for (i = 0; i < nsmpso; i++) {
	    *aLeft++ = outl[outfront + i];
	    *aRight++ = outr[outfront + i];
	  }
	  outcount -= nsmpso;
	  outfront += nsmpso;
	  nsmpso = 0;
	}
	else {
		  /* account for circular reference */
	  for (i = outfront; i < BUF_LEN; i++) {
	    *aLeft++ =  outl[i];
	    *aRight++ =  outr[i];
	  }
	  outcount -= nsmpso;
	  nsmpso -= (BUF_LEN - outfront);
	  for (i = 0; i < nsmpso; i++) {
	    *aLeft++ =  outl[i];
	    *aRight++ =  outr[i];
	  }
	  outfront = nsmpso;
	  nsmpso = 0;
	}
      }
      else {
	if ((outfront+nsmpso) < BUF_LEN) {
	  for (i = 0; i < outcount; i++) {
	    *aLeft++ =  outl[outfront + i];
	    *aRight++ =  outr[outfront + i];
	  }
	  nsmpso -= outcount;
	  outfront += outcount;
	  outcount = 0;
	}
	else {
	  /* account for circular reference */
	  for (i = outfront; i < BUF_LEN; i++) {
	    *aLeft++ =  outl[i];
	    *aRight++ =  outr[i];
	  }
	  nsmpso -= outcount;
	  outcount -= (BUF_LEN - outfront);
	  for (i = 0; i < outcount; i++)
	    {
	      *aLeft++ =  outl[i];
	      *aRight++ =  outr[i];
	    }
	  outfront = outcount;
	  outcount = 0;
	} 
      } /* end of audio processing loop - "if" */
    } /* end of control period loop - "while" */
	
	
	/* update state in p */
    p->outcount = outcount;
    p->incount = incount;
    p->outfront = outfront;
    p->outend = outend;
    p->oldel_index = el_index;
    p->oldaz_index = az_index;
	
    for (i=0; i<FILT_LEN; i++) {	/* archive current HRTFs */
      p->oldhrtf_data.left[i] = hrtf_data.left[i];
      p->oldhrtf_data.right[i] = hrtf_data.right[i];
    }
	
    for (i=0; i<BUF_LEN; i++) {
      p->outl[i] = outl[i];
      p->outr[i] = outr[i];
    }
	
    for (i=0; i<FILT_LENm1; i++) {
      p->bl[i] = bl[i];
      p->br[i] = br[i];
    }
	
    for (i=0; i<BUF_LEN; i++)
      p->x[i] = x[i];
	
    for (i=0; i<BUF_LEN; i++) {
      p->yl[i] = yl[i];
      p->yr[i] = yr[i];
    }	
}


/*********************** FFT functions *****************
 * The functions cfft, rfft, and cmult are taken from the
 * code in Elements of Computer Music by F. R. Moore, pp. 81-88
 *******************************************************/

/********************cfft**********************************/
/*
 * cfft replaces float array x containing NC complex values
 * (2*NC float alternating real, imaginary, and so on)
 * bi its Forrier transform if forward id true, or by its
 * inverse Fourrier transform if forward is false. NC must be
 * a power of 2.
 */
 
void cfft (float x[], int NC, int forward)
{
	float TWOPI = (float)(8.0*atan(1.0));	
	float wr, wi, wpr, wpi, theta, scale;
	int mmax, ND, m, i, j, delta;
	
	ND = NC<<1;
	bitreverse (x, ND);
	for (mmax = 2; mmax < ND; mmax = delta) {
		delta = mmax<<1;
		theta = TWOPI/(forward ? mmax : -mmax);
		wpr = -2.0f*(float)pow(sin(0.5*theta), 2.0);
		wpi = (float)sin(theta);
		wr = 1.0f;
		wi = 0.0f;
		for (m = 0; m < mmax; m += 2) {
			float rtemp, itemp;
			for (i = m; i < ND; i += delta) {
				j = i + mmax;
				rtemp = wr*x[j] - wi*x[j+1];
				itemp = wr*x[j+1] + wi*x[j];
				x[j] = x[i] - rtemp;
				x[j+1] = x[i+1] - itemp;
				x[i] += rtemp;
				x[i+1] += itemp;
			}
			wr = (rtemp = wr)*wpr - wi*wpi + wr;
			wi = wi*wpr + rtemp*wpi + wi;
		}
	}
	/* scale the output */
	scale = forward ? 1.0f/ND : 2.0f;
	for (i = 0; i < ND; i++)
		x[i] *= scale;
} 				


/* bitreverse places float array x containing N/2 complex values into
   bit-reversed order */
void bitreverse (float x[], int N)
{
	float rtemp, itemp;
	int i, j, m;
	
	for (i = j = 0; i < N; i += 2, j += m) {
		if (j>i) {
			rtemp = x[j]; itemp = x[j+1]; /* complex exchange */
			x[j] = x[i]; x[j+1] = x[i+1];
			x[i] = rtemp; x[i+1] = itemp;
			}
		for (m = N>>1; m >= 2 && j >= m; m >>= 1)
			   j -= m;
	}
}

/********************rfft**********************************/
/* If forward is true, rfft replaces 2*N real data points
   with N complex values representing the positive frequency half
   of their Fourrier spectrum, with x[1] replaced with the real
   part of the Nyquist frequency values. If forward is false, rfft
   expects x to contain a * posistive frequency spectrum arranged
   as before, and replaces it with 2*N real values. N must be a power 
   of 2.
*/

void rfft (float x[], int N, int forward)
{
	float PIE = (float)(4.0*atan(1.0));	
	float c1, c2, h1r, h1i, h2r, h2i, wr, wi, wpr, wpi, temp, theta;
	float xr, xi;
	int i,i1, i2, i3, i4, N2p1;
	
	theta = PIE/N;
	wr = 1.0f;
	wi = 0.0f;
	c1 = 0.5f;
	if (forward) {
		c2 = -0.5f;
		cfft (x,N,forward);
		xr = x[0];
		xi = x[1];
	}
	else {
		 c2 = 0.5f;
		 theta = -theta;
		 xr = x[1];
		 xi = 0.0f;
		 x[1] = 0.0f;
	}
	wpr = -2.0f*(float)pow(sin (0.5*theta), 2.0);
	wpi = (float)sin (theta);
	N2p1 = (N<<1) + 1;
	for (i = 0; i <= N>>1; i++) {
		i1 = i<<1;
		i2 = i1 + 1;
		i3 = N2p1 - i2;
		i4 = i3 + 1;
		if (i==0) {
			h1r = c1*(x[i1] + xr);
			h1i = c1*(x[i2] - xi);
			h2r = -c2*(x[i2] + xi);
			h2i = c2*(x[i1] - xr);
			x[i1] = h1r + wr*h2r - wi*h2i;
			x[i2] = h1i + wr*h2i + wi*h2r;
			xr = h1r - wr*h2r + wi*h2i;
			xi = -h1i + wr*h2i + wi*h2r;
		}
		else {
			h1r = c1*(x[i1] + x[i3]);
			h1i = c1*(x[i2] - x[i4]);
			h2r = -c2*(x[i2] + x[i4]);
			h2i = c2*(x[i1] - x[i3]);
			x[i1] = h1r + wr*h2r - wi*h2i;
			x[i2] = h1i + wr*h2i + wi*h2r;
			x[i3] = h1r - wr*h2r + wi*h2i;
			x[i4] = -h1i + wr*h2i + wi*h2r;
		}
		wr = (temp = wr)*wpr - wi*wpi + wr;
		wi = wi*wpr + temp*wpi + wi;
	}
	if (forward)
		x[1] = xr;
	else
		cfft (x,N,forward);
}


/* c, a, and b are rfft-format spectra each containging n floats--
 * place complex product of a and b into c 
 */
void cmult( float c[], float a[], float b[], int n)
{
	int i,j;
	float re, im;
	
	c[0] = a[0] * b[0];
	c[1] = a[1] * b[1];
	for (i=2, j=3; i<n; i+=2, j+=2)
	{
		re = a[i] * b[i]  -  a[j] * b[j];
		im = a[i] * b[j]  +  a[j] * b[i]; 
		c[i] = re;
		c[j] = im;
	}
}

