
/*
 * OpenCache()
 *
 *	return +1	if a valid cache file was found, set *pcfd, *psize
 *
 *	return 0	if a valid cache file was not found, set *pcfd to >= 0
 *			if we were able to open a temporary file for a new
 *			cache object, < 0 if we could not.
 *
 *	return -1	If we found a negatively cached entry.
 *
 */

#include "defs.h"

Prototype int OpenCache(const char *msgid, int *pcfd, int *psize);
Prototype void AbortCache(int fd, const char *msgid, int closefd);
Prototype void CommitCache(int fd, const  char *msgid, int closefd);
Prototype void DumpArticleFromCache(Connection *conn, const char *map, int size);

int
OpenCache(const char *msgid, int *pcfd, int *psize)
{
    int fd;
    int blen;
    hash_t hv = hhash(msgid);
    char path[256];

    *pcfd = -1;
    *psize = 0;

    sprintf(path, "%s/%03x", CacheHome, (int)(hv.h2 & 0x1FF));
    blen = strlen(path);
    {
	struct stat st;
	if (stat(path, &st) < 0) {
	    mkdir(path, 0755);
	}
    }
    sprintf(path + blen, "/%08x.%08x", (int)hv.h1, (int)hv.h2);

    /*
     * open cache file
     */

    if ((fd = open(path, O_RDWR)) >= 0) {
	struct stat st;

	if (fstat(fd, &st) == 0) {
	    if (st.st_size == 0) {	/* negatively cached */
		close(fd);
		return(-1);
	    }
	    *pcfd = fd;			/* positively cached */
	    *psize = st.st_size;
	    return(1);
	}
	close(fd);			/* error	     */
	return(0);
    }

    sprintf(path + blen, "/%08x.%08x.tmp", (int)hv.h1, (int)hv.h2);

    if ((fd = open(path, O_RDWR|O_CREAT, 0644)) < 0) {
	return(0);			/* error	     */
    }

    if (hflock(fd, 0, XLOCK_EX|XLOCK_NB) < 0) {
	close(fd);			/* someone else owns it */
	return(0);
    }
    {
	struct stat st;
	struct stat st2;
	if (fstat(fd, &st) < 0 || st.st_nlink == 0) {
	    close(fd);			/* delete race		*/
	    return(0);
	}
	if (stat(path, &st2) < 0 || st.st_ino != st2.st_ino) {
	    close(fd);			/* rename race		*/
	    return(0);
	}
	if (st.st_size != 0)		/* truncate if partial left over */
	    ftruncate(fd, 0);
    }

    /*
     * created new cache file.
     */
    *pcfd = fd;
    return(0);
}

/*
 * AbortCache() - cache not successfully written, destroy
 */

void
AbortCache(int fd, const char *msgid, int closefd)
{
    char path[256];
    hash_t hv = hhash(msgid);

    sprintf(path, "%s/%03x/%08x.%08x.tmp",
	CacheHome, 
	(int)(hv.h2 & 0x1FF),
	(int)hv.h1,
	(int)hv.h2
    );
    remove(path);
    ftruncate(fd, 0);
    hflock(fd, 0, XLOCK_UN);
    if (closefd)
	close(fd);
}

/*
 * CommitCache() - cache successfully written, commit to disk
 */

void
CommitCache(int fd, const  char *msgid, int closefd)
{
    char path1[256];
    char path2[256];
    hash_t hv = hhash(msgid);

    sprintf(path2, "%s/%03x/%08x.%08x",
	CacheHome, 
	(int)(hv.h2 & 0x1FF),
	(int)hv.h1,
	(int)hv.h2
    );
    strcpy(path1, path2);
    strcat(path1, ".tmp");
    if (rename(path1, path2) < 0)
	remove(path1);
    hflock(fd, 0, XLOCK_UN);
    if (closefd)
	close(fd);
}

/*
 * DUMPARTICLEFROMCACHE() - article buffer is passed as an argument.   The
 *			    buffer is already '.' escaped (but has no 
 *			    terminating '.\r\n'), and \r\n terminated.
 *
 *			    if (conn->co_ArtMode == COM_BODYNOSTAT), just
 *			    do the body.  Otherwise do the whole thing.
 */

void
DumpArticleFromCache(Connection *conn, const char *map, int size)
{
    int b = 0;
    int inHeader = 1;
    int nonl = 0;

    while (b < size) {
	int i;
	int yes = 0;

	for (i = b; i < size && map[i] != '\n'; ++i)
	    ;
	if (i < size) 
	    ++i;
	else
	    nonl = 1;
	switch(conn->co_ArtMode) {
	case COM_STAT:
	    break;
	case COM_HEAD:
	    if (inHeader == 1) {
		yes = 1;
		if (i == b + 2 && map[b] == '\r' && map[b+1] == '\n')
		    yes = 0;
	    }
	    break;
	case COM_ARTICLE:
	    yes = 1;
	    break;
	case COM_BODY:
	case COM_BODYNOSTAT:
	    if (inHeader == 0)
		yes = 1;
	    break;
	}
	if (yes)
	    MBWrite(&conn->co_TMBuf, map + b, i - b);

	if (inHeader && i == b + 2 && map[b] == '\r' && map[b+1] == '\n')
	    inHeader = 0;

	b = i;
    }
    if (nonl && conn->co_ArtMode != COM_STAT)
	MBPrintf(&conn->co_TMBuf, "\r\n", 2);
}

