#include <stdio.h>
#include <math.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>

#ifdef __STDC__
#include <stdlib.h>
#include <string.h>
typedef const void *qsort_arg; 
#else
extern char *getenv();
extern void *memset();
typedef void *qsort_arg;
#endif

#if defined(_POSIX_SOURCE)
#include <unistd.h>
#else
extern int close();
#endif

#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/MessageB.h>
#include <Xm/PushB.h>
#include <Xm/Separator.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>

#include "P_.h"
#include "astro.h"
#include "circum.h"
#include "net.h"

extern Widget toplevel_w;
#define XtD     XtDisplay(toplevel_w)
extern Colormap xe_cm;
extern XtAppContext xe_app;


#define	TOUT	180		/* max secs to wait for socket data.
				 * default http timeout is 3 minutes.
				 */

#define	DUPSEP		degrad(4/3600.)	/* dup if < this close, rads */
#define	DUPMAG		(3.0)		/* dup if mags differ < this much */
#define	GSCFOVLIM	degrad(15.0)	/* max fov we'll fetch */

extern Now *mm_get_now P_((void));
extern char *getBaseDir P_((void));
extern char *syserrstr P_((void));
extern double delra P_((double dra));
extern int GSCFetch P_((double ra0, double dec0, double fov, double fmag,
    ObjF **opp, int nopp, char msg[]));
extern int GSCSetup P_((char *cdp, char *chp, char msg[]));
extern int existsh P_((char *name));
static int fs_save P_((void));
extern int gscesofetch P_((Now *np, double ra, double dec, double fov,
    double mag, ObjF **opp, int nop, char msg[]));
extern int gscnetfetch P_((char *host, Now *np, double ra, double dec,
    double fov, double mag, ObjF **opp, int nop, char msg[]));
extern int obj_cir P_((Now *np, Obj *op));
extern int ppmchkfile P_((char *file, char *msg));
extern int ppmfetch P_((char *file, Now *np, double ra, double dec,
    double fov, double mag, int saohd, ObjF **opp, char *msg));
extern Obj *db_scan P_((DBScan *sp));  
extern void db_scaninit P_((DBScan *sp, int mask, ObjF *op, int nop));
extern void db_update P_((Obj *op));
extern void hlp_dialog P_((char *tag, char *deflt[], int ndeflt));
extern void prompt_map_cb P_((Widget w, XtPointer client, XtPointer call));
extern void set_something P_((Widget w, char *resource, XtArgVal value));
extern void set_xmstring P_((Widget w, char *resource, char *txt));
extern void watch_cursor P_((int want));
extern void wtip P_((Widget w, char *tip));
extern void xe_msg P_((char *msg, int app_modal));

static int objfd_cmp P_((qsort_arg v1, qsort_arg v2));
static int fs_nodbdups P_((ObjF *fop, int nfop, double ra, double dec,
    double fov));
static int scanchk P_((Obj *mop, Obj *op, double cdec));
static void fs_create_fsform P_((void));
static void defaultSOCKS P_((void));
static void fs_setup P_((void));
static void apply_cb P_((Widget w, XtPointer client, XtPointer call));
static void ok_cb P_((Widget w, XtPointer client, XtPointer call));
static void tb_cb P_((Widget w, XtPointer client, XtPointer call));
static void cancel_cb P_((Widget w, XtPointer client, XtPointer call));
static void help_cb P_((Widget w, XtPointer client, XtPointer call));
static int tout P_((int maxt, int fd, int w));
static char *herr P_((char *errmsg));
static int connect_to P_((int sockfd, struct sockaddr *serv_addr, int addrlen));
static int stopd_check P_((void));
static void stopd_cb P_((Widget w, XtPointer data, XtPointer call));

static Widget fsform_w;		/* the main form dialog */
static Widget cdtf_w, cdtb_w;	/* GSC CD-ROM text field and toggle button */
static Widget nettf_w, nettb_w;	/* GSC network text field and toggle button */
static Widget chtf_w, chtb_w;	/* GSC Cache text field and toggle button */
static Widget ndir_w;		/* net direct TB */
static Widget socks_w;		/* SOCKS option TB */
static Widget socksh_w, socksp_w;/* SOCKS host and port TF */
static Widget proxy_w;		/* Proxy option TB */
static Widget proxyh_w, proxyp_w;/* Proxy host and port TF */
static Widget etb_w;		/* ESO toggle button */
static Widget ppmtf_w, ppmtb_w;	/* PPM/SAO text field and toggle button */
static Widget saohdtb_w;	/* PPM TB on when want SAO/HD names */
static Widget nodupt_w;		/* no dups toggle button */
static Widget stopd_w;		/* user's stop dialog */
static int stopd_stopped;	/* set by the stopd PB */

/* code to pass to the TB callbacks.
 * used to enforce that net can't be used with CDROM or cache and only
 * one way to access the network overall.
 */
typedef enum {
    CDROMTB, CACHETB, NETTB, ESOTB, NETDIRTB, NETPROXTB, NETSOCKSTB
} TB;

static int cd_on;		/* whether GSC CDROM is currently on */
static char *cd_dn;		/* GSC CDROM dir */
static int ch_on;		/* whether GSC Cache is currently on */
static char *ch_dn;		/* GSC Cache dir */
static int net_on;		/* whether GSC network connection is on */
static char *net_host;		/* GSC network host name */
static int e_on;		/* whether ESO network connection is on */
static int proxy_on;		/* whether proxy network connection is on */
static char *proxy_host;	/* proxy host */
static int proxy_port;		/* proxy port */
static int socks_on;		/* whether SOCKS network connection is on */
static char *socks_host;	/* SOCKS host */
static char *socks_port;	/* SOCKS port (as a string) */
static int ppm_on;		/* whether ppm is currently on */
static char *ppm_fn;		/* ppm filename */
static int wantsaohd;		/* whether we want to use SAO/HD names */
static int nodup_on;		/* whether dup checking is on */

/* call to set up without actually bringing up the menus.
 */
void
fs_create()
{
	if (!fsform_w) {
	    fs_create_fsform();
	    (void) fs_save();	/* confirming here is just annoying */
	}
}

void
fs_manage()
{
	if (!fsform_w) {
	    fs_create_fsform();
	    (void) fs_save();
	}

	if (XtIsManaged(fsform_w))
	    XtUnmanageChild (fsform_w);
	else {
	    fs_setup();
	    XtManageChild (fsform_w);
	}
}

/* call to fetch a set of ObjF from the field star sources.
 * don't complain just if none are turned on at the moment.
 * we may eliminate any that seem to be dups of existing FIXED db objects.
 * we turn off any option that appears to not be working.
 * each o_flag in each resulting star is marked with FLDSTAR.
 * return -1 if something looks busted else count.
 * N.B. we only malloc *opp if we return > 0.
 */
int
fs_fetch (np, ra, dec, fov, mag, opp)
Now *np;	/* now */
double ra, dec;	/* at np->n_epoch */
double fov;	/* field of view, rads */
double mag;	/* limiting mag */
ObjF **opp;	/* we set *opp to a malloced list of ObjF */
{
	ObjF *op = NULL;
	int nop = 0;
	int s = 0;
	char msg[256];
	double gscfov;
	int i;

	if (!cd_on && !ch_on && !net_on && !e_on && !ppm_on)
	    return (-1);

	watch_cursor (1);

	/* the fetch tools all want and return J2000 values */
	if (epoch == EOD)
	    ap_as (np, J2000, &ra, &dec);
	else
	    precess (epoch, J2000, &ra, &dec);

	/* get the PPM stars, if desired */
	if (ppm_on) {
	    nop = ppmfetch (ppm_fn, np, ra, dec, fov, mag, wantsaohd, &op, msg);
	    if (nop < 0)
		ppm_on = 0;
	}

	/* now the GSC, if desired -- enforce max fov */
	gscfov = fov > GSCFOVLIM ? GSCFOVLIM : fov;
	if (net_on) {
	    nop = gscnetfetch (net_host, np, ra, dec, gscfov, mag, &op,nop,msg);
	    if (nop < 0)
		net_on = 0;
	} else if (e_on) {
	    nop = gscesofetch (np, ra, dec, gscfov, mag, &op, nop, msg);
	    if (nop < 0)
		e_on = 0;
	} else if (cd_on || ch_on) {
	    nop = GSCFetch (ra, dec, gscfov, mag, &op, nop, msg);
	    if (nop < 0)
		cd_on = 0;
	}
	if (nop < 0)
	    xe_msg (msg, 1);

	/* set the s_fields to np */
	for (i = 0; i < nop; i++)
	    obj_cir (np, (Obj *)&op[i]);

	/* squeeze out stars which duplicate each other or the main database.
	 */
	if (nodup_on && nop > 0) {
	    int newnop = fs_nodbdups (op, nop, ra, dec, fov);
	    if (newnop < nop) {
		/* shrink down to just newnop entries now */
		nop = newnop;
		if (nop > 0)
		    op = (ObjF *) realloc ((void *)op, nop*sizeof(ObjF));
		else {
		    free ((void *)op);	/* *all* are dups! */
		    op = NULL;
		    nop = 0;
		}
	    }
	} else {
	    /* still set the FLDSTAR flag on all entries */
	    for (i = 0; i < nop; i++)
		((Obj *)(&op[i]))->o_flags |= FLDSTAR;
	}

	/* pass back the result, if there's something left */
	if (nop > 0) {
	    *opp = op;
	    s = nop;
	} else if (nop < 0) {
	    fs_setup();
	    s = -1;
	}

	watch_cursor (0);

	return (s);
}

/* called to put up or remove the watch cursor.  */
void
fs_cursor (c)
Cursor c;
{
	Window win;

	if (fsform_w && (win = XtWindow(fsform_w)) != 0) {
	    if (c)
		XDefineCursor (XtDisplay(toplevel_w), win, c);
	    else
		XUndefineCursor (XtDisplay(toplevel_w), win);
	}
}

/* open the host, do the given GET cmd, and return a socket fd for the result.
 * return -1 and with excuse in msg[], else 0 if ok.
 */
int
httpGET (host, GETcmd, msg)
char *host;
char *GETcmd;
char msg[];
{
	char buf[1024];
	int fd;
	int n;

	/* fill buf[] with GET command */
	if (proxy_on) {
	    fd = mkconnection (proxy_host, proxy_port, msg);
	    if (fd < 0)
		return (-1);
	    (void) sprintf (buf, "GET http://%s", host);
	    if (GETcmd[0] != '/')
		(void) strcat (buf, "/");
	    (void) strcat (buf, GETcmd);
	} else {
	    /* SOCKS or direct are both handled by mkconnection() */
	    fd = mkconnection (host, 80, msg);
	    if (fd < 0)
		return (-1);
	    (void) sprintf (buf, "GET %s", GETcmd);
	}

	/* send it */
	n = strlen (buf);
	if (sendbytes(fd, (unsigned char *)buf, n) < 0) {
	    (void) sprintf (msg, "%s: send error: %s", host, syserrstr());
	    (void) close (fd);
	    return (-1);
	}

	/* caller can read response */
	return (fd);
}

/* qsort-style comparison of two ObjF's by dec */
static int
objfd_cmp (v1, v2)
qsort_arg v1, v2;
{
	Obj *op1 = (Obj *)v1;
	Obj *op2 = (Obj *)v2;
	double d = op1->s_dec - op2->s_dec;

	if (d < 0.0)
	    return (-1);
	if (d > 0.0)
	    return (1);
	return (0);
}

/* squeeze out all entries in fop[] which are located within DUPSEP and have a
 *   magnitude within DUPMAG of any FIXED object currently in the database. We
 *   also squeeze out any GSC stars which meet those same criteria for any PPM
 *   star.
 * when finished, all remaining entries will be contiguous at the front of the
 *   fop array and each will have FLDSTAR set in o_flag.
 * return the number of objects which remain.
 * N.B. we assume FLDSTAR bit is initially off in each o_flag.
 * N.B. we assume all fop[] have already been obj_cir()'d to mm_get_now().
 * N.B. we assume all entries in fop[] are within fov of ra/dec.
 */
static int
fs_nodbdups (fop, nfop, ra, dec, fov)
ObjF *fop;
int nfop;
double ra, dec, fov;
{
	double rov = fov/2;
	Obj *op;
	DBScan dbs;
	int l, u, m;
	Obj *mop;
	double diff;
	double cdec;

	/* sort fop by increasing dec */
	qsort ((void *)fop, nfop, sizeof(ObjF), objfd_cmp);

	/* mark all GSC stars with FLDSTAR flag which are close to any PPM.
	 * don't compare GSC and PPM stars with themselves.
	 */
	for (m = 0; m < nfop; m++) {
	    mop = (Obj *)&fop[m];

	    /* only check for GSC stars close to PPM stars, not v.v. */
	    if (mop->o_name[0] == 'G')
		continue;

	    /* scan each way from mop and mark close GSC stars */
	    cdec = cos(mop->s_dec);
	    for (u = m+1; u < nfop; u++) {
		op = (Obj *)&fop[u];
		if (fabs(mop->s_dec - op->s_dec) >= DUPSEP)
		    break;
		if (op->o_name[0] == 'G'
			&& cdec*delra(mop->s_ra - op->s_ra) < DUPSEP
			&& fabs(get_mag(mop) - get_mag(op)) < DUPMAG)
		    op->o_flags |= FLDSTAR;
	    }
	    for (l = m-1; l >= 0; l--) {
		op = (Obj *)&fop[l];
		if (fabs(mop->s_dec - op->s_dec) >= DUPSEP)
		    break;
		if (op->o_name[0] == 'G'
			&& cdec*delra(mop->s_ra - op->s_ra) < DUPSEP
			&& fabs(get_mag(mop) - get_mag(op)) < DUPMAG)
		    op->o_flags |= FLDSTAR;
	    }
	}

	/* scan all FIXED objects and mark all field stars with FLDSTAR
	 * which appears to be the same any one.
	 */
	for (db_scaninit(&dbs, FIXEDM, NULL, 0); (op = db_scan(&dbs)) != 0; ) {
	    /* update the s_ra/dec fields */
	    db_update (op);

	    /* skip ops outside the given field */
	    cdec = cos(op->s_dec);
	    if (fabs(op->s_dec - dec) > rov || cdec*delra(op->s_ra - ra) > rov)
		continue;

	    /* binary search to find the fop closest to op */
	    l = 0;
	    u = nfop - 1;
	    while (l <= u) {
		m = (l+u)/2;
		mop = (Obj *)(&fop[m]);
		diff = mop->s_dec - op->s_dec;
		if (diff < 0.0)
		    l = m+1;
		else
		    u = m-1;
	    }

	    /* scan each way from m and mark all that are dups with FLDSTAR.
	     * N.B. here, FLDSTAR marks *dups* (not the entries to keep)
	     * N.B. remember, u and l have crossed each other by now.
	     */
	    for (; l < nfop; l++)
		if (scanchk ((Obj *)(&fop[l]), op, cdec) < 0)
		    break;
	    for (; u >= 0; --u)
		if (scanchk ((Obj *)(&fop[u]), op, cdec) < 0)
		    break;
	}

	/* squeeze all entries marked with FLDSTAR to the front of fop[]
	 * and mark those that remain with FLDSTAR (yes, the bit now finally
	 * means what it says).
	 */
	for (u = l = 0; u < nfop; u++) {
	    mop = (Obj *)(&fop[u]);
	    if (!(mop->o_flags & FLDSTAR)) {
		mop->o_flags |= FLDSTAR; /* now the mark means a *keeper* */
		if (u > l)
		    memcpy ((void *)(&fop[l]), (void *)mop, sizeof(ObjF));
		l++;
	    }
	}

	return (l);
}

/* if mop is further than DUPSEP from op in dec, return -1.
 * then if mop is within DUPSEP in ra and within DUPMAG in mag too set FLDSTAR
 *   in mop->o_flags.
 * return 0.
 */
static int
scanchk (mop, op, cdec)
Obj *mop, *op;
double cdec;
{
	if (fabs(mop->s_dec - op->s_dec) >= DUPSEP)
	    return (-1);
	if (cdec*delra(mop->s_ra - op->s_ra) < DUPSEP
		&& fabs(get_mag(mop) - get_mag(op)) < DUPMAG)
	    mop->o_flags |= FLDSTAR;
	return (0);
}

static void
fs_create_fsform()
{
	char fname[1024];
	Widget f, w;
	Arg args[20];
	int n;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNmarginHeight, 10); n++;
	XtSetArg (args[n], XmNmarginWidth, 10); n++;
	XtSetArg (args[n], XmNverticalSpacing, 10); n++;
	XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	fsform_w = XmCreateFormDialog (toplevel_w, "FieldStars", args, n);
	set_something (fsform_w, XmNcolormap, (XtArgVal)xe_cm);
	XtAddCallback (fsform_w, XmNmapCallback, prompt_map_cb, NULL);
        XtAddCallback (fsform_w, XmNhelpCallback, help_cb, NULL);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle,"xephem Field Stars and Network Setup");n++;
	XtSetValues (XtParent(fsform_w), args, n);

	/* make the GSC section title */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	w = XmCreateLabel (fsform_w, "GSCT", args, n);
	set_xmstring (w, XmNlabelString, "Hubble Guide Star Catalog, GSC:");
	XtManageChild (w);

	    /* make the GSC CD-ROM toggle/text */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNcolumns, 60); n++;
	    cdtf_w = XmCreateTextField (fsform_w, "GSCCDDirectory", args, n);
	    wtip (cdtf_w, "Pathname to the root of the CDROM when mounted");
	    XtManageChild (cdtf_w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNrightWidget, cdtf_w); n++;
	    XtSetArg (args[n], XmNrightOffset, 10); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    cdtb_w = XmCreateToggleButton (fsform_w, "GSCCD", args, n);
	    XtAddCallback (cdtb_w, XmNvalueChangedCallback, tb_cb,
							    (XtPointer)CDROMTB);
	    set_xmstring (cdtb_w, XmNlabelString, "ASP CDROM Directory:");
	    wtip (cdtb_w, "Whether to use the ASP CDROM for GSC stars");
	    XtManageChild (cdtb_w);

	    /* make the GSC Cache toggle/text */

	    (void) sprintf (fname, "%s/catalogs/gsc",  getBaseDir());
	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, cdtf_w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNcolumns, 60); n++;
	    XtSetArg (args[n], XmNvalue, fname); n++;
	    chtf_w = XmCreateTextField (fsform_w, "GSCCacheDirectory", args, n);
	    wtip (chtf_w, "Pathname to the GSC subdirectory");
	    XtManageChild (chtf_w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, cdtf_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNrightWidget, chtf_w); n++;
	    XtSetArg (args[n], XmNrightOffset, 10); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    chtb_w = XmCreateToggleButton (fsform_w, "GSCCache", args, n);
	    XtAddCallback (chtb_w, XmNvalueChangedCallback, tb_cb,
							    (XtPointer)CACHETB);
	    set_xmstring (chtb_w, XmNlabelString,  "Local Cache Directory:");
	    wtip (chtb_w, "Whether to use a local disk for GSC stars");
	    XtManageChild (chtb_w);

	    /* make the GSC network toggle/text */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, chtf_w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNcolumns, 60); n++;
	    nettf_w = XmCreateTextField (fsform_w, "XEphemdbdURL", args, n);
	    wtip (nettf_w, "URL (/<host>/.../xephemdbd.pl) from which to fetch GSC stars");

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, chtf_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNrightWidget, nettf_w); n++;
	    XtSetArg (args[n], XmNrightOffset, 10); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    nettb_w = XmCreateToggleButton (fsform_w, "XEphemdbdNet", args, n);
	    XtAddCallback (nettb_w, XmNvalueChangedCallback, tb_cb,
							    (XtPointer)NETTB);
	    set_xmstring (nettb_w, XmNlabelString, "Internet to xephemdbd:");
	    wtip (nettb_w,"Whether to use an Internet XEphem server for stars");
	    XtManageChild (nettb_w);
	    XtManageChild (nettf_w);

	    /* make the ESO network toggle/text */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, nettf_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    etb_w = XmCreateToggleButton (fsform_w, "ESONet", args, n);
	    XtAddCallback (etb_w, XmNvalueChangedCallback, tb_cb,
							    (XtPointer)ESOTB);
	    set_xmstring (etb_w, XmNlabelString, "Internet to ESO");
	    wtip(etb_w,"Whether to use the Internet to ESO to fetch GSC stars");
	    XtManageChild (etb_w);

	/* make the PPM section title */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, etb_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	w = XmCreateLabel (fsform_w, "PPMT", args, n);
	set_xmstring (w, XmNlabelString,
				    "Position and Proper Motion Catalog, PPM:");
	XtManageChild (w);

	    /* make the PPM toggle/text */

	    (void) sprintf (fname, "%s/catalogs/ppm.xe",  getBaseDir());
	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNcolumns, 60); n++;
	    XtSetArg (args[n], XmNvalue, fname); n++;
	    ppmtf_w = XmCreateTextField (fsform_w, "PPMFilename", args, n);
	    wtip (ppmtf_w, "Pathname of ppm.xe file");
	    XtManageChild (ppmtf_w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNrightWidget, ppmtf_w); n++;
	    XtSetArg (args[n], XmNrightOffset, 10); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    ppmtb_w = XmCreateToggleButton (fsform_w, "PPM", args, n);
	    set_xmstring (ppmtb_w, XmNlabelString, "PPM Filename: ");
	    wtip (ppmtb_w, "Whether to access a PPM catalog");
	    XtManageChild (ppmtb_w);

	    /* make the PPM/SAO name toggle */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, ppmtf_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    saohdtb_w = XmCreateToggleButton (fsform_w, "SAOHD", args, n);
	    set_xmstring (saohdtb_w, XmNlabelString,
					    "Use SAO or HD Name if Assigned");
	    wtip (saohdtb_w,
	    		"Use SAO or HD designation instead of PPM if possible");
	    XtManageChild (saohdtb_w);

	    /* make the No dups toggle */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, saohdtb_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    nodupt_w = XmCreateToggleButton (fsform_w, "NoDups", args, n);
	    set_xmstring (nodupt_w, XmNlabelString, "No Duplicates");
	    wtip (nodupt_w,"Skip objects which appear to already be in memory");
	    XtManageChild (nodupt_w);

	/* make the Network section title */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, nodupt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	w = XmCreateLabel (fsform_w, "NetT", args, n);
	set_xmstring (w, XmNlabelString, "Network setup:");
	XtManageChild (w);

	    /* make the Direct toggle */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    ndir_w = XmCreateToggleButton (fsform_w, "Dir", args, n);
	    XtAddCallback (ndir_w, XmNvalueChangedCallback, tb_cb,
							(XtPointer)NETDIRTB);
	    set_xmstring (ndir_w, XmNlabelString, "Direct connect");
	    wtip (ndir_w, "Use direct internet connection (no proxy or SOCKS)");
	    XtManageChild (ndir_w);

	    /* make the Proxy toggle and info */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, ndir_w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNcolumns, 60); n++;
	    proxyh_w = XmCreateTextField (fsform_w, "ProxyHost", args, n);
	    wtip (proxyh_w, "Name of Proxy host");
	    XtManageChild (proxyh_w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, ndir_w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNrightWidget, proxyh_w); n++;
	    XtSetArg (args[n], XmNrightOffset, 10); n++;
	    XtSetArg (args[n], XmNcolumns, 5); n++;
	    proxyp_w = XmCreateTextField (fsform_w, "ProxyPort", args, n);
	    wtip (proxyp_w, "Proxy port number");
	    XtManageChild (proxyp_w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, ndir_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    proxy_w = XmCreateToggleButton (fsform_w, "Proxy", args, n);
	    set_xmstring (proxy_w, XmNlabelString, "via Proxy");
	    XtAddCallback (proxy_w, XmNvalueChangedCallback, tb_cb,
							(XtPointer)NETPROXTB);
	    wtip (proxy_w, "Reach the Internet through a proxy");
	    XtManageChild (proxy_w);

	    /* make the SOCKS toggle and info */

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, proxyh_w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNcolumns, 60); n++;
	    socksh_w = XmCreateTextField (fsform_w, "SOCKSHost", args, n);
	    wtip (socksh_w, "Name of SOCKS host");
	    XtManageChild (socksh_w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, proxyh_w); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNrightWidget, socksh_w); n++;
	    XtSetArg (args[n], XmNrightOffset, 10); n++;
	    XtSetArg (args[n], XmNcolumns, 5); n++;
	    socksp_w = XmCreateTextField (fsform_w, "SOCKSPort", args, n);
	    wtip (socksp_w, "SOCKS port number");
	    XtManageChild (socksp_w);

	    n = 0;
	    XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	    XtSetArg (args[n], XmNtopWidget, proxyh_w); n++;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	    XtSetArg (args[n], XmNleftOffset, 15); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    socks_w = XmCreateToggleButton (fsform_w, "SOCKS", args, n);
	    set_xmstring (socks_w, XmNlabelString, "via SOCKS");
	    XtAddCallback (socks_w, XmNvalueChangedCallback, tb_cb,
							(XtPointer)NETSOCKSTB);
	    wtip (socks_w, "Use SOCKS V4 to Internet");
	    XtManageChild (socks_w);

	    defaultSOCKS();

	/* make the controls across the bottom under a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, socksh_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (fsform_w, "Sep", args, n);
	XtManageChild (w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNfractionBase, 13); n++;
	f = XmCreateForm (fsform_w, "CF", args, n);
	XtManageChild (f);

	    n = 0;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 1); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 3); n++;
	    w = XmCreatePushButton (f, "Ok", args, n);
	    wtip (w, "Test settings and close");
	    XtAddCallback (w, XmNactivateCallback, ok_cb, NULL);
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 4); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 6); n++;
	    w = XmCreatePushButton (f, "Test", args, n);
	    wtip (w, "Test settings and remain up");
	    XtAddCallback (w, XmNactivateCallback, apply_cb, NULL);
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 7); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 9); n++;
	    w = XmCreatePushButton (f, "Close", args, n);
	    wtip (w, "Close this menu without doing anything");
	    XtAddCallback (w, XmNactivateCallback, cancel_cb, NULL);
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 10); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 12); n++;
	    w = XmCreatePushButton (f, "Help", args, n);
	    wtip (w, "More detailed descriptions");
	    XtAddCallback (w, XmNactivateCallback, help_cb, NULL);
	    XtManageChild (w);
}

/* init SOCKS host and port. first check SOCKS_PORT and SOCKS_NS env variables,
 * respectively (same ones used by netscape) then X resources.
 */
static void
defaultSOCKS()
{
	char *str;

	str = getenv ("SOCKS_PORT");
	if (str)
	    XmTextFieldSetString (socksp_w, str ? str : "1080");

	str = getenv ("SOCKS_NS");
	if (str)
	    XmTextFieldSetString (socksh_w, str);
}

/* set up the dialog according to our static state */
static void
fs_setup ()
{
	/* GSC */
	XmToggleButtonSetState (cdtb_w, cd_on, False);
	if (cd_dn)
	    XmTextFieldSetString (cdtf_w, cd_dn);

	XmToggleButtonSetState (chtb_w, ch_on, False);
	if (ch_dn)
	    XmTextFieldSetString (chtf_w, ch_dn);

	XmToggleButtonSetState (nettb_w, net_on, False);
	if (net_host)
	    XmTextFieldSetString (nettf_w, net_host);

	XmToggleButtonSetState (etb_w, e_on, False);

	/* PPM*/
	XmToggleButtonSetState (ppmtb_w, ppm_on, False);
	if (ppm_fn)
	    XmTextFieldSetString (ppmtf_w, ppm_fn);

	XmToggleButtonSetState (saohdtb_w, wantsaohd, False);

	XmToggleButtonSetState (nodupt_w, nodup_on, False);

	/* Net */
	XmToggleButtonSetState (ndir_w, !socks_on && !proxy_on, False);

	XmToggleButtonSetState (proxy_w, proxy_on, False);
	if (proxy_host)
	    XmTextFieldSetString (proxyh_w, proxy_host);
	if (proxy_port) {
	    char buf[32];
	    (void) sprintf (buf, "%d", proxy_port);
	    XmTextFieldSetString (proxyp_w, buf);
	}

	XmToggleButtonSetState (socks_w, socks_on, False);
	if (socks_host)
	    XmTextFieldSetString (socksh_w, socks_host);
	if (socks_port)
	    XmTextFieldSetString (socksp_w, socks_port);
}

/* save the dialog as our static state.
 * we test the new settings and might change them if they seem to be
 *   unreasonable.
 * if any major trouble, issue xe_msg and return -1, else return 0.
 */
static int
fs_save ()
{
	char *str, msg[1024];
	int allok = 1;
	int fd;

	watch_cursor (1);

	/* CDROM and Cache */
	cd_on = XmToggleButtonGetState (cdtb_w);
	if (cd_dn)
	    XtFree (cd_dn);
	cd_dn = XmTextFieldGetString (cdtf_w);
	ch_on = XmToggleButtonGetState (chtb_w);
	if (ch_dn)
	    XtFree (ch_dn);
	ch_dn = XmTextFieldGetString (chtf_w);
	if ((cd_on || ch_on) && GSCSetup (cd_on?cd_dn:NULL, ch_on?ch_dn:NULL,
								    msg) < 0) {
	    xe_msg (msg, 1);
	    cd_on = 0;
	    ch_on = 0;
	    fs_setup ();
	    allok = 0;
	}

	/* xephemdbd network connection */
	net_on = XmToggleButtonGetState (nettb_w);
	if (net_host)
	    XtFree (net_host);
	net_host = XmTextFieldGetString (nettf_w);
	if (net_on) {
	    /* test by actually fetching a small patch */
	    Now *np = mm_get_now();
	    double fov = degrad(1);
	    double mag = 13.0;
	    int nop;
	    ObjF *op;

	    op = NULL;
	    nop = gscnetfetch (net_host, np, 0.0, 0.0, fov, mag, &op, 0, msg);
	    if (nop < 0) {
		xe_msg (msg, 1);
		net_on = 0;
		fs_setup ();
		allok = 0;
	    } else if (nop > 0)
		free ((void *)op);
	}

	/* ESO connection */
	e_on = XmToggleButtonGetState (etb_w);
	if (e_on) {
	    /* test by actually fetching a small patch */
	    Now *np = mm_get_now();
	    double fov = degrad(1);
	    double mag = 13.0;
	    int nop;
	    ObjF *op;

	    op = NULL;
	    nop = gscesofetch (np, 0.0, 0.0, fov, mag, &op, 0, msg);
	    if (nop < 0) {
		xe_msg (msg, 1);
		e_on = 0;
		fs_setup ();
		allok = 0;
	    } else if (nop > 0)
		free ((void *)op);
	}

	/* PPM */
	ppm_on = XmToggleButtonGetState (ppmtb_w);
	if (ppm_fn)
	    XtFree (ppm_fn);
	ppm_fn = XmTextFieldGetString (ppmtf_w);
	if (ppm_on && ppmchkfile (ppm_fn, msg) < 0) {
	    xe_msg (msg, 1);
	    ppm_on = 0;
	    fs_setup();
	    allok = 0;
	}
	wantsaohd = XmToggleButtonGetState (saohdtb_w);
	nodup_on = XmToggleButtonGetState (nodupt_w);

	/* Network setup */
	proxy_on = XmToggleButtonGetState (proxy_w);
	if (proxy_host)
	    XtFree (proxy_host);
	proxy_host = XmTextFieldGetString (proxyh_w);
	str = XmTextFieldGetString (proxyp_w);
	proxy_port = atoi (str);
	XtFree (str);
	if (proxy_on) {
	    fd = mkconnection (proxy_host, proxy_port, msg);
	    if (fd < 0) {
		xe_msg (msg, 1);
		proxy_on = 0;
		fs_setup ();
		allok = 0;
	    } else
		(void) close (fd);
	}
	socks_on = XmToggleButtonGetState (socks_w);
	if (socks_host)
	    XtFree (socks_host);
	socks_host = XmTextFieldGetString (socksh_w);
	if (socks_port)
	    XtFree (socks_port);
	socks_port = XmTextFieldGetString (socksp_w);
	if (socks_on) {
	    /* TODO: how to test? */
	    fd = mkconnection (socks_host, atoi(socks_port), msg);
	    if (fd < 0) {
		xe_msg (msg, 1);
		socks_on = 0;
		fs_setup ();
		allok = 0;
	    } else
		(void) close (fd);
	}

	watch_cursor (0);

	return (allok ? 0 : -1);
}

/* called from Apply */
/* ARGSUSED */
static void
apply_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (fs_save() == 0)
	    xe_msg ("All options appear to be operating normally", 1);
}

/* called from Ok */
/* ARGSUSED */
static void
ok_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (fs_save() == 0)
	    XtUnmanageChild (fsform_w);
}

/* called from Ok */
/* ARGSUSED */
static void
cancel_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	/* outta here */
	XtUnmanageChild (fsform_w);
}

/* called from any of the CDROM, network or Cache toggle buttons.
 * client is one of the TB enums to tell us which.
 */
/* ARGSUSED */
static void
tb_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (XmToggleButtonGetState(w))
	    switch ((int)client) {
	    case CDROMTB:	/* FALLTHRU */
	    case CACHETB:
		/* turn off net and ESO if turning on CDROM or Cache */
		XmToggleButtonSetState (nettb_w, False, False);
		XmToggleButtonSetState (etb_w, False, False);
		break;
	    case NETTB:
		/* turn off CDROM and Cache and ESO if turning on network */
		XmToggleButtonSetState (cdtb_w, False, False);
		XmToggleButtonSetState (chtb_w, False, False);
		XmToggleButtonSetState (etb_w, False, False);
		break;
	    case ESOTB:
		/* turn off CDROM and Cache and net if turning on ESO */
		XmToggleButtonSetState (cdtb_w, False, False);
		XmToggleButtonSetState (chtb_w, False, False);
		XmToggleButtonSetState (nettb_w, False, False);
		break;
	    case NETDIRTB:
		/* turn off proxy and socks */
		XmToggleButtonSetState (proxy_w, False, False);
		XmToggleButtonSetState (socks_w, False, False);
		break;
	    case NETPROXTB:
		/* turn off direct and socks */
		XmToggleButtonSetState (ndir_w, False, False);
		XmToggleButtonSetState (socks_w, False, False);
		break;
	    case NETSOCKSTB:
		/* turn off direct and proxy */
		XmToggleButtonSetState (ndir_w, False, False);
		XmToggleButtonSetState (proxy_w, False, False);
		break;
	    default:
		printf ("FS: bad client: %d\n", (int)client);
		exit(1);
	    }
}

/* called from Ok */
/* ARGSUSED */
static void
help_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        static char *msg[] = {
"Set up which field star sources to use and where they are located."
};

	hlp_dialog ("FieldStars", msg, sizeof(msg)/sizeof(msg[0]));

}

/* establish a TCP connection to the named host on the given port.
 * if ok return file descriptor, else -1 with excuse in msg[].
 * try Socks if see socks_on.
 */
int
mkconnection (host, port, msg)
char *host;	/* name of server */
int port;	/* TCP port */
char msg[];	/* return diagnostic message here, if returning -1 */
{

	struct sockaddr_in serv_addr;
	struct hostent  *hp;
	int sockfd;

	/* don't want signal if loose connection to server */
	(void) signal (SIGPIPE, SIG_IGN);

	/* lookup host address.
	 * TODO: time out but even SIGALRM doesn't awaken this if it's stuck.
	 */
	hp = gethostbyname (host);
	if (!hp) {
	    (void) sprintf (msg, "Can not find IP of %s.\n%s", host, 
					    herr ("Try entering IP directly"));
	    return (-1);
	}

	/* connect -- use socks if on */
        if (socks_on) {
            /* Connection over SOCKS server */
            struct {
		unsigned char  VN;	/* version number */
		unsigned char  CD;	/* command code */
		unsigned short DSTPORT;	/* destination port */
		unsigned long  DSTIP;	/* destination IP addres */
	    } SocksPacket;

	    struct hostent *hs = gethostbyname (socks_host);
	    char *Socks_port_str = socks_port;
	    int Socks_port = Socks_port_str ? atoi (Socks_port_str) : 1080;

            SocksPacket.VN = 4;		/* version 4 */
	    SocksPacket.CD = 1;		/* Command code 1 = connect request */
            SocksPacket.DSTPORT = htons(port);/*Taken from function parameters*/

            if (!hs) {
		(void) sprintf (msg, "SOCKS: %s:\nCan not get server IP:\n%s.",
				socks_host, herr("Try entering IP directly"));
		return (-1);
            }

	    (void) memset ((char *)&serv_addr, 0, sizeof(serv_addr));
	    serv_addr.sin_family = AF_INET;
	    serv_addr.sin_addr.s_addr=
                              ((struct in_addr *)(hs->h_addr_list[0]))->s_addr;
	    serv_addr.sin_port = htons(Socks_port);
	    if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
	        (void) sprintf (msg, "SOCKS: %s/%d:\n%s", socks_host, port,
								syserrstr());
	        return (-1);
	    }

	    /* Yes, again. Some variables inside are static */
	    hp = gethostbyname (host);

            SocksPacket.DSTIP=((struct in_addr *)(hp->h_addr_list[0]))->s_addr;
	    if (connect_to (sockfd, (struct sockaddr *)&serv_addr,
						    sizeof(serv_addr)) < 0) {
	        (void) sprintf (msg, "SOCKS: %s: %s", socks_host, syserrstr());
	        (void) close(sockfd);
	        return (-1);
            }
            (void)write (sockfd, &SocksPacket, sizeof (SocksPacket));
            (void)write (sockfd, "xephem", 7);	/* yes, include trailing \0 */
            (void)read  (sockfd, &SocksPacket, sizeof (SocksPacket));
            switch (SocksPacket.CD) {
                case 90:
                    break; /* yes! */
                case 92:
                    (void) sprintf (msg, "SOCKS: cannot connect to client");
		    (void) close(sockfd);
                    return (-1);
                case 93:
                    (void) sprintf (msg, "SOCKS: client program and ident report different user-ids");
		    (void) close(sockfd);
                    return (-1);
                default:
                    (void) sprintf (msg, "SOCKS: Request rejected or failed");
		    (void) close(sockfd);
                    return (-1);
            }
            
	} else {
            /* normal connection without SOCKS server */
	    /* create a socket to the host's server */
	    (void) memset ((char *)&serv_addr, 0, sizeof(serv_addr));
	    serv_addr.sin_family = AF_INET;
	    serv_addr.sin_addr.s_addr =
			((struct in_addr *)(hp->h_addr_list[0]))->s_addr;
	    serv_addr.sin_port = htons(port);
	    if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
	        (void) sprintf (msg, "%s/%d: %s", host, port, syserrstr());
	        return (-1);
	    }
	    if (connect_to (sockfd, (struct sockaddr *)&serv_addr,
						    sizeof(serv_addr)) < 0) {
	        (void) sprintf (msg, "%s: %s", host, syserrstr());
	        (void) close(sockfd);
	        return (-1);
            }
	}

	return (sockfd);
}

/* send n bytes from buf to socket fd.
 * return 0 if ok else -1
 */
int
sendbytes (fd, buf, n)
int fd;
unsigned char buf[];
int n;
{
	int ns, tot;

	for (tot = 0; tot < n; tot += ns) {
	    if (tout (TOUT, fd, 1) < 0)
		return (-1);
	    ns = write (fd, (void *)(buf+tot), n-tot);
	    if (ns <= 0)
		return (-1);
	}
	return (0);
}

/* receive exactly n bytes from socket fd into buf.
 * return n if ok else -1
 */
int
recvbytes (fd, buf, n)
int fd;
unsigned char buf[];
int n;
{
	int ns, tot;

	for (tot = 0; tot < n; tot += ns) {
	    if (tout (TOUT, fd, 0) < 0)
		return (-1);
	    ns = read (fd, (void *)(buf+tot), n-tot);
	    if (ns <= 0)
		return (-1);
	}
	return (n);
}

/* read up to and including the next '\n' from socket fd into buf[max].
 * we silently ignore all '\r'. we add a trailing '\0'.
 * return line lenth (not counting \0) if all ok, else -1.
 * N.B. this never reads ahead -- if that's ok, recvlineb() is better
 */
int
recvline (fd, buf, max)
int fd;
char buf[];
int max;
{
	unsigned char c;
	int n;

	max--;	/* leave room for trailing \0 */

	for (n = 0; n < max && recvbytes (fd, &c, 1) == 1; ) {
	    if (c != '\r') {
		buf[n++] = c;
		if (c == '\n') {
		    buf[n] = '\0';
		    return (n);
		}
	    }
	}

	return (-1);
}

/* rather like recvline but reads ahead in big chunk for efficiency.
 * return length if read a line ok, 0 if hit eof, -1 if error.
 * N.B. we silently swallow all '\r'.
 * N.B. we read ahead and can hide bytes after each call.
 */
int
recvlineb (sock, buf, size)
int sock;
char *buf;
int size;
{
	static char linebuf[512];	/* [next .. bad-1] are good */
	static int next;		/* index of next good char */
	static int unk;			/* index of first unknown char */
	char *origbuf = buf;		/* save to prevent overfilling buf */
	char c = '\0';
	int nr = 0;

	/* always leave room for trailing \n */
	size -= 1;

	/* read and copy linebuf[next] to buf until buf fills or copied a \n */
	do {

	    if (next >= unk) {
		/* linebuf is empty -- refill */
		if (tout (TOUT, sock, 0) < 0) {
		    nr = -1;
		    break;
		}
		nr = read (sock, linebuf, sizeof(linebuf));
		if (nr <= 0)
		    break;
		next = 0;
		unk = nr;
	    }

	    if ((c = linebuf[next++]) != '\r')
		*buf++ = c;

	} while (buf-origbuf < size && c != '\n');

	/* always give back a real line regardless, else status */
	if (c == '\n') {
	    *buf = '\0';
	    nr = buf - origbuf;
	}

	return (nr);
}

/* wait at most maxt secs for the ability to read/write using fd and allow X
 *   processing in the mean time.
 * w is 0 is for reading, 1 for writing, 2 for either.
 * return 0 if ok to proceed, else -1 if trouble or timeout.
 */
static int
tout (maxt, fd, w)
int maxt;
int fd;
int w;
{
	int i;
	    
	for (i = 0; i < maxt && stopd_check() == 0; i++) {
	    fd_set rset, wset;
	    struct timeval tv;
	    int ret;

	    FD_ZERO (&rset);
	    FD_ZERO (&wset);
	    switch (w) {
	    case 0:
	    	FD_SET (fd, &rset);
		break;
	    case 1:
	    	FD_SET (fd, &wset);
		break;
	    case 2:
	    	FD_SET (fd, &rset);
	    	FD_SET (fd, &wset);
		break;
	    default:
		printf ("Bug: tout() called with %d\n", w);
		exit(1);
	    }

	    tv.tv_sec = 1;
	    tv.tv_usec = 0;

	    ret = select (fd+1, &rset, &wset, NULL, &tv);
	    if (ret > 0)
		return (0);
	    if (ret < 0)
		return (-1);
	}

	errno = i == maxt ? ETIMEDOUT : EINTR;
	return (-1);
}

/* a networking error has occured. if we can dig out more details about why
 * using h_errno, return its message, otherwise just return errmsg unchanged.
 * we do this because we don't know how portable is h_errno?
 */
static char *
herr (errmsg)
char *errmsg;
{
#if defined(HOST_NOT_FOUND) && defined(TRY_AGAIN)
	extern int h_errno;
	switch (h_errno) {
	case HOST_NOT_FOUND:
	    errmsg = "Host Not Found";
	    break;
	case TRY_AGAIN:
	    errmsg = "Might be a temporary condition -- try again later";
	    break;
	}
#endif
	return (errmsg);
}

/* just like connect(2) but tries to time out after TOUT yet let X continue.
 * return 0 if ok, else -1.
 */
static int
connect_to (sockfd, serv_addr, addrlen)
int sockfd;
struct sockaddr *serv_addr;
int addrlen;
{
#ifdef O_NONBLOCK               /* _POSIX_SOURCE */
#define NOBLOCK O_NONBLOCK
#else
#define NOBLOCK O_NDELAY
#endif
	int err, len;
	int flags;
	int ret;

	/* set socket non-blocking */
	flags = fcntl (sockfd, F_GETFL, 0);
	(void) fcntl (sockfd, F_SETFL, flags | NOBLOCK);

	/* start the connect */
	ret = connect (sockfd, serv_addr, addrlen);
	if (ret < 0 && errno != EINPROGRESS)
	    return (-1);

	/* wait for sockfd to become useable */
	ret = tout (TOUT, sockfd, 2);
	if (ret < 0)
	    return (-1);

	/* verify connection really completed */
	len = sizeof(err);
	err = 0;
	ret = getsockopt (sockfd, SOL_SOCKET, SO_ERROR, (char *) &err, &len);
	if (ret < 0)
	    return (-1);
	if (err != 0) {
	    errno = err;
	    return (-1);
	}

	/* looks good - restore blocking */
	(void) fcntl (sockfd, F_SETFL, flags);
	return (0);
}


/* a little dialog to let the user stop the net read */

/* pop up a modal dialog to allow aborting the pixel reading.
 */
void
stopd_up()
{
	if (!stopd_w) {
	    Arg args[20];
	    int n;

	    n = 0;
	    XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	    XtSetArg(args[n], XmNdefaultPosition, False);  n++;
	    XtSetArg(args[n], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL);  n++;
	    XtSetArg(args[n], XmNtitle, "xephem Net Stop");  n++;
	    stopd_w = XmCreateInformationDialog(toplevel_w, "NetStop", args, n);
	    set_something (stopd_w, XmNcolormap, (XtArgVal)xe_cm);
	    XtAddCallback (stopd_w, XmNokCallback, stopd_cb, (XtPointer)0);
	    XtAddCallback (stopd_w, XmNmapCallback, prompt_map_cb, NULL);
	    XtUnmanageChild(XmMessageBoxGetChild(stopd_w,
						    XmDIALOG_CANCEL_BUTTON));
	    XtUnmanageChild(XmMessageBoxGetChild(stopd_w,XmDIALOG_HELP_BUTTON));
	    set_xmstring(stopd_w, XmNmessageString, "Press Stop to cancel\nthe network read...");
	    set_xmstring(stopd_w, XmNokLabelString, "Stop");
	}

	XtManageChild (stopd_w);
	stopd_stopped = 0;
}

/* bring down the user stop dialog.
 */
void
stopd_down()
{
	if (stopd_w)
	    XtUnmanageChild (stopd_w);
	stopd_stopped = 0;
}

/* called when the user presses the Stop button */
/* ARGSUSED */
static void
stopd_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	stopd_stopped = 1;
}

/* poll whether the user wants to stop.
 * return -1 to stop, else 0.
 */
static int
stopd_check()
{
	/* check for user button presses */
	while ((XtAppPending (xe_app) & XtIMXEvent) == XtIMXEvent)
	    XtAppProcessEvent (xe_app, XtIMXEvent);

	return (stopd_stopped ? -1 : 0);
}
