module GraphicsBitmap
	( Bitmap
	, loadBitmap, readBitmap, deleteBitmap
	, getBitmapSize
	, bitmap, stretchBitmap, shearBitmap
	, withBitmap
	, createCompatibleBitmap, withCompatibleBitmap, withCompatibleDC
	, createBitmapFile
	) where

import qualified Win32
import GraphicsUtilities( bracket, bracket_ )
import Word
import Addr( plusAddr )
import Int

import GraphicsTypes

----------------------------------------------------------------
-- The interface
----------------------------------------------------------------

newtype Bitmap = MkBitmap Win32.HBITMAP

loadBitmap    :: String -> IO (Bitmap, (Int, Int))
readBitmap    :: String -> IO Bitmap
deleteBitmap  :: Bitmap -> IO ()

getBitmapSize :: Bitmap -> IO (Int, Int)

-- Bitmaps can be drawn in three ways:
-- a) with no transformation at  a point
-- b) stretched to fit           a rectangle
-- c) rotated and sheared to fit a parallelogram
--
-- Sadly, the latter isn't supported in Win'95

bitmap        :: Point                      -> Bitmap -> Picture
stretchBitmap :: Point  -> Point            -> Bitmap -> Picture
shearBitmap   :: Point  -> Point  -> Point  -> Bitmap -> Picture

withBitmap             :: Bitmap -> Picture -> Picture
createCompatibleBitmap :: Win32.HDC -> Int32 -> Int32 -> IO Bitmap
withCompatibleBitmap   :: Int32 -> Int32 -> (Bitmap -> Picture) -> Picture
withCompatibleDC       :: (Win32.HDC -> Picture) -> Picture
createBitmapFile       :: String -> Bitmap -> Picture

----------------------------------------------------------------
-- The implementation
----------------------------------------------------------------

deleteBitmap (MkBitmap bitmap) = Win32.deleteBitmap bitmap

loadBitmap fileName = do
     	--putStrLn ("<<DEBUG: loading bitmap \"" ++ fileName ++ "\">>")
     	bmp <- readBitmap fileName
     	sz  <- getBitmapSize bmp
     	return (bmp, sz)

getBitmapSize (MkBitmap bmp) = do
	(ty, w, h, wBytes, planes, bitsPixel, bits) <- Win32.getBitmapInfo bmp
    	return (toInt w, toInt h)

bitmap pt bmp =
  withCompatibleDC $ \ hdc ->
  withBitmap bmp   $ \ memdc -> do
    (width,height) <- getBitmapSize bmp
    Win32.bitBlt hdc x y (fromInt width) (fromInt height) 
      memdc 0 0 Win32.sRCCOPY
 where
  (x,y) = fromPoint pt

stretchBitmap p0 p1 bmp =
  withCompatibleDC $ \ hdc ->
  withBitmap bmp   $ \ memdc -> do
    (width,height) <- getBitmapSize bmp
    Win32.stretchBlt hdc   x0 y1 (x1-x0) (y0-y1)
	             memdc 0  0  (fromInt width) (fromInt height) Win32.sRCCOPY
 where
  (x0,y0) = fromPoint p0
  (x1,y1) = fromPoint p1

shearBitmap p0 p1 p2 bmp =
  withCompatibleDC $ \ hdc ->
  withBitmap bmp   $ \ memdc -> do
    (width,height) <- getBitmapSize bmp
    Win32.plgBlt hdc (fromPoint p0) (fromPoint p1) (fromPoint p2)
                     memdc 0 0 (fromInt width) (fromInt height) Nothing 0 0

withBitmap (MkBitmap bmp) p = \ hdc ->
  bracket_ (Win32.selectBitmap hdc bmp) (Win32.selectBitmap hdc) (p hdc)

-- Note that this DC is only "1 bit" in size - you have to call
-- "createCompatibleBitmap" before it is big enough to hold the bitmap
-- you want.
withCompatibleDC p = \ hdc ->
  bracket (Win32.createCompatibleDC (Just hdc)) Win32.deleteDC (p hdc)

withCompatibleBitmap w h p = \ hdc -> 
  bracket (createCompatibleBitmap hdc w h) deleteBitmap $ \ bitmap ->
  p bitmap hdc

createCompatibleBitmap hdc w h =
  Win32.createCompatibleBitmap hdc w h >>= return . MkBitmap

createBitmapFile fileName (MkBitmap bmp) hdc =
  Win32.createBMPFile fileName bmp hdc

----------------------------------------------------------------
-- Reading bitmaps from files
----------------------------------------------------------------

-- ToDo: the "bits" read are never freed but I think we can free them
-- as soon as we call createDIBitmap.

-- Summary of the Win32 documentation on BMP files:
--
-- A bitmap file consists of:
-- 
-- 	   +-------------------+
-- 	   | BITMAPFILEHEADER  |
-- 	   +-------------------+
-- 	   | BITMAPINFOHEADER  |
-- 	   +-------------------+
-- 	   | RGBQUAD array     |
-- 	   +-------------------+
-- 	   | Color-index array |
-- 	   +-------------------+
-- 
-- The file header tells you the size of the file and the offset of the
-- bitmap data from the header start.
-- 
-- The info header specifies the width and height, the colour format,
-- compression mode, number of bytes of data, resolution and the number
-- of colours.
-- 
-- The RGBQUAD array is a palette.
-- 
-- The Color-index array is the actual bitmap.

readBitmap fileName = 
  bracket 
    (Win32.createFile fileName 
	  	      Win32.gENERIC_READ Win32.fILE_SHARE_READ Nothing 
		      Win32.oPEN_EXISTING Win32.fILE_ATTRIBUTE_NORMAL Nothing)
    Win32.closeHandle
    $ \ file -> do
  (offset, size) <- readFileHeader file
  (infoHeader,bmi,bits) <- readBits file offset size
  hdc <- Win32.getDC Nothing  -- hdc for the screen
  bmp <- Win32.createDIBitmap hdc infoHeader Win32.cBM_INIT bits bmi Win32.dIB_RGB_COLORS
  return (MkBitmap bmp)

readFileHeader :: Win32.HANDLE -> IO (Word32, Word32)
readFileHeader file = do
  -- read the file header
  fileHeader <- Win32.malloc Win32.sizeofLPBITMAPFILEHEADER
  read <- Win32.readFile file fileHeader
            Win32.sizeofLPBITMAPFILEHEADER Nothing
  assert (read == Win32.sizeofLPBITMAPFILEHEADER) "Bitmap file lacks header"
  -- check the tag and get the size
  (tag, size, r1, r2, offset) <- Win32.getBITMAPFILEHEADER fileHeader
  assert (tag == intToWord16 (fromEnum 'B' + 256 * fromEnum 'M')) 
         "Bitmap file lacks tag"
  assert (r1 == 0 && r2 == 0) 
         "Bitmap header contains non-zero reserved words"
  Win32.free fileHeader
  return ( offset - Win32.sizeofLPBITMAPFILEHEADER
         , size - Win32.sizeofLPBITMAPFILEHEADER
         )

-- read the bits out of the rest of the file
-- assumes that you've just read the file header
readBits :: Win32.HANDLE -> Word32 -> Word32 -> 
              IO (Win32.LPBITMAPINFOHEADER, Win32.LPBITMAPINFO, Win32.LPVOID)
readBits file offset size = do
  header <- Win32.malloc size
  read <- Win32.readFile file header size Nothing
  assert (read == size) "Bitmap file ended unexpectedly"
  return ( header
         , header
         , header `plusAddr` word32ToInt offset
	 )

-- In the development system, this might print the error message
-- if the assertion fails.
assert :: Bool -> String -> IO ()
assert _ _ = return ()
{-
assert True _ = return ()
assert False why = do
  putStrLn "Assertion failed:"
  putStrLn why
  return ()
-}

----------------------------------------------------------------
-- End
----------------------------------------------------------------
