/*
 * X-mame config-file and commandline parsing
 */

#define __CONFIG_C_
#include "xmame.h"
#include "sound.h"
#include "driver.h"
#include "network.h"

/* some prototypes */
int parse_command_line (int argc, char *argv[]);
int parse_xmamerc_file();
void set_default_options (void);
extern char *title;

#ifndef MESS
/* from ... */
extern int beam, flicker;
extern int antialias;
#endif

/*
 * get configuration from configfile and env.
 */
int get_config (void)
{
   char *pt;
   
   set_default_options();
   
   if (parse_xmamerc_file()==OSD_NOT_OK)
   {
      fprintf(stderr,"Error in parsing xmamerc file\n");
      return (OSD_NOT_OK);
   }
   netstate = 0;
   netkeymap = FALSE;
   players = 0;

   
   /* get environment variables. This overrides xmamerc options */
   if ( (pt=getenv("DISPLAY")) ) displayname=pt;
#ifndef MESS
   if ( (pt=getenv("MAMEDIR")) ) mamedir=pt;
#endif
   
   return OSD_OK;
}

/* 
 * show help and exit
 */
void show_usage(void) 
{
char message[]="%s\n"
#ifdef MESS
  "Usage: xmess [system] [game] [options]\n"
#else
  "Usage: xmame [game] [options]\n"
#endif 
  "Options:\n"
  "\n"
  "*** Input ***\n"
  " -[no]joy                    Disable/enable joystick (if supported)\n"
  " -[no]mouse                  Disable/enable mouse (if supported)\n"
  " -[no]pad                    Disable/enable FM-TOWNS game pad (if supported)\n"
  " -[no]swapjoyaxis            Swap X and Y joystick axis\n"
  " -[no]analogstick            Use Joystick as analog for analog controls\n"
  " -x11joyname <name>          Name of X-based joystick device (if compiled in)\n"
  " -paddevname <name>          Name of pad device (defaults to /dev/pad00)\n"
  " -mapkey <Xcode>,<Mcode>     Set an specific key mapping\n"
  " -[no]cheat                  Disable/enable cheat mode if game supports it\n"
  "\n"
  "*** Network ***\n"
  " -master                     Enable master mode. Set number of players\n"
  " -slave                      Enable slave mode. Set master hostname\n"
  " -netmapkey                  All players use the player 1 keys. To use with *real* multiplayer games\n"
  "\n"
  "*** Video ***\n"
  " -heightscale<scale>         Set Y-Scale aspect ratio\n"
  " -widthscale	<scale>         Set X-Scale aspect ratio\n"
  " -scale 	<scale>         Set X-Y Scale to the same aspect ratio\n"
  " -frameskip	<#frames>       Skip <#frames> in video refresh\n"
#if !defined(ggi) && !defined(svgalib)
  " -display	<display>       To select display\n"
  " -[no]mitshm                 Disable/enable MIT Shared Mem (if supported)\n"
  " -[no]xsync                  Select XFlush/Xsync screen refresh method\n"
  " -root_window_id [id]        Create the xmame-window in an alternate\n"
  "                             root-window, only usefull for frontends!\n"
#endif
  " -[no]dirty                  Disable/enable use of dirty rectangles\n"
  " -[no]throttle               Disable/enable throttle\n"
  " -[no]privatecmap            Disable/enable use of private color maps\n"
#if defined(linux) && defined(svgalib)
  " -videomode  <mode>          Select svgalib video mode\n"
#endif
  " -[no]forcetruecolor         Disable/enable try to use pseudocolor X-Visuals\n"
#ifndef MESS
  " -norotate                   Disable rotation\n"
  " -ror                        Rotate display 90 degrees rigth\n"
  " -rol                        Rotate display 90 degrees left\n"
  " -flipx                      Flip X axis\n"
  " -flipy                      Flip Y axis\n"
  "\n"
  "*** Vector Games ***\n"
  " -beam                       Set the beam size for vector games\n"
  " -flicker                    Set the flicker for vector games\n"
  " -[no]antialias              Disable/enable antialiasing\n"
#endif
  "\n"
  "*** Sound ***\n"
  " -usedsp                     ( Linux only ) Select /dev/dsp as sound dev\n"
  " -useaudio                   ( Linux only ) Select /dev/audio as sound dev\n"
  " -[no]sound                  Disable/enable sound (if available)\n"
  " -samplefreq <samplefreq>    Set the sample frequency\n"
  " -timerfreq  <timerfreq>     Set the timer frequency (if timer based audio)\n"
  " -fakesound                  Generate sound even when sound is disabled, this is\n"
  "                             needed for some games which won't run without sound\n"
  "\n"
  "*** General ***\n"
#ifndef MESS
  " -mamedir    <dir>           Tell XMame where roms base tree resides\n"
#endif
  " -spooldir   <dir>           Tell XMame/XMess where to store high scores\n"
  " -showconfig                 Display Running parameters in rc style\n"
  " -help | -h | -?             Show this help\n"
  " -log                        Log debug info\n"
#ifdef MESS
  " -list          [regexp]     List supported systems matching regexp, or all\n"
  " -listfull      [regexp]     List supported systems with full description\n"
  " -listdetails                Detailed list of all supported systems\n"
#else
  " -list          [regexp]     List supported games matching regexp, or all\n"
  " -listfull      [regexp]     List supported games with full description\n"
  " -listgames     [regexp]     List supported games with manufacturer and year\n"
  " -listsamdir    [regexp]     List dir where samples should go\n"
  " -listroms      [regexp]     Show ROM names of games matching regexp\n"
  " -listsamples   [regexp]     Show audio samples of games matching regexp\n"
  " -listdetails                Detailed list of all supported games\n"
  " -verifyroms    [regexp]     Verify ROMS for games matching regexp\n"
  " -verifysamples [regexp]     Verify audio samples for games matching regexp\n"
  " -noclones                   Don't show bootlegs/clones in list commands\n"
#endif
  " [regexp]                    May contain * and ? wildcards\n"
  "\n"
  "Files: \n"
  " %s	Global configuration file\n"
#ifdef MESS
  " ${HOME}/xmess/xmessrc       User global configuration file\n"
  " ${HOME}/xmess/<game>.cnf    Game-dependent user configuration file\n"
#else
  " ${HOME}/xmame/xmamerc       User global configuration file\n"
  " ${HOME}/xmame/<game>.cnf    Game-dependent user configuration file\n"
#endif
  "\n"
  "Environment variables:\n"
  "\n"
  " HOME                        Users's home directory\n"
#ifndef MESS
  " MAMEDIR                     Arcade Roms base directory\n"
#endif
  " DISPLAY                     X-Server to display in\n"
  "\n"
#ifdef MESS
  "M.E.S.S.  -  Multi-Emulator Super System\n"
  "Copyright (C) 1998  by the MESS team\n"
  "X-Mess port maintained by Hans de Goede\n"
#else
  "Mame is an Original Idea of Nicola Salmoria and Mirko Buffoni\n"
  "X-Mame port maintained by Hans de Goede\n"
#endif
  "";
	fprintf(stderr,message,title,MAMERC);
}

/*
 * show running parameters in xmamerc / xmessrc style format ( using stdout )
 * if exitflag is true exit program
 *
 */
void show_config(void)
{
	fprintf(stderr,"##### %s Running parameters: #####\n",title);
	fprintf(stderr,"#\n");
	fprintf(stderr,"# Global parameters\n");
	fprintf(stderr,"#\n");
#ifndef MESS
	fprintf(stderr,"mamedir             %s\n",mamedir);
#endif
	fprintf(stderr,"spooldir            %s\n",spooldir);
#ifndef MESS
	fprintf(stderr,"defaultgame         %s\n",drivers[game_index]->name);
#endif
	fprintf(stderr,"#\n");
	fprintf(stderr,"# Video parameters\n");
	fprintf(stderr,"#\n");
	fprintf(stderr,"display             %s\n",displayname);
	fprintf(stderr,"videomode           %d\n",video_mode);
	fprintf(stderr,"heightscale         %d\n",heightscale);
	fprintf(stderr,"widthscale          %d\n",widthscale);
	fprintf(stderr,"frameskip           %d\n",options.frameskip);
	fprintf(stderr,"use_xsync           %d\n",(use_xsync)?1:0);
	fprintf(stderr,"use_dirty           %d\n",(use_dirty)?1:0);
	fprintf(stderr,"use_mitshm          %d\n",(mit_shm_avail)?1:0);
	fprintf(stderr,"private_colormap    %d\n",(use_private_cmap)?1:0);
	fprintf(stderr,"force_truecolor     %d\n",(force_truecolor)?1:0);
	fprintf(stderr,"throttle            %d\n",(throttle)?1:0);
#ifndef MESS
	fprintf(stderr,"#\n");
	fprintf(stderr,"# Vector parameters\n");
	fprintf(stderr,"#\n");
	fprintf(stderr,"antialias           %d\n",(antialias)?1:0);
	fprintf(stderr,"flicker             %f\n",(float)flicker/2.55);
	fprintf(stderr,"beam                %f\n",(float)beam/0x010000);
#endif
	fprintf(stderr,"#\n");
	fprintf(stderr,"# Audio parameters\n");
	fprintf(stderr,"#\n");
	fprintf(stderr,"play_sound          %d\n",(play_sound)?1:0);
	fprintf(stderr,"audiodevice         %s\n",audiodevice);
	fprintf(stderr,"samplefreq          %d\n",options.samplerate);
	fprintf(stderr,"timerfreq           %d\n",audio_timer_freq);
	fprintf(stderr,"#\n");
	fprintf(stderr,"# Mouse, Joystick and Trakball parameters\n");
	fprintf(stderr,"#\n");
	fprintf(stderr,"use_mouse           %d\n",(use_mouse)?1:0);
	fprintf(stderr,"use_joystick        %d\n",(use_joystick)?1:0);
	fprintf(stderr,"x11joyname          %s\n",x11joyname);
	fprintf(stderr,"swapjoyaxis         %d\n",(swapjoyaxis)?1:0);
	fprintf(stderr,"analogstick         %d\n",(analogstick)?1:0);
	fprintf(stderr,"towns_pad_name      %s\n",towns_pad_dev);
}

/*
 * check for existence of ${HOME}/xmame/xmamerc / ${HOME}/xmess/xmessrc
 * if found parse it, else try global MAMERC #defined file
 */
int parse_xmamerc_file()
{
	struct passwd   *pw;
	FILE            *file;
	int             lineno;
	char            buffer[2048];

	/* locate user's home directory */
	pw=getpwuid(getuid());
	if(!pw) { 
		fprintf(stderr,"Who are you? Not found in passwd database!!\n");
		return (OSD_NOT_OK);
	}
#ifdef MESS
	sprintf(buffer,"%s/xmess/xmessrc",pw->pw_dir);
#else
	sprintf(buffer,"%s/xmame/xmamerc",pw->pw_dir);
#endif
	/* try to open file. If so parse it */
	if ( (file=fopen(buffer,"r")) == (FILE *) NULL) {
		fprintf(stderr,"File %s not found.\n",buffer);
		/* No user config found - trying global file */
		strcpy(buffer, MAMERC );
		if ( (file=fopen(buffer,"r")) == (FILE *) NULL) {
			fprintf(stderr,"File %s not found. Skipping ...\n",buffer);
			return (OSD_OK);
		}
	}
	lineno=0;
	while(fgets( buffer,2047,file) ) {
	    char *p;
	    char *(q[5]);
	    int i,j;
	    lineno++;
	    /* skip comments */
	    if ( ( p=strchr(buffer,'#') ) ) *p='\0';
	    /* scan for words */
	    for(i=0;i<5;i++) q[i]=(char *)NULL;
	    for(i=0,j=0,p=buffer; *p; p++ ) {
		if ( isspace(*p) ) { *p='\0'; j=0; }
		else               { if(!j++) q[i++]=p; }
	    }   /* end of stripping words ( for i=0 )*/
	    /* test for wrong number of args */
	    if ( i==0 ) continue; /* empty line */
	    if ( i!=2 ) { 
		    fprintf(stderr,"Line %d: wrong number of parameters \n",lineno);
		    fclose(file);
		    return (OSD_NOT_OK);
	    }
	    /* now parse commands */
	    if (strcasecmp(q[0], "play_sound") == 0) {
		play_sound        = atoi( q[1] );
		continue;
	    } /* sound */
	    if (strcasecmp(q[0], "use_mouse") == 0) {
#ifdef USE_MOUSE
		use_mouse = atoi( q[1] ) ;
#endif
		continue; 
	    } /* mouse */
	    if (strcasecmp(q[0], "use_joystick") == 0) {
#ifdef USE_JOYSTICK
		use_joystick = atoi( q[1] ) ;
#endif
		continue; 
	    } /* joystick */
	    if (strcasecmp(q[0], "swapjoyaxis") == 0) {
#ifdef USE_JOYSTICK
		swapjoyaxis = atoi( q[1] ) ;
#endif
		continue; 
	    } /* swapjoyaxis */
	    if (strcasecmp(q[0], "analogstick") == 0) {
#ifdef USE_JOYSTICK
		analogstick = atoi( q[1] ) ;
#endif
		continue; 
	    } /* analogstick */
	    if (strcasecmp(q[0], "frameskip") == 0) {
		options.frameskip = atoi( q[1] ) ;
		if (options.frameskip > 5) options.frameskip = 5;
		continue; 
	    } /* frameskip */
	    if (strcasecmp(q[0], "widthscale") == 0) {
		widthscale = atoi ( q[1] );
		if (widthscale <= 0) {
			fprintf (stderr,"Line %d: Illegal widthscale (%d)\n", lineno,widthscale);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* widthscale */
	    if (strcasecmp(q[0], "heightscale") == 0) {
		heightscale = atoi ( q[1] );
		if (heightscale <= 0) {
			fprintf (stderr,"Line %d: Illegal heightscale (%d)\n", lineno,heightscale);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* heightscale */
	    if (strcasecmp(q[0], "force_truecolor") == 0) {
		force_truecolor = atoi(q[1]);
		continue;
	    } /* Force use of XSync() to refresh screen */
	    if (strcasecmp(q[0], "use_xsync") == 0) {
		use_xsync = atoi(q[1]);
		continue;
	    } /* use_xsync */
	    if (strcasecmp(q[0], "use_dirty") == 0) {
		use_dirty = atoi(q[1]);
		continue;
	    } /* use_dirty */
	    if (strcasecmp(q[0], "private_colormap") == 0) {
		use_private_cmap = atoi(q[1]);
		continue;
	    } /* Force use of private color maps */
	    if (strcasecmp(q[0], "use_mitshm") == 0) {
#ifdef USE_MITSHM
		mit_shm_avail = ( atoi(q[1]) ) ? 0 : -1 ;
#endif
		continue;
	    } /* mit shared memory */
            if (strcasecmp(q[0], "towns_pad_name") == 0) {
                towns_pad_dev=(char *)malloc(1+strlen(q[1]) );
                if( ! towns_pad_dev ) {
                        fprintf(stderr,"Malloc error: line %d\n",lineno);
                        return (OSD_NOT_OK);
                }
                strcpy(towns_pad_dev,q[1]);
#ifdef LIN_FM_TOWNS
                use_joystick = TRUE ;
#endif
               continue;
            } /* FM TOWNS game pad */
#ifndef MESS
	    if (strcasecmp(q[0], "defaultgame") == 0) {
		int j;
		/* do we have a drivers for this? */
		for (j = 0; drivers[j]; j++)
		   if (strcasecmp(q[1],drivers[j]->name) == 0) break;

		if (drivers[j] == 0)
		{
		   fprintf(stderr,"Game \"%s\" not supported\n", q[1]);
		   return OSD_NOT_OK;
		}
		else
		{
		   game_index = j;
	   	}
		continue; 
	    } /* defaultgame */
#endif
	    if (strcasecmp(q[0], "x11joyname") == 0) {
		x11joyname=(char *)malloc(1+strlen(q[1]) );
		if( ! x11joyname ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(x11joyname,q[1]);
		continue;
	    } /* x11joyname */
#ifndef MESS
	    if (strcasecmp(q[0], "mamedir") == 0) {
		mamedir=(char *)malloc(1+strlen(q[1]) );
		if( ! mamedir ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(mamedir,q[1]);
		continue;
	    } /* mamedir */
#endif
	    if (strcasecmp(q[0], "audiodevice") == 0) {
		audiodevice=(char *)malloc(1+strlen(q[1]) );
		if( ! audiodevice ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(audiodevice,q[1]);
		continue;
	    } /* audiodevice */
	    if (strcasecmp(q[0], "spooldir") == 0) {
		spooldir=(char *)malloc(1+strlen(q[1]) );
		if( ! spooldir ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(spooldir,q[1]);
		continue;
	    } /* spooldir */
	    if (strcasecmp(q[0], "display") == 0) {
		displayname=(char *)malloc(1+strlen(q[1]) );
		if( ! displayname ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(displayname,q[1]);
		continue;
	    } /* displayname */
	    if (strcasecmp(q[0],"mapkey") == 0) {
		int from,to;
#ifdef ultrix
		/* ultrix sscanf() requires explicit leading of 0x for hex numbers */
		if ( sscanf(q[1],"0x%x,0x%x",&from,&to) != 2 )
#else
		if ( sscanf(q[1],"%x,%x",&from,&to) != 2 )
#endif
		    fprintf(stderr,"Line %d: Invalid keymapping %s. Ignoring...\n",lineno,q[1]);
		else    if (sysdep_mapkey(from,to) ) {
			    fprintf(stderr,"Line %d: invalid keymapping\n",lineno);
			    return (OSD_NOT_OK);
			}
		continue;
	    } /* mapkey */
	    if (strcasecmp(q[0],"samplefreq") == 0) {
		options.samplerate = atoi(q[1]);
		if (options.samplerate < 8000 || options.samplerate > 45454) {
			fprintf (stderr,"Illegal audio sample frequency (%d)\n", options.samplerate);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* sample freq */
	    if (strcasecmp(q[0],"timerfreq") == 0) {
		audio_timer_freq = atoi(q[1]);
		if (audio_timer_freq < 1 || audio_timer_freq > 100) {
			fprintf (stderr,"Illegal audio timer frequency (%d)\n", audio_timer_freq);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* timer freq */
	    if (strcasecmp(q[0],"videomode") == 0) {
		video_mode = atoi(q[1]);
		if (video_mode < 0 || video_mode > 2) {
			fprintf (stderr,"illegal videomode (%d)\n", video_mode);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* video mode */
	    if (strcasecmp(q[0],"throttle") == 0) {
		throttle = atoi(q[1]);
		continue;
	    } /* throttle */
#ifndef MESS
	    if (strcasecmp(q[0],"antialias") == 0) {
		antialias = atoi(q[1]);
		continue;
	    } /* antialias */
	    if (strcasecmp(q[0], "beam") == 0) {
	      double f_beam = atof(q[1]);
	      if (f_beam == 0.0) {
		fprintf (stderr, "Error: beam requires an argument.\n");
		return (OSD_NOT_OK);
	      }
	      beam = (int)(f_beam * 0x00010000);
	      if (beam < 0x00010000)
		beam = 0x00010000;
	      if (beam > 0x00100000)
		beam = 0x00100000;
	      continue;
	    }
	    
	    if (strcasecmp(q[0], "flicker") == 0) {
	      double f_flicker = atof(q[1]);
	      flicker = (int)(f_flicker * 2.55);
	      if (flicker < 0)
		flicker = 0;
	      if (flicker > 255)
		flicker = 255;
	      continue;
	    }
#endif

	    /* if arrives here means unrecognized command line */
	    fprintf(stderr,"Line %d: unknown command %s\n",lineno,q[0]);
	    fclose(file);
	    return (OSD_NOT_OK);
	} /* while */
	fclose(file);
	return (OSD_OK);
}

/*
 * Set all options to there defaults
 */

void set_default_options (void)
{
	bitmap			= NULL;
	key			= NULL;
	use_joystick            = FALSE;
	swapjoyaxis             = FALSE;
	analogstick		= FALSE;
	use_mouse               = FALSE;
	use_private_cmap	= FALSE;
	play_sound              = FALSE;
	No_FM			= TRUE;
	root_window_id		= 0;
	widthscale              = 1;
	heightscale             = 1;
	mit_shm_avail           = 0;
	snapshot_no             = 0;
	game_index		= 0;
	video_mode		= 0;
	black_pen		= 0;
#ifndef MESS
	beam                    = (int)(1.0 * 0x00010000);
	flicker                 = 0;
	antialias		= 0;
#endif
	force_truecolor		= FALSE;
	use_xsync		= TRUE;
	use_dirty		= TRUE;
	use_layer		= FALSE;
	palette_dirty		= TRUE;
	black_pen_valid		= FALSE;
	throttle		= TRUE;
	audio_timer_freq        = AUDIO_TIMER_FREQ;
#ifndef MESS
	mamedir                 = MAMEDIR;
#endif
	spooldir                = SPOOLDIR;
	displayname             = DISPLAY;
	audiodevice		= "/dev/dsp";
	towns_pad_dev		= "/dev/pad00";
	options.frameskip	= 0;
#ifndef MESS
	options.norotate	= FALSE;
	options.rol		= FALSE;
	options.ror		= FALSE;
	options.flipx		= FALSE;
	options.flipy		= FALSE; 
#endif
	options.samplerate	= 22050;
	options.samplebits	= 8;
	options.cheat		= FALSE;
	options.mame_debug	= FALSE;
#ifdef X11_JOYSTICK
	x11joyname              = X11_JOYNAME;
#else
	x11joyname		= "";
#endif
#ifndef svgalib
#ifndef ggi
	display			= NULL;
	image			= NULL;
	window			= 0;
	dirty_lines		= NULL;
	old_dirty_lines		= NULL;
	copybuffer		= NULL;
	
	memset(xpixel_allocated, FALSE , sizeof(char) * 256);
#endif
#endif
}

int parse_command_line (int argc, char *argv[])
{
	int i, got_gamename	= 0;
	int showconfig		= 0;
	int showusage		= 0;
#ifdef MESS
	int num_roms		= 0;
	int num_floppies	= 0;
#endif
	
	/* parse argument invocation. */
	for (i = 1;i < argc;i++)
	{
		if (strcasecmp(argv[i], "-help") == 0) { 
			showusage = 1;
			continue;
		}
		if (strcasecmp(argv[i], "-h") == 0) {
			showusage = 1;
			continue;
		}
		if (strcasecmp(argv[i], "-?") == 0) {
			showusage = 1;
			continue;
		}
		if (strcasecmp(argv[i], "-master") == 0) {
			if (netstate == SLAVE)
			{
				fprintf(stderr, "Can't be Slave and Master\n");
				return(OSD_NOT_OK);
			}
			netstate = MASTER;
			players = atoi(argv[++i]);
			if (players > 4)
			  {
			    fprintf(stderr, "4 players max.\n");
			    return(OSD_NOT_OK);
			  }
			printf("XMame in network Master Mode\nWaiting for %d players.\n", players-1);
			continue;
		}
		if (strcasecmp(argv[i], "-slave") == 0) {
			if (netstate == MASTER)
			  {
			    fprintf(stderr, "Can't be Slave and Master\n");
			    return(OSD_NOT_OK);
			  }
			netstate = SLAVE;
			mastername = argv[++i];
			printf("Slave Mode; Registering to Master %s\n", mastername);
			continue;
		}
	        if (strcasecmp(argv[i], "-netmapkey") == 0) {
			netkeymap = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-pad") == 0) {
#ifdef LIN_FM_TOWNS
			use_joystick = TRUE; /* a joystick like device */
#else
			fprintf(stderr,"FM-TOWNS game pad support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-nopad") == 0) {
#ifdef LIN_FM_TOWNS
			use_joystick = FALSE; /* gamepad is joystick-like */
#else
			fprintf(stderr,"FM-TOWNS game pad support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-mouse") == 0) {
#ifdef USE_MOUSE
			use_mouse = TRUE;
#else
			fprintf(stderr,"Mouse support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-nomouse") == 0) {
#ifdef USE_MOUSE
			use_mouse = FALSE;
#else
			fprintf(stderr,"Mouse support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-joy") == 0) {
#ifdef USE_JOYSTICK
			use_joystick = TRUE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-nojoy") == 0) {
#ifdef USE_JOYSTICK
			use_joystick = FALSE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-swapjoyaxis") == 0) {
#ifdef USE_JOYSTICK
			swapjoyaxis = TRUE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-noswapjoyaxis") == 0) {
#ifdef USE_JOYSTICK
			swapjoyaxis = FALSE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-analogstick") == 0) {
#ifdef USE_JOYSTICK
			analogstick = TRUE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-noanalogstick") == 0) {
#ifdef USE_JOYSTICK
			analogstick = FALSE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-sound") == 0) {
			play_sound        = TRUE;
			audio_timer_freq  = AUDIO_TIMER_FREQ;
			continue;
		}
		if (strcasecmp(argv[i], "-usedsp") == 0) {
			audiodevice        = "/dev/dsp";
#ifndef linux
			fprintf(stderr,"Warning: \"-usedsp\" option ignored\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-useaudio") == 0) {
			audiodevice        = "/dev/audio";
#ifndef linux
			fprintf(stderr,"Warning: \"-usedsp\" option ignored\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-nosound") == 0) {
			play_sound        = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-fakesound") == 0) {
			fake_sound        = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-forcetruecolor") == 0) {
			force_truecolor = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-noforcetruecolor") == 0) {
			force_truecolor = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-xsync") == 0) {
			use_xsync = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-noxsync") == 0) {
			use_xsync = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-dirty") == 0) {
			use_dirty = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-nodirty") == 0) {
			use_dirty = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-privatecmap") == 0) {
			use_private_cmap = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-noprivatecmap") == 0) {
			use_private_cmap = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-widthscale") == 0) {
			widthscale = atoi (argv[++i]);
			if (widthscale <= 0) {
				fprintf (stderr,"illegal widthscale (%d)\n", widthscale);
				return (OSD_NOT_OK);
			}
			continue;
		}
		if (strcasecmp(argv[i], "-heightscale") == 0) {
			heightscale = atoi (argv[++i]);
			if (heightscale <= 0) {
				fprintf (stderr,"illegal heightscale (%d)\n", heightscale);
				return (OSD_NOT_OK);
			}
			continue;
		}
		if (strcasecmp(argv[i], "-frameskip") == 0) {
			options.frameskip = atoi (argv[++i]);
			if (options.frameskip < 0) {
				fprintf (stderr,"illegal frameskip value (%d)\n", options.frameskip);
				return (OSD_NOT_OK);
			}
			if (options.frameskip > 3) options.frameskip = 3;
			continue;
		}
		if (strcasecmp(argv[i], "-scale") == 0) {
			heightscale = widthscale = atoi (argv[++i]);
			if (heightscale <= 0) {
				fprintf (stderr,"illegal scale factor (%d)\n", heightscale);
				return (OSD_NOT_OK);
			}
			continue;
		}
		if (strcasecmp(argv[i], "-mitshm") == 0) {
			fprintf(stderr,"MIT Shared Memory use enabled\n");
			mit_shm_avail=1;
			continue;
		}
		if (strcasecmp(argv[i], "-nomitshm") == 0) {
			fprintf(stderr,"MIT Shared Memory use disabled\n");
			mit_shm_avail=-1;
			continue;
		}
		if (strcasecmp(argv[i], "-root_window_id") == 0) {
			/* strtol allows for e.g. 0x3c0000 */
			root_window_id = strtol (argv[++i], NULL, 0);
			continue;
		}
		if (strcasecmp(argv[i], "-showconfig") == 0) {
			showconfig=TRUE;
			continue;
		}
#ifndef MESS
		if (strcasecmp(argv[i], "-mamedir") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -mamedir option\n");
				return (OSD_NOT_OK);
			}
			mamedir=argv[++i];
			continue;
		}
#endif
		if (strcasecmp(argv[i], "-spooldir") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -spooldir option\n");
				return (OSD_NOT_OK);
			}
			spooldir=argv[++i];
			continue;
		}
		if (strcasecmp(argv[i], "-paddevname") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -paddevname option\n");
				return (OSD_NOT_OK);
			}
			towns_pad_dev=argv[++i];
			continue;
		}
		if (strcasecmp(argv[i], "-x11joyname") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -x11joyname option\n");
				return (OSD_NOT_OK);
			}
			x11joyname=argv[++i];
			continue;
		}
		if (strcasecmp(argv[i], "-display") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -display option\n");
				return (OSD_NOT_OK);
			}
			displayname=argv[++i];
			continue;
		}
		if (strcasecmp(argv[i],"-mapkey") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -mapkey option\n");
				return (OSD_NOT_OK);
			} else {
				int from,to;
				if ( sscanf(argv[++i],"%x,%x",&from,&to) != 2 )
					fprintf(stderr,"Invalid keymapping %s. Ignoring...\n",argv[i]);
				else    sysdep_mapkey(from,to);
			}
			continue;
		}
		if (strcasecmp(argv[i], "-samplefreq") == 0) {
			options.samplerate = atoi (argv[++i]);
			if (options.samplerate < 8000 || options.samplerate > 45454) {
				fprintf (stderr,"illegal audio sample frequention (%d)\n", options.samplerate);
				return (OSD_NOT_OK);
			}
			continue;
		}

		if (strcasecmp(argv[i], "-fm") == 0) {
		  fprintf (stderr, "Sorry, FM not supported at this point.\n");
		  continue;
		}

		if (strcasecmp(argv[i], "-nofm") == 0) {
		  fprintf (stderr, "Sorry, FM not supported at this point (-nofm has no effect).\n");
		  continue;
		}
		if (strcasecmp(argv[i], "-throttle") == 0) {
		  throttle = TRUE;
		  continue;
		}
		if (strcasecmp(argv[i], "-nothrottle") == 0) {
		  fprintf (stdout, "Turning off speed throttling....\n");
		  throttle = FALSE;
		  continue;
		}
#ifndef MESS
		if (strcasecmp(argv[i], "-antialias") == 0) {
		  antialias = 1;
		  continue;
		}
		if (strcasecmp(argv[i], "-noantialias") == 0) {
		  antialias = 0;
		  continue;
		}
		if (strcasecmp(argv[i], "-beam") == 0) {
		  double f_beam = atof(argv[++i]);
		  if (f_beam == 0.0) {
		    fprintf (stderr, "Error: -beam requires an argument.\n");
		    return (OSD_NOT_OK);
		  }
		  beam = (int)(f_beam * 0x00010000);
		  if (beam < 0x00010000)
		    beam = 0x00010000;
		  if (beam > 0x00100000)
		    beam = 0x00100000;
		  fprintf (stdout, "Setting beam size to %f\n", f_beam);
		  continue;
		}

		if (strcasecmp(argv[i], "-flicker") == 0) {
		  double f_flicker = atof(argv[++i]);
		  flicker = (int)(f_flicker * 2.55);
		  if (flicker < 0)
		    flicker = 0;
		  if (flicker > 255)
		    flicker = 255;
		  fprintf (stdout, "Setting flicker to %f (maps to %d)\n", f_flicker, flicker);
		  continue;
		}
#endif

		if (strcasecmp(argv[i], "-videomode") == 0) {
			video_mode = atoi (argv[++i]);
			if (video_mode < 0 || video_mode > 2) {
				fprintf (stderr,"illegal videomode (%d)\n", video_mode);
				return (OSD_NOT_OK);
			}
			continue;
		}
#ifndef MESS
		if (strcasecmp(argv[i],"-norotate") == 0) {
			options.norotate=TRUE;
			continue;
		}
		if (strcasecmp(argv[i],"-rol") == 0)	{
			options.rol=TRUE;
			continue;
		}
		if (strcasecmp(argv[i],"-ror") == 0)	{
			options.ror=TRUE;
			continue;
		}
		if (strcasecmp(argv[i],"-flipx") == 0) {
			options.flipx=TRUE;
			continue;
		}
		if (strcasecmp(argv[i],"-flipy") == 0) {
			options.flipy=TRUE;
			continue;
		}
#endif
		if (strcasecmp(argv[i],"-cheat") == 0) {
			options.cheat=TRUE;
			continue;
		}
		if (strcasecmp(argv[i],"-debug") == 0) {
			options.mame_debug=TRUE;
			continue;
		}
		if (strcasecmp(argv[i],"-log") == 0) {
			errorlog = options.errorlog = fopen("error.log","wa");
			continue;
		}
		if (strcasecmp(argv[i],"-record") == 0) {
			i++;
			if (i < argc)
			   options.record = osd_fopen(drivers[game_index]->name,argv[i],OSD_FILETYPE_INPUTLOG,1);
   			else
			{
			   fprintf(stderr, "-record requires an additional argument\n");
			   return OSD_NOT_OK;
			}
			continue;
		}

		if (strcasecmp(argv[i],"-playback") == 0) {
			i++;
			if (i < argc)
			   options.playback = osd_fopen(drivers[game_index]->name,argv[i],OSD_FILETYPE_INPUTLOG,0);
			else
			{
			   fprintf(stderr, "-playback requires an additional argument\n");
			   return OSD_NOT_OK;
			}
			continue;
		}
		
		/* unknown argument */
		if (argv[i][0]=='-')
		{
			fprintf(stderr,"Unknown option. Try %s -help\n",argv[0]);
			return (OSD_NOT_OK);
		}
		else
		{  /* if arrives here and not for the first time, 
		      means syntax error */
		   if (!got_gamename) /* notice: for MESS game means system */
		   {
			int j;
			/* do we have a drivers for this? */
			for (j = 0; drivers[j]; j++)
			   if (strcasecmp(argv[i],drivers[j]->name) == 0) break;

			if (drivers[j] == 0)
			{
#ifdef MESS
			   fprintf(stderr,"System \"%s\" not supported\n", argv[i]);
#else
			   fprintf(stderr,"Game \"%s\" not supported\n", argv[i]);
#endif
			   return OSD_NOT_OK;
			}
			else
			{
			   game_index = j;
			   got_gamename=1;
		   	}
		   }
		   else
#ifdef MESS
		   {
			/* Is it a floppy or a rom? */
			if ((strstr(argv[i],".dsk")!=NULL) || (strstr(argv[i],".DSK")!=NULL))
			{
				if (num_floppies >= MAX_FLOPPY)
				{
					printf("Too many floppy names specified!\n");
					return 1;
				}
				strcpy(options.floppy_name[num_floppies++],argv[i]);
				if (errorlog) fprintf(errorlog,"Using floppy image %s\n",argv[i]);
			}
			else
			{
				if (num_roms >= MAX_ROM)
				{
					printf("Too many image names specified!\n");
					return 1;
				}
				strcpy(options.rom_name[num_roms++],argv[i]);
				if (errorlog) fprintf(errorlog,"Using image %s\n",argv[i]);
			}
		   }
#else
		   {
			fprintf(stderr,"Duplicate gamename: %s\n",argv[i]);
			return (OSD_NOT_OK);
		   }
#endif
		}
	}
	
	if (showconfig)
	{
		show_config();
		return OSD_NOT_OK;
	}
	
#ifdef MESS
	if (showusage || !got_gamename || (!num_floppies && !num_roms))
#else
	if (showusage)
#endif
	{
		show_usage();
		return OSD_NOT_OK;
	}
	
	return (OSD_OK);
}
