/*
 * X-Mame system independent sound code
 */

#define __SOUND_C_

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef USE_TIMER
#include <sys/time.h>
#include <signal.h>
static int bytes_per_timer_alarm;

/* Set of signals to block during critical sections */
static sigset_t blockingset;
#endif

#include "xmame.h"
#include "sound.h"

/* #define SOUND_DEBUG */
static float master_volume_divider;
static int sound_active = 0;
VOICE_T voices[AUDIO_NUM_VOICES];
static unsigned char *buf=NULL;
static signed int *bigbuf=NULL;

/*********** code to quick alloc/free SAMPLE_T items **************
 * malloc/free is a too cpu-expensive process. When using a fixed size
 * malloc/free scheme , it's better to pre-allocate space and use this
 * algorithm
 *
 *   In our code SAMPLE_T is ideal for a quick malloc scheme. Of course
 * this is not applicable to data samples ( not fixed size ... )
 *
 * The SAMPLE_T code caches data blocks, which firstly reduces the
 * number of malloc() calls and secondly means we need not call free()
 * on memory allocated within SAMPLE_T.	 This is absolutely necessary
 * with timer based sound as there is no guarantee that a systems
 * malloc/free library is reentrant so free() must not be called from
 * within a signal handler.
 *
 * The cost of caching data blocks is to increase memory usage
 * slightly.
 *
 **************************************************************************/
 
/* this should be enough ....*/
#define ALLOC_TABLE_SIZE 64

/* Heuristic for deciding whether to shrink an existing allocated data
   block.  This should be a reasonably large number.  */
#define LARGE_SAMPLE 0x10000

static SAMPLE_T		SampleTTable[ALLOC_TABLE_SIZE];
static int		SampleTIndex[ALLOC_TABLE_SIZE];
static int		SampleTPointer;

int InitSampleTTable(void) {
    int i;
    for (i=0;i<ALLOC_TABLE_SIZE;i++) {
      SampleTIndex[i]=i;
      SampleTTable[i].data = NULL;
      SampleTTable[i].datasize = 0;
    }
    SampleTPointer=(ALLOC_TABLE_SIZE - 1 );
    return 0;
}

/* FreeSampleT is called from within the signal handler with timer
   based audio, so care must be taken that it does not modify any data
   that is not appropriately protected from access outside the signal
   handler */
static int FreeSampleT(SAMPLE_T *pt) {
    int item = (int) ( pt - SampleTTable );
    if (SampleTPointer>=(ALLOC_TABLE_SIZE-1) ){
	fprintf(stderr_file,"FreeSampleT() consistency error\n");
	exit(1);
    }
    SampleTIndex[++SampleTPointer] = item;

    return 0;
}

static SAMPLE_T	 *AllocSampleT(size_t blocksize) {
    SAMPLE_T *pt;

    if (SampleTPointer<0) return (SAMPLE_T *) NULL; /* no items available */
    pt = &SampleTTable[SampleTIndex[SampleTPointer]];

    /* Allocate new memory if the old memory block was too small or
       far larger than we require */
    if (pt->datasize < blocksize || pt->datasize > blocksize + LARGE_SAMPLE) {
      signed char *newblock;

#ifdef SOUND_DEBUG
      fprintf(stderr_file, "AllocSampleT(): New audio memory block, size %ld (oldsize=%ld)\n", (long)blocksize, (long)pt->datasize);
#endif

      newblock = (signed char *)malloc(blocksize);

      if (newblock == NULL) {
	/* malloc() failed */

	if (pt->datasize < blocksize)
	  /* If malloc() failed and the current block is too small, we
	     fail the operation */
	  return (SAMPLE_T *) NULL;
      } else {
	/* malloc() succeeded */
	if (pt->data != NULL)
	  free(pt->data);

	pt->data = newblock;
	pt->datasize = blocksize;
      }
    }

    SampleTPointer--;
    return pt;
}
   
/****************************************************************************
*
* Timer based sound routines
*
* Used in those systems that doesn't allow how many bytes can be send to the
* audio device without blocking
* 
*****************************************************************************/

#ifdef USE_TIMER
struct sigaction sig_action;

int start_timer() {
	struct itimerval timer_value ;
	void audio_timer(int arg);
	/* set and configure timer catcher */
	if (!play_sound) return OSD_OK;
	sig_action.sa_handler = audio_timer;
#ifdef hpux
	sig_action.sa_flags   = SA_RESETHAND;
#else
	sig_action.sa_flags   = SA_RESTART;
#endif

	if(sigaction (SIGALRM, &sig_action, NULL)< 0) {
		fprintf(stderr_file,"Sigaction Failed \n");
		return OSD_NOT_OK;
	}
	/* set and configure realtime alarm */
  	timer_value.it_interval.tv_sec =
 	timer_value.it_value.tv_sec    = 0L;
  	timer_value.it_interval.tv_usec=
  	timer_value.it_value.tv_usec   = 1000000L / audio_timer_freq;

	if (setitimer (ITIMER_REAL, &timer_value, NULL)) {
 		fprintf(stderr_file, "Setting the timer failed.\n");
  		return OSD_NOT_OK;
  	}
	return OSD_OK;
}

void audio_timer (int arg) {
        static int  in = FALSE;
        void mix_sound(void);
	if (!play_sound) return;
        if (!in) {
            in = TRUE;
            mix_sound();
            in = FALSE;
        } /* else {
          fprintf(stderr_file, ".");
        } */
#if defined solaris || defined irix
	/* rearm alarm timer */
	sigaction (SIGALRM, &sig_action, NULL);
#endif
}

#endif  /* ifdef USE_TIMER */
/*****************************************************************************
* 
* NON-Timer based sound routines
*
* the idea is simple: I inquire audio driver how many bytes can i send, 
* parse data through the mixer and send it to audio (or if the system is
* timer based, wait for a timer irq to arrive )
*
* Modified and hacked up for mame and sound by
*  Bill Lash - lash@tellabs.com
******************************************************************************/

#ifdef USE_TIMER
void osd_update_audio(void) /* make osd_update_audio do nothing, name it */
{                           /* mix_sound which is called by audio_timer  */
}                           /* a bit dirty but it does what we want.     */

void mix_sound(void)
#else
void osd_update_audio(void)
#endif
{
   int i,j;
   long size;
   SAMPLE_T *new_sample;
   signed char *current_data_pt;
   signed char *end_data_pt;
   float vol, pos_frac, freq_fac;
   static int warn=1;
   
   if ((play_sound == 0) || (sound_active == 0))  return;
   
   osd_profiler(OSD_PROFILE_SOUND);
   
#ifdef USE_TIMER
   size=bytes_per_timer_alarm;
#else
   if( (size=sysdep_get_audio_freespace()) <= 256) return;

   if (size > AUDIO_BUFF_SIZE)
   {
      size = AUDIO_BUFF_SIZE;
      if (warn)
      {
         warn=0;
         fprintf(stderr_file, "WARNING size > AUDIO_BUFF_SIZE in osd_update_audio\n"
                         "Sound quality might be worse then usual\n"
                         "please mail j.w.r.degoede@et.tudelft.nl about this message\n"
                         "include your makefile, machine+os and sounddriver version\n");
      }
   }
#endif

   memset(bigbuf, 0, size*sizeof(int));
   sound_active = 0;   

   for(j=0;j<AUDIO_NUM_VOICES;j++) {
      SAMPLE_T *pt = voices[j].sample;
      if (pt) {
         current_data_pt = voices[j].current_data_pt;
         pos_frac        = voices[j].pos_frac;
         end_data_pt     = pt->end_data_pt;
         vol             = pt->vol;
         freq_fac        = pt->freq_fac;
         sound_active    = 1;
      }
      else continue;   
      
      for (i=0;i<size;i++)
      {
#ifdef FANCY_SOUND	    
         bigbuf[i] += ( *current_data_pt + 
	  ( *(current_data_pt + 1) - *current_data_pt ) * pos_frac ) * vol;
#else
         bigbuf[i] += *current_data_pt * vol;
#endif
         pos_frac += freq_fac;
         while(pos_frac > 1) {
            pos_frac--;
            current_data_pt++;
         }
         if ( current_data_pt > end_data_pt ) {
            if( pt->loop_stream ) {
               if( pt->next_sample ) {
                  new_sample = pt->next_sample;
                  FreeSampleT(pt);
                  voices[j].sample = pt = new_sample;
                  end_data_pt = pt->end_data_pt;
                  vol         = pt->vol;
               }
               current_data_pt=pt->data;
            } else {
               osd_stop_sample(j);
               i = size;
            }
         } /* if at end of sample */
      } /* for i=0;i<size;i++) */
      voices[j].current_data_pt = current_data_pt;
      voices[j].pos_frac = pos_frac;
   } /* for j< AUDIO_NUM_VOICES */
   
#ifdef SOUND_DEBUG
   fprintf(stderr_file,"osd_update_audio(): %d avail %d used\n",size,i);
#endif

   if (sound_active)
   {
      int tmp;
      for (i=0;i<size;i++) {
	 tmp = bigbuf[i] + 128;
	 if (tmp>255) 
	    buf[i] = 255;
	 else if (tmp<0) 
	    buf[i] = 0;
	 else 
	    buf[i] = tmp;
      }
      sysdep_play_audio(buf,size);
   }
   osd_profiler(OSD_PROFILE_END);
}

/****************************************************************************
* 
* General sound routines
*
*****************************************************************************/

int sysdep_audio_initvars(void)
{
	int i;
	
	if (!play_sound ) return OSD_OK;
	
	osd_set_mastervolume(attenuation);
#ifdef USE_TIMER
	bytes_per_timer_alarm=options.samplerate/audio_timer_freq;
	if (bytes_per_timer_alarm > AUDIO_BUFF_SIZE)
	{
	   fprintf(stderr_file,"Error: bytes/timertick > audio-buf-size\n");
	   fprintf(stderr_file,"This is a configure error, try reasing fragsize (see -help)\n");
	   return OSD_NOT_OK;
	}

	/* Initialise the signal set to block during critical sections */
	sigemptyset(&blockingset);
	sigaddset(&blockingset, SIGALRM);
#ifndef svgalib
	/* in svgalib mode must start timer AFTER open display */
	if (play_sound)
	   if (start_timer()==OSD_NOT_OK) return(OSD_NOT_OK);
#endif
#endif
	buf    = malloc(frag_size * num_frags * sizeof(unsigned char));
	bigbuf = malloc(frag_size * num_frags * sizeof(signed int));
	if (!buf || !bigbuf)
	{
		if (buf) free(buf);
		if (bigbuf) free(buf);
		fprintf(stderr_file, "Error: couldn't allocate sound-buffers\n");
		return OSD_NOT_OK;
	}
	InitSampleTTable();
        for (i = 0; i < AUDIO_NUM_VOICES; i++) {
		voices[i].sample = NULL;
		voices[i].current_data_pt = NULL;
		voices[i].pos_frac = 0;
        }
	return OSD_OK;
}

void sysdep_audio_close(void)
{
	if (buf) free(buf);
	if (bigbuf) free(buf);
}

/* attenuation in dB */
void osd_set_mastervolume(int _attenuation)
{
#ifdef USE_TIMER
	sigset_t oset;
#endif
	int channel;
	float old_master_volume_divider;
	float attenuation_mult = 1.0;
	
	if (! play_sound ) return;
	
	attenuation = _attenuation;
	
	while (_attenuation++ < 0)
		attenuation_mult /= 1.122018454;	/* = (10 ^ (1/20)) = 1dB */
	
        old_master_volume_divider=master_volume_divider;
        master_volume_divider=(float)256 / attenuation_mult;
        
#ifdef USE_TIMER
	sigprocmask(SIG_BLOCK, &blockingset, &oset);
#endif

	for (channel=0; channel < AUDIO_NUM_VOICES; channel++) {
	   SAMPLE_T *pt=voices[channel].sample;
	   for(;pt;pt=pt->next_sample) 
	      pt->vol *= old_master_volume_divider/master_volume_divider;
	}

#ifdef USE_TIMER
	if (!sigismember(&oset, SIGALRM))
	  sigprocmask(SIG_UNBLOCK, &blockingset, NULL);
#endif
}

int osd_get_mastervolume(void)
{
	return attenuation;
}

void osd_sound_enable(int enable_it)
{
	static int orig_attenuation;

	if (enable_it)
	{
		osd_set_mastervolume(orig_attenuation);
	}
	else
	{
		/* taken from dos-port ?? maybe should be: */
		/* if (osd_get_mastervolume() > -40) */
		if (osd_get_mastervolume() != -40)
		{
			orig_attenuation = osd_get_mastervolume();
			osd_set_mastervolume(-48);
		}
	}
}


void osd_play_sample_generic(int channel,signed char *data,int len,int freq,int volume,int loop, int bits)
{
#ifdef USE_TIMER
	sigset_t oset;
#endif
	SAMPLE_T *new_samp;

        if ((play_sound == 0) || (channel >= AUDIO_NUM_VOICES)) return;
	if ((freq<10) || (freq> 100000)) 
	{ 
#ifdef SOUND_DEBUG
	fprintf(stderr_file,"osd_play_sample() call Ignored: chan:%d frec:%d vol:%d\n",channel,freq,volume);
#endif
	   return;
	}

#ifdef USE_TIMER
	sigprocmask(SIG_BLOCK, &blockingset, &oset);
#endif

	osd_stop_sample(channel);

	if (!(new_samp = AllocSampleT(len+1))) {
#ifdef USE_TIMER
	  if (!sigismember(&oset, SIGALRM))
	    sigprocmask(SIG_UNBLOCK, &blockingset, NULL);
#endif
		return;
	}

	sound_active=1;

	voices[channel].sample = new_samp;

#ifdef FANCY_SOUND
        /* calculate one sample more as the actual length for interpolation */
	if (loop)
	{
	  if (bits==8)
	    new_samp->data[len]=data[0];
	  else
	    new_samp->data[len]=((short *)data)[0] >> 8;
	}
	else new_samp->data[len]=0;
#endif	 
	voices[channel].current_data_pt = new_samp->data;
	new_samp->end_data_pt = new_samp->data + len - 1;
	voices[channel].pos_frac = 0;
	new_samp->next_sample = (SAMPLE_T *) NULL;
	new_samp->loop_stream = loop;
	new_samp->vol = (float)volume / master_volume_divider;
	new_samp->freq_fac = (float)freq / options.samplerate;
	if (bits==8) /* normal 8 bit data */
	{
	   memcpy(new_samp->data,data,len);
	}
	else         /* bummer 16 bit data, make it 8 bit since we don't */
	{            /* support 16 bit (yet ?) */
	   int i;
	   signed short *src=(signed short *)data;
	   for(i=0;i<len;i++)
	      /* we can shift safely since we're dividing and thus keep 
	         the signbit */
	      new_samp->data[i] = src[i] >> 8;
	}
#ifdef SOUND_DEBUG
        fprintf(stderr_file,"play() chan:%d len:%d frec:%d vol:%d loop:%d\n",channel,len,freq,volume,loop); 
#endif

#ifdef USE_TIMER
	if (!sigismember(&oset, SIGALRM))
	  sigprocmask(SIG_UNBLOCK, &blockingset, NULL);
#endif

	return;
}

void osd_play_sample(int channel,signed char *data,int len,int freq,int volume,int loop)
{
	osd_play_sample_generic(channel, data, len, freq, volume, loop, 8);
}

void osd_play_sample_16(int channel,signed short *data,int len,int freq,int volume,int loop)
{
	osd_play_sample_generic(channel,(signed char *)data, len/2, freq, volume, loop, 16);
}

void osd_play_streamed_sample_generic(int channel,signed char *data,int len,int freq,int volume,int bits)
{
#ifdef USE_TIMER
	sigset_t oset;
#endif

	SAMPLE_T *new_samp, *last_samp=NULL;

        if ((play_sound == 0) || (channel >= AUDIO_NUM_VOICES)) return;
	if ((freq<10) || (freq> 100000)) 
	{ 
#ifdef SOUND_DEBUG
	fprintf(stderr_file,"play_streamed_sample() call Ignored: chan:%d frec:%d vol:%d\n",channel,freq,volume);
#endif
	   return;
	}

#ifdef USE_TIMER
	sigprocmask(SIG_BLOCK, &blockingset, &oset);
#endif

	if (!(new_samp = AllocSampleT(len+1))) {

#ifdef USE_TIMER
	  if (!sigismember(&oset, SIGALRM))
	    sigprocmask(SIG_UNBLOCK, &blockingset, NULL);
#endif

	  return;
	}

        sound_active=1;

	if(voices[channel].sample == NULL) {
	    voices[channel].sample = new_samp;
	} else {
	    last_samp = voices[channel].sample;
	    while (last_samp->next_sample) last_samp=last_samp->next_sample;
	    last_samp->next_sample = new_samp;
	}
#ifdef FANCY_SOUND
	if (bits==16) /* make short sample a char sample */
	   data[0]=((short *)data)[0] >> 8;
        /* calculate one sample more as the actual length for interpolation */
	new_samp->data[len]=data[0];
	if(last_samp) 
	   *(last_samp->end_data_pt + 1)=data[0];
	else
#else
	if(last_samp == NULL)
#endif
	   voices[channel].current_data_pt=new_samp->data;
	new_samp->loop_stream = 1;
	new_samp->next_sample = NULL;
	new_samp->end_data_pt = new_samp->data + len - 1;
	new_samp->vol = (float)volume / master_volume_divider;
	new_samp->freq_fac = (float)freq / options.samplerate;
	if (bits==8) /* normal 8 bit data */
	{
	   memcpy(new_samp->data,data,len);
	}
	else         /* bummer 16 bit data, make it 8 bit since we don't */
	{            /* support 16 bit (yet ?) */
	   int i;
	   signed short *src=(signed short *)data;
	   for(i=0;i<len;i++)
	      /* we can shift safely since we're dividing and thus keep 
	         the signbit */
	      new_samp->data[i] = src[i] >> 8;
	}

#ifdef USE_TIMER
	if (!sigismember(&oset, SIGALRM))
	  sigprocmask(SIG_UNBLOCK, &blockingset, NULL);
#endif
}

void osd_play_streamed_sample(int channel,signed char *data,int len,int freq,int volume)
{
	osd_play_streamed_sample_generic(channel, data, len, freq, volume, 8);
}

void osd_play_streamed_sample_16(int channel,signed short *data,int len,int freq,int volume)
{
	osd_play_streamed_sample_generic(channel,(signed char *)data, len/2, freq, volume, 16);
}

void osd_adjust_sample(int channel,int freq,int volume)
{
#ifdef USE_TIMER
	sigset_t oset;
#endif

	SAMPLE_T *next_samp;
        if (play_sound == 0 || channel >= AUDIO_NUM_VOICES) return;
	if ((freq<10) || (freq> 100000)) 
	{ 
#ifdef SOUND_DEBUG
	fprintf(stderr_file,"osd_adjust_sample() call Ignored: chan:%d frec:%d vol:%d\n",channel,freq,volume);
#endif
	   return;
	}
	
	if ( freq==-1 && volume==-1 ) return;

#ifdef USE_TIMER
	sigprocmask(SIG_BLOCK, &blockingset, &oset);
#endif

	if(voices[channel].sample != NULL) {
	    next_samp = voices[channel].sample;
	    while(next_samp != NULL) {
	    	if (volume != -1)
	    	   next_samp->vol = (float)volume / master_volume_divider;
	    	if (freq   != -1)
	    	   next_samp->freq_fac = (float)freq / options.samplerate;
	    	next_samp = next_samp->next_sample;
	    }
	}

#ifdef USE_TIMER
	if (!sigismember(&oset, SIGALRM))
	  sigprocmask(SIG_UNBLOCK, &blockingset, NULL);
#endif
}

void osd_stop_sample(int channel)
{
#ifdef USE_TIMER
	sigset_t oset;
#endif

	SAMPLE_T *next_samp;
	SAMPLE_T *curr_samp;
        if (play_sound == 0 || channel >= AUDIO_NUM_VOICES) return;

#ifdef USE_TIMER
	sigprocmask(SIG_BLOCK, &blockingset, &oset);
#endif

	if(voices[channel].sample != NULL) {
	    next_samp = voices[channel].sample;
	    voices[channel].sample = NULL;
	    voices[channel].current_data_pt = NULL;
	    voices[channel].pos_frac = 0;
	    while(next_samp != NULL) {
	    	curr_samp = next_samp;
	    	next_samp = next_samp->next_sample;
	    	FreeSampleT(curr_samp);
	    }
	}

#ifdef USE_TIMER
	if (!sigismember(&oset, SIGALRM))
	  sigprocmask(SIG_UNBLOCK, &blockingset, NULL);
#endif
}

void osd_restart_sample(int channel)
{
#ifdef USE_TIMER
	sigset_t oset;
#endif

	if (play_sound == 0 || channel >= AUDIO_NUM_VOICES) return;

#ifdef USE_TIMER
	sigprocmask(SIG_BLOCK, &blockingset, &oset);
#endif

	voices[channel].current_data_pt = voices[channel].sample->data;

#ifdef USE_TIMER
	if (!sigismember(&oset, SIGALRM))
	  sigprocmask(SIG_UNBLOCK, &blockingset, NULL);
#endif
}

int osd_get_sample_status(int channel)
{
        int stopped=0;
        
	if (play_sound == 0 || channel >= AUDIO_NUM_VOICES) return -1;
	if (voices[channel].sample == NULL) stopped=1;

        return stopped;
}

/* handle ugly dos fm-stuff */

void osd_ym2203_write(int n, int r, int v) {
}

void osd_ym2203_update(void) {
}

int osd_ym3812_status(void)
{
	return 0;
}

int osd_ym3812_read(void)
{
	return 0;
}

void osd_ym3812_control(int reg)
{
}

void osd_ym3812_write(int data)
{
}
