/*
 * Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.
 *
 * (c) Copyright 1996, 1997, 1998 Gary Henderson (gary@daniver.demon.co.uk) and
 *                                Jerremy Koot (jkoot@snes9x.com)
 *
 * Super FX C emulator code 
 * (c) Copyright 1997, 1998 Ivar (Ivar@snes9x.com) and
 *                          Gary Henderson.
 * Super FX assembler emulator code (c) Copyright 1998 zsKnight and _Demo_.
 *
 * DSP1 emulator code (c) Copyright 1998 Ivar, _Demo_ and Gary Henderson.
 * DOS port code contains the works of other authors. See headers in
 * individual files.
 *
 * Snes9x homepage: www.snes9x.com
 *
 * Permission to use, copy, modify and distribute Snes9x in both binary and
 * source form, for non-commercial purposes, is hereby granted without fee,
 * providing that this license information and copyright notice appear with
 * all copies and any derived work.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event shall the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Snes9x is freeware for PERSONAL USE only. Commercial users should
 * seek permission of the copyright holders first. Commercial use includes
 * charging money for Snes9x or software derived from Snes9x.
 *
 * The copyright holders request that bug fixes and improvements to the code
 * should be forwarded to them so everyone can benefit from the modifications
 * in future versions.
 *
 * Super NES and Super Nintendo Entertainment System are trademarks of
 * Nintendo Co., Limited and its subsidiary companies.
 */

#include "snes9x.h"

#include "memmap.h"
#include "ppu.h"
#include "cpuexec.h"
#include "missing.h"
#include "dma.h"
#include "apu.h"
#include "gfx.h"

extern int HDMA_ModeByteCounts [8];
extern uint8 *HDMAMemPointers [8];
extern uint8 *HDMABasePointers [8];

/**********************************************************************************************/
/* S9xDoDMA()                                                                                   */
/* This function preforms the general dma transfer                                            */
/**********************************************************************************************/
void S9xDoDMA (uint8 Channel)
{
    uint8 Work;

    if (Channel > 7 || CPU.InDMA)
	return;

    CPU.InDMA = TRUE;
    SDMA *d = &DMA[Channel];

    switch (d->BAddress)
    {
    case 0x18:
    case 0x19:
	if (IPPU.RenderThisFrame)
	    FLUSH_REDRAW ();
	break;
    }

    int count = d->TransferBytes;

    if (count == 0)
	count = 0x10000;

    int inc = d->AAddressFixed ? 0 : (!d->AAddressDecrement ? 1 : -1);

#ifdef DEBUGGER
    if (Settings.TraceDMA)
    {
	sprintf (String, "DMA[%d]: %s Mode: %d 0x%02X%04X->0x21%02X Bytes: %d (%s) V-Line:%ld",
		Channel, d->TransferDirection ? "read" : "write",
		d->TransferMode, d->ABank, d->AAddress,
		d->BAddress, d->TransferBytes,
		d->AAddressFixed ? "fixed" :
		(d->AAddressDecrement ? "dec" : "inc"),
		CPU.V_Counter);
	if (d->BAddress == 0x18)
	    sprintf (String, "%s VRAM: %04X (%d,%d) %s", String,
		     PPU.VMA.Address,
		     PPU.VMA.Increment, PPU.VMA.FullGraphicCount,
		     PPU.VMA.High ? "word" : "byte");
	else
	if (d->BAddress == 0x22)
	    sprintf (String, "%s CGRAM: %02X (%x)", String, PPU.CGADD,
		     PPU.CGFLIP);
	S9xMessage (S9X_TRACE, S9X_DMA_TRACE, String);
    }
#endif

    if (!d->TransferDirection)
    {
#ifdef VAR_CYCLES
	CPU.Cycles += 8 * count;
#else
	CPU.Cycles += count + (count >> 2);
#endif
	uint8 *base = GetBasePointer ((d->ABank << 16) + d->AAddress);
	uint16 p = d->AAddress;

	if (!base)
	    base = Memory.ROM;

	if (inc > 0)
	    d->AAddress += count;
	else
	if (inc < 0)
	    d->AAddress -= count;

	if (d->TransferMode == 0 || d->TransferMode == 2)
	{
	    switch (d->BAddress)
	    {
	    case 0x04:
		do
		{
		    Work = *(base + p);
		    REGISTER_2104(Work);
		    p += inc;
		    CHECK_SOUND();
		} while (--count > 0);
		break;
	    case 0x18:
		if (!PPU.VMA.FullGraphicCount)
		{
		    do
		    {
			Work = *(base + p);
			REGISTER_2118_linear(Work);
			p += inc;
			CHECK_SOUND();
		    } while (--count > 0);
		}
		else
		{
		    do
		    {
			Work = *(base + p);
			REGISTER_2118_tile(Work);
			p += inc;
			CHECK_SOUND();
		    } while (--count > 0);
		}
		break;
	    case 0x22:
		do
		{
		    Work = *(base + p);
		    REGISTER_2122(Work);
		    p += inc;
		    CHECK_SOUND();
		} while (--count > 0);
		break;
	    case 0x80:
		do
		{
		    Work = *(base + p);
		    REGISTER_2180(Work);
		    p += inc;
		    CHECK_SOUND();
		} while (--count > 0);
		break;
	    default:
		do
		{
		    Work = *(base + p);
		    S9xSetPPU (Work, 0x2100 + d->BAddress);
		    p += inc;
		    CHECK_SOUND();
		} while (--count > 0);
		break;
	    }
	}
	else
	if (d->TransferMode == 1)
	{
	    if (d->BAddress == 0x18)
	    {
		// Write to V-RAM
		if (!PPU.VMA.FullGraphicCount)
		{
		    do
		    {
			Work = *(base + p);
			REGISTER_2118_linear(Work);
			p += inc;
			if (count <= 1)
			    break;

			Work = *(base + p);
			REGISTER_2119_linear(Work);
			p += inc;
			CHECK_SOUND();
			count -= 2;
		    } while (count > 0);
		}
		else
		{
		    do
		    {
			Work = *(base + p);
			REGISTER_2118_tile(Work);
			p += inc;
			if (count <= 1)
			    break;

			Work = *(base + p);
			REGISTER_2119_tile(Work);
			p += inc;
			CHECK_SOUND();
			count -= 2;
		    } while (count > 0);
		}
	    }
	    else
	    {
		// DMA mode 1 general case
		do
		{
		    Work = *(base + p);
		    S9xSetPPU (Work, 0x2100 + d->BAddress);
		    p += inc;
		    if (count <= 1)
			break;

		    Work = *(base + p);
		    S9xSetPPU (Work, 0x2101 + d->BAddress);
		    p += inc;
		    CHECK_SOUND();
		    count -= 2;
		} while (count > 0);
	    }
	}
	else
	if (d->TransferMode == 3)
	{
	    do
	    {
		Work = *(base + p);
		S9xSetPPU (Work, 0x2100 + d->BAddress);
		p += inc;
		if (count <= 1)
		    break;

		Work = *(base + p);
		S9xSetPPU (Work, 0x2100 + d->BAddress);
		p += inc;
		if (count <= 2)
		    break;

		Work = *(base + p);
		S9xSetPPU (Work, 0x2101 + d->BAddress);
		p += inc;
		if (count <= 3)
		    break;

		Work = *(base + p);
		S9xSetPPU (Work, 0x2101 + d->BAddress);
		p += inc;
		CHECK_SOUND();
		count -= 4;
	    } while (count > 0);
	}
	else
	if (d->TransferMode == 4)
	{
	    do
	    {
		Work = *(base + p);
		S9xSetPPU (Work, 0x2100 + d->BAddress);
		p += inc;
		if (count <= 1)
		    break;

		Work = *(base + p);
		S9xSetPPU (Work, 0x2101 + d->BAddress);
		p += inc;
		if (count <= 2)
		    break;

		Work = *(base + p);
		S9xSetPPU (Work, 0x2102 + d->BAddress);
		p += inc;
		if (count <= 3)
		    break;

		Work = *(base + p);
		S9xSetPPU (Work, 0x2103 + d->BAddress);
		p += inc;
		CHECK_SOUND();
		count -= 4;
	    } while (count > 0);
	}
	else
	{
#ifdef DEBUGGER
	    if (Settings.TraceDMA)
	    {
		sprintf (String, "Unknown DMA transfer mode: %d on channel %d\n",
			 d->TransferMode, Channel);
		S9xMessage (S9X_TRACE, S9X_DMA_TRACE, String);
	    }
#endif
	}

#ifdef SPC700_C
	APU_EXECUTE ();
#endif
	while (CPU.Cycles > CPU.NextEvent)
	    S9xDoHBlankProcessing ();

	// Check for reaching the start of v-blank then skipping on right
	// past. Rise of the Robots needs this.
//	if (!(Memory.FillRAM [0x4210] & 0x80))
//	    CPU.Flags &= ~DELAYED_NMI_FLAG;
    }
    else
    {
	do
	{
	    switch (d->TransferMode)
	    {
	    case 0:
	    case 2:
#ifndef VAR_CYCLES
		CPU.Cycles += 1;
#endif
		Work = S9xGetPPU (0x2100 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		--count;
		break;

	    case 1:
#ifndef VAR_CYCLES
		CPU.Cycles += 3;
#endif
		Work = S9xGetPPU (0x2100 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		if (!--count)
		    break;

		Work = S9xGetPPU (0x2101 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		count--;
		break;
		
	    case 3:
#ifndef VAR_CYCLES
		CPU.Cycles += 6;
#endif
		Work = S9xGetPPU (0x2100 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		if (!--count)
		    break;
		    
		Work = S9xGetPPU (0x2100 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		if (!--count)
		    break;
		    
		Work = S9xGetPPU (0x2101 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		if (!--count)
		    break;
		    
		Work = S9xGetPPU (0x2101 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		count--;
		break;

	    case 4:
#ifndef VAR_CYCLES
		CPU.Cycles += 6;
#endif
		Work = S9xGetPPU (0x2100 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		if (!--count)
		    break;
		    
		Work = S9xGetPPU (0x2101 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		if (!--count)
		    break;

		Work = S9xGetPPU (0x2102 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		if (!--count)
		    break;
		    
		Work = S9xGetPPU (0x2103 + d->BAddress);
		S9xSetByte (Work, (d->ABank << 16) + d->AAddress);
		d->AAddress += inc;
		count--;
		break;

	    default:
#ifdef DEBUGGER
		if (1) //Settings.TraceDMA)
		{
		    sprintf (String, "Unknown DMA transfer mode: %d on channel %d\n",
			     d->TransferMode, Channel);
		    S9xMessage (S9X_TRACE, S9X_DMA_TRACE, String);
		}
#endif
		count = 0;
		break;
	    }
	    CHECK_SOUND();
	    DO_HBLANK_CHECK();
	} while (count);
    }
    
    Memory.FillRAM[0x4302 + (Channel << 4)] = (uint8) d->AAddress;
    Memory.FillRAM[0x4303 + (Channel << 4)] = d->AAddress >> 8;
    Memory.FillRAM[0x4305 + (Channel << 4)] = 0;
    Memory.FillRAM[0x4306 + (Channel << 4)] = 0;
    DMA[Channel].IndirectAddress = 0;
    d->TransferBytes = 0;
    
    CPU.InDMA = FALSE;
}

void S9xStartHDMA ()
{
    if (Settings.DisableHDMA)
	IPPU.HDMA = 0;
    else
	missing.hdma_this_frame = IPPU.HDMA = Memory.FillRAM [0x420c];

    if (IPPU.HDMA)
    {
	IPPU.HDMAStarted = TRUE;
	static bool8 warn1 = FALSE;
	static bool8 warn2 = FALSE;
	static bool8 warn3 = FALSE;
	for (uint8 i = 0; i < 8; i++)
	{
	    DMA [i].LineCount = 0;
	    DMA [i].Address = DMA [i].AAddress;
	}
    }
}

uint8 S9xDoHDMA (uint8 byte)
{
    struct SDMA *p = &DMA [0];
    
    int d = 0;

    for (uint8 mask = 1; mask; mask <<= 1, p++, d++)
    {
	if (byte & mask)
	{
	    if (!p->LineCount)
	    {
		uint8 line = S9xGetByte ((p->ABank << 16) + p->Address);
		if (line == 0x80)
		{
		    p->Repeat = TRUE;
		    p->LineCount = 128;
		}
		else
		{
		    p->Repeat = !(line & 0x80);
		    p->LineCount = line & 0x7f;
		}
		
		if (!p->LineCount)
		{
		    byte &= ~mask;
		    p->IndirectAddress += HDMAMemPointers [d] - HDMABasePointers [d];
		    Memory.FillRAM [0x4305 + (d << 4)] = (uint8) p->IndirectAddress;
		    Memory.FillRAM [0x4306 + (d << 4)] = p->IndirectAddress >> 8;
		    continue;
		}

		p->Address++;
		p->FirstLine = 1;
		if (p->HDMAIndirectAddressing)
		{
		    p->IndirectBank = Memory.FillRAM [0x4307 + ((p - DMA) << 4)];
		    p->IndirectAddress = S9xGetWord ((p->ABank << 16) + p->Address);
		    p->Address += 2;
		}
		else
		{
		    p->IndirectBank = p->ABank;
		    p->IndirectAddress = p->Address;
		}
		HDMABasePointers [d] = HDMAMemPointers [d] = 
				    S9xGetMemPointer ((p->IndirectBank << 16) + p->IndirectAddress);
	    }
	    if (p->Repeat && !p->FirstLine)
	    {
		p->LineCount--;
		continue;
	    }
	    if (!HDMAMemPointers [d])
	    {
		byte &= ~mask;
		continue;
	    }
#ifdef DEBUGGER
	    if (Settings.TraceHDMA)
	    {
		sprintf (String, "H-DMA[%d] (%d) 0x%02X%04X->0x21%02X %s, Count: %d, Rep: %s, V-LINE: %ld %02X%04X\n",
			 p-DMA, p->TransferMode, p->IndirectBank,
			 p->IndirectAddress,
			 p->BAddress,
			 p->HDMAIndirectAddressing ? "indirect" : "absolute",
			 p->LineCount,
			 p->Repeat ? "yes" : "no", CPU.V_Counter,
			 p->ABank, p->Address);
		S9xMessage (S9X_TRACE, S9X_HDMA_TRACE, String);
	    }
#endif

	    switch (p->TransferMode)
	    {
	    case 0:
#ifndef VAR_CYCLES
		CPU.Cycles += 1;
#else
		CPU.Cycles += 8;
#endif
		S9xSetPPU (*HDMAMemPointers [d]++, 0x2100 + p->BAddress);
		break;
	    case 1:
	    case 5:
#ifndef VAR_CYCLES
		CPU.Cycles += 3;
#else
		CPU.Cycles += 16;
#endif
		S9xSetPPU (*(HDMAMemPointers [d] + 0), 0x2100 + p->BAddress);
		S9xSetPPU (*(HDMAMemPointers [d] + 1), 0x2101 + p->BAddress);
		HDMAMemPointers [d] += 2;
		break;
	    case 2:
	    case 6:
#ifndef VAR_CYCLES
		CPU.Cycles += 3;
#else
		CPU.Cycles += 16;
#endif
		S9xSetPPU (*(HDMAMemPointers [d] + 0), 0x2100 + p->BAddress);
		S9xSetPPU (*(HDMAMemPointers [d] + 1), 0x2100 + p->BAddress);
		HDMAMemPointers [d] += 2;
		break;
	    case 3:
	    case 7:
#ifndef VAR_CYCLES
		CPU.Cycles += 6;
#else
		CPU.Cycles += 32;
#endif
		S9xSetPPU (*(HDMAMemPointers [d] + 0), 0x2100 + p->BAddress);
		S9xSetPPU (*(HDMAMemPointers [d] + 1), 0x2100 + p->BAddress);
		S9xSetPPU (*(HDMAMemPointers [d] + 2), 0x2101 + p->BAddress);
		S9xSetPPU (*(HDMAMemPointers [d] + 3), 0x2101 + p->BAddress);
		HDMAMemPointers [d] += 4;
		break;
	    case 4:
#ifndef VAR_CYCLES
		CPU.Cycles += 6;
#else
		CPU.Cycles += 32;
#endif
		S9xSetPPU (*(HDMAMemPointers [d] + 0), 0x2100 + p->BAddress);
		S9xSetPPU (*(HDMAMemPointers [d] + 1), 0x2101 + p->BAddress);
		S9xSetPPU (*(HDMAMemPointers [d] + 2), 0x2102 + p->BAddress);
		S9xSetPPU (*(HDMAMemPointers [d] + 3), 0x2103 + p->BAddress);
		HDMAMemPointers [d] += 4;
		break;
	    }
	    if (!p->HDMAIndirectAddressing)
		p->Address += HDMA_ModeByteCounts [p->TransferMode];
	    p->FirstLine = 0;
	    p->LineCount--;
	}
    }
    return (byte);
}

void S9xResetDMA ()
{
    int d;
    for (d = 0; d < 8; d++)
    {
	DMA [d].TransferDirection = FALSE;
	DMA [d].HDMAIndirectAddressing = FALSE;
	DMA [d].AAddressFixed = TRUE;
	DMA [d].AAddressDecrement = FALSE;
	DMA [d].TransferMode = 0xff;
	DMA [d].ABank = 0xff;
	DMA [d].AAddress = 0xffff;
	DMA [d].Address = 0xffff;
	DMA [d].BAddress = 0xff;
	DMA [d].TransferBytes = 0xffff;
    }
    for (int c = 0x4300; c < 0x4380; c += 0x10)
    {
	for (d = c; d < c + 12; d++)
	    Memory.FillRAM [d] = 0xff;

	Memory.FillRAM [c + 0xf] = 0xff;
    }
}
