
/* search.c
 *   Handles the search window popup
 */

#include <stdio.h>

#include <Xos.h>
#include <Intrinsic.h>
#include <StringDefs.h>

#include <Shell.h>

#include <Xaw/Command.h>
#include <Xaw/Label.h>
#include <Xaw/Form.h>
#include <Xaw/Box.h>
#include <Xaw/AsciiText.h>

#include "defs.h"
#include "externs.h"
#include "game.h"
#include "widgets.h"



enum {G_LABEL,G_INPUT,F_LABEL,F_INPUT,POUND_LABEL,POUND_INPUT,
		H_LABEL,H_INPUT,N_LABEL,N_INPUT,U_LABEL,U_INPUT,
		NUM_OF_N };

enum {NFORM_W,KANA_W,ENGLISH_W,SEARCH_W,NUM_OF_W};


Widget search_popup;

Widget searchwidgets[NUM_OF_W];
Widget searchnumbers[NUM_OF_N];
Widget searchkanjiW,searchokuW; /* display current kanji here */

static int lastsearch=-1;	/* holds index of last kanji searched */

static char *searchAccel = 
 " <Key>Return:  do-find()";


static up = -1; /* this has to be visible by printsearch() */

/* This gets called after clicking on the "search" button, or
 *  using the keyboard accelerator. It pops up the search window.
 */
void SearchCallback(Widget w,XtPointer client_data, XtPointer call_data)
{
	static Position rel_x,rel_y;
	Position x,y;

	if(up == -1){
		/* first time init.. */
		rel_x = GetXtNumber("search_popupx","Search_popupx");
		rel_y = GetXtNumber("search_popupy","Search_popupy");
		up = 0;
	}

	if(up == 0)
	{
		XtTranslateCoords(toplevel,rel_x,rel_y,&x,&y);
		XtVaSetValues(search_popup,
		      XtNx,x,
		      XtNy,y,
		      NULL);
		XtVaSetValues(searchwidgets[SEARCH_W],
			      XtNstring," Search String Here ",
			      NULL);
		XtPopup(search_popup,XtGrabNone);
		setstatus("Bringing up search window...");
		up = 1;
	} else
	{
		XtPopdown(search_popup);
		up = 0;
	}
}

/* printsearch:
*	Given an index, make search window print out appropriate stuff.
*	printkanji(0) means "clear windows".
*	ASSUME passed valid index to translations[].
*/
void printsearch(int kindex){
	XChar2b kchars[2];

	if(up != 1){
		/* pop up window, if it isnt already */
		SearchCallback(NULL,NULL,NULL);
	}
	
	kchars[1].byte1 = kchars[1].byte2 = 0;

	kchars[0].byte1 = (kindex & 0xff00)>>8;
	kchars[0].byte2 = (kindex & 0x00ff);

	XtVaSetValues(searchkanjiW,XtNlabel,kchars,
			XtNwidth,KANJIWIDTH,XtNheight,KANJIWIDTH,NULL);


	if(kindex==0){
		/* BLANK EVERYTHING */
#ifdef USE_OKU
		XtVaSetValues(searchokuW,XtNlabel,"",NULL);
#endif
		XtVaSetValues(searchkanjiW,
			      XtNlabel,"                               ",
			      NULL);
		XtVaSetValues(searchwidgets[ENGLISH_W],XtNlabel,"",NULL);
		SetWidgetNumberval(searchnumbers[G_INPUT],0);
		SetWidgetNumberval(searchnumbers[F_INPUT],0);
		SetWidgetNumberval(searchnumbers[POUND_INPUT],0);
		SetWidgetNumberval(searchnumbers[H_INPUT],0);
		SetWidgetNumberval(searchnumbers[N_INPUT],0);
		SetWidgetNumberval(searchnumbers[U_INPUT],0);
	} else {
		struct translationstruct *item;
		item = translations[kindex];

		SetWidgetNumberval(searchnumbers[G_INPUT],item->grade_level);
		SetWidgetNumberval(searchnumbers[F_INPUT],item->frequency);
		SetWidgetNumberval(searchnumbers[POUND_INPUT],kindex);
		SetWidgetNumberval(searchnumbers[H_INPUT],item->Hindex);
		SetWidgetNumberval(searchnumbers[N_INPUT],item->Nindex);
		SetWidgetHexval(searchnumbers[U_INPUT],item->Uindex);
#ifdef USE_OKU
		if(item->ON_extra != NULL){
			XtVaSetValues(searchokuW,
 				      XtNlabel,item->ON_extra,NULL);
		} else {
			XtVaSetValues(searchokuW,XtNlabel,"",NULL);
		}
#endif
		XtVaSetValues(searchwidgets[KANA_W],
			      XtNlabel,item->pronunciation,NULL);
		XtVaSetValues(searchwidgets[ENGLISH_W],
			      XtNlabel,item->english,NULL);
	}

	lastsearch = kindex;

}


/* doenglishfind:
 *	Assume what we want to look for is in the search ENGLISH INPUT widget.
 *	First, read in that value.
 *	Start from last place searched, to allow for similar
 *	characters.
 *	Return index of kanji, or -1 if not found
 */
int doenglishfind()
{
	String str;
	int searchindex;
	if((lastsearch<lowestkanji) ||
	   (lastsearch >highestkanji)){
		lastsearch = lowestkanji;
	}

	searchindex = lastsearch;
	
	XtVaGetValues(searchwidgets[SEARCH_W],XtNstring,&str,NULL);

	if((str[0] == '\0') || (str[0] == '\n')){
		return -1;
	}


	do {
		searchindex++;
		if(searchindex >highestkanji)
			searchindex = lowestkanji;
		if(translations[searchindex] == NULL)
			continue;

		if(strstr(translations[searchindex]->english,str) != NULL){
			return searchindex;
		}
	
	} while(searchindex !=lastsearch);

	return 0;
	

}

/* DoFind
 *   (do-find)accelerator hook
 *    Handles <return> press for numerical input windows
 *	on the search popup.
 */
void 
DoFind(Widget w,XEvent *event,String *params,Cardinal *num_parags){
	int newindex;
	if(w == searchwidgets[SEARCH_W]){
		newindex = doenglishfind();
		if(newindex == -1){
			printsearch(lastsearch);
		} else {
			printsearch(newindex);
		}
		
	}

	else if (w == searchnumbers[POUND_INPUT])
	{
		/* want closest index match */
		newindex = GetWidgetNumberval(w);
		if((newindex <lowestkanji) || (newindex> highestkanji)){
			char tempstr[100];
			printsearch(lastsearch);
			return;
		} 

		if(newindex > highestkanji){
			newindex = highestkanji;
		} else if (newindex <lowestkanji){
			newindex = lowestkanji;
		} else {
			/* find closest non-blank */
			while(translations[newindex] == NULL){
				newindex++;
				/* this should never be triggered, but... */
				if(newindex >highestkanji)
					newindex = lowestkanji;
			}
		}

		printsearch(newindex);
	} else 	if (w == searchnumbers[F_INPUT])
	{
		/* want closest frequency match */
		/* "F:" input widget */
		int findfreq;
		int curindex = lowestkanji+1;
		

		newindex = lowestkanji;
		findfreq = GetWidgetNumberval(w);

		if(findfreq <1){
			setstatus("frequencies must be greater than 0");
			printsearch(lastsearch);
			return;			
		}

		while(curindex <=highestkanji){
			if(translations[curindex] == NULL){
				curindex++;
				continue;
			}
			if(translations[curindex]->frequency == findfreq){
				newindex = curindex;
				break;
			}
			if((findfreq - translations[curindex]->frequency) <
			   (findfreq - translations[newindex]->frequency)){
				newindex = curindex;
			}
			curindex++;
			   
		}
		
		printsearch(newindex);

	} else {
		char append='\0';
		char findstring[100];
		String str;

		
		if(w == searchnumbers[H_INPUT]){
			append = 'H';
		} else if(w == searchnumbers[N_INPUT]){
			append = 'N';
		} else if(w == searchnumbers[U_INPUT]){
			append = 'U';
		} else {
			puts("Where did that input come from??");
			return;
		}
		
		XtVaGetValues(w,XtNstring,&str,NULL);

		sprintf(findstring,"%c%.99s",append,str);

		/* FindIndex sets status appropriately on error */

		newindex = FindIndex(findstring);
		printsearch(newindex);
		
	}
	

}


/*************************************************************
 *    Widget Creation Below Here                             *
 *************************************************************/

/* make the G, F, # , etc,  for the search popup */
void makenumbers(Widget parent){
	searchnumbers[G_LABEL] = XtVaCreateWidget("0",labelWidgetClass,parent,
		       XtNlabel,"G:",      XtNborderWidth,0,
		       NULL);
	searchnumbers[G_INPUT] = XtVaCreateWidget("1",asciiTextWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[G_LABEL],
		       XtNstring,"   ",  XtNdisplayCaret,False,
		       XtNwidth,30,	/* leave this short */
		       NULL);
	searchnumbers[H_LABEL] = XtVaCreateWidget("6",labelWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNlabel,"H:",   XtNborderWidth,0,
		       NULL);
	searchnumbers[H_INPUT] = XtVaCreateWidget("7",asciiTextWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[H_LABEL],
		       XtNstring," ",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);

	searchnumbers[F_LABEL] = XtVaCreateWidget("2",labelWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[G_INPUT],
		       XtNfromHoriz,searchnumbers[H_INPUT],
		       XtNlabel,"F:",  XtNborderWidth,0,
		       NULL);
	searchnumbers[F_INPUT] = XtVaCreateWidget("3",asciiTextWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[F_LABEL],
		       XtNstring,"",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);
	searchnumbers[N_LABEL] = XtVaCreateWidget("8",labelWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[G_INPUT],
		       XtNfromHoriz,searchnumbers[H_INPUT],
		       XtNlabel,"N:",   XtNborderWidth,0,
		       NULL);
	searchnumbers[N_INPUT] = XtVaCreateWidget("9",asciiTextWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[N_LABEL],
		       XtNstring," ",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);

	searchnumbers[POUND_LABEL] = XtVaCreateWidget("4",labelWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[F_INPUT],
		       XtNfromHoriz,searchnumbers[N_INPUT],
		       XtNlabel,"#:  ",  XtNborderWidth,0,
		       NULL);
	searchnumbers[POUND_INPUT] = XtVaCreateWidget("5",asciiTextWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[POUND_LABEL],
		       XtNstring,"",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);
	searchnumbers[U_LABEL] = XtVaCreateWidget("10",labelWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[F_INPUT],
		       XtNfromHoriz,searchnumbers[N_INPUT],
		       XtNlabel,"Ux:",   XtNborderWidth,0,
		       NULL);
	searchnumbers[U_INPUT] = XtVaCreateWidget("11",asciiTextWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[U_LABEL],
		       XtNstring," ",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);

	XtManageChildren(searchnumbers,NUM_OF_N);
}


/* make topstuff:
 * Make widgets dealing with "current kanji", at top of search popup
 */
void maketopstuff(Widget parent){
	Widget numbersform;

	searchkanjiW = XtVaCreateManagedWidget("kanjilarge",
						  labelWidgetClass,
						  parent,
			XtNlabel,"",
			XtNwidth,KANJIWIDTH,XtNheight,KANJIWIDTH,
			XtNencoding,XawTextEncodingChar2b,
			XtNfont,largekfont,
			NULL);
	
	numbersform = XtVaCreateManagedWidget("numform",
					formWidgetClass,
					parent,
			XtNfromHoriz,searchkanjiW,
       			NULL);

	makenumbers(numbersform);

#ifdef USE_OKU
	searchokuW = XtVaCreateManagedWidget("searchoku",
					     labelWidgetClass,
					     parent,
			XtNlabel,"                               ",
			XtNencoding,XawTextEncodingChar2b,
			XtNfont,smallkfont,
			XtNjustify,XtJustifyLeft,
			XtNwidth,FULLWIDTH,
			XtNfromVert,searchkanjiW,
			XtNfromVert,numbersform,
			NULL);
#endif

}

/* This is the only exported "make widgets" routine */
void MakeSearchPopup()
{
	 Widget searchform;
	 XtAccelerators Accel;

	search_popup = XtVaCreatePopupShell("kdrill_search",
		transientShellWidgetClass,
		toplevel,
		NULL);

	searchform = XtVaCreateManagedWidget("searchform",
					     formWidgetClass,
					     search_popup,
					     NULL);

	searchwidgets[NFORM_W] = XtVaCreateManagedWidget("northform",formWidgetClass,
			searchform,
			XtNborderWidth,2,
			NULL);


	maketopstuff(searchwidgets[NFORM_W]);


	searchwidgets[KANA_W] = XtVaCreateWidget("searchkana",
						 labelWidgetClass,
						 searchform,
			XtNlabel,"                               ",
			XtNfromVert,searchwidgets[NFORM_W],
			XtNvertDistance,20,
			XtNencoding,XawTextEncodingChar2b,
			XtNfont,smallkfont,
			XtNwidth,FULLWIDTH,
			NULL);

	searchwidgets[ENGLISH_W] = XtVaCreateWidget("searchenglish",
						    labelWidgetClass,
						    searchform, 
			XtNlabel,"                                       ",
			XtNfromVert,searchwidgets[KANA_W],
			XtNfont,englishfont,
			XtNwidth,FULLWIDTH,
			NULL);

	searchwidgets[SEARCH_W] = XtVaCreateWidget("searchenglish",
						   asciiTextWidgetClass,
						   searchform, 
			XtNlabel,"                                       ",
			XtNfromVert,searchwidgets[ENGLISH_W],
			XtNvertDistance,20,
			XtNfont,englishfont,
			XtNwidth,FULLWIDTH,
			XtNeditType,XawtextEdit,
			XtNstring," Search String Here ",
			NULL);

	XtManageChildren(searchwidgets,NUM_OF_W);


	Accel = XtParseAcceleratorTable(searchAccel);
	XtOverrideTranslations(searchwidgets[SEARCH_W],Accel);


	XtOverrideTranslations(searchnumbers[F_INPUT],Accel);
	XtOverrideTranslations(searchnumbers[POUND_INPUT],Accel);
	XtOverrideTranslations(searchnumbers[H_INPUT],Accel);
	XtOverrideTranslations(searchnumbers[N_INPUT],Accel);
	XtOverrideTranslations(searchnumbers[U_INPUT],Accel);


}
