// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: widget.h,v 1.21 1998/06/21 03:25:12 jgg Exp $
/* ######################################################################

   Widget - Base class for the internal widget set.
   
   The idea here is to have a mostly-general windowing system that we can 
   apply to multiple target entities. 'Widget' is the name given to any
   screen entity that has an internal class instance. Ie a Button, Window
   Menu, etc.
   
   A Widget exists inside of a tripply linked tree, with a parent, child
   and sibling. Clipping regions are applied to the widget so that it 
   cannot draw outside of it's parent widget. Handling of keyboard 
   focus, extenting and notifications are all handled by the widget class.
   
   Other classes subclass widget to provide the needed kinds of GUI
   elements (like button). After much thought I decided that there
   is exactly one big difference between Text mode and Graphics mode
   operation, how the widgets are drawn and what widgets are drawn.
   To accomidate this I chose to have a general draw function, Render()
   which will draw the widget in the two different modes. The current mode 
   can be queried with gGC != 0. For many of the widgets drawing of the two 
   types differes only slightly so there is simply several if's.
   
   Different layout is handled by the client of the library which should
   provide different functions for layout of graphic widgets and text
   widgets. Graphic widgets look the SAME on all platforms as they do
   not depend on the platforms custom widgets, only on the specialization
   of the platforms draw routines and 'region' creation routines.
   
   Another element a widget has is a platform specific 'region'. A 
   region is a general term for a window that has no borders or any
   special decorations or controls. A widget can be flagged to have
   a region in which case during it's realization platform specific
   calls will be performed to create the region and route input from
   the region to the widget. The advantage here is that the widget
   library can get the benifits of optimized graphics clipping and
   optimized event routing that the target GUI provides.
   
   So the net effect is that the only platform specific elements of
   the whole system are the Graphics class, the Input class(s) and the
   attach function.

   Please note that all widgets must be constructed on the heap and 
   could be destroyed by the library if a parent is destroyed.
   
   See the description of the graphics classes for information about
   how the graphics IO system is handled

   Some other things to note about the widget is that the base class
   provides for BorderX/BorderY. These are used for child widget offsetting
   and clipping. Basically the child widgets are placed inside the borders
   of the parent. This is all the BorderX/Y variables control. BasicWidget
   and all classes derived from it use the values for the width of
   the border.
   
   ##################################################################### */
									/*}}}*/
// Header section: deity
#ifndef DEITY_WIDGET_H
#define DEITY_WIDGET_H

#include <deity/event.h>
#include <deity/notify.h>
#include <deity/thread.h>

// Notifications generated by Widget
#define Nt_Resize Notifyer::Tag(Widget::Type,1)     // Data -> Rect *NewSize
#define Nt_Sync Notifyer::Tag(Widget::Type,2)       // Data -> 0
#define Nt_Action Notifyer::Tag(Widget::Type,2)     // Data -> 0

class Widget
{
   // Internal helpers
   void RealizeChildren();
   void DoSync();
   void DoRender(bool NoDraw);
   
   // Internal grab tracking class
   struct GrabItem
   {
      Widget *Target;
      unsigned long Flags;

      // Flags
      inline bool IsFlag(unsigned long F) const {return (Flags & F) == F;};
      void Flag(unsigned long State, unsigned long F) {Flags = (Flags & (~F)) | State;};
      inline void Flag(unsigned long F) {Flags |= F;};
   };

   // Internal, global Grab information
   static GrabItem *GrabList;
   static int GrabSize;
   
   // G++ Bug, should be protected
   public:

   /* Do not directly manipulate any of these parameters. They are
      protected so that derived classes can have easy access to Widget
      attributes */

   // Tree Pointers
   Widget *Parent;
   Widget *Child;
   Widget *Brother;
   Widget *FocusWidget;

   // State
   bool Realized;
   Rect Pos;
   long BorderX;
   long BorderY;
   unsigned long Flags;
   unsigned long ActionFlags;
   unsigned short ExtentXFlags;
   unsigned short ExtentYFlags;
   NotifyerList Notifications;

   // Region specific data
   void *RegionData;
   
   // Actually performs the drawing.
   virtual void Render(CombinedGC &/*GC*/) {};

   // Input handlers
   virtual bool Key(const KeyEvent &,Widget *) {return false;};
   virtual void Mouse(const MouseEvent &) {};

   // Focus notification
   virtual void FocusGained(Widget *) {};
   virtual void FocusLost(Widget *) {};
   virtual void FocusRestore() {};

   // Notification
   inline bool Trigger(Notifyer::Tag ID,void *Data)
      {return Notifications.Trigger(this,ID,Data);};

   // Aids
   long ExtentDo(long New,long Old,unsigned short &Flags);
   
   public:

   static void *Type;
   static Widget *LastWidget;
   static Mutex WidgetLock;
   
   // Grab::Flag values (GrabMouse)
   enum GrabFlags
   {
      GrabWhenFocused = (1 << 0),    // Grab takes effect only when target has focus
      GrabPermanent = (1 << 1),      // Grab is not removed on button release (if widget is realized)
      GrabChildren = (1 << 2),       // Grab should follow to children
      GrabAllGrab = (1 << 3)         // Grab is always active (popup menus)
   };      

   // Flag defines
   enum BasicFlags
   {
      ShowFocus = (1 << 0),          // The widget renders focus
      Focusable = (1 << 1),          // The widget can get focus
      ParentKeys = (1 << 2),         // Widget gets keys the parent doesnt want
      DelayRealize = (1 << 3),       // Widget isnt realized by *Family
      MenuDrop = (1 << 4),           // The popup menu is descended
      ForceFocus = (1 << 5),         // Widget should look like it has focus
      Region = (1 << 6),             // Widget has a region
      Window = (1 << 7),             // Widget should have window decoration
      Building = (1 << 8),           // Region is being built by the GUI
      AutoExtent = (1 << 9),         // Widget gets extented on parent resize
      Set = (1 << 10),               // Widget is set (depressed)
      Toggle = (1 << 11),            // Widget doesnt auto-unset
      MouseTransparent = (1 << 12)   // Mouse events go through the widget+children
   };

   // ActionFlag defines
   enum ActionFlags
   {
      NeedsRender = (1 << 0),        // Widget needs to be drawn (damaged)
      NeedsSync = (1 << 1),          // Widget needs to be syncd with client
      RedrawPending = (1 << 2)       // Region is going to get a draw event soon
   };

   // ExtentFlag defines, diff for x/y directions
   enum ExtentFlags
   {
      ExtentOnce = 0,               // Extent only if Done not set
      ExtentAlways = 1,             // Extent always to ideal size
      ExtentLarger = 2,             // Extent to become largeronly
      ExtentNever = 3,              // Never extent this dimension
      ExtentDone = 1 << 3,          // Set after each extent
      ExtentModeMask = 7
   };

   // A Tree iterator
   class iterator
   {
      Widget *Current;
      public:

      // Iteration
      inline void operator ++(int) {Current = Current->Brother;};
      inline void operator ++() {Current = Current->Brother;};
      inline void down() {Current = Current->Brother;};
      inline void up() {Current = Current->Parent;};
      inline bool end() const {return Current == 0;};
      
      // Comparison
      inline bool operator == (const iterator &B) const {return Current == B.Current;};
      inline bool operator != (const iterator &B) const {return Current != B.Current;};
		
      // Accessors
      inline Widget const *operator ->() const {return Current;};
      inline Widget *operator ->() {return Current;};
      inline Widget const &operator *() const {return *Current;};
      inline Widget &operator *() {return *Current;};
      inline operator Widget const *() const {return Current;};
      inline operator Widget *() {return Current;};
      
      // Constructors
      inline iterator(Widget *Start) : Current(Start) {}; 
   };
   friend iterator;
   
   // Create the iterator
   iterator begin() {return iterator(this);};
   iterator children() {return iterator(Child);};
   unsigned int NumChildren();
      
   // Realization control
   virtual void Realize();
   virtual void UnRealize();
   void RealizeFamily();
   void ChangeParent(Widget *Parent);
   
   // Size control
   virtual void Extent();
   virtual Point IdealSize() {return Point(1,1);};
   virtual void Layout() {return;};
   void ExtentFamily();
   void ExtentMode(unsigned long X,unsigned long Y);
   virtual void Resize(Rect NewSize);
   inline Rect Size() const {return Pos;};
   inline AbsRect Loc() const {return Pos;};
   inline Point CanvasSize() const {return Point(Pos.w - BorderX*2,Pos.h - BorderY*2);};
   Rect AbsoluteRect(bool ToRegion = false);
   inline Point Borders() const {return Point(BorderX,BorderY);};
   
   // Drawing
   void Draw();
   bool ConnectGC(CombinedGC &GC);
   inline void Damage() {ActionFlags |= NeedsRender;}; // Schedual drawing
   inline bool IsDamaged() const {return (ActionFlags & NeedsRender) == NeedsRender;};
   inline void Sync() {ActionFlags |= NeedsSync;};
   void DoActions();
      
   // Input
   void RouteKey(const KeyEvent &Key);
   void RouteMouse(MouseEvent &Event,bool Direct = false);
   void GrabMouse(unsigned long Flag = 0,bool Primary = false);
   
   // Flags
   inline bool IsFlag(unsigned long F) const {return (Flags & F) == F;};
   void Flag(unsigned long State, unsigned long F) {Flags = (Flags & (~F)) | State;};
   inline void Flag(unsigned long F) {Flags |= F;};
   
   // Focus
   bool Focus();
   void GiveFocus();
   void FocusNext();
   void FocusLast();

   // Order
   void ToBack();
   void ToFront();
   
   // Notification 
   inline void Add(Notifyer *Notify) {Notifications.Add(Notify);};
		
   // Construct/Destruct
   Widget(Widget *Parent = 0);
   virtual ~Widget();
};

#endif
