/*
 *  wmconfig.c:		Input of WindowMaker config file	
 *
 *  Written by:		Ullrich Hafner
 *		
 *  Copyright (C) 1998 Ullrich Hafner <hafner@informatik.uni-wuerzburg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1998/10/08 22:04:58 $
 *  $Author: hafner $
 *  $Revision: 1.32 $
 *  $State: Exp $
 */

#include "config.h"

#if HAVE_STDLIB_H
#	include <stdlib.h>
#endif /* not HAVE_STDLIB_H */
#if HAVE_STRING_H
#	include <string.h>
#else /* not HAVE_STRING_H */
#	include <strings.h>
#endif /* not HAVE_STRING_H */
#include <proplist.h>

#include "misc.h"
#include "window.h"

#include "error.h"

/*******************************************************************************

			       global variables
  
*******************************************************************************/

char		*wmpath        = NULL;	/* path to WindowMaker attributes file */
proplist_t	*plist_changed = NULL;	/* modified proplist keys */
proplist_t	*pl_yes        = NULL;	/* dummy */
proplist_t	*orig_wmaker   = NULL;	/* original WindowMaker file */

/*******************************************************************************

			       public code
  
*******************************************************************************/

int
main (int argc, char **argv)
{
   proplist_t	*global_windowmaker;	/* global defined options */
   proplist_t	*windowmaker;		/* user defined options in WindowMaker */
   proplist_t   *wmconfig = NULL;	/* WMConfig options */
   bool_t	previews  = YES;
   char		*font = "-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*";
   
   init_error_handling (argv [0]);

   /*
    *  Parse command line
    */
   if (argc > 3)
   {
      message (PACKAGE " " VERSION "\nusage: %s [-nopreviews] [filename]\n"
	       "filename: path to WindowMaker configuration file", argv [0]);
      return 1;
   }
   else if (argc == 3)
   {
      if (streq (argv [1], "-nopreviews"))
      {
	 previews = NO;
	 wmpath	  = argv [2];
      }
      else if (streq (argv [2], "-nopreviews"))
      {
	 previews = NO;
	 wmpath	  = argv [1];
      }
      else
      {
	 message (PACKAGE " " VERSION "\nusage: %s [-nopreviews] [filename]\n"
		  "filename: path to WindowMaker configuration file", argv [0]);
	 return 1;
      }
   }
   else if (argc == 2)
   {
      if (streq (argv [1], "-nopreviews"))
	 previews = NO;
      else if (argv [1][0] == '-')
      {
	 message (PACKAGE " " VERSION "\nusage: %s [-nopreviews] [filename]\n"
		  "filename: path to WindowMaker configuration file", argv [0]);
	 return 1;
      }
      else
	 wmpath = argv [1];
   }
   
   /*
    *  Open 'WindowMaker' file
    */
   if (!wmpath)
      wmpath = get_gnustep_path ("WindowMaker");
   windowmaker = PLGetProplistWithPath (wmpath);

   plist_changed = PLMakeDictionaryFromEntries (NULL, NULL, NULL);
   pl_yes	 = PLRetain (PLMakeString ("Yes"));
   
   if (windowmaker)
   {
      proplist_t *keys  = PLGetAllDictionaryKeys (windowmaker);
      unsigned	 n;
      
      for (n = 0; n < PLGetNumberOfElements (keys); n++)
	 PLInsertDictionaryEntry (plist_changed, PLGetArrayElement (keys, n),
				  pl_yes);

      PLRelease (keys);
      orig_wmaker = PLDeepCopy (windowmaker);
   }

   /*
    *  Open system 'WindowMaker' file
    */
   {
      char *global_file = Calloc (strlen (WMAKERDIR)
				  + strlen ("/Defaults/WindowMaker") + 1,
				  sizeof (char));
      char *filename;
      
      strcpy (global_file, WMAKERDIR);
      strcat (global_file, "/Defaults/WindowMaker");
      filename = expand_tilde (global_file);
      Free (global_file);
      
      global_windowmaker = PLGetProplistWithPath (filename);
      if (!global_windowmaker)
      {
	 warning ("Can't open global WindowMaker configuration file `%s'.",
		  filename);
	 global_windowmaker = PLMakeDictionaryFromEntries (NULL, NULL, NULL);
      }
      Free (filename);
   }

   if (!windowmaker)
   {
      windowmaker = global_windowmaker;
      PLSetFilename (windowmaker, PLMakeString (wmpath));
      if (PLSave (windowmaker, YES))
	 warning ("New WindowMaker configuration file `%s' created.", wmpath);
      else
	 error ("Can't generate configuration file `%s'.", wmpath);
   }
   else
   {
      char		*orig;		/* original filename */
      proplist_t	*porig;		/* original fname in PropList format */
      char		*new;		/* backup filename */
      proplist_t	*pnew;		/* backup fname in PropList format */

      /*
       *  Generate backup file "'path'.bak"
       */
      porig = PLGetFilename (windowmaker);
      orig  = PLGetString (porig);
      new   = Calloc (strlen (orig) + strlen (".bak") + 1, sizeof (char));
      strcpy (new, orig);
      strcat (new, ".bak");
      pnew = PLMakeString (new);
      PLSetFilename (windowmaker, pnew);
      if (PLSave (windowmaker, YES))
	 message ("Backupfile `%s' generated.", new);
      else
	 error ("Can't write backupfile `%s'.", new);
      PLSetFilename (windowmaker, porig);
      PLRelease (pnew);
      Free (new);

      PLSetFilename (global_windowmaker, porig);
      PLMergeDictionaries(global_windowmaker, windowmaker);
      PLRelease (windowmaker);
      windowmaker = global_windowmaker;
   }
   
   /*
    *  Read english option description file
    */
   {
      char *path;
   
      path = Calloc (strlen (PKGDATADIR) + strlen ("/WMWmakerconf") + 1,
		     sizeof (char));
      strcpy (path, PKGDATADIR);
      strcat (path, "/WMWmakerconf");

      wmconfig = PLGetProplistWithPath (path);
      if (!wmconfig)
	 error ("Can't read file '%s'", path);
      Free (path);
   }
   /*
    *  Read optional option description file
    */
   {
      char		*path;
      char		*lang   = getenv ("LANG");
      char		locale [3];
      proplist_t	*langconfig = NULL;	

      if (lang)				/* try users language */
      {
	 strncpy (locale, lang, 2);
	 locale [2] = 0;
            
	 path = Calloc (strlen (PKGDATADIR) + strlen ("/WMWmakerconf.de") + 1,
			sizeof (char));
	 strcpy (path, PKGDATADIR);
	 strcat (path, "/WMWmakerconf.");
	 strcat (path, locale);
      
	 langconfig = PLGetProplistWithPath (path);
	 if (langconfig)		/* language supported */
	 {
	    proplist_t	*keys = PLGetAllDictionaryKeys (wmconfig);
	    proplist_t	*info = PLMakeString ("Info");
	    unsigned	n;
	    bool_t	NLS_entry_missing = NO;
	    
	    for (n = 0; n < PLGetNumberOfElements (keys); n++)
	    {
	       proplist_t *attribute = PLGetArrayElement (keys, n);
	       proplist_t *entry;
	       
	       if (entry = PLGetDictionaryEntry (langconfig, attribute))
	       {
		  proplist_t *info_entry = PLGetDictionaryEntry (entry, info);

		  if (info_entry)
		  {
		     proplist_t *dict;

		     dict = PLGetDictionaryEntry (wmconfig, attribute);
		     PLInsertDictionaryEntry (dict, info, info_entry);
		  }
		  else
		  {
		     NLS_entry_missing = YES;
		     warning ("NLS tooltips: missing translation of attribute '%s'.",
			      PLGetString (attribute));
		  }
	       }
	       else
	       {
		  NLS_entry_missing = YES;
		  warning ("NLS tooltips: missing translation of attribute '%s'.",
			   PLGetString (attribute));
	       }
	    }

	    if (NLS_entry_missing)
	       message ("**********************************************"
			"**********************************"
			"Dear Wmakerconf user. At least one of the new "
			"WindowMaker attributes hasn't\nbeen "
			"translated to your language yet. "
			"I'll appreciate your help to update the\ntooltips: "
			"please send the translation of the attributes shown"
			" above to\n"
			"hafner@informatik.uni-wuerzburg.de.\n"
			"**********************************************"
			"**********************************");
	    PLRelease (info);
	    PLRelease (keys);
	    /*
	     *  Get display font of native language
	     */
	    {
	       proplist_t *wmakerconf = PLMakeString ("Wmakerconf");
	       proplist_t *global     = PLGetDictionaryEntry (langconfig,
							      wmakerconf);

	       if (global)
	       {
		  proplist_t *fontstr = PLMakeString ("Font");
		  proplist_t *fntname = PLGetDictionaryEntry (global, fontstr);

		  if (fntname)
		     font = strdup (PLGetString (fntname));

		  PLRelease (fontstr);
	       }
	       PLRelease (wmakerconf);
	    }
	    PLRelease (langconfig);
	 }
	 else
	    message ("**********************************************"
		     "**********************************"
		     "Dear Wmakerconf user, this program is already equipped "
		     "with native language\nsupport. However, no volunteer has "
		     "been found to translate the tooltips to\nyour language yet."
		     " If you like to translate some of the tooltips texts"
		     "\nplease contact Ullrich Hafner "
		     "<hafner@informatik.uni-wuerzburg.de>\n"
		     "for further information.\n"
		     "**********************************************"
		     "**********************************");
	 
	 Free (path);
      }
   }

   /*
    *  Fill WindowMaker attributes not defined by the user with system defaults
    */
   {
      proplist_t *keys  = PLGetAllDictionaryKeys (wmconfig);
      proplist_t *pldef = PLMakeString ("Default");
      unsigned	 n;
      
      for (n = 0; n < PLGetNumberOfElements (keys); n++)
      {
	 proplist_t *attribute = PLGetArrayElement (keys, n);
	 
	 if (!PLGetDictionaryEntry (windowmaker, attribute)) /* not defined */
	 {
	    proplist_t *apl = PLGetDictionaryEntry (wmconfig, attribute);
	    proplist_t *val = PLGetDictionaryEntry (apl, pldef);
	    
	    PLInsertDictionaryEntry (windowmaker, attribute, val);
	 }
      }

      PLRelease (pldef);
      PLRelease (keys);
   }

   root_window (argc, argv, previews, font, wmconfig, windowmaker);

   PLRelease (wmconfig);
   PLRelease (windowmaker);
   
   return 0;
}
