/*
 * IceWM
 *
 * Copyright (C) 1997,1998 Marko Macek
 */

#include "icewm.h"

YWindow::YWindow(YWindow *parent, Window win) {
    fParentWindow = parent;
    fHandle = win;
    fFirstWindow = fLastWindow = fPrevWindow = fNextWindow = 0;
    flags = 0;
    fX = fY = 0;
    fWidth = fHeight = 1;
    fPointer = None;
    fStyle = 0;
    fGraphics = 0;
    fEnabled = true;
    fToplevel = false;
    fFocusedWindow = 0;
    unmapCount = 0;
    accel = 0;
#ifdef CONFIG_TOOLTIP
    fToolTip = 0;
#endif
    fDND = false;
    XdndDragSource = None;
    XdndDropTarget = None;
    fEventMask =
        KeyPressMask | KeyReleaseMask |
        LeaveWindowMask | EnterWindowMask | FocusChangeMask;

    if (fHandle != 0) {
        MSG(("adopting window %lX\n", fHandle));
        flags |= wfAdopted;
        create();
    } else if (fParentWindow == 0) {
        assert(desktop != 0);
        fParentWindow = desktop;
    }
    insertWindow();
}

YWindow::~YWindow() {
    removeWindow();
    while (accel) {
        YAccelerator *next = accel->next;
        delete accel;
        accel = next;
    }
#ifdef CONFIG_TOOLTIP
    if (fToolTip) {
        fToolTip->hide();
        if (fToolTipTimer && fToolTipTimer->getListener() == fToolTip) {
            fToolTipTimer->stopTimer();
            fToolTipTimer->setListener(0);
        }
        delete fToolTip;
    }
#endif
    if (fGraphics) {
        delete fGraphics;
        fGraphics = 0;
    }
    if (flags & wfCreated)
        destroy();
}

void YWindow::setFocus() {
    XSetInputFocus(app->display(),
                   handle(),
                   RevertToParent,
                   CurrentTime);
}

void YWindow::setStyle(unsigned long aStyle) {
    if (fStyle != aStyle) {
        fStyle = aStyle;

        if (flags & wfCreated) {
            if (aStyle & wsManager)
                fEventMask |=
                    SubstructureRedirectMask | SubstructureNotifyMask |
                    ButtonPressMask | ButtonReleaseMask | ButtonMotionMask;

            XSelectInput(app->display(), fHandle, fEventMask);
        }
    }
}

Graphics &YWindow::getGraphics() {
    if (fGraphics == 0)
        fGraphics = new Graphics(this);
    return *fGraphics;
}

void YWindow::repaint() {
    if ((flags & (wfCreated | wfVisible)) == (wfCreated | wfVisible)) {
        Graphics &g = getGraphics();

        paint(g, 0, 0, width(), height());
    }
}

void YWindow::repaintFocus() {
    if ((flags & (wfCreated | wfVisible)) == (wfCreated | wfVisible)) {
        Graphics &g = getGraphics();

        paintFocus(g, 0, 0, width(), height());
    }
}

void YWindow::create() {
    if (flags & wfCreated)
        return ;
    if (fHandle == 0) {
        XSetWindowAttributes attributes;
        unsigned int attrmask = CWEventMask;

        fEventMask |=
            ExposureMask |
            ButtonPressMask | ButtonReleaseMask | ButtonMotionMask;
        if (fParentWindow == desktop && !(fStyle & wsOverrideRedirect))
            fEventMask |= StructureNotifyMask;
        if (fStyle & wsManager)
            fEventMask |=
                SubstructureRedirectMask | SubstructureNotifyMask |
                ButtonPressMask | ButtonReleaseMask | ButtonMotionMask;
        
        if (fStyle & wsSaveUnder) {
            attributes.save_under = True;
            attrmask |= CWSaveUnder;
        }
        if (fStyle & wsOverrideRedirect) {
            attributes.override_redirect = True;
            attrmask |= CWOverrideRedirect;
        }
        if (fPointer != None) {
            attrmask |= CWCursor;
            attributes.cursor = fPointer;
        }
        attributes.event_mask = fEventMask;
        fHandle = XCreateWindow(app->display(),
                                parent()->handle(),
                                x(), y(), width(), height(),
                                0,
                                CopyFromParent,
                                InputOutput,
                                CopyFromParent,
                                attrmask,
                                &attributes);

        MSG(("window created: 0x%lX at (%d:%d %dx%d)", fHandle, x(), y(), width(), height()));
        if (flags & wfVisible)
            XMapWindow(app->display(), fHandle);
    } else {
        XWindowAttributes attributes;

        XGetWindowAttributes(app->display(),
                             fHandle,
                             &attributes);
        fX = attributes.x;
        fY = attributes.y;
        fWidth = attributes.width;
        fHeight = attributes.height;

        MSG(("window initial geometry (%d:%d %dx%d)",
             fX, fY, fWidth, fHeight));

        if (attributes.map_state != IsUnmapped)
            flags |= wfVisible;
        else
            flags &= ~wfVisible;

        fEventMask |=
            StructureNotifyMask |
            ColormapChangeMask |
            PropertyChangeMask;

        if (fStyle & wsManager)
            fEventMask |=
                SubstructureRedirectMask | SubstructureNotifyMask |
                ButtonPressMask | ButtonReleaseMask | ButtonMotionMask;
        
        XSelectInput(app->display(), fHandle, fEventMask);
    }
    XSaveContext(app->display(), fHandle, windowContext, (XPointer)this);
    flags |= wfCreated;
}

void YWindow::destroy() {
    if (flags & wfCreated) {
        XDeleteContext(app->display(), fHandle, windowContext);
        if (!(flags & wfDestroyed)) {
            if (!(flags & wfAdopted)) {
                XDestroyWindow(app->display(), fHandle);
            } else {
                XSelectInput(app->display(), fHandle, NoEventMask); 
            }
            flags |= wfDestroyed;
        }
        flags &= ~wfCreated;
    }
}
void YWindow::removeWindow() {
    if (fParentWindow) {
        if (fParentWindow->fFocusedWindow == this)
            fParentWindow->fFocusedWindow = 0;
        if (fPrevWindow)
            fPrevWindow->fNextWindow = fNextWindow;
        else
            fParentWindow->fFirstWindow = fNextWindow;
        
        if (fNextWindow)
            fNextWindow->fPrevWindow = fPrevWindow;
        else
            fParentWindow->fLastWindow = fPrevWindow;
    }
    fPrevWindow = fNextWindow = 0;
}

void YWindow::insertWindow() {
    if (fParentWindow) {
        fNextWindow = fParentWindow->fFirstWindow;
        fPrevWindow = 0;

        if (fNextWindow)
            fNextWindow->fPrevWindow = this;
        else
            fParentWindow->fLastWindow = this;
        fParentWindow->fFirstWindow = this;
    }
}

void YWindow::reparent(YWindow *parent, int x, int y) {
    //flags &= ~wfVisible; // don't unmap when we get UnmapNotify
    if (flags & wfVisible) {
        flags |= wfUnmapped;
        unmapCount++;
    }

    removeWindow();
    fParentWindow = parent;
    insertWindow();

    XReparentWindow(app->display(),
                    handle(),
                    parent->handle(),
                    x, y);
    fX = x;
    fY = y;
}

Window YWindow::handle() {
    if (!(flags & wfCreated))
        create();

    return fHandle;
}

void YWindow::show() {
    if (!(flags & wfVisible)) {
        flags |= wfVisible;
        XMapWindow(app->display(), handle());
    }
}

void YWindow::hide() {
    if (flags & wfVisible) {
        flags &= ~wfVisible;
        flags |= wfUnmapped;
        unmapCount++;
        XUnmapWindow(app->display(), handle());
    }
}

void YWindow::raise() {
    XRaiseWindow(app->display(), handle());
}

void YWindow::lower() {
    XLowerWindow(app->display(), handle());
}

void YWindow::handleEvent(const XEvent &event) {
    switch (event.type) {
    case KeyPress:
    case KeyRelease: handleKey(event.xkey); break;
    case ButtonPress:
        captureEvents();
        handleButton(event.xbutton);
        break;
    case ButtonRelease:
        releaseEvents();
        handleButton(event.xbutton);
        break;
    case MotionNotify: 
         {
             XEvent new_event, old_event;

             old_event = event;
             while (XPending(app->display()) > 0 &&
                    XCheckMaskEvent(app->display(),
                                 KeyPressMask |
                                 KeyReleaseMask |
                                 ButtonPressMask | 
                                 ButtonReleaseMask |
                                 ButtonMotionMask,
                                 &new_event) == True)
             {
                 if (event.type != new_event.type) {
                    XPutBackEvent(app->display(), &new_event);
                    handleMotion(event.xmotion);
                    return ;
                 } else {
                     old_event = new_event;
                 }
             }
             handleMotion(old_event.xmotion);
         } 
         break;

    case EnterNotify:
    case LeaveNotify: handleCrossing(event.xcrossing); break;
    case FocusIn:
    case FocusOut: handleFocus(event.xfocus); break;
    case PropertyNotify: handleProperty(event.xproperty); break;
    case ColormapNotify: handleColormap(event.xcolormap); break;
    case MapRequest: handleMapRequest(event.xmaprequest); break;
    case ConfigureRequest:
        handleConfigureRequest(event.xconfigurerequest);
        break;
    case ConfigureNotify: handleConfigure(event.xconfigure); break;
    case DestroyNotify: handleDestroyWindow(event.xdestroywindow); break;
    case Expose: handleExpose(event.xexpose); break;
    case GraphicsExpose: handleGraphicsExpose(event.xgraphicsexpose); break;
    case MapNotify: handleMap(event.xmap); break;
    case UnmapNotify: handleUnmap(event.xunmap); break;
    case ClientMessage: handleClientMessage(event.xclient); break;
#ifdef SHAPE
    default:
        if (shapesSupported && event.type == (shapeEventBase + ShapeNotify))
            handleShapeNotify(*(XShapeEvent *)&event);
        break;
#endif
    }
}

void YWindow::handleExpose(const XExposeEvent &expose) {
    Graphics &g = getGraphics();
    XRectangle r;

    r.x = expose.x;
    r.y = expose.y;
    r.width = expose.width;
    r.height = expose.height;
    
    XSetClipRectangles(app->display(), g.handle(),
                       0, 0, &r, 1, Unsorted);
    paint(g,
          expose.x,
          expose.y,
          expose.width,
          expose.height);

    XSetClipMask(app->display(), g.handle(), None);
    //XFlush(app->display());
}

void YWindow::handleGraphicsExpose(const XGraphicsExposeEvent &graphicsExpose) {
    Graphics &g = getGraphics();
    XRectangle r;

    r.x = graphicsExpose.x;
    r.y = graphicsExpose.y;
    r.width = graphicsExpose.width;
    r.height = graphicsExpose.height;
    
    XSetClipRectangles(app->display(), g.handle(),
                       0, 0, &r, 1, Unsorted);
    paint(g,
          graphicsExpose.x,
          graphicsExpose.y,
          graphicsExpose.width,
          graphicsExpose.height);

    XSetClipMask(app->display(), g.handle(), None);
}

void YWindow::handleConfigure(const XConfigureEvent &configure) {
    if (configure.window == handle())
        if (configure.x != fX ||
            configure.y != fY ||
            (unsigned int)configure.width != fWidth ||
            (unsigned int)configure.height != fHeight)
        {
            fX = configure.x;
            fY = configure.y;
            fWidth = configure.width;
            fHeight = configure.height;
            
            this->configure(fX, fY, fWidth, fHeight);
        }
}

bool YWindow::handleKey(const XKeyEvent &key) {
    if (key.type == KeyPress) {
        KeySym k = XKeycodeToKeysym(app->display(), key.keycode, 0);
        unsigned int m = KEY_MODMASK(key.state);

        if (accel) {
            YAccelerator *a;

            for (a = accel; a; a = a->next) {
                //printf("%c %d - %c %d %d\n", k, k, a->key, a->key, a->mod);
                if (m == a->mod && k == a->key)
                    if (a->win->handleKey(key) == true)
                        return true;
            }
            if (ISLOWER(k)) {
                k = TOUPPER(k);
                for (a = accel; a; a = a->next)
                    if (m == a->mod && k == a->key)
                        if (a->win->handleKey(key) == true)
                            return true;
            }
        }
        if (k == XK_Tab && fFirstWindow) {
            if (m == 0) {
                nextFocus();
                return true;
            } else if (m == ShiftMask) {
                prevFocus();
                return true;
            }
        }
    }
    return false;
}

YWindow *YWindow::fClickWindow = 0;
Time YWindow::fClickTime = 0;
int YWindow::fClickCount = 0;
XButtonEvent YWindow::fClickEvent = { 0 };
int YWindow::fClickDrag = 0;
unsigned int YWindow::fClickButton = 0;
unsigned int YWindow::fClickButtonDown = 0;

void YWindow::handleButton(const XButtonEvent &button) {
    int x_dif = button.x_root - fClickEvent.x_root;
    int y_dif = button.y_root - fClickEvent.y_root;
    
    unsigned int motionDelta =
        ((x_dif < 0) ? - x_dif : x_dif) +
        ((y_dif < 0) ? - y_dif : y_dif);
    
    if (button.type == ButtonPress) {
        fClickDrag = 0;

        if (fClickWindow != this) {
            fClickWindow = this;
            fClickCount = 1;
        } else {
            if ((button.time - fClickTime < MultiClickTime) &&
                fClickButton == button.button &&
                motionDelta < ClickMotionDistance &&
                button.x >= 0 && button.y >= 0 &&
                button.x < int(width()) && button.y < int(height()))
                fClickCount++;
            else
                fClickCount = 1;
        }
        fClickEvent = button;
        fClickButton = fClickButtonDown = button.button;
        fClickTime = button.time;
    } else if (button.type == ButtonRelease) {
        if ((fClickWindow == this) &&
            !fClickDrag &&
            fClickCount > 0 &&
            fClickButtonDown == button.button &&
            motionDelta < ClickMotionDistance &&
            button.x >= 0 && button.y >= 0 &&
            button.x < int(width()) && button.y < int(height()))
        {
            fClickButtonDown = 0;
            handleClick(fClickEvent, button, fClickCount);
        } else {
            fClickWindow = 0;
            fClickCount = 1;
            fClickButtonDown = 0;
            fClickButton = 0;
            if (fClickDrag)
                handleEndDrag(fClickEvent, button);
        }
    }
#ifdef CONFIG_TOOLTIP
    if (fToolTip) {
        fToolTip->hide();
        if (fToolTipTimer && fToolTipTimer->getListener() == fToolTip) {
            fToolTipTimer->stopTimer();
            fToolTipTimer->setListener(0);
        }
    }
#endif
}

void YWindow::handleMotion(const XMotionEvent &motion) {
    if (fClickButtonDown) {
        if (fClickDrag) {
            handleDrag(fClickEvent, motion);
        } else {
            int x_dif = motion.x_root - fClickEvent.x_root;
            int y_dif = motion.y_root - fClickEvent.y_root;
            
            unsigned int motionDelta =
                ((x_dif < 0) ? - x_dif : x_dif) +
                ((y_dif < 0) ? - y_dif : y_dif);
            
            if ((motion.time - fClickTime > ClickMotionDelay) || // !!! test
                (motionDelta >= ClickMotionDistance)) {
                fClickDrag = 1;
                handleBeginDrag(fClickEvent, motion);
            }
        }
    }
}

#ifdef CONFIG_TOOLTIP
YTimer *YWindow::fToolTipTimer = 0;
#endif

void YWindow::setToolTip(const char *tip) {
#ifdef CONFIG_TOOLTIP
    if (fToolTip) {
        fToolTip->hide();
        if (fToolTipTimer && fToolTipTimer->getListener() == fToolTip) {
            fToolTipTimer->stopTimer();
            fToolTipTimer->setListener(0);
        }
        if (!tip) {
            delete fToolTip;
            fToolTip = 0;
        }
    }
    if (tip) {
        if (!fToolTip)
            fToolTip = new YToolTip();
        if (fToolTip)
            fToolTip->setText(tip);
    }
#endif
}

void YWindow::handleCrossing(const XCrossingEvent &crossing) {
#ifdef CONFIG_TOOLTIP
    if (fToolTip) {
        if (crossing.type == EnterNotify) {
            if (fToolTipTimer == 0)
                fToolTipTimer = new YTimer();
            if (fToolTipTimer) {
                fToolTipTimer->setInterval(ToolTipDelay);
                fToolTipTimer->setListener(fToolTip);
                fToolTipTimer->startTimer();
                fToolTip->locate(this, crossing);
            }
        } else if (crossing.type == LeaveNotify) {
            fToolTip->hide();
            if (fToolTipTimer && fToolTipTimer->getListener() == fToolTip) {
                fToolTipTimer->stopTimer();
                fToolTipTimer->setListener(0);
            }
        }
    }
#endif
}

void YWindow::handleProperty(const XPropertyEvent &) {
}

void YWindow::handleColormap(const XColormapEvent &) {
}

void YWindow::handleFocus(const XFocusChangeEvent &) {
}

void YWindow::handleClientMessage(const XClientMessageEvent &message) {
    if (message.message_type == _XA_WM_PROTOCOLS 
        && message.format == 32
        && message.data.l[0] == (long)_XA_WM_DELETE_WINDOW)
    {
        handleClose();
    } else if (message.message_type == XA_XdndEnter ||
               message.message_type == XA_XdndLeave ||
               message.message_type == XA_XdndPosition ||
               message.message_type == XA_XdndStatus ||
               message.message_type == XA_XdndDrop ||
               message.message_type == XA_XdndFinished)
    {
        handleXdnd(message);
    }
}

#if 0
    virtual void handleVisibility(const XVisibilityEvent &visibility);
    virtual void handleCreateWindow(const XCreateWindowEvent &createWindow);
#endif

void YWindow::handleMap(const XMapEvent &) {
    //flags |= wfVisible;
}

void YWindow::handleUnmap(const XUnmapEvent &) {
    if (flags & wfUnmapped) {
        unmapCount--;
        if (unmapCount == 0)
            flags &= ~wfUnmapped;
    }
    else
        flags &= ~wfVisible;
}

void YWindow::handleDestroyWindow(const XDestroyWindowEvent &destroyWindow) {
    if (destroyWindow.window == fHandle)
        flags |= wfDestroyed;
}

void YWindow::handleConfigureRequest(const XConfigureRequestEvent &) {
}

void YWindow::handleMapRequest(const XMapRequestEvent &) {
}

#ifdef SHAPE
void YWindow::handleShapeNotify(const XShapeEvent &) {
}
#endif

void YWindow::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &/*up*/, int /*count*/) {
}

void YWindow::handleBeginDrag(const XButtonEvent &/*down*/, const XMotionEvent &/*motion*/) {
}

void YWindow::handleDrag(const XButtonEvent &/*down*/, const XMotionEvent &/*motion*/) {
}

void YWindow::handleEndDrag(const XButtonEvent &/*down*/, const XButtonEvent &/*up*/) {
}

void YWindow::paint(Graphics &g, int x, int y, unsigned int w, unsigned int h) {
    g.fillRect(x, y, w, h);
}

void YWindow::paintFocus(Graphics &/*g*/, int /*x*/, int /*y*/, unsigned int /*w*/, unsigned int /*h*/) {
}

void YWindow::setGeometry(int x, int y, unsigned int width, unsigned int height) {
    if (x != fX ||
        y != fY ||
        width != fWidth ||
        height != fHeight)
    {
        fX = x;
        fY = y;
        fWidth = width;
        fHeight = height;

        if (flags & wfCreated) {
            XMoveResizeWindow(app->display(),
                              fHandle,
                              fX, fY, fWidth, fHeight);
        }

        configure(fX, fY, fWidth, fHeight);
    }
}

void YWindow::setPosition(int x, int y) {
    if (x != fX || y != fY) {
        fX = x;
        fY = y;

        if (flags & wfCreated)
            XMoveWindow(app->display(), fHandle, fX, fY);

        configure(fX, fY, width(), height());
    }
}

void YWindow::setSize(unsigned int width, unsigned int height) {
    if (width != fWidth || height != fHeight) {
        fWidth = width;
        fHeight = height;

        if (flags & wfCreated)
            XResizeWindow(app->display(), fHandle, fWidth, fHeight);

        configure(x(), y(), fWidth, fHeight);
    }
}

void YWindow::mapToGlobal(int &x, int &y) {
    int dx, dy;
    Window child;

    XTranslateCoordinates(app->display(),
                          handle(),
                          desktop->handle(),
                          x, y,
                          &dx, &dy, &child);
    x = dx;
    y = dy;
}

void YWindow::mapToLocal(int &x, int &y) {
    int dx, dy;
    Window child;

    XTranslateCoordinates(app->display(),
                          desktop->handle(),
                          handle(),
                          x, y,
                          &dx, &dy, &child);
    x = dx;
    y = dy;
}

void YWindow::configure(int, int, unsigned int, unsigned int) {
}

void YWindow::setPointer(Cursor pointer) {
    fPointer = pointer;

    if (flags & wfCreated) {
        XSetWindowAttributes attributes;

        attributes.cursor = fPointer;

        XChangeWindowAttributes(app->display(),
                                handle(),
                                CWCursor,
                                &attributes);
    }
}

void YWindow::setGrabPointer(Cursor pointer) {
    XChangeActivePointerGrab(app->display(),
                             ButtonPressMask | PointerMotionMask | ButtonReleaseMask,
                             pointer, CurrentTime);//app->getEventTime());
}

void YWindow::grabKey(int key, unsigned int modifiers) {
    KeyCode keycode = XKeysymToKeycode(app->display(), key);
    if (keycode != 0) {
        XGrabKey(app->display(), keycode, modifiers, handle(), True,
                 GrabModeAsync, GrabModeAsync);
        XGrabKey(app->display(), keycode, modifiers | LockMask, handle(), True,
                 GrabModeAsync, GrabModeAsync);

        if (app->NumLockMask != 0) {
            XGrabKey(app->display(), keycode, modifiers | app->NumLockMask, handle(), True,
                     GrabModeAsync, GrabModeAsync);
            XGrabKey(app->display(), keycode, modifiers | LockMask | app->NumLockMask, handle(), True,
                     GrabModeAsync, GrabModeAsync);
        }
    }
}

void YWindow::grabButton(int button, unsigned int modifiers) {
    // !!! fix Lock+Num
    XGrabButton(app->display(),
                button, modifiers,
                handle(), True,
                ButtonPressMask | ButtonReleaseMask | ButtonMotionMask,
                GrabModeAsync, GrabModeAsync,
                None, None);
}

void YWindow::captureEvents() {
    app->captureGrabEvents(this);
}

void YWindow::releaseEvents() {
    app->releaseGrabEvents(this);
}

void YWindow::donePopup(YPopupWindow * /*command*/) {
}

void YWindow::handleClose() {
}

bool YWindow::isFocusTraversable() {
    return false;
}

bool YWindow::isFocused() {
    if (parent() == 0)
        return true;
    else
        return (parent()->fFocusedWindow == this) && parent()->isFocused();
}

void YWindow::setEnabled(bool enable) {
    if (enable != fEnabled) {
        fEnabled = enable;
        repaint();
    }
}

void YWindow::nextFocus() {
    YWindow *w1;
    YWindow *w = fFocusedWindow ? fFocusedWindow : fLastWindow;

    w1 = w;
    if (w) do {
        w = w->fPrevWindow;
        if (w == 0)
            w = fLastWindow;
        if (w->isFocusTraversable())
            break;
    } while (w != w1);
    if (w) w->requestFocus();
}

void YWindow::prevFocus() {
    YWindow *w1;
    YWindow *w = (fFocusedWindow ? fFocusedWindow : fFirstWindow);

    w1 = w; 
    if (w) do {
        w = w->fNextWindow;
        if (w == 0)
            w = fFirstWindow;
        if (w->isFocusTraversable())
            break;
    } while (w != w1);
    if (w) w->requestFocus();
}

void YWindow::requestFocus() {
    if (parent())
        parent()->setFocus(this);
}

void YWindow::setFocus(YWindow *window) {
    if (window != fFocusedWindow) {
        YWindow *oldFocus = fFocusedWindow;

        fFocusedWindow = window;

        if (oldFocus)
            oldFocus->lostFocus();
        if (fFocusedWindow)
            fFocusedWindow->gotFocus();
    }
}
void YWindow::gotFocus() {
    if (parent() && parent()->isFocused())
        setFocus();
    repaintFocus();
}

void YWindow::lostFocus() {
    repaintFocus();
}

void YWindow::installAccelerator(unsigned int key, unsigned int mod, YWindow *win) {
    key = TOUPPER(key);
    if (fToplevel || fParentWindow == 0) {
        YAccelerator **pa = &accel, *a;
        
        while (*pa) {
            a = *pa;
            if (a->key == key &&
                a->mod == mod &&
                a->win == win)
            {
                assert(1 == 0);
                return ;
            } else
                pa = &(a->next);
        }
        
        a = new YAccelerator;
        if (a == 0)
            return ;
        
        a->key = key;
        a->mod = mod;
        a->win = win;
        a->next = accel;
        accel = a;
    } else parent()->installAccelerator(key, mod, win);
}

void YWindow::removeAccelerator(unsigned int key, unsigned int mod, YWindow *win) {
    key = TOUPPER(key);
    if (fToplevel || fParentWindow == 0) {
        YAccelerator **pa = &accel, *a;
        
        while (*pa) {
            a = *pa;
            if (a->key == key &&
                a->mod == mod &&
                a->win == win)
            {
                *pa = a->next;
                delete a;
            } else
                pa = &(a->next);
        }
    } else parent()->removeAccelerator(key, mod, win);
}

void YWindow::setWinGravity(int gravity) {
    unsigned long eventmask = CWWinGravity;
    XSetWindowAttributes attributes;
    attributes.win_gravity = gravity;

    XChangeWindowAttributes(app->display(), handle(), eventmask, &attributes);
}

const Atom XdndCurrentVersion = 3;

void YWindow::setDND(bool enabled) {
    if (fDND != enabled) {
        fDND = enabled;

        if (fDND) {
            XChangeProperty(app->display(), handle(),
                            XA_XdndAware, XA_ATOM, // ??? ATOM
                            32, PropModeReplace,
                            (unsigned char *)&XdndCurrentVersion, 1);
        } else {
            XDeleteProperty(app->display(), handle(), XA_XdndAware);
        }
    }
}

void YWindow::XdndStatus(bool acceptDrop, Atom dropAction) {
    XClientMessageEvent msg;
    int x_root = 0, y_root = 0;

    mapToGlobal(x_root, y_root);
    
    msg.type = ClientMessage;
    msg.display = app->display();
    msg.window = XdndDragSource;
    msg.message_type = XA_XdndStatus;
    msg.format = 32;
    msg.data.l[0] = handle();
    msg.data.l[1] = (acceptDrop ? 0x00000001 : 0x00000000) | 2;
    msg.data.l[2] = (x_root << 16) + y_root;
    msg.data.l[3] = (width() << 16) + height();
    msg.data.l[4] = dropAction;
    XSendEvent(app->display(), XdndDragSource, False, 0L, (XEvent *)&msg);
}

void YWindow::handleXdnd(const XClientMessageEvent &message) {
    if (message.message_type == XA_XdndEnter) {
        printf("XdndEnter source=%lX\n", message.data.l[0]);
        XdndDragSource = message.data.l[0];
    } else if (message.message_type == XA_XdndLeave) {
        printf("XdndLeave source=%lX\n", message.data.l[0]);
        if (XdndDropTarget) {
            YWindow *win;

            if (XFindContext(app->display(), XdndDropTarget, windowContext,
                             (XPointer *)&win) == 0)
                win->handleDNDLeave();
        }
        XdndDragSource = None;
    } else if (message.message_type == XA_XdndPosition &&
               XdndDragSource != 0)
    {
        Window target, child;
        int x, y, nx, ny;

        XdndDragSource = message.data.l[0];
        x = int(message.data.l[2] >> 16);
        y = int(message.data.l[2] & 0xFFFF);

        target = handle();
        
        printf("XdndPosition source=%lX %d:%d time=%ld action=%ld window=%ld\n",
                 message.data.l[0],
                 x, y,
                 message.data.l[3],
                 message.data.l[4],
                 XdndDropTarget);


        do {
            if (XTranslateCoordinates(app->display(),
                                      desktop->handle(), target,
                                      x, y, &nx, &ny, &child))
            {
                if (child)
                    target = child;
            } else {
                target = 0;
                break;
            }
        } while (child);

        if (target) {
            if (XdndDropTarget) {
                YWindow *win;
                
                if (XFindContext(app->display(), XdndDropTarget, windowContext,
                                 (XPointer *)&win) == 0)
                    win->handleDNDLeave();
            }
            XdndDropTarget = target;
            if (XdndDropTarget) {
                YWindow *win;
                
                if (XFindContext(app->display(), XdndDropTarget, windowContext,
                                 (XPointer *)&win) == 0)
                    win->handleDNDEnter();
            }
        }
        printf("XdndPosition %d:%d target=%ld\n",
               nx, ny, XdndDropTarget);
        XdndStatus(false, None);
        /*{
            XClientMessageEvent msg;

            msg.data.l[0] = handle();
            msg.data.l[1] = (false ? 0x00000001 : 0x00000000) | 2;
            msg.data.l[2] = 0; //(x << 16) + y;
            msg.data.l[3] = 0;//(1 << 16) + 1;
            msg.data.l[4] = None;
            XSendEvent (app->display(), XdndDragSource, True, 0L, (XEvent *)&msg);
        }*/
    } else if (message.message_type == XA_XdndStatus) {
        printf("XdndStatus\n");
    } else if (message.message_type == XA_XdndDrop) {
        printf("XdndDrop\n");
    } else if (message.message_type == XA_XdndFinished) {
        printf("XdndFinished\n");
    }
}

void YWindow::handleDNDEnter() {
}

void YWindow::handleDNDLeave() {
}

YDesktop::YDesktop(YWindow *aParent = 0, Window win = 0):
    YWindow(aParent, win)
{
}
YDesktop::~YDesktop() {
}
    
void YDesktop::resetColormap(bool /*active*/) { /// !!!
}

void YDesktop::resetInputFocus() {
}
