/*
 * IceWM
 *
 * Copyright (C) 1997,1998 Marko Macek
 */

#include "icewm.h"

YColor *menuBg = 0;
YColor *menuItemFg = 0;
YColor *activeMenuItemBg = 0;
YColor *activeMenuItemFg = 0;
YColor *disabledMenuItemFg = 0;

YFont *menuFont = 0;

bool YApplication::popup(YWindow *forWindow, YPopupWindow *popup) {
    assert(popup != 0);
    if (fPopup == 0) {
        Cursor changePointer = (popup->popupFlags() & YPopupWindow::pfNoPointerChange) ? None : rightPointer;
        
        if (!grabEvents(forWindow ? forWindow : popup, changePointer,
                        ButtonPressMask | ButtonReleaseMask |
                        (menuMouseTracking ? PointerMotionMask : ButtonMotionMask)))
        {
            popup->cancelPopup();
            return false;
        }
    }
    popup->setPrevPopup(fPopup);
    fPopup = popup;
    return true;
}

void YApplication::popdown(YPopupWindow *popdown) {
    if (popdown != fPopup) {
	MSG(("popdown: 0x%lX  fPopup: 0x%lX", popdown, fPopup));
        return ;
    }
    assert(fPopup != 0);
    fPopup = fPopup->prevPopup();
    
    if (fPopup == 0)
        releaseEvents();
}

YPopupWindow::YPopupWindow(YWindow *aParent): YWindow(aParent) {
    fForWindow = 0;
    fPrevPopup = 0;
    fCommandListener = 0;
    fFlags = 0;
    setStyle(wsSaveUnder | wsOverrideRedirect);
}

YPopupWindow::~YPopupWindow() {
}

void YPopupWindow::updatePopup() {
}

void YPopupWindow::sizePopup() {
}

void YPopupWindow::activatePopup() {
}

void YPopupWindow::deactivatePopup() {
}

bool YPopupWindow::popup(YWindow *forWindow,
                         CommandListener *commandListener,
                         PopDownListener *popDown,
                         unsigned int flags)
{
    fPrevPopup = 0;
    fFlags = flags;
    fForWindow = forWindow;
    fCommandListener = commandListener;
    fPopDownListener = popDown;

    raise();
    show();

    app->popup(forWindow, this);
    activatePopup();
    return true;
}

bool YPopupWindow::popup(YWindow *forWindow,
                         CommandListener *commandListener,
                         PopDownListener *popDown,
                         int x, int y, int x_delta, int y_delta, unsigned int flags) {

    if ((flags & pfPopupMenu) &&
        (wmLook == lookWarp3 || wmLook == lookWarp4))
        flags |= pfFlipVertical;
    
    fFlags = flags;

    updatePopup();

    sizePopup();

    /* !!! FIX this to maximize visible area */
    if ((x + width() > desktop->width()) || (fFlags & pfFlipHorizontal))
        if (fFlags & (pfCanFlipHorizontal | pfFlipHorizontal)) {
            x -= width() + x_delta;
            fFlags |= pfFlipHorizontal;
        } else
            x = desktop->width() - width();
    if ((y + height() > desktop->height()) || (fFlags & pfFlipVertical))
        if (fFlags & (pfCanFlipVertical | pfFlipVertical)) {
            y -= height() + y_delta;
            fFlags |= pfFlipVertical;
        } else
            y = desktop->height() - height();
    if (x < 0)
        if (fFlags & pfCanFlipHorizontal)
            x += width() + x_delta;
        else
            x = 0;
    if (y < 0)
        if (fFlags & pfCanFlipVertical)
            y += height() + y_delta;
        else
            y = 0;

    if (forWindow == 0) {
        if ((x + width() > desktop->width()))
            x = desktop->width() - width();

        if (x < 0)
            x = 0;

        if ((y + height() > desktop->height()))
            y = desktop->height() - height();

        if (y < 0)
            y = 0;
    }

    setPosition(x, y);

    return popup(forWindow, commandListener, popDown, fFlags);
}

void YPopupWindow::popdown() {
    if (fPopDownListener)
        fPopDownListener->handlePopDown(this);
    
    deactivatePopup();
    app->popdown(this);
    hide();

    fFlags = 0;
    fForWindow = 0;
    fCommandListener = 0;
}

void YPopupWindow::cancelPopup() {
    if (fForWindow) {
        fForWindow->donePopup(this);
    } else {
        popdown();
    }
}

void YPopupWindow::finishPopup(WMCommand command, void *context, unsigned int modifiers) {
    CommandListener *cmd = fCommandListener;
    
    while (app->popup())
        app->popup()->cancelPopup();

    if (cmd)
        cmd->handleCommand(command, context, modifiers);
    else //!!!
        fprintf(stderr, "no menu listener.\n");
}

bool YPopupWindow::handleKey(const XKeyEvent &/*key*/) {
    return true;
}

void YPopupWindow::handleButton(const XButtonEvent &button) {
    if (button.x_root >= x() &&
        button.y_root >= y() &&
        button.x_root < int (x() + width()) &&
        button.y_root < int (y() + height()) &&
        button.window == handle())
    {
        YWindow::handleButton(button);
    } else {
        if (fForWindow) {
            XEvent xev;

            xev.xbutton = button;

            app->handleGrabEvent(fForWindow, xev);
        } else {
            popdown();
        }       
    }
}

void YPopupWindow::handleMotion(const XMotionEvent &motion) {
    if (motion.x_root >= x() &&
        motion.y_root >= y() &&
        motion.x_root < int (x() + width()) &&
        motion.y_root < int (y() + height()) &&
       motion.window == handle())
    {
        YWindow::handleMotion(motion);
    } else {
        if (fForWindow) {
            XEvent xev;

            xev.xmotion = motion;
            
            app->handleGrabEvent(fForWindow, xev);
        }
    }
}

YMenu::YMenu(YWindow *parent): YPopupWindow(parent) {
    fItems = 0;
    fItemCount = 0;
    paintedItem = selectedItem = -1;
    fPopup = 0;
    fPopupActive = 0;
}

YMenu::~YMenu() {
    if (fPopup) {
        fPopup->popdown();
        fPopup = 0;
    }

    for (int i = 0; i < fItemCount; i++)
        delete fItems[i];
    FREE(fItems);
}

void YMenu::activatePopup() {
    if (popupFlags() & pfButtonDown)
        selectedItem = -1;
    else
        focusItem(findActiveItem(itemCount() - 1, 1), 0, 0);
}

void YMenu::deactivatePopup() {
    if (fPopup) {
        fPopup->popdown();
        fPopup = 0;
    }
}

void YMenu::donePopup(YPopupWindow *popup) {
    if (fPopup)
        assert(popup == fPopup);
    if (fPopup) {
        fPopup->popdown();
        fPopup = 0;
        if (selectedItem != -1)
            if (item(selectedItem)->submenu() == popup)
                paintItems();
    }
}

int YMenu::onCascadeButton(int selItem, int x, int /*y*/) {
    if (selItem != -1 &&
        item(selItem)->command() != cmdSubmenu &&
        item(selItem)->submenu())
    {
        int fontHeight = menuFont->height() + 1;

        unsigned int h = fontHeight;

        if (item(selItem)->getPixmap() && item(selItem)->getPixmap()->height() > h)
            h = item(selItem)->getPixmap()->height();

        if (x <= int(width() - h - 4))
            return 1;
    }
    return 0;
}

void YMenu::focusItem(int itemNo, int submenu, int byMouse) {
    selectedItem = itemNo;

    YMenu *sub = 0;
    if (selectedItem != -1)
        sub = item(selectedItem)->submenu();

    if (sub != fPopup) {
        int repaint = 0;
        
        if (fPopup) {
            fPopup->popdown();
            fPopup = 0;
            repaint = 1;
        }

        if (submenu && sub && item(selectedItem)->isEnabled()) {
            int xp, yp;
            int l, t, r, b;

            getOffsets(l, t, r, b);
            findItemPos(selectedItem, xp, yp);
            sub->popup(this, commandListener(), 0,
                       x() + width() - r, y() + yp - t,
                       width() - r - l, -1,
                       YPopupWindow::pfCanFlipHorizontal |
                       (popupFlags() & YPopupWindow::pfFlipHorizontal) |
                       (byMouse ? (unsigned int)YPopupWindow::pfButtonDown : 0U));
            fPopup = sub;
            repaint = 1;
        }
        if (repaint && selectedItem != -1 && paintedItem == selectedItem &&
            item(selectedItem)->command() != cmdSubmenu)
            paintItems();
    }
    if (paintedItem != selectedItem)
        paintItems();
}

int YMenu::findActiveItem(int cur, int direction) {
    assert(direction == -1 || direction == 1);
    
    if (itemCount() == 0)
        return -1;

    if (cur == -1)
        if (direction == 1)
            cur = itemCount() - 1;
        else
            cur = 0;

    assert(cur >= 0 && cur < itemCount());

    int c = cur;
    do {
        c += direction;
        if (c < 0) c = itemCount() - 1;
        if (c >= itemCount()) c = 0;
    } while (c != cur && item(c)->command() == cmdSeparator);
    return c;
}

int YMenu::activateItem(int no, int byMouse, unsigned int modifiers) {
    assert(selectedItem == no && selectedItem != -1);
    
    if (item(selectedItem)->isEnabled()) {
        if (item(selectedItem)->command() == cmdSubmenu) {
            focusItem(selectedItem, 1, byMouse);
        } else {
            finishPopup(WMCommand(item(selectedItem)->command()),
                        item(selectedItem)->context(), modifiers);
        }
    } else {
        //XBell(app->display(), 50);
        return -1;
    }
    return 0;
}

bool YMenu::handleKey(const XKeyEvent &key) {
    if (key.type == KeyPress) {
        KeySym k = XKeycodeToKeysym(app->display(), key.keycode, 0);
        int m = KEY_MODMASK(key.state);

        if ((m & ~ShiftMask) == 0) {
            if (itemCount() > 0) {
                if (k == XK_Up)
                    focusItem(findActiveItem(selectedItem, -1), 0, 0);
                else if (k == XK_Down)
                    focusItem(findActiveItem(selectedItem, 1), 0, 0);
                else if (k == XK_Home)
                    focusItem(findActiveItem(itemCount() - 1, 1), 0, 0);
                else if (k == XK_End)
                    focusItem(findActiveItem(0, -1), 0, 0);
                else if (k == XK_Right)
                    focusItem(selectedItem, 1, 0);
                else if (k == XK_Left) {
                    if (prevPopup())
                        cancelPopup();
                } else if (k == XK_Return || k == XK_KP_Enter) {
                    if (selectedItem != -1 && item(selectedItem)->command() != cmdSeparator) {
                        activateItem(selectedItem, 0, key.state);
                        return true;
                    }
                } else if (k == XK_Escape) {
                    cancelPopup();
                } else if ((k < 256) && ((m & ~ShiftMask) == 0)) {
                    for (int i = 0; i < itemCount(); i++) {
                        int hot = item(i)->hotChar();
                        if (hot != -1 &&
                            TOUPPER(char(hot)) == TOUPPER(char(k)))
                        {
                            focusItem(i, 0, 0);
                            if (!(m & ShiftMask))
                                activateItem(i, 0, key.state);
                            return true;
                        }
                    }
                }
            }
        }
    }
    return YPopupWindow::handleKey(key);
}

void YMenu::handleButton(const XButtonEvent &button) {
    if (button.button != 0) {
        int selItem = findItem(button.x_root - x(), button.y_root - y());
        int submenu = onCascadeButton(selItem,
                                      button.x_root - x(),
                                      button.y_root - y()) ? 0 : 1;
        if (button.type == ButtonRelease && fPopupActive == fPopup && fPopup != 0 && submenu) {
            fPopup->popdown();
            fPopupActive = fPopup = 0;
            focusItem(selItem, 0, 1);
            if (submenu)
                paintItems();
            return ;
        } else if (button.type == ButtonPress) {
            fPopupActive = fPopup;
        }
        focusItem(selItem, submenu, 1);
        if (selectedItem != -1 &&
            button.type == ButtonRelease && 
            item(selectedItem)->command() != cmdSeparator &&
            (item(selectedItem)->command() == cmdSubmenu ||
             !item(selectedItem)->submenu() || !submenu)
           )
        {
            activateItem(selectedItem, 1, button.state);
            return ;
        }
        if (button.type == ButtonRelease &&
            (selectedItem == -1 || item(selectedItem)->command() == cmdSeparator))
            focusItem(findActiveItem(itemCount() - 1, 1), 0, 0);
    }
    YPopupWindow::handleButton(button);
}

void YMenu::handleMotion(const XMotionEvent &motion) {
    bool isButton =
        (motion.state & (Button1Mask |
                         Button2Mask |
                         Button3Mask |
                         Button4Mask |
                         Button5Mask)) ? true : false;
    
    if (menuMouseTracking || isButton) {
        int selItem = findItem(motion.x_root - x(), motion.y_root - y());
        if (selItem != -1 || app->popup() == this) {
            int submenu = onCascadeButton(selItem,
                                          motion.x_root - x(),
                                          motion.y_root - y()) 
                          && !(motion.state & ControlMask) ? 0 : 1;
            //if (selItem != -1)
                focusItem(selItem, submenu, 1);
        }
    }

    YPopupWindow::handleMotion(motion);
}

YMenu::YMenuItem::YMenuItem(const char *name, int aHotCharPos, const char *param, WMCommand command, YMenu *submenu, void *context) {
    if (name)
        fName = new char[strlen(name) + 1];
    else
        fName = 0;
    if (fName) strcpy(fName, name);
    if (param)
        fParam = new char[strlen(param) + 1];
    else
        fParam = 0;
    if (fParam) strcpy(fParam, param);
    fCommand = command;
    fContext = context;
    fSubmenu = submenu;
    fEnabled = 1;
    fHotCharPos = aHotCharPos;
    fPixmap = 0;
    fChecked = 0;
    if (!fName || fHotCharPos >= int(strlen(name)) || fHotCharPos < -1)
        fHotCharPos = -1;
}

YMenu::YMenuItem::~YMenuItem() {
    if (fSubmenu && !fSubmenu->isShared())
        delete fSubmenu;
    delete fName;
    delete fParam;
}

void YMenu::YMenuItem::setChecked(bool c) {
    fChecked = c;
}

void YMenu::YMenuItem::setPixmap(YPixmap *pixmap) {
    fPixmap = pixmap;
}

YMenu::YMenuItem *YMenu::addItem(const char *name, int hotCharPos, const char *param, WMCommand command, void *context) {
    return add(new YMenuItem(name, hotCharPos, param, command, 0, context));
}

YMenu::YMenuItem * YMenu::addSubmenu(const char *name, int hotCharPos, YMenu *submenu) {
    return add(new YMenuItem(name, hotCharPos, 0, cmdSubmenu, submenu, 0));
}

YMenu::YMenuItem * YMenu::addSeparator() {
    return add(new YMenuItem());
}

void YMenu::removeAll() {
    if (fPopup) {
        fPopup->popdown();
        fPopup = 0;
    }
    for (int i = 0; i < itemCount(); i++)
        delete fItems[i];
    FREE(fItems);
    fItems = 0;
    fItemCount = 0;
    paintedItem = selectedItem = -1;
    fPopup = 0;
}

YMenu::YMenuItem * YMenu::add(YMenuItem *item) {
    if (item) {
        YMenuItem **newItems = (YMenuItem **)REALLOC(fItems, sizeof(YMenuItem *) * ((fItemCount) + 1));
        if (newItems) {
            fItems = newItems;
            fItems[fItemCount++] = item;
            return item;
        } else {
            delete item;
            return 0;
        }
    }
    return item;
}

YMenu::YMenuItem *YMenu::findItem(WMCommand cmd) {
    for (int i = 0; i < itemCount(); i++)
        if (cmd == item(i)->command())
            return item(i);
    return 0;
}
YMenu::YMenuItem *YMenu::findSubmenu(YMenu *sub) {
    for (int i = 0; i < itemCount(); i++)
        if (sub == item(i)->submenu())
            return item(i);
    return 0;
}

void YMenu::enableCommand(WMCommand cmd) {
    for (int i = 0; i < itemCount(); i++)
        if (cmd == cmdNone || cmd == item(i)->command())
            item(i)->setEnabled(true);
}

void YMenu::disableCommand(WMCommand cmd) {
    for (int i = 0; i < itemCount(); i++)
        if (cmd == cmdNone || cmd == item(i)->command())
            item(i)->setEnabled(false);
}

int YMenu::getItemHeight(int itemNo, int &h, int &top, int &bottom, int &pad) {
    h = top = bottom = pad = 0;
    
    if (itemNo < 0 || itemNo > itemCount())
        return -1;

    if (item(itemNo)->command() == cmdSeparator) {
        top = 0;
        bottom = 0;
        pad = 1;
        if (wmLook == lookMetal)
            h = 3;
        else
            h = 4;
    } else {
        int fontHeight = menuFont->height() + 1;
        unsigned int ih = fontHeight;

        if (fontHeight < 16)
            fontHeight = 16;

        if (item(itemNo)->getPixmap() &&
            item(itemNo)->getPixmap()->height() > ih)
            ih = item(itemNo)->getPixmap()->height();

        if (wmLook == lookWarp4 || wmLook == lookWin95) {
            top = bottom = 0;
            pad = 1;
        } else if (wmLook == lookMetal) {
            top = bottom = 1;
            pad = 1;
        } else if (wmLook == lookMotif) {
            top = bottom = 2;
            pad = 1;
        } else if (wmLook == lookGtk) {
            top = bottom = 2;
            pad = 1;
        } else {
            top = 1;
            bottom = 2;
            pad = 1;
        }
        h = top + pad + ih + pad + bottom;
    }
    return 0;
}

void YMenu::getItemWidth(int i, int &iw, int &nw, int &pw) {
    iw = nw = pw = 0;

    if (item(i)->command() == cmdSeparator) {
        iw = 0;
        nw = 0;
        pw = 0;
    } else {
        YPixmap *p = item(i)->getPixmap();
        if (p)
            iw = p->height();

        char *name = item(i)->name();
        if (name)
            nw = menuFont->textWidth(name);

        char *param = item(i)->param();
        if (param)
            pw = menuFont->textWidth(param);
    }
}

void YMenu::getOffsets(int &left, int &top, int &right, int &bottom) {
    if (wmLook == lookMetal) {
        left = 1;
        right = 1;
        top = 2;
        bottom = 2;
    } else {
        left = 2;
        top = 2;
        right = 3;
        bottom = 3;
    }
}

void YMenu::getArea(int &x, int &y, int &w, int &h) {
    getOffsets(x, y, w, h);
    w = width() - 1 - x - w;
    h = height() - 1 - y - h;
}

int YMenu::findItemPos(int itemNo, int &x, int &y) {
    x = -1;
    y = -1;
    
    if (itemNo < 0 || itemNo > itemCount())
        return -1;

    int w, h;

    getArea(x, y, w, h);
    for (int i = 0; i < itemNo; i++) {
        int ih, top, bottom, pad;
        
        getItemHeight(i, ih, top, bottom, pad);
        y += ih;
        /*if (item(i)->command() == cmdSeparator)
            y += 4;
        else {
            unsigned int h = fontHeight;

            if (item(i)->getPixmap() && item(i)->getPixmap()->height() > h)
                h = item(i)->getPixmap()->height();

            if (wmLook == lookGtk)
                h++;
            y += h + 3;
        }*/
    }
    return 0;
}

int YMenu::findItem(int mx, int my) {
    int x, y, w, h;

    getArea(x, y, w, h);
    for (int i = 0; i < itemCount(); i++) {
        int top, bottom, pad;

        getItemHeight(i, h, top, bottom, pad);
        if (my >= y && my < y + h && mx > 0 && mx < int(width()) - 1)
            return i;

        y += h;
    }
    return -1;
}

void YMenu::sizePopup() {
    int width, height;
    int maxName = 0;
    int maxParam = 0;
    int maxIcon = 16;
    int l, t, r, b;
    int padx = 1;
    int left = 1;

    getOffsets(l, t, r, b);

    width = l;
    height = t;

    for (int i = 0; i < itemCount(); i++) {
        int ih, top, bottom, pad;
        
        getItemHeight(i, ih, top, bottom, pad);
        height += ih;

        if (pad > padx)
            padx = pad;
        if (top > left)
            left = top;

        int iw, nw, pw;
        
        getItemWidth(i, iw, nw, pw);

        if (item(i)->submenu())
            pw += 2 + ih;

        if (iw > maxIcon)
            maxIcon = iw;
        if (nw > maxName)
            maxName = nw;
        if (pw > maxParam)
            maxParam = pw;
    }

    //maxWidth = maxName + maxParam + (maxParam ? 8 : 0);
    //width += 3 + maxh + 2 + 2 + maxWidth + 4 + 4;

    namePos = l + left + padx + maxIcon + 2;
    paramPos = namePos + 2 + maxName + 6;
    width = paramPos + maxParam + 4 + r;
    height += b;
    
    setSize(width, height);
}

void YMenu::paintItems() {
    Graphics &g = getGraphics();
    int l, t, r, b;
    getOffsets(l, t, r, b);
    
    for (int i = 0; i < itemCount(); i++)
        paintItem(g, i, l, t, r, (i == selectedItem || i == paintedItem) ? 1 : 0);
    paintedItem = selectedItem;
}
void YMenu::drawSeparator(Graphics &g, int x, int y, int w) {
    if (wmLook == lookMetal) {
        g.setColor(menuBg);
        g.drawLine(x, y + 0, w, y + 0);
        g.setColor(activeMenuItemBg);
        g.drawLine(x, y + 1, w, y + 1);;
        g.setColor(white);
        g.drawLine(x, y + 2, w, y + 2);;
        g.drawLine(x, y, x, y + 2);
        g.setColor(menuBg);
    } else {
        //g.setColor(menuBg); // ASSUMED
        g.drawLine(x, y + 0, w, y + 0);
        g.setColor(menuBg->darker());
        g.drawLine(x, y + 1, w, y + 1);
        g.setColor(menuBg->brighter());
        g.drawLine(x, y + 2, w, y + 2);
        g.setColor(menuBg);
        g.drawLine(x, y + 3, w, y + 3); y++;
    }
}

void YMenu::paintItem(Graphics &g, int i, int &l, int &t, int &r, int paint) {
    int fontHeight = menuFont->height() + 1;
    int fontBaseLine = menuFont->ascent();
    YMenuItem *mitem = item(i);
    char *name = mitem->name();
    char *param = mitem->param();

    g.setColor(menuBg);
    if (mitem->command() == cmdSeparator) {
        if (paint)
            drawSeparator(g, 1, t, width() - 2);
        t += (wmLook == lookMetal) ? 3 : 4;
    } else {
        int eh, top, bottom, pad, ih;

        getItemHeight(i, eh, top, bottom, pad);
        ih = eh - top - bottom - pad - pad;
        
        if (i == selectedItem)
            g.setColor(activeMenuItemBg);
        
        if (paint) {
            g.fillRect(l, t, width() - r - l, eh);

            
            if (wmLook == lookMetal && i != selectedItem) {
                g.setColor(white);
                g.drawLine(1, t, 1, t + eh - 1);
                g.setColor(menuBg);
            }

            if (wmLook != lookWin95 && wmLook != lookWarp4 &&
                i == selectedItem)
            {
                bool raised = false;
#ifdef OLDMOTIF
                if (wmLook == lookMotif)
                    raised = true;
#endif

                g.setColor(menuBg);
                if (wmLook == lookGtk)
                    g.drawBorderW(l, t, width() - r - l - 1, eh - 1, true);
                else if (wmLook == lookMetal) {
                    g.setColor(activeMenuItemBg->darker());
                    g.drawLine(l, t, width() - r - l, t);
                    g.setColor(activeMenuItemBg->brighter());
                    g.drawLine(l, t + eh - 1, width() - r - l, t + eh - 1);
                } else
                    g.draw3DRect(l, t, width() - r - l - 1, eh - 1, raised);
                    

                if (wmLook == lookMotif)
                    g.draw3DRect(l + 1, t + 1,
                                 width() - r - l - 3, eh - 3, raised);
            }

            YColor *fg;
            if (!mitem->isEnabled())
                fg = disabledMenuItemFg;
            else if (i == selectedItem)
                fg = activeMenuItemFg;
            else
                fg = menuItemFg;
            g.setColor(fg);
            g.setFont(menuFont);
            
            int delta = (i == selectedItem) ? 1 : 0;
            if (wmLook == lookMotif || wmLook == lookGtk ||
                wmLook == lookWarp4 || wmLook == lookWin95 ||
                wmLook == lookMetal)
                delta = 0;
            int baseLine = t + top + pad + (ih - fontHeight) / 2 + fontBaseLine + delta;
                //1 + 1 + t + (eh - fontHeight) / 2 + fontBaseLine + delta;

            if (mitem->isChecked()) {
                XPoint points[4];

                points[0].x = l + 3 + (16 - 10) / 2;
                points[1].x = 5;
                points[2].x = 5;
                points[3].x = -5;
                points[0].y = t + top + pad + ih / 2;
                points[1].y = -5;
                points[2].y = 5;
                points[3].y = 5;

                g.fillPolygon(points, 4, Convex, CoordModePrevious);
            } else {
                if (mitem->getPixmap())
                    g.drawPixmap(mitem->getPixmap(),
                                 l + 1 + delta,
                                 t + delta + top + pad + (eh - top - pad * 2 - bottom - mitem->getPixmap()->height()) / 2);
            }

            if (name) {
                if (!mitem->isEnabled()) {
                    g.setColor(white);
                    g.drawChars((char *)name, 0, strlen((char *)name),
                                1 + delta + namePos, 1 + baseLine);
                    
                    if (mitem->hotCharPos() != -1) {
                        g.drawCharUnderline(1 + delta +  namePos, 1 + baseLine,
                                            (char *)name, mitem->hotCharPos());
                    }
                }
                g.setColor(fg);
                g.drawChars((char *)name, 0, strlen((char *)name),
                            delta + namePos, baseLine);

                if (mitem->hotCharPos() != -1) {
                    g.drawCharUnderline(delta + namePos, baseLine, 
                                        (char *)name, mitem->hotCharPos());
                }
            }
            
            if (param) {
                if (!mitem->isEnabled()) {
                    g.setColor(white);
                    g.drawChars((char *)param, 0, strlen((char *)param),
                                paramPos + delta + 1,
                                baseLine + 1);
                }
                g.setColor(fg);
                g.drawChars((char *)param, 0, strlen((char *)param),
                            paramPos + delta,
                            baseLine);
            }
            else if (mitem->submenu() || mitem->command() == cmdSubmenu) {
                int active = ((mitem->command() == cmdSubmenu && i == selectedItem) ||
                              fPopup == mitem->submenu()) ? 1 : 0;
                if (mitem->command() != cmdSubmenu) {
                    g.setColor(menuBg);
                    g.fillRect(width() - r - 1 - ih - pad, t + top + pad, ih, ih);
                    g.drawBorderW(width() - r - 1 - ih - pad, t + top + pad, ih - 1, ih - 1,
                                  active ? false : true);
                    delta = delta ? active ? 1 : 0 : 0;
                }

                if (wmLook == lookGtk || wmLook == lookMotif) {
                    int asize = 9;
                    int ax = delta + width() - r - 1 - asize * 3 / 2;
                    int ay = delta + t + top + pad + (ih - asize) / 2;
                    g.setColor(menuBg);
                    g.drawArrow(0, active ? 1 : -1, ax, ay, asize);
                } else {
                    int asize = 9;
                    int ax = width() - r - 1 - asize;
                    int ay = delta + t + top + pad + (ih - asize) / 2;

                    if (mitem->submenu() && mitem->command() != cmdSubmenu)
                        g.setColor(menuItemFg);
                    else 
                        g.setColor(fg);
                    g.drawArrow(0, 0, ax, ay, asize);
                }

            }
        }
        t += eh;
    }
}

void YMenu::paint(Graphics &g, int /*_x*/, int /*_y*/, unsigned int /*_width*/, unsigned int /*_height*/) {
    if (wmLook == lookMetal) {
        g.setColor(activeMenuItemBg);
        g.drawLine(0, 0, width() - 1, 0);
        g.drawLine(0, 0, 0, height() - 1);
        g.drawLine(width() - 1, 0, width() - 1, height() - 1);
        g.drawLine(0, height() - 1, width() - 1, height() - 1);
        g.setColor(white);
        g.drawLine(1, 1, width() - 2, 1);
        g.setColor(menuBg);
        g.drawLine(1, height() - 2, width() - 2, height() - 2);
    } else {
        g.setColor(menuBg);
        g.drawBorderW(0, 0, width() - 1, height() - 1, true);
        g.drawLine(1, 1, width() - 3, 1);
        g.drawLine(1, 1, 1, height() - 3);
        g.drawLine(1, height() - 3, width() - 3, height() - 3);
        g.drawLine(width() - 3, 1, width() - 3, height() - 3);
    }

    int l, t, r, b;
    getOffsets(l, t, r, b);
    
    for (int i = 0; i < itemCount(); i++)
        paintItem(g, i, l, t, r, 1);
    paintedItem = selectedItem;
}
