/*
 * IceWM
 *
 * Copyright (C) 1997,1998 Marko Macek
 *
 * Window list
 */

#include "icewm.h"

YColor *listBoxBg = 0;
YColor *listBoxFg = 0;
YColor *listBoxSelBg = 0;
YColor *listBoxSelFg = 0;

YListItem::YListItem() {
    fPrevItem = fNextItem = 0;
    fSelected = 0;
}

YListItem::~YListItem() {
}

YListItem *YListItem::getNext() {
    return fNextItem;
}

YListItem *YListItem::getPrev() {
    return fPrevItem;
}

void YListItem::setNext(YListItem *next) {
    fNextItem = next;
}

void YListItem::setPrev(YListItem *prev) {
    fPrevItem = prev;
}

int YListItem::getSelected() {
    return fSelected;
}

void YListItem::setSelected(int aSelected) {
    fSelected = aSelected;
}

unsigned char *YListItem::getText() {
    return 0;
}

YIcon *YListItem::getIcon() {
    return 0;
}

int YListItem::getOffset() {
    return 0;
}

YListBox::YListBox(YScrollBar *vert, YWindow *aParent, Window win): YWindow(aParent, win) {
    fVerticalScroll = vert;
    if (vert)
        vert->setListener(this);
    fOffsetY = 0;
    fFirst = fLast = 0;
    fFocusedItem = 0;
    fSelectStart = fSelectEnd = -1;
    fSelecting = -1;
}

YListBox::~YListBox() {
    fFirst = fLast = 0;
}

bool YListBox::isFocusTraversable() {
    return true;
}

int YListBox::addAfter(YListItem *prev, YListItem *item) {
    assert(item->getPrev() == 0);
    assert(item->getNext() == 0);

    item->setNext(prev->getNext());
    if (item->getNext())
        item->getNext()->setPrev(item);
    else
        fLast = item;
    
    item->setPrev(prev);
    prev->setNext(item);
    repaint();
    return 1;
}

int YListBox::addItem(YListItem *item) {
    assert(item->getPrev() == 0);
    assert(item->getNext() == 0);

    item->setNext(0);
    item->setPrev(fLast);
    if (fLast)
        fLast->setNext(item);
    else
        fFirst = item;
    fLast = item;
    repaint();
    return 1;
}

void YListBox::removeItem(YListItem *item) {
    if (item->getPrev())
        item->getPrev()->setNext(item->getNext());
    else
        fFirst = item->getNext();

    if (item->getNext())
        item->getNext()->setPrev(item->getPrev());
    else
        fLast = item->getPrev();
    item->setPrev(0);
    item->setNext(0);
    repaint();
}

YListItem *YListBox::findItem(int mouseY, int &no) {
    int y = 0;
    int lh, fh = windowListFont->height();
    YListItem *a = fFirst;
    
    no = 0;
    if (ICON_SMALL > fh)
        lh = ICON_SMALL;
    else
        lh = fh;
    lh += 2;
    while (a) {
        if (mouseY >= y - fOffsetY && mouseY < y - fOffsetY + lh)
            return a;
        y += lh;
        a = a->getNext();
        no++;
    }
    no = -1;
    return 0;
}

void YListBox::selectItem(YListItem *i) {
    if (i == 0) {
        fFocusedItem = -1;
        fSelectStart = 0;
        fSelectEnd = 0;
    } else {
        int n = 0;
        YListItem *f = fFirst;
        for(; f ; f = f->getNext(), n++)
            if (i == f) {
                fFocusedItem = n;
                fSelectStart = fSelectEnd = n;
            }
        ensureVisibility(fFocusedItem);
    }
    fSelecting = -1;
    setSelection();
    repaint();
}

void YListBox::updateSelection(int apply) {
    if (fSelectStart != -1) {
        assert(fSelectEnd != -1);
        
        int beg = (fSelectStart < fSelectEnd) ? fSelectStart : fSelectEnd;
        int end = (fSelectStart < fSelectEnd) ? fSelectEnd : fSelectStart;

        YListItem *i;
        int n;

        for (i = fFirst, n = 0; i; i = i->getNext(), n++) {
            int s = i->getSelected();
            int t = (n >= beg) && (n <= end);

            if (fSelecting) {
                s &= 1;
                if (t)
                    s |= apply ? 1 : 2;
            } else {
                s &= 1;
                if (t) {
                    if (s & 1)
                        if (apply)
                            s = 0;
                        else
                            s |= 2;
                }
            }
            if (apply)
                s &= 1;
            i->setSelected(s);
        }
    }
}

void YListBox::setSelection() {
    if (fSelectStart != -1) {
        assert(fSelectEnd != -1);
        
        int beg = (fSelectStart < fSelectEnd) ? fSelectStart : fSelectEnd;
        int end = (fSelectStart < fSelectEnd) ? fSelectEnd : fSelectStart;

        YListItem *i;
        int n;

        for (i = fFirst, n = 0; i; i = i->getNext(), n++) {
            int t = ((n >= beg) && (n <= end)) ? 1 : 0;
            i->setSelected(t);
        }
    }
}

int YListBox::itemCount() {
    YListItem *i;
    int n;
    
    for (i = fFirst, n = 0; i; i = i->getNext(), n++)
        ;
    return n;
}

YListItem *YListBox::item(int no) {
    YListItem *a = fFirst;
    int n;
    
    for (n = 0; a; a = a->getNext(), n++)
        if (n == no)
            return a;
    return 0;
}

int YListBox::getLineHeight() {
    int lh, fh = windowListFont->height();
    if (ICON_SMALL > fh)
        lh = ICON_SMALL;
    else
        lh = fh;
    lh += 2;
    return lh;
}

void YListBox::ensureVisibility(int item) {
    if (item >= 0) {
        int lh = getLineHeight();
        int fy = item * lh;
        
        if (fy + lh >= fOffsetY + (int)height())
            fOffsetY = fy + lh - height();
        if (fy <= fOffsetY)
            fOffsetY = fy;
    }
}

void YListBox::focusVisible() {
    int ty = fOffsetY;
    int by = ty + height();

    if (fFocusedItem >= 0) {
        int lh = getLineHeight();

        while (ty > fFocusedItem * lh)
            fFocusedItem++;
        while (by - lh < fFocusedItem * lh)
            fFocusedItem--;
    }
}

bool YListBox::handleKey(const XKeyEvent &key) {
    if (key.type == KeyPress) {
        KeySym k = XKeycodeToKeysym(app->display(), key.keycode, 0);
        int m = KEY_MODMASK(key.state);

        int SelPos, OldPos = fFocusedItem, count = itemCount();

        if (m & ShiftMask) {
            SelPos = fFocusedItem;
        } else {
            SelPos = -1;
        }
        
        switch (k) {
        case XK_Return:
            {
                YListItem *i = item(fFocusedItem);
                if (i)
                    activateItem(i);
            }
            break;
        case ' ':
            {
                YListItem *i = item(fFocusedItem);
                if (i) {
                    i->setSelected(i->getSelected() ? 0 : 1);
                    repaint();
                }
            }
            break;
        case XK_Up:
            if (fFocusedItem > 0)
                fFocusedItem--;
            break;
        case XK_Down:
            if (fFocusedItem < count - 1)
                fFocusedItem++;
            break;
#if 0
        case XK_Prior:
            fVerticalScroll->setValue(fVerticalScroll->getValue() -
                                      fVerticalScroll->getBlockIncrement());
            fOffsetY = fVerticalScroll->getValue();
            fFocusedItem -= height() / getLineHeight();
            if (fFocusedItem < 0)
                if (count > 0)
                    fFocusedItem = 0;
                else
                    fFocusedItem = -1;
            repaint();
            break;
        case XK_Next:
            fVerticalScroll->setValue(fVerticalScroll->getValue() +
                                      fVerticalScroll->getBlockIncrement());
            fOffsetY = fVerticalScroll->getValue();
            fFocusedItem += height() / getLineHeight();
            if (fFocusedItem > count - 1)
                fFocusedItem = count - 1;
            repaint();
            break;
#endif
        case XK_Home:
            fFocusedItem = 0;
            break;
        case XK_End:
            fFocusedItem = count - 1;
            break;
        default:
            if (k < 256) {
                unsigned char c = TOUPPER(k);
                int count = itemCount();
                int i = fFocusedItem;
                YListItem *it = 0;
                unsigned char *title;

                for (int n = 0; n < count; n++) {
                    i = (i + 1) % count;
                    it = item(i);
                    title = it->getText();
                    if (title && TOUPPER(title[0]) == c) {
                        fFocusedItem = i;
                        break;
                    }
                }
            } else {
                if (fVerticalScroll->handleScrollKeys(key) == false
                    //&& fHorizontalScroll->handleScrollKeys(key) == false
                   )
                    return YWindow::handleKey(key);
            }
        }
        if (fFocusedItem != OldPos) {
            if (SelPos == -1) {
                fSelectStart = fFocusedItem;
                fSelectEnd = fFocusedItem;
                fSelecting = 1;
            } else {
                if (fSelectStart == OldPos)
                    fSelectStart = fFocusedItem;
                else if (fSelectEnd == OldPos)
                    fSelectEnd = fFocusedItem;
                else
                    fSelectStart = fSelectEnd = fFocusedItem;
                if (fSelectStart == -1)
                    fSelectStart = fSelectEnd;
                if (fSelectEnd == -1)
                    fSelectEnd = fSelectStart;

                fSelecting = 1;
            }
            setSelection();
            ensureVisibility(fFocusedItem);
            repaint();
        }
        return true;
    }
    return YWindow::handleKey(key);
}

void YListBox::handleButton(const XButtonEvent &button) {
    if (button.button == 1) {
        int no;
        YListItem *i = findItem(button.y, no);
        fFocusedItem = no;
        
        if (button.type == ButtonPress) {
            if (!(button.state & ControlMask))
                for (YListItem *j = fFirst; j; j = j->getNext())
                    j->setSelected(0);
            if (i) {
                fSelectStart = no;
                fSelectEnd = no;
                fFocusedItem = no;
                fSelecting = i->getSelected() ? 0 : 1;
                updateSelection(0);
                ensureVisibility(fFocusedItem);
                repaint();
            } else {
                fSelectStart = -1;
                fSelectEnd = -1;
                fSelecting = 1;
                repaint();
            }
        } else if (button.type == ButtonRelease) {
            if (no != -1 && no != fSelectEnd) {
                fSelectEnd = no;
                updateSelection(0);
            }
            updateSelection(1);
            fSelecting = -1;
            ensureVisibility(fFocusedItem);
            repaint();
        }
    }
    if (fVerticalScroll->handleScrollMouse(button) == false)
        YWindow::handleButton(button);
}

void YListBox::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &up, int count) {
    if (up.button == 1 && count == 2) {
        int no;
        YListItem *i = findItem(up.y, no);
        if (i)
            activateItem(i);
    }
}

void YListBox::handleMotion(const XMotionEvent &motion) {
    if (motion.state & Button1Mask) {
        int no;

        findItem(motion.y, no);

        fFocusedItem = no;
        if (no != -1 && fSelectEnd != no) {
            if (fSelectStart == -1)
                fSelectStart = no;
            fSelectEnd = no;
            updateSelection(0);
            ensureVisibility(fFocusedItem);
            repaint();
        }
    }
    YWindow::handleMotion(motion);
}

void YListBox::handleDrag(const XButtonEvent &down, const XMotionEvent &motion) {
    if (down.button == 2) {
        int dx = motion.y - down.y;

        fVerticalScroll->setValue(fVerticalScroll->getValue() - dx);
        fOffsetY = fVerticalScroll->getValue();
        repaint();
    }
}

void YListBox::scroll(YScrollBar *scroll, int delta) {
    if (scroll == fVerticalScroll) {
        fOffsetY += delta;
        focusVisible();
        repaint();
    }
}

void YListBox::move(YScrollBar *scroll, int pos) {
    if (scroll == fVerticalScroll) {
        fOffsetY = pos;
        focusVisible();
        repaint();
    }
}

void YListBox::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    int y = 0, x = 3;
    int lh = getLineHeight();
    int fh = windowListFont->height();
    YListItem *a = fFirst;
    int n = 0;

    g.setColor(listBoxBg);
    g.fillRect(0, 0, width(), height());

    while (a) {
        unsigned char *title = a->getText();
        YIcon *icon = a->getIcon();
        
        int xpos = 0;
        int yPos = y + lh - (lh - fh) / 2 - windowListFont->descent();
        
        xpos += a->getOffset();
        
        int s = a->getSelected();
        if (fSelecting == -1)
            s = (s == 1) ? 1 : 0;
        if (fSelecting == 0 && (s & 2))
            s = 0;
        
        if (s)
            g.setColor(listBoxSelBg);
        else
            g.setColor(listBoxBg);
        g.fillRect(0, y - fOffsetY, width(), lh);
        if (fFocusedItem == n) {
            g.setColor(black);
            g.drawRect(0, y - fOffsetY, width() - 1, lh - 1);
        }
        if (icon && icon->small())
            g.drawPixmap(icon->small(), xpos + x, y - fOffsetY + 1);
        if (s)
            g.setColor(listBoxSelFg);
        else
            g.setColor(listBoxFg);
        g.setFont(windowListFont);
        if (title) {
            g.drawChars((char *)title, 0, strlen((char *)title),
                        xpos + x + 20, yPos - fOffsetY);
        }
        y += lh;
        a = a->getNext();
        n++;
    }
    fVerticalScroll->setValues(fOffsetY, height(), 0, y);
    fVerticalScroll->setBlockIncrement(height());
    fVerticalScroll->setUnitIncrement(fh);
}

void YListBox::activateItem(YListItem */*item*/) {
}

bool YListBox::hasSelection() {
    if (fSelectStart != -1 && fSelectEnd != -1)
        return true;
    return false;
}
