/*
 * IceWM
 *
 * Copyright (C) 1997,1998 Marko Macek
 */

#include "icewm.h"

void YFrameControl::setFrame(YFrameWindow *newFrame) {
    fFrame = newFrame;
}

YFrameClient::YFrameClient(YWindow *parent, YFrameWindow *frame, Window win): YFrameControl(parent, frame, win) {
    fClient = win ? win : handle();
    fBorder = 0;
    fProtocols = 0;
    fWindowTitle = 0;
    fIconTitle = 0;
    fColormap = None;
    fShaped = 0;
    fHints = 0;
    //fSavedFrameState =
    fSizeHints = XAllocSizeHints();
    fClassHint = XAllocClassHint();
    fTransientFor = 0;
#ifndef NO_MWM_HINTS
    fMwmHints = 0;
#endif

    getProtocols();
    getNameHint();
    getIconNameHint();
    getSizeHints();
    getClassHint();
    getTransient();
    getWMHints();
#ifndef NO_MWM_HINTS
    getMwmHints();
#endif
    
#ifdef SHAPE
    if (shapesSupported) {
        XShapeSelectInput (app->display(), handle(), ShapeNotifyMask);
        queryShape();
    }
#endif
    XSaveContext(app->display(),
                 clientWindow(),
                 frame ? frameContext : clientContext,
                 (XPointer)(frame ? (YWindow *)frame : (YWindow *)this));
}

YFrameClient::~YFrameClient() {
    XDeleteContext(app->display(),
                   clientWindow(),
                   frame() ? frameContext : clientContext);
    FREE(fWindowTitle); fWindowTitle = 0;
    FREE(fIconTitle); fIconTitle = 0;
    if (fSizeHints) { XFree(fSizeHints); fSizeHints = 0; }
    if (fClassHint) {
        if (fClassHint->res_name) {
            XFree(fClassHint->res_name);
            fClassHint->res_name = 0;
        }
        if (fClassHint->res_class) {
            XFree(fClassHint->res_class);
            fClassHint->res_class = 0;
        }
        XFree(fClassHint);
        fClassHint = 0;
    }
    if (fHints) { XFree(fHints); fHints = 0; }
    if (fMwmHints) { XFree(fMwmHints); fMwmHints = 0; }
}

void YFrameClient::getProtocols() {
    Atom *wmp = 0;
    int count;

#ifdef CONFIG_WINLIST
    if (this == windowList) {
        fProtocols = wpDeleteWindow;
        return ;
    }
#endif

    fProtocols = fProtocols & wpDeleteWindow; // always keep WM_DELETE_WINDOW

    if (XGetWMProtocols (app->display(),
                         handle(),
                         &wmp, &count) && wmp)
    {
        for (int i = 0; i < count; i++) {
            if (wmp[i] == _XA_WM_DELETE_WINDOW) fProtocols |= wpDeleteWindow;
            if (wmp[i] == _XA_WM_TAKE_FOCUS) fProtocols |= wpTakeFocus;
	}
        XFree (wmp);
    }
}

void YFrameClient::getSizeHints() {
    if (fSizeHints) {
        long supplied;
        
        if (!XGetWMNormalHints(app->display(),
                               handle(),
                               fSizeHints, &supplied))
            fSizeHints->flags = 0;

        if (fSizeHints->flags & PResizeInc) {
            if (fSizeHints->width_inc == 0) fSizeHints->width_inc = 1;
            if (fSizeHints->height_inc == 0) fSizeHints->height_inc = 1;
        } else
            fSizeHints->width_inc = fSizeHints->height_inc = 1;


        if (!(fSizeHints->flags & PBaseSize)) {
            if (fSizeHints->flags & PMinSize) {
                fSizeHints->base_width = fSizeHints->min_width;
                fSizeHints->base_height = fSizeHints->min_height;
            } else
                fSizeHints->base_width = fSizeHints->base_height = 0;
        }
        if (!(fSizeHints->flags & PMinSize)) {
            fSizeHints->min_width = fSizeHints->base_width;
            fSizeHints->min_height = fSizeHints->base_height;
        }
        if (!(fSizeHints->flags & PMaxSize)) {
            fSizeHints->max_width = manager->maxWidth(WinLayerAboveDock);
            fSizeHints->max_height = manager->maxHeight(WinLayerAboveDock);
        }
        if (fSizeHints->max_width < fSizeHints->min_width)
            fSizeHints->max_width = manager->maxWidth(WinLayerAboveDock);
        if (fSizeHints->max_height < fSizeHints->min_height)
            fSizeHints->max_height = manager->maxHeight(WinLayerAboveDock);
        
        if (fSizeHints->min_height <= 0)
            fSizeHints->min_height = 1;
        if (fSizeHints->min_width <= 0)
            fSizeHints->min_width = 1;

        if (!(fSizeHints->flags & PWinGravity)) {
            fSizeHints->win_gravity = NorthWestGravity;
            fSizeHints->flags |= PWinGravity;
        }
    }
}

void YFrameClient::getClassHint() {
    if (fClassHint) {
        if (fClassHint->res_name) {
            XFree(fClassHint->res_name);
            fClassHint->res_name = 0;
        }
        if (fClassHint->res_class) {
            XFree(fClassHint->res_class);
            fClassHint->res_class = 0;
        }
        XGetClassHint(app->display(), handle(), fClassHint);
    }
}

void YFrameClient::getTransient() {
    Window newTransientFor;
            
    if (XGetTransientForHint(app->display(),
                             handle(),
                             &newTransientFor))
    {
        if (newTransientFor == manager->handle() || /* bug in xfm */
            newTransientFor == desktop->handle() ||
            newTransientFor == handle()                 /* bug in fdesign */
            /* !!! TODO: check for recursion */
           )
            newTransientFor = 0;

        if (newTransientFor != fTransientFor) {
            if (fTransientFor)
                if (frame())
                    frame()->removeAsTransient();
            fTransientFor = newTransientFor;
            if (fTransientFor)
                if (frame())
                    frame()->addAsTransient();
        }
    }
}

void YFrameClient::constrainSize(int &w, int &h, CARD32 layer, int flags) {
    if (fSizeHints) {
        int wm = fSizeHints->min_width;
        int hm = fSizeHints->min_height;
        int wM = fSizeHints->max_width;
        int hM = fSizeHints->max_height;
        int wb = fSizeHints->base_width;
        int hb = fSizeHints->base_height;
        int wf = fSizeHints->width_inc;
        int hf = fSizeHints->height_inc;

        if (fSizeHints->flags & PAspect) { // aspect ratios
            int xm = fSizeHints->min_aspect.x;
            int ym = fSizeHints->min_aspect.y;
            int xM = fSizeHints->max_aspect.x;
            int yM = fSizeHints->max_aspect.y;

            // !!! fix handling of KeepX and KeepY together
            if (xm * h > ym * w) { // min aspect
                if (flags & csKeepX) {
                    if (h < hm) h = hm;
                    if (h > hM) h = hM;
                    h = w * ym / xm;
                    if (h < hm) h = hm;
                    if (h > hM) h = hM;
                    w = h * xm / ym;
                } else {
                    if (w > wM) w = wM; // maximum size
                    if (w < wm) w = wm; // minimum size
                    w = h * xm / ym;
                    if (w < wm) w = wm; // minimum size
                    if (w > wM) w = wM;
                    h = w * ym / xm;
                }
            }
            if (xM * h < yM * w) { // max aspect
                if (flags & csKeepX) {
                    if (h < hm) h = hm;
                    if (h > hM) h = hM;
                    h = w * yM / xM;
                    if (h > hM) h = hM;
                    if (h < hm) h = hm;
                    w = h * xM / yM;
                } else {
                    if (w > wM) w = wM; // maximum size
                    if (w < wm) w = wm; // minimum size
                    w = h * xM / yM;
                    if (w < wm) w = wm; // minimum size
                    if (w > wM) w = wM;
                    h = w * yM / xM;
                }
            }
        }

        if (w < wm) w = wm; // minimum size
        if (h < hm) h = hm;

        if (w > wM) w = wM; // maximum size
        if (h > hM) h = hM;

        if (w >= int(manager->maxWidth(layer))) w = manager->maxWidth(layer);
        if (h >= int(manager->maxHeight(layer))) h = manager->maxHeight(layer);

        w = wb + (w - wb + ((flags & csRound) ? wf / 2 : 0)) / wf * wf;
        h = hb + (h - hb + ((flags & csRound) ? hf / 2 : 0)) / hf * hf;
    }
    if (w <= 0) w = 1;
    if (h <= 0) h = 1;
}

struct _gravity_offset 
{
  int x, y;
};

void YFrameClient::gravityOffsets (int &xp, int &yp) {
    xp = 0;
    yp = 0;

    if (fSizeHints == 0)
        return ;

    static struct {
        int x, y;
    } gravOfsXY[11] = {
        {  0,  0 },  /* ForgetGravity */
        { -1, -1 },  /* NorthWestGravity */
        {  0, -1 },  /* NorthGravity */
        {  1, -1 },  /* NorthEastGravity */
        { -1,  0 },  /* WestGravity */
        {  0,  0 },  /* CenterGravity */
        {  1,  0 },  /* EastGravity */
        { -1,  1 },  /* SouthWestGravity */
        {  0,  1 },  /* SouthGravity */
        {  1,  1 },  /* SouthEastGravity */
        {  0,  0 },  /* StaticGravity */
    };

    int g = fSizeHints->win_gravity;
  
    if (!(g < ForgetGravity || g > StaticGravity)) {
        xp = (int)gravOfsXY[g].x;
        yp = (int)gravOfsXY[g].y;
    }
}

void YFrameClient::sendMessage(Atom msg, Time timeStamp) {
    XClientMessageEvent xev;
  
    xev.type = ClientMessage;
    xev.window = handle();
    xev.message_type = _XA_WM_PROTOCOLS;
    xev.format = 32;
    xev.data.l[0] = msg;
    xev.data.l[1] = timeStamp;
    XSendEvent (app->display(), handle(), False, 0L, (XEvent *) &xev);
}

void YFrameClient::setFrame(YFrameWindow *newFrame) {
    XDeleteContext(app->display(),
                   clientWindow(),
                   frame() ? frameContext : clientContext);
    YFrameControl::setFrame(newFrame);
    if (newFrame == 0) {
        //msg("remove frame");
    } else {
        //msg("got frame");
    }
    XSaveContext(app->display(),
                 clientWindow(),
                 frame() ? frameContext : clientContext,
                 (XPointer)(frame() ? (YWindow *)frame() : (YWindow *)this));
}

void YFrameClient::setFrameState(FrameState state) {
    unsigned long arg[2];

    arg[0] = (unsigned long) state;
    arg[1] = (unsigned long) None;

    if (state == WithdrawnState) {
        if (phase != phaseRestart && phase != phaseShutdown) {
            MSG(("deleting window properties id=%lX", handle()));
            XDeleteProperty(app->display(), clientWindow(), _XA_WIN_WORKSPACE);
            XDeleteProperty(app->display(), clientWindow(), _XA_WIN_LAYER);
            XDeleteProperty(app->display(), clientWindow(), _XA_WIN_STATE);
            XDeleteProperty(app->display(), clientWindow(), _XA_WM_STATE);
        }
    } else {
        XChangeProperty(app->display(), handle(),
                        _XA_WM_STATE, _XA_WM_STATE,
                        32, PropModeReplace,
                        (unsigned char *)arg, 2);
    }
}

FrameState YFrameClient::getFrameState() {
    FrameState st = WithdrawnState;
    Atom type;
    int format;
    unsigned long nitems, lbytes;
    unsigned char *propdata;

    if (XGetWindowProperty(app->display(), handle(),
                           _XA_WM_STATE, 0, 3, False, _XA_WM_STATE,
                           &type, &format, &nitems, &lbytes,
                           &propdata) == Success && propdata)
    {
        st = *(long *)propdata;
        XFree(propdata);
    }
    return st;
}

void YFrameClient::handleUnmap(const XUnmapEvent &unmap) {
    MSG(("Unmap: unmapped %d visible %d", unmapped(), visible()));
    if (!unmapped()) {
        MSG(("UnmapWindow"));
#ifdef NEED_GRAB
        XGrabServer(app->display());
        XSync(app->display(), 0);
#endif
        XEvent ev;
        if (XCheckTypedWindowEvent(app->display(), unmap.window, DestroyNotify, &ev)) {
            manager->destroyedWindow(unmap.window);
            XUngrabServer(app->display());
            return ; // gets destroyed
        } else {
            manager->unmanageWindow(unmap.window, false);
        }
#ifdef NEED_GRAB
        XUngrabServer(app->display());
#endif
    }
    YWindow::handleUnmap(unmap);
}

void YFrameClient::handleProperty(const XPropertyEvent &property) {
    switch (property.atom) {
    case XA_WM_NAME:
        getNameHint();
        break;
    case XA_WM_ICON_NAME:
        getIconNameHint();
        break;
    case XA_WM_CLASS:
        getClassHint();
        if (frame())
            frame()->getFrameHints();
        break;
    case XA_WM_HINTS:
        getWMHints();
        break;
    case XA_WM_NORMAL_HINTS:
        getSizeHints();
        break;
    case XA_WM_TRANSIENT_FOR:
        getTransient();
        break;
    default:
        if (property.atom == _XA_WM_PROTOCOLS)
            getProtocols();
        else if (property.atom == _XA_KWM_WIN_ICON) {
            if (frame())
                frame()->updateIcon();
        } else if (property.atom == _XA_WIN_ICONS) {
            if (frame())
                frame()->updateIcon();
        }
#ifndef NO_MWM_HINTS
        else if (property.atom == _XATOM_MWM_HINTS) {
            getMwmHints();
            if (frame())
                frame()->updateMwmHints();
        }
#endif
        break;
    }
}

void YFrameClient::handleColormap(const XColormapEvent &colormap) {
    setColormap(colormap.colormap); //(colormap.state == ColormapInstalled && colormap.c_new == True) 
//                ? colormap.colormap 
//                : None);
}


void YFrameClient::handleDestroyWindow(const XDestroyWindowEvent &destroyWindow) {
    //msg("DESTROY: %lX", destroyWindow.window);
    YWindow::handleDestroyWindow(destroyWindow);

    if (destroyed())
        manager->destroyedWindow(destroyWindow.window);
}

#ifdef SHAPE
void YFrameClient::handleShapeNotify(const XShapeEvent &shape) {
    if (shapesSupported) {
        if (shape.kind == ShapeBounding) {
            fShaped = shape.shaped ? 1 : 0;
            if (frame())
                frame()->setShape();
        }
    }
}
#endif

void YFrameClient::setWindowTitle(unsigned char *aWindowTitle) {
    FREE(fWindowTitle);
    fWindowTitle = aWindowTitle ? (unsigned char *)strdup((char *)aWindowTitle) : 0;
    if (frame())
        frame()->updateTitle();
}

#ifdef I18N
void YFrameClient::setWindowTitle(XTextProperty  *prop) {
    Status status;
    char **cl;
    int n;
    if (!prop->value || prop->encoding == XA_STRING) {
	setWindowTitle((unsigned char *)prop->value);
	return;
    }
    status = XmbTextPropertyToTextList (app->display(), prop, &cl, &n);
    if (status >= Success && n > 0 && cl[0]) {
	setWindowTitle((unsigned char *)cl[0]);
	XFreeStringList(cl);
    } else {
	setWindowTitle((unsigned char *)prop->value);
    }
}
#endif

void YFrameClient::setIconTitle(unsigned char *aIconTitle) {
    FREE(fIconTitle);
    fIconTitle = aIconTitle ? (unsigned char *)strdup((char *)aIconTitle) : 0;
    if (frame())
        frame()->updateIconTitle();
}

#ifdef I18N
void YFrameClient::setIconTitle(XTextProperty  *prop) {
    Status status;
    char **cl;
    int n;
    if (!prop->value || prop->encoding == XA_STRING) {
	setIconTitle((unsigned char *)prop->value);
	return;
    }
    status = XmbTextPropertyToTextList (app->display(), prop, &cl, &n);
    if (status >= Success && n > 0 && cl[0]) {
	setIconTitle((unsigned char *)cl[0]);
	XFreeStringList(cl);
    } else {
	setIconTitle((unsigned char *)prop->value);
    }
}
#endif

void YFrameClient::setColormap(Colormap cmap) {
    fColormap = cmap;
    if (frame() && manager->colormapWindow() == frame())
        manager->installColormap(cmap);
}

#ifdef SHAPE
void YFrameClient::queryShape() {
    fShaped = 0;
    
    if (shapesSupported) {
        int xws, yws, xbs, ybs;
        unsigned wws, hws, wbs, hbs;
        Bool boundingShaped, clipShaped;

        XShapeQueryExtents(app->display(), handle(),
                           &boundingShaped, &xws, &yws, &wws, &hws,
                           &clipShaped, &xbs, &ybs, &wbs, &hbs);
        fShaped = boundingShaped ? 1 : 0;
  }
}
#endif

YClientContainer::YClientContainer(YWindow *parent, YFrameWindow *frame)
:YFrameControl(parent, frame)
{
    setStyle(wsManager);
    haveButtonGrab = 0;
}

YClientContainer::~YClientContainer() {
    releaseButtons();
}


void YClientContainer::handleButton(const XButtonEvent &button) {
    if (haveButtonGrab) {
        if (haveButtonGrab)
            MSG(("buttonGrab"));

        if (!(button.state & ControlMask) &&
            (!useMouseWheel || (button.button != 4 && button.button != 5)))
        {
            if (focusOnClickClient)
                frame()->activate();
            if (raiseOnClickClient)
                frame()->wmRaise();
        }
#if 0
        if (button.button == 3 &&
            ((button.state &
              (ControlMask | ShiftMask | app->AltMask) ==
              (ControlMask | ShiftMask))))
        {
            frame()->wmMove();
            XAllowEvents(app->display(), AsyncPointer, CurrentTime);
            return ;
        }
#endif
        if (passFirstClickToClient)
            XAllowEvents(app->display(), ReplayPointer, CurrentTime);
        else
            XAllowEvents(app->display(), AsyncPointer, CurrentTime);
    }
    return ;
    // YFrameControl::handleButton(button);
}

// manage button grab on frame window to capture clicks to client window
// we want to keep the grab when:
//    focusOnClickClient && not focused
// || raiseOnClickClient && not on top

// !!! fix ('not on top'  should be  'can be raised').
//     the difference is when we have transients and explicitFocus
//     also there is the difference with layers

void YClientContainer::grabButtons() {
    if (!haveButtonGrab &&
        (clickFocus ||
         focusOnClickClient || raiseOnClickClient))
    {
        haveButtonGrab = 1;

        XGrabButton(app->display(),
                    AnyButton, AnyModifier,
                    handle(), True,
                    ButtonPressMask,
                    GrabModeSync, GrabModeAsync, None, None);
    }
}

void YClientContainer::releaseButtons() {
    if (haveButtonGrab) {
        haveButtonGrab = 0;

        XUngrabButton(app->display(), AnyButton, AnyModifier,
                      handle());
    }
}

void YClientContainer::handleConfigureRequest(const XConfigureRequestEvent &configureRequest) {
    MSG(("configure request in frame"));

    if (frame() && 
        configureRequest.window == frame()->client()->clientWindow()) 
    {
        XConfigureRequestEvent cre = configureRequest;
        
        frame()->configureClient(cre);
    }
}

void YClientContainer::handleMapRequest(const XMapRequestEvent &mapRequest) {
    if (mapRequest.window == frame()->client()->clientWindow()) {
        frame()->setState(WinStateMinimized |
                          WinStateHidden |
                          WinStateRollup,
                          0);
        frame()->focusOnMap();
    }
}

void YClientContainer::handleCrossing(const XCrossingEvent &crossing) {
    if (frame() && pointerColormap) {
        if (crossing.type == EnterNotify)
            manager->setColormapWindow(frame());
        else if (crossing.type == LeaveNotify &&
                 crossing.detail != NotifyInferior &&
                 crossing.mode == NotifyNormal &&
                 manager->colormapWindow() == frame())
        {
            manager->setColormapWindow(0);
        }
    }
}

void YFrameClient::handleClientMessage(const XClientMessageEvent &message) {
    if (message.message_type == _XA_WM_CHANGE_STATE) {
        YFrameWindow *frame = manager->getFrame(message.window);

        if (message.data.l[0] == IconicState) {
            if (frame && !(frame->isMinimized() || frame->isRollup()))
                frame->wmMinimize();
        } else if (message.data.l[0] == NormalState) {
            if (frame)
                frame->setState(WinStateHidden |
                                WinStateRollup |
                                WinStateMinimized, 0);
        } // !!! handle WithdrawnState if needed

    } else if (message.message_type == _XA_WIN_WORKSPACE) {
        if (frame())
            frame()->setWorkspace(message.data.l[0]);
        else
            setWinWorkspaceHint(message.data.l[0]);
    } else if (message.message_type == _XA_WIN_LAYER) {
        if (frame())
            frame()->setLayer(message.data.l[0]);
        else
            setWinLayerHint(message.data.l[0]);
    } else if (message.message_type == _XA_WIN_STATE) {
        if (frame())
            frame()->setState(message.data.l[0], message.data.l[1]);
        else
            setWinStateHint(message.data.l[0], message.data.l[1]);
    } else
        YWindow::handleClientMessage(message);
}

void YFrameClient::getNameHint() {
    XTextProperty prop;
    
    if (XGetWMName(app->display(), handle(), &prop)) {
#ifndef I18N
        setWindowTitle(prop.value);
#else
        setWindowTitle(&prop);
#endif
        if (prop.value) XFree(prop.value);
    } else {
        setWindowTitle((unsigned char *)0);
    }
}

void YFrameClient::getIconNameHint() {
    XTextProperty prop;
    
    if (XGetWMIconName(app->display(), handle(), &prop)) {
#ifndef I18N
        setIconTitle(prop.value);
#else
        setIconTitle(&prop);
#endif
        if (prop.value) XFree(prop.value);
    } else {
        setIconTitle((unsigned char *)0);
    }
}

void YFrameClient::getWMHints() {
    if (fHints)
        XFree(fHints);
    fHints = XGetWMHints(app->display(), handle());
}

#ifndef NO_MWM_HINTS
void YFrameClient::getMwmHints() {
    int retFormat;
    Atom retType;
    unsigned long retCount, remain;

    if (fMwmHints) {
        XFree(fMwmHints);
        fMwmHints = 0;
    }
    if (XGetWindowProperty(app->display(), handle(),
                           _XATOM_MWM_HINTS, 0L, 20L, False, _XATOM_MWM_HINTS,
                           &retType, &retFormat, &retCount,
                           &remain,(unsigned char **)&fMwmHints) == Success && fMwmHints)
        if (retCount >= PROP_MWM_HINTS_ELEMENTS)
            return;
        else
            XFree(fMwmHints);
    fMwmHints = 0;
}

CARD32 YFrameClient::mwmFunctions() {
    CARD32 functions = ~0U;
    
    if (fMwmHints && (fMwmHints->flags & MWM_HINTS_FUNCTIONS)) {
        if (fMwmHints->functions & MWM_FUNC_ALL)
            functions = ~fMwmHints->functions;
        else
            functions = fMwmHints->functions;
    }
    functions &= (MWM_FUNC_RESIZE | MWM_FUNC_MOVE |
                  MWM_FUNC_MINIMIZE | MWM_FUNC_MAXIMIZE |
                  MWM_FUNC_CLOSE);
    return functions;
}

CARD32 YFrameClient::mwmDecors() {
    CARD32 decors = ~0U;
    CARD32 func = mwmFunctions();
    
    if (fMwmHints && (fMwmHints->flags & MWM_HINTS_DECORATIONS)) {
        if (fMwmHints->decorations & MWM_FUNC_ALL)
            decors = ~fMwmHints->decorations;
        else
            decors = fMwmHints->decorations;
    }
    decors &= (MWM_DECOR_BORDER | MWM_DECOR_RESIZEH |
               MWM_DECOR_TITLE | MWM_DECOR_MENU |
               MWM_DECOR_MINIMIZE | MWM_DECOR_MAXIMIZE);

    /// !!! add disabled buttons
    decors &=
        ~(/*((func & MWM_FUNC_RESIZE) ? 0 : MWM_DECOR_RESIZEH) |*/
          ((func & MWM_FUNC_MINIMIZE) ? 0 : MWM_DECOR_MINIMIZE) |
          ((func & MWM_FUNC_MAXIMIZE) ? 0 : MWM_DECOR_MAXIMIZE));

    return decors;
}
#endif

bool YFrameClient::getKwmIcon(int *count, Pixmap **pixmap) {
    Atom r_type;
    int r_format;
    unsigned long nitems;
    unsigned long bytes_remain;
    unsigned char *prop;
    
    if (XGetWindowProperty(app->display(), handle(),
                           _XA_KWM_WIN_ICON, 0, 2, False, _XA_KWM_WIN_ICON,
                           &r_type, &r_format, &nitems, &bytes_remain,
                           &prop) == Success && prop)
    {
        if (r_format == 32 &&
            r_type == _XA_KWM_WIN_ICON &&
            nitems == 2)
        {
            *count = nitems;
            *pixmap = (Pixmap *)prop;

            if (fHints)
                XFree(fHints);
            if ((fHints = XGetWMHints(app->display(), handle())) != 0) {
            }
            return true;
        } else {
            XFree(prop);
        }
    }
    return false;
}

bool YFrameClient::getWinIcons(Atom *type, int *count, CARD32 **elem) {
    Atom r_type;
    int r_format;
    unsigned long nitems;
    unsigned long bytes_remain;
    unsigned char *prop;
    
    if (XGetWindowProperty(app->display(), handle(),
                           _XA_WIN_ICONS, 0, 4096, False, AnyPropertyType,
                           &r_type, &r_format, &nitems, &bytes_remain,
                           &prop) == Success && prop)
    {
        if (r_format == 32 && nitems > 0) {

            if (r_type == _XA_WIN_ICONS ||
                r_type == XA_PIXMAP) // for compatibility, obsolete, (will be removed?)
            {
                *type = r_type;
                *count = nitems;
                *elem = (CARD32 *)prop;
                return true;
            }
        }
        XFree(prop);
    }
    return false;
}

void YFrameClient::setWinWorkspaceHint(CARD32 wk) {
    XChangeProperty(app->display(),
                    handle(),
                    _XA_WIN_WORKSPACE,
                    XA_CARDINAL,
                    32, PropModeReplace,
                    (unsigned char *)&wk, 1);
}

bool YFrameClient::getWinWorkspaceHint(CARD32 *workspace) {
    Atom r_type;
    int r_format;
    unsigned long count;
    unsigned long bytes_remain;
    unsigned char *prop;

    if (XGetWindowProperty(app->display(),
                           handle(),
                           _XA_WIN_WORKSPACE,
                           0, 1, False, XA_CARDINAL,
                           &r_type, &r_format,
                           &count, &bytes_remain, &prop) == Success && prop)
    {
        if (r_type == XA_CARDINAL && r_format == 32 && count == 1U) {
            CARD32 ws = *(CARD32 *)prop;
            if (ws < workspaceCount) {
                *workspace = ws;
                XFree(prop);
                return true;
            }
        }
        XFree(prop);
    }
    return false;
}

void YFrameClient::setWinLayerHint(CARD32 layer) {
    XChangeProperty(app->display(),
                    handle(),
                    _XA_WIN_LAYER,
                    XA_CARDINAL,
                    32, PropModeReplace,
                    (unsigned char *)&layer, 1);
}

bool YFrameClient::getWinLayerHint(CARD32 *layer) {
    Atom r_type;
    int r_format;
    unsigned long count;
    unsigned long bytes_remain;
    unsigned char *prop;

    if (XGetWindowProperty(app->display(),
                           handle(),
                           _XA_WIN_LAYER,
                           0, 1, False, XA_CARDINAL,
                           &r_type, &r_format,
                           &count, &bytes_remain, &prop) == Success && prop)
    {
        if (r_type == XA_CARDINAL && r_format == 32 && count == 1U) {
            CARD32 l = *(CARD32 *)prop;
            if (l < WinLayerCount) {
                *layer = l;
                XFree(prop);
                return true;
            }
        }
        XFree(prop);
    }
    return false;
}

bool YFrameClient::getWinStateHint(CARD32 *mask, CARD32 *state) {
    Atom r_type;
    int r_format;
    unsigned long count;
    unsigned long bytes_remain;
    unsigned char *prop;

    if (XGetWindowProperty(app->display(),
                           handle(),
                           _XA_WIN_STATE,
                           0, 2, False, XA_CARDINAL,
                           &r_type, &r_format,
                           &count, &bytes_remain, &prop) == Success && prop)
    {
        MSG(("got state"));
        if (r_type == XA_CARDINAL && r_format == 32 && count == 2U) {
            CARD32 m = ((CARD32 *)prop)[0];
            CARD32 s = ((CARD32 *)prop)[1];

            *state = s;
            *mask = m;
            XFree(prop);
            return true;
        }
        MSG(("bad state"));
        XFree(prop);
    }
    return false;
}

void YFrameClient::setWinStateHint(CARD32 mask, CARD32 state) {
    CARD32 s[2];

    s[0] = mask;
    s[1] = state;

    MSG(("set state=%lX mask=%lX", state, mask));

    XChangeProperty(app->display(),
                    handle(),
                    _XA_WIN_STATE,
                    XA_CARDINAL,
                    32, PropModeReplace,
                    (unsigned char *)&s, 2);
}
