#include "hypermail.h"
#include <ctype.h>

/*
** printfile - print html header/footer file and fill in values 
**             substituting for magic cookies. Return allocated
**             memory with the header/footer filled in.
**
**      Substitution cookies supported
**
**              %% - '%' character
**              %~ - storage directory
**              %a - Other Archives URL
**              %b - About Archive URL
**              %B - Body line specified
**              %e - email addr of message author - Not valid on index pages
**              %g - date and time archive generated
**              %h - HMURL
**              %i - Message-id - Not valid on index pages
**              %l - archive label
**              %m - Mailto address
**              %p - PROGNAME
**              %s - Subject of message or Index Title
**              %v - VERSION
**              %u - Expanded version link (HMURL,PROGNAME,VERSION)
**              %S - Subject META TAG - Not valid on index pages
**              %A - Author META TAG - Not valid on index pages
**              \n - newline character
**              \t - tab character
**
*/

int printfile(FILE *fp, char *format, char *label, char *subject,
                  char *dir, char *name, char *email, char *message_id)
{
    register char *cp;
    register char *aptr;
    char c;

    aptr = format;

    while ((c = *aptr++)) {
        if (c == '\\') {
           switch (*aptr++) {
               case 'n':  /* Add the \n character */
                   putc('\n', fp);
                   continue;
               case 't':  /* Add the \t character */
                   putc('\t', fp);
                   continue;
               default:
                   break;
           }    /* end switch */
        }
        else if (c == '%') {
           switch (*aptr++) {
               case '%':  /* Add the % character */
                   putc('%', fp);
                   continue;
               case '~':     /* %~ - storage directory */
                   for (cp = dir; *cp; cp++) 
                       putc(*cp, fp);
                   continue;
               case 'A':     /* %e - email address of message author */
                   if (email && name) {
                       fprintf(fp,"<META NAME=\"Author\" CONTENT=\"%s (%s)\">", 
                               name, email);
                   }
                   continue;
               case 'a':     /* %a - Other Archives URL */
                   if (*archives) {
                       for (cp = archives; *cp; cp++) 
                           putc(*cp, fp);
                   }
                   continue;
               case 'B':     /* %B - BODY specified */
                   if (*htmlbody) {
                       for (cp = htmlbody; *cp; cp++) 
                           putc(*cp, fp);
                   }
                   continue;
               case 'b':     /* %b - About this archive URL */
                   if (*about) {
                       for (cp = about; *cp; cp++) 
                           putc(*cp, fp);
                   }
                   continue;
               case 'e':     /* %e - email address of message author */
                   if (email) {
                       for (cp = email; *cp; cp++) 
                           putc(*cp, fp);
                   }
                   continue;
               case 'g':     /* %g - date and time archive generated */
                   for (cp = getlocaltime(); *cp; cp++)
                       putc(*cp, fp);
                   continue;
               case 'h':   /* %h - Hypermail Resource Center */
                   for (cp = HMURL; *cp; cp++) 
                       putc(*cp, fp);
                   continue;
               case 'i':     /* %m - Message-ID of message*/
                   if (message_id) {
                       for (cp = message_id; *cp; cp++) 
                           putc(*cp, fp);
                   }
                   continue;
               case 'l':     /* %l - Archive label  */
                   for (cp = label; *cp; cp++) 
                       putc(*cp, fp);
                   continue;
               case 'm':     /* %m - mailto*/
                   for (cp = mailto; *cp; cp++) 
                       putc(*cp, fp);
                   continue;
               case 'p':   /* %p - PROGNAME */
                   for (cp = PROGNAME; *cp; cp++) 
                       putc(*cp, fp);
                   continue;
               case 's':     /* %s - Subject of message or Index Title */
                   for (cp = convchars(subject); *cp; cp++) 
                       putc(*cp, fp);
                   continue;
               case 'S':     /* %s - Subject of message or Index Title */
                   fprintf(fp,"<META NAME=\"Subject\" CONTENT=\"%s\">", 
                           convchars(subject));
                   continue;
               case 'v':   /* %v - VERSION */
                   for (cp = VERSION; *cp; cp++) 
                       putc(*cp, fp);
                   continue;
               case 'u':   /* %u - Expanded Version link */
                   fprintf(fp,"<A HREF=\"%s\">%s %s</A>", 
                                HMURL, PROGNAME, VERSION);
                   continue;
               default:
                   putc('%', fp);
                   putc('?', fp);
                   continue;
           }    /* end switch */
        }
        putc(c, fp);
     }  /* end while */

     fflush(fp);
     return(0);
}

/*
** Prints the standard page header 
*/

void print_main_header(FILE *fp,char *label,char *name,
                       char *email,char *subject) 
{
    char title[256];
    char *rp;

    fprintf(fp,"<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\">\n");
    fprintf(fp,"<HTML>\n");
    fprintf(fp,"<HEAD>\n");

    /* 
    ** Assure the title is not longer than 64 characters as is
    ** recommended by the HTML specs. Also, strip off any trailing
    ** whitespace in TITLE so weblint is happy. 
    */

    sprintf(title, "%s: %s", label, convchars(subject));

    if (strlen(title) > TITLESTRLEN) {
        title[TITLESTRLEN-1] = '\0';
        rp = title+(TITLESTRLEN-2);
    }
    else 
        rp = title+(strlen(title)-1);

    while(isspace(*rp))
         *rp-- = '\0';

    fprintf(fp,"<TITLE>%s</TITLE>\n", title);

    if (name && email)
        fprintf(fp,"<META NAME=\"Author\" CONTENT=\"%s (%s)\">\n",name,email);
    fprintf(fp,"<META NAME=\"Subject\" CONTENT=\"%s\">\n", convchars(subject));
    if (use_mailto)
        fprintf(fp,"<LINK REV=\"made\" HREF=\"mailto:%s\">\n",mailto);
    fprintf(fp,"</HEAD>\n");
    
    if (*htmlbody)
        fprintf(fp,"%s\n",htmlbody);
    else
        fprintf(fp,"<BODY BGCOLOR=\"#FFFFFF\" TEXT=\"#000000\">\n");
}

/*
** Prints the header for an index page.
*/

void print_msg_header(FILE *fp, char *label, char *subject, 
                      char *dir, char *name, char *email, char *msgid)
{
    if (mhtmlheader)
        printfile(fp,mhtmlheader,label,subject,dir,name,email,msgid);
    else {
       print_main_header(fp, label, name, email, subject);
       fprintf(fp, "<H1 ALIGN=CENTER>%s</H1>\n", convchars(subject));
       if (!usetable)
           fprintf(fp,"<HR>\n<P>\n");
    }
}

/*
** Prints the header for the by_date index page.
*/

void print_index_header(FILE *fp,char *label,char *dir,char *subject) 
{
    if (ihtmlheader)
        printfile(fp,ihtmlheader,label,subject,dir,NULL,NULL,NULL);
    else {
        print_main_header(fp, label, NULL, NULL, subject);
        fprintf(fp, "<H1 ALIGN=CENTER>%s<BR>%s</H1>\n", label, subject);
        if (!usetable)
            fprintf(fp,"<HR WIDTH=400>\n");
    }
}

/*
** Prints the HTML page footer.
*/

void printfooter(FILE *fp,char *htmlfooter,char *label,char *dir,char *subject)
{
    printcomment(fp, "trailer", "footer");

    if (htmlfooter)
        printfile(fp, htmlfooter,label,subject,dir,NULL,NULL,NULL);
    
    else {
        if (showhr && !usetable)
            fprintf(fp,"<HR>\n");
        fprintf(fp,"<P>\n<SMALL>\n<EM>\n");
        fprintf(fp, "This archive was generated by ");
        fprintf(fp, "<A HREF=\"%s\">%s %s</A> \n", HMURL, PROGNAME, VERSION);
        fprintf(fp,"on <EM>%s</EM>\n",getlocaltime());
        fprintf(fp,"</EM>\n</SMALL>\n");
        fprintf(fp, "</BODY>\n</HTML>\n");
    }
}
