/* 
** Copyright (C) 1994, 1995 Enterprise Integration Technologies Corp.
**         VeriFone Inc./Hewlett-Packard. All Rights Reserved.
** Kevin Hughes, kev@kevcom.com 3/11/94
** Kent Landfield, kent@landfield.com 4/6/97
** 
** This program and library is free software; you can redistribute it and/or 
** modify it under the terms of the GNU (Library) General Public License 
** as published by the Free Software Foundation; either version 2 
** of the License, or any later version. 
** 
** This program is distributed in the hope that it will be useful, 
** but WITHOUT ANY WARRANTY; without even the implied warranty of 
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
** GNU (Library) General Public License for more details. 
** 
** You should have received a copy of the GNU (Library) General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
*/

#include "hypermail.h"
#include <ctype.h>

/*
** Prints a program error string and stops. If errorstr is NULL,
** this prints whatever happens to be in errstr[] at the time.
*/

void progerr(char *errorstr)
{
    if (errorstr != NULL)
        fprintf(stderr, "%s: %s\n", PROGNAME, errorstr);
    else
        fprintf(stderr, "%s: %s\n", PROGNAME, errmsg);
    fprintf(stderr, "%s: type \"%s -z\" for options.\n", PROGNAME, PROGNAME);
    exit(-1);
}

/*
** Output a menu line with hyperlinks for table display
**
** All parameters are inputs only.  Parameters:
**   fp       : HTML output file pointer
**   idx      : Type of page this menu is for.
**   about    : "NONE" or else the URL of info about this archive.
**   archives : "NONE" or else the URL of more hypermail archives.
**   currentid: "" or else the id of the "current" message.
**   cursub   : "" or else the subject of the "current" message.
**   pos      : Called at the top or bottom of the page.
*/

void fprint_menu(FILE *fp, mindex_t idx, char *about, char *archives,
                 char *currentid, char *cursub, int pos)
{
   fprintf(fp, "<DIV ALIGN=CENTER>\n<TABLE BORDER=2 WIDTH=\"100%%\">\n<TR>\n");

   if (mailcommand) {
       if (*hmail) {
           fprintf(fp, "<TH><A HREF=\"%s\">New Message</A></TH>\n",
                  makemailcommand("mailto:$TO", hmail, "", ""));

           if((currentid != NULL && currentid[0] != '\0') ||
              (cursub    != NULL && cursub[0]    != '\0'))
              fprintf(fp, "<TH><A HREF=\"%s\">Reply</A></TH>\n",
                   makemailcommand(mailcommand,hmail,currentid,cursub));
       }
   }

   if (*about)
      fprintf(fp,"<TH><A HREF=\"%s\">About this list</A></TH>\n",about);

   if (idx != NO_INDEX && !reverse) {
      if (pos == PAGE_TOP)
          fprintf(fp,"<TH><A HREF=\"#end\">End of Messages</A></TH>\n");
      else
          fprintf(fp,"<TH><A NAME=\"end\" HREF=\"#\">Start of Messages</A></TH>\n");
   }
           
   if(idx != DATE_INDEX)
      fprintf(fp,"<TH><A HREF=\"%s\">Date view</A></TH>\n",datename);

   if(idx != THREAD_INDEX)
      fprintf(fp,"<TH><A HREF=\"%s\">Thread view</A></TH>\n",thrdname);

   if(idx != SUBJECT_INDEX)
      fprintf(fp,"<TH><A HREF=\"%s\">Subject view</A></TH>\n",subjname);

   if(idx != AUTHOR_INDEX)
      fprintf(fp,"<TH><A HREF=\"%s\">Author view</A></TH>\n",authname);

   if (*archives)
      fprintf(fp,"<TH><A HREF=\"%s\">Other groups</A></TH>\n", archives);

   fprintf(fp, "</TR>\n</TABLE>\n</DIV>\n");
}

/*----------------------------------------------------------------------------*/

void fprint_summary(FILE *fp, int first_d, int last_d, int num)
{
   fprintf(fp, "<DIV ALIGN=CENTER>\n");
   fprintf(fp, "<TABLE>\n<TR>\n  <TH COLSPAN=4>%d Messages</TH>\n</TR>\n", num);
   fprintf(fp, "<TR>\n");
   fprintf(fp, "  <TH>Starting:</TH><TD><EM>%s</EM></TD>\n",getdatestr(first_d));
   fprintf(fp, "  <TH>Ending:</TH><TD><EM>%s</EM></TD>\n",getdatestr(last_d));
   fprintf(fp, "</TR>\n</TABLE>\n</DIV>\n");
}

/*----------------------------------------------------------------------------*/

void print_index_header_links(FILE *fp,char *archives,char *about,int called)
{
    /* 
    ** Print out the links for
    ** 
    **      About this archive 
    **      Most recent messages 
    **      Messages sorted by: [ date ][ subject ][ author ] 
    **      Other mail archives 
    **
    ** as appropriate.
    */
    
    fprintf(fp, "<UL>\n");

    if (*about) {
        fprintf(fp, "<LI><STRONG><A HREF=\"%s\">About this ", about);
        fprintf(fp, "archive</A></STRONG>\n");
    }

    if (!reverse && (called != FROM_AUTHOR && called != FROM_SUBJECT))
        fprintf(fp,
        "<LI><STRONG><A HREF=\"#end\">Most recent messages</A></STRONG>\n");

    fprintf(fp, "<LI><STRONG>Messages sorted by:</STRONG> \n");

    if (called != FROM_AUTHOR)
        fprintf(fp, "<A HREF=\"%s\">[ author ]</A>\n", authname);

    if (called != FROM_DATE)
         fprintf(fp, "<A HREF=\"%s\">[ date ]</A>\n", datename);

    if (called != FROM_THREAD)
         fprintf(fp, "<A HREF=\"%s\">[ thread ]</A>\n", thrdname);

    if (called != FROM_SUBJECT)
         fprintf(fp, "<A HREF=\"%s\">[ subject ]</A>\n", subjname);

    if (*archives) {
        fprintf(fp, "<LI><STRONG><A HREF=\"%s\">Other mail ", archives);
        fprintf(fp, "archives</A></STRONG>\n");
    }
    fprintf(fp, "</UL>\n<P>\n");
}

void print_index_footer_links(FILE *fp, char *archives, int called)
{
    /* 
    ** Print out the links for
    ** 
    **      Messages sorted by: [ date ][ subject ][ author ] 
    **      Other mail archives 
    **
    ** as appropriate.
    */

    fprintf(fp, "<UL>\n");
    fprintf(fp, "<LI><STRONG>Messages sorted by:</STRONG> \n");

    if (called != FROM_AUTHOR)
        fprintf(fp, "<A HREF=\"%s\">[ author ]</A>\n", authname);

    if (called != FROM_DATE)
        fprintf(fp, "<A HREF=\"%s\">[ date ]</A>\n", datename);

    if (called != FROM_THREAD)
        fprintf(fp, "<A HREF=\"%s\">[ thread ]</A>\n", thrdname);

    if (called != FROM_SUBJECT)
        fprintf(fp, "<A HREF=\"%s\">[ subject ]</A>\n", subjname);

    if (*archives) {
        fprintf(fp, "<LI><STRONG><A HREF=\"%s\">Other mail ", archives);
        fprintf(fp, "archives</A></STRONG>\n");
    }
    fprintf(fp, "</UL>\n\n");
}

/*
** Prints a comment in the file.
*/

void printcomment(FILE *fp, char *label, char *value)
{
    fprintf(fp, "<!-- %s=\"%s\" -->\n", label, value);
}

/*
** Prints a line of HTML. Used to filter off unwanted tags from the output.
*/

void printhtml(FILE *fp, char *line)
{
  /* the banned tags ignore all of them that begin exactly like "<tag" or
     "</tag" (case insensitive). Any other letters within the tags destroy
     this filter capability. */

  char *banned_tags[]={"html>"}; /* ending >-letter must be included */
  int i;
  int index;
  
  while(*line) {
    if('<' != *line)
      fputc(*line++, fp);
    else {
      if('/' == line[1])
        index=2;
      else
        index=1;
      for(i=0; i<sizeof(banned_tags)/sizeof(banned_tags[0]);i++) {
        if(!strncasecmp(&line[index],
                        banned_tags[i],
                        strlen(banned_tags[i]))) {
          line+=strlen(banned_tags[i])+index;
          break;
        }
      }
      if(sizeof(banned_tags)/sizeof(banned_tags[0]) == i) {
        fputc(*line++, fp);
        if(2 == index)
          fputc(*line++, fp);
      }
    }
  }

}

/*
** Pretty-prints the dates in the index files.
*/

void printdates(FILE *fp, struct header *hp)
{
    if (hp != NULL) {
        printdates(fp, hp->left);
        fprintf(fp, "<LI><A HREF=\"%.4d.html\"><STRONG>%s</STRONG></A> ",
                    hp->msgnum, convchars(hp->subject));
        fprintf(fp, "<A NAME=\"%d\"><EM>%s</EM></A>\n", hp->msgnum, hp->name);
        printdates(fp, hp->right);
    }
}

int showheader(char *header)
{
   return (inlist(show_headers, header));
}


/*
** The heuristics for displaying an otherwise ordinary line (a non-quote,
** non-sig, non-inhtml, non-blank line) if 'showhtml' is true (which really
** means we're *not* doing <PRE> to format the text) have changed so that
** indented text shows up correctly.  First, leading spaces become HTML
** non-breaking spaces (&nbsp;).  In order for this to work, those lines
** must have been preceded by a <BR> or <P>.  We accomplish this by checking
** ahead when we print each line... if we determine that the next line
** should be broken before, we do so.
*/

void printbody(FILE *fp, struct body *bp, char *id, char *subject)
{
    int insig, inblank;
    int inhtml;
    char inheader=FALSE; /* we always start in a mail header */
    int pre=FALSE;

    if (showhr)
        fprintf(fp, "<HR>\n");

    printcomment(fp, "body", "start");

    if (!showhtml) {
      fprintf(fp, "<PRE>\n");
      pre=TRUE;
    }
#if 0
    else
      printf("SHOWHTML is ON\n");
#endif
    inblank = 1;
    insig = 0;

    inhtml = 0;

    while (bp != NULL) {
#if 1
          if (bp->html) {
            /* already in HTML, don't touch */
            printhtml(fp, bp->line);
#else
            fprintf(fp, "%s", convchars(bp->line));
          if (bp->html && (strncasecmp(bp->line, "<HTML>", 6) && 
                           strncasecmp(bp->line, "</HTML>", 7))) {
            fprintf(fp, "%s", convchars(bp->line));
#endif
            bp = bp->next;
            continue;
          }

          if(bp->header) {
            char head[128];
            if(!inheader) {
              if(!pre && showhtml && showheaders) {
                fprintf(fp, "<PRE>\n");
                pre=TRUE;
              }
            }
            inheader=TRUE;
            if(sscanf(bp->line, "%127[^:]", head) &&
               show_headers && !showheader(head)) {
              /* the show header keyword has been used, then we skip all those
                 that aren't mentioned! */
              if(isalnum(*head) || !showheaders) {
                /* this check is only to make sure that the last line among 
                   the headers (the "\n" one) won't be filtered off */
                bp = bp->next;
                continue;
              }
            }
          }
          else {
            if(inheader) {
              if(showhtml && showheaders && pre) {
                fprintf(fp, "</PRE>\n");
                pre=FALSE;
              }
              fprintf(fp, "<P>\n");
            }
            inheader=FALSE;
          }

          if (bp->attached && bp->header) {
#if 0
            /*
             * This is an attached mail's headers. Only output the few
             * ones we might be interested in. Hmmm... Are there others ?
             */
            if (!strncmp(bp->line, "From:", 5) ||
               !strncmp(bp->line, "To:", 3) ||
               !strncmp(bp->line, "Date:", 5) ||
               !strncmp(bp->line, "Subject:", 8) ) {
#endif
              fprintf(fp, "<STRONG>%s</STRONG><BR>\n", bp->line);
#if 0
            }
#endif
            bp = bp->next;
            continue;            
          }

        if ((bp->line)[0] == '\n') {
#if 0
            if (bp->header && showhtml && showheaders)
                fprintf(fp, "</PRE>\n");
#endif
        }
        else if (bp->header && !showheaders) {
            bp = bp->next;
            continue;
        }

        if ((bp->line)[0] == '\n' && inblank) {
            bp = bp->next;
            continue;
        }
        else
            inblank = 0;

        if (!strncasecmp(bp->line, "<HTML>", 6)) {
            inhtml = 1;
            if (!showhtml && pre) {
                fprintf(fp, "</PRE>\n");
                pre=FALSE;
            }
        }
        else if (!strncasecmp(bp->line, "</HTML>", 7)) {
            inhtml = 0;
            if (showhtml)
                fprintf(fp, "%s", bp->line);
            else
                fprintf(fp, "%s<BR>", convchars(bp->line));
            if (!showhtml && !pre) {
                fprintf(fp, "<PRE>\n");
                pre=TRUE;
            }
            bp = bp->next;
            continue;
        }

        if (inhtml) {
            if (showhtml)
                fprintf(fp, "%s", bp->line);
            else
                fprintf(fp, "%s<BR>", convchars(bp->line));
            bp = bp->next;
            continue;
        }
        if (showhtml) {
            if (!strcmp(bp->line, "--\n") ||
                !strcmp(bp->line, "-- \n") ||
                !strcmp(bp->line, "---\n")) {
                insig = 1;
                fprintf(fp, "<PRE>\n");
                pre=TRUE;
            }

            if ((bp->line)[0] == '\n')
                fprintf(fp, "<P>");
            else {
                if (insig)
                    fprintf(fp, "%s", (char *)convurls(bp->line,id,subject));
                else if (isquote(bp->line)) {
                    fprintf(fp, "%s%s%s<BR>\n",
                           (iquotes)?"<EM>":"",
                            /* Daniel changed this 10th of July 1998 from
                               convurls(rmcr(bp->line), id, subject),  */
                            /* Why ? */
                           convurls(bp->line, id, subject),
                           (iquotes)?"</EM>":"");
                }
                else if ((bp->line)[0] != '\0') {
#if 1
                    char *sp;
                    sp = bp->line;
                    while (*sp && (*sp == ' ' || *sp == '\t')) {
                        if (*sp == '\t')
                            fprintf(fp, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;");
                        else
                            fprintf(fp, "&nbsp;");
                        sp++;
                    }
                    fprintf(fp, "%s",convurls(sp,id,subject));
#else
                    int sp;
                    for (sp = 0;isspace(bp->line[sp]);sp++){
                        if (bp->line[sp] == '\t')
                            fprintf(fp, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;");
                        else
                            fprintf(fp, "&nbsp;");
                    }
                    fprintf(fp, "%s",convurls(&bp->line[sp],id,subject));
#endif

                    /*
                    ** Determine whether we should break.
                    ** We could check for leading spaces
                    ** or quote lines, but in general,
                    ** non-alphanumeric lines should be
                    ** broken before.
                    */
 
                    if ((showbr && !bp->header) ||
                        ((bp->next != NULL) && !isalnum(bp->next->line[0])))
                      fprintf(fp, "<BR>");
                    if(!bp->header) {
                      fprintf(fp, "\n");
                    }
                }

            }
        }
        else if ((bp->line)[0] != '\0') {
            fprintf(fp, "%s", (char *) convurls(bp->line, id, subject));
        }
        bp = bp->next;

    }

    if (pre) 
        fprintf(fp, "</PRE>\n");

    fprintf(fp, "<P>");
    printcomment(fp, "body", "end");
    if (showhr)
        fprintf(fp, "<HR>\n");
}

/*
** Printing...the other main part of this program!
** This writes out the articles, beginning with the number startnum.
*/

void writearticles(char *dir,char *label,char *about,int overwrite,int startnum)
{
    int num, statusnum, skip, subjmatch, newfile;
    char name[NAMESTRLEN], email[MAILSTRLEN], subject[SUBJSTRLEN],
        inreply[REPYSTRLEN], date[DATESTRLEN], fromdate[DATESTRLEN],
        msgid[MSGDSTRLEN];
    char name2[NAMESTRLEN], email2[MAILSTRLEN], subject2[SUBJSTRLEN],
        inreply2[REPYSTRLEN], date2[DATESTRLEN], fromdate2[DATESTRLEN],
        msgid2[MSGDSTRLEN];
    char filename[MAXFILELEN];
    struct body *bp, *status;
    struct reply *rp;
    FILE *fp;

    num = startnum;
    skip = 0;

    if (showprogress)
        printf("Writing articles to \"%s\"...    ", dir);

    while ((bp = (struct body *) hashnumlookup(num, name, email,
            subject, inreply, date, fromdate, msgid)) != NULL) {

        sprintf(filename, "%s%s%.4d.html", 
                dir, (dir[strlen(dir) - 1] == '/') ? "" : "/", num);

        /*
        ** Determine to overwrite files or not
        */

        if (isfile(filename))
            newfile = 0;
        else
            newfile = 1;

        if (!newfile && !overwrite) {
            skip = 1; /* is this really necessary with continue ??? */
            num++;
            continue;
        }
        else {
            if ((fp = fopen(filename, "w")) == NULL) {
                sprintf(errmsg, "Couldn't write \"%s\".", filename);
                progerr((char *) NULL);
            }
        }

        /*
        ** Create the comment fields necessary for incremental updating
        */

        print_msg_header(fp, label, subject, dir, name, email, msgid);

        printcomment(fp, "received", fromdate);
        printcomment(fp, "sent", date);
        printcomment(fp, "name", name);
        printcomment(fp, "email", email);
        printcomment(fp, "subject", convchars(subject));
        printcomment(fp, "id", msgid);
        printcomment(fp, "inreplyto", convchars(inreply));

        if (usetable) {
            fprint_menu(fp,NO_INDEX,about,"",msgid,subject,PAGE_TOP);
            fprintf(fp, "<P>\n");
        }

        /*
        ** Print the message's author info and date.
        */

        if (!strcmp(name, email)) {
            if (use_mailcommand) {
                fprintf(fp, "<A HREF=\"%s\">", 
                         makemailcommand(mailcommand,email,msgid,subject));
                fprintf(fp, "<EM>%s</EM></A><BR>\n", name);
            }
            else
                fprintf(fp, "<EM>%s</EM><BR>\n", name);
        }
        else {
            if (use_mailcommand) {  
                fprintf(fp,"<STRONG>%s</STRONG> (<A HREF=\"%s\">",
                        name,makemailcommand(mailcommand,email,msgid,subject));
                fprintf(fp, "<EM>%s</EM></A>)<BR>\n", email);
            }
            else
                fprintf(fp, "%s (<EM>%s</EM>)<BR>\n", name, email);
        }
        fprintf(fp, "<EM>%s</EM>\n<P>\n", date);
 
        /*
        ** This is here because it looks better here. The table looks
        ** better before the Author info. This stuff should be in 
        ** printfile() so it could be laid out as the user wants...
        */

        if (!usetable) {
            fprintf(fp, "<UL>\n");
            fprintf(fp, "<LI><STRONG>Messages sorted by:</STRONG> \n");
            fprintf(fp, "<A HREF=\"%s#%d\">[ date ]</A>\n", datename, num);
            fprintf(fp, "<A HREF=\"%s#%d\">[ thread ]</A>\n", thrdname, num);
            fprintf(fp, "<A HREF=\"%s#%d\">[ subject ]</A>\n", subjname, num);
            fprintf(fp, "<A HREF=\"%s#%d\">[ author ]</A>\n", authname,num);
        }

        /*
        ** Should we print message links ?
        */

        if (show_msg_links) {
            printcomment(fp, "next", "start");
            if (usetable)
                fprintf(fp, "<UL>\n");
    
            /*
            ** Is there a next message?
            */
    
            status = (struct body *) hashnumlookup(num + 1, name2, email2,
                      subject2, inreply2, date2, fromdate2, msgid2);
            if (status != NULL) {
                fprintf(fp, "<LI><STRONG>Next message:</STRONG> ");
                fprintf(fp, "<A HREF=\"%.4d.html\">%s: \"%s\"</A>\n",
                num + 1, name2, convchars(subject2));
            }
    
            /*
            ** Is there a previous message?
            */
    
            status = (struct body *) hashnumlookup(num - 1, name2, email2,
                      subject2, inreply2, date2, fromdate2, msgid2);
            if (status != NULL) {
                fprintf(fp, "<LI><STRONG>Previous message:</STRONG> ");
                fprintf(fp, "<A HREF=\"%.4d.html\">%s: \"%s\"</A>\n",
                num - 1, name2, convchars(subject2));
            }
    
            /*
            ** Is this message a reply to another?
            */
    
            if (inreply[0] != '\0') {
                statusnum = hashreplylookup(inreply2,name2,subject2,&subjmatch);
                if (statusnum != -1) {
                    if (subjmatch)
                        fprintf(fp,"<LI><STRONG>Maybe in reply to:</STRONG>");
                    else
                        fprintf(fp,"<LI><STRONG>In reply to:</STRONG>");
                    fprintf(fp," <A HREF=\"%.4d.html\">",statusnum);
                    fprintf(fp,"%s: \"%s\"</A>\n",name2,convchars(subject2));
                }
            }
    
            /*
            ** Is there a message next in the thread?
            */
    
            printcomment(fp, "nextthread", "start");
    
            for (rp = threadlist; rp != NULL; rp = rp->next)
                if (rp->msgnum == num && rp->next != NULL &&
                    rp->next->msgnum != -1) {
                    fprintf(fp, "<LI><STRONG>Next in thread:</STRONG> ");
                    fprintf(fp, "<A HREF=\"%.4d.html\">", rp->next->msgnum);
                    fprintf(fp, "%s: \"%s\"</A>\n",
                            rp->next->name, convchars(rp->next->subject));
                }
    
                /*
                ** Does this message have replies? If so, print them all!
                */
    
                if (showreplies) {
                    for (rp = replylist; rp != NULL; rp = rp->next) {
                        if (rp->msgnum == num) {
                            if (rp->maybereply)
                                fprintf(fp,"<LI><STRONG>Maybe reply:</STRONG>");
                            else
                                fprintf(fp, "<LI><STRONG>Reply:</STRONG>");
                            fprintf(fp," <A HREF=\"%.4d.html\">",rp->frommsgnum);
                            fprintf(fp,"%s: \"%s\"</A>\n",rp->name,convchars(rp->subject));
                        }
                    }
                    printcomment(fp, "reply", "end");
                }
        }

        fprintf(fp, "</UL>\n");


        /*
        ** Finally...print the body!
        */

        printbody(fp, bp, msgid, subject);

        /*
        ** Should we print out the message links ?
        */

        if (show_msg_links) {
            fprintf(fp, "<P>\n<UL>\n");
    
            printcomment(fp, "next", "start");
    
            status = (struct body *) hashnumlookup(num + 1, name2,
                      email2, subject2, inreply2, date2, fromdate2, msgid2);
            if (status != NULL) {
                fprintf(fp, "<LI><STRONG>Next message:</STRONG> ");
                fprintf(fp, "<A HREF=\"%.4d.html\">%s: \"%s\"</A>\n",
                        num + 1, name2, convchars(subject2));
            }
    
            status = (struct body *) hashnumlookup(num - 1, name2,
                      email2, subject2, inreply2, date2, fromdate2, msgid2);
            if (status != NULL) {
                fprintf(fp, "<LI><STRONG>Previous message:</STRONG> ");
                fprintf(fp, "<A HREF=\"%.4d.html\">%s: \"%s\"</A>\n",
                        num - 1, name2, convchars(subject2));
            }
    
            if (inreply[0] != '\0') {
                statusnum = hashreplylookup(inreply2,name2,subject2,&subjmatch);
                if (statusnum != -1) {
                    if (subjmatch)
                        fprintf(fp,"<LI><STRONG>Maybe in reply to:</STRONG>");
                    else
                        fprintf(fp,"<LI><STRONG>In reply to:</STRONG>");
                    fprintf(fp," <A HREF=\"%.4d.html\">",statusnum);
                    fprintf(fp,"%s: \"%s\"</A>\n",name2,convchars(subject2));
                }
            }
    
            printcomment(fp, "nextthread", "start");
    
            for (rp = threadlist; rp != NULL; rp = rp->next) {
                if (rp->msgnum == num && rp->next != NULL &&
                    rp->next->msgnum != -1) {
                    fprintf(fp, "<LI><STRONG>Next in thread:</STRONG> ");
                    fprintf(fp, "<A HREF=\"%.4d.html\">", rp->next->msgnum);
                    fprintf(fp, "%s: \"%s\"</A>\n",
                            rp->next->name, convchars(rp->next->subject));
                }
            }
    
            if (showreplies) {
                for (rp = replylist; rp != NULL; rp = rp->next) {
                    if (rp->msgnum == num) {
                        if (rp->maybereply)
                            fprintf(fp, "<LI><STRONG>Maybe reply:</STRONG>");
                        else
                            fprintf(fp, "<LI><STRONG>Reply:</STRONG>");
                        fprintf(fp, " <A HREF=\"%.4d.html\">", rp->frommsgnum);
                        fprintf(fp, "%s: \"%s\"</A>\n",
                                rp->name, convchars(rp->subject));
                    }
                }
                printcomment(fp, "reply", "end");
            }
            fprintf(fp, "</UL>\n");
    
        }

        if (usetable)
            fprint_menu(fp,NO_INDEX,about,"",msgid,subject,PAGE_BOTTOM);

        printfooter(fp, mhtmlfooter, label, dir, subject);

        fclose(fp);

        if (newfile && chmod(filename, filemode) == -1) {
            sprintf(errmsg, "Couldn't chmod \"%s\" to %o.",
            filename, filemode);
            progerr((char *) NULL);
        }

        if (!(num % 5) && showprogress && !skip) {
            printf("\b\b\b\b%3.0f%c",((float) num / (float) bignum) * 100,'%');
            fflush(stdout);
        }

        num++;
    }
    if (showprogress)
        printf("\b\b\b\b    \n");
    if (!overwrite)
        bignum = num - 1;
}


/*
** Write the date index...
*/

void writedates(char *dir, char *label, char *archives, char *about)
{
    int newfile;
    char filename[MAXFILELEN];
    FILE *fp;

    sprintf(filename, "%s%s%s", dir,
	(dir[strlen(dir) - 1] == '/') ? "" : "/", datename);

    if (isfile(filename))
        newfile = 0;
    else
        newfile = 1;

    if ((fp = fopen(filename, "w")) == NULL) {
        sprintf(errmsg, "Couldn't write \"%s\".", filename);
        progerr((char *) NULL);
    }

    if (showprogress)
        printf("Writing date index to \"%s\"...", filename);

    /*
    ** Print out the index file header 
    */
    print_index_header(fp, label, dir, "By Date");

    if (!usetable) {
        /* 
        ** Print out archive information links at the top of the index
        */ 
        print_index_header_links(fp, archives, about, FROM_DATE);
    
        if (showhr)
            fprintf(fp, "<HR>\n");

        /*
        ** Printout the Dates for the Starting and Ending messages 
        ** in the archive, along with a count of the messages.
        */

        fprintf(fp, "<STRONG>Starting:</STRONG> <EM>%s</EM><BR>\n", 
                getdatestr(firstdatenum));
        fprintf(fp, "<STRONG>Ending:</STRONG> <EM>%s</EM><BR>\n", 
                getdatestr(lastdatenum));
        fprintf(fp, "<STRONG>Messages:</STRONG> %d\n<P>\n", bignum + 1);
    }
    else {
        fprint_menu(fp, DATE_INDEX, about, archives, "", "", PAGE_TOP);
        fprint_summary(fp, firstdatenum, lastdatenum, bignum + 1);
        if (showhr)
            fprintf(fp, "<HR>\n");
    }

    /*
    ** Print out the actual message index lists. Here's the beef.
    */

    fprintf(fp, "<UL>\n");
    printdates(fp, datelist);
    fprintf(fp, "</UL>\n<P>\n");

    if (!usetable) {
        /*
        ** Printout the dates for the last message in the archive
        ** along with the date the archive was generated on.
        */

        fprintf(fp,"<A NAME=\"end\"><STRONG>Last message date:</STRONG></A> ");
        fprintf(fp,"<EM>%s</EM><BR>\n", getdatestr(lastdatenum));
        fprintf(fp,"<STRONG>Archived on:</STRONG> <EM>%s</EM><P>\n",getlocaltime());
        if (showhr)
            fprintf(fp, "<HR>\n");

        /* 
        ** Print out archive information links at the bottom of the index
        */ 
        print_index_footer_links(fp, archives, FROM_DATE);
    }
    else {
        fprint_menu(fp, DATE_INDEX, about, archives, "", "", PAGE_BOTTOM);
    }

    /* 
    ** Print the index page footer.
    */
    printfooter(fp, ihtmlfooter, label, dir, "By Date");

    fclose(fp);

    if (newfile && chmod(filename, filemode) == -1) {
        sprintf(errmsg, "Couldn't chmod \"%s\" to %o.", filename,
        filemode);
        progerr((char *) NULL);
    }

    if (showprogress)
        putchar('\n');
}

/*
** While printing the thread index, prints any replies to replies.
*/

void checkreplies(FILE *fp, int num, int level)
{
    struct reply *rp;

    for (rp = replylist; rp != NULL; rp = rp->next) {
        if (rp->msgnum == num) {
            if (level < thrdlevels)
                fprintf(fp, "<UL>\n");
            fprintf(fp,"<LI><A HREF=\"%.4d.html\">",rp->frommsgnum);
            printedlist = (struct printed *)
                           markasprinted(printedlist, rp->frommsgnum);
            fprintf(fp,"<STRONG>%s </STRONG></A> ",convchars(rp->subject));
            fprintf(fp, "<A NAME=\"%d\"><EM>%s</EM></A>\n",
                           rp->frommsgnum, rp->name);
            checkreplies(fp, rp->frommsgnum, level + 1);
            if (level < thrdlevels)
                fprintf(fp, "</UL>\n");
        }
    }
}

/*
** Prints the article pointers in the thread index by date.
*/

void printthreads(FILE *fp, struct header *hp)
{
    int hasreply;
    struct reply *rp;

    if (hp != NULL) {
        printthreads(fp, hp->left);

        for (hasreply = 0, rp = replylist; rp != NULL; rp = rp->next) {
            if (rp->frommsgnum == hp->msgnum) {
                hasreply = 1;
                break;
            }
        }

        if (!hasreply && !wasprinted(printedlist, hp->msgnum)) {
            fprintf(fp, "<LI><A HREF=\"%.4d.html\"><STRONG>%s</STRONG></A> ",
                    hp->msgnum, convchars(hp->subject));
            fprintf(fp,"<A NAME=\"%d\"><EM>%s</EM></A>\n",hp->msgnum,hp->name);
            printedlist = (struct printed *)
                    markasprinted(printedlist, hp->msgnum);

            checkreplies(fp, hp->msgnum, 1);
        }
        printthreads(fp, hp->right);
    }
}

/*
** Write the thread index...
*/

void writethreads(char *dir, char *label, char *archives, char *about)
{
    int newfile;
    char filename[MAXFILELEN];
    FILE *fp;

    struct printed *pp;

    while (printedlist) { /* cleanup needed ?? */
         pp = printedlist;
         printedlist = printedlist->next;
         free(pp);
    }

    printedlist = NULL;

    sprintf(filename, "%s%s%s", 
            dir, (dir[strlen(dir) - 1] == '/') ? "" : "/", thrdname);

    if (isfile(filename))
        newfile = 0;
    else
        newfile = 1;

    if ((fp = fopen(filename, "w")) == NULL) {
        sprintf(errmsg, "Couldn't write \"%s\".", filename);
        progerr((char *) NULL);
    }

    if (showprogress)
        printf("Writing thread index to \"%s\"...", filename);

    print_index_header(fp, label, dir, "By Thread");

    if (!usetable) {
        /* 
        ** Print out the index page links 
        */ 
        print_index_header_links(fp, archives, about, FROM_THREAD);
        
        if (showhr)
            fprintf(fp, "<HR>\n");
    
        fprintf(fp, "<STRONG>Starting:</STRONG> <EM>%s</EM><BR>\n",
                getdatestr(firstdatenum));
        fprintf(fp, "<STRONG>Ending:</STRONG> <EM>%s</EM><BR>\n", 
                getdatestr(lastdatenum));
        fprintf(fp, "<STRONG>Messages:</STRONG> %d\n<P>\n", bignum + 1);
    }
    else {
        fprint_menu(fp, THREAD_INDEX, about, archives, "", "", PAGE_TOP);
        fprint_summary(fp, firstdatenum, lastdatenum, bignum + 1);
        if (showhr)
            fprintf(fp, "<HR>\n");
    }

    fprintf(fp, "<UL>\n");
    printthreads(fp, datelist);
    fprintf(fp, "</UL>\n<P>\n");

    if (!usetable) {
        fprintf(fp, "<A NAME=\"end\"><STRONG>Last message date:</STRONG></A> ");
        fprintf(fp, "<EM>%s</EM><BR>\n", getdatestr(lastdatenum));
        fprintf(fp, "<STRONG>Archived on:</STRONG> <EM>%s</EM><P>\n", getlocaltime());
             
        if (showhr)
            fprintf(fp, "<HR>\n");
    
        /* 
        ** Print out archive information links at the bottom of the index
        */ 
        print_index_footer_links(fp, archives, FROM_THREAD);
    
    }
    else {
        fprint_menu(fp, THREAD_INDEX, about, archives, "", "", PAGE_BOTTOM);
    }

    printfooter(fp, ihtmlfooter, label, dir, "By Thread");

    fclose(fp);

    if (newfile && chmod(filename, filemode) == -1) {
        sprintf(errmsg, "Couldn't chmod \"%s\" to %o.", filename, filemode);
        progerr((char *) NULL);
    }

    if (showprogress)
        putchar('\n');
}

/*
** Print the subject index pointers alphabetically.
*/

void printsubjects(FILE *fp, struct header *hp)
{
    static char oldsubject[SUBJSTRLEN];

    if (hp != NULL) {
        printsubjects(fp, hp->left);
        if (strcasecmp(hp->subject, oldsubject))
            fprintf(fp, "<LI><STRONG>%s</STRONG>\n",convchars(hp->subject));
        fprintf(fp, "<UL>\n");
        fprintf(fp,"<LI><A HREF=\"%.4d.html\">%s</A> ",hp->msgnum,hp->name);
        fprintf(fp,"<A NAME=\"%d\"><EM>%s</EM></A>\n", hp->msgnum,hp->datestr);
        fprintf(fp, "</UL>\n");
        strcpy(oldsubject, hp->subject);
        printsubjects(fp, hp->right);
    }
}

/*
** Prints the subject index.
*/

void writesubjects(char *dir, char *label, char *archives, char *about)
{
    int newfile;
    char filename[MAXFILELEN];
    FILE *fp;

    sprintf(filename, "%s%s%s", 
            dir, (dir[strlen(dir) - 1] == '/') ? "" : "/", subjname);

    if (isfile(filename))
        newfile = 0;
    else
        newfile = 1;

    if ((fp = fopen(filename, "w")) == NULL) {
        sprintf(errmsg, "Couldn't write \"%s\".", filename);
        progerr((char *) NULL);
    }

    if (showprogress)
        printf("Writing subject index to \"%s\"...", filename);

    print_index_header(fp, label, dir, "By Subject");

    if (!usetable) {
        /* 
        ** Print out the index page links 
        */ 
        print_index_header_links(fp, archives, about, FROM_SUBJECT);
    
        if (showhr)
            fprintf(fp, "<HR>\n");

        fprintf(fp, "<STRONG>Starting:</STRONG> <EM>%s</EM><BR>\n",
                getdatestr(firstdatenum));
        fprintf(fp, "<STRONG>Ending:</STRONG> <EM>%s</EM><BR>\n", 
                getdatestr(lastdatenum));
        fprintf(fp, "<STRONG>Messages:</STRONG> %d\n<P>\n", bignum + 1);
    }
    else {
        fprint_menu(fp, SUBJECT_INDEX, about, archives, "", "", PAGE_TOP);
        fprint_summary(fp, firstdatenum, lastdatenum, bignum + 1);
        if (showhr)
            fprintf(fp, "<HR>\n");
    }

    fprintf(fp, "<UL>\n");
    printsubjects(fp, subjectlist);
    fprintf(fp, "</UL>\n<P>\n");

    if (!usetable) {
        fprintf(fp, "<STRONG>Last message date:</STRONG> <EM>%s</EM><BR>\n",
                getdatestr(lastdatenum));
        fprintf(fp, "<STRONG>Archived on:</STRONG> <EM>%s</EM><P>\n", 
                getlocaltime());

        if (showhr)
            fprintf(fp, "<HR>\n");

        /* 
        ** Print out archive information links at the bottom of the index
        */ 
        print_index_footer_links(fp, archives, FROM_SUBJECT);
    }
    else {
        fprint_menu(fp, SUBJECT_INDEX, about, archives, "", "", PAGE_BOTTOM);
    }

    printfooter(fp, ihtmlfooter, label, dir, "By Subject");

    fclose(fp);

    if (newfile && chmod(filename, filemode) == -1) {
        sprintf(errmsg, "Couldn't chmod \"%s\" to %o.", filename,
        filemode);
        progerr((char *) NULL);
    }

    if (showprogress)
        putchar('\n');
}

/*
** Prints the author index links sorted alphabetically.
*/

void printauthors(FILE *fp, struct header *hp)
{
    static char oldname[SUBJSTRLEN];

    if (hp != NULL) {
        printauthors(fp, hp->left);
        if (strcasecmp(hp->name, oldname))
            fprintf(fp, "<LI><STRONG>%s</STRONG>\n", hp->name);
        fprintf(fp,"<UL>\n");
        fprintf(fp,"<LI><A HREF=\"%.4d.html\">%s</A> ",
                     hp->msgnum, convchars(hp->subject));
        fprintf(fp,"<A NAME=\"%d\"><EM>%s</EM></A>\n", hp->msgnum,hp->datestr);
        fprintf(fp,"</UL>\n");
        strcpy(oldname, hp->name);
        printauthors(fp, hp->right);
    }
}

/*
** Prints the author index file and links sorted alphabetically.
*/

void writeauthors(char *dir, char *label, char *archives, char *about)
{
    int newfile;
    char filename[MAXFILELEN];
    FILE *fp;

    sprintf(filename, "%s%s%s", 
            dir, (dir[strlen(dir) - 1] == '/') ? "" : "/", authname);

    if (isfile(filename))
        newfile = 0;
    else
        newfile = 1;

    if ((fp = fopen(filename, "w")) == NULL) {
        sprintf(errmsg, "Couldn't write \"%s\".", filename);
        progerr((char *)NULL);
    }

    if (showprogress)
        printf("Writing author index to \"%s\"...", filename);

    print_index_header(fp, label, dir, "By Author");

    if (!usetable) {
        /* 
        ** Print out the index page links 
        */ 
        print_index_header_links(fp, archives, about, FROM_AUTHOR);
        
        if (showhr)
            fprintf(fp, "<HR>\n");
    
        fprintf(fp, "<STRONG>Starting:</STRONG> <EM>%s</EM><BR>\n",
                getdatestr(firstdatenum));
        fprintf(fp, "<STRONG>Ending:</STRONG> <EM>%s</EM><BR>\n", 
                getdatestr(lastdatenum));
        fprintf(fp, "<STRONG>Messages:</STRONG> %d\n<P>\n", bignum + 1);
    }
    else {
        fprint_menu(fp, AUTHOR_INDEX, about, archives, "", "", PAGE_TOP);
        fprint_summary(fp, firstdatenum, lastdatenum, bignum + 1);        
        if (showhr)
            fprintf(fp, "<HR>\n");
    }

    fprintf(fp, "<UL>\n");
    printauthors(fp, authorlist);
    fprintf(fp, "</UL>\n<P>\n");

    if (!usetable) {
        fprintf(fp, "<STRONG>Last message date:</STRONG> <EM>%s</EM><BR>\n",
                    getdatestr(lastdatenum));
        fprintf(fp,"<STRONG>Archived on:</STRONG> <EM>%s</EM><P>\n",getlocaltime());

        if (showhr)
            fprintf(fp, "<HR>\n");

        /* 
        ** Print out archive information links at the bottom of the index
        */ 
        print_index_footer_links(fp, archives, FROM_AUTHOR);
    }
    else {
        fprint_menu(fp, AUTHOR_INDEX, about, archives, "", "", PAGE_BOTTOM);
    }

    printfooter(fp, ihtmlfooter, label, dir, "By Author");

    fclose(fp);

    if (newfile && chmod(filename, filemode) == -1) {
        sprintf(errmsg, "Couldn't chmod \"%s\" to %o.", filename, filemode);
        progerr((char*)NULL);
    }

    if (showprogress)
        putchar('\n');
}
