/* -*-Mode: C;-*-
 * $Id: serializeio.c 1.17 Thu, 01 Oct 1998 04:56:35 -0700 jmacd $
 * ser.c:
 *
 * Copyright (C) 1998, Josh MacDonald.
 * All Rights Reserved.
 *
 * Author: Josh MacDonald <jmacd@CS.Berkeley.EDU>
 */

#include <stdlib.h>
#include <sys/param.h>  /* for ntohs, htons on some */
#include <netinet/in.h> /* on others */

#include "serializeio.h"

/* Sink methods
 */

gboolean
sink_next_uint16 (SerialSink* sink, guint16 num)
{
  num = htons (num);

  return sink->sink_write (sink, (guint8*) &num, sizeof (num));
}

gboolean
sink_next_uint32 (SerialSink* sink, guint32 num)
{
  num = htonl (num);

  return sink->sink_write (sink, (guint8*) &num, sizeof (num));
}

gboolean
sink_next_uint (SerialSink* sink, guint32 num)
{
  /* This is mostly because I dislike endian, and less to save space
   * on small ints.  However, the uint32 and uint16 functions are used
   * when the number is expected to be large, in which case this
   * format can expand the number. */

  guint8 sink_buf[16];  /* this is enough room for a 12-byte int */
  guint  sink_count = 0;

  do
    {
      guint left = num & 0x7f;
      guint outnum;

      num >>= 7;

      outnum = left | (num ? 0x80 : 0);

      sink_buf[sink_count++] = outnum;
    }
  while (num);

  return sink->sink_write (sink, sink_buf, sink_count);
}

gboolean
sink_next_uint8 (SerialSink* sink, guint8 val)
{
  return sink->sink_write (sink, &val, 1);
}

gboolean
sink_next_bool (SerialSink* sink, gboolean val)
{
  guint8 sink_buf[1];
  sink_buf[0] = val;
  return sink->sink_write (sink, sink_buf, 1);
}

gboolean
sink_next_string (SerialSink* sink, const char   *ptr)
{
  return sink->next_bytes (sink, ptr, strlen (ptr));
}

gboolean
sink_next_bytes (SerialSink* sink, const guint8   *ptr, guint32 len)
{
  return sink->next_uint (sink, len) &&
         sink->sink_write (sink, ptr, len);
}

gboolean
sink_next_bytes_known (SerialSink* sink, const guint8   *ptr, guint32 len)
{
  return sink->sink_write (sink, ptr, len);
}

/* Source methods
 */

gboolean
source_next_uint32 (SerialSource* source, guint32 *ptr)
{
  guint32 x;

  if (! source->source_read (source, (guint8*) &x, sizeof (x)))
    return FALSE;

  (*ptr) = ntohl (x);

  return TRUE;
}

gboolean
source_next_uint16 (SerialSource* source, guint16 *ptr)
{
  guint16 x;

  if (! source->source_read (source, (guint8*) &x, sizeof (x)))
    return FALSE;

  (*ptr) = ntohs (x);

  return TRUE;
}

gboolean
source_next_uint (SerialSource* source, guint32 *ptr)
{
  /* This is mostly because I dislike endian, and less to save space
   * on small ints */
  guint8 c;
  guint8 arr[16];
  gint i = 0;
  gint donebit = 1;
  gint bits;

  while (source->next_uint8 (source, &c))
    {
      donebit = c & 0x80;
      bits = c & 0x7f;

      arr[i++] = bits;

      if (!donebit)
	break;
    }

  if (donebit)
    return FALSE;

  *ptr = 0;

  for (i -= 1; i >= 0; i -= 1)
    {
      *ptr <<= 7;
      *ptr |= arr[i];
    }

  return TRUE;
}

gboolean
source_next_uint8 (SerialSource* source, guint8 *ptr)
{
  return source->source_read (source, ptr, 1);
}

gboolean
source_next_bool (SerialSource* source, gboolean *ptr)
{
  guint8 sink_buf[1];

  if (! source->source_read (source, sink_buf, 1))
    return FALSE;

  if (sink_buf[0])
    *ptr = TRUE;
  else
    *ptr = FALSE;

  return TRUE;
}

gboolean
source_next_string (SerialSource* source, const char **ptr)
{
  guint32 len;
  guint8* buf;

  if (! source->next_uint (source, &len))
    return FALSE;

  buf = source->source_alloc (source, len+1);

  buf[len] = 0;

  (*ptr) = buf;

  return source->source_read (source, buf, len);
}

gboolean
source_next_bytes (SerialSource* source, const guint8 **ptr, guint32 *len_ptr)
{
  guint32 len;
  guint8* buf;

  if (! source->next_uint (source, &len))
    return FALSE;

  buf = source->source_alloc (source, len);

  (*len_ptr) = len;
  (*ptr) = buf;

  return source->source_read (source, buf, len);
}

gboolean
source_next_bytes_known (SerialSource* source, guint8 *ptr, guint32 len)
{
  return source->source_read (source, ptr, len);
}

void
source_default_reset (SerialSource* source)
{
  source->alloc_pos = 0;

  if (source->alloc)
    {
      g_free (source->alloc);
      source->alloc = NULL;
    }
}

void*
source_default_alloc (SerialSource* source, guint32 len)
{
  void* ret;

  if (! source->alloc)
    {
      source->alloc = g_new (guint8, source->total_alloc + 8);

      ALIGN_8 ((long)source->alloc);
    }

  g_assert (len+source->alloc_pos <= source->total_alloc);

  ret = ((guint8*)source->alloc) + source->alloc_pos;

  source->alloc_pos += len;

  ALIGN_8 (source->alloc_pos);

  g_assert (((long)ret) % 8 == 0);
  g_assert (source->alloc_pos % 8 == 0);

  return ret;
}

typedef struct {
  SerialStatus (*func) ();
  guint32 val;
} SerEntry;

static GArray   *ser_array;
static gboolean  ser_array_sorted;

void
serializeio_initialize_type (guint32 val, SerialStatus (*func) ())
{
  SerEntry it;

  it.val = val;
  it.func = func;

  if (ser_array == NULL)
    ser_array = g_array_new (FALSE, FALSE, sizeof (SerEntry));

  g_array_append_val (ser_array, it);

  ser_array_sorted = FALSE;
}

static int
ser_entry_compare (const void* va, const void* vb)
{
  SerEntry* a = (SerEntry*) va;
  SerEntry* b = (SerEntry*) vb;

  return a->val - b->val;
}

SerialStatus
unserialize_generic_acceptable (SerialSource* source, guint32 accept, SerialType* object_type, void** object)
{
  SerialStatus s;

  s = unserialize_generic (source, object_type, object);

  if (s == SerialSuccess)
    {
      if (((*object_type) & SER_LIBRARY_OFFSET_MASK) != (accept & SER_LIBRARY_OFFSET_MASK))
	{
	  source->status = SerialIncorrectType;
	  return source->status;
	}

      if (! (((*object_type) & ~SER_LIBRARY_OFFSET_MASK) |
	     (accept         & ~SER_LIBRARY_OFFSET_MASK)))
	{
	  source->status = SerialIncorrectType;
	  return source->status;
	}

      return s;
    }
  else
    return s;
}

SerialStatus
unserialize_generic (SerialSource* source, SerialType* object_type, void** object)
{
  SerialType type = (* source->source_type) (source);
  gint high_index, low_index = 0, index;
  gboolean found;
  gboolean par = TRUE;
  gint count = 0;

  if (source->status != SerialSuccess)
    return source->status;

  high_index = ser_array->len;
  found = FALSE;

  if (! ser_array_sorted)
    {
      ser_array_sorted = TRUE;
      qsort (ser_array->data, ser_array->len, sizeof (SerEntry), ser_entry_compare);
    }

  /* Binary search. */
  for (;;)
    {
      gint this_val;

      index = low_index + (high_index-low_index+par)/2;

      par = !par;

      this_val = g_array_index (ser_array, SerEntry, index).val;

      if (this_val == type)
	{
	  found=TRUE;
	  (* object_type) = type;
	  (* g_array_index (ser_array, SerEntry, index).func) (source, object);
	  break;
	}

      if (low_index == high_index-1 && count++ == 1)
	break;

      if (this_val < type)
	low_index = index;
      else
	high_index = index;
    }

  if (! found)
    source->status = SerialIncorrectType;

  if (source->status != SerialSuccess)
    (*object) = NULL;
  else
    {
      source->alloc = NULL;
      (* source->source_reset) (source);
    }

  return source->status;
}

void
ser_source_init (SerialSource* it)
{
  it->next_bytes_known = source_next_bytes_known;
  it->next_bytes = source_next_bytes;
  it->next_uint = source_next_uint;
  it->next_uint32 = source_next_uint32;
  it->next_uint16 = source_next_uint16;
  it->next_uint8 = source_next_uint8;
  it->next_bool = source_next_bool;
  it->next_string = source_next_string;
  it->source_alloc = source_default_alloc;
}

void
ser_sink_init (SerialSink* it)
{
  it->next_bytes_known = sink_next_bytes_known;
  it->next_bytes = sink_next_bytes;
  it->next_uint = sink_next_uint;
  it->next_uint32 = sink_next_uint32;
  it->next_uint16 = sink_next_uint16;
  it->next_uint8 = sink_next_uint8;
  it->next_bool = sink_next_bool;
  it->next_string = sink_next_string;
}


/* Segment source/sink impls
 */
typedef struct _HandleSerialSource HandleSerialSource;
typedef struct _HandleSerialSink HandleSerialSink;

struct _HandleSerialSource {
  SerialSource source;

  FileHandle* fh;
};

struct _HandleSerialSink {
  SerialSink sink;

  FileHandle* fh;
};

static SerialType handle_source_type           (SerialSource* source);
static gboolean   handle_source_close          (SerialSource* source);
static gboolean   handle_source_read           (SerialSource* source, guint8 *ptr, guint32 len);
static void       handle_source_free           (SerialSource* source);

static gboolean   handle_sink_type             (SerialSink* sink, SerialType type, guint len);
static gboolean   handle_sink_close            (SerialSink* sink);
static gboolean   handle_sink_write            (SerialSink* sink, const guint8 *ptr, guint32 len);
static void       handle_sink_free             (SerialSink* sink);
static SerialStatus handle_sink_quantum        (SerialSink* sink);

static SerialSource* handle_source_int (FileHandle* fh, const HandleFuncTable* table, guint size);
static SerialSink* handle_sink_int (FileHandle* fh, const HandleFuncTable* table, guint size);

void
handle_source_reset (SerialSource* source)
{
  source_default_reset (source);
}

SerialSource*
serializeio_handle_source (FileHandle* fh, const HandleFuncTable* table)
{
  return handle_source_int (fh, table, sizeof (HandleSerialSource));
}

SerialSink*
serializeio_handle_sink (FileHandle* fh, const HandleFuncTable* table)
{
  return handle_sink_int (fh, table, sizeof (HandleSerialSink));
}

SerialSource*
handle_source_int (FileHandle* fh, const HandleFuncTable* table, guint size)
{
  HandleSerialSource* it = g_malloc0(size);

  ser_source_init (&it->source);

  it->fh = fh;

  it->source.table = table;

  it->source.source_type = handle_source_type;
  it->source.source_close = handle_source_close;
  it->source.source_read = handle_source_read;
  it->source.source_free = handle_source_free;
  it->source.source_reset = handle_source_reset;

  return &it->source;
}

SerialSink*
handle_sink_int (FileHandle* fh, const HandleFuncTable* table, guint size)
{
  HandleSerialSink* it = g_malloc0(size);

  ser_sink_init (&it->sink);

  it->fh = fh;

  it->sink.table = table;

  it->sink.sink_type = handle_sink_type;
  it->sink.sink_close = handle_sink_close;
  it->sink.sink_write = handle_sink_write;
  it->sink.sink_free = handle_sink_free;
  it->sink.sink_quantum = handle_sink_quantum;

  return &it->sink;
}

SerialStatus
handle_sink_quantum (SerialSink* sink)
{
  return SerialSuccess;
}

SerialType
handle_source_type (SerialSource* source)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;
  guint x;

  handle_source_reset (source);

  if (! source->table->table_handle_getui (ssource->fh, &x))
    return ST_Error;

  if (! source->table->table_handle_getui (ssource->fh, &source->total_alloc))
    return ST_Error;

  return x;
}

gboolean
handle_source_close (SerialSource* source)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;

  return source->table->table_handle_close (ssource->fh, 0);
}

gboolean
handle_source_read (SerialSource* source, guint8 *ptr, guint32 len)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;

  gboolean it = source->table->table_handle_read (ssource->fh, ptr, len);

  if (! it)
    {
      handle_source_reset (source);
      source->status = SerialFailure;
    }

  return it;
}

void
handle_source_free (SerialSource* source)
{
  HandleSerialSource* ssource = (HandleSerialSource*) source;

  handle_source_reset (source);

  g_free (ssource);
}

gboolean
handle_sink_type (SerialSink* sink, SerialType type, guint len)
{
  HandleSerialSink* ssink = (HandleSerialSink*) sink;

  if (! sink->table->table_handle_putui (ssink->fh, type) ||
      ! sink->table->table_handle_putui (ssink->fh, len))
    {
      sink->status = SerialFailure;
      return FALSE;
    }

  return TRUE;
}

gboolean
handle_sink_close (SerialSink* sink)
{
  HandleSerialSink* ssink = (HandleSerialSink*) sink;

  if (! sink->table->table_handle_close (ssink->fh, 0))
    {
      sink->status = SerialFailure;
      return FALSE;
    }

  return TRUE;
}

gboolean
handle_sink_write (SerialSink* sink, const guint8 *ptr, guint32 len)
{
  HandleSerialSink* ssink = (HandleSerialSink*) sink;

  if (! sink->table->table_handle_write (ssink->fh, ptr, len))
    {
      sink->status = SerialFailure;
      return FALSE;
    }

  return TRUE;
}

void
handle_sink_free (SerialSink* sink)
{
  g_free (sink);
}

#ifdef IN_REPO
#include "repo.h"

/* File source/sink impls
 */
typedef struct _FileSerialSource FileSerialSource;
typedef struct _FileSerialSink FileSerialSink;

struct _FileSerialSource {
  HandleSerialSource source;

  gboolean (* real_close) (SerialSource* sink);
  FileHandle* handle;
};

struct _FileSerialSink {
  HandleSerialSink sink;

  gboolean (* real_close) (SerialSink* sink);
  FileHandle* handle;
};

static gboolean
file_sink_close (SerialSink* sink)
{
  FileSerialSink* ssink = (FileSerialSink*) sink;

  gboolean res1 = ssink->real_close (sink);
  gboolean res2 = handle_close (ssink->handle, 0);

  return res1 && res2;
}

static gboolean
file_source_close (SerialSource* source)
{
  FileSerialSource* ssource = (FileSerialSource*) source;

  gboolean res1 = ssource->real_close (source);
  gboolean res2 = handle_close (ssource->handle, 0);

  return res1 && res2;
}

SerialSink*
file_sink       (File *file)
{
  FileHandle* handle = file_open (file, HV_Replace);
  FileSerialSink* sink;

  if (! handle)
    return NULL;

  sink = (FileSerialSink*) handle_sink_int (handle, &repo_handle_table, sizeof(FileSerialSink));

  sink->real_close = sink->sink.sink.sink_close;
  sink->handle = handle;
  sink->sink.sink.sink_close = file_sink_close;

  return &sink->sink.sink;
}

SerialSource*
file_source     (File *file)
{
  FileHandle* handle = file_open (file, HV_Replace);
  FileSerialSource* source;

  if (! handle)
    return NULL;

  source = (FileSerialSource*) handle_source_int (handle, &repo_handle_table, sizeof(FileSerialSource));

  source->real_close = source->source.source.source_close;
  source->handle = handle;
  source->source.source.source_close = file_source_close;

  return &source->source.source;
}
#endif
