/*
 Selects rows from an rdbtable.

 Author: Carlo Strozzi <carlos@linux.it>
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define MAX_ESCAPE_LENGTH       256
#define MAX_ARG_LENGTH          1024
#define EMPTY                   ""

void    show_help( char *my_name) {
  printf("
        NoSQL operator: %s
    
Usage:  %s  [options] [expression]

Options:
    -help    Print this help info.
    -n       Strip header from output.
    -x       Debug option.
    -e \"abc\" Escape characters in set [a,b,c] by prepending a
             backslash to each of them on STDOUT.
    
Selects rows from the input rdbtable that satisfy an arbitrary AWK
expression using column names.  Chars that are special to the UNIX
shell must be quoted.
    
Column names in the AWK expression must be referred to with the construct
'$P[\"column_name\"]'. For example, to select all rows that have the \"NAME\"
column equal to \"Hobbs\" the selection expression is:
    
                    '$P[\"NAME\"] == \"Hobbs\"' 
    
Likewise, more complex expressions can be used:\n
    
         '$P[\"NAME\"] == \"Hobbs\" && $P[\"AGE\"] ~ /^2[0-9]$/'
    
Ordinary AWK field numbers may be used in expressions, i.e.:
    
           '$(3) == \"Hobbs\" || $(0) ~ /abcd/', and so on.
    
Any other valid AWK statement can be used, like 'NR == 5' , etc.
    
Long expessions may be folded over multiple lines by ending each
line with a backslash, i.e.:

    '$P[\"NAME\"] == \"Hobbs\" && $P[\"JOB\"] == \"Programmer\" %s
     && $P[\"AGE\"] ~ /^2[0-9]$/'

This operator reads an rdbtable via STDIN and writes an rdbtable
via STDOUT.  If no selection expression is provided, then the whole
table is written to STDOUT.
    
    
'$Id: nsq-frow.c,v 1.1 1998/05/29 20:43:01 carlos Exp $'
    
            ----------------------
NoSQL RDBMS, Copyright (C) 1998 Carlo Strozzi.
This program comes with ABSOLUTELY NO WARRANTY; for details
refer to the GNU General Public License.
    
You should have received a copy of the GNU General Public License
along with this program;  if not, write to the Free Software
Foundation, Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307
USA.
            ----------------------", my_name, my_name, "\\");
}

int main( int  argc, char *argv[] ) {

  /* For getopt() */
  extern char* optarg;
  extern int optind;

  /* for the rest of the program. */
  register int
    a_loop;
  char
    *my_name=argv[0],
    e_set[MAX_ESCAPE_LENGTH] = EMPTY,
    awk_expr[MAX_ARG_LENGTH] = EMPTY,
    cmd_buf[MAX_COMMAND_LENGTH];

  int no_hdr=0, debug=0;

  while ((a_loop = getopt(argc, argv, "nhxe:")) != EOF) {
    switch (a_loop) {
      case 'h':
        show_help(my_name);
        exit(0);
        break;
      case 'n': 
        no_hdr=1;
        break;
      case 'x': 
        debug=1;
        break;
      case 'e':
        snprintf(e_set,MAX_ESCAPE_LENGTH,"%s", optarg);
        break;
      default:
        show_help(my_name);
        exit(1);
    }
  }

  if( optind < argc ) 
    snprintf(awk_expr,MAX_ARG_LENGTH,"%s", argv[optind++]);
  else
    /* No AWK expression was specified. */
    snprintf(awk_expr,MAX_ARG_LENGTH,"%s", "");

  snprintf(cmd_buf,MAX_COMMAND_LENGTH,"#
#
BEGIN{
  NULL=\"\"
  FS=\"\\t\"; OFS = FS;
  split( \"%s\", e_set, NULL)
  # Honour the 'debug' switch.
  if( %d ) {
    arg_vec = \"# ARGC = \" ARGC

    for( arg in ARGV )
      arg_vec = arg_vec \"\\n# ARGV[\" arg \"] = \" ARGV[arg]

    print arg_vec > \"/dev/stderr\"
  }
}
# Table comments.
r == 0 && $0 ~ /^ *#/ { if( ! %d ) print; next }
# Column names and positions.
r == 0 {
  while( ++p <= NF ) {
    # Make sure we pick the first occurrence of duplicated column
    # names (it may happen after a join).
    if( P[$p] == NULL ) {
      P[$p] = p
      N[p] = $p
    }
  }
  r++
  if( ! %d ) print
  next
}
# Column definitions.
r == 1 { if( ! %d ) print; NR=0; r++; next }
%s {
  # Apply any requested escaping
  for( e in e_set ) gsub( e_set[e], \"\\\\\\\\\"e_set[e], $0 )
  print }", e_set, debug, no_hdr, no_hdr, no_hdr, awk_expr);

  if( debug )
    fprintf (stderr, "Generated AWK program:
    ----------\n%s\n----------\n",cmd_buf);

  execlp(AWK,"awk",cmd_buf,NULL);
  exit(0);
}
