/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <math.h>

#ifdef linux
#include <endian.h>
#elif defined (FreeBSD)
#include <machine/endian.h>
#elif defined (sgi)
#include <sys/endian.h>
#elif defined (sun)
#include <sys/byteorder.h>
#endif

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Scrollbar.h>

#include "types.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "misc.h"
#include "status.h"
#include "button.h"
#include "menu.h"
#include "graphics.h"
#include "audio_file.h"
#include "endian.h"

extern Main_Data *MD;
extern AppResources app_resources;

static void update_display_mem(Main_Data*,Display*,Window);
static void update_display_file(Main_Data*,Display*,Window);
static void update_canvas_mem(Next_Wave*,Display*,Window,Dimension,int);
static void update_canvas_mem_1(Next_Wave*,Display*,Window,Dimension,int);
static void update_canvas_mem_lines(Next_Wave*,Display*,Window,Dimension,int);
static void update_canvas_file(Next_Wave*,Display*,Window,Dimension,int);
static void update_canvas_file_1(Next_Wave*,Display*,Window,Dimension,int);
static void update_canvas_file_lines(Next_Wave*,Display*,Window,Dimension,int);
static void set_canvas_mark(Next_Wave *nw);
static Boolean test_focus(Main_Data*,Widget,Widget);
static Next_Wave *get_nw(Main_Data*,Widget);

static int firstmark=-1;
static bool resizing=FALSE;
static bool line_saved=FALSE;

void update_display(Main_Data *md)
{
    Window win = XtWindow(md->mw->graph); /* get window ID */
    Display *dpy=XtDisplay(md->mw->graph);
    
    update_status(md);
    XClearArea(dpy, win, 0, 0, 0, 0, False);  /* clear display with bg-color */
    
    if (md->wd==NULL) return;
    if ((md->wd->inmem) && (md->wd->buffer==NULL)) return;
    
    if (md->wd->inmem) update_display_mem(md,dpy,win);
    else {
	update_display_file(md,dpy,win);
    }
    if (md->wd->isplay) set_line(md->wd->markbeg/md->mg->step);
    set_whole_mark(md);
    XFlush(dpy);
}

void update_display_mem(Main_Data *md,Display *dpy,Window win)
{
    int i,j,k,l,length,height;
    Dimension width;				/* dimension of display */  
    byte *data=NULL,y;			        /* to go through the data */
    char *data16=NULL;
    byte base;
    byte *lines_l;
    byte *lines_r=NULL;
    float step,step_add;
    
    XtVaGetValues(md->mw->graph,XtNwidth, &width, NULL);
    
    if ((!md->cb->update)&&(width==md->mg->width)) {
	/* draw old lines */
	if (md->wd->lines_l!=NULL) {
	    switch (md->wd->channels) {
	     case MONO:
		for (i=0,j=0;i<width;i++,j+=2)
		  XDrawLine(dpy,win,md->mg->gcl,i,md->wd->lines_l[j],i,
			    md->wd->lines_l[j+1]);
		break;
	     case STEREO:
		for (i=0,j=0;i<width;i++,j+=2) {
		    XDrawLine(dpy,win,md->mg->gcl,i,md->wd->lines_l[j],i,
			      md->wd->lines_l[j+1]);
		    XDrawLine(dpy,win,md->mg->gcr,i,md->wd->lines_r[j],i,
			      md->wd->lines_r[j+1]);
		}
		break;
	    }
	    return;
	} else {
	    if ((md->wd->lines_l=malloc(2*width))==NULL) return;
	    switch(md->wd->channels) {
	     case STEREO:
		if ((md->wd->lines_r=malloc(2*width))==NULL) return;
	    }
	}
    } else {
	/* something has changed, rescan data */
	if (md->wd->lines_l!=NULL) free(md->wd->lines_l);
	if ((md->wd->lines_l=malloc(2*width))==NULL) return;
	switch(md->wd->channels) {
	 case STEREO:
	    if (md->wd->lines_r!=NULL) free(md->wd->lines_r);
	    if ((md->wd->lines_r=malloc(2*width))==NULL) return;
	}
	md->cb->update=False;
    }
    
    lines_l=md->wd->lines_l;
    for (i=0;i<2*width;i+=2) {
	lines_l[i]=255;lines_l[i+1]=0;
    }
    switch(md->wd->channels) {
     case STEREO:
	lines_r=md->wd->lines_r;
	for (i=0;i<2*width;i+=2) {
	    lines_r[i]=255;lines_r[i+1]=0;
	}
	break;
    }
    
    md->mg->width=width;
    
    length=md->wd->length;
    height=PREVIEWHEIGHT-1;
    
    step_add=(float)md->wd->tlength/(float)width;
    md->mg->step=step_add;
    
    if (md->wd->res == 8) {
	data=(byte*)md->wd->buffer;
	
	if (md->wd->channels==MONO) {
	    for (i=0,j=0,k=0,step=step_add;i<length;i++) {
		y=data[i];
		if (y<lines_l[j]) lines_l[j]=y;
		if (y>lines_l[j+1]) lines_l[j+1]=y;
		if ((float)i>step) {
		    lines_l[j]=height-lines_l[j];
		    lines_l[j+1]=height-lines_l[j+1];
		    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
		    step+=step_add;
		    k++;
		    j+=2;
		}
	    }
	    lines_l[j]=height-lines_l[j];
	    lines_l[j+1]=height-lines_l[j+1];
	    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
	} else {
	    base=height/2;
	    for (i=0,j=0,k=0,l=0,step=step_add;l<length;i++) {
		y=data[l];
		l++;
		if (y<lines_l[j]) lines_l[j]=y;
		if (y>lines_l[j+1]) lines_l[j+1]=y;
		y=data[l];
		l++;
		if (y<lines_r[j]) lines_r[j]=y;
		if (y>lines_r[j+1]) lines_r[j+1]=y;
		if ((float)i>step) {
		    lines_l[j]>>=1;lines_l[j+1]>>=1;
		    lines_l[j]=base-lines_l[j];
		    lines_l[j+1]=base-lines_l[j+1];
		    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
		    lines_r[j]>>=1;lines_r[j+1]>>=1;
		    lines_r[j]=base-lines_r[j];
		    lines_r[j+1]=base-lines_r[j+1];
		    lines_r[j]=PREVIEWHEIGHT/2+lines_r[j];
		    lines_r[j+1]=PREVIEWHEIGHT/2+lines_r[j+1];
		    XDrawLine(dpy,win,md->mg->gcr,k,lines_r[j],k,lines_r[j+1]);
		    step+=step_add;
		    k++;
		    j+=2;
		}
	    }
	    lines_l[j]>>=1;lines_l[j+1]>>=1;
	    lines_l[j]=base-lines_l[j];
	    lines_l[j+1]=base-lines_l[j+1];
	    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
	    lines_r[j]>>=1;lines_r[j+1]>>=1;
	    lines_r[j]=base-lines_r[j];
	    lines_r[j+1]=base-lines_r[j+1];
	    lines_r[j]=PREVIEWHEIGHT/2+lines_r[j];
	    lines_r[j+1]=PREVIEWHEIGHT/2+lines_r[j+1];
	    XDrawLine(dpy,win,md->mg->gcr,k,lines_r[j],k,lines_r[j+1]);
	}  
    } else {
	
	data16=(char*)md->wd->buffer;
	
#ifdef little_endian
	l=1;
#else
	l=0;
#endif
	if (md->wd->channels==MONO) {
	    base=PREVIEWHEIGHT/2;
	    for (i=0,j=0,k=0,step=step_add;l<length;i++,l+=2) {
		y=data16[l]+base;
		if (y<lines_l[j]) lines_l[j]=y;
		if (y>lines_l[j+1]) lines_l[j+1]=y;
		if ((float)i>step) {
		    lines_l[j]=height-lines_l[j];
		    lines_l[j+1]=height-lines_l[j+1];
		    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
		    step+=step_add;
		    k++;
		    j+=2;
		}
	    }
	    lines_l[j]=height-lines_l[j];
	    lines_l[j+1]=height-lines_l[j+1];
	    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
	} else { 
	    base=height/2;
	    for (i=0,j=0,k=0,step=step_add;l<length;i++) {
		y=data16[l]+PREVIEWHEIGHT/2;
		l+=2;
		if (y<lines_l[j]) lines_l[j]=y;
		if (y>lines_l[j+1]) lines_l[j+1]=y;
		y=data16[l]+PREVIEWHEIGHT/2;
		l+=2;
		if (y<lines_r[j]) lines_r[j]=y;
		if (y>lines_r[j+1]) lines_r[j+1]=y;
		if ((float)i>step) {
		    lines_l[j]>>=1;lines_l[j+1]>>=1;
		    lines_l[j]=base-lines_l[j];
		    lines_l[j+1]=base-lines_l[j+1];
		    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
		    lines_r[j]>>=1;lines_r[j+1]>>=1;
		    lines_r[j]=base-lines_r[j];
		    lines_r[j+1]=base-lines_r[j+1];
		    lines_r[j]=PREVIEWHEIGHT/2+lines_r[j];
		    lines_r[j+1]=PREVIEWHEIGHT/2+lines_r[j+1];
		    XDrawLine(dpy,win,md->mg->gcr,k,lines_r[j],k,lines_r[j+1]);
		    step+=step_add;
		    k++;
		    j+=2;
		}
	    }
	    lines_l[j]>>=1;lines_l[j+1]>>=1;
	    lines_l[j]=base-lines_l[j];
	    lines_l[j+1]=base-lines_l[j+1];
	    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
	    lines_r[j]>>=1;lines_r[j+1]>>=1;
	    lines_r[j]=base-lines_r[j];
	    lines_r[j+1]=base-lines_r[j+1];
	    lines_r[j]=PREVIEWHEIGHT/2+lines_r[j];
	    lines_r[j+1]=PREVIEWHEIGHT/2+lines_r[j+1];
	    XDrawLine(dpy,win,md->mg->gcr,k,lines_r[j],k,lines_r[j+1]);
	}  
    }
}

void update_display_file(Main_Data *md,Display *dpy,Window win)
{
    int i,j,k,l,length,height;
    Dimension width;				/* dimension of display */  
    byte *data=NULL,y;			        /* to go through the data */
    char *data16=NULL;
    byte base;
    Audio_File af;
    byte *lines_l;
    byte *lines_r=NULL;
    float step,step_add;
    
    wd2af(md->wd,&af);
    if (af_rewind(af)==AF_ERROR) return;
    
    XtVaGetValues(md->mw->graph,XtNwidth, &width, NULL);
    
    if ((!md->cb->update)&&(width==md->mg->width)) {
	/* draw old lines */
	if (md->wd->lines_l!=NULL) {
	    switch (md->wd->channels) {
	     case MONO:
		for (i=0,j=0;i<width;i++,j+=2)
		  XDrawLine(dpy,win,md->mg->gcl,i,md->wd->lines_l[j],i,
			    md->wd->lines_l[j+1]);
		break;
	     case STEREO:
		for (i=0,j=0;i<width;i++,j+=2) {
		    XDrawLine(dpy,win,md->mg->gcl,i,md->wd->lines_l[j],i,
			      md->wd->lines_l[j+1]);
		    XDrawLine(dpy,win,md->mg->gcr,i,md->wd->lines_r[j],i,
			      md->wd->lines_r[j+1]);
		}
		break;
	    }
	    return;
	} else {
	    if ((md->wd->lines_l=malloc(2*width))==NULL) return;
	    switch(md->wd->channels) {
	     case STEREO:
		if ((md->wd->lines_r=malloc(2*width))==NULL) return;
	    }
	}
    } else {
	/* something has changed, rescan data */
	if (md->wd->lines_l!=NULL) free(md->wd->lines_l);
	if ((md->wd->lines_l=malloc(2*width))==NULL) return;
	switch(md->wd->channels) {
	 case STEREO:
	    if (md->wd->lines_r!=NULL) free(md->wd->lines_r);
	    if ((md->wd->lines_r=malloc(2*width))==NULL) return;
	}
	md->cb->update=False;
    }
    
    lines_l=md->wd->lines_l;
    for (i=0;i<2*width;i+=2) {
	lines_l[i]=255;lines_l[i+1]=0;
    }
    switch(md->wd->channels) {
     case STEREO:
	lines_r=md->wd->lines_r;
	for (i=0;i<2*width;i+=2) {
	    lines_r[i]=255;lines_r[i+1]=0;
	}
	break;
    }
    
    md->mg->width=width;
    
    length=md->wd->tlength;
    height=PREVIEWHEIGHT-1;
    base=PREVIEWHEIGHT/2;
    
    step_add=(float)md->wd->tlength/(float)width;
    md->mg->step=step_add;
    
    if (md->wd->res == 8) {
	
	data=(byte*)md->mg->fbuf;
	if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
	
	if (md->wd->channels==MONO) {
	    for (i=0,j=0,k=0,l=0,step=step_add;i<length;i++,l++) {
		/* l should really equal MAXUSHORT */
		if (l>=MAXUSHORT) {
		    if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
		    l=0;
		}
		y=data[l];
		if (y<lines_l[j]) lines_l[j]=y;
		if (y>lines_l[j+1]) lines_l[j+1]=y;
		if ((float)i>step) {
		    lines_l[j]=height-lines_l[j];
		    lines_l[j+1]=height-lines_l[j+1];
		    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
		    step+=step_add;
		    k++;
		    j+=2;
		}
	    }
	    lines_l[j]=height-lines_l[j];
	    lines_l[j+1]=height-lines_l[j+1];
	    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
	} else { 
	    base=height/2;
	    for (i=0,j=0,k=0,l=0,step=step_add;i<length;i++) {
		if (l>=MAXUSHORT) {
		    if (af_read(af,(char*) data,MAXUSHORT)==AF_ERROR) return;
		    l=0;
		}
		y=data[l];
		l++;
		if (y<lines_l[j]) lines_l[j]=y;
		if (y>lines_l[j+1]) lines_l[j+1]=y;
		y=data[l];
		l++;
		if (y<lines_r[j]) lines_r[j]=y;
		if (y>lines_r[j+1]) lines_r[j+1]=y;
		if ((float)i>step) {
		    lines_l[j]>>=1;lines_l[j+1]>>=1;
		    lines_l[j]=base-lines_l[j];
		    lines_l[j+1]=base-lines_l[j+1];
		    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
		    lines_r[j]>>=1;lines_r[j+1]>>=1;
		    lines_r[j]=base-lines_r[j];
		    lines_r[j+1]=base-lines_r[j+1];
		    lines_r[j]=PREVIEWHEIGHT/2+lines_r[j];
		    lines_r[j+1]=PREVIEWHEIGHT/2+lines_r[j+1];
		    XDrawLine(dpy,win,md->mg->gcr,k,lines_r[j],k,lines_r[j+1]);
		    step+=step_add;
		    k++;
		    j+=2;
		}
	    }
	    lines_l[j]>>=1;lines_l[j+1]>>=1;
	    lines_l[j]=base-lines_l[j];
	    lines_l[j+1]=base-lines_l[j+1];
	    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
	    lines_r[j]>>=1;lines_r[j+1]>>=1;
	    lines_r[j]=base-lines_r[j];
	    lines_r[j+1]=base-lines_r[j+1];
	    lines_r[j]=PREVIEWHEIGHT/2+lines_r[j];
	    lines_r[j+1]=PREVIEWHEIGHT/2+lines_r[j+1];
	    XDrawLine(dpy,win,md->mg->gcr,k,lines_r[j],k,lines_r[j+1]);
	}  
    } else {
	data16=(char*)md->mg->fbuf;
	if (af_read(af,data16,MAXUSHORT)==AF_ERROR) return;
#ifdef little_endian
	l=1;
#else
	l=0;
#endif
	if (md->wd->channels==MONO) { 
	    for (i=0,j=0,k=0,step=step_add;i<length;i++,l+=2) {
		if (l>=MAXUSHORT) {
		    if (af_read(af,data16,MAXUSHORT)==AF_ERROR) return;
#ifdef little_endian
		    l=1;
#else
		    l=0;
#endif
		}
		y=data16[l]+base;
		if (y<lines_l[j]) lines_l[j]=y;
		if (y>lines_l[j+1]) lines_l[j+1]=y;
		if ((float)i>step) {
		    lines_l[j]=height-lines_l[j];
		    lines_l[j+1]=height-lines_l[j+1];
		    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
		    step+=step_add;
		    k++;
		    j+=2;
		}
	    }
	    lines_l[j]=height-lines_l[j];
	    lines_l[j+1]=height-lines_l[j+1];
	    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
	} else { 
	    base=height/2;
	    for (i=0,j=0,k=0,step=step_add;i<length;i++) {
		if (l>=MAXUSHORT) {
		    if (af_read(af,data16,MAXUSHORT)==AF_ERROR) return;
#ifdef little_endian
		    l=1;
#else
		    l=0;
#endif
		}
		y=data16[l]+PREVIEWHEIGHT/2;
		l+=2;
		if (y<lines_l[j]) lines_l[j]=y;
		if (y>lines_l[j+1]) lines_l[j+1]=y;
		y=data16[l]+PREVIEWHEIGHT/2;
		l+=2;
		if (y<lines_r[j]) lines_r[j]=y;
		if (y>lines_r[j+1]) lines_r[j+1]=y;
		if ((float)i>step) {
		    lines_l[j]>>=1;lines_l[j+1]>>=1;
		    lines_l[j]=base-lines_l[j];
		    lines_l[j+1]=base-lines_l[j+1];
		    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
		    lines_r[j]>>=1;lines_r[j+1]>>=1;
		    lines_r[j]=base-lines_r[j];
		    lines_r[j+1]=base-lines_r[j+1];
		    lines_r[j]=PREVIEWHEIGHT/2+lines_r[j];
		    lines_r[j+1]=PREVIEWHEIGHT/2+lines_r[j+1];
		    XDrawLine(dpy,win,md->mg->gcr,k,lines_r[j],k,lines_r[j+1]);
		    step+=step_add;
		    k++;
		    j+=2;
		}
	    }
	    lines_l[j]>>=1;lines_l[j+1]>>=1;
	    lines_l[j]=base-lines_l[j];
	    lines_l[j+1]=base-lines_l[j+1];
	    XDrawLine(dpy,win,md->mg->gcl,k,lines_l[j],k,lines_l[j+1]);
	    lines_r[j]>>=1;lines_r[j+1]>>=1;
	    lines_r[j]=base-lines_r[j];
	    lines_r[j+1]=base-lines_r[j+1];
	    lines_r[j]=PREVIEWHEIGHT/2+lines_r[j];
	    lines_r[j+1]=PREVIEWHEIGHT/2+lines_r[j+1];
	    XDrawLine(dpy,win,md->mg->gcr,k,lines_r[j],k,lines_r[j+1]);
	}  
    }
}

void update_canvas(Main_Data *md,Next_Wave *nw)
{
    Window win;
    Display *dpy;
    Dimension width,height;	
    int length=0;
    float top,shown;					

    /* if we get no nw take the one who is actual in main window */
    if (nw==NULL) {
	nw=md->nw;
	while(md->wd!=nw->wd) nw=nw->nw;
    }
    
    if (nw->wd->inmem) if (nw->wd->buffer==NULL) return;
    
    win = XtWindow(nw->cw->graph);
    dpy = XtDisplay(nw->cw->graph);
    
    XtVaGetValues(nw->cw->graph,XtNwidth,&width,XtNheight,&height,NULL);
    
    length=nw->wd->tlength;
    
    if (nw->cg->pos<0) nw->cg->pos=0;
    
    if ((length/width+1)<nw->cg->step) {
        nw->cg->step=length/width+1;
        set_zoom_label(nw->cg->step);
    }
    
    length=width*nw->cg->step;
    if (nw->cg->pos+length*nw->wd->bpspl>nw->wd->length) { 
	length=(nw->wd->length-nw->cg->pos)/nw->wd->bpspl;
    }
    
    if (nw->cg->pos>=(nw->wd->tlength-(width/2*nw->cg->step))*nw->wd->bpspl) {
	nw->cg->pos=(nw->wd->tlength-(width/2*nw->cg->step))*nw->wd->bpspl;
	length=(width/2)*nw->cg->step;
    } 
    
    nw->cg->width=width;
    /* clear display with bg-color */
    XClearArea(dpy, win, 0, 0, 0, 0, False);     
    
    if (nw->wd->inmem) update_canvas_mem(nw,dpy,win,height,length);
    else {
	update_canvas_file(nw,dpy,win,height,length);
    }
    
    /* set the position scrollbar of canvas */
    top=(float)nw->cg->pos/(float)nw->wd->length;
    shown=((float)width*(float)nw->cg->step)/(float)nw->wd->tlength;
    XawScrollbarSetThumb(nw->cw->scroll,top,shown);
    
    /* set the zoom scrollbar of canvas */
    shown=((float)width/(float)nw->wd->tlength);
    top=1.0-shown*(float)nw->cg->step;
    if (top<0) top=0.0;
    XawScrollbarSetThumb(nw->cw->zscroll,top,shown);
    set_canvas_mark(nw);
    
    XFlush(dpy);
}

void update_canvas_mem(Next_Wave *nw,Display *dpy,Window win,
		       Dimension height,int length)
{
    XPoint *pointsl=NULL,*pointsr=NULL;					
    byte *ispointl=NULL,*ispointr=NULL;		/* for test, if we should set
						 a point at this x-coord */
    int i,j,k,l=0,m=0,y=0;
    byte *data=NULL;				/* to go through the data */
    ushort *data16=NULL,value;
    int base,base_s,height_s;
    float factor;
    
    if (nw->cg->step==1) {
	update_canvas_mem_1(nw,dpy,win,height,length);
	return;
    }
    
    if (nw->cg->step>app_resources.cv_maxpoints) {
	update_canvas_mem_lines(nw,dpy,win,height,length);
	return;
    }
    
    factor=(float) height/(float) pow(2.0,nw->wd->res);
    
    base=height/2;
    height_s=height-1;
    base_s=base-1;
    
    switch (nw->wd->channels) {
     case MONO:
	if ((ispointl=calloc(height,sizeof(byte)))==NULL) return;
	if ((pointsl=(XPoint*) XtCalloc(height,sizeof(XPoint)))==NULL) {
	    free(ispointl);
	    return;
	}
	break;
     case STEREO:
	if ((ispointl=calloc(height/2,sizeof(byte)))==NULL) return;
	if ((pointsl=(XPoint*) XtCalloc(height/2,sizeof(XPoint)))==NULL) {
	    free(ispointl);
	    return;
	}
	if ((ispointr=calloc(height/2,sizeof(byte)))==NULL) {
	    free(ispointl);
	    free(pointsl);
	    return;
	}
	if ((pointsr=(XPoint*) XtCalloc(height/2,sizeof(XPoint)))==NULL) {
	    free(ispointl);
	    free(pointsl);
	    free(ispointr);
	    return;
	}
	factor/=2;
	break;
    }
    
    if (nw->wd->res == 8) {
	data = nw->wd->buffer;
	data+= nw->cg->pos;
	
	if (nw->wd->channels==MONO) { 
	    for (i=0,j=0,k=0,m=0;i<length;i++,j++) {	
		if (j==nw->cg->step) {
		    for (j=0;j<height;j++) ispointl[j]=FALSE;   
		    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
		    j=0;
		    k=0;
		    m++;
		}
		y=data[i]*factor;
		if (!ispointl[y]) { 
		    ispointl[y]=True;
		    pointsl[k].y=height_s-y;
		    pointsl[k].x=m;
		    k++;
		}
	    }
	    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
	} else { 
	    for (i=0,j=0,k=0,l=0,m=0;i<length*2;j++) {	
		if (j==nw->cg->step) {
		    for (j=0;j<height/2;j++) {
			ispointl[j]=FALSE;   /* clear test array */
			ispointr[j]=FALSE;   /* clear test array */
		    }   
		    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
		    XDrawPoints(dpy,win,nw->cg->gcr,pointsr,l,CoordModeOrigin);
		    j=0;
		    k=0;
		    l=0;
		    m++;
		}
		y=data[i]*factor;
		i++;
		if (!ispointl[y]) { 
		    ispointl[y]=True;
		    pointsl[k].y=base_s-y;
		    pointsl[k].x=m;
		    k++;
		}
		y=data[i]*factor;
		i++;
		if (!ispointr[y]) { 
		    ispointr[y]=True;
		    pointsr[l].y=base+(base_s-y);
		    pointsr[l].x=m;
		    l++;
		}
	    }
	    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
	    XDrawPoints(dpy,win,nw->cg->gcr,pointsr,l,CoordModeOrigin);
	}  
    } else {
	data16=(ushort*)nw->wd->buffer;
	data16+=nw->cg->pos/2;
	
	if (nw->wd->channels==MONO) {
	    for (i=0,j=0,k=0,m=0; i<length;i++,j++) {
		if (j==nw->cg->step) {
		    /* clear test array */
		    for (j=0;j<height;j++) ispointl[j]=False;  
		    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
		    j=0;
		    k=0;
		    m++;
		}
		value=data16[i]+32767;
		y=value*factor;
		if (!ispointl[y]) { 
		    ispointl[y]=True;
		    pointsl[k].y=height_s-y;
		    pointsl[k].x=m;
		    k++;
		}
	    }
	    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
	} else { 
	    for (i=0,j=0,k=0,l=0,m=0; i<length*2;j++) {	
		if (j==nw->cg->step) {
		    for (j=0;j<=height/2;j++) { 
			ispointl[j]=False;   /* clear test array */
			ispointr[j]=False;   /* clear test array */
		    }
		    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
		    XDrawPoints(dpy,win,nw->cg->gcr,pointsr,l,CoordModeOrigin);
		    j=0;
		    k=0;
		    l=0;
		    m++;
		}
		value=data16[i]+32767;
		y=value*factor;
		i++;
		if (!ispointl[y]) { 
		    ispointl[y]=True;
		    pointsl[k].y=base_s-y;
		    pointsl[k].x=m;
		    k++;
		}
		value=data16[i]+32767;
		y=value*factor;
		i++;
		if (!ispointr[y]) { 
		    ispointr[y]=True;
		    pointsr[l].y=base+(base_s-y);
		    pointsr[l].x=m;
		    l++;
		}
	    }
	    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
	    XDrawPoints(dpy,win,nw->cg->gcr,pointsr,l,CoordModeOrigin);
	}
    }
    
    /* XtFree returns immediately if pointer is NULL */ 
    XtFree((char*) ispointl);
    XtFree((char*) pointsl);
    XtFree((char*) ispointr);
    XtFree((char*) pointsr);
}

void update_canvas_mem_1(Next_Wave *nw,Display *dpy,Window win,
			 Dimension height,int length)
{
    XPoint *pointsl=NULL,*pointsr=NULL;
    int i,j,y;
    byte *data=NULL;				/* to go through the data */
    unsigned short *data16=NULL,value;
    int base,base_s,height_s;
    float factor;
    
    if ((pointsl=(XPoint*) XtCalloc(nw->cg->width,sizeof(XPoint)))==NULL)
      return;
    
    factor=(float) height/(float) pow(2.0,nw->wd->res);
    base=height/2;
    height_s=height-1;
    base_s=base-1;
    
    switch (nw->wd->channels) {
     case STEREO:
	if ((pointsr=(XPoint*) XtCalloc(nw->cg->width,sizeof(XPoint)))==NULL) {
	    free(pointsl);
	    return;
	}
	factor/=2.0;
	break;
    }
    
    if (nw->wd->res == 8) {
	
	data = nw->wd->buffer;
	data+= nw->cg->pos;
	
	if (nw->wd->channels==MONO) { 
	    for (i=0;i<length;i++) {
		y=height_s-data[i]*factor;
		pointsl[i].y=y;
		pointsl[i].x=i;
	    }
	} else { 
	    
	    for (i=0,j=0;i<length;i++) {
		y=base_s-data[j]*factor;
		j++;
		pointsl[i].y=y;
		pointsl[i].x=i;
		y=base+(base_s-data[j]*factor);
		j++;
		pointsr[i].y=y;
		pointsr[i].x=i;
	    }
	}  
    } else {
	
	data16=(unsigned short*)nw->wd->buffer;
	data16+=nw->cg->pos/2;
	
	if (nw->wd->channels==MONO) {
	    for (i=0;i<length;i++) {
		value=data16[i]+32767;
		y=value*factor;
		pointsl[i].y=height_s-y;
		pointsl[i].x=i;
	    }
	} else { 
	    for (i=0,j=0;i<length;i++) {
		value=data16[j]+32767;
		y=value*factor;
		pointsl[i].y=base_s-y;
		pointsl[i].x=i;
		j++;
		value=data16[j]+32767;
		y=value*factor;
		pointsr[i].y=base+(base_s-y);
		pointsr[i].x=i;
		j++;
	    }
	}
    }
    
    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,nw->cg->width,CoordModeOrigin);
    switch(nw->wd->channels) {
     case STEREO:
	XDrawPoints(dpy,win,nw->cg->gcr,pointsr,nw->cg->width,CoordModeOrigin);
	free(pointsr);
	break;
    }
    free(pointsl);
}

void update_canvas_mem_lines(Next_Wave *nw,Display *dpy,Window win,
			     Dimension height,int length)
{
    int i,j,k;
    ushort *data16=NULL,left_u,left_b,right_u,right_b,y;
    byte *data=NULL;
    int base,base_s,height_s;
    float factor;
    
    left_u=right_u=0;
    left_b=right_b=65535;
    
    factor=(float) height/(float) pow(2.0,nw->wd->res);
    base=height/2;
    height_s=height-1;
    base_s=base-1;
    
    switch (nw->wd->channels) {
     case STEREO:
	factor/=2.0;
	break;
    }
    if (nw->wd->res == 8) {
	data = nw->wd->buffer;
	data+= nw->cg->pos;
	
	if (nw->wd->channels==MONO) { 
	    for (i=0,j=0,k=0;i<length;i++,j++) {	
		y=data[i];
		if (y>left_u) left_u=y;if (y<left_b) left_b=y;
		if (j==nw->cg->step) {
		    left_u*=factor;left_b*=factor;
		    left_u=height_s-left_u;left_b=height_s-left_b;
		    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
		    j=0;k++;
		    left_u=right_u=0;
		    left_b=right_b=65535;
		}
	    }
	    left_u*=factor;left_b*=factor;
	    left_u=height_s-left_u;left_b=height_s-left_b;
	    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
	} else { 
	    for (i=0,j=0,k=0;i<length*2;j++) {	
		y=data[i];i++;
		if (y>left_u) left_u=y;if (y<left_b) left_b=y;
		y=data[i];i++;
		if (y>right_u) right_u=y;if (y<right_b) right_b=y;
		if (j==nw->cg->step) {
		    left_u*=factor;left_b*=factor;
		    left_u=base_s-left_u;left_b=base_s-left_b;
		    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
		    right_u*=factor;right_b*=factor;
		    right_u=base_s-right_u;right_b=base_s-right_b;
		    right_u+=base;right_b+=base;
		    XDrawLine(dpy,win,nw->cg->gcr,k,right_b,k,right_u);
		    j=0;k++;
		    left_u=right_u=0;
		    left_b=right_b=65535;
		}
	    }
	    left_u*=factor;left_b*=factor;
	    left_u=base_s-left_u;left_b=base_s-left_b;
	    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
	    right_u*=factor;right_b*=factor;
	    right_u=base_s-right_u;right_b=base_s-right_b;
	    right_u+=base;right_b+=base;
	    XDrawLine(dpy,win,nw->cg->gcr,k,right_b,k,right_u);
	}  
    } else {
	data16=(ushort*)nw->wd->buffer;
	data16+=nw->cg->pos/2;
	
	if (nw->wd->channels==MONO) {
	    for (i=0,j=0,k=0; i<length;i++,j++) {
		y=data16[i]+32767;
		if (y>left_u) left_u=y;if (y<left_b) left_b=y;
		if (j==nw->cg->step) {
		    left_u*=factor;left_b*=factor;
		    left_u=height_s-left_u;left_b=height_s-left_b;
		    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
		    j=0;k++;
		    left_u=right_u=0;
		    left_b=right_b=65535;
		}
	    }
	    left_u*=factor;left_b*=factor;
	    left_u=height_s-left_u;left_b=height_s-left_b;
	    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
	} else { 
	    for (i=0,j=0,k=0; i<length*2;j++) {
		y=data16[i]+32767;i++;
		if (y>left_u) left_u=y;if (y<left_b) left_b=y;
		y=data16[i]+32767;i++;
		if (y>right_u) right_u=y;if (y<right_b) right_b=y;
		if (j==nw->cg->step) {
		    left_u*=factor;left_b*=factor;
		    left_u=base_s-left_u;left_b=base_s-left_b;
		    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
		    right_u*=factor;right_b*=factor;
		    right_u=base_s-right_u;right_b=base_s-right_b;
		    right_u+=base;right_b+=base;
		    XDrawLine(dpy,win,nw->cg->gcr,k,right_b,k,right_u);
		    j=0;k++;
		    left_u=right_u=0;
		    left_b=right_b=65535;
		}
	    }
	    left_u*=factor;left_b*=factor;
	    left_u=base_s-left_u;left_b=base_s-left_b;
	    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
	    right_u*=factor;right_b*=factor;
	    right_u=base_s-right_u;right_b=base_s-right_b;
	    right_u+=base;right_b+=base;
	    XDrawLine(dpy,win,nw->cg->gcr,k,right_b,k,right_u);
	}
    }
}

void update_canvas_file(Next_Wave *nw,Display *dpy,Window win,
			Dimension height,int length)
{
    XPoint *pointsl=NULL,*pointsr=NULL;					
    byte *ispointl=NULL,*ispointr=NULL;		/* for test, if we should set
						 a point at this x-coord */
    int i,j,k,l=0,m=0,n,y=0;
    byte *data=NULL;				/* to go through the data */
    ushort *data16=NULL,value;
    int base,base_s,height_s;
    float factor;
    Audio_File af;
    
    
    if (nw->cg->step==1) {
	update_canvas_file_1(nw,dpy,win,height,length);
	return;
    }
    
    if (nw->cg->step>app_resources.cv_maxpoints) {
	update_canvas_file_lines(nw,dpy,win,height,length);
	return;
    }
    
    wd2af(nw->wd,&af);
    if (af_rewind(af)==AF_ERROR) return;
    
    factor=(float) height/(float) pow(2.0,nw->wd->res);
    
    base=height/2;
    height_s=height-1;
    base_s=base-1;
    
    switch (nw->wd->channels) {
     case MONO:
	if ((ispointl=calloc(height,sizeof(byte)))==NULL) return;
	if ((pointsl=(XPoint*) XtCalloc(height,sizeof(XPoint)))==NULL) {
	    free(ispointl);
	    return;
	}
	break;
     case STEREO:
	if ((ispointl=calloc(height/2,sizeof(byte)))==NULL) return;
	if ((pointsl=(XPoint*) XtCalloc(height/2,sizeof(XPoint)))==NULL) {
	    free(ispointl);
	    return;
	}
	if ((ispointr=calloc(height/2,sizeof(byte)))==NULL) {
	    free(ispointl);
	    free(pointsl);
	    return;
	}
	if ((pointsr=(XPoint*) XtCalloc(height/2,sizeof(XPoint)))==NULL) {
	    free(ispointl);
	    free(pointsl);
	    free(ispointr);
	    return;
	}
	factor/=2;
	break;
    }
    
    if (af_seek(af,nw->cg->pos,SEEK_CUR)==AF_ERROR) return;
    
    if (nw->wd->res == 8) {
	data = (byte*) MD->mg->fbuf;
	if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
	
	if (nw->wd->channels==MONO) { 
	    for (i=0,j=0,k=0,m=0,n=0;i<length;i++,j++,n++) {	
		if (j==nw->cg->step) {
		    for (j=0;j<height;j++) ispointl[j]=FALSE;   
		    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
		    j=0;
		    k=0;
		    m++;
		}
		if (n>=MAXUSHORT) {
		    if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		y=data[n]*factor;
		if (!ispointl[y]) { 
		    ispointl[y]=True;
		    pointsl[k].y=height_s-y;
		    pointsl[k].x=m;
		    k++;
		}
	    }
	    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
	} else { 
	    for (i=0,j=0,k=0,l=0,m=0,n=0;i<length;i++,j++) {	
		if (j==nw->cg->step) {
		    for (j=0;j<height/2;j++) {
			ispointl[j]=FALSE;   /* clear test array */
			ispointr[j]=FALSE;   /* clear test array */
		    }   
		    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
		    XDrawPoints(dpy,win,nw->cg->gcr,pointsr,l,CoordModeOrigin);
		    j=0;
		    k=0;
		    l=0;
		    m++;
		}
		if (n>=MAXUSHORT) {
		    if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		y=data[n]*factor;
		n++;
		if (!ispointl[y]) { 
		    ispointl[y]=True;
		    pointsl[k].y=base_s-y;
		    pointsl[k].x=m;
		    k++;
		}
		y=data[n]*factor;
		n++;
		if (!ispointr[y]) { 
		    ispointr[y]=True;
		    pointsr[l].y=base+(base_s-y);
		    pointsr[l].x=m;
		    l++;
		}
	    }
	    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
	    XDrawPoints(dpy,win,nw->cg->gcr,pointsr,l,CoordModeOrigin);
	}  
    } else {
	data16=(ushort*)MD->mg->fbuf;
	if (af_read(af,(char*)data16,MAXUSHORT)==AF_ERROR) return;
	
	if (nw->wd->channels==MONO) {
	    for (i=0,j=0,k=0,m=0,n=0; i<length;i++,j++,n++) {
		if (j==nw->cg->step) {
		    /* clear test array */
		    for (j=0;j<height;j++) ispointl[j]=False;  
		    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
		    j=0;
		    k=0;
		    m++;
		}
		if (n>=MAXUSHORT/2) {
		    if (af_read(af,(char*)data16,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		value=data16[n]+32767;
		y=value*factor;
		if (!ispointl[y]) { 
		    ispointl[y]=True;
		    pointsl[k].y=height_s-y;
		    pointsl[k].x=m;
		    k++;
		}
	    }
	    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
	} else { 
	    for (i=0,j=0,k=0,l=0,m=0,n=0; i<length;i++,j++) {	
		if (j==nw->cg->step) {
		    for (j=0;j<=height/2;j++) { 
			ispointl[j]=False;   /* clear test array */
			ispointr[j]=False;   /* clear test array */
		    }
		    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
		    XDrawPoints(dpy,win,nw->cg->gcr,pointsr,l,CoordModeOrigin);
		    j=0;
		    k=0;
		    l=0;
		    m++;
		}
		if (n>=MAXUSHORT/2) {
		    if (af_read(af,(char*)data16,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		value=data16[n]+32767;
		y=value*factor;
		n++;
		if (!ispointl[y]) { 
		    ispointl[y]=True;
		    pointsl[k].y=base_s-y;
		    pointsl[k].x=m;
		    k++;
		}
		value=data16[n]+32767;
		y=value*factor;
		n++;
		if (!ispointr[y]) { 
		    ispointr[y]=True;
		    pointsr[l].y=base+(base_s-y);
		    pointsr[l].x=m;
		    l++;
		}
	    }
	    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,k,CoordModeOrigin);
	    XDrawPoints(dpy,win,nw->cg->gcr,pointsr,l,CoordModeOrigin);
	}
    }
    
    /* XtFree returns immediately if pointer is NULL */ 
    XtFree((char*) ispointl);
    XtFree((char*) pointsl);
    XtFree((char*) ispointr);
    XtFree((char*) pointsr);
}

void update_canvas_file_1(Next_Wave *nw,Display *dpy,Window win,
			  Dimension height,int length)
{
    XPoint *pointsl=NULL,*pointsr=NULL;
    int i,n,y;
    byte *data=NULL;				/* to go through the data */
    unsigned short *data16=NULL,value;
    int base,base_s,height_s;
    float factor;
    Audio_File af;
    
    wd2af(nw->wd,&af);
    if (af_rewind(af)==AF_ERROR) return;
    
    if ((pointsl=(XPoint*) XtCalloc(nw->cg->width,sizeof(XPoint)))==NULL)
      return;
    
    factor=(float) height/(float) pow(2.0,nw->wd->res);
    base=height/2;
    height_s=height-1;
    base_s=base-1;
    
    switch (nw->wd->channels) {
     case STEREO:
	if ((pointsr=(XPoint*) XtCalloc(nw->cg->width,sizeof(XPoint)))==NULL) {
	    free(pointsl);
	    return;
	}
	factor/=2.0;
	break;
    }
    
    if (af_seek(af,nw->cg->pos,SEEK_CUR)==AF_ERROR) return;
    
    if (nw->wd->res == 8) {
	
	data=(byte*)MD->mg->fbuf;
	if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
	
	if (nw->wd->channels==MONO) { 
	    for (i=0,n=0;i<length;i++,n++) {
		if (n==MAXUSHORT) {
		    if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		y=height_s-data[n]*factor;
		pointsl[i].y=y;
		pointsl[i].x=i;
	    }
	} else { 
	    
	    for (i=0,n=0;i<length;i++) {
		if (n>=MAXUSHORT) {
		    if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		y=base_s-data[n]*factor;
		n++;
		pointsl[i].y=y;
		pointsl[i].x=i;
		y=base+(base_s-data[n]*factor);
		n++;
		pointsr[i].y=y;
		pointsr[i].x=i;
	    }
	}  
    } else {
	
	data16=(ushort*) MD->mg->fbuf;
	if (af_read(af,(char*)data16,MAXUSHORT)==AF_ERROR) return;
	
	if (nw->wd->channels==MONO) {
	    for (i=0,n=0;i<length;i++,n++) {
		if (n>=MAXUSHORT/2) {
		    if (af_read(af,(char*)data16,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		value=data16[n]+32767;
		y=value*factor;
		pointsl[i].y=height_s-y;
		pointsl[i].x=i;
	    }
	} else { 
	    for (i=0,n=0;i<length;i++) {
		if (n>=MAXUSHORT/2) {
		    if (af_read(af,(char*)data16,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		value=data16[n]+32767;
		y=value*factor;
		pointsl[i].y=base_s-y;
		pointsl[i].x=i;
		n++;
		value=data16[n]+32767;
		y=value*factor;
		pointsr[i].y=base+(base_s-y);
		pointsr[i].x=i;
		n++;
	    }
	}
    }
    
    XDrawPoints(dpy,win,nw->cg->gcl,pointsl,nw->cg->width,CoordModeOrigin);
    switch(nw->wd->channels) {
     case STEREO:
	XDrawPoints(dpy,win,nw->cg->gcr,pointsr,nw->cg->width,CoordModeOrigin);
	free(pointsr);
	break;
    }
    free(pointsl);
}

void update_canvas_file_lines(Next_Wave *nw,Display *dpy,Window win,
			      Dimension height,int length)
{
    int i,j,k,n;
    ushort *data16=NULL,left_u,left_b,right_u,right_b,y;
    byte *data=NULL;
    int base,base_s,height_s;
    float factor;
    Audio_File af;
    
    wd2af(nw->wd,&af);
    if (af_rewind(af)==AF_ERROR) return;
    
    left_u=right_u=0;
    left_b=right_b=65535;
    
    factor=(float) height/(float) pow(2.0,nw->wd->res);
    base=height/2;
    height_s=height-1;
    base_s=base-1;
    
    switch (nw->wd->channels) {
     case STEREO:
	factor/=2.0;
	break;
    }
    
    if (af_seek(af,nw->cg->pos,SEEK_CUR)==AF_ERROR) return;
    
    if (nw->wd->res == 8) {
	data = (byte*) MD->mg->fbuf;
	if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
	
	if (nw->wd->channels==MONO) { 
	    for (i=0,j=0,k=0,n=0;i<length;i++,j++,n++) {
		if (n==MAXUSHORT) {
		    if (af_read(af,(char*)data,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		y=data[n];
		if (y>left_u) left_u=y;if (y<left_b) left_b=y;
		if (j==nw->cg->step) {
		    left_u*=factor;left_b*=factor;
		    left_u=height_s-left_u;left_b=height_s-left_b;
		    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
		    j=0;k++;
		    left_u=right_u=0;
		    left_b=right_b=65535;
		}
	    }
	    left_u*=factor;left_b*=factor;
	    left_u=height_s-left_u;left_b=height_s-left_b;
	    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
	} else { 
	    for (i=0,j=0,k=0,n=0;i<length;i++,j++) {	
		if (n==MAXUSHORT) {
		    if (af_read(af,(char*) data,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		y=data[n];n++;
		if (y>left_u) left_u=y;if (y<left_b) left_b=y;
		y=data[n];n++;
		if (y>right_u) right_u=y;if (y<right_b) right_b=y;
		if (j==nw->cg->step) {
		    left_u*=factor;left_b*=factor;
		    left_u=base_s-left_u;left_b=base_s-left_b;
		    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
		    right_u*=factor;right_b*=factor;
		    right_u=base_s-right_u;right_b=base_s-right_b;
		    right_u+=base;right_b+=base;
		    XDrawLine(dpy,win,nw->cg->gcr,k,right_b,k,right_u);
		    j=0;k++;
		    left_u=right_u=0;
		    left_b=right_b=65535;
		}
	    }
	    left_u*=factor;left_b*=factor;
	    left_u=base_s-left_u;left_b=base_s-left_b;
	    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
	    right_u*=factor;right_b*=factor;
	    right_u=base_s-right_u;right_b=base_s-right_b;
	    right_u+=base;right_b+=base;
	    XDrawLine(dpy,win,nw->cg->gcr,k,right_b,k,right_u);
	}  
    } else {
	data16=(ushort*)MD->mg->fbuf;
	if (af_read(af,(char*)data16,MAXUSHORT)==AF_ERROR) return;
	
	if (nw->wd->channels==MONO) {
	    for (i=0,j=0,k=0,n=0; i<length;i++,j++,n++) {
		if (n>=MAXUSHORT/2) {
		    if (af_read(af,(char*) data16,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		y=data16[n]+32767;
		if (y>left_u) left_u=y;if (y<left_b) left_b=y;
		if (j==nw->cg->step) {
		    left_u*=factor;left_b*=factor;
		    left_u=height_s-left_u;left_b=height_s-left_b;
		    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
		    j=0;k++;
		    left_u=right_u=0;
		    left_b=right_b=65535;
		}
	    }
	    left_u*=factor;left_b*=factor;
	    left_u=height_s-left_u;left_b=height_s-left_b;
	    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
	} else { 
	    for (i=0,j=0,k=0,n=0; i<length;i++,j++) {
		if (n>=MAXUSHORT/2) {
		    if (af_read(af,(char*) data16,MAXUSHORT)==AF_ERROR) return;
		    n=0;
		}
		y=data16[n]+32767;n++;
		if (y>left_u) left_u=y;if (y<left_b) left_b=y;
		y=data16[n]+32767;n++;
		if (y>right_u) right_u=y;if (y<right_b) right_b=y;
		if (j==nw->cg->step) {
		    left_u*=factor;left_b*=factor;
		    left_u=base_s-left_u;left_b=base_s-left_b;
		    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
		    right_u*=factor;right_b*=factor;
		    right_u=base_s-right_u;right_b=base_s-right_b;
		    right_u+=base;right_b+=base;
		    XDrawLine(dpy,win,nw->cg->gcr,k,right_b,k,right_u);
		    j=0;k++;
		    left_u=right_u=0;
		    left_b=right_b=65535;
		}
	    }
	    left_u*=factor;left_b*=factor;
	    left_u=base_s-left_u;left_b=base_s-left_b;
	    XDrawLine(dpy,win,nw->cg->gcl,k,left_b,k,left_u);
	    right_u*=factor;right_b*=factor;
	    right_u=base_s-right_u;right_b=base_s-right_b;
	    right_u+=base;right_b+=base;
	    XDrawLine(dpy,win,nw->cg->gcr,k,right_b,k,right_u);
	}
    }
}

void expose_action (Widget w, Main_Data *md, XExposeEvent *event, Boolean *flg)
{
    if (md->wd==NULL) return;
    
    /*
    if (md->mb->exposing) return;
    */
    
#if defined (linux) || defined (FreeBSD)
    if (resizing) {
	resizing=FALSE;
	return;
    }
#endif
    
    if (md->mg->pixmap==None) {
	if (event->count) return;
	md->mb->exposing=TRUE;
	update_display(md);
	md->mb->exposing=FALSE;
	return;
    } 
}

void config_action (Widget w, Main_Data *md, XConfigureEvent *event,
		    Boolean *flg)
{
    if (md->wd==NULL) return;
    resizing=True;
    md->mb->exposing=True;
    update_display(md);
    md->mb->exposing=False;
}

/* some windowmanagers doesn't send a message if window got focus */
Boolean test_focus(Main_Data *md,Widget w1,Widget w2)
{
    if (w1!=w2) {
	Next_Wave *nw=md->nw;
	while (nw!=NULL) {
	    if (nw->cw->graph==w2 || nw->cw->scroll==w2 || nw->cw->zscroll==w2) {
		set_actual_canvas(nw);
		return(False);
	    }
	    nw=nw->nw;
	}
    }
    return(True);
}

Next_Wave *get_nw(Main_Data *md,Widget w)
{
    Next_Wave *nw=md->nw;
    while (nw!=NULL) {
	if (nw->cw->graph==w || nw->cw->scroll==w || nw->cw->zscroll==w) {
	    return(nw);
	}
	nw=nw->nw;
    }
    return(NULL);
}

void expose_canvas (Widget w, Main_Data *md, XExposeEvent *event, Boolean *flg)
{
#if defined (linux) || defined (FreeBSD)
    if (resizing) {
	if (!event->count) resizing=FALSE;
	return;
    }
#endif
    /*
    if (md->cb->exposing) return;
    */
    if (md->cg->pixmap==None) {
	if (event->count) return;
	md->cb->exposing=TRUE;
	update_canvas(md,get_nw(md,w));
	md->cb->exposing=FALSE;
	return;
    } 
}

void config_canvas (Widget w, Main_Data *md, XConfigureEvent *event,
		    Boolean *flg)
{
    /*
     if (md->cb->exposing) return;
     */
     
    resizing=TRUE;
    md->cb->exposing=TRUE;
    update_canvas(md,get_nw(md,w));
    md->cb->exposing=FALSE;
}

void jump_canvas(Widget w,XtPointer client_data,XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    Next_Wave *nw;
    Dimension width;
    float top=*((float*)call_data);
    int newpos;

    if ((nw=get_nw(md,w))==NULL) return;
    
    /* we are at the begin and drag to begin */
    if (top<0.001) {
	top=0.0;
	if (nw->cg->pos==0) return;
    }
    
    newpos=(int)((float)nw->wd->tlength*top);
    newpos*=nw->wd->bpspl;
    
    XtVaGetValues(nw->cw->graph,XtNwidth,&width,NULL);
    
    /* top is sometimes above 1.0 or below 0.999, so we ask not for */
    /* width/2, but for width */
    if (newpos>(nw->wd->tlength-(width*nw->cg->step))*nw->wd->bpspl)
      newpos=(nw->wd->tlength-(width/2*nw->cg->step))*nw->wd->bpspl;
    
    if (newpos==nw->cg->pos) return;
    
    nw->cg->pos=newpos;
    md->mb->exposing=TRUE;
    
    XClearArea(XtDisplay(nw->cw->graph),XtWindow(nw->cw->graph),
	       0, 0, 0, 0, False);
    
    update_canvas(md,nw);
    
    md->mb->exposing=FALSE;
}

void scroll_canvas(Widget w,XtPointer client_data,XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int scroll=(int)call_data*5;
    Dimension width;
    Next_Wave *nw;

    if ((nw=get_nw(md,w))==NULL) return;
    
    if (scroll==0) return;
    if ((nw->cg->pos==0)&&(scroll<0)) return;
    
    XtVaGetValues(nw->cw->graph,XtNwidth,&width,NULL);
    
    if (scroll>0) {
	if (nw->cg->pos==(nw->wd->tlength-(width/2*nw->cg->step))*nw->wd->bpspl)
	  return;
	nw->cg->pos+=scroll*nw->wd->bpspl;
	if (nw->cg->pos>(nw->wd->tlength-(width/2*nw->cg->step))*nw->wd->bpspl)
	  nw->cg->pos=(nw->wd->tlength-(width/2*nw->cg->step))*nw->wd->bpspl;
	
    } else {
	nw->cg->pos+=scroll*nw->wd->bpspl;
    }
    
    md->mb->exposing=TRUE;
    update_canvas(md,nw);
    md->mb->exposing=FALSE;
}

void zjump_canvas(Widget w,XtPointer client_data,XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    float top=*((float*)call_data);
    int newzoom;
    Next_Wave *nw;
    
    if ((nw=get_nw(md,w))==NULL) return;
    
    newzoom=((float)nw->wd->tlength/(float)nw->cg->width)-
      top*((float)nw->wd->tlength/(float)nw->cg->width) +1;
    if (newzoom<1) newzoom=1;
    if (newzoom==nw->cg->step) return;   
    md->mb->exposing=TRUE;
    nw->cg->step=newzoom;
    update_canvas(md,nw);
    if (nw->wd==md->wd) set_zoom_label(md->cg->step);
    md->mb->exposing=FALSE;
}

void zscroll_canvas(Widget w,XtPointer client_data,XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int scroll=(int)call_data;
    Next_Wave *nw;
    
    if ((nw=get_nw(md,w))==NULL) return;

    if ((nw->cg->step==1)&&(scroll>0)) return;
    if ((nw->cg->step==nw->wd->tlength/nw->cg->width+1)&&(scroll<0)) return; 
    
    md->mb->exposing=TRUE;
    if (scroll>0) nw->cg->step--;
    else nw->cg->step++;
    update_canvas(md,nw);
    if (nw->wd==md->wd) set_zoom_label(md->cg->step);
    md->mb->exposing=FALSE;
}

void zoom_canvas(Main_Data *md)
{
    Dimension width;

    md->mb->exposing=TRUE;
    XtVaGetValues(md->cw->graph,XtNwidth,&width,NULL);
    md->cg->pos=md->wd->markbeg*md->wd->bpspl;
    md->cg->step=md->wd->marklength/width+1;
    if (!md->cg->step) md->cg->step=1;
    /* sub 5% */
    md->cg->pos-=(width/20)*md->cg->step;
    /* are we at begin of a sample pair ? */
    md->cg->pos+=md->cg->pos % md->wd->bpspl;
    update_canvas(md,NULL);
    set_zoom_label(md->cg->step);
    md->mb->exposing=FALSE;
}

void buttonp_action (Widget w, Main_Data *md, XButtonEvent *event,
		     Boolean *flg)
{
    if (md->no==0) return;
    if (md->wd->length==0) return;
    if (md->mb->playing) return;
    if (event->button>1) return;
    
    if (md->wd->isplay) { 
	md->wd->isplay=False;
	clear_line(md->wd->markbeg/md->mg->step);
    }
    if (md->wd->ismark) {
	set_whole_mark(md);
	set_canvas_mark(get_actual_nw(md));
	md->wd->ismark=False;
    }
    firstmark=md->wd->markbeg=event->x;
    set_beginmark_label((int) ((float)firstmark)*md->mg->step);
    set_cursor(md->mw->graph,CURSOR_MARK);
}

void mark_action (Widget w, Main_Data *md, XMotionEvent *event, Boolean *flg)
{
    Window win = XtWindow(md->mw->graph);		
    Display *dpy=XtDisplay(md->mw->graph);
    int x;
    
    if (md->no==0) return;
    if (md->mb->playing) return;
    
    x=event->x;
    if (x>=md->mg->width) x=md->mg->width;
    if (x<0) x=0;
    
    if (firstmark==-1) return;
    if (x>=md->wd->markbeg) {
	XFillRectangle(dpy,win,md->mg->markgc,md->wd->markbeg,0,
		       x-md->wd->markbeg,PREVIEWHEIGHT);
	md->wd->markbeg=x;
    } else {
	XFillRectangle(dpy,win,md->mg->markgc,x,0,
		       md->wd->markbeg-x,PREVIEWHEIGHT);
	md->wd->markbeg=x;
    }
    if (x>firstmark) {
	set_beginmark_label((int)((float)firstmark*md->mg->step));
	set_lengthmark_label((int)((float)(x-firstmark)*md->mg->step));
    } else {
	set_beginmark_label((int)((float)x*md->mg->step));
	set_lengthmark_label((int)((float)(firstmark-x)*md->mg->step));
    }
}


void buttonr_action (Widget w, Main_Data *md, XButtonEvent *event, 
		     Boolean *flg)
{
    if (md->no==0) return;
    if (md->mb->playing) return;
    
    if (event->button==1) {
	Dimension width;
	if (firstmark==-1) return;
	XtVaGetValues(md->mw->graph,XtNwidth,&width,NULL);
	if (event->x<0) event->x=0;
	if (event->x>width) event->x=width;
	set_cursor(md->mw->graph,CURSOR_DEFAULT);
	
	if (event->x==firstmark) {
	    md->wd->markbeg=ceil(firstmark*md->mg->step);
	    set_line(md->wd->markbeg/md->mg->step);
	    md->wd->isplay=True;
	    md->wd->ismark=False;
	    md->cg->pos=md->wd->markbeg*md->wd->bpspl;
	} else {
	    if (event->x>firstmark) {
		md->wd->markbeg=ceil(firstmark*md->mg->step);
		md->wd->marklength=ceil((event->x-firstmark)*md->mg->step);
	    } else {
		md->wd->markbeg=ceil(event->x*md->mg->step);
		md->wd->marklength=ceil((firstmark-event->x)*md->mg->step);
	    }
	    md->wd->ismark=True;
	    md->wd->isplay=False;
	    md->cg->pos=(md->wd->markbeg+md->wd->marklength)*md->wd->bpspl;
	    md->cg->pos-=md->cg->width/2*md->wd->bpspl*md->cg->step;
	}
	update_canvas(md,NULL);
    } else {
	if (md->wd->ismark) {
	    set_whole_mark(md);
	    set_canvas_mark(get_actual_nw(md));
	} 
	if (md->wd->isplay) clear_line(md->wd->markbeg/md->mg->step);
	
	md->wd->markbeg=0;
	md->wd->marklength=0;
	md->wd->isplay=False;
	md->wd->ismark=False;
    }
    update_status(md);
    firstmark=-1;
}

void set_mark_beg(Main_Data *md,int x)
{
    Window win = XtWindow(md->mw->graph);
    Display *dpy=XtDisplay(md->mw->graph);
    
    
    XFillRectangle(dpy,win,md->mg->markgc,x,0,1,PREVIEWHEIGHT);
    
    if (md->wd->markbeg+md->wd->marklength<md->wd->tlength)
      XFillRectangle(dpy,win,md->mg->markgc,
		     x+md->wd->marklength/md->mg->step,0,1,PREVIEWHEIGHT);
}

void set_mark_length(Main_Data *md,int x)
{
    Window win = XtWindow(md->mw->graph);		
    Display *dpy=XtDisplay(md->mw->graph);
    
    
    XFillRectangle(dpy,win,md->mg->markgc,x,0,1,PREVIEWHEIGHT);
}  

void set_whole_mark(Main_Data *md)
{
    Window win = XtWindow(md->mw->graph);		/* get window ID */
    Display *dpy=XtDisplay(md->mw->graph);
    int x,l;
    
    if (!md->wd->ismark) return;

    x=md->wd->markbeg/md->mg->step;
    l=md->wd->marklength/md->mg->step;
    XFillRectangle(dpy,win,md->mg->markgc,x,0,l,PREVIEWHEIGHT);
}  

void set_line(int value)
{
    Window win = XtWindow(MD->mw->graph);		/* get window ID */
    Display *dpy=XtDisplay(MD->mw->graph);
    
    XCopyArea(dpy, XtWindow(MD->mw->graph), MD->mg->lpixmap, MD->mg->gcl, value,
	      0, 1, PREVIEWHEIGHT, 0, 0);
    XDrawLine(dpy, win, MD->mg->playgc,value, 0, value, PREVIEWHEIGHT);
    XFlush(dpy);
    line_saved=True;
}

void clear_line(int value)
{
    Window win = XtWindow(MD->mw->graph);		/* get window ID */
    Display *dpy=XtDisplay(MD->mw->graph);
    
    if (!line_saved) return;
    XClearArea(dpy, win, value, 0, 1, 0, FALSE);
    XCopyArea(dpy, MD->mg->lpixmap, XtWindow(MD->mw->graph), MD->mg->gcl, 
	      0, 0, 1, PREVIEWHEIGHT, value , 0);
    XFlush(dpy);
    line_saved=False;
}

void set_canvas_mark(Next_Wave *nw)
{
    Window win = XtWindow(nw->cw->graph);		/* get window ID */
    Display *dpy=XtDisplay(nw->cw->graph);
    Dimension width,height;
    int beg,length;
    
    if (!nw->wd->ismark) return;
    
    beg=(nw->wd->markbeg/nw->cg->step)-(nw->cg->pos/nw->wd->bpspl/nw->cg->step);
    length=nw->wd->marklength/nw->cg->step;
    XtVaGetValues(nw->cw->graph,XtNwidth,&width,XtNheight,&height,NULL);
    if (beg>(short)width) return;
    if (beg+length>(short)width) length=width-beg;
    if (beg<0) {
	if (beg+length<0) return;
	length+=beg;
	beg=0;
    }
    XFillRectangle(dpy,win,nw->cg->markgc,beg,0,
		   length,height);
}

void mark_canvas (Widget w, Main_Data *md, XEvent *event, Boolean *flg)
{
    Window win = XtWindow(md->cw->graph);		/* get window ID */
    Display *dpy=XtDisplay(md->cw->graph);
    Dimension width,height;
    XEvent pev;
    XButtonEvent *ev;
    Boolean scrolled=False;
    int x,firstpos,firstx,pos,beg,length;
    
    if (md->mb->playing) return;
    /* some windowmanagers doesn't send a message if window got focus */
    if (!test_focus(md,md->cw->graph,w)) return;
    
    ev=(XButtonEvent*)event;
    if (ev->button>2) return;
    
    if (md->wd->isplay) clear_line(md->wd->markbeg/md->mg->step);
    if (md->wd->ismark) {
	set_whole_mark(md);
	set_canvas_mark(get_actual_nw(md));
    }
    
    md->wd->isplay=False;
    md->wd->ismark=False;
    
    /* if middle button pressed, we don't mark */
    if (ev->button==2) {
	md->wd->markbeg=0;
	md->wd->marklength=0;
	update_status(md);
	return;
    }
    
    /* remove all events from queue */
    XSync(dpy,True);
    
    /* pointer outside the points/lines */
    pos=md->cg->pos/md->wd->bpspl/md->cg->step;
    if (ev->x + pos > md->wd->tlength/md->cg->step) {
	update_status(md);
	return;
    }
    
    XtVaGetValues(md->cw->graph,XtNwidth,&width,XtNheight,&height,NULL);
    
    firstpos=md->cg->pos;
    x=ev->x;
    if (x<0) x=0;
    if (x>width) x=width;
    firstx=x;
    md->wd->markbeg=x;
    
    set_cursor(md->cw->graph,CURSOR_MARK);
    
    /* wait for first event arrive after button press */
    XPeekEvent(dpy,&pev);
    
    while (True) {
	
	if ((x<0)||(x>width)) {
	    if (x<0) {
		if (md->cg->pos!=0) {
		    md->cg->pos+=x*md->cg->step*md->wd->bpspl;
		    update_canvas(md,NULL);
		    md->wd->markbeg=x;
		    scrolled=True;
		} else {
		    if (md->wd->markbeg>0) {
			XFillRectangle(dpy,win,md->mg->markgc,0,0,
				       md->wd->markbeg,height);
			md->wd->markbeg=x;
		    }
		}
	    }
	    if ((x>0)&& 
		(md->cg->pos<
		 (md->wd->tlength-(width/2*md->cg->step))*md->wd->bpspl)) {
		md->cg->pos+=(x-width)*md->cg->step*md->wd->bpspl;
		if (md->cg->pos>
		    (md->wd->tlength-(width/2*md->cg->step))*md->wd->bpspl)
		  md->cg->pos=
		  (md->wd->tlength-(width/2*md->cg->step))*md->wd->bpspl;
		update_canvas(md,NULL);
		md->wd->markbeg=x;
		scrolled=True;
	    }
	} else {
	    pos=md->cg->pos/md->cg->step/md->wd->bpspl;
	    if (scrolled) {
		int firstp=firstpos/md->cg->step/md->wd->bpspl;
		if (firstp+firstx>pos) {
		    if (firstp-pos>width) md->wd->markbeg=width;
		    else md->wd->markbeg=firstx+firstp-pos;
		} else md->wd->markbeg=0; 
		scrolled=False;
	    } 
	    if (x+pos>md->wd->tlength/md->cg->step)
	      x=md->wd->tlength/md->cg->step-pos;
	    if (x>=md->wd->markbeg) {
		XFillRectangle(dpy,win,md->mg->markgc,md->wd->markbeg,0,
			       x-md->wd->markbeg,height);
	    } else {
		XFillRectangle(dpy,win,md->mg->markgc,x,0,
			       md->wd->markbeg-x,height);
	    }
	    md->wd->markbeg=x;
	}
	
	/* new events arrived */
	if (XPending(dpy)) {
	    XPeekEvent(dpy,&pev);
	    switch (pev.type) {
	     case MotionNotify: {
		 XMotionEvent *ev;
		 
		 /* remove event from queue */
		 XNextEvent(dpy,&pev);
		 ev=(XMotionEvent*) &pev;
		 x=ev->x;
		 break;
	     } 
	     case ButtonRelease: {
		 XButtonEvent *ev;
		 Boolean update_cvmark=False;
		 
		 /* remove event from queue */
		 XNextEvent(dpy,&pev);
		 ev=(XButtonEvent*) &pev;
		 x=ev->x;
		 set_cursor(md->cw->graph,CURSOR_DEFAULT);
		 
		 if ((firstpos==md->cg->pos)&&(firstx==x)) {
		     md->wd->markbeg=
		       md->cg->pos/md->wd->bpspl+firstx*md->cg->step;
		     set_line(md->wd->markbeg/md->mg->step);
		     md->wd->isplay=True;
		     update_status(md);
		     return;
		 }
		 if ((x<0)||(x>=width)) {
		     if (x<0) x=0;
		     if (x>width) x=width;
		     update_cvmark=True;
		 }
		 if (firstpos<md->cg->pos) {
		     beg=firstpos/md->wd->bpspl;
		     length=(md->cg->pos-firstpos)/md->wd->bpspl;
		 } else {
		     beg=md->cg->pos/md->wd->bpspl;
		     length=(firstpos-md->cg->pos)/md->wd->bpspl;
		 }
		 pos=md->cg->pos/md->cg->step/md->wd->bpspl;
		 if (x+pos>md->wd->tlength/md->cg->step)
		   x=md->wd->tlength/md->cg->step-pos;
		 if (firstx<x) {
		     beg+=firstx*md->cg->step;
		     length+=(x-firstx)*md->cg->step;
		 } else {
		     beg+=x*md->cg->step;
		     length+=(firstx-x)*md->cg->step;
		 }
		 md->wd->ismark=True;
		 md->wd->markbeg=beg;
		 md->wd->marklength=length;
		 if (update_cvmark&&scrolled) 
		   set_canvas_mark(get_actual_nw(md));
		 set_whole_mark(md);
		 update_status(md);
		 return;
	     } /* ButtonRelease */
	    } /* switch */
	} /* XPending */
    } /* while(True) */
}

