%{

/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <malloc.h>

#include "insgus.h"

	/* insgus_lexer.c */

int yylex( void );

extern int line_count;
extern FILE *yyin;
extern int static_dma;
	
	/* local functions */

static void yyerror( char *, ... );
static void card_start( void );
static void card_stop( void );
static void driver_command( char *driver );

%}

%start lines

%union {
    int i_value;
    char *s_value;
  };

%token <i_value> L_INTEGER
%token <s_value> L_STRING

%token L_FALSE L_TRUE
%token L_STATIC_DMA L_CARD L_ID L_ORDER L_DRIVER
%token L_PORT L_IRQ L_ESS_IRQ L_DMA1 L_DMA1_SIZE L_DMA2 L_DMA2_SIZE
%token L_DB_PORT L_DB_IRQ L_DB_DMA L_DB_DMA_SIZE
%token L_JOYSTICK_DAC

%type <i_value> order dma_size gf1_port gf1_irq ess_irq gf1_dma db_port db_irq db_dma bool jdac

%%

lines	: line
	| lines line
	;

line	: L_CARD
		{ card_start(); }
		'{' cflags '}'
		{ card_stop(); }
	| L_STATIC_DMA bool	{ static_dma = $2; }
	| error			{ yyerror( "unknown keyword in top level\n" ); }
	;

cflags	: cflag
	| cflags cflag
	;

cflag	: L_ID L_STRING		{ strncpy( card.id, $2, sizeof( card.id ) ); free( $2 ); }
	| L_ORDER order		{ card.logical = $2; }
	| L_PORT gf1_port	{ card.port = $2; }
	| L_IRQ gf1_irq		{ card.irq = $2; }
	| L_ESS_IRQ ess_irq	{ card.ess_irq = $2; }
	| L_DMA1 gf1_dma	{ card.dma1 = $2; }
	| L_DMA1_SIZE dma_size	{ card.dma1_size = $2; }
	| L_DMA2 gf1_dma	{ card.dma2 = $2; }
	| L_DMA2_SIZE dma_size	{ card.dma2_size = $2; }
	| L_DB_PORT db_port	{ card.daughter_port = $2; }
	| L_DB_IRQ db_irq	{ card.daughter_irq = $2; }
	| L_DB_DMA db_dma	{ card.daughter_dma = $2; }
	| L_DB_DMA_SIZE dma_size { card.daughter_dma_size = $2; }
	| L_JOYSTICK_DAC jdac	{ card.joystick_dac = $2; }
	| L_DRIVER L_STRING	{ driver_command( $2 ); }
	| error			{ yyerror( "uknown keyword" ); }
	;

bool	: L_TRUE		{ $$ = 1; }
	| L_FALSE		{ $$ = 0; }
	| error			{ yyerror( "unknown boolean value" ); }
	;

order	: L_INTEGER		{ if ( $1 < 1 || $1 > GUS_CARDS ) yyerror( "bad order value %i (must be from 1 to %i)", $1, GUS_CARDS );
                                  $$ = $1; }
	| error			{ yyerror( "uknown order value" ); }
	;

dma_size : L_INTEGER		{ if ( $1 != 0x100 && ( $1 < 0 || $1 > 128 || ( $1 % 8 ) != 0 ) ) yyerror( "bad dma size value (must be from auto,8,16,32,64 or 128)" );
				  $$ = $1; }
	| error			{ yyerror( "unknown dma size value" ); }
	;

gf1_port : L_INTEGER		{ if ( $1 != 0x100 && ( $1 < 0x210 || $1 > 0x260 || ( $1 % 0x10 ) != 0 ) ) yyerror( "bad gf1 port value (must be from auto,0x210,0x220,0x230,0x240,0x250 or 0x260)" );
				  $$ = $1; }
	| error			{ yyerror( "uknown gf1 port value" ); }
	;

gf1_irq : L_INTEGER		{ if ( $1 != 0x100 && $1 != 3 && $1 != 5 && $1 != 7 && $1 != 9 && $1 != 11 && $1 != 12 && $1 != 15 ) yyerror( "bad gf1 irq value (must be from auto,3,5,7,9,11,12,15)" );
				  $$ = $1; }
	| error			{ yyerror( "uknown gf1 irq value" ); }
	;

ess_irq : L_INTEGER		{ if ( $1 != 0x100 && $1 != 5 && $1 != 7 && $1 != 9 && $1 != 10 ) yyerror( "bad ess irq value (must be from auto,5,7,9,10)" );
				  $$ = $1; }
	| error			{ yyerror( "uknown ess irq value" ); }
	;

gf1_dma : L_INTEGER		{ if ( $1 != 0x100 && $1 != 0 && $1 != 1 && $1 != 3 && $1 != 5 && $1 != 6 && $1 != 7 ) yyerror( "bad dma value (must be from auto,0,1,3,5,6 or 7)" );
				  $$ = $1; }
	| error			{ yyerror( "unknown gf1 dma value" ); }
	;

db_port : L_INTEGER		{ if ( $1 != 0xffff && $1 != 0x530 && $1 != 0x604 && $1 != 0xe80 && $1 != 0xf40 ) yyerror( "bad gf1 port value (must be from disable,0x530,0x604,0xe80 or 0xf40)" );
				  $$ = $1; }
	| error			{ yyerror( "uknown daughter port value" ); }
	;

db_irq	: L_INTEGER		{ if ( $1 != 3 && $1 != 4 && $1 != 5 && $1 != 6 && $1 != 7 && $1 != 9 ) yyerror( "bad gf1 irq value (must be from 3,4,5,6,7,9)" );
				  $$ = $1; }
	| error			{ yyerror( "uknown daughter irq value" ); }
	;

db_dma	: L_INTEGER		{ if ( $1 != 1 && $1 != 2 && $1 != 3 ) yyerror( "bad dma value (must be from 1,2 or 3)" );
				  $$ = $1; }
	| error			{ yyerror( "unknown daughter dma value" ); }
	;

jdac	: L_INTEGER		{ if ( $1 < 0 || $1 > 31 ) yyerror( "bad joystick DAC level (must be from 0 to 31)" );
				  $$ = $1; }
	| error			{ yyerror( "unknown joystick DAC level" ); }
	;


%%

static void yyerror( char *string, ... )
{
  va_list vars;
  va_start( vars, string );
  fprintf( stderr, "insgus: error in configuration file '%s': (line %i) ", cfg_file, line_count + 1 );
  vfprintf( stderr, string, vars );
  fprintf( stderr, "\n" );
  va_end( vars );
  exit( 1 );
}

static void card_start( void )
{
  card_number++;
  card_init();
}

static void card_stop( void )
{
  card_next();
}

static void driver_command( char *command )
{
  struct driver_command *cmd, *cmd1;

  cmd = malloc( sizeof( struct driver_command ) );
  cmd -> command = command;
  cmd -> next = NULL;
  if ( !command ) return;
  if ( !card.user_data )
    card.user_data = cmd;
   else
    {
      cmd1 = card.user_data;
      while ( cmd1 -> next ) cmd1 = cmd1 -> next;
      cmd1 -> next = cmd;
    }
}
