/*
    tkmixer.C, source code of tkmixer
    Copyright (C) 1996  Radek Pospisil ( rpos1368@ss1000.ms.mff.cuni.cz )

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include <tcl.h>
#include <tk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <linux/soundcard.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
  

extern char *mix_init, *mix_cw;

// misc functions & macros
void err( int e, char *t ) {
   if ( e ) {
      if ( t ) fprintf( stderr, "%s\n", t );
      perror( "tkmixer" );
      exit( 1 );
   }
}

#define IOCTL( req, val ) {\
int ret = ioctl( hmix, (req), (val) );\
if ( ret < 0 ) {\
   fprintf( stderr,"tkmixer: ioctl error on device \"%s\"\nreason: %s\n",\
           mixdevname, strerror( errno ) );\
   exit( 1 );\
}\
}

#define TEXEC( what, errstr ) if ( (what) != TCL_OK ) {\
   fprintf( stderr, errstr": %s\n", interp->result );\
   exit( 1 );\
};

#define TEXEC2( what, errstr ) (what); if ( *interp->result != 0 ) {\
   fprintf( stderr, errstr": %s\n", interp->result );\
   exit( 1 );\
};

#define TKERR( e, t ) { if ( e ) {\
char p[200];\
sprintf( p, "after idle { .dialog1.msg configure -justify center; grab .dialog1 }; tk_dialog .dialog1 {tkmixer error} {tkmixer error\n"t"\n%s} error 0 OK\n", strerror( errno ) );\
TEXEC( Tcl_Eval( interp, p ), "restore error" );\
return;\
}}


/*
 * class MixDev - mixer device - process changes from user and set mixer device
 * to reflect it and so on
 */
class MixDev {
 private:
   typedef enum {
        I_BASE = 0,
	I_PVOL = 1, I_VVOL = 2,
	I_PBAL = 3, I_VBAL = 4, 
	I_PREC = 5, I_VREC = 6, 
	I_PMUTE = 7, I_VMUTE = 8,
   } VARS;
   
   char **names, *mixdevname;
   Tcl_Interp *interp;
   int stereo, rec;
   int left, right, recon, muteon;
   int hmix, bit;
   
 public:
   MixDev( Tcl_Interp *_interp, char *name, int _hmix, char *mdn, int _bit, 
	  int _str, int _rec, int _vol, int _recon, int _muteon ) {
      hmix = _hmix;
      bit = _bit;
      mixdevname = mdn;
      interp = _interp;
      stereo = (_str)?1:0;
      rec = (_rec)?1:0;
      char p[200];
      sprintf( p, "create_widget %s %s %p %d %d", ".", name, this,  stereo, rec );
      TEXEC( Tcl_Eval( interp, p ), "Create widget failed," );
      int argc;
      TEXEC( Tcl_SplitList( interp, interp->result, &argc, &names ), "Create widget failed" );
      setlrr( _vol, _recon );
      setmute( _muteon );
   }

   
   ~MixDev() {}
   
   
   char *getbase() {
      return names[I_BASE];
   }
   
   
   void hide() {
      char p[200];
      sprintf( p, "pack forget %s", names[I_BASE] );
      TEXEC( Tcl_Eval( interp, p ), "hide widget failed," );
   }
   
   
   void show() {
      char p[200];
      sprintf( p, "pack %s -side left -fill y -expand yes", names[I_BASE] );
      TEXEC( Tcl_Eval( interp, p ), "show widget failed," );
   }
   
   
 private:
   int update() {
      int vol, bal;
      char *val;
      val = Tcl_GetVar( interp, names[I_VVOL], TCL_GLOBAL_ONLY );
      TEXEC( Tcl_GetInt( interp, val, &vol ), "Can't convert volume" );
      val = Tcl_GetVar( interp, names[I_VBAL], TCL_GLOBAL_ONLY  );
      TEXEC( Tcl_GetInt( interp, val, &bal ), "Can't convert balance" );
      val = Tcl_GetVar( interp, names[I_VREC], TCL_GLOBAL_ONLY  );
      TEXEC( Tcl_GetBoolean( interp, val, &recon ), "Can't convert record" );
      val = Tcl_GetVar( interp, names[I_VMUTE], TCL_GLOBAL_ONLY  );
      TEXEC( Tcl_GetBoolean( interp, val, &muteon ), "Can't convert mute" );
      
      if ( bal < 0 ) {
	 left = vol;
	 right = vol+bal*vol/5;
      } else {
	 right = vol;
	 left = vol-bal*vol/5;
      }
      
      set( 0 );
      
      return TCL_OK;
   }

 public:
   void set( int setgui ) {
// set GUI      
      char p[200];
      if ( setgui ) {
	 sprintf( p, "%s set %d", names[I_PVOL], getvol() );
	 TEXEC( Tcl_Eval( interp, p ), "set failed" );
	 sprintf( p, "%s set %d", names[I_PBAL], getbal() );
	 TEXEC( Tcl_Eval( interp, p ), "set failed" );
	 sprintf( p, "if { %d } { %s select } else { %s deselect }", getrec(), names[I_PREC], names[I_PREC] );
	 TEXEC( Tcl_Eval( interp, p ), "set failed" );
	 sprintf( p, "if { %d } { %s select } else { %s deselect }", getmute(), names[I_PMUTE], names[I_PMUTE] );
	 TEXEC( Tcl_Eval( interp, p ), "set failed" );
      }
      
      char *mutev = (muteon)?"disabled":"normal";
      sprintf( p, "%s configure -state %s; %s configure -state %s; %s configure -state %s", 
	      names[I_PVOL], mutev, names[I_PBAL], mutev, names[I_PREC], mutev );
      TEXEC( Tcl_Eval( interp, p ), "mute failed" );
      
      if ( !setgui ) {
// set mixer device
	 int pm = getlr();
	 IOCTL( MIXER_WRITE( bit ), &pm );
	 IOCTL( SOUND_MIXER_READ_RECSRC, &pm );
	 pm = (pm&~(1<<bit))|((recon)?(1<<bit):0);
	 IOCTL( SOUND_MIXER_WRITE_RECSRC, &pm );
      }
   }

   void setmute( int m ) {
      muteon = (m)?1:0;
      set( 0 );
      set( 1 );
   }
   
   
   void setlrr( int v, int r, int tot = 0 ) {
      r = (r)?1:0;
      if ( !tot && getlr() == v && recon == r ) return;
      left = v>>8;
      right = v&0xff;
      recon = r;
      if ( tot ) set( 0 );
      set( 1 );
   }
   
   
   int getvol() {
      return left >? right;
   }
   
   int getbal() {
      int l = left, r = right;
      int sign = (l<r)?-1:1, bal;
      if ( sign < 0 ) {
	 bal = r; r = l; l = bal;
      }
      if ( l == 0 ) return 0;
      bal = 5*(l-r)/l;
      return sign*bal;
   }
   
   int getlr( int real = 0) {
      return (muteon && !real )?0:(right<<8)|left;
   }
   
   int getrec() {
      return recon;
   }
   
   int getmute() {
      return muteon;
   }
   
   
   static int MD_change( ClientData, Tcl_Interp *interp, int argc, char **argv ) {
      MixDev *p;
      if ( argc != 2 && argc != 3 ) {
	 Tcl_SetResult( interp, "wrong # args", TCL_STATIC );
	 return TCL_ERROR;
      }
   
      sscanf( argv[1], "%p", &p );
      if ( p == NULL ) {
	 fprintf( stderr,"Unknown Mixdev\n" );
	 exit( 1 );
      }
      return p->update();
   }
};



/*
 * class Mixer cares about MixDev classes
 */
class Mixer {
 private:
   const int view_max =  (1<<SOUND_MIXER_NRDEVICES)-1;
   const int view_min =  SOUND_MASK_VOLUME;
   int view_user, view;
   int fullrestore;
   
   static Mixer *ptr;
   static Tcl_AsyncHandler tah;
   static char *labels[];
   Tcl_Interp *interp;
   char *mixdevname;
   char *conffile;
   int hmix, mixdev, mixstdev, mixrecdev;
   MixDev *md[SOUND_MIXER_NRDEVICES];
   
 public:
   Mixer( Tcl_Interp *_interp, char *_mixdevname, int r ) {
      if ( ptr != NULL ) {
	 fprintf( stderr, "Another instance of Mixer is not allowed\n" );
	 exit( 1 );
      }
      interp = _interp;
      
// init view
      view_user = view_max;
      view = 0;
      
      fullrestore = 0;
      
// configuration file
#define CFNAME "/.tkmixerconf"      
      char *cf = getenv( "HOME" );
      err( cf==NULL, "HOME is not set" );
      conffile = new char[strlen( cf )+strlen( CFNAME )+1];
      strcat( strcpy( conffile, cf ), CFNAME );
#undef CFNAME      
      
// open mixer device
      mixdevname = strdup( _mixdevname );
      hmix = open( mixdevname, O_RDWR );
      err( hmix < 0, "error opening mixer device" );
      
      IOCTL( SOUND_MIXER_READ_DEVMASK, &mixdev );
      IOCTL( SOUND_MIXER_READ_STEREODEVS, &mixstdev );
      IOCTL( SOUND_MIXER_READ_RECMASK, &mixrecdev );
      if ( mixdev == 0 ) {      
	 fprintf( stderr, "tkmixer: there are no devices on mixer device \"%s\"\n", mixdevname );
	 exit( 1 );
      }
  
      int recon;
      IOCTL( SOUND_MIXER_READ_RECSRC, &recon );
      for ( int i = 0; ( i < SOUND_MIXER_NRDEVICES ); i++ ) {
	 if ( mixdev & ( 1<<i ) ) {
	    int v;
	    IOCTL( MIXER_READ( i ), &v );
	    md[i] = new MixDev( interp, labels[i], hmix, mixdevname, i,
			       mixstdev&(1<<i), mixrecdev&(1<<i),
			       v, recon&(1<<i), 0 );
	 } else
	   md[i] = NULL;
      }
      
      review( view_user );
      if ( r ) restore();
      
// 'unlock' update processing
      ptr = this;
   }
   
   ~Mixer() {
      close( hmix );
      delete conffile;
      for ( int i = 0; ( i < SOUND_MIXER_NRDEVICES ); i++ ) {
	 if ( md[i] ) delete md[i];
      }
   }
   
   void review( int nview ) {
      if ( nview == view ) return;
      for ( int i = 0; ( i < SOUND_MIXER_NRDEVICES ); i++ ) {
	 if ( md[i] ) md[i]->hide();
      }
      
      for ( int i = 0; ( i < SOUND_MIXER_NRDEVICES ); i++ ) {
	 if ( md[i] && nview&(1<<i) ) md[i]->show();
      }
      
      view = nview;
   }
   
   void mixupdate() {
      int r;
      IOCTL( SOUND_MIXER_READ_RECSRC, &r );
      for ( int i = 0; ( i < SOUND_MIXER_NRDEVICES ); i++ ) {
	 int v;
	 if ( md[i] ) {
	    IOCTL( MIXER_READ( i ), &v );
	    md[i]->setlrr( v, (r&(1<<i))?1:0 );
	 }
      }
   }
   
   void minimal() {
      review( view_min );
   }
   
   void maximal() {
      review( view_max );
   }
   
   void user() {
      review( view_user );
   }
   
   void save() {
      FILE *f = fopen( conffile, "wt" );
      TKERR( f == NULL, "can't save configuration" );
      TKERR( 0 == fprintf( f, "%#04x %#04x %d\n", view, view_user, fullrestore ),
	    "error saving configuration" );
	    
      for ( int i = 0; ( i < SOUND_MIXER_NRDEVICES ); i++ ) {
	 if ( md[i] ) 
	   TKERR( 0 == fprintf( f, "%#04x %d %d\n", md[i]->getlr( 1 ), md[i]->getrec(), md[i]->getmute() ),
		 "error saving configuration" )
	 else 
	   TKERR( 0 == fprintf( f, "%#04x %d %d\n", 0, 0, 0 ),
		 "error saving configuration" );
      }
      fclose( f );
   }
   
   typedef struct {
      int vol, rec, mute;
   } MixDevStore;
   
   void restore() {
      FILE *f = fopen( conffile, "rt" );
      TKERR( f == NULL, "can't load configuration" );
      int _view, _view_user, _fullrestore;
      MixDevStore *mds = new MixDevStore[ SOUND_MIXER_NRDEVICES ];
      TKERR( 3 != fscanf( f, "%x %x %d", &_view, &_view_user, &_fullrestore ),
	    "configuration file corrupted" );
      for ( int i = 0; ( i < SOUND_MIXER_NRDEVICES ); i++ )
	TKERR( 3 != fscanf( f, "%x %x %d", &(mds[i].vol), &(mds[i].rec), &(mds[i].mute) ),
	      "configuration file corrupted" );
      
      if ( (fullrestore=_fullrestore) != 0 ) {
	 for ( int i = 0; ( i < SOUND_MIXER_NRDEVICES ); i++ )
	   if ( md[i] ) {
	      md[i]->setmute( 0 );
	      md[i]->setlrr( mds[i].vol, mds[i].rec, 1 );
	      md[i]->setmute( mds[i].mute );
	   }
      }
      view_user = _view_user;
      review( _view );
      
      delete mds;
      fclose( f );
   }
   
   
   void configure() {
      static char *nn[] = SOUND_DEVICE_NAMES;
      char p[4000], *pp = p;
      pp += sprintf( pp, "after idle { grab .config }\n" );
      pp += sprintf( pp, "set tc [toplevel .config]\n" );
      pp += sprintf( pp, "wm resizable $tc 0 0\n" );
      
      int cnt = 0;
      for ( int j = 0; ( j < SOUND_MIXER_NRDEVICES ); j++ ) 
	if ( md[j] ) cnt++;
 
      
      pp += sprintf( pp, "\
set fm [frame $tc.fm]\n\
pack $fm -fill both -side left\n\
label $fm.lbl -text \"What you want to see?\"\n\
pack $fm.lbl -fill both -side top\n" );
      
      pp += sprintf( pp, "checkbutton $fm.rest -text \"keep volume setting\" -indicatoron 0 -selectcolor \"\"\n" );
      pp += sprintf( pp, "$fm.rest %s\n", ((fullrestore)?"select":"deselect") );
      pp += sprintf( pp, "pack $fm.rest -fill both -side bottom\n" );
      
      pp += sprintf( pp, "set fms [frame $fm.fms]\n" );
      pp += sprintf( pp, "pack $fms -fill both -side top\n" );

      pp += sprintf( pp, "set fml [frame $fms.fml]\n" );
      pp += sprintf( pp, "pack $fml -fill both -side left\n" );
      
      int j = 0, i = 0;
      for (; ( i <= cnt/2 ); j++ ) {
	 if ( md[j] ) {
	    i++;
	    pp += sprintf( pp, "checkbutton $fml.%s -width 9 -text %s -indicatoron 0 -selectcolor \"\"\n",
			  nn[j], nn[j] );
	    pp += sprintf( pp, "$fml.%s %s\n", 
			  nn[j], ((view_user&(1<<j))?"select":"deselect") );
	    pp += sprintf( pp, "pack $fml.%s -fill x -side top\n", nn[j] );
	 }
      }

      pp += sprintf( pp, "set fmr [frame $fms.fmr]\n" );
      pp += sprintf( pp, "pack $fmr -fill both -side left\n" );
      for (; ( j < SOUND_MIXER_NRDEVICES ); j++ ) {
	 if ( md[j] ) {
	    pp += sprintf( pp, "checkbutton $fmr.%s -width 9 -text %s -indicatoron 0 -selectcolor \"\"\n",
			  nn[j], nn[j] );
	    pp += sprintf( pp, "$fmr.%s %s\n", 
			  nn[j], ((view_user&(1<<j))?"select":"deselect") );
	    pp += sprintf( pp, "pack $fmr.%s -fill x -side top\n", nn[j] );
	 }
      }

      pp += sprintf( pp, "set fm2 [frame $tc.fm2]\n" );
      pp += sprintf( pp, "pack $fm2 -side bottom -fill both\n" );

      pp += sprintf( pp, "button $fm2.btok -text ok -command { MD_confack }\n" );
      pp += sprintf( pp, "pack $fm2.btok -fill x\n" );

      pp += sprintf( pp, "button $fm2.btcn -text cancel -command { destroy $tc }\n" );
      pp += sprintf( pp, "pack $fm2.btcn -fill x\n" );

      TEXEC( Tcl_Eval( interp, p ), "configure error" );
   }

   
   void confack() {
      static char *nn[] = SOUND_DEVICE_NAMES;
      char p[1000], *v;
      int x;
      view_user = 0;
      for ( int i = 0 ; ( i < SOUND_MIXER_NRDEVICES ); i++ ) {
	 if ( md[i] ) {
	    v = Tcl_GetVar( interp, nn[i], TCL_GLOBAL_ONLY );
	    TEXEC( Tcl_GetBoolean( interp, v, &x ), "config error" );
	    view_user |= (x)?(1<<i):0;
	 }
      }
      
      v = Tcl_GetVar( interp, "rest", TCL_GLOBAL_ONLY );
      TEXEC( Tcl_GetBoolean( interp, v, &fullrestore ), "config error" );

      user();
      
      sprintf( p, "destroy .config\n" );
      TEXEC( Tcl_Eval( interp, p ), "configure error" );
   }
   
   
#define DEFCMD( name ) static \
int MD_##name( ClientData, Tcl_Interp *, int, char ** ) {\
if ( ptr != NULL ) ptr->name();\
return TCL_OK; }
   
DEFCMD( mixupdate );   
DEFCMD( minimal );   
DEFCMD( maximal );   
DEFCMD( user );   
DEFCMD( save );   
DEFCMD( restore );
DEFCMD( configure );
DEFCMD( confack );
#undef DEFCMD   
};

Mixer *Mixer::ptr = NULL;
char *Mixer::labels[] = SOUND_DEVICE_LABELS;
int restor = 0;
char *mixdev = "/dev/mixer";
char **pargv;

Tk_ArgvInfo argTable[] = {
     { "-r", TK_ARGV_CONSTANT, (char*)1, (char*)&restor,
	  "restore setting when starting mixer" },
     
     { "-m", TK_ARGV_STRING, NULL, (char*)&mixdev,
	  "Name of mixer device ( def /dev/mixer )"},
     
     {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL,
                      (char *) NULL }
};


/*
 * My Tcl/Tk application init
 */
int MyAppInit( Tcl_Interp *interp ) {
// parse command line arguments
   int argc = 0;
   for (; ( pargv[argc] ); argc++ );
   TEXEC( Tk_ParseArgv(interp, Tk_MainWindow( interp ), &argc, pargv, argTable, 0), "" );
   
// init tcl/tk   
   TEXEC( Tcl_Init( interp ), "Tcl_Init failed" );
   TEXEC( Tk_Init( interp ), "Tk_Init failed" );

// create commands
   TEXEC2( Tcl_CreateCommand( interp, "MD_change", MixDev::MD_change, NULL, NULL ), "Create command failed" );
   TEXEC2( Tcl_CreateCommand( interp, "MD_mixupdate", Mixer::MD_mixupdate, NULL, NULL ), "Create command failed" );
   TEXEC2( Tcl_CreateCommand( interp, "MD_minimal", Mixer::MD_minimal, NULL, NULL ), "Create command failed" );
   TEXEC2( Tcl_CreateCommand( interp, "MD_maximal", Mixer::MD_maximal, NULL, NULL ), "Create command failed" );
   TEXEC2( Tcl_CreateCommand( interp, "MD_user", Mixer::MD_user, NULL, NULL ), "Create command failed" );
   TEXEC2( Tcl_CreateCommand( interp, "MD_save", Mixer::MD_save, NULL, NULL ), "Create command failed" );
   TEXEC2( Tcl_CreateCommand( interp, "MD_restore", Mixer::MD_restore, NULL, NULL ), "Create command failed" );
   TEXEC2( Tcl_CreateCommand( interp, "MD_configure", Mixer::MD_configure, NULL, NULL ), "Create command failed" );
   TEXEC2( Tcl_CreateCommand( interp, "MD_confack", Mixer::MD_confack, NULL, NULL ), "Create command failed" );

// init GUI
   TEXEC( Tcl_Eval( interp, mix_init ), "Init failed" );

   new Mixer( interp, mixdev, restor );
   
// run tk engine   
   Tk_MainLoop();

// for good feeling...   
   TEXEC( Tcl_Eval( interp, "exit 0" ), "exit failed" );
   
   return TCL_OK;
}

int main( int argc, char **argv ) {
   pargv = argv;
   Tk_Main( argc, argv, MyAppInit );
   return 0;
}



/*
 * Init script
 */

char *mix_init = "
# 
# mixer update
#
proc mix_update {} {
   MD_mixupdate
   after 200 mix_update
}

  
#
# widget proc   
#
proc create_widget { wbase name ptr stereo rec } {
if { $wbase == \".\" } { set wbase \"\" }
set base [ frame $wbase.frame_$name ]
set recvar \"record_$name\"
set mutevar \"mute_$name\"
set volvar \"vol_$name\"
set balvar \"bal_$name\"
# create mute button
set mute [ checkbutton $base.mute \
       -indicatoron 0 \
       -selectcolor \"\" \
       -text $name \
       -variable $mutevar \
       -width 5 \
       -disabledforeground \"\" \
       -command \"MD_change $ptr\" ]

# create scale for volume adjusting
set volume [ scale $base.volume \
	-bigincrement 0.0 \
	-from 100 \
	-orient v \
	-resolution 1.0 \
	-showvalue 0 \
	-sliderlength 20 \
	-tickinterval 0.0 \
	-to 0 \
	-width 10 \
        -relief groove \
	-length 125 \
	-variable $volvar \
	-command \"MD_change $ptr\" ]


# create scale for balance
set balance [ scale $base.balance \
	-bigincrement 0.0 \
	-from -5.0 \
	-length 25 \
	-orient h \
	-resolution 1.0 \
	-showvalue 0 \
	-sliderlength 10 \
	-tickinterval 0.0 \
	-to 5.0 \
	-width 10 \
        -relief groove \
	-variable $balvar \
 	-command \"MD_change $ptr\" ]

# create rec button
set record [ checkbutton $base.rec \
       -selectimage recon \
       -image recoff \
       -indicatoron 0 \
       -selectcolor \"\" \
       -variable $recvar \
       -borderwidth 1 \
       -height 9 \
       -width 15 \
       -disabledforeground \"\" \
       -command \"MD_change $ptr\" ]

# bind 
bind $balance <ButtonRelease-2> \"$balance set 0\"
bind $balance <ButtonPress-2> \"$balance set 0\"
bind $balance <KeyPress-c> \"$balance set 0\"

#pack $base -side left -fill both -expand yes
pack $mute -fill both
pack $volume

upvar #0 bwidth bwidth bheight bheight     
if { $stereo } { pack $balance -fill both }
if { $rec } { pack $record -side bottom -fill both }
   
return [list $base $volume $volvar $balance $balvar $record $recvar $mute $mutevar ]
}
 
#  
#init proc
#
uplevel #0 {
tk appname mixer
wm title . \"mixer\"
wm resizable . 0 0

# create images for rec button
set recon \"
#define recon_width 13
#define recon_height 7
static unsigned char recon_bits[] = {
   0x00, 0x00, 0xe7, 0x0c, 0x29, 0x02, 0x67, 0x02, 0x25, 0x02, 0xe9, 0x0c,
   0x00, 0x00};
\"
set recoff \"
#define recoff_width 13
#define recoff_height 7
static unsigned char recoff_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00};
\"
image create bitmap recon -data $recon -maskdata $recon
image create bitmap recoff -data $recoff -maskdata $recoff

# create menu     
set mpop [ menu .mpop -tearoff 0 ]
$mpop add command -label \"Save setting\" -underline 0 -accelerator C-s -command { MD_save }
$mpop add command -label \"Restore setting\" -underline 0 -accelerator C-r -command { MD_restore }
$mpop add separator
$mpop add command -label \"Minimal view\" -underline 2 -accelerator C-n -command { MD_minimal }
$mpop add command -label \"Maximal view\" -underline 2 -accelerator C-x -command { MD_maximal }
$mpop add command -label \"User view\" -underline 0 -accelerator C-u -command { MD_user }   
$mpop add separator
$mpop add command -label \"Configure\" -underline 0 -accelerator C-c -command { MD_configure }
$mpop add separator
$mpop add command -label \"Quit\" -underline 0 -accelerator C-q -command { exit 0 }
$mpop add separator
$mpop add command -label \"Help\" -underline 0 -accelerator C-h -command { help }

bind . <Control-s> { $mpop invoke Save* }
bind . <Control-r> { $mpop invoke Rest* }
bind . <Control-n> { $mpop invoke Min* }
bind . <Control-x> { $mpop invoke Max* }
bind . <Control-u> { $mpop invoke User*}
bind . <Control-c> { $mpop invoke Conf* }
bind . <Control-q> { $mpop invoke Quit* }
bind . <Control-h> { $mpop invoke Help* }
   
# bind menu popup events   
bind . <Button-3> { tk_popup $mpop %X %Y }
bind . <Control-KeyPress-space> { tk_popup $mpop %X %Y }
   
# run mixer update
after 100 mix_update   
}

proc help {} {
if { [catch { toplevel .help }] } { raise .help; return }
set fm .help
button .help.btok -text ok -command { destroy .help }
set bf [ .help.btok cget -font ]   
text $fm.text -wrap word -yscrollcommand \"$fm.sb set\" \
      -font $bf
scrollbar $fm.sb -command \"$fm.text yview\"
     
$fm.text insert 0.0 {
tkmixer v1.0
(c) 1996 Radek Pospisil
email:rpos1368@ss1000.ms.mff.cuni.cz
} t1
   
$fm.text insert end {
features:
} t2
$fm.text insert end {
   - volume ( 1-100 ) & balance ( -5..5 ) setting
   - mute control
   - record source control
   - configureable view 
   - save/restore view setting and mixer device setting ( volume, balance, mute & record source )
   - full mouse and/or keyboard control ( yes, you can use tkmixer without mouse )
}
   
$fm.text insert end {
mouse binding:
} t3
$fm.text insert end {
   - left button control mute and record checkbuttons and volume and balance scales
   - middle button reset balance to 0 
   - right button wake up popup menu
}
   
$fm.text insert end {
keys binding:
} t4
$fm.text insert end {
   - (Shift+)TAB cycle throw all widgets
   - SPACE or Enter on/off mute and record checkbutton
   - cursor arrows control scales of volume and balance
   - C on balance reset balance to 0 ( center )
   - Ctrl+SPACE wake up popup menu
   - Ctrl+s save setting
   - Ctrl+r restore setting
   - Ctrl+n minmum view
   - Ctrl+x maximum view
   - Ctrl+u user view
   - Ctrl+c run configure dialog
   - Ctrl+h show help window
   - Ctrl+q QUIT tkmixer
}

foreach t { t2 t3 t4 } {
   $fm.text tag configure $t -underline 1 -font -*-Courier-Bold-O-Normal--*-120-*-*-*-*-*-*
}
   
$fm.text tag configure t1 -justify center

$fm.text configure -state disabled     
     
pack .help.btok -side bottom -fill x 
pack $fm.sb -side right -fill y
pack $fm.text -side left -fill both -expand y
}
";
