// datarep.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include <assert.h>
#ifdef NeXT
#include <m68k/math.h>
#else
#include <values.h>
#endif

#include "byteswap.h"
#include "compressor.h"
#include "datafile.h"
#include "localdefs.h"
#include "iofunction.h"
#include "range.h"
#include "repclone.h"
#include "randomnoise.h"
#include "smartmem.h"

DataRep::Array::Array(int size, int length) : memPtr(nil), bytes(0) {
	changeSize(size, length);
}

DataRep::Array::~Array() { SmartMemory::free(memPtr, bytes); }

unsigned 
DataRep::Array::changeSize(int size, int requestedLen) {
	int newlen = requestedLen;
	unsigned newbytes = newlen * size;
	if(memPtr == nil)			// just alloc if not yet alloc'd
		memPtr = addr(SmartMemory::allocate(&newbytes));
	else memPtr = addr(SmartMemory::changeAllocation(memPtr, bytes, &newbytes));
	if(newbytes == bytes) 		// failure
		return 0;
	else
		return bytes = newbytes;
}

//********

// static initialization of ctor tables for DataRep

DataRep::Constructor
DataRep::ctor_table[] = {
	nil,
	SignedCharDataRep::new_Rep,
	UnsignedCharDataRep::new_Rep,
	MuLawDataRep::new_Rep,
	ShortDataRep::new_Rep,
	IntDataRep::new_Rep,
	FloatDataRep::new_Rep,
	DoubleDataRep::new_Rep
};

inline int typeToIndex(DataType type) {
	return powerOfTwoToLinearEnum((int) type);
}

// static meta-constructors

DataRep *
DataRep::create(DataType type, int length, int chans) {
	static const int tableEnd = sizeof(ctor_table) / sizeof(*ctor_table);
	int index = typeToIndex(type);
	assert(index < tableEnd && index > 0);
	return (*ctor_table[index])(length, chans);
}

DataRep::DataRep(int size, int length, int chans)
		: data_size(short(size)), len(length), nchans(chans), offset(0) {
	a = new Array(size, length * chans);
	a->ref();
	if(a->size() < sizeInBytes())	// couldnt allocate whole thing
		len = a->size() / (data_size * nchans);
}

DataRep *
DataRep::newRep() {
	return newLengthRep(length());
}

DataRep *
DataRep::clone(const Range& selection) {
	return clone(selection, Range(0, nchans-1));
}

// protected ctor used only by derived classes for making clones

DataRep::DataRep(const DataRep* rep)
		: data_size(rep->data_size), len(rep->len), nchans(rep->nchans),
		  offset(rep->offset), a(rep->a) {
	a->ref();
}

DataRep::~DataRep() {
	Resource::unref(a);
} 

int
DataRep::sizeInBytes() const {
	return length() * channels() * dataSize();
}

int
DataRep::frameOffset() {
	return int(offset/(dataSize() * nchans));
}

int
DataRep::read(DataFile *f) {
	int status = false;
	addr inbuff = getAddressOfContiguousData();
	if(inbuff) {
		f->read(inbuff, min(f->readSize(), sizeInBytes()));
		if((status = f->good()) == true) {
			if(f->bytesAreSwapped())
				swapBytes();
		}
		else {
			char msg[64];
			sprintf(msg, "%s%s", "Error reading data from disk",
				f->iocount() ? "." : ":  unexpected EOF");
			Application::alert(msg);
		}
	}
	else Application::alert("DataRep::read: contiguous data not available.");
	return status;
}

// for the moment, the DataRep swaps its bytes internally before writing to
// disk (if necessary), and then swaps them back.  This will be done more
// efficiently in the future.

int
DataRep::write(DataFile *f) const {
	int status = false;
	addr outbuff = getAddressOfContiguousData();
	if(outbuff) {
		if(f->bytesAreSwapped())
			((DataRep *) this)->swapBytes();	// casting away const
		f->write(outbuff, sizeInBytes());
		if(f->bytesAreSwapped())
			((DataRep *) this)->swapBytes();	// FIX ME -- inefficient!
		if((status = f->good()) != true)
			f->error();
	}
	else Application::alert("DataRep::write: contiguous data not available.");
	return status;
}

void
DataRep::copyFrom(const DataRep *source) {
	if(source == this) return;
	int chans = min(source->channels(), channels());
	int end = min(source->length(), length());
	for (int chan = 0; chan < chans; ++chan) {
		Handle src_h = source->getHandle(0, chan);
		Handle dest_h = getHandle(0, chan);
		for (int j = 0; j < end; ++j) {
			double vs;
			src_h = source->getAndAdvance(src_h, &vs);
			dest_h = setAndAdvance(dest_h, vs);
		}
	}
}

void
DataRep::copyRescaledFrom(const DataRep *source, double scalingFactor, boolean dither) {
	if(source == this) return;
	int chans = min(source->channels(), channels());
	int end = min(source->length(), length());
	RandomNoise r1(23, .5);
	RandomNoise r2(17, .5);
	for (int chan = 0; chan < chans; ++chan) {
		Handle src_h = source->getHandle(0, chan);
		Handle dest_h = getHandle(0, chan);
		for (int j = 0; j < end; ++j) {
			double vs;
			src_h = source->getAndAdvance(src_h, &vs);
			double ditherAmp = dither ? r1() + r2() : 0;
			dest_h = setAndAdvance(dest_h, vs * scalingFactor + ditherAmp);
		}
	}
}

// retrieve an interpolated location value 
// this calls the virtual get(int,int) functions

double
DataRep::get(double loc, int chan) const {
	int iloc = int(loc);
	double fraction = loc - iloc;
	double current = get(iloc, chan);
	return current + fraction * (get(iloc + 1, chan) - current);
}

// load one frame (all channels for single horizontal moment) into double array

void
DataRep::getFrame(float *array, int arraySize, int frameno) const {
	int chansToRead = min(arraySize, channels());
	Handle h = getHandle(frameno);
	int chan;
	for(chan = 0; chan < chansToRead; ++chan) {
		double v;
		h = getAndShift(h, &v);
		*array++ = v;
	}
	for (int loc = chan; loc < arraySize; loc++)
		*array++ = 0;	// zero remainder, if any
}

// reverse:  set values in one frame

void
DataRep::setFrame(float *array, int arraySize, int frameno) {
	int chansToWrite = min(arraySize, channels());
	Handle h = getHandle(frameno);
	for(int chan = 0; chan < chansToWrite; ++chan)
		h = setAndShift(h, *array++);
}

void
DataRep::getFrame(double *array, int arraySize, int frameno) const {
	int chansToRead = min(arraySize, channels());
	Handle h = getHandle(frameno);
	int chan;
	for(chan = 0; chan < chansToRead; ++chan)
		h = getAndShift(h, array++);
	for (int loc = chan; loc < arraySize; loc++)
		*array++ = 0;	// zero remainder, if any
}

// reverse:  set values in one frame

void
DataRep::setFrame(double *array, int arraySize, int frameno) {
	int chansToWrite = min(arraySize, channels());
	Handle h = getHandle(frameno);
	for(int chan = 0; chan < chansToWrite; ++chan)
		h = setAndShift(h, *array++);
}

// load one frame (every incr channels for single horizontal moment)

void
DataRep::getFrame(DataRep *frame, int frameno, int offset, int incr) const {
	int trueChans = channels();
	int flen = frame->length();
	Handle h = getHandle(frameno, offset);
	Handle frame_h = frame->getHandle();
	for (int chan = offset, outloc=0;
			chan < trueChans && outloc < flen; chan += incr, ++outloc) {
		double v;
		h = getAndShift(h, &v, incr);
		frame_h = frame->setAndAdvance(frame_h, v);
	}
}

void
DataRep::setFrame(const DataRep *frame, int frameno, int incr) {
	int chansToWrite = min(frame->length(), channels());
	Handle h = getHandle(frameno, offset);
	Handle frame_h = frame->getHandle();
	for (int chan = 0; chan < chansToWrite; chan += incr) {
		double v;
		frame_h = frame->getAndAdvance(frame_h, &v);
		h = setAndShift(h, v, incr);
	}
}

void
DataRep::add(const DataRep *source) {
	int chan, chans = min(source->channels(), channels());
	int end = min(source->length(), length());
	for (chan = 0; chan < chans; ++chan) {
		Handle src_h = source->getHandle(0, chan);
		Handle dest_h = getHandle(0, chan);
		for (int j = 0; j < end; ++j) {
			double vs, vd;
			src_h = source->getAndAdvance(src_h, &vs);
			getAndAdvance(dest_h, &vd);
			dest_h = setAndAdvance(dest_h, vs + vd);
		}
	}
}

int
DataRep::getArray(int *array, int size, int channel) const {
	Handle h = getHandle(0, channel);
	int end = min(size, length());
	int i;
	for(i = 0; i < end; i++) {
		double out;
		h = getAndAdvance(h, &out);
		*array++ = int(out);
	}
	while(i++ < size) *array++ = 0;	// zero out remainder
	return end;
}

int
DataRep::getArray(float *array, int size, int channel) const {
	Handle h = getHandle(0, channel);
	int end = min(size, length());
	int i;
	for(i = 0; i < end; i++) {
		double out;
		h = getAndAdvance(h, &out);
		*array++ = out;
	}
	while(i++ < size) *array++ = 0.0;	// zero out remainder
	return end;
}

int
DataRep::getArray(double *array, int size, int channel) const {
	return getArray(array, size, channel, 0);
}

int
DataRep::getArray(double *array, int size, int channel, int offset) const {
	Handle h = getHandle(offset, channel);
	int end = min(size, length() - offset);
	int i;
	for(i = 0; i < end; i++)
		h = getAndAdvance(h, array++);
	while(i++ < size) *array++ = 0.0;
	return end;
}

int
DataRep::setFromArray(float *array, int size, int channel, int offset) {
	Handle h = getHandle(offset, channel);
	int end = min(size, length() - offset);
	for(int i = 0; i < end; i++)
		h = setAndAdvance(h, *array++);
	return end;
}

int
DataRep::setFromArray(double *array, int size, int channel, int offset) {
	Handle h = getHandle(offset, channel);
	int end = min(size, length() - offset);
	for(int i = 0; i < end; i++)
		h = setAndAdvance(h, *array++);
	return end;
}

void
DataRep::valueRange(Range &r) const {
	valueRange(r, 0, realLength() - 1, 0, realChannels() - 1, true);
}

double
DataRep::rmsAmplitude(int start, int end, int chfirst, int chlast) const {
	double mean = 0.0;
	long nvals = 0;
	for(int chan = chfirst; chan <= chlast; chan++) {
		for(int i = start; i <= end; i++) {
			double sig = get(i, chan);
			mean += sig * sig;
			nvals++;
		}
	}
	return nvals ? sqrt(mean/nvals) : 0.0;
}

// extract max, min peak values from data and store in another data object
// if EnvelopeType is Decibels, amps are scaled into Decibels between
// 0 dB (minimum 16-bit level) and ~90 dB (max 16-bit level)
// return value is number of channels in envelope

int
DataRep::getEnvelope(DataRep *envelope, int channel, EnvelopeType etype, double reflevel) const {
	int numberOfSections = envelope->length();
	double realIncrement = length() / double(numberOfSections);
	double currentVal = 0, nextVal = 0;
	register int currentLoc, nextLoc;
	int oldLoc = -1;
	Range currentRange;
	double minLevel = reflevel/32767.0;
	static double maxDecibel = -20.0 * log10(1.0/32767.0);
	register DataRep *envrep = envelope;
	Handle h0 = envrep->getHandle(0, 0);
	Handle h1 = envrep->getHandle(0, 1);
	for(int i = 0; i < numberOfSections; i++) {
		nextVal = currentVal + realIncrement;
		currentLoc = int(currentVal);
		if(int(currentLoc) != oldLoc) {
			nextLoc = (int(nextVal) > currentLoc) ? 
				int(nextVal) - 1 : int(nextVal);
			if(etype != RMSAmplitude)
				valueRange(currentRange, currentLoc, nextLoc, channel, channel);
			else
				currentRange.set(0.0, rmsAmplitude(currentLoc, nextLoc, channel, channel));
			currentLoc = oldLoc;
		}
		switch(etype) {
		case PeakToPeakAmplitude:
			h0= envrep->setAndAdvance(h0, currentRange.min());
			h1= envrep->setAndAdvance(h1, currentRange.max());
			break;
		case AbsoluteMagnitude:
		case RMSAmplitude:
			h0=envrep->setAndAdvance(h0,currentRange.absoluteMax());
			break;
		case Decibels:
			double absmax;
			absmax = max(minLevel, currentRange.absoluteMax());
			h0=envrep->setAndAdvance(h0,
				maxDecibel + 20.0 * log10(absmax/reflevel));
			break;
		default:
			Application::die("Invalid Envelope type.");
			break;
		};			
		currentVal = nextVal;
	}
	return (etype == PeakToPeakAmplitude && realIncrement > 1.0) ? 2 : 1;
}
	
int
DataRep::spliceIn(const Range& selection) {
	int start = selection.intMin();
	int slength = int(selection.spread());
	return shiftOver(start, slength);
}

int
DataRep::spliceOut(const Range& selection) {
	int start = selection.intMin();
	int slength = selection.size();
	return shiftOver(start, -slength);
}

int
DataRep::shiftOver(int start, int shift) {
	assert(start >= 0);
	addr array = nil;
	int status = true;
	int newlen = (start < realLength()) ? 
		realLength() + shift : start + shift;
	int oldsize = sizeInBytes();
	if(newlen < 1) {
		Application::alert("DataRep::shiftOver error: new length < 1");
		status = false;
	}
	if(shift > 0 && !changeLength(newlen))
		status = false;
	if(status && shift != 0 && (array = getAddressOfContiguousData()) != nil) {
		// make sure blocksize does not use clone settings
		int blocksize = realChannels() * dataSize();	// NOTE: was channels()
		int bshift = shift * blocksize;
		int offset = start * blocksize;
		int bstart = offset - ((bshift < 0) ? bshift : 0);
		int bend = bstart + bshift;
		int bytesToMove = oldsize - bstart;
		if(bytesToMove)
			SmartMemory::copy(array + bstart, array + bend, bytesToMove);
		if(bshift > 0) zero(array + bstart, bshift);
	}
	if(status && shift < 0 && !changeLength(newlen))
		status = false;
	return status;
}

void
DataRep::zero(addr startAddress, int bytes) {
	SmartMemory::zero(startAddress, bytes, zeroValue());
}

int
DataRep::applyFunction(SimpleFunction& fun, int start, int at_chan) {
	int end = length();
	Handle h = getHandle(start, at_chan);
	Handle dest_h = h;
	// if start >= end, this falls through -- automatic safety
	for (int j = start; j < end; ++j) {
		double vs;
		h = getAndAdvance(h, &vs);
		dest_h = setAndAdvance(dest_h, fun(vs));
	}
	return true;
}

int
DataRep::giveSamplesTo(InputOutputFunction &fun) const {
	BUG("DataRep::giveSamplesTo()");
	int end = length();
	Index& indx = fun.getIndex();
	if(indx.chan < channels()) {
		Handle h = getHandle(indx.loc, indx.chan);
		while(indx.loc < end) {
			double vs;
			h = getAndAdvance(h, &vs);
			++indx.loc;
			if(fun.addToInQueue(vs) != true) // if queue full
				return indx.loc < end ? true : false;
		}
		++indx.chan;
	}
	return false;				// indicates no more samples to read
}

RunStatus
DataRep::getSamplesFrom(OutputFunction& fun, int start, int at_chan) {
	BUG("DataRep::getSamplesFrom()");
	int end = length();
	RunStatus status = Run;
	Handle h = getHandle(start, at_chan);
	for(int j = start; j < end && status == Run; ++j)
		h = setAndAdvance(h, fun(&status));
	return status;
}

int
DataRep::checkLength(int end) {
	int realend = end + frameOffset();
	return (realend > realLength()) ? changeLength(realend) : 1;
}

// user gets error mesg, but array still usable if realloc fails

int
DataRep::changeLength(int newlen) {
	int status = true;
	unsigned newbytes = array().changeSize(dataSize(), newlen * nchans);
	if(newbytes > 0)		// until Exceptions enabled
		setLength(int(newbytes/(dataSize() * nchans)));
	else
		status = false;
	return status;
}

void
DataRep::changeNChans(int newchans) {
	int oldchans = nchans;
	double ratio = oldchans / double(newchans);
	nchans = newchans;
	setLength(int(len * ratio));
}

//////////////////////////////////

CharDataRep::CharDataRep(int length, int chans, boolean doZero)
	: DataRep(sizeof(char), length, chans) {
		if(doZero)
			zero(array()[0], length*chans*dataSize());
}

void
CharDataRep::reverse(){
	for(int chan = 0; chan < channels(); chan++) {
		char *front = (char *) getHandle(0, chan);
		char *back = (char *) getHandle(length() - 1, chan);
		while(front < back) {
			char tmp = *front;
			*front = *back;
			*back = tmp;
			front += nchans;
			back -= nchans;
		}
	}
}

int
CharDataRep::changeLength(int newlen) {
	int oldlen = length();
	int status = true;
	if((status = DataRep::changeLength(newlen)) && newlen > oldlen) {
		int oldEndByte = oldlen * nchans * dataSize();
		int newEndByte = newlen * nchans * dataSize();
		zero(array()[oldEndByte], newEndByte - oldEndByte);
	}
	return status;
}

//////////////////////////////////

SignedCharDataRep::SignedCharDataRep(int length, int chans)
	: CharDataRep(length, chans) {}

DataRep *
SignedCharDataRep::newLengthRep(int newlen) {
	return new SignedCharDataRep(newlen, channels());
}

// static constructors used to initialize virtual ctor arrays

DataRep*
SignedCharDataRep::new_Rep(int len, int chans) {
	return new SignedCharDataRep(len, chans);
}

// cloning (shallow copying) routines

DataRep *
SignedCharDataRep::clone(const Range& selection, const Range& chans) {
	return new Cloned<SignedCharDataRep>(this, selection, chans);
}

// iteration routines

void
SignedCharDataRep::set(double val, int pos, int chan) {
	*((char *) getHandle(pos, chan)) = (Schar(val));
}

double
SignedCharDataRep::get(int pos, int chan) const {
	Schar *ptr = (Schar *) (char *) getHandle(pos, chan); 
	return(pos < len ? *ptr : 0);
}

DataRep::Handle
SignedCharDataRep::setAndShift(DataRep::Handle h, double val, int incr) {
	Schar *ap = (Schar *) (char *) h; *ap = (Schar(val));
	return (ap += incr);
}

DataRep::Handle
SignedCharDataRep::getAndShift(DataRep::Handle h, double *val, int incr) const {
	Schar *ap = (Schar *)(char *)h; *val = *ap;
	return (ap += incr);
}

void
SignedCharDataRep::valueRange(Range &r, int start, int end, int chfirst, int chlast, boolean real) const {
	double max = -128, min = 127;
	for(int chan = chfirst; chan <= chlast; chan++) {
		register Schar *ptr = (Schar *) (char *) getHandle(start, chan);
		for(int i = start; i <= end; i++) {
			double val = *ptr;
			max = val > max ? val : max;
			min = val < min ? val : min;
			ptr += nchans;
		}
	}
	r.set(min, max);
}

//////////////////////////////////

inline Schar changeSigned(Uchar u) {
	return (Schar) (u ^ 0x80);
}

UnsignedCharDataRep::UnsignedCharDataRep(int length, int chans)
		: CharDataRep(length, chans, false /* dont zero */ ) {
	zero(array()[0], length*chans*dataSize());
}

// since '0' is not zero when using unsigned chars, we must
// explicitly zero new data

int
UnsignedCharDataRep::zeroValue() const {
	return changeSigned(0);
}

DataRep *
UnsignedCharDataRep::newLengthRep(int newlen) {
	return new UnsignedCharDataRep(newlen, channels());
}

// static constructors used to initialize virtual ctor arrays

DataRep*
UnsignedCharDataRep::new_Rep(int len, int chans) {
	return new UnsignedCharDataRep(len, chans);
}

// cloning (shallow copying) routines

DataRep *
UnsignedCharDataRep::clone(const Range& selection, const Range& chans) {
	return new Cloned<UnsignedCharDataRep>(this, selection, chans);
}

// iteration routines

void
UnsignedCharDataRep::set(double val, int pos, int chan) {
	*((char *) getHandle(pos, chan)) = (char) changeSigned(Uchar(val));
}

double
UnsignedCharDataRep::get(int pos, int chan) const {
	Uchar *ptr = (Uchar *) (char *) getHandle(pos, chan); 
	return(pos < len ? changeSigned(*ptr) : 0);
}

DataRep::Handle
UnsignedCharDataRep::setAndShift(DataRep::Handle h, double val, int incr) {
	char *ap = (char *)h; *ap = changeSigned(char(val));
	return (ap += incr);
}

DataRep::Handle
UnsignedCharDataRep::getAndShift(DataRep::Handle h, double *val, int incr) const {
	char* ap = (char *)h; *val = changeSigned(*ap);
	return (ap += incr);
}

void
UnsignedCharDataRep::valueRange(Range &r, int start, int end, int chfirst, int chlast, boolean real) const {
	double max = -128, min = 127;
	for(int chan = chfirst; chan <= chlast; chan++) {
		register Uchar *ptr = (Uchar *) (char *) getHandle(start, chan);
		for(int i = start; i <= end; i++) {
			double val = changeSigned(*ptr);
			max = val > max ? val : max;
			min = val < min ? val : min;
			ptr += nchans;
		}
	}
	r.set(min, max);
}

//////////////////////////////////

MuLawDataRep::MuLawDataRep(int length, int chans)
		: CharDataRep(length, chans, false /* dont zero */ ) {
	zero(array()[0], length*chans*dataSize());
}

// since '0' is not zero when mu-law encoded, we must
// explicitly zero new data

int
MuLawDataRep::zeroValue() const {
	return Compressor::toMuLaw(0);
}

DataRep *
MuLawDataRep::newLengthRep(int newlen) {
	return new MuLawDataRep(newlen, channels());
}

// static constructors used to initialize virtual ctor arrays

DataRep*
MuLawDataRep::new_Rep(int len, int chans) {
	return new MuLawDataRep(len, chans);
}

// cloning (shallow copying) routines

DataRep *
MuLawDataRep::clone(const Range& selection, const Range& chans) {
	return new Cloned<MuLawDataRep>(this, selection, chans);
}

// iteration routines

void
MuLawDataRep::set(double val, int pos, int chan) {
	*((char *) getHandle(pos, chan)) = Compressor::toMuLaw(round(val));
}

double
MuLawDataRep::get(int pos, int chan) const {
	char *ptr = (char *) getHandle(pos, chan); 
	return(pos < len ? Compressor::fromMuLaw(*ptr) : zeroValue());
}

DataRep::Handle
MuLawDataRep::setAndShift(DataRep::Handle h, double val, int incr) {
	char *ap = (char *)h;
	*ap = Compressor::toMuLaw(round(val));
	return (ap += incr);
}

DataRep::Handle
MuLawDataRep::getAndShift(DataRep::Handle h, double *val, int incr) const {
	char *ap = (char *)h;
	*val = Compressor::fromMuLaw(*ap);
	return (ap += incr);
}

void
MuLawDataRep::valueRange(Range &r, int start, int end, int chfirst, int chlast, boolean real) const {
	double max = -32768, min = 32767;
	for(int chan = chfirst; chan <= chlast; chan++) {
		register char *ptr = (char *) getHandle(start, chan);
		for(int i = start; i <= end; i++) {
			double val = Compressor::fromMuLaw(*ptr);
			max = val > max ? val : max;
			min = val < min ? val : min;
			ptr += nchans;
		}
	}
	r.set(min, max);
}

//////////////////////////////////

DataRep *
ShortDataRep::newLengthRep(int newlen) {
	return new ShortDataRep(newlen, channels());
}

// static constructors used to initialize virtual ctor arrays

DataRep*
ShortDataRep::new_Rep(int len, int chans) {
	return new ShortDataRep(len, chans);
}

// cloning (shallow copying) routines

DataRep *
ShortDataRep::clone(const Range& selection, const Range& chans) {
	return new Cloned<ShortDataRep>(this, selection, chans);
}

// iteration routines

void
ShortDataRep::set(double val, int pos, int chan) {
	*((short *) getHandle(pos, chan)) = short(int(val));
}

double
ShortDataRep::get(int pos, int chan) const {
	short *ptr = (short *) getHandle(pos, chan);
	return(pos < len ? *ptr : 0);
}

DataRep::Handle
ShortDataRep::setAndShift(DataRep::Handle h, double val, int incr) {
	short *ap = (short *)h;
	*ap = short(val);
	return (ap += incr);
}

DataRep::Handle
ShortDataRep::getAndShift(DataRep::Handle h, double *val, int incr) const {
	short *ap = (short *)h;
	*val = *ap;
	return (ap += incr);
}

void
ShortDataRep::valueRange(Range &r, int start, int end, int chfirst, int chlast, boolean real) const {
	double max = -(MAXSHORT), min = MAXSHORT;
	for(int chan = chfirst; chan <= chlast; chan++) {
		register short *ptr = (short *) getHandle(start, chan);
		for(int i = start; i <= end; i++) {
			double val = *ptr;
			max = val > max ? val : max;
			min = val < min ? val : min;
			ptr += nchans;
		}
	}
	r.set(min, max);
}

void
ShortDataRep::reverse(){
	for(int chan = 0; chan < channels(); chan++) {
		short *front = (short *) getHandle(0, chan);
		short *back = (short *) getHandle(length() - 1, chan);
		while(front < back) {
			short tmp = *front;
			*front = *back;
			*back = tmp;
			front += nchans;
			back -= nchans;
		}
	}
}

void
ShortDataRep::swapBytes(){
	for(int chan = 0; chan < channels(); chan++) {
		int samps = length();
		short *samp = (short *) getHandle(0, chan);
		while(samps--) {
			*samp = swap(*samp);
			samp += nchans;
		}
	}
}

//////////////////////////////////

DataRep *
IntDataRep::newLengthRep(int newlen) {
	return new IntDataRep(newlen, channels());
}

// static constructors used to initialize virtual ctor arrays

DataRep*
IntDataRep::new_Rep(int len, int chans) {
	return new IntDataRep(len, chans);
}

// cloning (shallow copying) routines

DataRep *
IntDataRep::clone(const Range& selection, const Range& chans) {
	return new Cloned<IntDataRep>(this, selection, chans);
}

// iteration routines

void
IntDataRep::set(double val, int pos, int chan) {
	*((int *) getHandle(pos, chan)) = int(val);
}

double
IntDataRep::get(int pos, int chan) const {
	int *ptr = (int *) getHandle(pos, chan);
	return(pos < len ? *ptr : 0);
}

DataRep::Handle
IntDataRep::setAndShift(DataRep::Handle h, double val, int incr) {
	int *ap = (int *)h;
	*ap = int(val);
	return (ap += incr);
}

DataRep::Handle
IntDataRep::getAndShift(DataRep::Handle h, double *val, int incr) const {
	int *ap = (int *)h;
	*val = *ap;
	return (ap += incr);
}

void
IntDataRep::valueRange(Range &r, int start, int end, int chfirst, int chlast, boolean real) const {
	double max = -(MAXINT), min = MAXINT;
	for(int chan = chfirst; chan <= chlast; chan++) {
		register int *ptr = (int *) getHandle(start, chan);
		for(int i = start; i <= end; i++) {
			double val = *ptr;
			max = val > max ? val : max;
			min = val < min ? val : min;
			ptr += nchans;
		}
	}
	r.set(min, max);
}

void
IntDataRep::reverse(){
	for(int chan = 0; chan < channels(); chan++) {
		int *front = (int *) getHandle(0, chan);
		int *back = (int *) getHandle(length() - 1, chan);
		while(front < back) {
			int tmp = *front;
			*front = *back;
			*back = tmp;
			front += nchans;
			back -= nchans;
		}
	}
}

void
IntDataRep::swapBytes(){
	for(int chan = 0; chan < channels(); chan++) {
		int samps = length();
		int *samp = (int *) getHandle(0, chan);
		while(samps--) {
			*samp = swap(*samp);
			samp += nchans;
		}
	}
}

//////////////////////////////////

DataRep *
FloatDataRep::newLengthRep(int newlen) {
	return new FloatDataRep(newlen, channels());
}

// static constructors used to initialize virtual ctor arrays

DataRep*
FloatDataRep::new_Rep(int len, int chans) {
	return new FloatDataRep(len, chans);
}

DataRep *
FloatDataRep::clone(const Range& selection, const Range& chans) {
	return new Cloned<FloatDataRep>(this, selection, chans);
}

// iteration routines

void
FloatDataRep::set(double val, int pos, int chan) {
	*((float *) getHandle(pos, chan)) = val;
}

double
FloatDataRep::get(int pos, int chan) const {
	float *ptr = (float *) getHandle(pos, chan);
	return(pos < len ? *ptr : 0);
}

DataRep::Handle
FloatDataRep::setAndShift(DataRep::Handle h, double val, int incr) {
	float *ap = (float *)h; *ap = val;
	return (ap += incr);
}

DataRep::Handle
FloatDataRep::getAndShift(DataRep::Handle h, double *val, int incr) const {
	float *ap = (float *)h; *val = *ap;
	return (ap += incr);
}

void
FloatDataRep::valueRange(Range &r, int start, int end, int chfirst, int chlast, boolean real) const {
	double max = -(MAXFLOAT), min = MAXFLOAT;
	for(int chan = chfirst; chan <= chlast; chan++) {
		register float *ptr = (float *) getHandle(start, chan);
		for(int i = start; i <= end; i++) {
			double val = *ptr;
			max = val > max ? val : max;
			min = val < min ? val : min;
			ptr += nchans;
		}
	}
	r.set(min, max);
}

void
FloatDataRep::reverse(){
	for(int chan = 0; chan < channels(); chan++) {
		float *front = (float *) getHandle(0, chan);
		float *back = (float *) getHandle(length() - 1, chan);
		while(front < back) {
			float tmp = *front;
			*front = *back;
			*back = tmp;
			front += nchans;
			back -= nchans;
		}
	}
}

void
FloatDataRep::swapBytes(){
	for(int chan = 0; chan < channels(); chan++) {
		int samps = length();
		float *samp = (float *) getHandle(0, chan);
		while(samps--) {
			*samp = swap(*samp);
			samp += nchans;
		}
	}
}

//////////////////////////////////

DataRep *
DoubleDataRep::newLengthRep(int newlen) {
	return new DoubleDataRep(newlen, channels());
}

// static constructors used to initialize virtual ctor arrays

DataRep*
DoubleDataRep::new_Rep(int len, int chans) {
	return new DoubleDataRep(len, chans);
}

DataRep *
DoubleDataRep::clone(const Range& selection, const Range& chans) {
	return new Cloned<DoubleDataRep>(this, selection, chans);
}

// iteration routines

void
DoubleDataRep::set(double val, int pos, int chan) {
	*((double *) getHandle(pos, chan)) = val;
}

double
DoubleDataRep::get(int pos, int chan) const {
	double *ptr = (double *) getHandle(pos, chan);
	return(pos < len ? *ptr : 0);
}

DataRep::Handle
DoubleDataRep::setAndShift(DataRep::Handle h, double val, int incr) {
	double *ap = (double *)h; *ap = val;
	return (ap += incr);
}

DataRep::Handle
DoubleDataRep::getAndShift(DataRep::Handle h, double *val, int incr) const {
	double *ap = (double *)h; *val = *ap;
	return (ap += incr);
}

void
DoubleDataRep::valueRange(Range &r, int start, int end, int chfirst, int chlast, boolean real) const {
	double max = -(MAXDOUBLE), min = MAXDOUBLE;
	for(int chan = chfirst; chan <= chlast; chan++) {
		register double *ptr = (double *) getHandle(start, chan);
		for(int i = start; i <= end; i++) {
			double val = *ptr;
			max = val > max ? val : max;
			min = val < min ? val : min;
			ptr += nchans;
		}
	}
	r.set(min, max);
}

void
DoubleDataRep::reverse(){
	for(int chan = 0; chan < channels(); chan++) {
		double *front = (double *) getHandle(0, chan);
		double *back = (double *) getHandle(length() - 1, chan);
		while(front < back) {
			double tmp = *front;
			*front = *back;
			*back = tmp;
			front += nchans;
			back -= nchans;
		}
	}
}

void
DoubleDataRep::swapBytes(){
	for(int chan = 0; chan < channels(); chan++) {
		int samps = length();
		double *samp = (double *) getHandle(0, chan);
		while(samps--) {
			*samp = swap(*samp);
			samp += nchans;
		}
	}
}
