/*==============================================================================

  $Id: load_669.c,v 1.6 1998/09/20 21:45:18 miod Exp $

  Composer 669 module loader

==============================================================================*/

/*
	This library is free software; you can redistribute it and/or modify
	it under the terms of the GNU Library General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.
 
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Library General Public License for more details.
 
	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <string.h>

#include <mikmod.h>

/*========== Module structure */

/* header */
typedef struct S69HEADER {   
	UBYTE	marker[2];
	CHAR	message[108];
	UBYTE	nos;
	UBYTE	nop;
	UBYTE	looporder;
	UBYTE	orders[0x80];
	UBYTE	tempos[0x80];
	UBYTE	breaks[0x80];
} S69HEADER;

/* sample information */
typedef struct S69SAMPLE {
	CHAR	filename[13];
	SLONG	length;
	SLONG	loopbeg;
	SLONG	loopend;
} S69SAMPLE;

/* encoded note */
typedef struct S69NOTE {
	UBYTE	a,b,c;
} S69NOTE;

/*========== Loader variables */

/* current pattern */
static	S69NOTE* s69pat=NULL;
/* Module header */
static	S69HEADER* mh=NULL;

/* file type identification */
static	CHAR* S69_Version[]={
	"Composer 669",
	"Extended 669"
};

/*========== Loader code */

BOOL S69_Test(void)
{
	UBYTE id[2];

	if(!_mm_read_UBYTES(id,2,modfp))
		return 0;
	/* look for id */
	if(!memcmp(id,"if",2) || !memcmp(id,"JN",2)) {
		/* skip song message */
		_mm_fseek(modfp,108,SEEK_CUR);
		/* sanity checks */
		if(_mm_read_UBYTE(modfp) > 64) return 0;
		if(_mm_read_UBYTE(modfp) > 128) return 0;
		if(_mm_read_UBYTE(modfp) > 120) return 0;
	} else
		return 0;

	return 1;
}

BOOL S69_Init(void)
{
	if(!(s69pat=(S69NOTE *)_mm_malloc(64*8*sizeof(S69NOTE)))) return 0;
	if(!(mh=(S69HEADER *)_mm_malloc(sizeof(S69HEADER)))) return 0;

	return 1;
}

void S69_Cleanup(void)
{
	if(s69pat) free(s69pat);
	if(mh) free(mh);

	s69pat=NULL;
	mh=NULL;
}

BOOL S69_LoadPatterns(void)
{
	int track,row,channel;
	UBYTE note,inst,vol,effect;
	S69NOTE *cur;
	int tracks=0;
	
	if(!AllocPatterns()) return 0;
	if(!AllocTracks()) return 0;

	for(track=0;track<of.numpat;track++) {
		/* set pattern break locations */
		of.pattrows[track]=mh->breaks[track]+1;

		/* load the 669 pattern */
		cur=s69pat;
		for(row=0;row<64;row++) {
			for(channel=0;channel<8;channel++,cur++) {
				cur->a = _mm_read_UBYTE(modfp);
				cur->b = _mm_read_UBYTE(modfp);
				cur->c = _mm_read_UBYTE(modfp);
			}
		}

		if(feof(modfp)) {
			_mm_errno = MMERR_LOADING_PATTERN;
			return 0;
		}

		/* translate the pattern */
		for(channel=0;channel<8;channel++) {
			UniReset();
			/* set pattern tempo */
			UniPTEffect(0xf,78);
			UniPTEffect(0xf,mh->tempos[track]);

			for(row=0;row<=mh->breaks[track];row++) {
				int a,b,c;

				/* fetch the encoded note */
				a=s69pat[(row*8)+channel].a;
				b=s69pat[(row*8)+channel].b;
				c=s69pat[(row*8)+channel].c;

				/* decode it */
				note=a>>2;
				inst=((a&0x3)<<4)|((b&0xf0)>>4);
				vol=b&0xf;

				if (a<0xff) {
					if (a<0xfe) {
						UniInstrument(inst);
						UniNote(note+24);
					}
					UniPTEffect(0xc,vol<<2);
				}

				if (c!=0xff) {
					effect=c&0xf;
					switch(c>>4) {
						case 0: /* porta up */
							UniPTEffect(0x1,effect);
							break;
						case 1: /* porta down */
							UniPTEffect(0x2,effect);
							break;
						case 2: /* porta to note */
							UniPTEffect(0x3,effect);
							break;
#ifdef MIKMOD_DEBUG
						case 3: /* frequency adjust */
							fprintf(stderr,"669: unsupported frequency adjust effect at pat=%d row=%d chan=%d\n",
							       track,row,channel);
							break;
#endif
						case 4: /* vibrato */
							UniPTEffect(0x4,effect);
							break;
						case 5: /* set speed */
							if (effect) {
								UniPTEffect(0xf,78);
								UniPTEffect(0xf,effect);
							} else 
							  if(mh->marker[0]!=69) {
#ifdef MIKMOD_DEBUG
								fprintf(stderr,"669: unsupported super fast tempo at pat=%d row=%d chan=%d\n",
								       track,row,channel);
#endif
							}
							break;
					}
				}
				UniNewline();
			}
			if(!(of.tracks[tracks++]=UniDup())) return 0;
		}
	}

	return 1;
}

BOOL S69_Load(void)
{
	int i;
	SAMPLE *current;
	S69SAMPLE sample;

	/* module header */
	_mm_read_UBYTES(mh->marker,2,modfp);
	_mm_read_UBYTES((UBYTE *)mh->message,108,modfp);
	mh->nos=_mm_read_UBYTE(modfp);
	mh->nop=_mm_read_UBYTE(modfp);
	mh->looporder=_mm_read_UBYTE(modfp);
	_mm_read_UBYTES(mh->orders,0x80,modfp);
	_mm_read_UBYTES(mh->tempos,0x80,modfp);
	_mm_read_UBYTES(mh->breaks,0x80,modfp);

	/* set module variables */
	of.initspeed=4;
	of.inittempo=78;
	of.songname=DupStr(mh->message,36);
	of.modtype=strdup(S69_Version[memcmp(mh->marker,"JN",2)==0]);
	of.numchn=8;
	of.numpat=mh->nop;
	of.numins=of.numsmp=mh->nos;
	of.numtrk=of.numchn*of.numpat;
	of.flags=UF_XMPERIODS;
	if(mh->message) 
		if((of.comment=(CHAR*)_mm_malloc(3*(36+1)+1))) {
			strncpy(of.comment,mh->message,36);
			strcat(of.comment,"\r");
			if (strlen(mh->message)>36) strncat(of.comment,mh->message+36,36);
			strcat(of.comment,"\r");
			if (strlen(mh->message)>72) strncat(of.comment,mh->message+72,36);
			strcat(of.comment,"\r");
			of.comment[3*(36+1)]=0;
		}

	if(!AllocPositions(0x80)) return 0;
	for(i=0;i<0x80;i++) {
		if(mh->orders[i]==0xff) break;
		of.positions[i]=mh->orders[i];
	}
	of.numpos=i;

	if(!AllocSamples()) return 0;
	current=of.samples;

	for(i=0;i<of.numins;i++) {
		/* sample information */
		_mm_read_UBYTES((UBYTE*)sample.filename,13,modfp);
		sample.length=_mm_read_I_SLONG(modfp);
		sample.loopbeg=_mm_read_I_SLONG(modfp);
		sample.loopend=_mm_read_I_SLONG(modfp);

		if((sample.length<0)||(sample.loopbeg<-1)||(sample.loopend<-1)) {
			_mm_errno = MMERR_LOADING_HEADER;
			return 0;
		}

		current->samplename=DupStr(sample.filename,13);
		current->seekpos=0;
		current->speed=0;
		current->length=sample.length;
		current->loopstart=sample.loopbeg;
		current->loopend=(sample.loopend<sample.length)?sample.loopend:sample.length;
		current->flags=(sample.loopbeg<sample.loopend)?SF_LOOP:0;
		current->volume=64;

		current++;
	}

	if(!S69_LoadPatterns()) return 0;

	return 1;
}

/*========== Loader information */

MLOADER load_669={
	NULL,
	"669",
	"669 loader v0.2",
	S69_Init,
	S69_Test,
	S69_Load,
	S69_Cleanup,
	NULL
};
