/*==============================================================================

  $Id: mikmod.h,v 1.8 1998/09/20 21:45:13 miod Exp $

  MikMod sound library main include file

==============================================================================*/

/*
	This library is free software; you can redistribute it and/or modify
	it under the terms of the GNU Library General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.
 
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Library General Public License for more details.
 
	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _MIKMOD_H
#define _MIKMOD_H

#include <tdefs.h>

#ifdef __cplusplus
extern "C" {
#endif

#include <mmio.h>

/*========== Wrapper macros */

#define MikMod_RegisterDriver(x) MD_RegisterDriver(&x)
#define MikMod_RegisterLoader(x) ML_RegisterLoader(&x)
#define MikMod_RegisterErrorHandler(x) _mm_RegisterErrorHandler(x)

/*========== Samples */

#define SFX_CRITICAL 1

/* Sample format [loading and in-memory] flags: */
#define SF_16BITS       0x0001
#define SF_SIGNED       0x0002
#define SF_STEREO       0x0004
#define SF_DELTA        0x0008
#define SF_BIG_ENDIAN   0x0010
#define SF_ITPACKED		0x0020

#define	SF_FORMATMASK	0x003F

/* General Playback flags */

#define SF_LOOP         0x0040
#define SF_BIDI         0x0080
#define SF_SUSTAIN      0x0100
#define SF_REVERSE      0x0200

#define SF_PLAYBACKMASK	0x03C0

/* Module-only Playback Flags */

#define SF_OWNPAN		0x0400
#define SF_UST_LOOP     0x0800

#define SF_EXTRAPLAYBACKMASK	0x0C00

#define MUTE_EXCLUSIVE  32000
#define MUTE_INCLUSIVE  32001

/* Panning constants */
#define PAN_LEFT       0
#define PAN_CENTER   128
#define PAN_RIGHT    255
#define PAN_SURROUND 512 /* panning value for Dolby Surround */

typedef struct SAMPLE {
	ULONG  speed;       /* Base playing speed/frequency of note */
	UBYTE  volume;      /* volume 0-64 */
	UWORD  panning;     /* panning (0-255 or PAN_SURROUND) */
	ULONG  length;      /* length of sample (in samples!) */
	ULONG  loopstart;   /* repeat position (relative to start, in samples) */
	ULONG  loopend;     /* repeat end */
	ULONG  susbegin;    /* sustain loop begin (in samples) \  Not Supported */
	ULONG  susend;      /* sustain loop end                /      Yet! */
	UWORD  flags;       /* sample format in memory */

    /* Variables used by the module player only! (ignored for sound effects) */
	UBYTE  globvol;     /* global volume */
	UBYTE  vibflags;    /* autovibrato flag stuffs */
	UBYTE  vibtype;     /* Vibratos moved from INSTRUMENT to SAMPLE */
	UBYTE  vibsweep;
	UBYTE  vibdepth;
	UBYTE  vibrate;
	CHAR   *samplename; /* name of the sample */

    /* Values used internally only (not saved in disk formats) */
	UWORD  avibpos;     /* autovibrato pos [player use] */
	UBYTE  divfactor;   /* for sample scaling, maintains proper period slides */
	ULONG  seekpos;     /* seek position in file */
	SWORD  handle;      /* sample handle used by individual drivers */
} SAMPLE;

/* This is a handle of sorts attached to any sample registered with
   SL_RegisterSample.  Generally, this only need be used or changed by the
   loaders and drivers of mikmod. */
typedef struct SAMPLOAD {
	struct SAMPLOAD *next;

	ULONG  length;        /* length of sample (in samples!) */
	ULONG  loopstart;     /* repeat position (relative to start, in samples) */
	ULONG  loopend;       /* repeat end */
	UWORD  infmt,outfmt;
	int    scalefactor;
	SAMPLE *sample;
	FILE   *fp;
} SAMPLOAD;

/*========== Sample and waves loading interface */

extern void SL_HalveSample(SAMPLOAD*);
extern void SL_Sample8to16(SAMPLOAD*);
extern void SL_Sample16to8(SAMPLOAD*);
extern void SL_SampleSigned(SAMPLOAD*);
extern void SL_SampleUnsigned(SAMPLOAD*);
extern BOOL SL_LoadSamples(void);
extern SAMPLOAD *SL_RegisterSample(SAMPLE*,int,FILE*);
extern BOOL SL_Load(void*,SAMPLOAD*,int);
extern BOOL SL_LoadStream(void*,UWORD,UWORD,int,FILE*);
extern BOOL SL_Init(SAMPLOAD*);
extern void SL_Exit(SAMPLOAD*);

extern SAMPLE *WAV_LoadFP(FILE*);
extern SAMPLE *WAV_LoadFN(CHAR*);
extern void WAV_Free(SAMPLE*);

/*========== Drivers */

#include <ptform.h>

/* max. number of handles a driver has to provide. (not strict) */
#define MAXSAMPLEHANDLES 384

enum {
	MD_MUSIC = 0,
	MD_SNDFX
};

enum {
	MD_HARDWARE = 0,
	MD_SOFTWARE
};

/* Mixing flags */

/* These ones take effect only after MikMod_Init or MikMod_Reset */
#define DMODE_16BITS       1 /* enable 16 bit output */
#define DMODE_SURROUND     2 /* enable surround sound */
#define DMODE_SOFT_SNDFX   4 /* Process sound effects via software mixer */
#define DMODE_SOFT_MUSIC   8 /* Process music via software mixer */
/* These take effect immidiately. */
#define DMODE_STEREO      16 /* enable stereo output */
#define DMODE_REVERSE     32 /* reverse stereo */
#define DMODE_INTERP      64 /* enable interpolation */

/* Driver structure */

typedef struct MDRIVER {
	struct MDRIVER *next;
	CHAR  *Name;
	CHAR  *Version;
	UBYTE HardVoiceLimit, /* Limit of hardware mixer voices for this driver */
	      SoftVoiceLimit; /* Limit of software mixer voices for this driver */

	BOOL  (*IsPresent)          (void);
	SWORD (*SampleLoad)         (SAMPLOAD*,int);
	void  (*SampleUnLoad)       (SWORD);
	ULONG (*FreeSampleSpace)    (int);
	ULONG (*RealSampleLength)   (int,SAMPLE*);
	BOOL  (*Init)               (void);
	void  (*Exit)               (void);
	BOOL  (*Reset)              (void);
	BOOL  (*SetNumVoices)       (void);
	BOOL  (*PlayStart)          (void);
	void  (*PlayStop)           (void);
	void  (*Update)             (void);
	void  (*VoiceSetVolume)     (UBYTE,UWORD);
	void  (*VoiceSetFrequency)  (UBYTE,ULONG);
	void  (*VoiceSetPanning)    (UBYTE,ULONG);
	void  (*VoicePlay)          (UBYTE,SWORD,ULONG,ULONG,ULONG,ULONG,UWORD);
	void  (*VoiceStop)          (UBYTE);
	BOOL  (*VoiceStopped)       (UBYTE);
	void  (*VoiceReleaseSustain)(UBYTE);
	SLONG (*VoiceGetPosition)   (UBYTE);
	ULONG (*VoiceRealVolume)    (UBYTE);
	BOOL  (*StreamInit)         (ULONG,UWORD);
	void  (*StreamExit)         (void);
	void  (*StreamSetSpeed)     (ULONG);
	SLONG (*StreamGetPosition)  (void);
	void  (*StreamLoadFP)       (FILE*);
} MDRIVER;

/* These variables can be changed at ANY time and results will be immediate */
extern UBYTE md_bpm;         /* current song / hardware BPM rate */
extern UBYTE md_volume;      /* Global sound volume (0-128) */
extern UBYTE md_musicvolume; /* volume of song */
extern UBYTE md_sndfxvolume; /* volume of sound effects */
extern UBYTE md_reverb;      /* 0 = none;  15 = chaos */
extern UBYTE md_pansep;      /* 0 = mono;  128 == 100% (full left/right) */

/* The variables below can be changed at any time, but changes will not be
   implemented until MikMod_Reset is called. A call to MikMod_Reset may result
   in a skip or pop in audio (depending on the soundcard driver and the settings
   changed). */
extern UWORD md_device;      /* Device */
extern UWORD md_mixfreq;     /* mixing frequency.  Valid 5000 -> 44100 */
extern UWORD md_dmabufsize;  /* DMA buffer size.  Valid 512 -> 32000 */
extern UWORD md_mode;        /* Mode.  See DMODE_? flags above */

/* Variables below can be changed via MD_SetNumVoices at any time. However, a
   call to MD_SetNumVoicess while the driver is active will cause the sound to
   skip slightly. */
extern UBYTE md_numchn;      /* number of song + sound effects voices */
extern UBYTE md_sngchn;      /* number of song voices */
extern UBYTE md_sfxchn;      /* number of sound effects voices */
extern UBYTE md_hardchn;     /* number of hardware mixed voices */
extern UBYTE md_softchn;     /* number of software mixed voices */

/* Following variables should not be changed! */
extern MDRIVER *md_driver;   /* Current driver in use. */

/* This is for use by the hardware drivers only.  It points to the registered
   tickhandler function. */
extern void (*md_player)(void);

/*========== Driver interface */
extern void   MikMod_RegisterAllDrivers(void);
extern void   MikMod_RegisterAllLoaders(void);

extern BOOL   MikMod_Init(void);
extern void   MikMod_Exit(void);
extern BOOL   MikMod_Reset(void);
extern SBYTE  MikMod_PlaySample(SAMPLE*,ULONG,UBYTE);
extern BOOL   MikMod_SetNumVoices(int,int);
extern BOOL   MikMod_Active(void);
extern BOOL   MikMod_EnableOutput(void);
extern void   MikMod_DisableOutput(void);
extern void   MikMod_RegisterPlayer(void (*)(void));
extern void   MikMod_Update(void);

extern void   Voice_SetVolume(SBYTE,UWORD);
extern void   Voice_SetFrequency(SBYTE,ULONG);
extern void   Voice_SetPanning(SBYTE,ULONG);
extern void   Voice_Play(SBYTE,SAMPLE*,ULONG);
extern void   Voice_Stop(SBYTE);
extern void   Voice_ReleaseSustain(SBYTE);
extern BOOL   Voice_Stopped(SBYTE);
extern SLONG  Voice_GetPosition(SBYTE);
extern ULONG  Voice_RealVolume(SBYTE);

extern CHAR*  MD_InfoDriver(void);
extern void   MD_RegisterDriver(MDRIVER*);
extern SWORD  MD_SampleLoad(SAMPLOAD*,int);
extern void   MD_SampleUnLoad(SWORD);
extern void   MD_SetBPM(UBYTE);
extern ULONG  MD_SampleSpace(int);
extern ULONG  MD_SampleLength(int,SAMPLE*);

/*========== Known drivers list */
extern MDRIVER drv_nos;    /* nosound driver */
extern MDRIVER drv_raw;    /* raw file output driver [music.raw] */
extern MDRIVER drv_stdout;
extern MDRIVER drv_wav;    /* RIFF WAVE file output driver [music.wav] */

extern MDRIVER drv_AF;     /* Dec Alpha AudioFile driver */
extern MDRIVER drv_aix;    /* AIX driver */
extern MDRIVER drv_esd;    /* Enlightenment sound daemon (EsounD) driver */
extern MDRIVER drv_hp;     /* HP-UX driver */
extern MDRIVER drv_oss;    /* OpenSound System (Linux,FreeBSD...) driver */
extern MDRIVER drv_sgi;    /* SGI driver */
extern MDRIVER drv_sun;    /* Sun driver */

/*========== Streaming audio */

typedef struct MSTREAM {
	struct MSTREAM *next;
	CHAR    *type;
	CHAR    *version;
	BOOL    (*Init)(void);
	BOOL    (*Test)(void);
	BOOL    (*Load)(void);
	void    (*Cleanup)(void);
} MSTREAM;

extern int   stream_bufsize;
extern FILE *stream_fp;
extern SLONG stream_seekpos;
extern SLONG stream_reppos;

/*========== Virtual channel mixer interface */

extern BOOL  VC_Init(void);
extern void  VC_Exit(void);
extern BOOL  VC_SetNumVoices(void);
extern ULONG VC_SampleSpace(int);
extern ULONG VC_SampleLength(int,SAMPLE*);

extern BOOL  VC_PlayStart(void);
extern void  VC_PlayStop(void);

extern SWORD VC_SampleLoad(SAMPLOAD*,int);
extern void  VC_SampleUnload(SWORD);

extern void  VC_WriteSamples(SBYTE*,ULONG);
extern ULONG VC_WriteBytes(SBYTE*,ULONG);
extern void  VC_SilenceBytes(SBYTE*,ULONG);

extern void  VC_VoiceSetVolume(UBYTE,UWORD);
extern void  VC_VoiceSetFrequency(UBYTE,ULONG);
extern void  VC_VoiceSetPanning(UBYTE,ULONG);
extern void  VC_VoicePlay(UBYTE,SWORD,ULONG,ULONG,ULONG,ULONG,UWORD);

extern void  VC_VoiceStop(UBYTE);
extern BOOL  VC_VoiceStopped(UBYTE);
extern void  VC_VoiceReleaseSustain(UBYTE);
extern SLONG VC_VoiceGetPosition(UBYTE);
extern ULONG VC_VoiceRealVolume(UBYTE);

#ifdef __cplusplus
}
#endif
#endif
