
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include "widget.h"
#include "../econfig.h"

void HideTextBox(TextBox * box)
{
   EDBUG(5, "HideTextBox");
   if (box->visible) {
      XUnmapWindow(disp, box->win);
      box->visible = FALSE;
   }
   EDBUG_RETURN_;
}

void UnhideTextBox(TextBox * box)
{
   EDBUG(5, "UnhideTextBox");
   if (!box->visible) {
      XMapWindow(disp, box->win);
      box->visible = TRUE;
   }
   EDBUG_RETURN_;
}

void DefaultsTextBox(TextBox * box)
{
   box->x = box->y = box->w = box->h = 0;
   box->back = NULL;
   box->state = 1;
   box->exposed = 0;
   box->interactive = 0;
   box->visible = TRUE;
   box->action_left = box->action_middle = box->action_right = 0;
   box->fontheight = 0;
   box->font_info = NULL;
   box->fontset = 0;
   box->items = Emalloc(sizeof(XTextItem));
   box->items->chars = Emalloc(1);
   box->items->delta = 0;
   box->name_x = 5;
   box->name_width = 0;
   box->selected = FALSE;
   box->bounce = FALSE;
   box->scroll = FALSE;
   box->scroll_time = 0;
   box->fontname = duplicate("fixed");
   box->use_fn = FALSE;
   box->fn = NULL;
   box->multiline = FALSE;
}

void CreateTextBox(TextBox * box, Window parent)
{
   char text[20];

   EDBUG(5, "CreateTextBox");
   box->parent = parent;
   if ((box->fn = Fnlib_load_font(fnlib, box->fontname))) {
      box->use_fn = TRUE;
      box->fnstyle.mode = MODE_VERBATIM;
      box->fnstyle.orientation = FONT_TO_RIGHT;
      box->fnstyle.justification = 0;
      box->fnstyle.spacing = 0;
   }
   else if (strchr(box->fontname, ',')) {
      int missing_cnt;
      char **missing_list, *def_str;

      box->fontset = XCreateFontSet(disp, box->fontname, &missing_list,
				    &missing_cnt, &def_str);
      if (missing_cnt)
	 XFreeStringList(missing_list);
      if (!box->fontset || missing_cnt) {
	 Efree(box->fontname);
	 box->fontname = duplicate("fixed");
      }
      else {
	 XRectangle r1, r2;

	 XmbTextExtents(box->fontset, "A", 1, &r1, &r2);
	 box->fontheight = r2.height;
	 box->h = box->fontheight * 1.5;
      }
   }
   if (!box->use_fn && !box->fontset) {
      if ((box->font_info = XLoadQueryFont(disp, box->fontname)) == NULL) {
	 fprintf(stderr, "The font %s could not be loaded."
		 " Using fixed instead.\n", box->fontname);
	 if ((box->font_info = XLoadQueryFont(disp, "fixed")) == NULL) {
	    fprintf(stderr, "Couldn't load fixed either, quitting\n");
	    exit(-1);
	 }
      }
      box->items->font = box->font_info->fid;
      box->fontheight = box->font_info->max_bounds.ascent + box->font_info->max_bounds.descent;
      box->h = box->fontheight * 1.5;
   }
   if (!box->parent)
      box->parent = root;
   if (box->back) {
      box->win = CreateWin(box->parent, box->x, box->y, box->w, box->h);
      XMapWindow(disp, box->win);
      Imlib_apply_image(imlib, box->back->im, box->win);
   }
   else {
      box->win = CreateClearWin(box->parent, box->x, box->y, box->w, box->h);
      XMapWindow(disp, box->win);
   }
   XClearWindow(disp, box->win);
   if (!box->state)
      HideTextBox(box);
   sprintf(text, "eMusic %s", VERSION);
   SetTextTextBox(box, text);
   if (box->interactive)
      box->scroll = FALSE;
   EDBUG_RETURN_;
}

TextBox *ConfigTextBox(Window parent, FILE * file)
{
   char st[FILEPATH_LEN_MAX];
   char s1[FILEPATH_LEN_MAX], s2[FILEPATH_LEN_MAX];
   int end = FALSE;
   TextBox *box = Emalloc(sizeof(TextBox));

   EDBUG(5, "ConfigTextBox");
   DefaultsTextBox(box);
   box->parent = parent;
   while (!end) {
      if (GetNextLine(st, file)) {
	 splitstring(s1, s2, st);
	 if (!strncmp("END", s1, 3))
	    end = TRUE;
	 else if (!strncmp("BACKGROUND", s1, 10)) {
	    box->back = LoadImage(s2);
	    box->w = box->back->w;
	    box->h = box->back->h;
	 }
	 else if (!strncmp("SIZE", s1, 4)) {
	    if (box->back)
	       FreeImageDat(box->back);
	    box->back = NULL;
	    getcoords(s2, &box->w, &box->h);
	 }
	 else if (!strncmp("LOCATION", s1, 8))
	    getcoords(s2, &box->x, &box->y);
	 else if (!strncmp("INTERACTIVE", s1, 11))
	    box->interactive = parsebool(s2);
	 else if (!strncmp("VISIBLE", s1, 7))
	    box->state = parsebool(s2);
	 else if (!strncmp("FONTNAME", s1, 8)) {
	    Efree(box->fontname);
	    box->fontname = duplicate(s2);
	 }
	 else if (!strncmp("FONTHEIGHT", s1, 10))
	    box->fontheight = atoi(s2);
	 else if (!strncmp("SCROLL", s1, 6))
	    box->scroll = parsebool(s2);
	 else if (!strncmp("ACTION_R", s1, 8))
	    box->action_right = parse_action(s2);
	 else if (!strncmp("ACTION_M", s1, 8))
	    box->action_middle = parse_action(s2);
	 else if (!strncmp("ACTION_L", s1, 8))
	    box->action_left = parse_action(s2);
	 else if (!strncmp("MULTILINE", s1, 9))
	    box->multiline = parsebool(s2);
	 else
	    fprintf(stderr, "Unknown line in TextBox config\n");
      }
      else
	 end = TRUE;
   }
   if (!deferred)
      CreateTextBox(box, parent);
   EDBUG_RETURN(box);
}

void EventTextBox(TextBox * box, XEvent ev)
{
   EDBUG(5, "EventTextBox");
   if (ev.type == Expose)
      box->exposed = TRUE;
   else if (ev.type == ButtonRelease) {
      if (!box->interactive)
	 box->scroll = !box->scroll;
      box->selected = TRUE;
      if ((ev.xbutton.button == 3) && (!box->action_right))
	 ev.xbutton.button = 1;
      else if ((ev.xbutton.button == 2) && (!box->action_middle))
	 ev.xbutton.button = 1;
      if (ev.xbutton.button == 3)
	 do_action(box->action_right);
      else if (ev.xbutton.button == 2)
	 do_action(box->action_middle);
      else
	 do_action(box->action_left);
      UpdateTextBox(box);
   }
   else if (ev.type == KeyPress);
   EDBUG_RETURN_;
}

void MoveTextBox(TextBox * box, int x, int y)
{
   EDBUG(5, "MoveTextBox");
   box->x = x;
   box->y = y;
   XMoveWindow(disp, box->win, x, y);
   UpdateTextBox(box);
   EDBUG_RETURN_;
}

void ScaleTextBox(TextBox * box, double scale_w, double scale_h)
{
   EDBUG(5, "ScaleTextBox");
   box->x *= scale_w;
   box->y *= scale_h;
   box->w *= scale_w;
   box->h *= scale_h;
   XMoveResizeWindow(disp, box->win, box->x, box->y, box->w, box->h);
   if (box->back) {
      Imlib_render(imlib, box->back->im, box->w, box->h);
      Imlib_apply_image(imlib, box->back->im, box->win);
   }
   XClearWindow(disp, box->win);
   UpdateTextBox(box);
   EDBUG_RETURN_;
}

int isEventTextBox(TextBox * box, XEvent ev)
{
   EDBUG(5, "isEventTextBox");
   if (ev.xbutton.subwindow == box->win) {
      EDBUG_RETURN(TRUE);
   }
   else if (ev.xany.window == box->win)
      EDBUG_RETURN(TRUE);
   EDBUG_RETURN(FALSE);
}

void SetTextTextBox(TextBox * box, char *text)
{
   EDBUG(5, "SetTextTextBox");
   if (!text) {
      text = Emalloc(20);
      sprintf(text, "eMusic %s", VERSION);
   }
   if (!box->items) {
      box->items = Emalloc(sizeof(XTextItem));
      box->items->chars = NULL;
   }
   if (!strcmp(text, box->items->chars))
      EDBUG_RETURN_;
   box->items->nchars = strlen(text);
   box->items->chars = Erealloc(box->items->chars, strlen(text) + 1);
   strcpy(box->items->chars, text);
   if (!box->use_fn) {
      if (box->fontset) {
	 XRectangle r1, r2;

	 XmbTextExtents(box->fontset, text, strlen(text), &r1, &r2);
	 box->name_width = r2.width;
      }
      else
	 box->name_width = XTextWidth(box->font_info, text, strlen(text));
   }
   else
      box->name_width = strlen(text) * 5;
   box->drawn = FALSE;
   EDBUG_RETURN_;
}

char *ReturnTextTextBox(TextBox * box)
{
   return box->items->chars;
}

void UpdateTextBox(TextBox * box)
{
   int x = 0, y = 0, rx = 0, ry = 0;
   static int old_x = -1;
   struct timeval tv;
   Window root_ret, child_ret;
   unsigned int mask_ret;
   Pixmap pmap;
   char *text;

   EDBUG(5, "UpdateTextBox");
   if (!box || !box->items->chars)
      EDBUG_RETURN_;
   if (!box->interactive) {
      gettimeofday(&tv, NULL);
      if (abs(tv.tv_usec - box->scroll_time) < 100000)
	 EDBUG_RETURN_;
      box->scroll_time = tv.tv_usec;
   }
   if ((box->name_width <= box->w) || (!box->scroll)) {
      if (box->name_width > box->w && box->interactive)
	 box->name_x = box->w - box->name_width - 1;
      else
	 box->name_x = box->items->delta = 0;
   }
   else if (box->bounce) {
      if (box->name_x > 0) {
	 box->bounce = FALSE;
	 box->name_x = 0;
      }
      else
	 box->name_x += 1;
   }
   else {
      if ((box->name_x + box->name_width) < box->w)
	 box->bounce = TRUE;
      else
	 box->name_x -= 1;
   }
   if (box->drawn && (box->name_x == old_x))
      EDBUG_RETURN_;
   old_x = box->name_x;
   box->drawn = TRUE;
   if ((box->name_x < 0) && ((box->name_x + box->name_width) < box->w))
      box->name_x = box->w - box->name_width;
   if (box->name_x > 0)
      box->name_x = 0;
   XClearWindow(disp, box->win);
   if (box->use_fn) {
      text = Emalloc(strlen(box->items->chars) + 400);
      strcpy(text, box->items->chars);
      strcat(text, "                                                        ");
      pmap = XCreatePixmap(disp, box->win, box->w, box->h, imlib->x.depth);
      Fnlib_draw(fnlib, box->fn, pmap, 0, 0, 0, box->w, box->h, box->name_x, 0, box->h, &box->fnstyle, (unsigned char *) text);
      XSetWindowBackgroundPixmap(disp, box->win, pmap);
      XClearWindow(disp, box->win);
      XFreePixmap(disp, pmap);
      Efree(text);
   }
   else if (box->fontset) {
      XmbDrawString(disp, box->win, box->fontset, fg, box->name_x,
		    box->fontheight - box->fontheight / 3,
		    box->items->chars, strlen(box->items->chars));
   }
   else
      XDrawText(disp, box->win, fg, box->name_x, box->fontheight - box->fontheight / 3, box->items, 1);
   if (!box->interactive)
      EDBUG_RETURN_;
   if (box->selected)
      if (!XQueryPointer(disp, box->win, &root_ret, &child_ret, &rx, &ry, &x, &y, &mask_ret))
	 box->selected = FALSE;
   if (box->selected) {
      if (!box->use_fn) {
	 if (box->fontset) {
	    XRectangle r1, r2;

	    XmbTextExtents(box->fontset, box->items->chars,
			   strlen(box->items->chars), &r1, &r2);
	    x = r2.width;
	 }
	 else
	    x = XTextWidth(box->font_info, box->items->chars, strlen(box->items->chars)) + 2;
      }
      else
	 x = strlen(box->items->chars) * box->h + 2;
      XDrawLine(disp, box->win, fg, x, 0, x, box->fontheight);
   }
   EDBUG_RETURN_;
}

void FreeTextBox(TextBox * text)
{
   EDBUG(5, "FreeTextBox");
   if (!text)
      EDBUG_RETURN_;
   FreeImageDat(text->back);
   if (text->font_info)
      Efree(text->font_info);
   else if (text->fontset)
      XFreeFontSet(disp, text->fontset);
   if (text->items) {
      if (text->items->chars)
	 Efree(text->items->chars);
      Efree(text->items);
   }
   if (text->fn)
      Fnlib_free_font(fnlib, text->fn);
   Efree(text);
   text = NULL;
   EDBUG_RETURN_;
}
