
#include "audio.h"

#include <sys/types.h>
#include <signal.h>
#ifdef HAVE_MLOCK
#ifdef OS_Linux
#include <sys/mman.h>
#endif
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xthreads.h>
#include <StringList.h>
#include "../analyze.h"

struct audioType {
   int frequency;
   int stereo;
   int sign;
   int big;
   int sixteen;
};

pthread_mutex_t buffer_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_t buffer_thread;

struct ringBuffer audioBuffer;

int buffer_running;
static int doFlush;
int inputFinished;
int AUDIO_BUFFER_SIZE;
extern int AUSIZ;

void *audioThread(void *a_type);

int audioBufferWrite(char *buf, int bytes)
{
   int temp;

   EDBUG(2, "audioBufferWrite");
   if (!buf || !bytes)
      EDBUG_RETURN(0);
   while ((!audioBuffer.bufferPtr) || (bufferFree(&audioBuffer) < bytes))
      usleep(350);
   pthread_mutex_lock(&buffer_mutex);
   temp = AUDIO_BUFFER_SIZE - audioBuffer.inPos;
   if (bytes <= temp) {
      memcpy((audioBuffer.bufferPtr + audioBuffer.inPos), buf, bytes);
      audioBuffer.inPos = (audioBuffer.inPos + bytes) % AUDIO_BUFFER_SIZE;
      pthread_mutex_unlock(&buffer_mutex);
      EDBUG_RETURN(bytes);
   }
   memcpy((audioBuffer.bufferPtr + audioBuffer.inPos), buf, temp);
   audioBuffer.inPos = (audioBuffer.inPos + temp) % AUDIO_BUFFER_SIZE;
   pthread_mutex_unlock(&buffer_mutex);
   EDBUG_RETURN(temp + audioBufferWrite(buf + temp, bytes - temp));
}

int audio_play(char *buffer, int size)
{
   EDBUG(2, "audio_play");
   if (!AUDIO_BUFFER_SIZE) {
      EDBUG_RETURN(audioWrite(buffer, size));
   }
   else {
      EDBUG_RETURN(audioBufferWrite(buffer, size));
   }
}

void initBuffer(void)
{
   EDBUG(2, "initBuffer");
   pthread_mutex_lock(&buffer_mutex);
   audioBuffer.bufferPtr = Emalloc(AUDIO_BUFFER_SIZE);
   if (audioBuffer.bufferPtr == NULL) {
      fprintf(stderr, "Unable to allocate write buffer\n");
      exit(1);
   }
   memset(audioBuffer.bufferPtr, 0, AUDIO_BUFFER_SIZE);
#ifdef HAVE_MLOCK
   mlock(audioBuffer.bufferPtr, AUDIO_BUFFER_SIZE);
#endif
   audioBuffer.inPos = 0;
   audioBuffer.outPos = 0;
   pthread_mutex_unlock(&buffer_mutex);
   EDBUG_RETURN_;
}

void freeBuffer(void)
{
   EDBUG(2, "freeBuffer");
   pthread_mutex_lock(&buffer_mutex);
#ifdef HAVE_MLOCK
   munlock(audioBuffer.bufferPtr, AUDIO_BUFFER_SIZE);
#endif
   Efree(audioBuffer.bufferPtr);
   audioBuffer.bufferPtr = NULL;
   pthread_mutex_unlock(&buffer_mutex);
   EDBUG_RETURN_;
}

void flush_audio(void)
{
   EDBUG(2, "flush_audio");
   if (AUDIO_BUFFER_SIZE != 0)
      doFlush = TRUE;
   else
      audioFlush();
   EDBUG_RETURN_;
}

int audioBufferOpen(int frequency, int stereo, int sign, int big, int sixteen)
{
   struct audioType *type = Emalloc(sizeof(struct audioType));

   EDBUG(2, "audioBufferOpen");
   inputFinished = FALSE;
   audioBuffer.bufferPtr = NULL;
   type->frequency = frequency;
   type->stereo = stereo;
   type->sign = sign;
   type->big = big;
   type->sixteen = sixteen;
   pthread_create(&buffer_thread, NULL, audioThread, (void *) type);
   sleep(1);
   EDBUG_RETURN(1);
}

void *audioThread(void *a_type)
{
   int outFd, cnt, bytesToEnd;
   fd_set outFdSet;
   fd_set *outFdPtr;

#ifdef DEBUG_BUFFER
   int cntr = 0, percentFull;

#endif
   struct timeval timeout;
   struct audioType *type = a_type;

   EDBUG(2, "audioThread");
   if (audioOpen(type->frequency, type->stereo, type->sign, type->big,
		 type->sixteen)) {
      fprintf(stderr, "Urghggghghghg... can't open EsounD...\n");
      exit(0);
   }

   outFd = getAudioFd();
   audioFlush();
   initBuffer();
   Efree(type);
   buffer_running = TRUE;
   while (1) {
      FD_ZERO(&outFdSet);
      FD_SET(outFd, &outFdSet);
      timeout.tv_sec = 0;
      timeout.tv_usec = 10;
      pthread_mutex_lock(&buffer_mutex);
#ifdef DEBUG_BUFFER
      fprintf(stderr, "BS=%d inPos=%d outPos=%d bufferSize=%d bufferFree=%d\n",
	      AUSIZ, audioBuffer.inPos, audioBuffer.outPos,
	      bufferSize(&audioBuffer), bufferFree(&audioBuffer));
#endif
      if (bufferSize(&audioBuffer) < AUSIZ) {
	 outFdPtr = NULL;
	 if (inputFinished) {
	    pthread_mutex_unlock(&buffer_mutex);
	    break;
	 }
      }
      else
	 outFdPtr = &outFdSet;

      pthread_mutex_unlock(&buffer_mutex);
      if (select(outFd + 1, NULL, outFdPtr, NULL, &timeout) > -1) {
	 if (outFdPtr && (FD_ISSET(outFd, outFdPtr))) {
	    pthread_mutex_lock(&buffer_mutex);
	    if (bufferSize(&audioBuffer) >= AUSIZ) {
	       bytesToEnd = AUDIO_BUFFER_SIZE - audioBuffer.outPos;
#ifdef DEBUG_BUFFER
	       percentFull = 100 * bufferSize(&audioBuffer) / AUDIO_BUFFER_SIZE;
	       if ((cntr++ % (16 / (AUSIZ / 4096))) == 0)
		  fprintf(stderr, "\rBuffer (%2d%%) %6d", percentFull,
			  bufferSize(&audioBuffer));
#endif
	       if (AUSIZ > bytesToEnd) {
		  cnt = audioWrite(audioBuffer.bufferPtr + audioBuffer.outPos, bytesToEnd);
		  cnt += audioWrite(audioBuffer.bufferPtr, AUSIZ - bytesToEnd);
		  audioBuffer.outPos = AUSIZ - bytesToEnd;
	       }
	       else {
		  cnt = audioWrite(audioBuffer.bufferPtr +
				   audioBuffer.outPos, AUSIZ);
		  audioBuffer.outPos += AUSIZ;
	       }
	    }
	    pthread_mutex_unlock(&buffer_mutex);
#ifdef DEBUG_BUFFER
	    fprintf(stderr, "buffer: wrote %d bytes (to audio)\n", cnt);
#endif
	 }
      }
      else {
	 perror("select");
	 exit(-1);
      }
      if (doFlush) {
	 pthread_mutex_lock(&buffer_mutex);
	 audioBuffer.inPos = 0;
	 audioBuffer.outPos = 0;
	 memset(audioBuffer.bufferPtr, 0, AUDIO_BUFFER_SIZE);
	 audioFlush();
	 doFlush = FALSE;
	 pthread_mutex_unlock(&buffer_mutex);
      }
   }
   freeBuffer();
   audioClose();
   buffer_running = FALSE;
   EDBUG_RETURN(NULL);
}

void audioBufferClose()
{
   EDBUG(2, "audioBufferClose");
   if (!end_of_song) {
      if (AUDIO_BUFFER_SIZE)
	 doFlush = TRUE;
      else
	 audioFlush();
   }
   EDBUG_RETURN_;
}
