/*
 *   ALSA sequencer Ports
 *   Copyright (c) 1998 by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"
#include "seq_ports.h"

/*

   registration of client ports

 */


/* 

NOTE: the current implementation of the port structure as a linked list is
not optimal for clients that have many ports. For sending messages to all
subscribers of a port we first need to find the address of the port
structure, which means we have to traverse the list. A direct access table
(array) would be better, but big preallocated arrays waste memory.

Possible actions:

1) leave it this way, a client does normaly does not have more than a few
ports

2) replace the linked list of ports by a array of pointers which is
dynamicly kmalloced. When a port is added or deleted we can simply allocate
a new array, copy the corresponding pointers, and delete the old one. We
then only need a pointer to this array, and an integer that tells us how
much elements are in array.

*/


/*---------------------------------------------------------------------------*/
/* Subscription entry */

/* constructor */
static subscription_t *snd_seq_subscription_new(void)
{
	subscription_t *subs;
	
	/* create a new subscriber entry */
	subs = snd_malloc(sizeof(subscription_t));
	if (subs) {
		memset(subs, 0, sizeof(subscription_t));
		return subs;
	}
	return NULL;
}
 
/* destructor */
static int snd_seq_subscription_delete(subscription_t **subs)
{
	subscription_t *s = *subs;
	*subs = NULL;

	/* FIXME: release resources...*/
	
	snd_free(s,sizeof(subscription_t));
	return 1; /* success */
}


/*---------------------------------------------------------------------------*/
/* List of subscribers */

/* constructor */
static subscribers_t *snd_seq_subscribers_new(void)
{
	return NULL;
}
 
/* destructor */
static int snd_seq_subscribers_delete(subscribers_t **subs)
{
	subscribers_t *s = *subs;
	subscribers_t *o;
	*subs = NULL;

	/* release resources...*/
	while (s) {
		if (s->data)
			snd_seq_subscription_delete(&s->data);
		o = s;
		s = s->next;
		snd_free(o, sizeof(subscribers_t));
	}
	return 1;	/* success */
}

/* add entry to subscription list */
static int snd_seq_subscribers_add_entry(subscribers_t **subs, subscription_t *entry)
{
	subscribers_t *s = *subs;
	if (s == NULL) {
		/* first subscriber */
		*subs = snd_malloc(sizeof(subscribers_t));
		s = *subs;	
	} else {
		/* add to end of list */
		while (s->next) {
			s = s->next;
		}
		s->next = snd_malloc(sizeof(subscribers_t));
		s = s->next;
	}
	
	if (s != NULL) {
		memset (s, 0 , sizeof(subscribers_t));
		
		/* fill in data */
		s->data = entry;	
		return 1;
	} else {
		return 0;	/* failed */
	}
}

/* remove entry from subscription list */
static int snd_seq_subscribers_remove_entry(subscribers_t **subs,  snd_seq_addr_t *dest)
{
	snd_printk("FIXME: snd_seq_subscribers_remove_entry is NOT IMPLEMENTED\n");
	return 0;
}

/* get pointer to subscription entry */
static subscription_t *snd_seq_subscribers_get_ptr(subscribers_t **subs, snd_seq_addr_t *dest)
{
	snd_printk("FIXME: snd_seq_subscribers_get_ptr is NOT IMPLEMENTED\n");
	return NULL;
}


/*---------------------------------------------------------------------------*/



/* return pointer to port structure */
client_port_t *snd_seq_get_port_ptr(client_port_t ** ports, int num)
{
	client_port_t *p = *ports;

	if (p != NULL) {
		if (p->port == num)
			return p;
		while (p->next) {
			p = p->next;
			if (p->port == num)
				return p;
		}
	}
	return NULL;		/* not found */
}


/* create a port, port number is returned (-1 on failure) */
int snd_seq_create_port(client_port_t ** ports)
{
	client_port_t *new_port;
	int num = -1;
	
	/* sanity check */
	if (ports == NULL) {
		snd_printk("oops: snd_seq_create_port() got NULL\n");
		return -1;
	}

	/* create a new port */
	new_port = snd_malloc(sizeof(client_port_t));
	if (new_port) {
		memset(new_port, 0, sizeof(client_port_t));
		new_port->port = -1;
		strcpy(new_port->name, "");
		new_port->next = NULL;
	} else {
		snd_printk("malloc failed for registering client port\n");
		return -1;	/* failure, out of memory */
	}

	/* add the port to the list of registed ports for this client */
	if (*ports == NULL) {
		/* first port in the list */
		*ports = new_port;
		num = 0;
	} else {
		/* add to end of list */
		client_port_t *p = *ports;
		num = p->port;
		while (p->next) {
			p = p->next;
			if (p->port > num)
				num = p->port;
		}
		p->next = new_port;
		num++;
	}

	/* init port data */
	new_port->port = num;	/* store the port number in the port */
	sprintf(new_port->name, "port-%d", num);
	new_port->subscribers = snd_seq_subscribers_new();

	return num;
}


/* delete port data */
static int snd_seq_port_delete(client_port_t *port)
{
	/* sanity check */
	if (port == NULL) {
		snd_printk("oops: port_delete() got NULL\n");
		return -1;
	}
	
	/* clear subscribers info */
	snd_seq_subscribers_delete(&port->subscribers);
	
	snd_free(port, sizeof(client_port_t));
	return 1;
}


/* delete a port from port structure */
int snd_seq_delete_port(client_port_t ** ports, int port)
{
	client_port_t *p = *ports;

	if (p != NULL) {
		if (p->port == port) {
			*ports = p->next;
			return snd_seq_port_delete(p);
		}
		while (p->next) {
			if (p->next) {
				if (p->next->port == port) {
					client_port_t *del = p->next;
					p->next = del->next;
					return snd_seq_port_delete(del);
				}
			}
			p = p->next;
		}
	}
	return -1; 	/* error, port not found */
}


/* set port info fields */
int snd_seq_set_port_info(client_port_t * port, snd_seq_port_info_t * info)
{
	if ((port == NULL) || (info == NULL))
		return -1;

	/* set port name */
	strncpy(port->name, info->name, SND_SEQ_MAX_PORT_NAME);
	
	/* set capabilities */
	port->capability = info->capability;
	
	/* set port type */
	port->port_type = info->port_type;
	
	return 0;
}

/* get port info fields */
int snd_seq_get_port_info(client_port_t * port, snd_seq_port_info_t * info)
{
	if ((port == NULL) || (info == NULL))
		return -1;

	/* get port name */
	strncpy(info->name, port->name, SND_SEQ_MAX_PORT_NAME);
	
	/* get capabilities */
	info->capability = port->capability;
	
	/* get port type */
	info->port_type = port->port_type;

	return 0;
}


/* add subscriber to subscription list */
int snd_seq_port_add_subscriber(client_port_t *port, snd_seq_addr_t *dest)
{
	subscription_t	*new_subs;

	if ((port == NULL) || (dest == NULL))
		return -1;
	
	/* create a new subscriber entry */
	new_subs = snd_seq_subscription_new();
	if (new_subs) {
		memcpy(&new_subs->dest, dest, sizeof(snd_seq_addr_t));
	} else {
		snd_printk("malloc failed for registering subscriber\n");
		return -1;	/* failure, out of memory */
	}

	/* add to the list */
	snd_seq_subscribers_add_entry(&port->subscribers, new_subs);

	snd_printk("Added subscription, %s will send to %d:%d via queue %d\n",
		port->name,
		dest->client, dest->port, dest->queue);
			
	return 0;	/* ok */
}

/* remove subscriber from subscription list */ 
int snd_seq_port_remove_subscriber(client_port_t *port, snd_seq_addr_t *dest)
{
	return snd_seq_subscribers_remove_entry(&port->subscribers, dest);
}


