/* tomidi.c  version 1.5.7
 * This file is part of the code for abc2midi a converter from abc format 
 * to MIDI.  
 * 
 * 19th August 1996
 * James Allwright
 * Department of Electronics and Computer Science,
 * University of Southampton, UK
 *
 * Macintosh Port 30th July 1996 
 * Wil Macaulay (wil@syndesis.com)
 
 */

#include "abc.h"
#include "midifile.h"
#include <stdio.h>
#ifdef __MWERKS__
#define __MACINTOSH__ 1
#endif /* __MWERKS__ */

char* index();
#ifdef __MACINTOSH__
#include <string.h>
int setOutFileCreator(char *fileName,unsigned long theType,
                      unsigned long theCreator);
#define index(a,b) (int)strchr((a),(b))
#endif /* __MACINTOSH__ */ 
#define DIV 480
#define MAXPARTS 100
#define MAXLINE 500
#define INITTEXTS 20
#define INITWORDS 20

FILE *fp;
extern int lineno;
int check;
char basemap[7], workmap[7];
int  basemul[7], workmul[7];
int maxtexts = INITTEXTS;
char** atext;
int ntexts = 0;
int maxnotes;
int *pitch, *num, *denom, *feature;
int notes;
int verbose = 0;
int xmatch;
int dotune, pastheader;
long tempo;
int sf, mi;
int tuplecount, tfact_num, tfact_denom, tnote_num, tnote_denom;
int specialtuple;
int inchord, chordcount, ingrace;
int inslur;
int hasgchords;
char part[MAXPARTS];
int parts, partno;
int part_start[26];
int time_num, time_denom, tempo_num, tempo_denom;
int err_num, err_denom;
int relative_tempo, Qtempo;
int default_length;
int division = DIV;
int div_factor;
long delta_time;
long tracklen, tracklen1;
char outname[40], outbase[40];
int bar_num, bar_denom, barno, barsize, beat;
int b_num, b_denom;
int loudnote, mednote, softnote, channel, program;
int channels[16];
int transpose;
int hornpipe, last_num, last_denom;
int gracenotes;
int gfact_num, gfact_denom;
int middle_c;
int gchords, g_started;
int basepitch, seventh, force_fun;
int karaoke, wcount, ktied, kspace, koverflow;
int maxvoice;
int maxwords = INITWORDS;
char** words;
struct notetype {
  int base; 
  int chan;
  int vel;
};
enum chordmode {maj, min, aug, dim} mode;
struct notetype gchord, fun;
int g_num, g_denom;
int g_next;
char gchord_seq[40];
int gchord_len[40];
int g_ptr;
/* queue for notes waiting to end */
/* allows us to do general polyphony */
#define QSIZE 40
struct Qitem {
  int delay;
  int pitch;
  int chan;
  int next;
};
struct Qitem Q[QSIZE];
int Qhead, freehead, freetail;
#ifdef NOFTELL
char buffer[8000];
int buffcount, buffering;
#endif

int getarg(option, argc, argv)
/* look for argument 'option' in command line */
char *option;
char *argv[];
int argc;
{
  int j, place;

  place = -1;
  for (j=0; j<argc; j++) {
    if (strcmp(option, argv[j]) == 0) {
      place = j + 1;
    };
  };
  return (place);
}

extern int* checkmalloc();

char* addstring(s)
/* create space for string and store it in memory */
char* s;
{
  char* p;

  p = (char*) checkmalloc(strlen(s)+1);
  strcpy(p, s);
  return(p);
}

event_init(argc, argv, filename)
/* this routine is called first by parseabc.c */
int argc;
char* argv[];
char **filename;
{
  int j;

  /* look for code checking option */
  if (getarg("-c", argc, argv) != -1) {
    check = 1;
  } else {
    check = 0;
  };
  /* look for verbose option */
  if (getarg("-v", argc, argv) != -1) {
    verbose = 1;
  } else {
    verbose = 0;
  };
  maxnotes = 500;
  /* allocate space for notes */
  pitch = checkmalloc(maxnotes*sizeof(int));
  num = checkmalloc(maxnotes*sizeof(int));
  denom = checkmalloc(maxnotes*sizeof(int));
  feature = checkmalloc(maxnotes*sizeof(int));
  /* and for text */
  atext = (char**) checkmalloc(maxtexts*sizeof(char*));
  words = (char**) checkmalloc(maxwords*sizeof(char*));
  if ((getarg("-h", argc, argv) != -1) || (argc < 2)) {
    printf("abc2midi version 1.5.7\n");
    printf("Usage : abc2midi <abc file> [reference number] [-c] [-v]\n");
    printf("        [reference number] selects a tune\n");
    printf("        -c selects checking only\n");
    printf("        -v selects verbose option\n");
    exit(0);
  } else {
    xmatch = 0;
    if ((argc >= 3) && (isdigit(*argv[2]))) {
      xmatch = readnumf(argv[2]);
    };
    *filename = argv[1];
    strcpy(outbase, argv[1]);
    for (j = 0; j<strlen(outbase); j++) {
      if (outbase[j] == '.') outbase[j] = '\0';
    };
  };
  dotune = 0;
  parseroff();
}

event_text(s)
/* text found in abc file */
char *s;
{
  char msg[200];

  sprintf(msg, "Ignoring text: %s\0", s);
  event_warning(msg);
}

event_reserved(p)
/* reserved character H-Z found in abc file */
char p;
{
  char msg[200];

  sprintf(msg, "Ignoring reserved character %c\0", p);
  event_warning(msg);
}

event_tex(s)
/* TeX command found - ignore it */
char *s;
{
}

event_linebreak()
{
  addfeature(LINENUM, lineno, 0, 0);
}

event_musiclinebreak()
{
  if (dotune) {
    addfeature(NEWLINE, 0, 0, 0);
  };
}

event_eof()
/* end of abc file encountered */
{
  if (dotune) {
    dotune = 0;
    parseroff();
    finishfile();
  };
  if (verbose) {
    printf("End of File reached\n");
  };
  free(pitch);
  free(num);
  free(denom);
  free(feature);
}

event_error(s)
/* generic error handler */
char *s;
{
  printf("Error in line %d : %s\n", lineno, s);
}

event_warning(s)
/* generic warning handler - for flagging possible errors */
char *s;
{
  printf("Warning in line %d : %s\n", lineno, s);
}

textfeature(type, s)
int type;
char* s;
{
  atext[ntexts] = addstring(s);
  addfeature(type, ntexts, 0, 0);
  ntexts = ntexts + 1;
  if (ntexts >= maxtexts) {
    maxtexts = textextend(maxtexts, &atext);
  };
}

event_comment(s)
/* comment found in abc */
char *s;
{
  if (dotune) {
    if (pastheader) {
      textfeature(TEXT, s);
    } else {
      textfeature(TEXT, s);
    };
  };
}

event_specific(package, s)
/* package-specific command found i.e. %%NAME */
/* only %%MIDI commands are actually handled */
char *package, *s;
{
  char msg[200], command[40];
  char *p;

  if (strcmp(package, "MIDI") == 0) {
    int ch;
    int done;

    p = s;
    done = 0;
    skipspace(&p);
    readstr(command, &p);
    if (strcmp(command, "channel") == 0) {
      skipspace(&p);
      ch = readnump(&p) - 1;
      addfeature(CHANNEL, ch, 0, 0);
      done = 1;
    };
    if (strcmp(command, "transpose") == 0) {
      int neg, val;

      skipspace(&p);
      neg = 0;
      if (*p == '+') p = p + 1;
      if (*p == '-') {
        p = p + 1;
        neg = 1;
      };
      skipspace(&p);
      val = readnump(&p);
      if (neg) val = - val;
      if (pastheader) {
        addfeature(TRANSPOSE, val, 0, 0);
      } else {
        transpose = val;
      };
      done = 1;
    };
    if (strcmp(command, "C") == 0) {
      int val;

      skipspace(&p);
      val = readnump(&p);
      middle_c = val;
      done = 1;
    };

    if (strcmp(command, "grace") == 0) {
      int a, b;
      char msg[40];

      skipspace(&p);
      a = readnump(&p);
      if (*p != '/') {
        event_error("Need / in MIDI grace command");
      } else {
        p = p + 1;
      };
      b = readnump(&p);
      if ((a < 1) || (b < 1) || (a >= b)) {
        sprintf(msg, "%d/%d is not a suitable fraction", a, b);
        event_error(msg);
      } else {
        if (pastheader) {
          addfeature(SETGRACE, a, 0, b);
        } else {
          gfact_num = a;
          gfact_denom = b;
        };
      };
      done = 1;
    };
    if (strcmp(command, "gchordon") == 0) {
      addfeature(GCHORDON, 0, 0, 0);
      done = 1;
    };
    if (strcmp(command, "gchordoff") == 0) {
      addfeature(GCHORDOFF, 0, 0, 0);
      done = 1;
    };
    if (done == 0) {
      /* add as a command to be interpreted later */
      textfeature(DYNAMIC, s);
    };
  } else {
    strcpy(msg, "%");
    strcpy(msg+strlen(msg), package);
    strcpy(msg+strlen(msg),s);
    event_comment(msg);
  };
}

set_gchords(s)
/* set up a string which indicates how to generate accompaniment from */
/* guitar chords (i.e. "A", "G" in abc). */
/* called from dodeferred(), startfile() and setbeat() */
char* s;
{
  int seq_len;
  char* p;
  int j;

  p = s;
  j = 0;
  seq_len = 0;
  while (((*p == 'z') || (*p == 'c') || (*p == 'f')) && (j<39)) {
    gchord_seq[j] = *p;
    p = p + 1;
    if ((*p >= '0') && (*p <= '9')) {
      gchord_len[j] = readnump(&p);
    } else {
      gchord_len[j] = 1;
    };
    seq_len = seq_len + gchord_len[j];
    j = j + 1;
  };
  if (seq_len == 0) {
    event_error("Bad gchord");
    gchord_seq[0] = 'z';
    gchord_len[0] = 1;
    seq_len = 1;
  };
  gchord_seq[j] = '\0';
  if (j == 39) {
    event_error("Sequence string too long");
  };
  /* work out unit delay in 1/4 notes*/
  g_num = time_num * 4;
  g_denom = time_denom * seq_len;
  reduce(&g_num, &g_denom);
}

dodeferred(s)
/* handle package-specific command which has been held over to be */
/* interpreted as MIDI is being generated */
char* s;
{
  char* p;
  char command[40];
  int done;

  p = s;
  skipspace(&p);
  readstr(command, &p);
  skipspace(&p);
  done = 0;
  if (strcmp(command, "program") == 0) {
    int chan, prog;

    skipspace(&p);
    prog = readnump(&p);
    chan = channel;
    skipspace(&p);
    if ((*p >= '0') && (*p <= '9')) {
      chan = prog - 1;
      prog = readnump(&p);
    };
    write_program(prog, chan);
    done = 1;
  };
  if (strcmp(command, "gchord") == 0) {
    set_gchords(p);
    done = 1;
  };
  if (strcmp(command, "chordprog") == 0) {
    int prog;

    prog = readnump(&p);
    write_program(prog, gchord.chan);
    done = 1;
  };
  if (strcmp(command, "bassprog") == 0) {
    int prog;

    prog = readnump(&p);
    write_program(prog, fun.chan);
    done = 1;
  };
  if (strcmp(command, "chordvol") == 0) {
    gchord.vel = readnump(&p);
    done = 1;
  };
  if (strcmp(command, "bassvol") == 0) {
    fun.vel = readnump(&p);
    done = 1;
  };
  if (strcmp(command, "beat") == 0) {
    skipspace(&p);
    loudnote = readnump(&p);
    skipspace(&p);
    mednote = readnump(&p);
    skipspace(&p);
    softnote = readnump(&p);
    skipspace(&p);
    beat = readnump(&p);
    if (beat == 0) {
      beat = barsize;
    };
    done = 1;
  };
  if (strcmp(command, "control") == 0) {
    int chan, n, datum;
    char data[20];
    char sel[40];

    skipspace(&p);
    chan = channel;
    if (isalpha(*p)) {
      readstr(sel, &p);
      skipspace(&p);
      if (strcmp(sel, "bass") == 0) {
        chan = fun.chan;
      };
      if (strcmp(sel, "chord") == 0) {
        chan = gchord.chan;
      };
    };
    n = 0;
    while ((n<20) && (*p >= '0') && (*p <= '9')) {
      datum = readnump(&p);
      if (datum > 127) {
        event_error("data must be in the range 0 - 127");
        datum = 0;
      };
      data[n] = (char) datum;
      n = n + 1;
      skipspace(&p);
    };
    write_control(chan, data, n);
    done = 1;
  };
  if (done == 0) {
    event_error("Command not recognized");
  }; 
} 

event_field(k, f) 
/* Handles R: T: and any other field not handled elsewhere */
char k; 
char *f;
{
  if (dotune) {
    switch (k) {
    case 'T':
      textfeature(TITLE, f);
      break;
    case 'R':
      {
        char* p;

        p = f;
        skipspace(&p);
        if ((strncmp(p, "Hornpipe", 8) == 0) || 
            (strncmp(p, "hornpipe", 8) == 0)) {
          hornpipe = 1;
        };
      };
      break;
    default:
      {
        char buff[100];

        sprintf(buff, "%c:%s", k, f);
        textfeature(TEXT, buff);
      };
    };
  };
}

event_words(p)
/* handles a w: field in the abc */
char* p;
{
  int l;
  char* s;

  karaoke = 1;
  /* strip off leading spaces */
  s = p;
  while (*s == ' ') {
    s = s + 1;
  };
  words[wcount] = addstring(s);
  wcount = wcount + 1;
  if (wcount >= maxwords) {
    maxwords = textextend(maxwords, &words);
  };
  /* strip off any trailing spaces */
  l = strlen(s) - 1;
  while (*(words[wcount-1]+l) == ' ') {
    *(words[wcount-1]+l) = '\0';
    l = l - 1;
  };
}

char_out(list, out, ch)
/* routine for building up part list */
char* list;
char** out;
char ch;
{
  if (*out - list >= MAXPARTS) {
    event_error("Expanded part is too large");
  } else {
    **out = ch;
    *out = *out + 1;
    parts = parts + 1;
  };
}

read_spec(spec, part)
/* converts a P: field to a list of part labels */
/* e.g. P:A(AB)3(CD)2 becomes P:AABABABCDCD */
char spec[];
char part[];
{
  char* in;
  char* out;
  int i, j;
  int stackptr;
  char* stack[10];

  stackptr = 0;
  in = spec;
  out = part;
  while (((*in >= 'A') && (*in <= 'Z')) || (*in == '(') || (*in == '.') ||
         (*in == ')') || ((*in >= '0') && (*in <= '9'))) {
    if (*in == '.') {
      in = in + 1;
    };
    if ((*in >= 'A') && (*in <= 'Z')) {
      char_out(part, &out, *in);
      in = in + 1;
    };
    if (*in == '(') {
      if (stackptr < 10) {
        stack[stackptr] = out;
        stackptr = stackptr + 1;
      } else {
        event_error("nesting too deep in part specification");
      };
      in = in + 1;
    };
    if (*in == ')') {
      in = in + 1;
      if (stackptr > 0) {
        int repeats;
        char* start;
        char* stop;

        if ((*in >= '0') && (*in <= '9')) {
          repeats = readnump(&in);
        } else {
          repeats = 1;
        };
        stackptr = stackptr - 1;
        start = stack[stackptr];
        stop = out;
        for (i=1; i<repeats; i++) {
          for (j=0; j<((int) (stop-start)); j++) {
            char_out(part, &out, *(start+j));
          };
        };
      } else {
        event_error("Too many )'s in part specification");
      };
    };
    if ((*in >= '0') && (*in <= '9')) {
      int repeats;
      char ch;

      repeats = readnump(&in);
      if (out > part) {
        ch = *(out-1);
        for (i = 1; i<repeats; i++) {
          char_out(part, &out, ch);
        };
      } else {
        event_error("No part to repeat in part specification");
      };
    };
  };
  if (stackptr != 0) {
    event_error("Too many ('s in part specification");
  };
}
  
event_part(s)
/* handles a P: field in the abc */
char* s;
{
  char* p;

  if (dotune) {
    p = s;
    skipspace(&p);
    if (pastheader) {
      if (((int)*p < 'A') || ((int)*p > 'Z')) {
        event_error("Part must be one of A-Z");
        return;
      };
      part_start[(int)*p - (int)'A'] = notes;
      addfeature(PART, (int)*p, 0, 0);
    } else {
      parts = 0;
      read_spec(p, part);
    };
  };
}

event_voice(n, s)
/* handles a V: field in the abc */
int n;
char *s;
{
  addfeature(VOICE, n, 0, 0);
  if (n > maxvoice) {
    maxvoice = n;
  };
}

event_length(n)
/* handles an L: field in the abc */
int n;
{
  default_length = n;
}

event_blankline()
/* blank line found in abc signifies the end of a tune */
{
  if (dotune) {
    finishfile();
    parseroff();
    dotune = 0;
  };
}

event_refno(n)
/* handles an X: field (which indicates the start of a tune) */
int n;
{
  if (dotune) {
    finishfile();
    parseroff();
    dotune = 0;
  };
  if (verbose) {
    printf("Reference X: %d\n", n);
  };
  if ((n == xmatch) || (xmatch == 0)) {
    parseron();
    dotune = 1;
    pastheader = 0;
    strcpy(outname, outbase);
    numstr(n, outname+strlen(outname));
    strcpy(outname+strlen(outname), ".mid");
    startfile();
  };
}

event_tempo(n, a, b, rel)
/* handles a Q: field e.g. Q: a/b = n  or  Q: Ca/b = n */
int n;
int a, b, rel;
{
  int t_num, t_denom;
  int new_div;
  long new_tempo;

  if (dotune) {
    if (pastheader) {
      tempo_num = a;
      tempo_denom = b;
      relative_tempo = rel;
      tempounits(&t_num, &t_denom);
      new_tempo = (long) 60*1000000*t_denom/(n*4*t_num);
      new_div = (int) ((float)DIV*(float)new_tempo/(float)tempo + 0.5);
      addfeature(TEMPO, new_div, 0, 0);
    } else {
      Qtempo = n;
      tempo_num = a;
      tempo_denom = b;
      relative_tempo = rel;
    };
  };
}

event_timesig(n, m)
/* handles an M: field  M:n/m */
int n, m;
{
  if (dotune) {
    if (pastheader) {
      addfeature(TIME, n, 0, m);
    } else {
      time_num = n;
      time_denom = m;
    };
  };
}

event_key(sharps, s, minor, modmap, modmul)
/* handles a K: field */
int sharps; /* sharps is number of sharps in key signature */
int minor; /* a boolean 0 or 1 */
char *s; /* original string following K: */
char modmap[7]; /* array of accidentals to be applied */
int  modmul[7]; /* array giving multiplicity of each accent (1 or 2) */
{
  if (dotune) {
    setmap(sharps, basemap, basemul);
    altermap(basemap, basemul, modmap, modmul);
    copymap();
    if (pastheader) {
      addfeature(KEY, sharps, 0, minor);
    } else {
      addfeature(DOUBLE_BAR, 0, 0, 0);
      sf = sharps;
      mi = minor;
      pastheader = 1;
      headerprocess();
    };
  };
}

event_graceon()
/* a { in the abc */
{
  if (gracenotes) {
    event_error("Nested grace notes not allowed");
  };
  gracenotes = 1;
  addfeature(GRACEON, 0, 0, 0);
  ingrace = 1;
}

event_graceoff()
/* a } in the abc */
{
  if (!gracenotes) {
    event_error("} without matching {");
  };
  gracenotes = 0;
  addfeature(GRACEOFF, 0, 0, 0);
  ingrace = 0;
}

event_rep1()
/* |1 in the abc */
{
  if ((notes == 0) || (feature[notes-1] != SINGLE_BAR)) {
    event_error("[1 must follow a single bar");
  } else {
    feature[notes-1] = BAR1;
  };
}

event_rep2()
/* :|2 in the abc */
{
  if ((notes == 0) || (feature[notes-1] != REP_BAR)) {
    event_error("[2 must follow a :| ");
  } else {
    feature[notes-1] = REP_BAR2;
  };
}

event_slur(t)
/* called when s in encountered in the abc */
int t;
{
  if (t) {
    addfeature(SLUR_ON, 0, 0, 0);
    inslur = 1;
  } else {
    slurtotie();
    addfeature(SLUR_OFF, 0, 0, 0);
    inslur = 0;
  };
}

event_sluron(t)
/* called when ( is encountered in the abc */
int t;
{
  if (t == 1) {
    addfeature(SLUR_ON, 0, 0, 0);
    inslur = 1;
  };
}

event_sluroff(t)
/* called when ) is encountered */
int t;
{
  if (t == 0) {
    slurtotie();
    addfeature(SLUR_OFF, 0, 0, 0);
    inslur = 0;
  };
}

slurtotie()
{
  int last1, slurtie, last2, failed;
  int j;

  if ((!ingrace) && (!inchord)) {
    j = notes-1;
    failed = 0;
    last1 = -1;
    while ((j>=0) && (!failed) && (last1 == -1)) {
      if (feature[j] == NOTE) {
        last1 = j;
      };
      if ((j<=0) || (feature[j] == REST) || (feature[j] == CHORDOFF) || 
          (feature[j] == GRACEOFF) || (feature[j] == SLUR_ON)) {
        failed = 1;
      };
      j = j - 1;
    };
    slurtie = -1;
    while ((j>=0) && (!failed) && (slurtie == -1)) {
      if (feature[j] == SLUR_TIE) {
        slurtie = j;
      };
      if ((j<=0) || (feature[j] == REST) || (feature[j] == CHORDOFF) || 
          (feature[j] == GRACEOFF) || (feature[j] == SLUR_ON) ||
          (feature[j] == NOTE)) {
        failed = 1;
      };
      j = j - 1;
    };
    last2 = -1;
    while ((j>=0) && (!failed) && (last2 == -1)) {
      if (feature[j] == NOTE) {
        last2 = j;
      };
      if ((j<=0) || (feature[j] == REST) || (feature[j] == CHORDOFF) || 
          (feature[j] == GRACEOFF) || (feature[j] == SLUR_ON)) {
        failed = 1;
      };
      j = j - 1;
    };
    if ((!failed) && (pitch[last1] == pitch[last2])) {
      /* promote SLUR_TIE to tie */
      feature[slurtie] = TIE;
      event_warning("Slur in abc taken to mean a tie");
    } else {
      if (verbose) {
        event_warning("Slur ignored");
      };
    };
  };
}

event_tie()
/* a tie - has been encountered in the abc */
{
  addfeature(TIE, 0, 0, 0);
}

event_rest(n,m)
/* rest of n/m in the abc */
int n, m;
{
  int num, denom;

  num = n;
  denom = m;
  if (inchord) chordcount = chordcount + 1;
  if (tuplecount > 0) {
    num = num * tfact_num;
    denom = denom * tfact_denom;
    if (tnote_num == 0) {
      tnote_num = num;
      tnote_denom = denom;
    } else {
      if (tnote_num * denom != num * tnote_denom) {
        if (!specialtuple) {
          event_warning("Different length notes in tuple");
        };
      };
    };
    if ((!gracenotes) &&
        ((!inchord) || ((inchord) && (chordcount == 1)))) {
      tuplecount = tuplecount - 1;
    };
  };
  if ((!ingrace) && ((!inchord)||(chordcount==1))) {
    addunits(num, denom*default_length);
  };
  last_num = 3; /* hornpiping (>) cannot follow rest */
  addfeature(REST, 0, num*4, denom*default_length);
}

event_bar(type)
/* handles bar lines of various types in the abc */
int type;
{
  int newtype;

  if ((type == THIN_THICK) || (type == THICK_THIN)) {
    newtype = DOUBLE_BAR;
  } else {
    newtype = type;
  };
  addfeature(newtype, 0, 0, 0);
  copymap();
  zerobar();
}

event_space()
/* space character in the abc is ignored by abc2midi */
{
  /* ignore */
  /* printf("Space event\n"); */
}

event_lineend(ch, n)
/* called when \ or ! or * or ** is encountered at the end of a line */
char ch;
int n;
{
  /* ignore */
}

event_broken(type, mult)
/* handles > >> >>> < << <<< in the abc */
int type, mult;
{
  if ((hornpipe) && (feature[notes-1] == GT)) {
    /* remove any superfluous hornpiping */
    notes = notes - 1;
  };
  addfeature(type, mult, 0, 0);
}

event_tuple(n, q, r)
/* handles triplets (3 and general tuplets (n:q:r in the abc */
int n, q, r;
{
  if (tuplecount > 0) {
    event_error("nested tuples");
  } else {
    if (r == 0) {
      specialtuple = 0;
      tuplecount = n;
    } else {
      specialtuple = 1;
      tuplecount = r;
    };
    if (q != 0) {
      tfact_num = q;
      tfact_denom = n;
    } else {
      if ((n < 2) || (n > 9)) {
        event_error("Only tuples (2 - (9 allowed");
        tfact_num = 1;
        tfact_denom = 1;
        tuplecount = 0;
      } else {
        /* deduce tfact_num using standard abc rules */
        if ((n == 2) || (n == 4) || (n == 8)) tfact_num = 3;
        if ((n == 3) || (n == 6)) tfact_num = 2;
        if ((n == 5) || (n == 7) || (n == 9)) {
          if ((time_num % 3) == 0) {
            tfact_num = 3;
          } else {
            tfact_num = 2;
          };
        };
        tfact_denom = n;
      };
    };
    tnote_num = 0;
    tnote_denom = 0;
  };
}

event_chord()
/* a + has been encountered in the abc */
{
  if (inchord) {
    event_chordoff();
  } else {
    event_chordon();
  };
}

event_chordon()
/* handles a chord start [ in the abc */
{
  if (inchord) {
    event_error("Attempt to nest chords");
  } else {
    addfeature(CHORDON, 0, 0, 0);
  };
  inchord = 1;
  chordcount = 0;
}

event_chordoff()
/* handles a chord close ] in the abc */
{
  if (!inchord) {
    event_error("Chord already finished");
  } else {
    addfeature(CHORDOFF, 0, 0, 0);
  };
  inchord = 0;
  chordcount = 0;
}

splitstring(s, sep, handler)
/* breaks up string into fields with sep as the field separator */
/* and calls handler() for each sub-string */
char* s;
char sep;
void (*handler)();
{
  char* out;
  char* p;
  int fieldcoming;

  p = s;
  fieldcoming = 1;
  while (fieldcoming) {
    out = p;
    while ((*p != '\0') && (*p != sep)) p = p + 1;
    if (*p == sep) {
      *p = '\0';
      p = p + 1;
    } else {
      fieldcoming = 0;
    };
    (*handler)(out);
  };
}

void event_handle_instruction(), event_handle_gchord();

event_instruction(s)
/* handles a ! ... ! event in the abc */
char* s;
{
  splitstring(s, ';', event_handle_instruction);
}

event_gchord(s)
/* handles guitar chords " ... " */
char* s;
{
  splitstring(s, ';', event_handle_gchord);
}

void event_handle_gchord(s)
/* handler for the guitar chords */
char* s;
{
  int basepitch;
  char accidental, note;
  char* p;
  int seventh, force_fun;
  enum chordmode mode;

  if ((*s >= '0') && (*s <= '5')) {
    event_finger(s);
    return;
  };
  p = s;
  if ((*p >= 'A') && (*p <= 'G')) {
    note = *p - (int) 'A' + (int) 'a';
    force_fun = 0;
    p = p + 1;
  } else {
    if ((*p >= 'a') && (*p <= 'g')) {
      note = *p;
      force_fun = 1;
      p = p + 1;
    } else {
      event_error("Guitar chord does not start with A-G or a-g");
      return;
    };
  };
  accidental = '=';
  if (*p == '#') {
    accidental = '^';
    p = p + 1;
  };
  if (*p == 'b') {
    accidental = '_';
    p = p + 1;
  };
  basepitch = pitchof(note, accidental, 1, 0) - middle_c;
  mode = maj;
  seventh = 0;
  if (*p == '7') {
    seventh = 1;
    p = p + 1;
  };
  if (*p == 'm') {
    mode = min;
    p = p + 1;
  };
  if ((*p == 'a') && (*(p+1) == 'u') && (*(p+2) == 'g')) {
    mode = aug;
    p = p + 3;
  };
  if ((*p == 'd') && (*(p+1) == 'i') && (*(p+2) == 'm')) {
    mode = dim;
    p = p + 3;
  };
  if (*p == '7') {
    seventh = 1;
    p = p + 1;
  };
  hasgchords = 1;
  addfeature(GCHORD, basepitch, 0, 
        (force_fun*8) + (((int) mode)*2) + seventh);
}

void event_handle_instruction(s)
/* handler for ! ! instructions */
/* does ppp pp p mp mf f ff fff */
char* s;
{
  char buff[MAXLINE];
  char* p;
  char* q;
  int done;

  p = s;
  /* remove any leading spaces */
  skipspace(&p);
  /* remove any trailing spaces */
  q = p;
  while ((*q != '\0') && (*q != ' ')) {
    q = q + 1;
  };
  if (*q == ' ') {
    *q = '\0';
  };
  done = 0;
  if (strcmp(p, "ppp") == 0) {
    event_specific("MIDI", "beat 30 20 10 1");
    done = 1;
  };
  if (strcmp(p, "pp") == 0) {
    event_specific("MIDI", "beat 45 35 20 1");
    done = 1;
  };
  if (strcmp(p, "p") == 0) {
    event_specific("MIDI", "beat 60 50 35 1");
    done = 1;
  };
  if (strcmp(p, "mp") == 0) {
    event_specific("MIDI", "beat 75 65 50 1");
    done = 1;
  };
  if (strcmp(p, "mf") == 0) {
    event_specific("MIDI", "beat 90 80 65 1");
    done = 1;
  };
  if (strcmp(p, "f") == 0) {
    event_specific("MIDI", "beat 105 95 80 1");
    done = 1;
  };
  if (strcmp(p, "ff") == 0) {
    event_specific("MIDI", "beat 120 110 95 1");
    done = 1;
  };
  if (strcmp(p, "fff") == 0) {
    event_specific("MIDI", "beat 127 125 110 1");
    done = 1;
  };
  if (done == 0) {
    sprintf(buff, "instruction !%s! ignored", s);
    event_warning(buff);
  };
}

event_finger(p)
/* a 1, 2, 3, 4 or 5 has been found in a guitar chord field */
char *p;
{
  /* does nothing */
}

hornp(num, denom)
int num, denom;
{
  if ((hornpipe) && (notes > 0) && (feature[notes-1] != GT)) {
    if ((num*last_denom == last_num*denom) && (num == 1) && 
        (denom*time_num == 32)) {
      if (((time_num == 4) && (bar_denom == 8)) ||
          ((time_num == 2) && (bar_denom == 16))) {
           addfeature(GT, 1, 0, 0);
      };
    };
    last_num = num;
    last_denom = denom;
  };
}

event_note(decorators, accidental, mult, note, octave, n, m)
/* handles a note in the abc */
int decorators[DECSIZE];
int mult;
char accidental, note;
int octave, n, m;
{
  int pitch;
  int num, denom;

  num = n;
  denom = m;
  if (inchord) chordcount = chordcount + 1;
  if (tuplecount > 0) {
    num = num * tfact_num;
    denom = denom * tfact_denom;
    if (tnote_num == 0) {
      tnote_num = num;
      tnote_denom = denom;
    } else {
      if (tnote_num * denom != num * tnote_denom) {
        if (!specialtuple) {
          event_warning("Different length notes in tuple");
        };
      };
    };
    if ((!gracenotes) &&
        ((!inchord) || ((inchord) && (chordcount == 1)))) {
      tuplecount = tuplecount - 1;
    };
  };
  if ((!ingrace) && (!inchord)) {
    hornp(num, denom*default_length);
  } else {
    last_num = 3; /* hornpiping (>) cannot follow chord or grace notes */
  };
  if ((!ingrace) && ((!inchord)||(chordcount==1))) {
    addunits(num, denom*default_length);
  };
  pitch = pitchof(note, accidental, mult, octave);
  if (decorators[FERMATA]) {
    num = num*2;
  };
  if ((decorators[ROLL]) || (decorators[ORNAMENT])) {
    if (inchord) {
      event_error("Roll in chord not supported");
    };
    doroll(note, octave, num, denom, pitch);
  } else {
    if (decorators[STACCATO]) {
      if (inchord) {
        if (chordcount == 1) {
          addfeature(REST, pitch, num*4, denom*default_length);
        };
        addfeature(NOTE, pitch, num*4, denom*2*default_length);
      } else {
        addfeature(NOTE, pitch, num*4, denom*2*default_length);
        addfeature(REST, pitch, num*4, denom*2*default_length);
      };
    } else {
      addfeature(NOTE, pitch, num*4, denom*default_length);
      if ((inslur) && (!ingrace)) {
        addfeature(SLUR_TIE, 0, 0, 0);
      };
    };
  };
}

doroll(note, octave, n, m, pitch)
/* applies a roll to a note */
char note;
int octave, n, m;
int pitch;
{
  char up, down;
  int t;
  int upoct, downoct, pitchup, pitchdown;
  char *anoctave = "cdefgab";

  upoct = octave;
  downoct = octave;
  t = (int) ((long) index(anoctave, note)  - (long) anoctave);
  up = *(anoctave + ((t+1) % 7));
  down = *(anoctave + ((t+6) % 7));
  if (up == 'c') upoct = upoct + 1;
  if (down == 'b') downoct = downoct - 1;
  pitchup = pitchof(up, basemap[(int)up - 'a'], 1, upoct);
  pitchdown = pitchof(down, basemap[(int)down - 'a'], 1, downoct);
  addfeature(NOTE, pitch, n*4, m*default_length*5);
  addfeature(NOTE, pitchup, n*4, m*default_length*5);
  addfeature(NOTE, pitch, n*4, m*default_length*5);
  addfeature(NOTE, pitchdown, n*4, m*default_length*5);
  addfeature(NOTE, pitch, n*4, m*default_length*5);
}

int pitchof(note, accidental, mult, octave)
/* finds MIDI pitch value for note */
char note, accidental;
int mult, octave;
{
  int p;
  char acc;
  int mul, noteno;
  static int scale[7] = {0, 2, 4, 5, 7, 9, 11};
  char *anoctave = "cdefgab";

  p = (int) ((long) index(anoctave, note) - (long) anoctave);
  p = scale[p];
  acc = accidental;
  mul = mult;
  noteno = (int)note - 'a';
  if (acc == ' ') {
    acc = workmap[noteno];
    mul = workmul[noteno];
  } else {
    workmap[noteno] = acc;
    workmul[noteno] = mul;
  };
  if (acc == '^') p = p + mul;
  if (acc == '_') p = p - mul;
  return p + 12*octave + middle_c;
}

setmap(sf, map, mult)
/* work out accidentals to be applied to each note */
int sf; /* number of sharps in key signature -7 to +7 */
char map[7];
int mult[7];
{
  int j;

  for (j=0; j<7; j++) {
    map[j] = '=';
    mult[j] = 1;
  };
  if (sf >= 1) map['f'-'a'] = '^';
  if (sf >= 2) map['c'-'a'] = '^';
  if (sf >= 3) map['g'-'a'] = '^';
  if (sf >= 4) map['d'-'a'] = '^';
  if (sf >= 5) map['a'-'a'] = '^';
  if (sf >= 6) map['e'-'a'] = '^';
  if (sf >= 7) map['b'-'a'] = '^';
  if (sf <= -1) map['b'-'a'] = '_';
  if (sf <= -2) map['e'-'a'] = '_';
  if (sf <= -3) map['a'-'a'] = '_';
  if (sf <= -4) map['d'-'a'] = '_';
  if (sf <= -5) map['g'-'a'] = '_';
  if (sf <= -6) map['c'-'a'] = '_';
  if (sf <= -7) map['f'-'a'] = '_';
}

altermap(basemap, basemul, modmap, modmul)
/* apply modifiers to a set of accidentals */
char basemap[7], modmap[7];
int basemul[7], modmul[7];
{
  int i;

  for (i=0; i<7; i++) {
    if (modmap[i] != ' ') {
      basemap[i] = modmap[i];
      basemul[i] = modmul[i];
    };
  };
}

copymap()
{
  int j;

  for (j=0; j<7; j++) {
    workmap[j] = basemap[j];
    workmul[j] = basemul[j];
  };
}

addfeature(f, p, n, d) 
/* place feature in internal table */
int f, p, n, d;
{
  feature[notes] = f;
  pitch[notes] = p;
  num[notes] = n;
  denom[notes] = d;
  if ((f == NOTE) || (f == REST)) {
    reduce(&num[notes], &denom[notes]);
  };
  notes = notes + 1;
  if (notes >= maxnotes) {
    maxnotes = autoextend(maxnotes);
  };
}

int autoextend(maxnotes)
/* increase the number of abc elements the program can cope with */
int maxnotes;
{
  int newlimit;
  int *ptr;
  int i;

  if (verbose) {
    event_warning("Extending note capacity");
  };
  newlimit = maxnotes*2;
  ptr = checkmalloc(newlimit*sizeof(int));
  for(i=0;i<maxnotes;i++){
    ptr[i] = feature[i];
  };
  free(feature);
  feature = ptr;
  ptr = checkmalloc(newlimit*sizeof(int));
  for(i=0;i<maxnotes;i++){
    ptr[i] = pitch[i];
  };
  free(pitch);
  pitch = ptr;
  ptr = checkmalloc(newlimit*sizeof(int));
  for(i=0;i<maxnotes;i++){
    ptr[i] = num[i];
  };
  free(num);
  num = ptr;
  ptr = checkmalloc(newlimit*sizeof(int));
  for(i=0;i<maxnotes;i++){
    ptr[i] = denom[i];
  };
  free(denom);
  denom = ptr;
  return(newlimit);
}

int textextend(maxstrings, stringarray)
/* resize an array of pointers to strings */
/* used with arrays words and atext */
int maxstrings;
char*** stringarray;
{
  int i, newlimit;
  char** ptr;

  newlimit = maxstrings*2;
  if (verbose) {
    event_warning("Extending text capacity");
  };
  ptr = (char**) checkmalloc(newlimit*sizeof(char*));
  for(i=0;i<maxstrings;i++){
    ptr[i] = (*stringarray)[i];
  };
  free(*stringarray);
  *stringarray = ptr;
  return(newlimit);
}

numstr(n, out)
/* convert number to string */
int n;
char *out;
{
  char s[10];
  char* p;
  int i, t;

  t = n;
  i = 9;
  while ((t > 0) || (i == 9)){
    s[i] = (char) (t % 10 + (int)'0');
    t = t/10;
    i = i - 1;
  };
  i = i + 1;
  p = out;
  while (i < 10) {
    *p = s[i];
    p = p + 1;
    i = i + 1;
  };
  *p = '\0';
}
  
int nullputc(c)
/* dummy putc for abc checking option */
char c;
{
  int t;

  t = ((int) c) & 0xFF;
  return (t);
}

/* workaround for problems with PCC compiler */
/* data may be written to an internal buffer */

#ifdef NOFTELL
int myputc(c)
char c;
{
  int t;

  if (buffering) {
    buffer[buffcount] = c;
    buffcount = buffcount + 1;
    t = ((int) c) & 0xFF;
  } else {
    t = putc(c, fp);
  };
  return (t);
}
#else
myputc(c)
char c;
{
  return (putc(c,fp));
}
#endif

#ifdef NOFTELL
flushbuffer()
{
  int t;

  t = 0;
  while (t < buffcount) {
    putc(buffer[t], fp);
    t = t + 1;
  };
  buffcount = 0;
  buffering = 0;
}

setbuffering(t)
int t;
{
  buffering = t;
}
#endif

chordfix()
/* set up chord length in CHORDOFF field */
{
  int j;
  int inchord;
  int chord_num, chord_denom;

  j = 0;
  inchord = 0;
  chord_num = 0;
  chord_denom = 1;
  while (j<notes) {
    switch (feature[j]) {
    case CHORDON:
      if (inchord) {
        event_error("Nested chords");
      };
      inchord = 1;
      chord_num = -1;
      j = j + 1;
      break;
    case CHORDOFF:
      if ((!inchord) || (chord_num == -1)) {
        event_error("Unexpected end of chord");
      } else {
        num[j] = chord_num;
        denom[j] = chord_denom;
      };
      inchord = 0;
      j = j + 1;
      break;
    case NOTE:
      if ((inchord) && (chord_num == -1)) {
        chord_num = num[j];
        chord_denom = denom[j];
      };
      j = j + 1;
      break;
    case REST:
      if ((inchord) && (chord_num == -1)) {
        chord_num = num[j];
        chord_denom = denom[j];
      };
      j = j + 1;
      break;
    case LINENUM:
      lineno = pitch[j];
      j = j + 1;
      break;
    default:
      j = j + 1;
      break;
    };
  };
}

tiefix()
/* connect up tied notes */
{
  int j;
  int inchord;
  int chord_num, chord_denom;

  j = 0;
  inchord = 0;
  while (j<notes) {
    switch (feature[j]) {
    case CHORDON:
      inchord = 1;
      chord_num = -1;
      j = j + 1;
      break;
    case CHORDOFF:
      if (!((!inchord) || (chord_num == -1))) {
        num[j] = chord_num;
        denom[j] = chord_denom;
      };
      inchord = 0;
      j = j + 1;
      break;
    case NOTE:
      if ((inchord) && (chord_num == -1)) {
        chord_num = num[j];
        chord_denom = denom[j];
      };
      j = j + 1;
      break;
    case REST:
      if ((inchord) && (chord_num == -1)) {
        chord_num = num[j];
        chord_denom = denom[j];
      };
      j = j + 1;
      break;
    case TIE:
      dotie(j, inchord);
      j = j + 1;
      break;
    case LINENUM:
      lineno = pitch[j];
      j = j + 1;
      break;
    default:
      j = j + 1;
      break;
    };
  };
}

dotie(j, xinchord)
/* called by preprocess() to handle ties */
int j, xinchord;
{
  int tienote, place;
  int tietodo, done;
  int lastnote, lasttie;
  int inchord;
  int tied_num, tied_denom;

  /* find note to be tied */
  tienote = j;
  while ((tienote > 0) && (feature[tienote] != NOTE) && 
         (feature[tienote] != REST)) {
    tienote = tienote - 1;
  };
  if (feature[tienote] != NOTE) {
    event_error("Cannot find note before tie");
  } else {
    inchord = xinchord;
    /* change NOTE + TIE to TNOTE + REST */
    feature[tienote] = TNOTE;
    feature[j] = REST;
    num[j] = num[tienote];
    denom[j] = denom[tienote];
    place = j;
    tietodo = 1;
    lasttie = j;
    tied_num = num[tienote];
    tied_denom = denom[tienote];
    done = 0;
    while ((place < notes) && (tied_num >=0) && (done == 0)) {
      switch (feature[place]) {
        case NOTE:
          lastnote = place;
          if ((tied_num == 0) && (tietodo == 0)) {
            done = 1;
          };
          if ((pitch[place] == pitch[tienote]) && (tietodo == 1)) {
            /* tie in note */
            if (tied_num != 0) {
              event_error("Time mismatch at tie");
            };
            tietodo = 0;
            /* add time to tied time */
            addfract(&tied_num, &tied_denom, num[place], denom[place]);
            /* add time to tied note */
            addfract(&num[tienote], &denom[tienote], num[place], denom[place]);
            /* change note to a rest */
            feature[place] = REST;
            /* get rid of tie */
            if (lasttie != j) {
              feature[lasttie] = NONOTE;
            };
          };
          if (inchord == 0) {
            /* subtract time from tied time */
            addfract(&tied_num, &tied_denom, -num[place], denom[place]);
          };
          break;
        case REST:
          if ((tied_num == 0) && (tietodo == 0)) {
            done = 1;
          };
          if (inchord == 0) {
            /* subtract time from tied time */
            addfract(&tied_num, &tied_denom, -num[place], denom[place]);
          };
          break;
        case TIE:
          if (pitch[lastnote] == pitch[tienote]) {
            lasttie = place;
            tietodo = 1;
          };
          break;
        case CHORDON:
          inchord = 1;
          break;
        case CHORDOFF:
          inchord = 0;
          /* subtract time from tied time */
          addfract(&tied_num, &tied_denom, -num[place], denom[place]);
          break;
        default:
          break;
      };
      place = place + 1;
    };
    if (tietodo == 1) {
      event_error("Could not find note to be tied");
    };
  };
}
      
addfract(xnum, xdenom, a, b)
/* add a/b to the count of units in the bar */
int *xnum;
int *xdenom;
int a, b;
{
  *xnum = (*xnum)*b + a*(*xdenom);
  *xdenom = (*xdenom) * b;
  reduce(xnum, xdenom);
}

applybroken(place, type, n)
int place, type, n;
/* adjust lengths of broken notes */
{
  int num1, num2, denom;
  int j;
  int forechord, forestart, foreend, backchord, backstart, backend;
  int failed, lastnote;

  j = place;
  failed = 0;
  forestart = -1;
  foreend = -1;
  forechord = 0;
  /* find following note or chord */
  while ((!failed) && (forestart == -1)) {
    if ((feature[j] == NOTE) || (feature[j] == REST)) {
      forestart = j;
      if (forechord) {
        lastnote = forestart;
      } else {
        foreend = forestart;
      };
    };
    if ((feature[j] == GRACEON) || (feature[j] == TIE)) {
      event_error("Unexpected item following broken rhythm");
    };
    if (feature[j] == CHORDON) {
      forechord = 1;
    };
    j = j + 1;
    if (j>=notes) {
      failed = 1;
    };
  };
  /* look for extend of chord if there is one */
  while ((!failed) && (foreend == -1)) {
    if ((feature[j] == NOTE) || (feature[j] == REST)) {
      lastnote = j;
    };
    if ((feature[j] == GRACEON) || (feature[j] == TIE)) {
      event_error("Unexpected item following broken rhythm");
    };
    if (feature[j] == CHORDOFF) {
      foreend = lastnote;
    };
    j = j + 1;
    if (j>=notes) {
      failed = 1;
    };
  };
  /* look for note or chord before broken rhythm symbol */
  j = place;
  backend = -1;
  backstart = -1;
  backchord = 0;
  while ((!failed) && (backend == -1)) {
    if ((feature[j] == NOTE) || (feature[j] == REST)) {
      backend = j;
      if (backchord) {
        lastnote = backend;
      } else {
        backstart = backend;
      };
    };
    if ((feature[j] == GRACEOFF) || (feature[j] == TIE)) {
      event_error("Unexpected item preceding broken rhythm");
    };
    if (feature[j] == CHORDOFF) {
      backchord = 1;
    };
    j = j - 1;
    if (j<0) {
      failed = 1;
    };
  };
  /* look for extend of chord if there is one */
  while ((!failed) && (backstart == -1)) {
    if ((feature[j] == NOTE) || (feature[j] == REST)) {
      lastnote = j;
    };
    if ((feature[j] == GRACEON) || (feature[j] == TIE)) {
      event_error("Unexpected item following broken rhythm");
    };
    if (feature[j] == CHORDON) {
      backstart = lastnote;
    };
    j = j - 1;
    if (j<0) {
      failed = 1;
    };
  };
  switch(n) {
    case 1:
      num1 = 4;
      num2 = 2;
      break;
    case 2:
      num1 = 7;
      num2 = 1;
      break;
    case 3:
      num1 = 15;
      num2 = 1;
      break;
  };
  denom = (num1 + num2)/2;
  if (type == LT) {
    j = num1;
    num1 = num2;
    num2 = j;
  };
  if (failed) {
    event_error("Cannot apply broken rhythm");
  } else {
    for (j=backstart; j<=backend; j++) {
      lenmul(j, num1, denom);
    };
    for (j=forestart; j<=foreend; j++) {
      lenmul(j, num2, denom);
    };
  };
}

applygrace(place)
int place;
/* assign lengths to grace notes before generating MIDI */
{
  int start, end, p;
  int next_num, next_denom;
  int fact_num, fact_denom;
  int grace_num, grace_denom;
  int j;
  int nextinchord;
  int hostnotestart, hostnoteend;

  j = place;
  start = -1;
  while ((j < notes) && (start == -1)) {
    if (feature[j] == GRACEON) {
      start = j;
    };
    if (feature[j] == GRACEOFF) {
      event_error("} with no matching {");
    };
    j = j + 1;
  };
  /* now find end of grace notes */
  end = -1;
  while ((j < notes) && (end == -1)) {
    if (feature[j] == GRACEOFF) {
      end = j;
    };
    if ((feature[j] == GRACEON) && (j != start - 1)) {
      event_error("nested { not allowed");
    };
    j = j + 1;
  };
  /* now find following note */
  nextinchord = 0;
  hostnotestart = -1;
  while ((hostnotestart == -1) && (j < notes)) {
    if ((feature[j] == NOTE) || (feature[j] == REST)) {
      hostnotestart = j;
    };
    if (feature[j] == GRACEON) {
      event_error("Intervening note needed between grace notes");
    };
    if (feature[j] == CHORDON) {
      nextinchord = 1;
    };
    j = j + 1;
  };
  hostnoteend = -1;
  if (nextinchord) {
    while ((hostnoteend == -1) && (j < notes)) {
      if (feature[j] == CHORDOFF) {
        hostnotestart = j;
      };
      j = j + 1;
    };
  } else {
    hostnoteend = hostnotestart;
  };
  if (hostnotestart == -1) {
    event_error("No note found to follow grace notes");
  } else {
    /* count up grace units */
    grace_num = 0;
    grace_denom = 1;
    p = start;
    while (p <= end) {
      if ((feature[p] == NOTE) || (feature[p] == REST)) {
        grace_num = grace_num * denom[p] + grace_denom * num[p];
        grace_denom = grace_denom * denom[p];
        reduce(&grace_num, &grace_denom);
      };
      p = p + 1;
    };
    /* adjust host note or notes */
    p = hostnotestart;
    while (p <= hostnoteend) {
      if ((feature[p] == NOTE) || (feature[p] == REST)) {
        next_num = num[p];
        next_denom = denom[p];
        num[p] = num[p] * (gfact_denom - gfact_num);
        denom[p] = next_denom * gfact_denom;
        reduce(&num[p], &denom[p]);
      };
      p = p + 1;
    };
    fact_num = next_num * grace_denom * gfact_num;
    fact_denom = next_denom * grace_num * gfact_denom;
    reduce(&fact_num, &fact_denom);
    /* adjust length of grace notes */
    p = start;
    while (p <= end) {
      lenmul(p, fact_num, fact_denom);
      p = p + 1;
    };
  };
}

dograce()
/* assign lengths to grace notes before generating MIDI */
{
  int j;

  j = 0;
  while (j < notes) {
    if (feature[j] == GRACEON) {
      applygrace(j);
    };
    if (feature[j] == SETGRACE) {
      gfact_num = pitch[j];
      gfact_denom = denom[j];
    };
    if (feature[j] == LINENUM) {
      lineno = pitch[j];
    };
    j = j + 1;
  };
}

dobroken()
/* adjust note lengths for broken rhythm > < >> << >>> <<< */
{
  int j;

  j = 0;
  while (j < notes) {
    if ((feature[j] == GT) || (feature[j] == LT)) {
      applybroken(j, feature[j], pitch[j]);
    };
    if (feature[j] == LINENUM) {
      lineno = pitch[j];
    };
    j = j + 1;
  };
}

lenmul(n, a, b)
/* multiply note length by a/b */
int n, a, b;
{
  if ((feature[n] == NOTE) || (feature[n] == REST)) {
    num[n] = num[n] * a;
    denom[n] = denom[n] * b;
    reduce(&num[n], &denom[n]);
  };
}

softcheckbar(pass)
/* allows repeats to be in mid-bar */
int pass;
{
  if ((bar_num-barsize*(bar_denom) >= 0) || (barno <= 0)) {
    checkbar(pass);
  };
}

checkbar(pass) 
/* check to see we have the right number of notes in the bar */
int pass;
{
  char msg[80];

  /* allow zero length bars for typesetting purposes */
  if ((bar_num-barsize*(bar_denom) != 0) && 
      (bar_num != 0) && ((pass == 2) || (barno != 0))) {
    sprintf(msg, "Bar %d has \0", barno);
    numstr(bar_num, msg+strlen(msg));
    if (bar_denom != 1) {
      strcpy(msg+strlen(msg), "/");
      numstr(bar_denom, msg+strlen(msg));
    };
    strcpy(msg+strlen(msg), " units instead of ");
    numstr(barsize, msg+strlen(msg)); 
    if (pass == 2) {
      strcpy(msg+strlen(msg), " in repeat");
    };
    event_warning(msg);
  };
  if (bar_num > 0) {
    barno = barno + 1;
  };
  bar_num = 0;
  bar_denom = 1;
  /* zero place in gchord sequence */
  g_ptr = 0;
  addtoQ(0, g_denom, -1, g_ptr, 0);
  g_next = 0;
}

addunits(a, b)
/* add a/b to the count of units in the bar */
int a, b;
{
  bar_num = bar_num*(b*b_denom) + (a*b_num)*bar_denom;
  bar_denom = bar_denom * (b*b_denom);
  reduce(&bar_num, &bar_denom);
}

reduce(a, b)
/* elimate common factors in fraction a/b */
int *a, *b;
{
  int sign;
  int t, n, m;

  if (*a < 0) {
    sign = -1;
    *a = -*a;
  } else {
    sign = 1;
  };
  /* find HCF using Euclid's algorithm */
  if (*a > *b) {
    n = *a;
    m = *b;
  } else {
    n = *b;
    m = *a;
  };
  while (m != 0) {
    t = n % m;
    n = m;
    m = t;
  };
  *a = (*a/n)*sign;
  *b = *b/n;
}

save_state(vec, a, b, c, d, e)
/* save status when we go into a repeat */
int vec[5];
int a, b, c, d, e;
{
  vec[0] = a;
  vec[1] = b;
  vec[2] = c;
  vec[3] = d;
  vec[4] = e;
}

restore_state(vec, a, b, c, d, e)
/* restore status when we loop back to do second repeat */
int vec[5];
int *a, *b, *c, *d, *e;
{
  *a = vec[0];
  *b = vec[1];
  *c = vec[2];
  *d = vec[3];
  *e = vec[4];
}

zerobar()
/* start a new count of beats in the bar */
{
  bar_num = 0;
  bar_denom = 1;
  barno = barno + 1;
}

placerep(j)
/* patch up missing repeat */
int j;
{
  event_warning("Assuming repeat");
  switch(feature[j]) {
  case DOUBLE_BAR:
    feature[j] = BAR_REP;
    break;
  case SINGLE_BAR:
    feature[j] = BAR_REP;
    break;
  case REP_BAR:
    feature[j] = DOUBLE_REP;
    break;
  case BAR_REP:
    event_error("Too many end repeats");
    break;
  case DOUBLE_REP:
    event_error("Too many end repeats");
    break;
  default:
    event_error("Internal error - please report");
    break;
  };
}

fixreps()
/* find and correct missing repeats in music */
{
  int j;
  int rep_point;
  int expect_repeat;
  int use_next;

  expect_repeat = 0;
  use_next = 0;
  j = 0;
  while (j < notes) {
    switch(feature[j]) {
    case SINGLE_BAR:
      if (use_next) {
        rep_point = j;
        use_next = 0;
      };
      break;
    case DOUBLE_BAR:
      rep_point = j;
      use_next = 0;
      break;
    case BAR_REP:
      expect_repeat = 1;
      use_next = 0;
      break;
    case REP_BAR:
      if (!expect_repeat) {
        placerep(rep_point);
      };
      expect_repeat = 0;
      rep_point = j;
      use_next = 0;
      break;
    case REP_BAR2:
      if (!expect_repeat) {
        placerep(rep_point);
      };
      expect_repeat = 0;
      use_next = 1;
      break;
    case DOUBLE_REP:
      if (!expect_repeat) {
        placerep(rep_point);
      };
      expect_repeat = 1;
      break;
    default:
      break;
    };
    j = j + 1;
  };
}

int findchannel()
/* work out next available channel */
{
  int j;

  j = 0;
  while ((j<16) && (channels[j] != 0)) {
    j = j + 1;
  };
  if (j == 16) {
    event_error("Not enough channels");
    j = 0;
  };
  return (j);
}

findpart(j, partno)
/* find out where part starts */
int *j;
int *partno;
{
  char newpart;
  int  n;

  *partno = *partno + 1;
  while ((*partno <= parts) && 
         (part_start[(int)part[*partno] - (int) 'A'] == -1)) {
    event_error("Part not defined");
    *partno = *partno + 1;
  };
  if (*partno > parts) {
    *j = notes;
  } else {
    newpart = part[*partno];
    n = (int) newpart - (int) 'A';
    *j = part_start[n];
  };
}

checkline(j, w)
int *j;
int *w;
/* make sure number of syllables in w: matches number of notes */
{
  char msg[100];

  if (koverflow > 0) {
    sprintf(msg, "%d more syllables than notes for word line %d", 
                 koverflow, *w);
    event_error(msg);
  } else {
    while ((*j < notes) && (feature[*j] != NEWLINE)) {
      koverflow = koverflow - advancenote(j);
    };
    if (koverflow < 0) {
      sprintf(msg, "%d more notes than syllables for word line %d", 
                 -koverflow, *w);
      event_error(msg);
    };
  };     
  *j = *j + 1;
  koverflow = 0;
}

char getword(syllable, place, w, j)
/* picks up next syllable out of w: field */
char syllable[200];
int* place;
int* w;
int* j;
{
  char c;
  int i;

  i = 0;
  if (*w >= wcount) {
    syllable[i] = '\0';
    return ('\0');
  };
  if (*place == 0) {
    if ((*w % 2) == 0) {
      syllable[i] = '/';
    } else {
      syllable[i] = '\\';
    };
    i = i + 1;
  };
  if (kspace) {
    syllable[i] = ' ';
    i = i + 1;
  };
  c = *(words[*w]+(*place));
  while ((c != ' ') && (c != '-') && (c != '\0') && (c != '|') && (c != '*') &&
         (c != '_') && (c != '\\')) {
    syllable[i] = c;
    if (syllable[i] == '~') syllable[i] = ' ';
    if ((syllable[i] == '\\') && (*(words[*w]+(*place+1)) == '-')) {
      syllable[i] = '-';
      *place = *place + 1;
    };
    i = i + 1;
    *place = *place + 1;
    c = *(words[*w]+(*place));
  };
  switch (c)  {
  case ' ':
    /* skip over any other spaces */
    while (*(words[*w]+(*place)) == ' ') {
      *place = *place + 1;
    };
    kspace = 1;
    break;
  case '-':
    *place = *place + 1;
    kspace = 0;
    break;
  case '_':
    *place = *place + 1;
    kspace = 0;
    break;
  case '\0':
    *w = *w + 1;
    kspace = 0;
    *place = 0;
    break;
  case '\\':
    *w = *w + 1;
    kspace = 1;
    *place =0;
    break;
  case '|':
    *place = *place + 1;
    break;
  case '*':
    kspace = 0;
    *place = *place + 1;
    break;
  };
  syllable[i] = '\0';
  return (c);
}

int advancenote(j)
/* part of w: field handling - steps on one note */
int* j;
{
  int done;

  if (feature[*j] == NEWLINE) {
    koverflow = koverflow + 1;
    return(0);
  };
  done = 0;
  if (ktied) {
    ktied = 0;
    return(0);
  };
  while ((*j < notes) && (done == 0) && (feature[*j] != NEWLINE)) {
    switch(feature[*j]) {
    case NOTE:
      if (!inchord) {
        delay(num[*j], denom[*j], 0);
        done = 1;
      };
      break;
    case REST:
      if (!inchord) {
        delay(num[*j], denom[*j], 0);
      };
      break;
    case CHORDON:
      inchord = 1;
      break;
    case CHORDOFF:
      inchord = 0;
      delay(num[*j], denom[*j], 0);
      done = 1;
      break;
    case VOICE:
      if (pitch[*j] != 1) {
        int foundvoice;

        foundvoice = 0;
        while ((*j < notes) && (!foundvoice)) {
          if (((feature[*j] == VOICE) && (pitch[*j] == 1)) ||
              (feature[*j] == PART)) {
            foundvoice = 1;
          } else {
            *j = *j + 1;
          };
        };
      };
      break;
    case TEMPO:
      if (*j > 0) {
        div_factor = pitch[*j];
      };
      break;
    case TITLE:
      break;
    case LINENUM:
      lineno = pitch[*j];
      break;
    default:
      break;
    };
    *j = *j + 1;
  };
  if (done == 0) {
    koverflow = koverflow + 1;
  };
  return(done);
}

advancebar(j)
/* part of w: field handling - goes to start of next bar */
int* j;
{
  int done;

  if (feature[*j] == NEWLINE) {
    koverflow = koverflow + 1;
    return(0);
  };
  ktied = 0;
  done = 0;
  while ((*j < notes) && (done == 0) && (feature[*j] != NEWLINE)) {
    switch(feature[*j]) {
    case NOTE:
      if (!inchord) {
        delay(num[*j], denom[*j], 0);
      };
      break;
    case REST:
      if (!inchord) {
        delay(num[*j], denom[*j], 0);
      };
      break;
    case CHORDON:
      inchord = 1;
      break;
    case CHORDOFF:
      inchord = 0;
      delay(num[*j], denom[*j], 0);
      break;
    case VOICE:
      if (pitch[*j] != 1) {
        int foundvoice;

        foundvoice = 0;
        while ((*j < notes) && (!foundvoice)) {
          if (((feature[*j] == VOICE) && (pitch[*j] == 1)) ||
              (feature[*j] == PART)) {
            foundvoice = 1;
          } else {
            *j = *j + 1;
          };
        };
      };
      break;
    case SINGLE_BAR:
      done = 1;
      break;
    case DOUBLE_BAR:
      done = 1;
      break;
    case BAR_REP:
      done = 1;
      break;
    case REP_BAR:
      done = 1;
      break;
    case BAR1:
      done = 1;
      break;
    case REP_BAR2:
      done = 1;
      break;
    case DOUBLE_REP:
      done = 1;
      break;
    case TEMPO:
      if (*j > 0) {
        div_factor = pitch[*j];
      };
      break;
    case LINENUM:
      lineno = pitch[*j];
      break;
    default:
      break;
    };
    *j = *j + 1;
  };
  if (done == 0) {
    koverflow = koverflow + 1;
  };
}

karaoketrack()
/* write out karaoke track based on w: fields */
{
  int w;
  int j;
  char syllable[40];
  int place;
  char term;

  text_data("@LENGL");
  j = 0;
  while ((j < notes) && (feature[j] != TITLE)) j = j+1;
  if (feature[j] == TITLE) {
    char atitle[200];

    strcpy(atitle, "@T");
    strcpy(atitle+2, atext[pitch[j]]);
    text_data(atitle);
  };
  w = 0;
  place = 0;
  j = 0;
  inchord = 0;
  kspace = 0;
  ktied = 0;
  koverflow = 0;
  while (w < wcount) {
    term = getword(syllable, &place, &w, &j);
    if (strlen(syllable) > 0) text_data(syllable);
    switch (term) {
      case ' ':
        advancenote(&j);
        break;
      case '-':
        advancenote(&j);
        break;
      case '_':
        advancenote(&j);
        break;
      case '\0':
        advancenote(&j);
        checkline(&j, &w);
        break;
      case '\\':
        advancenote(&j);
        break;
      case '|':
        advancebar(&j);
        break;
      case '*':
        advancenote(&j);
        advancenote(&j);
        break;
    };
  };
}

long mywritetrack(xtrack)
/* this routine writes a MIDI track  */
int xtrack;
{
  int track;
  int j, pass;
  int expect_repeat;
  int slurring;
  int state[5];
  int foundvoice;
  void mf_write_tempo();

  track = xtrack;
  if (karaoke) {
    if (track == 2) {
      delta_time = 0;
      karaoketrack();
      return(0L);
    } else {
      if (track > 2) track = track - 1;
    };
  };
  if (track == 0) {
    if (karaoke) {
      text_data("@KMIDI KARAOKE FILE");
    };
    mf_write_tempo(tempo);
    /* write key */
    write_keysig(sf, mi);
    /* write timesig */
    write_meter(time_num, time_denom);
    if (maxvoice > 0) {
       return(0L);
    };
  } else {
    if (verbose) {
      printf("track %d\n", track);
    };
  };
  tracklen = 0;
  partno = -1;
  g_started = 0;
  g_ptr = 0;
  addtoQ(0, g_denom, -1, g_ptr, 0);
  g_next = 0;
  tuplecount = 0;
  inchord = 0;
  channel = findchannel();
  channels[channel] = 1;
  if (hasgchords) {
    fun.chan = findchannel();
    channels[fun.chan] = 1;
    gchord.chan = findchannel();
    channels[fun.chan] = 0;
  };
  channels[channel] = 0;
  if (program != -1) {
    write_program(program, channel);
  };
  /* write notes */
  j = 0;
  if ((maxvoice == 0) || (track == 1)) {
    foundvoice = 1;
  } else {
    foundvoice = 0;
    while ((j < notes) && (!foundvoice)) {
      if ((feature[j] == VOICE) && (pitch[j] == track)){
        foundvoice = 1;
      };
      j = j + 1;
    };
  };
  barno = 0;
  bar_num = 0;
  bar_denom = 1;
  delta_time = 0;
  err_num = 0;
  err_denom = 1;
  pass = 1;
  save_state(state, j, barno, div_factor, transpose, channel);
  slurring = 0;
  expect_repeat = 0;
  while (j < notes) {
    switch(feature[j]) {
    case NOTE:
      noteon(j);
      /* set up note off */
      addtoQ(num[j], denom[j], pitch[j] + transpose, channel, -1);
      if (!inchord) {
        delay(num[j], denom[j], 0);
        addunits(num[j], denom[j]);
      };
      break;
    case TNOTE:
      noteon(j);
      /* set up note off */
      addtoQ(num[j], denom[j], pitch[j] + transpose, channel, -1);
      break;
    case REST:
      if (!inchord) {
        delay(num[j], denom[j], 0);
        addunits(num[j], denom[j]);
      };
      break;
    case CHORDON:
      inchord = 1;
      break;
    case CHORDOFF:
      inchord = 0;
      delay(num[j], denom[j], 0);
      addunits(num[j], denom[j]);
      break;
    case LINENUM:
      /* get correct line number for diagnostics */
      lineno = pitch[j];
      break;
    case PART:
      if (parts == -1) {
        char msg[1];

        msg[0] = (char) pitch[j];
        mf_write_meta_event((long) 0, marker, msg, 1);
      } else {
        findpart(&j, &partno);
      };
      break;
    case VOICE:
      if (pitch[j] == track) {
        foundvoice = 1;
      } else {
        foundvoice = 0;
        while ((j < notes) && (!foundvoice)) {
          if (((feature[j] == VOICE) && (pitch[j] == track)) ||
              ((feature[j] == PART) && (track == 1))) {
            foundvoice = 1;
          } else {
            j = j + 1;
          };
        };
      };
      break;
    case TEXT:
      mf_write_meta_event((long) 0, text_event, atext[pitch[j]],
                          strlen(atext[pitch[j]]));
      break;
    case TITLE:
      mf_write_meta_event((long) 0, sequence_name, atext[pitch[j]],
                          strlen(atext[pitch[j]]));
      break;
    case SINGLE_BAR:
      checkbar(pass);
      break;
    case DOUBLE_BAR:
      checkbar(pass);
      break;
    case BAR_REP:
      softcheckbar(pass);
      if (expect_repeat) {
        event_error("Expected end repeat not found at |:");
      };
      save_state(state, j, barno, div_factor, transpose, channel);
      expect_repeat = 1;
      pass = 1;
      break;
    case REP_BAR:
      softcheckbar(pass);
      if (pass == 1) {
        if (!expect_repeat) {
          event_error("Found unexpected :|");
        } else {
          pass = 2;
          restore_state(state, &j, &barno, &div_factor, &transpose, &channel);
          slurring = 0;
        };
      } else {
        pass = 1;
        expect_repeat = 0;
      };
      break;
    case BAR1:
      checkbar(pass);
      if (pass == 2) {
        while ((j<notes) && (feature[j] != REP_BAR2)) {
          j = j + 1;
        };
        barno = barno + 1;
        if (feature[j] != REP_BAR2) {
          event_error("Not found :|2");
        };
        pass = 1;
        expect_repeat = 0;
      } else {
        if (!expect_repeat) {
          event_error("|1 or | [1 found outside repeat");
        };
      };
      break;
    case REP_BAR2:
      checkbar(pass);
      if (pass == 1) {
        if (!expect_repeat) {
          event_error(":|2 found outside repeat context");
        } else {
          restore_state(state, &j, &barno, &div_factor, &transpose, &channel);
          slurring = 0;
          pass = 2;
        };
      } else {
        event_error("Found unexpected :|2");
      };
      break;
    case DOUBLE_REP:
      softcheckbar(pass);
      if (pass == 2) {
        expect_repeat = 1;
        save_state(state, j, barno, div_factor, transpose, channel);
        pass = 1;
      } else {
        if (!expect_repeat) {
          event_error("Found unexpected ::");
          expect_repeat = 1;
          save_state(state, j, barno, div_factor, transpose, channel);
          pass = 1;
        } else {
          restore_state(state, &j, &barno, &div_factor, &transpose, &channel);
          slurring = 0;
          pass = 2;
        };
      };
      break;
    case GCHORD:
      basepitch = pitch[j];
      seventh = denom[j] & 1;
      mode = (enum chordmode) ((denom[j] & 6) >> 1);
      force_fun = (denom[j] & 8) >> 3;
      g_started = 1;
      break;
    case GCHORDON:
      gchords = 1;
      break;
    case GCHORDOFF:
      gchords = 0;
      break;
    case DYNAMIC:
      dodeferred(atext[pitch[j]]);
      break;
    case KEY:
      write_keysig(pitch[j], denom[j]);
      break;
    case TIME:
      write_meter(pitch[j], denom[j]);
      break;
    case TEMPO:
      if (j > 0) {
        div_factor = pitch[j];
      };
      break;
    case CHANNEL:
      channel = pitch[j];
      break;
    case PROGRAM:
      write_program(pitch[j], denom[j]);
      break;
    case TRANSPOSE:
      transpose = pitch[j];
      break;
    case SLUR_ON:
      if (slurring) {
        event_error("Unexpected start of slur found");
      };
      slurring = 1;
      break;
    case SLUR_OFF:
      if (!slurring) {
        event_error("Unexpected end of slur found");
      };
      slurring = 0;
      break;
    default:
      break;
    };
    j = j + 1;
  };
  if (expect_repeat) {
    event_error("Missing :| at end of tune");
  };
  clearQ();
  tracklen = tracklen + delta_time;
  if (track == 1) {
    tracklen1 = tracklen;
  } else {
    if ((track != 0) && (tracklen != tracklen1)) {
      char msg[100];

      sprintf(msg, "Track %d is %Ld units long not %Ld", 
              track, tracklen, tracklen1);
      event_warning(msg);
    };
  };
  return (delta_time);
}

set_meter(n, m)
int n, m;
{
  /* set up barsize */
  barsize = n;
  if (barsize % 3 == 0) {
    beat = 3;
  } else {
    if (barsize % 2 == 0) {
      beat = 2;
    } else {
      beat = barsize;
    };
  };
  /* correction factor to make sure we count in the right units */
  if (m > 4) {
    b_num = m/4;
    b_denom = 1;
  } else {
   b_num = 1;
   b_denom = 4/m;
  };
}

write_meter(n, m)
int n, m;
{
  int t, dd;
  char data[4];

  set_meter(n, m);

  dd = 0;
  t = m;
  while (t > 1) {
    dd = dd + 1;
    t = t/2;
  };
  data[0] = (char)n;
  data[1] = (char)dd;
  if (n%2 == 0) {
    data[2] = (char)(24*2*n/m);
  } else {
    data[2] = (char)(24*n/m);
  };
  data[3] = 8;
  mf_write_meta_event((long) 0, time_signature, data, 4);
}

write_keysig(sf, mi)
int sf, mi;
{
  char data[2];

  data[0] = (char) (0xff & sf);
  data[1] = (char) mi;
  mf_write_meta_event((long) 0, key_signature, data, 2);
}

noteon(n)
/* compute note data and call noteon_data to write MIDI note event */
int n;
{
  int vel;

  /* set velocity */
  if (bar_num == 0) {
    vel = loudnote;
  } else {
    if ((bar_denom == 1) && ((bar_num % beat) == 0)) {
      vel = mednote;
    } else {
      vel = softnote;
    };
  };
  noteon_data(pitch[n] + transpose, channel, vel);
}

text_data(s)
/* write text event */
char* s;
{
  mf_write_meta_event(delta_time, text_event, s, strlen(s));
  tracklen = tracklen + delta_time;
  delta_time = (long) 0;
}

noteon_data(pitch, channel, vel)
int pitch, channel, vel;
{
  midi_noteon(delta_time, pitch, channel, vel);
  tracklen = tracklen + delta_time;
  delta_time = (long) 0;
}

write_program(p, channel)
int p, channel;
{
  char data[1];

  data[0] = p;
  mf_write_midi_event(delta_time, program_chng, channel, data, 1);
  tracklen = tracklen + delta_time;
  delta_time = 0;
}

write_control(channel, data, n)
/* write MIDI control event */
int channel, n;
char data[];
{
  mf_write_midi_event(delta_time, control_change, channel, data, n);
}

delay(a, b, c)
/* wait for time a/b */
int a, b, c;
{
  int dt;

  dt = (div_factor*a)/b + c;
  err_num = err_num * b + ((div_factor*a)%b)*err_denom;
  err_denom = err_denom * b;
  reduce(&err_num, &err_denom);
  dt = dt + (err_num/err_denom);
  err_num = err_num%err_denom;
  timestep(dt, 0);
}

midi_noteon(delta_time, pitch, chan, vel)
long delta_time;
int pitch, chan, vel;
{
  char data[2];

  data[0] = (char) pitch;
  data[1] = (char) vel;
  mf_write_midi_event(delta_time, note_on, chan, data, 2);
  channels[chan] = 1;
}

midi_noteoff(delta_time, pitch, chan)
long delta_time;
int pitch, chan;
{
  char data[2];

  data[0] = (char) pitch;
  data[1] = (char) 0;
  mf_write_midi_event(delta_time, note_off, chan, data, 2);
}

save_note(num, denom, pitch, chan, vel)
/* queue up note */
int num, denom;
int pitch, chan, vel;
{
  noteon_data(pitch, chan, vel);
  addtoQ(num, denom, pitch, chan, -1);
}

dogchords(i)
/* generate accompaniment notes */
int i;
{
  if ((i == g_ptr) && (g_ptr < strlen(gchord_seq))) {
    int len;
    char action;

    action = gchord_seq[g_ptr];
    len = gchord_len[g_ptr];
    if (force_fun && (action == 'c')) {
      action = 'f';
    };
    switch (action) {
    case 'z':
      break;
    case 'c':
      /* do chord */
      if (g_started && gchords) {
        save_note(g_num*len, g_denom, basepitch+gchord.base, gchord.chan, 
                  gchord.vel);
        if ((mode == min) || (mode == dim)) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+3, 
                    gchord.chan, gchord.vel);
        } else {
          save_note(g_num*len, g_denom, basepitch+gchord.base+4, 
                    gchord.chan, gchord.vel);
        };
        if ((mode == maj) || (mode == min)) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+7, gchord.chan,
                  gchord.vel);
        };
        if (mode == dim) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+6, gchord.chan,
                  gchord.vel);
        };
        if (mode == aug) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+8, gchord.chan,
                  gchord.vel);
        };
        if (seventh) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+11, 
                    gchord.chan, gchord.vel);
        };
      };
      break;
    case 'f':
      if (g_started && gchords) {
        /* do fundamental */
        save_note(g_num*len, g_denom, basepitch+fun.base, fun.chan, fun.vel);
      };
    };
    g_ptr = g_ptr + 1;
    addtoQ(g_num*len, g_denom, -1, g_ptr, 0);
  };
}

/* routines to handle note queue */

addtoQ(num, denom, pitch, chan, d) 
int num, denom, pitch, chan, d;
{
  int i, done;
  int wait;
  int *ptr;

  wait = ((div_factor*num)/denom) + d;
  /* find free space */
  if (freehead == -1) {
    event_error("Note off queue overflow");
  } else {
    i = freehead;
    freehead = Q[freehead].next;
  };
  Q[i].pitch = pitch;
  Q[i].chan = chan;
  /* find place in queue */
  ptr = &Qhead;
  done = 0;
  while (!done) {
    if (*ptr == -1) {
      *ptr = i;
      Q[i].next = -1;
      Q[i].delay = wait;
      done = 1;
    } else {
      if (Q[*ptr].delay > wait) {
        Q[*ptr].delay = Q[*ptr].delay - wait;
        Q[i].next = *ptr;
        Q[i].delay = wait;
        *ptr = i;
        done = 1;
      } else {
        wait = wait - Q[*ptr].delay;
        ptr = &Q[*ptr].next;
      };
    };
  };
}

removefromQ(i)
int i;
{
  if (i == -1) {
    event_error("Internal error - nothing to remove from queue");
  } else {
    if (Q[Qhead].delay != 0) {
      event_error("Internal error - queue head has non-zero time");
    };
    Qhead = Q[i].next;
    Q[i].next = freehead;
    freehead = i;
  };
}

clearQ()
{
  int time;

  /* remove gchord requests */
  time = 0;
  while ((Qhead != -1) && (Q[Qhead].pitch == -1)) {
    time = time + Q[Qhead].delay;
    Qhead = Q[Qhead].next;
  };
  if (Qhead != -1) {
    timestep(time, 1);
  };
  /* do any remaining note offs, but don't do chord request */
  while (Qhead != -1) {
    event_error("Sustained notes beyond end of track");
    timestep(Q[Qhead].delay+1, 1);
  };
}

printQ()
{
  int t;

  t = Qhead;
  printf("Q:");
  while (t != -1) {
    printf("p(%d)-%d->", Q[t].pitch, Q[t].delay);
    t = Q[t].next;
  };
  printf("\n");
}

timestep(t, atend)
int t;
int atend;
{
  int time;
  int headtime;

  time = t;
  /* process any notes waiting to finish */
  while ((Qhead != -1) && (Q[Qhead].delay < time)) {
    headtime = Q[Qhead].delay;
    delta_time = delta_time + (long) headtime;
    time = time - headtime;
    advanceQ(headtime);
    if (Q[Qhead].pitch == -1) {
      if (!atend) {
        dogchords(Q[Qhead].chan);
      };
    } else {
      midi_noteoff(delta_time, Q[Qhead].pitch, Q[Qhead].chan);
      tracklen = tracklen + delta_time;
      delta_time = (long) 0;
    };
    removefromQ(Qhead);
  };
  if (Qhead != -1) {
    advanceQ(time);
  };
  delta_time = delta_time + (long) time;
}

advanceQ(t)
int t;
{
  if (Qhead == -1) {
    event_error("Internal error - empty queue");
  } else {
    Q[Qhead].delay = Q[Qhead].delay - t;
  };
}

startfile()
/* called at the beginning of an abc tune by event_refno */
/* This sets up all the default values */
{
  int j;

  if (verbose) {
    printf("scanning tune\n");
  };
  /* set up defaults */
  sf = 0;
  mi = 0;
  setmap(0, basemap, basemul);
  copymap();
  time_num = 4;
  time_denom = 4;
  default_length = -1;
  event_tempo(120, 1, 4, 0);
  notes = 0;
  ntexts = 0;
  loudnote = 105;
  mednote = 95;
  softnote = 80;
  program = -1;
  gfact_num = 1;
  gfact_denom = 3;
  transpose = 0;
  hornpipe = 0;
  karaoke = 0;
  wcount = 0;
  parts = -1;
  middle_c = 60;
  for (j=0; j<26; j++) {
    part_start[j] = -1;
  };
  for (j=0; j<16;j++) {
    channels[j] = 0;
  };
  hasgchords = 0;
  gchords = 1;
  set_gchords("z");
  fun.base = 36;
  fun.vel = 80;
  gchord.base = 48;
  gchord.vel = 75;
#ifdef NOFTELL
  buffcount = 0;
  buffering = 0;
#endif
}

tempounits(t_num, t_denom)
/* interprets Q: once default length is known */
int *t_num, *t_denom;
{
  /* calculate unit for tempo */
  if (tempo_num == 0) {
    *t_num = 1;
    *t_denom = default_length;
  } else {
    if (relative_tempo) {
      *t_num = tempo_num;
      *t_denom = tempo_denom*default_length;
    } else {
      *t_num = tempo_num;
      *t_denom = tempo_denom;
    };
  };
}

setbeat()
/* default accompaniment patterns for various time signatures */
{
  /* set up chord/fundamental sequence if not already set */
  if ((time_num == 2) && (time_denom == 2)) {
    set_gchords("fzczfzcz");
  };
  if (((time_num == 2) || (time_num == 4)) && (time_denom == 4)) {
    set_gchords("fzczfzcz");
  };
  if ((time_num == 3) && (time_denom == 4)) {
    set_gchords("fzczcz");
  };
  if ((time_num == 6) && (time_denom == 8)) {
    set_gchords("fzcfzc");
  };
  if ((time_num == 9) && (time_denom == 8)) {
    set_gchords("fzcfzcfzc");
  };
}

headerprocess()
/* called after the K: field has been reached, signifying the end of */
/* the header and the start of the tune */
{
  int t_num, t_denom;
  int i;

  gracenotes = 0; /* not in a grace notes section */
  inchord = 0; /* not in a chord */
  inslur = 0; /* not in a slur */
  /* calculate time for a default length note */
  if (default_length == -1) {
    if (((float) time_num)/time_denom < 0.75) {
      default_length = 16;
    } else {
      default_length = 8;
    };
  };
  bar_num = 0;
  bar_denom = 1;
  set_meter(time_num, time_denom);
  if (hornpipe) {
    if ((time_denom != 4) || ((time_num != 2) && (time_num != 4))) {
      event_error("Hornpipe must be in 2/4 or 4/4 time");
      hornpipe = 0;
    };
  };
  barno = 0;
  ingrace = 0;
  inchord = 0;

  tempounits(&t_num, &t_denom);
  /* make tempo in terms of 1/4 notes */
  tempo = (long) 60*1000000*t_denom/(Qtempo*4*t_num);
  div_factor = division;
  setbeat();
  maxvoice = 0;

  /* initialize queue of notes waiting to finish */
  Qhead = -1;
  freehead = 0;
  for (i=0; i<QSIZE-1; i++) {
    Q[i].next = i + 1;
  };
  Q[QSIZE-1].next = -1;
  freetail = QSIZE-1;
}

finishfile()
/* end of tune has been reached - write out MIDI file */
{
  if (!pastheader) {
    event_error("No K: field found at start of tune");
  } else {
    int i; 

    if (parts > -1) {
      addfeature(PART, 'Z', 0, 0);
    };
    if (verbose) {
      printf("handling broken rhythm\n");
    };
    dobroken();
    if (verbose) {
      printf("handling grace notes\n");
    };
    dograce();
    if (verbose) {
      printf("preprocessing\n");
    };
    chordfix();
    tiefix();
    /* preprocess(); */
    if (verbose) {
      printf("fixing repeats\n");
    };
    if ((parts == -1) && (maxvoice <= 1)) {
      fixreps();
    };
    if (check) {
      Mf_putc = nullputc;
      for (i=0; i<= maxvoice; i++) {
        mywritetrack(i);
      };
    } else {
      if ((fp = fopen(outname, "wb")) == NULL) {
        event_error("File open failed");
        exit(1);
      };
      printf("writing MIDI file %s\n", outname);
      Mf_putc = myputc;
      Mf_writetrack = mywritetrack;
      if ((maxvoice == 0) && (!karaoke)) {
        mfwrite(0, 1, division, fp);
      } else {
        if (maxvoice == 0) maxvoice = 1;
        mfwrite(1, maxvoice+karaoke+1, division, fp);
      };
      fclose(fp);
#ifdef __MACINTOSH__
      (void) setOutFileCreator(outname,'Midi','ttxt');
#endif /* __MACINTOSH__ */

    };
    for (i=0; i<ntexts; i++) {
      free(atext[i]);
    };
    for (i=0; i<wcount; i++) {
      free(words[i]);
    };
    free(words);
  };
}

