"""
  PDA.Palm.App.XWord - Crossword Puzzles
  $Id: XWord.py,v 1.5 1998/08/29 01:09:01 rob Exp $
  
  Copyright 1998 Rob Tillotson <rob@io.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License, version 2,
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.


  This module handles databases in the format used by XWord,
  a crossword puzzle program for PalmOS by Penguin Software
  (http://www.roadkill.com/penguin/).

  Crosswords are stored one to a database.  Each database consists
  of a single record which includes the board layout, solution, clues,
  and space to store the player's work in progress.

  In addition to the usual packing and unpacking functions, the Record
  class in this module includes a way to read USA Today format
  crosswords.  Elsewhere in this package there is a small sample
  program which automatically fetches a crossword from the USA Today
  web site and saves it as an XWord database -- this was, in fact, my
  intent in doing this module at all, because I would eventually like
  to have the USA Today puzzle delivered automatically to my Pilot
  every morning.
"""

__version__ = '$Id: XWord.py,v 1.5 1998/08/29 01:09:01 rob Exp $'

import PDA.Palm
from PDA.Palm import FLD_STRING, FLD_INT, FLD_UNKNOWN, FLD_LIST

import string,struct,array,re

class Record(PDA.Palm.Record):
    def __init__(self, raw='', index=0, id=0, attr=0, category=0):
	self.fields = { 'name': (FLD_STRING, ''),
			'description': (FLD_STRING, ''),
			'sizex': (FLD_INT, 0),
			'sizey': (FLD_INT, 0),
			'cluepos': (FLD_UNKNOWN, None),
			'solution': (FLD_UNKNOWN, None),
			'entries': (FLD_UNKNOWN, None),
			'across': (FLD_LIST, []),
			'down': (FLD_LIST, []),
			'random_int_0': (FLD_INT, 1),
			'random_int_1': (FLD_INT, 1)
			}
	PDA.Palm.Record.__init__(self, raw, index, id, attr, category)

    def unpack(self, raw):
	self.raw = raw

	self['random_int_0'], self['random_int_1'], n, d, \
	self['sizex'], self['sizey'] = struct.unpack('>hh32s80shh',raw[0:120])
	
	self['name'] = filter(lambda x: x != '\000', n)
	self['description'] = filter(lambda x: x != '\000', d)

	x = self['sizex']*self['sizey']
	self['cluepos'] = array.array('c', raw[120:120+x])
	self['solution'] = array.array('c', raw[120+x: 120+(2*x)])
	self['entries'] = array.array('c', raw[120+(2*x):120+(3*x)])
	raw = raw[120+(3*x):]

	# scan for a 'fd'
	x = string.index(raw, '\xfd')
	raw = raw[x+1:]
	# and a fe to end the across clues
	y = string.index(raw, '\xfe')
	self['across'] = string.split(raw[x:y], '\0')
	# and now an ff or the end of record
	x = string.find(raw, '\xff')
	if x < 0: self['down'] = string.split(raw[y+1:], '\0')
	else: self['down'] = string.split(raw[y+1:x], '\0')

    def pack(self):
	r = self.packfields('>hh32s80shh',
			    ['random_int_0','random_int_1',
			    'name', 'description', 'sizex', 'sizey'])
	r = r + self['cluepos'].tostring()
	r = r + self['solution'].tostring()
	r = r + self['entries'].tostring()
	r = r + '\xfd'
	r = r + string.join(self['across'], '\0')
	r = r + '\xfe'
	r = r + string.join(self['down'], '\0')
	r = r + '\xff'
	self.raw = r
	return r

    def read_usa_today(self, f):
	"""Load this record from a USA Today format crossword file."""
	l = map(string.strip, f.readlines())
	m = re.match('USA Today\s+([0-9A-Z/]+)\s+(?:"|'')(.+)(?:"|'')\s+.*By\s+(.*)', l[0])
	if not m: raise ValueError, 'not a USA Today puzzle file?'
	date, name, author = m.group(1,2,3)
	self['name'] = 'UT %s "%s"' % (date, name)
	self['description']= \
			 'By %s [Conv: PDA.Palm.App.XWord by Rob Tillotson]' % author

	self['sizex'], self['sizey'] = \
		       tuple(map(string.atoi,
				 map(string.strip, string.split(l[1], ','))))
	l = l[2:]
	p = ''
	for r in range(0, self['sizey']):
	    p = p + string.join(map(lambda x: x < 0 and chr(255) or chr(x),
				    map(string.atoi,
					map(string.strip,
					    string.split(l[r], ',')))),'')
	self['cluepos'] = array.array('c', p)

	l = l[self['sizey']:]
	p = ''
	for r in range(0, self['sizey']):
	    p = p + string.join(map(lambda x: x and x or ' ',
				    map(string.strip,string.split(l[r], ','))), '')
	self['solution'] = array.array('c', p)
	self['entries'] = array.array('c', ' '*self['sizex']*self['sizey'])

	cluesize = ord(max(filter(lambda x: ord(x) < 255, self['cluepos'].tostring())))
	self['across'] = [''] * (cluesize+1)
	self['down'] = [''] * (cluesize+1)
	for clue in l[self['sizey']:]:
	    dir, num, text = tuple(map(string.strip,string.split(clue,',')))
	    num = string.atoi(num)
	    if dir == 'a':
		self['across'][num-1] = '%2dA. %s' % (num, text)
	    elif dir == 'd':
		self['down'][num-1] = '%2dD. %s' % (num, text)

	# chop down so that only one '' is left at the end
	while len(self['across']) > 1 and not self['across'][-1] and \
	      not self['across'][-2]:
	    del self['across'][-1]
	while len(self['down']) > 1 and not self['down'][-1] and not self['down'][-2]:
	    del self['down'][-2]
	    
class Database(PDA.Palm.Database):
    def __init__(self, db, info):
	PDA.Palm.Database.__init__(self, db, info)
	self.record_class = Record
	

PDA.Palm.Types.register(Database, {'type': 'DATa', 'creator':'XWrd'})
