/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: PlextorReaderScan.cc,v $
 * Revision 1.2  1998/09/27 19:20:05  mueller
 * Added retrieval of control nibbles for track with 'analyzeTrack()'.
 *
 * Revision 1.1  1998/09/07 15:15:40  mueller
 * Initial revision
 *
 */

static char rcsid[] = "$Id: PlextorReaderScan.cc,v 1.2 1998/09/27 19:20:05 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>

#include "PlextorReaderScan.h"
#include "PWSubChannel96.h"

#include "Toc.h"
#include "util.h"

PlextorReaderScan::PlextorReaderScan(ScsiIf *scsiIf, Toc *toc)
  : PlextorReader(scsiIf, toc)
{
  driverName_ = "Plextor CD-ROM Reader (scanning) - Version 1.0";
}

// static constructor
CdrDriver *PlextorReaderScan::instance(ScsiIf *scsiIf, Toc *toc)
{
  return new PlextorReaderScan(scsiIf, toc);
}

int PlextorReaderScan::analyzeTrack(int trackNr, long startLba,
				    long endLba, Msf *indexIncrements,
				    int *indexIncrementCnt, long *pregap,
				    char *isrcCode, unsigned char *ctl)
{
  unsigned char cmd[12];
  unsigned char *data;
  int blockLength = AUDIO_BLOCK_LEN + 96;
  int blocksPerRead = scsiIf_->maxDataLen() / blockLength;
  int n, i;
  int actIndex = 1;
  long length;
  long crcErrCnt = 0;
  long timeCnt = 0;
  int ctlSet = 0;

  readIsrc(trackNr, isrcCode);

  if (pregap != NULL)
    *pregap = 0;

  *indexIncrementCnt = 0;
  *ctl = 0;

  data = new (unsigned char)[blocksPerRead * blockLength];

  startLba -= 75;
  if (startLba < 0) {
    startLba = 0;
  }
  length = endLba - startLba;

  memset(cmd, 0, 12);
  cmd[0] = 0xd8;  // READ CDDA
  cmd[10] = 0x02;

  while (length > 0) {
    n = (length > blocksPerRead ? blocksPerRead : length);

    cmd[2] = startLba >> 24;
    cmd[3] = startLba >> 16;
    cmd[4] = startLba >> 8;
    cmd[5] = startLba;
    cmd[7] = n >> 16;
    cmd[8] = n >> 8;
    cmd[9] = n;

    if (sendCmd(cmd, 12, NULL, 0, data, n * blockLength) != 0) {
      delete[] data;
      return 1;
    }

    for (i = 0; i < n; i++) {
      PWSubChannel96 chan(data + i * blockLength + AUDIO_BLOCK_LEN);
      
      if (chan.checkCrc()) {
	if (chan.type() == SubChannel::QMODE1DATA) {
	  int t = chan.trackNr();
	  Msf time(chan.min(), chan.sec(), chan.frame()); // track rel time

	  if (timeCnt > 74) {
	    message(0, "%s\r", time.str());
	    timeCnt = 0;
	  }

	  if (t == trackNr && !ctlSet) {
	    *ctl = chan.ctl();
	    *ctl |= 0x80;
	    ctlSet = 1;
	  }
	  if (t == trackNr && chan.indexNr() > actIndex) {
	    actIndex = chan.indexNr();
	    message(0, "Found index %d at: %s", actIndex, time.str());
	    if ((*indexIncrementCnt) < 98) {
	      indexIncrements[*indexIncrementCnt] = time;
	      *indexIncrementCnt += 1;
	    }
	  }
	  else if (t == trackNr + 1) {
	    if (chan.indexNr() == 0) {
	      if (pregap != NULL)
		*pregap = time.lba();
	    }
	    if (crcErrCnt != 0)
	      message(0, "Found %ld Q sub-channels with CRC errors.",
		      crcErrCnt);
	    
	    delete[] data;
	    return 0;
	  }
	}
      }
      else {
	crcErrCnt++;
#if 0
	if (chan.type() == SubChannel::QMODE1DATA) {
	  message(0, "Q sub-channel data at %02d:%02d:%02d failed CRC check - ignored",
		 chan.min(), chan.sec(), chan.frame());
	}
	else {
	  message(0, "Q sub-channel data failed CRC check - ignored.");
	}
	chan.print();
#endif
      }

      timeCnt++;
    }

    length -= n;
    startLba += n;
  }

  if (crcErrCnt != 0)
    message(0, "Found %ld Q sub-channels with CRC errors.", crcErrCnt);

  delete[] data;
  return 0;
}
