/*
 * pftp -- sends files from host to host through free choosable ports
 *
 * Copyright (C) 1996, 1997, 1998 Ben Schluricke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the emplied warranty of MERCHANT-
 * ABILITY OF FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *    Written by Ben Schluricke
 *    E-Mail:    bhor0533@lehr.chem.TU-Berlin.DE
 *
 * This program is dedicated to my girl-friend, Heather O'Rourke.
 *
 *
 */
#ifdef USE_POSIX_THREAD
#define _REENTRANT
#include <pthread.h>
#endif
#if defined HP_UX || defined OSF1
#include <sys/termio.h>
#endif
#if defined unicos
#include <sys/termios.h>
#endif
#if defined FreeBSD || defined OpenBSD
#include <sys/errno.h>
#include <sys/ioctl_compat.h>
#endif
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <signal.h>
#include "main.h"
#include "connect.h"

#ifdef NEXTSTEP
int putenv(const char *);
#endif

extern void reply_from_server(void);

#ifndef TCGETA
#define TCGETA TIOCGETP
#define TCSETA TIOCSETP
#else
#define SYSV
#endif

#ifdef SYSV
#include <termios.h>
#endif

#define DEFAULTROWS 24 /* defaul number of rows, if window size is unknown */
#define DEFAULTCOLS 80 /* defaul number of columns, if window size is unknown */

void free_memory(void);

void set_tty(int i)
{
#if !defined SYSV
   static struct sgttyb tori;
   struct sgttyb tnew;
#else
   static struct termio tori;
   struct termio tnew;
#endif
   volatile struct winsize ws;

   if (i && isatty(2)) {
      /*
       * Get window size.
       */
      if (i==3) {
         ioctl(2, TIOCGWINSZ, &ws);
         _WINROWS_= ws.ws_row ? ws.ws_row : DEFAULTROWS;
         _WINCOLS_= ws.ws_col ? ws.ws_col : DEFAULTCOLS;

         /*
          * Get current tty settings.
          */
         if (ioctl(2, TCGETA, &tori) < 0) {
            fprintf(stderr, "** ioctl: %s\n", _PFTP_ERROR_ARRAY_);
            exit(1);
         }
         return;
      }

      /*
       * Change terminal modes.
       */
      tnew = tori;
#ifndef SYSV
      tnew.sg_flags |= CBREAK;
      if (i==2) tnew.sg_flags &= ~ECHO;
#else
      if (i==2) tnew.c_lflag &= ~ECHO;
      tnew.c_lflag &= ~ICANON;
      tnew.c_lflag &= ~ISIG;
      tnew.c_cc[VMIN] = 1;
      tnew.c_cc[VTIME] = 0;
#endif

      /*
       * Set new tty modes.
       */
      if (ioctl(2, TCSETA, &tnew) < 0) {
         fprintf(stderr, "** ioctl: %s\n", _PFTP_ERROR_ARRAY_);
         exit(1);
      }
   }
   else if (isatty(2)) {
      /*
       * Set new tty modes.
       */
      if (ioctl(2, TCSETA, &tori) < 0) {
         fprintf(stderr, "** ioctl: %s\n", _PFTP_ERROR_ARRAY_);
         exit(1);
      }
   }
}

/*
 * handler--handler signals.
 */
void handler(int sig)
{
   FILE *fe=stderr;

   signal(SIGPIPE, SIG_IGN);
   signal(SIGINT, SIG_IGN);

   if (sig == SIGPIPE) {
      if (slfp) {
         if ((*statstr)->version) {
            reply_from_server();
         }
         fprintf(slfp, "\n** Connection closed by foreign host.\n");
         if (pftplog) {
            if ((fe = fopen(pftplog, "a")) != NULL) {
               fprintf(fe, "\n** Connection closed by foreign host.\n");
               fclose(fe);
            }
         }
         if (slfp != stderr) fclose(slfp);
      }
      free_memory();
      exit(1);
   }
#ifdef USE_POSIX_THREAD
   else if (sig == SIGINT && CHILDNUM > 0) {
      if (slfp) {
         fprintf(slfp, "\n** Connection was aborded.");
         fprintf(slfp, " There %s %d pending connection%s\n", \
         CHILDNUM==1? "was": "were", CHILDNUM, CHILDNUM==1? ".":"s.");
         if (slfp != stderr) fclose(slfp);
      }
      if (pftplog) {
         if ((fe = fopen(pftplog, "a")) != NULL) {
            fprintf(fe, "\n** Connection was aborded.");
            fprintf(fe, " There %s %d pending connection%s\n", \
            CHILDNUM==1? "was": "were", CHILDNUM, CHILDNUM==1? ".":"s.");
            fclose(fe);
         }
      }
      free_memory();
      exit(1);
   }
   else if (sig == SIGINT && CHILDNUM < 0) {
      if (slfp) {
         fprintf(slfp, "\n** Connection was aborded.\n");
         if (slfp != stderr) fclose(slfp);
      }
      if (pftplog) {
         if ((fe = fopen(pftplog, "a")) != NULL) {
            fprintf(fe, "\n** Connection was aborded.");
            fclose(fe);
         }
      }
      free_memory();
      exit(1);
   }
   else if (sig == SIGINT) {
      if (slfp) {
#ifndef SYSV
         fprintf(slfp, "\b\b  \n*** Socket closed. No further connection request detected.\n");
#else
         fprintf(slfp, "\n*** Socket closed. No further connection request detected.\n");
#endif
         if (slfp != stderr) fclose(slfp);
      }
      free_memory();
      exit(0);
   }
#else
   else if (sig == SIGINT) {
      if (slfp) {
#ifndef SYSV
         fprintf(slfp, "\b\b  \n*** Socket closed.\n");
#else
         fprintf(slfp, "\n*** Socket closed.\n");
#endif
         if (slfp != stderr) fclose(slfp);
      }
      free_memory();
      exit(0);
   }
#endif
}

void free_memory(void)
{
   if (*_CLIENTHOSTNAME_) {
      int cl;

      for (cl=0; _CLIENTHOSTNAME_[cl]; cl++) free(_CLIENTHOSTNAME_[cl]);
      *_CLIENTHOSTNAME_ = NULL;
   }
   if (*filter) {
      int j;
      for (j=0; *(filter+j); j++) free(*(filter+j));
      *filter = NULL;
   }
   if ((*statstr)->_STDIN_BUFFER_) free((*statstr)->_STDIN_BUFFER_);
   if (_STDOUT_BUFFER_) free(_STDOUT_BUFFER_);
}

void init_user_id(short strnum)
{
   /*
    * Check if server is running with su privileges
    * and if so set new gid and uid.
    */
   if (!getuid()) {
      if (setgid((gid_t)(*(statstr+strnum))->pw_gid) != 0) {
         if (slfp) fprintf(slfp, "** setregid: %s\n", _PFTP_ERROR_ARRAY_);
         shutdown((*(statstr+strnum))->ns, 2);
         exit(1);
      }
      if (setuid((uid_t)(*(statstr+strnum))->pw_uid) != 0) {
         if (slfp) fprintf(slfp, "** setreuid: %s\n", _PFTP_ERROR_ARRAY_);
         shutdown((*(statstr+strnum))->ns, 2);
         exit(1);
      }
   }
}

#ifdef HAVE_GLIB
/* This is to work around a bug in glibc */
extern int putenv(const char *);
#endif

void init_user_dir(short strnum)
{
   char *homedir=NULL;
   char *user=NULL;

   /*
    * Get name of resource file.
    */
   MEM_CHECK((homedir = (char *)calloc(SONAME, sizeof(char))));
   MEM_CHECK((user = (char *)calloc(SONAME, sizeof(char))));
   MEM_CHECK(((*(statstr+strnum))->home = (char *)calloc(SONAME, sizeof(char))));
   strcpy(homedir, "HOME=");
   strcat(homedir, (*(statstr+strnum))->pw_dir);
   strcpy(user, "USER=");
   strcat(user, (*(statstr+strnum))->pw_name);
   strcpy((*(statstr+strnum))->home, (*(statstr+strnum))->pw_dir);
   strcat((*(statstr+strnum))->home, PFTPRESOURCE);

   /*
    * Set home directory.
    */
   if (putenv(homedir) != 0) {
      if (slfp) fprintf(slfp, "** putenv: insufficient space in the environment\n");
      shutdown((*(statstr+strnum))->ns, 2);
      exit(1);
   }

   /*
    * Set environment variable USER.
    */
   if (putenv(user) != 0) {
      if (slfp) fprintf(slfp, "** putenv: insufficient space in the environment\n");
      shutdown((*(statstr+strnum))->ns, 2);
      exit(1);
   }

   /*
    * Change current working directory
    * to user's home directory.
    */
   if (chdir((*(statstr+strnum))->pw_dir) < 0) {
      if (slfp) fprintf(slfp, "** %s: %s\n", (*(statstr+strnum))->pw_dir, _PFTP_ERROR_ARRAY_);
      shutdown((*(statstr+strnum))->ns, 2);
      exit(1);
   }
}

#if defined HAVE_LIBCRYPT && !defined HP_UX && !defined FreeBSD && !defined OpenBSD && !defined unicos
#include <crypt.h>
#endif

void create_passwd(void)
{
   char *passwd=NULL, *p=NULL, *t=NULL;
   char salt[8], *s=NULL;
   int i=0;
   
   fprintf(stderr, "Please give me some salt! (tow characters out of [a-zA-Z0-9./]): ");
   s = salt;
   set_tty(1);
   for (i=0; i < 2; i++, s++) {
      if (!((*s = fgetc(stdin)) >= 'a' && *s <= 'z') \
       && !(*s >= 'A' && *s <= 'Z') \
       && !(*s >= '0' && *s <= '9') \
       && *s != '.' && *s != '/') {
         fprintf(stderr, "\nHmmm...read the last line again.\n");
         set_tty(0);
         exit(1);
      }
   }
   *s = '\0';
   set_tty(0);
   set_tty(2);
   MEM_CHECK((passwd = (char *)calloc(SONAME, sizeof(char))));
   MEM_CHECK((p = (char *)calloc(SONAME, sizeof(char))));
   fprintf(stderr, "\nPassword: ");
   for (t=fgets(p, SONAME, stdin); *t && *t != '\n'; t++);
   *t = '\0';
   if (*p) strcpy(passwd, p);
   else {
      fprintf(stderr, "\n** No password entered.\n");
      set_tty(0);
      exit(1);
   }
   fprintf(stderr, "\nRetype the password: ");
   for (t=fgets(p, SONAME, stdin); *t && *t != '\n'; t++);
   *t = '\0';
   if (*p) {
      if (!strcmp(passwd, p)) {
         free(p);
         for (p=passwd, i=0; *p && *p != '\n' && i < 8; i++, p++);
         *p = '\0';
         fprintf(stderr, "\n");
         fprintf(stdout, "%s\n", (char *)crypt(passwd, salt));
         p = NULL;
      }
      else {
         fprintf(stderr, "\n** Mistyped. Try again.\n");
         set_tty(0);
         exit(1);
      }
   }
   set_tty(0);
   if (p) free(p);
   if (passwd) free(passwd);
   exit(0);
}
