
/*
 * term.c -- termios and termcap handlers
 *
 * Copyright 1990 Michael Sandrof
 * Copyright 1995 Matthew Green
 * Coypright 1996 EPIC Software Labs
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */

#include "irc.h"
#include "struct.h"

#include "ircaux.h"

#ifdef __EMX__
static BYTE pair[2];
char default_pair[2];
VIOMODEINFO vmode;
int	vio_screen = 0;
#include "pmbitchx.h"
#endif
#ifdef __EMXPM__
int     avio_redraw=0;
int     avio_resized=0;
int     VIO_staticx=81;
int     VIO_staticy=50;
int     VIO_font_width=6;
int     VIO_font_height=10;
int     titleh, borderw, borderh;
static HAB hab = 0;			/* handle to the anchor block			 */
static HMQ hmq = 0;			/* handle to the message queue			 */
static HWND hwndClient = 0;	        /* handle to the client 				 */
static HWND hwndFrame = 0; 	        /* handle to the frame window			 */
static QMSG qmsg = { 0 };	        /* message-queue structure				 */
static HDC hdc = 0;			/* handle to the device context 		 */
static HVPS hvps = 0;	                /* handle to the AVIO presentation space */
static HEV sem = 0;
ULONG  scrbuf, scrlen; /* rosmo */
VIOMODEINFO viomode;
FILE *debug;
static unsigned char output_buf[16];
int    output_tail = 0;
LONG   menuY;
clock_t flush_counter;
HMTX   mutex = 0;
HMTX   screenMutex = 0;
ULONG  cx, cy;
int    just_resized = 0;
#endif

#ifdef WINNT
HANDLE ghstdout;
CONSOLE_SCREEN_BUFFER_INFO gscrbuf;
CONSOLE_CURSOR_INFO gcursbuf;
DWORD gdwPlatform;
#endif

#include "vars.h"
#include "struct.h"
#include "ircterm.h"
#include "window.h"
#include "screen.h"
#include "output.h"
#ifdef TRANSLATE
#include "translat.h"
#endif


#if defined(_ALL_SOURCE) || defined(__EMX__)
#include <termios.h>
#else
#include <sys/termios.h>
#endif

#include <sys/ioctl.h>

static	int		tty_des;		/* descriptor for the tty */
static	struct	termios	oldb, newb;
	char		my_PC, *BC, *UP;
	int		BClen, UPlen;
#ifdef WTERM_C
int use_socks = 0;
char *get_string_var(enum VAR_TYPES var) { return NULL; }
char username[40];
void put_it(const char *str, ...) { return; }
#endif

                
#ifndef WTERM_C

/* Systems cant seem to agree where to put these... */
#ifdef HAVE_TERMINFO
extern	int		setupterm();
extern	char		*tigetstr();
extern	int		tigetnum();
extern	int		tigetflag();
#define Tgetstr(x, y) 	tigetstr(x.iname)
#define Tgetnum(x) 	tigetnum(x.iname);
#define Tgetflag(x) 	tigetflag(x.iname);
#else
extern	int		tgetent();
extern	char		*tgetstr();
extern	int		tgetnum();
extern	int		tgetflag();
#define Tgetstr(x, y) 	tgetstr(x.tname, &y)
#define Tgetnum(x) 	tgetnum(x.tname)
#define Tgetflag(x) 	tgetflag(x.tname)
#endif

extern  char    *getenv();
extern	char	*tparm();

/*
 * The old code assumed termcap. termcap is almost always present, but on
 * many systems that have both termcap and terminfo, termcap is deprecated
 * and its databases often out of date. Configure will try to use terminfo
 * if at all possible. We define here a mapping between the termcap / terminfo
 * names, and where we store the information.
 * NOTE: The terminfo portions are NOT implemented yet.
 */
#define CAP_TYPE_BOOL   0
#define CAP_TYPE_INT    1
#define CAP_TYPE_STR    2

typedef struct cap2info
{
	const char *	iname;
	const char *	tname;
	int 		type;
	char **		ptr;
} cap2info;

/*      Our variable name          Cap / Info   Description             */
int     TI_xmc,                 /* sg  / xmc    Space left by smso/rmso */
        TI_km,                  /* km  / km     Keyboard has meta mode  */
        TI_am,                  /* am  / am     Has auto margins        */
        TI_colors,              /* Co  / colors Number of colors        */
        TI_pairs,               /* pa  / pairs  Number of color pairs   */
        TI_cols,                /* co  / cols   Number of columns       */
        TI_lines                /* li  / lines  Number of lines         */
        ;
char    *TI_cup,                /* cm  / cup    Set cursor position     */
        *TI_mrcup,              /* CM  / mrcup  Relative cursor position*/
        *TI_hpa,                /* ch  / hpa    Set horizontal position */
        *TI_vpa,                /* cv  / vpa    Set vertical position   */
        *TI_ed,                 /* cd  / ed     Erase to end of display */
        *TI_clear,              /* cl  / clear  Clear the screen        */
        *TI_cr,                 /* cr  / cr     Carriage return         */
        *TI_nl,                 /* nl  / nl     New line (linefeed)     */
        *TI_el,                 /* ce  / el     Clear to end of line    */
        *TI_cuf1,               /* nd  / cuf1   Forward one space       */
        *TI_cub1,               /* le  / cub1   Backward one space      */
        *TI_cuf,                /* RI  / cuf    Forwards X spaces       */
        *TI_cub,                /* LE  / cub    Backwards X spaces      */
        *TI_bs,                 /* bs  / bs     Backspace               */
        *TI_ind,                /* sf  / ind    Scroll forwards         */
        *TI_ri,                 /* sr  / ri     Scroll backwards        */
        *TI_indn,               /* SF  / indn   Scroll forwards X lines */
        *TI_rin,                /* SR  / rin    Scroll back X lines     */
        *TI_csr,                /* cs  / csr    Change scrolling region */
        *TI_wind,               /* wi  / wind   Set scrolling window    */
        *TI_il,                 /* AL  / il     Insert lines            */
        *TI_il1,                /* al  / il1    Insert 1 line           */
        *TI_dl,                 /* DL  / dl     Delete lines            */
        *TI_dl1,                /* dl  / dl     Delete 1 line           */
        *TI_ich,                /* IC  / ich    Insert X characters     */
        *TI_ich1,               /* ic  / ich1   Insert 1 character      */
        *TI_smir,               /* im  / smir   Enter insert mode       */
        *TI_rmir,               /* ei  / rmir   Exit insert mode        */
        *TI_smdc,               /* dm  / smdc   Enter delete mode       */
        *TI_rmdc,               /* ed  / rmdc   Exit delete mode        */
        *TI_dch,                /* DC  / dch    Delete X characters     */
        *TI_dch1,               /* dc  / dch1   Delete 1 character      */
        *TI_smso,               /* so  / smso   Start standout (rev) mode */
        *TI_rmso,               /* se  / rmso   End standout (rev) mode */
        *TI_smul,               /* us  / smul   Start underline mode    */
        *TI_rmul,               /* ue  / rmul   End underline mode      */
        *TI_bel,                /* bl  / bel    Sound the bell          */
        *TI_blink,              /* mb  / blink  Start blink mode        */
        *TI_bold,               /* md  / bold   Start bold mode         */
        *TI_sgr0,               /* me  / sgr0   No bold / blink         */
        *TI_op,                 /* op  / op     Original color pair     */
        *TI_rep,                /* rp  / rep    Repeat char X Y times   */
        *TI_setf,               /* Sf  / setf   Set foreground color    */
        *TI_setb,               /* Sb  / setb   Set background color    */
        *TI_setaf,              /* AF  / setaf  Set ANSI foreground     */
        *TI_setab,              /* AB  / setab  Set ANSI background     */
        *TI_dispc,              /* S1  / dispc  Display PC ROM char     */
        *TI_smcup,              /* ti  / smcup  Start using CUP mode    */
        *TI_rmcup,              /* te  / rmcup  Stop using CUP mode     */
        *TI_smacs,              /* as  / smacs  Start ACS mode          */
        *TI_rmacs,              /* ae  / rmacs  End ACS mode            */
        *TI_pad,                /* pc  / pad    Pad character           */
        *TI_cuu1,               /* up  / cuu1   Move up one line        */
        *TI_smam,               /* SA  / smam   Turn on auto margins    */
        *TI_rmam                /* RA  / rmam Turn off auto margins   */
        ;

cap2info tcaps[] =
{
	{ "xmc",      "sg",   CAP_TYPE_INT,   (char **)&TI_xmc },
	{ "km",       "km",   CAP_TYPE_BOOL,  (char **)&TI_km },
	{ "am",       "am",   CAP_TYPE_BOOL,  (char **)&TI_am },
	{ "colors",   "Co",   CAP_TYPE_INT,   (char **)&TI_colors },
	{ "pairs",    "pa",   CAP_TYPE_INT,   (char **)&TI_pairs },
	{ "cols",     "co",   CAP_TYPE_INT,   (char **)&TI_cols },
	{ "lines",    "li",   CAP_TYPE_INT,   (char **)&TI_lines },
	{ "cup",      "cm",   CAP_TYPE_STR,   &TI_cup },
	{ "mrcup",    "CM",   CAP_TYPE_STR,   &TI_mrcup },
	{ "hpa",      "ch",   CAP_TYPE_STR,   &TI_hpa },
	{ "vpa",      "cv",   CAP_TYPE_STR,   &TI_vpa },
	{ "ed",       "cd",   CAP_TYPE_STR,   &TI_ed },
	{ "clear",    "cl",   CAP_TYPE_STR,   &TI_clear },
	{ "cr",       "cr",   CAP_TYPE_STR,   &TI_cr },
	{ "nl",       "nl",   CAP_TYPE_STR,   &TI_nl },
	{ "el",       "ce",   CAP_TYPE_STR,   &TI_el },
	{ "cuf1",     "nd",   CAP_TYPE_STR,   &TI_cuf1 },
	{ "cub1",     "le",   CAP_TYPE_STR,   &TI_cub1 },
	{ "cuf",      "RI",   CAP_TYPE_STR,   &TI_cuf },
	{ "cub",      "LE",   CAP_TYPE_STR,   &TI_cub },
	{ "bs",       "bs",   CAP_TYPE_STR,   &TI_bs },
	{ "ind",      "sf",   CAP_TYPE_STR,   &TI_ind },
	{ "ri",       "sr",   CAP_TYPE_STR,   &TI_ri },
	{ "indn",     "SF",   CAP_TYPE_STR,   &TI_indn },
	{ "rin",      "SR",   CAP_TYPE_STR,   &TI_rin },
	{ "csr",      "cs",   CAP_TYPE_STR,   &TI_csr },
	{ "wind",     "wi",   CAP_TYPE_STR,   &TI_wind },
	{ "il",       "AL",   CAP_TYPE_STR,   &TI_il },
	{ "il1",      "al",   CAP_TYPE_STR,   &TI_il1 },
	{ "dl",       "DL",   CAP_TYPE_STR,   &TI_dl },
	{ "dl1",      "dl",   CAP_TYPE_STR,   &TI_dl1 },
	{ "ich",      "IC",   CAP_TYPE_STR,   &TI_ich },
	{ "ich1",     "ic",   CAP_TYPE_STR,   &TI_ich1 },
	{ "smir",     "im",   CAP_TYPE_STR,   &TI_smir },
	{ "rmir",     "ei",   CAP_TYPE_STR,   &TI_rmir },
	{ "smdc",     "dm",   CAP_TYPE_STR,   &TI_smdc },
	{ "rmdc",     "ed",   CAP_TYPE_STR,   &TI_rmdc },
	{ "dch",      "DC",   CAP_TYPE_STR,   &TI_dch },
	{ "dch1",     "dc",   CAP_TYPE_STR,   &TI_dch1 },
	{ "smso",     "so",   CAP_TYPE_STR,   &TI_smso },
	{ "rmso",     "se",   CAP_TYPE_STR,   &TI_rmso },
	{ "smul",     "us",   CAP_TYPE_STR,   &TI_smul },
	{ "rmul",     "ue",   CAP_TYPE_STR,   &TI_rmul },
	{ "bel",      "bl",   CAP_TYPE_STR,   &TI_bel },
	{ "blink",    "mb",   CAP_TYPE_STR,   &TI_blink },
	{ "bold",     "md",   CAP_TYPE_STR,   &TI_bold },
	{ "sgr0",     "me",   CAP_TYPE_STR,   &TI_sgr0 },
	{ "op",       "op",   CAP_TYPE_STR,   &TI_op },
	{ "rep",      "rp",   CAP_TYPE_STR,   &TI_rep },
	{ "setf",     "Sf",   CAP_TYPE_STR,   &TI_setf },
	{ "setb",     "Sb",   CAP_TYPE_STR,   &TI_setb },
	{ "setaf",    "AF",   CAP_TYPE_STR,   &TI_setaf },
	{ "setab",    "AB",   CAP_TYPE_STR,   &TI_setab },
	{ "dispc",    "S1",   CAP_TYPE_STR,   &TI_dispc },
	{ "smcup",    "ti",   CAP_TYPE_STR,   &TI_smcup },
	{ "rmcup",    "te",   CAP_TYPE_STR,   &TI_rmcup },
	{ "smacs",    "as",   CAP_TYPE_STR,   &TI_smacs },
	{ "rmacs",    "ae",   CAP_TYPE_STR,   &TI_rmacs },
	{ "pad",      "pc",   CAP_TYPE_STR,   &TI_pad },
	{ "cuu1",     "up",   CAP_TYPE_STR,   &TI_cuu1 },
	{ "smam",     "SA",   CAP_TYPE_STR,   &TI_smam },
	{ "rmam",     "RA",   CAP_TYPE_STR,   &TI_rmam }
};
	int 	numcaps = sizeof(tcaps) / sizeof(cap2info);
	char 	TI_normal[256];
	char *	TI_sgrstrs[TERM_SGR_MAXVAL];
	char *	TI_forecolors[16];
	char *	TI_backcolors[16];

	int	meta_mode = 2;
	int	can_color = 0;
	int	need_redraw = 0;
static	int	term_echo_flag = 1;
static	int	li;
static	int	co;
#ifndef HAVE_TERMINFO
static	char	termcap[2048];	/* bigger than we need, just in case */
#endif
static	char	termcap2[2048];	/* bigger than we need, just in case */

static	char	*tptr = termcap2;

/*
 * term_echo: if 0, echo is turned off (all characters appear as blanks), if
 * non-zero, all is normal.  The function returns the old value of the
 * term_echo_flag 
 */
int	term_echo (int flag)
{
	int	echo;

	echo = term_echo_flag;
	term_echo_flag = flag;
	return (echo);
}

/*
 * term_putchar: puts a character to the screen, and displays control
 * characters as inverse video uppercase letters.  NOTE:  Dont use this to
 * display termcap control sequences!  It won't work! 
 *
 * Um... well, it will work if DISPLAY_ANSI_VAR is set to on... (hop)
 */
void	term_putchar (unsigned char c)
{
	if (term_echo_flag)
	{
		/* Sheer, raving paranoia */
#ifndef __EMX__
		if (!(newb.c_cflag & CS8) && (c & 0x80))
			c &= ~0x80;
#endif
		/* 
		 * This mangles all the unprintable control characters
		 * except for the escape character.  This will only support
		 * escape sequences that use ESC + any normal, printable
		 * characters (this is an assumption throughout the program).
		 * While you can use a non-vt100 emulation, the output will
		 * probably be mangled in several places.
		 */
		if (c < 0x20 && c == 0x9b) /* all ctl chars except ESC */
		{
			term_standout_on();
			putchar_x((c | 0x40) & 0x7f); /* Convert to printable */
			term_standout_off();
		}
		else if (c == 0x7f) 	/* delete char */
		{
			term_standout_on();
			putchar_x('?');
			term_standout_off();
		}
		else
			putchar_x(c);
	}
	else
		putchar_x(' ');
}

/*
 * term_reset: sets terminal attributed back to what they were before the
 * program started 
 */
void term_reset (void)
{
	tcsetattr(tty_des, TCSADRAIN, &oldb);

	if (TI_csr)
		tputs_x((char *)tparm(TI_csr, 0, TI_lines - 1));
	term_gotoxy(0, TI_lines - 1);
#if use_alt_screen
	if (TI_rmcup)
		tputs_x(TI_rmcup);
#endif
#if use_automargins
	if (TI_am && TI_smam)
		tputs_x(TI_smam);
#endif
	term_flush();
}

/*
 * term_cont: sets the terminal back to IRCII stuff when it is restarted
 * after a SIGSTOP.  Somewhere, this must be used in a signal() call 
 */
RETSIGTYPE term_cont (int unused)
{
extern int foreground;
	use_input = foreground = (tcgetpgrp(0) == getpgrp());
	if (foreground)
	{
#if use_alt_screen
		if (TI_smcup)
			tputs_x(TI_smcup);
#endif
#if use_automargins
		if (TI_rmam)
			tputs_x(TI_rmam);
#endif
		need_redraw = 1;
		tcsetattr(tty_des, TCSADRAIN, &newb);
	}
}

/*
 * term_pause: sets terminal back to pre-program days, then SIGSTOPs itself. 
 */
extern void term_pause (char unused, char *not_used)
{
#ifdef WINNT
char *shell;
DWORD dwmode;
HANDLE hstdin=GetStdHandle(STD_INPUT_HANDLE);
STARTUPINFO si = { 0 };
PROCESS_INFORMATION pi = { 0 };

	si.cb = sizeof(si);

	if (!(shell = get_string_var(SHELL_VAR)))
	{
		if (gdwPlatform == VER_PLATFORM_WIN32_WINDOWS)
			shell = "command.com";
		else
			shell = "cmd.exe";
	}
	if (!(GetConsoleMode(hstdin,&dwmode)))
		return;

	if (!SetConsoleMode(hstdin, dwmode | ((ENABLE_LINE_INPUT|ENABLE_ECHO_INPUT|ENABLE_PROCESSED_INPUT) & ~ENABLE_WINDOW_INPUT)))
		return;

	if(!CreateProcess(NULL, shell, NULL, NULL, TRUE, 0, NULL, NULL, &si, &pi) )
		return;
	CloseHandle(pi.hThread);
	WaitForSingleObject(pi.hProcess,INFINITE);
	CloseHandle(pi.hProcess);

	if (!SetConsoleMode(hstdin, dwmode ) )
		return;
	refresh_screen(0,NULL);
#else

	term_reset();
#ifndef __EMX__
	kill(getpid(), SIGSTOP);
#endif
#endif
}
#endif /* NOT IN WTERM_C */

#ifdef __EMXPM__
void avio_set_var(int aviovar, int value) {
   if (aviovar==AVIOREDRAW) 
        avio_redraw=value;
   if (aviovar==AVIORESIZED) 
         avio_resized=value;
}

int avio_get_var(int aviovar) {
   if (aviovar==AVIOREDRAW) 
      return avio_redraw;
   if (aviovar==AVIORESIZED) 
      return avio_resized;
}

void avio_refresh_screen(void) {
   /*WinQueryTaskSizePos(hab, 0, &winpos);
   winpos.cx=co*VIO_font_width;
   winpos.cy=li*VIO_font_height;
   WinSetWindowPos(hwndFrame, HWND_TOP, winpos.x, winpos.y, winpos.cx, winpos.cy, SWP_ACTIVATE | SWP_SIZE | SWP_SHOW | SWP_MOVE);*/
   if (avio_redraw==1) 
      refresh_screen(0, NULL);
   avio_set_var(AVIORESIZED,0);
}

void avio_settitle(char *titletext) {
  WinSetWindowText(hwndFrame, titletext);
}

int aflush() {
 HMTX h = mutex;

  /* flush */
  if (output_tail > 0)
   {
    DosOpenMutexSem(NULL, &h);
    DosRequestMutexSem(h, SEM_INDEFINITE_WAIT);

    VioWrtTTY((PCH)output_buf, output_tail, hvps);
	output_tail = 0;
    DosReleaseMutexSem(h);
    DosCloseMutexSem(h);
   }
}

void flush_thread()
{
 clock_t foo;
 while (1 & 1)
  {
   foo = clock();
   if ((foo - flush_counter) > 10)
    { if (output_tail > 0) { aflush(); } }
   DosSleep(100L);
  }
}

int aputc(int c) {
   static char tmpos2[2];
   HMTX h;

   DosOpenMutexSem(NULL, &h);
   DosRequestMutexSem(h, SEM_INDEFINITE_WAIT);

   /* buffer */
   if (output_tail < 16)
    { output_buf[output_tail] = (unsigned char) c; output_tail++; flush_counter = clock(); }

   /* flush */
   if (output_tail == 16 || c == '\n' || c == '\r')
	{
     VioWrtTTY((PCH)output_buf, output_tail, hvps);
	 output_tail = 0;
	}

   DosReleaseMutexSem(h);
   DosCloseMutexSem(h);
}

int areadkbd(int windowhandle,char *buffer, int maxbufsize) {
   int i;   /* Ignore the window handle for now because there is only one window. */
   if (strlen(aviokbdbuffer)> maxbufsize) {
      for(i=0;i<=strlen(aviokbdbuffer);i++)
         {
            if (i<maxbufsize) {
               buffer[i]=aviokbdbuffer[i];
            }
            else if(i==maxbufsize) {
               buffer[i]=0;
               aviokbdbuffer[0]=aviokbdbuffer[i];
               }
            else {
               aviokbdbuffer[i-maxbufsize]=aviokbdbuffer[i];
            }
      }
   } else {
         strcpy(buffer,aviokbdbuffer);
         aviokbdbuffer[0]='\0';
   }
   return strlen(buffer);
} 

int aprintf(char *format, ...) {
  va_list args;
  char	  putbuf[AVIO_BUFFER+1], bluf[AVIO_BUFFER];
  USHORT  x, y, i, o;

  va_start(args, format);
  vsprintf(putbuf, format, args);
  va_end(args);

  i = o = 0;
  while (putbuf[i] != '\n' && putbuf[i] != 0) {
	i++;
    if (putbuf[i] == '\n' || putbuf[i] == 0) {
         strcpy(bluf, "");
         strncat(bluf, &putbuf[o], i - o);
      if (putbuf[i] != 0) strcat(bluf, "\r\n");
	  VioWrtTTY((PCH)bluf,
				strlen(bluf),
				hvps);
      i++; o = i;
	}
  }

}

MRESULT EXPENTRY FontDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
 ULONG               Remfonts = 0, Metlen = 0, Fonts = 0, i;
 static PFONTMETRICS fonts;
 char                buf[256];

 switch (msg) {
  case WM_DESTROY:
    free(fonts);
    break;
  case WM_INITDLG:
    VioQueryFonts(&Remfonts,
                  NULL,
                  0,
                  &Fonts,
                  "System VIO",
                  VQF_PUBLIC,
                  hvps);

    fonts = malloc(sizeof(FONTMETRICS) * Remfonts);

    Fonts = Remfonts;
    VioQueryFonts(&Remfonts,
                  fonts,
                  sizeof(FONTMETRICS),
                  &Fonts,
                  "System VIO",
                  VQF_PUBLIC,
                  hvps);

    for (i = 0; i < Fonts; i++)
     {
      sprintf(buf, "%s %ux%u", fonts[i].szFacename, fonts[i].lMaxBaselineExt, fonts[i].lAveCharWidth);
      WinSendMsg(WinWindowFromID(hwnd, 101),
                 LM_INSERTITEM,
                 MPFROMSHORT(LIT_END),
                 MPFROMP(buf));
     }
    break;
  case WM_COMMAND:
    switch (COMMANDMSG(&msg)->cmd)
     {
      case DID_CANCEL:
        WinDismissDlg(hwnd, 0);
        break;
      case DID_OK:
        i = (ULONG)WinSendMsg(WinWindowFromID(hwnd, 101),
                              LM_QUERYSELECTION,
                              MPFROMSHORT(LIT_CURSOR),
                              0);
        VIO_font_width  = fonts[i].lAveCharWidth;
        VIO_font_height = fonts[i].lMaxBaselineExt;

        VioSetDeviceCellSize(VIO_font_height, VIO_font_width, hvps);

        cx = (co - 1) * VIO_font_width;
        cy = li * VIO_font_height;
        WinSendMsg(hwndFrame, WM_MOVE, 0, MPFROM2SHORT(cx, cy));
        WinSetWindowPos(hwndFrame, HWND_TOP, 0, 0, cx+(borderw*2), cy+(borderh*2)+titleh+menuY, SWP_SIZE);
        avio_refresh_screen();

        WinDismissDlg(hwnd, 0);
        break;
      default: break;
    }
   return((MRESULT)0);
   break;
  default: return (WinDefDlgProc(hwnd, msg, mp1, mp2));
 }
}

void font_dialog(HWND hwnd)
{
 WinDlgBox(HWND_DESKTOP, HWND_DESKTOP, FontDlgProc, NULLHANDLE, FONTDIALOG, NULL);
}

MRESULT EXPENTRY GenericWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{

 HPS   hps;
 int tmp;
 SHORT x, y, i;
 SWP winpos;
 RECTL rcl;
 char *Reason;
 FILE *debug;

	switch (msg) {
         case WM_COMMAND:
            switch (COMMANDMSG(&msg)->cmd)
             {
              case IDM_M_FONT:
                font_dialog(hwnd);
                break;
              case IDM_M_QUIT:
                irc_exit(1, "NuKe & Rosmo owns me!", NULL);
                break;
              default: break;
             }
            return((MRESULT)0);
            break;
         case WM_CLOSE:
                 irc_exit(1, "PMBitchX rocks your world!", NULL);
                 break;
         case WM_PAINT:
            hps = WinBeginPaint(hwnd, NULLHANDLE, NULL);
            VioShowPS(VIO_staticy, VIO_staticx, 0, hvps);
            WinEndPaint(hps);
            return (0L);
         case WM_CHAR:
            if(!(SHORT1FROMMP(mp1) & KC_KEYUP)) break;
                 if(strlen(aviokbdbuffer)==255) {
                   DosBeep(1000, 200);
                 } else {
                if(SHORT1FROMMP(mp2) != 0) {
                   tmp=strlen(aviokbdbuffer);
                   if (SHORT1FROMMP(mp2) == 224) {
                      aviokbdbuffer[tmp]=0xFF;
                      aviokbdbuffer[tmp+1]=SHORT1FROMMP(mp1);
                      aviokbdbuffer[tmp+2]=0;
                   } else {
                      if (!(SHORT1FROMMP(mp1) & KC_CTRL) && (SHORT1FROMMP(mp2) > 96) && (SHORT1FROMMP(mp2) < 122)) 
                         aviokbdbuffer[tmp]=SHORT1FROMMP(mp2)-96;
                      else
                         aviokbdbuffer[tmp]=SHORT1FROMMP(mp2);
                      aviokbdbuffer[tmp+1]=0;
                   }
                }
             }
             make_window_current(last_input_screen->current_window);
             do_screens(NULL);
             break;
    case WM_USER:
             just_resized = 1;
             WinSetWindowPos(hwndFrame, HWND_TOP, 0, 0, cx+(borderw*2), cy+(borderh*2)+titleh+menuY, SWP_SIZE);
             break;
	case WM_SIZE:
             if (just_resized > 0) { just_resized = 0; return((MRESULT)0); }
             x=SHORT1FROMMP(mp2); y=SHORT2FROMMP(mp2);
             if(x != 0 && y != 0) {
             co=((int)(x/VIO_font_width));
             li=((int)(y/VIO_font_height));
             if(co < 20) co=20;
             if(li < 10) li=10;
             cx = (co-1) * VIO_font_width;
             cy = li * VIO_font_height;
             VioAssociate((HDC)NULL, hvps);      /* disassociates the AVIO PS */
             VioDestroyPS(hvps);                 /* destroys the AVIO PS      */

             VIO_staticy = li+1;
             VIO_staticx = co+1;

             hdc = WinOpenWindowDC(hwndClient);                                    /* opens device context */
             VioCreatePS(&hvps, VIO_staticy, VIO_staticx, 1,                       /* creates AVIO PS */
                         1, 0);
             VioAssociate(hdc, hvps);                                      /* associates DC and AVIO PS */

             VioSetDeviceCellSize(VIO_font_height, VIO_font_width, hvps);              /* Set the font size */

             avio_refresh_screen();

             WinPostMsg(hwnd, WM_USER, 0, 0);
            }
			return (WinDefAVioWindowProc(hwnd, msg, (ULONG)mp1, (ULONG)mp2));
	}
	return (WinDefWindowProc(hwnd, msg, mp1, mp2));
}

void avio_exit(void) {
	VioAssociate((HDC)NULL, hvps);		/* disassociates the AVIO PS */
	VioDestroyPS(hvps); 				/* destroys the AVIO PS 	 */
	WinDestroyWindow(hwndFrame);
	WinDestroyMsgQueue(hmq);
	WinTerminate(hab);
        DosCloseMutexSem(mutex);
	DosExit(EXIT_PROCESS, 0);
}

void avio_init(void) {

        HEV   mySem = sem;
        SWP winpos;

        ULONG flStyle = FCF_MINMAX | FCF_SYSMENU | FCF_TITLEBAR |
                   FCF_SIZEBORDER | FCF_SHELLPOSITION | FCF_TASKLIST | FCF_MENU;

        DosOpenEventSem(NULL, (PHEV)&mySem);

	hab = WinInitialize(0);
	hmq = WinCreateMsgQueue(hab, 0);

        if (!WinRegisterClass(hab, "AVIO",
			GenericWndProc, CS_SIZEREDRAW, 0))
		DosExit(EXIT_PROCESS, 1);

	hwndFrame = WinCreateStdWindow(HWND_DESKTOP,
								   WS_VISIBLE,
								   &flStyle,
                                   "AVIO",
                                   irc_version,
								   0L,
								   NULLHANDLE,
                                   IDM_MAINMENU,
								   &hwndClient);

        co = 81; li = 50;
        cx = (co-1) * VIO_font_width;
        cy = li * VIO_font_height;
        menuY = WinQuerySysValue(HWND_DESKTOP, SV_CYMENU); /* rosmo */
        borderw=WinQuerySysValue(HWND_DESKTOP, SV_CXSIZEBORDER);
        borderh=WinQuerySysValue(HWND_DESKTOP, SV_CYSIZEBORDER);
        titleh=WinQuerySysValue(HWND_DESKTOP, SV_CYTITLEBAR);
        WinSetWindowPos(hwndFrame, HWND_TOP, 0, 0, cx+(borderw*2), cy+(borderh*2)+titleh + menuY, SWP_SIZE);

	hdc = WinOpenWindowDC(hwndClient);	                                  /* opens device context */
	VioCreatePS(&hvps, VIO_staticy, VIO_staticx, 1,		                  /* creates AVIO PS */
				1, 0);
	VioAssociate(hdc, hvps);			                          /* associates DC and AVIO PS */

        VioSetDeviceCellSize(VIO_font_height, VIO_font_width, hvps);              /* Set the font size */
        VioGetBuf(&scrbuf, &scrlen, hvps);

        DosPostEventSem(mySem);
        DosCloseEventSem(mySem);

        /*WinQueryTaskSizePos(hab, 0, &winpos);
        winpos.cx=80*VIO_font_width;
        winpos.cy=50*VIO_font_height;
        WinSetWindowPos(hwndFrame, HWND_TOP, winpos.x, winpos.y, winpos.cx, winpos.cy, SWP_ACTIVATE | SWP_SIZE | SWP_SHOW | SWP_MOVE);*/

/* rosmo */
        DosCreateMutexSem(NULL, &mutex, 0, FALSE);

/* rosmo loves mutexes nowadays */
        DosCreateMutexSem("\\SEM32\\BITCHX_OWNS_ME", &screenMutex, DC_SEM_SHARED, FALSE);

        while (WinGetMsg(hab, &qmsg, (HWND)NULL, 0, 0))
             WinDispatchMsg(hab, &qmsg);

}
#endif



/*
 * term_init: does all terminal initialization... reads termcap info, sets
 * the terminal to CBREAK, no ECHO mode.   Chooses the best of the terminal
 * attributes to use ..  for the version of this function that is called for
 * wserv, we set the termial to RAW, no ECHO, so that all the signals are
 * ignored.. fixes quite a few problems...  -phone, jan 1993..
 */
int termfeatures = 0;

int term_init (void)
{
#ifndef WTERM_C
	int	i;
	int	desired;
	char	*term;
#if !defined(__EMX__) && !defined(WINNT)

	if (dumb_mode)
		ircpanic("term_init called in dumb_mode");

	if ((term = getenv("TERM")) == (char *) 0)
	{
		fprintf(stderr, "\n");
		fprintf(stderr, "You do not have a TERM environment variable.\n");
		fprintf(stderr, "So we'll be running in dumb mode...\n");
		return -1;
	}
	else
		fprintf(stdout, "Using terminal type [%s]\n", term);

#ifdef HAVE_TERMINFO
	setupterm(NULL, 1, &i);
	if (i != 1)
	{
		fprintf(stderr, "setupterm failed: %d\n", i);
		fprintf(stderr, "So we'll be running in dumb mode...\n");
		return -1;
	}
#else
	if (tgetent(termcap, term) < 1)
	{
		fprintf(stderr, "\n");
		fprintf(stderr, "Your current TERM setting (%s) does not have a termcap entry.\n", term);
		fprintf(stderr, "So we'll be running in dumb mode...\n");
		return -1;
	}
#endif

	for (i = 0; i < numcaps; i++)
	{
		int *iptr, ival;

		if (tcaps[i].type == CAP_TYPE_INT)
		{
			iptr = (int *)tcaps[i].ptr;
			ival = Tgetnum(tcaps[i]);
			*iptr = ival;
		}
		else if (tcaps[i].type == CAP_TYPE_BOOL)
		{
			iptr = (int *)tcaps[i].ptr;
			ival = Tgetflag(tcaps[i]);
			*iptr = ival;
		}
		else
		{
			if ((*tcaps[i].ptr = Tgetstr(tcaps[i], tptr)) == (char *) -1)
				*tcaps[i].ptr = NULL;
		}
	}

	BC = TI_cub1;
	UP = TI_cuu1;
	if (TI_pad)
		my_PC = TI_pad[0];
	else
		my_PC = 0;

	if (BC)
		BClen = strlen(BC);
	else
		BClen = 0;
	if (UP)
		UPlen = strlen(UP);
	else
		UPlen = 0;

	li = TI_lines;
	co = TI_cols;
	if (!co)
		co = 79;
	if (!li)
		li = 24;

	if (!TI_nl)
		TI_nl = "\n";
	if (!TI_cr)
		TI_cr = "\r";

	TI_normal[0] = 0;
	if (TI_sgr0)
	{
		strcat(TI_normal, TI_sgr0);
	}
	if (TI_rmso)
	{
		if (TI_sgr0 && strcmp(TI_rmso, TI_sgr0))
			strcat(TI_normal, TI_rmso);
	}
	if (TI_rmul)
	{
		if (TI_sgr0 && strcmp(TI_rmul, TI_sgr0))
			strcat (TI_normal, TI_rmul);
	}


#else

#ifdef WINNT
	CONSOLE_SCREEN_BUFFER_INFO scrbuf;
	HANDLE hinput =GetStdHandle(STD_INPUT_HANDLE);
	DWORD dwmode;
	OSVERSIONINFO osver;

	ghstdout = GetStdHandle(STD_OUTPUT_HANDLE);

	if(!GetConsoleScreenBufferInfo(ghstdout, &gscrbuf) ) 
	{
		fprintf(stderr,"Error from getconsolebufinfo %d\n",GetLastError());
		abort();
	}

	GetConsoleMode(hinput,&dwmode);
	SetConsoleMode(hinput,dwmode & (~(ENABLE_LINE_INPUT |ENABLE_ECHO_INPUT
				| ENABLE_PROCESSED_INPUT)| ENABLE_WINDOW_INPUT/* | ENABLE_WRAP_AT_EOL_OUTPUT*/)
				);

	osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	GetVersionEx(&osver);
	gdwPlatform = osver.dwPlatformId;

	GetConsoleCursorInfo(hinput, &gcursbuf);

	GetConsoleScreenBufferInfo(ghstdout, &scrbuf);
	li = scrbuf.srWindow.Bottom - scrbuf.srWindow.Top + 1;
	co = scrbuf.srWindow.Right - scrbuf.srWindow.Left + 1;
	TI_lines = li;
	TI_cols = co - 1;
#elif defined(__EMXPM__)
	/* Create an unnamed semaphore */
	DosCreateEventSem(NULL,
					  (PHEV)&sem,
					  DC_SEM_SHARED,
					  FALSE);

	TI_lines = 50;
	TI_cols = 80;
	li = TI_lines;
	co = TI_cols;

	_beginthread(avio_init, NULL, (4*0xFFFF), NULL);
        _beginthread(flush_thread, NULL, 0xFFFF, NULL);

	DosWaitEventSem(sem, SEM_INDEFINITE_WAIT);
	DosCloseEventSem(sem);


	default_pair[0] = ' ';
	default_pair[1] = 7;
#else
	vmode.cb = sizeof(VIOMODEINFO);
	VioGetMode(&vmode, 0);
	default_pair[0] = ' ';
	default_pair[1] = 7;

	TI_lines = vmode.row;
	TI_cols = vmode.col+1;
	li = TI_lines;
	co = TI_cols;
#endif
	TI_cup = strdup("\e[%i%d;%dH");
	TI_cub1 = strdup("\e[D");
	TI_clear = strdup("\e[H\e[2J");
	TI_el = strdup("\e[K");
	TI_mrcup = strdup("\e[%i%d;%dH");
	TI_cub = strdup("\e[D");
	TI_cuf = strdup("\e[C");
	TI_cuf1 = strdup("\e[C");
	TI_smso = strdup("\e[7m");
	TI_rmso = strdup("\e[0m");
	TI_smul = strdup("\e[1;34m");
	TI_rmul = strdup("\e[0;37m");
	TI_bold = strdup("\e[1;37m");
	TI_sgr0 = strdup("\e[0;37m");
	TI_blink = strdup("\e[5m");
	TI_normal[0]=0;

	TI_csr = strdup("\e[%i%p1%d;%p2%dr");
	TI_ri = strdup("\eM");
	TI_ind = strdup("\n");
	TI_il = strdup("\e[%p1%dL");
	TI_il1 = strdup("\e[L");
	TI_dl = strdup("\e[%p1%dM");
	TI_dl1 = strdup("\e[M");
	
	strcat(TI_normal, TI_sgr0);
	strcat(TI_normal, TI_rmso);

	TI_cr = "\r";
	TI_nl = "\n";
	TI_bel = "\007";
	TI_cub = "\010";

#endif

	/*
	 * Finally set up the TI_sgrstrs array.  Clean it out first...
	 */
	for (i = 0; i < TERM_SGR_MAXVAL; i++)
		TI_sgrstrs[i] = "";


	if (TI_bold)
		TI_sgrstrs[TERM_SGR_BOLD_ON-1] = TI_bold;
	if (TI_sgr0)
	{
		TI_sgrstrs[TERM_SGR_BOLD_OFF - 1] = TI_sgr0;
		TI_sgrstrs[TERM_SGR_BLINK_OFF - 1] = TI_sgr0;
	}
	if (TI_blink)
		TI_sgrstrs[TERM_SGR_BLINK_ON - 1] = TI_blink;
	if (TI_smul)
		TI_sgrstrs[TERM_SGR_UNDL_ON - 1] = TI_smul;
	if (TI_rmul)
		TI_sgrstrs[TERM_SGR_UNDL_OFF - 1] = TI_rmul;
	if (TI_smso)
		TI_sgrstrs[TERM_SGR_REV_ON - 1] = TI_smso;
	if (TI_rmso)
		TI_sgrstrs[TERM_SGR_REV_OFF - 1] = TI_rmso;
	if (TI_normal[0])
	{
		TI_sgrstrs[TERM_SGR_NORMAL - 1] = TI_normal;
		TI_sgrstrs[TERM_SGR_RESET - 1] = TI_normal;
	}


	/*
	 * Now figure out whether or not this terminal is "capable enough"
	 * for the client. This is a rather complicated set of tests, as
	 * we have many ways of doing the same thing.
	 * To keep the set of tests easier, we set up a bitfield integer
	 * which will have the desired capabilities added to it. If after
	 * all the checks we dont have the desired mask, we dont have a
	 * capable enough terminal.
	 */
	desired = TERM_CAN_CUP | TERM_CAN_CLEAR | TERM_CAN_CLREOL |
		TERM_CAN_RIGHT | TERM_CAN_LEFT | TERM_CAN_SCROLL;

	termfeatures = 0;
	if (TI_cup)
		termfeatures |= TERM_CAN_CUP;
	if (TI_hpa && TI_vpa)
		termfeatures |= TERM_CAN_CUP;
	if (TI_clear)
		termfeatures |= TERM_CAN_CLEAR;
	if (TI_ed)
		termfeatures |= TERM_CAN_CLEAR;
	if (TI_dl || TI_dl1)
		termfeatures |= TERM_CAN_CLEAR;
	if (TI_il || TI_il1)
		termfeatures |= TERM_CAN_CLEAR;
	if (TI_el)
		termfeatures |= TERM_CAN_CLREOL;
	if (TI_cub || TI_mrcup || TI_cub1 || TI_bs)
		termfeatures |= TERM_CAN_LEFT;
	if (TI_cuf  || TI_cuf1 || TI_mrcup)
		termfeatures |= TERM_CAN_RIGHT;
	if (TI_dch || TI_dch1)
		termfeatures |= TERM_CAN_DELETE;
	if (TI_ich || TI_ich1)
		termfeatures |= TERM_CAN_INSERT;
	if (TI_rep)
		termfeatures |= TERM_CAN_REPEAT;
	if (TI_csr && (TI_ri || TI_rin) && (TI_ind || TI_indn))
		termfeatures |= TERM_CAN_SCROLL;
	if (TI_wind && (TI_ri || TI_rin) && (TI_ind || TI_indn))
		termfeatures |= TERM_CAN_SCROLL;
	if ((TI_il || TI_il1) && (TI_dl || TI_dl1))
		termfeatures |= TERM_CAN_SCROLL;
	if (TI_bold && TI_sgr0)
		termfeatures |= TERM_CAN_BOLD;
	if (TI_blink && TI_sgr0)
		termfeatures |= TERM_CAN_BLINK;
	if (TI_smul && TI_rmul)
		termfeatures |= TERM_CAN_UNDL;
	if (TI_smso && TI_rmso)
		termfeatures |= TERM_CAN_REVERSE;
	if (TI_dispc)
		termfeatures |= TERM_CAN_GCHAR;
	if ((TI_setf && TI_setb) || (TI_setaf && TI_setab))
		termfeatures |= TERM_CAN_COLOR;

	if ((termfeatures & desired) != desired)
	{
		fprintf(stderr, "\nYour terminal (%s) cannot run IRC II in full screen mode.\n", term);
		fprintf(stderr, "The following features are missing from your TERM setting.\n");
		if (!(termfeatures & TERM_CAN_CUP))
			fprintf (stderr, "\tCursor movement\n");
		if (!(termfeatures & TERM_CAN_CLEAR))
			fprintf(stderr, "\tClear screen\n");
		if (!(termfeatures & TERM_CAN_CLREOL))
			fprintf(stderr, "\tClear to end-of-line\n");
		if (!(termfeatures & TERM_CAN_RIGHT))
			fprintf(stderr, "\tCursor right\n");
		if (!(termfeatures & TERM_CAN_LEFT))
			fprintf(stderr, "\tCursor left\n");
		if (!(termfeatures & TERM_CAN_SCROLL))
			fprintf(stderr, "\tScrolling\n");

		fprintf(stderr, "So we'll be running in dumb mode...\n");
		return -1;
	}

	if (!TI_cub1)
	{
		if (TI_bs)
			TI_cub1 = TI_bs;
		else
			TI_cub1 = "\b";
	}
	if (!TI_bel)
		TI_bel = "\007";

	/*
	 * Default to no colors. (ick)
	 */
	for (i = 0; i < 16; i++)
		TI_forecolors[i] = TI_backcolors[i] = "";

	/*
	 * Set up colors.
	 * Absolute fallbacks are for ansi-type colors
	 */
	for (i = 0; i < 16; i++)
	{
		char cbuf[128];

		cbuf[0] = '\0';
		if (i >= 8)
			strcpy(cbuf, TI_sgrstrs[TERM_SGR_BOLD_ON-1]);
		if (TI_setf) 
			strcat(cbuf, (char *)tparm(TI_setf, i & 0x07, 0));
		else if (TI_setaf)
			strcat(cbuf, (char *)tparm(TI_setaf, i & 0x07, 0));
		else if (i >= 8)
			sprintf(cbuf, "\033[1;%dm", (i & 0x07) + 30);
		else
			sprintf(cbuf, "\033[%dm", (i & 0x07) + 30);

		TI_forecolors[i] = m_strdup(cbuf);
	}
            
	for (i = 0; i < 16; i++)
	{
		char cbuf[128];

		cbuf[0] = '\0';
		if (i >= 8)
			strcpy(cbuf, TI_sgrstrs[TERM_SGR_BLINK_ON - 1]);

		if (TI_setab)
			strcat (cbuf, (char *)tparm(TI_setab, i & 0x07, 0));
		else if (TI_setb)
			strcat (cbuf, (char *)tparm(TI_setb, i & 0x07, 0));
		else if (i >= 8)
			sprintf(cbuf, "\033[1;%dm", (i & 0x07) + 40);
		else
			sprintf(cbuf, "\033[%dm", (i & 0x07) + 40);

		TI_backcolors[i] = m_strdup(cbuf);
	}
#endif /* NOT IN WTERM_C */
	tty_des = 0;
	reinit_term(tty_des);
	return 0;
}

void reinit_term(int tty)
{
	/* Set up the terminal discipline */
	tcgetattr(tty, &oldb);

	newb = oldb;
	newb.c_lflag &= ~(ICANON | ECHO); /* set equ. of CBREAK, no ECHO */
	newb.c_cc[VMIN] = 1;	          /* read() satified after 1 char */
	newb.c_cc[VTIME] = 0;	         /* No timer */

#       if !defined(_POSIX_VDISABLE)
#               if defined(HAVE_FPATHCONF)
#                       define _POSIX_VDISABLE fpathconf(tty, _PC_VDISABLE)
#               else
#                       define _POSIX_VDISABLE 0
#               endif
#       endif

	newb.c_cc[VQUIT] = _POSIX_VDISABLE;
#	ifdef VDSUSP
		newb.c_cc[VDSUSP] = _POSIX_VDISABLE;
# 	endif
#	ifdef VSUSP
		newb.c_cc[VSUSP] = _POSIX_VDISABLE;
#	endif

#ifndef WTERM_C
	if (!use_flow_control)
		newb.c_iflag &= ~IXON;	/* No XON/XOFF */
	/* Ugh. =) This is annoying! */
#if use_alt_screen
	if (TI_smcup)
		tputs_x(TI_smcup);
#endif
#if use_automargins
	if (TI_rmam)
		tputs_x(TI_rmam);
#endif
#endif

	tcsetattr(tty, TCSADRAIN, &newb);
	tty_des = tty;
	return;
}


#ifndef WTERM_C

void tty_dup(int tty)
{
	close (tty_des);
	dup2(tty, tty_des);
}

/*
 * term_resize: gets the terminal height and width.  Trys to get the info
 * from the tty driver about size, if it can't... uses the termcap values. If
 * the terminal size has changed since last time term_resize() has been
 * called, 1 is returned.  If it is unchanged, 0 is returned. 
 */
int term_resize (void)
{
	static	int	old_li = -1,
			old_co = -1;

#if 1
#	if defined (TIOCGWINSZ)
	{
		struct winsize window;

		if (ioctl(tty_des, TIOCGWINSZ, &window) < 0)
		{
			TI_lines = li;
			TI_cols = co;
		}
		else
		{
			if ((TI_lines = window.ws_row) == 0)
				TI_lines = li;
			if ((TI_cols = window.ws_col) == 0)
				TI_cols = co;
		}
	}
#	else
	{
		TI_lines = li;
		TI_cols = co;
	}
#	endif

#if use_automargins
	if (!TI_am || !TI_rmam)
	{
		TI_cols--;
	}
#else
	TI_cols--;
#endif

	if ((old_li != TI_lines) || (old_co != TI_cols))
	{
		if (last_input_screen)
		{
			last_input_screen->li = TI_lines;
			last_input_screen->co = TI_cols;
		}
		old_li = TI_lines;
		old_co = TI_cols;
		return (1);
	}
#else
#	if defined (TIOCGWINSZ)
	{
		struct winsize window;

		if (ioctl(tty_des, TIOCGWINSZ, &window) < 0)
		{
			old_li = TI_lines;
			old_co = TI_cols - 1;
		}
		else
		{
			if ((old_li = window.ws_row) == 0)
				old_li = TI_lines;
			if ((old_co = (window.ws_col - 1)) == 0)
				old_co = TI_cols - 1;
		}
	}
#	else
	{
		old_li = TI_lines;
		old_co = TI_cols - 1;
	}
#	endif

	if (main_screen)
	{
		if ((main_screen->li != old_li) || (main_screen->co != old_co))
		{
			main_screen->li = old_li;
			main_screen->co = old_co;
			return (1);
		}
	}
	else
	{
		TI_lines = old_li;
		TI_cols = old_co;
	}
#endif
	return (0);
}


/* term_CE_clear_to_eol(): the clear to eol function, right? */
void term_clreol (void)
{
#ifdef __EMXPM__ 
USHORT row, col;
    aflush();
    VioGetCurPos(&row, &col, hpvs);
    pair[0] = ' '; pair[1] = 7;
    VioWrtNChar((PCH)&pair[0], TI_cols - col + 1, row, col, hpvs);
    VioSetCurPos(row, col, hpvs);
#else
	tputs_x(TI_el);
#endif
	return;
}

/* Set the cursor position */
void term_gotoxy (int col, int row)
{
#ifdef __EMXPM__
	aflush();
	VioSetCurPos(row, col, hpvs);
#else
	if (TI_cup)
		tputs_x((char *)tparm(TI_cup, row, col));
	else
	{
		tputs_x((char *)tparm(TI_hpa, col));
		tputs_x((char *)tparm(TI_vpa, row));
	}
#endif
}

/* A no-brainer. Clear the screen. */
void term_clrscr (void)
{
#ifdef __EMX__
#ifdef __EMXPM__
	aflush();
	VioScrollUp(0, 0, -1, -1, -1, &default_pair, hvps);
#else
	VioScrollUp(0, 0, -1, -1, -1, &default_pair, vio_screen);
#endif
#else

	int i;

	/* We have a specific cap for clearing the screen */
	if (TI_clear)
	{
		tputs_x(TI_clear);
		term_gotoxy (0, 0);
		return;
	}

	term_gotoxy (0, 0);
	/* We can clear by doing an erase-to-end-of-display */
	if (TI_ed)
	{
		tputs_x (TI_ed);
		return;
	}
	/* We can also clear by deleteing lines ... */
	else if (TI_dl)
	{
		tputs_x((char *)tparm(TI_dl, TI_lines));
		return;
	}
	/* ... in this case one line at a time */
	else if (TI_dl1)
	{
		for (i = 0; i < TI_lines; i++)
			tputs_x (TI_dl1);
		return;
	}
	/* As a last resort we can insert lines ... */
	else if (TI_il)
	{
		tputs_x ((char *)tparm(TI_il, TI_lines));
		term_gotoxy (0, 0);
		return;
	}
	/* ... one line at a time */
	else if (TI_il1)
	{
		for (i = 0; i < TI_lines; i++)
			tputs_x (TI_il1);
		term_gotoxy (0, 0);
	}
#endif
}

/*
 * Move the cursor NUM spaces to the left, non-destruvtively if we can.
 */
void term_left (int num)
{
#ifdef __EMXPM__
        int row,col;
        aflush();
        VioGetCurPos(&row, &col, hvps);
        if(col>num)
           col=col-num;
        else
           col=0;
	VioSetCurPos(row, col, hvps);
#else
	if (TI_cub)
		tputs_x ((char *)tparm(TI_cub, num));
	else if (TI_mrcup)
		tputs_x ((char *)tparm(TI_mrcup, -num, 0));
	else if (TI_cub1)
		while (num--)
			tputs_x(TI_cub1);
	else if (TI_bs)
		while (num--)
			tputs_x (TI_bs);
#endif
}

/*
 * Move the cursor NUM spaces to the right
 */
void term_right (int num)
{
#ifdef __EMXPM__
        int row,col;
        aflush();
        VioGetCurPos(&row, &col, hvps);
        if(TI_cols>num)
           col=col+num;
        else
           col=TI_cols;
	VioSetCurPos(row, col, hvps);
#else
	if (TI_cuf)
		tputs_x ((char *)tparm(TI_cuf, num));
	else if (TI_mrcup)
		tputs_x ((char *)tparm(TI_mrcup, num, 0));
	else if (TI_cuf1)
		while (num--)
			tputs_x(TI_cuf1);
#endif
}

/*
 * term_delete (int num)
 * Deletes NUM characters at the current position
 */
void term_delete (int num)
{
	if (TI_smdc)
		tputs_x(TI_smdc);

	if (TI_dch)
		tputs_x ((char *)tparm (TI_dch, num));
	else if (TI_dch1)
		while (num--)
			tputs_x (TI_dch1);

	if (TI_rmdc)
		tputs_x (TI_rmdc);
}

/*
 * Insert character C at the curent cursor position.
 */
void term_insert (unsigned char c)
{
	if (TI_smir)
		tputs_x (TI_smir);
	else if (TI_ich1)
		tputs_x (TI_ich1);
	else if (TI_ich)
		tputs_x ((char *)tparm(TI_ich, 1));

	term_putchar (c);

	if (TI_rmir)
		tputs_x(TI_rmir);
}


/*
 * Repeat the character C REP times in the most efficient way.
 */
void term_repeat (unsigned char c, int rep)
{
	if (TI_rep)
		tputs_x((char *)tparm (TI_rep, (int)c, rep));
	else
		while (rep--)
			putchar_x (c);
}



/*
 * Scroll the screen N lines between lines TOP and BOT.
 */
void term_scroll (int top, int bot, int n)
{
	int i,oneshot=0,rn,sr,er;
	char thing[128], final[128], start[128];

#ifdef __EMX__
	pair[0] = ' '; pair[1] = 7;
#ifdef __EMXPM__
	aflush();
	if (n > 0) VioScrollUp(top, 0, bot, TI_cols, n, (PBYTE)&pair, hvps);
	else if (n < 0) { n = -n; VioScrollDn(top, 0, bot, TI_cols, n, (PBYTE)&pair, hvps); }
#else
	if (n > 0) VioScrollUp(top, 0, bot, TI_cols, n, (PBYTE)&pair, (HVIO) vio_screen);
	else if (n < 0) { n = -n; VioScrollDn(top, 0, bot, TI_cols, n, (PBYTE)&pair, (HVIO) vio_screen); }
#endif
#else

	/* Some basic sanity checks */
	if (n == 0 || top == bot || bot < top)
		return;

	sr = er = 0;
	final[0] = start[0] = thing[0] = 0;

	if (n < 0)
		rn = -n;
	else
		rn = n;

	/*
	 * First thing we try to do is set the scrolling region on a 
	 * line granularity.  In order to do this, we need to be able
	 * to have termcap 'cs', and as well, we need to have 'sr' if
	 * we're scrolling down, and 'sf' if we're scrolling up.
	 */
	if (TI_csr && (TI_ri || TI_rin) && (TI_ind || TI_indn))
	{
		/*
		 * Previously there was a test to see if the entire scrolling
		 * region was the full screen.  That test *always* fails,
		 * because we never scroll the bottom line of the screen.
		 */
		strcpy(start, (char *)tparm(TI_csr, top, bot));
		strcpy(final, (char *)tparm(TI_csr, 0, TI_lines-1));

		if (n > 0)
		{
			sr = bot;
			er = top;
			if (TI_indn)
			{
				oneshot = 1;
				strcpy(thing, (char *)tparm(TI_indn, rn, rn));
			}
			else
				strcpy(thing, TI_ind);
		}
		else
		{
			sr = top;
			er = bot;
			if (TI_rin)
			{
				oneshot = 1;
				strcpy (thing, (char *)tparm(TI_rin, rn, rn));
			}
			else
				strcpy (thing, TI_ri);
		}
	}

	else if (TI_wind && (TI_ri || TI_rin) && (TI_ind || TI_indn))
	{
		strcpy(start, (char *)tparm(TI_wind, top, bot, 0, TI_cols-1));
		strcpy(final, (char *)tparm(TI_wind, 0, TI_lines-1, 0, TI_cols-1));

		if (n > 0)
		{
			sr = bot;
			er = top;
			if (TI_indn)
			{
				oneshot = 1;
				strcpy (thing, (char *)tparm(TI_indn, rn, rn));
			}
			else
				strcpy (thing, TI_ind);
		}
		else
		{
			sr = top;
			er = bot;
			if (TI_rin)
			{
				oneshot = 1;
				strcpy (thing,(char *)tparm(TI_rin, rn, rn));
			}
			else
				strcpy (thing, TI_ri);
		}
	}

	else if ((TI_il || TI_il1) && (TI_dl || TI_dl1))
	{
		if (n > 0)
		{
			sr = top;
			er = bot;

			if (TI_dl)
			{
				oneshot = 1;
				strcpy (thing, (char *)tparm(TI_dl, rn, rn));
			}
			else
				strcpy (thing, TI_dl1);

			if (TI_il)
			{
				oneshot = 1;
				strcpy(final, (char *)tparm(TI_il, rn, rn));
			}
			else
				strcpy(final, TI_il1);
		}
		else
		{
			sr = bot;
			er = top;
			if (TI_il)
			{
				oneshot = 1;
				strcpy (thing, (char *)tparm(TI_il, rn, rn));
			}
			else
				strcpy (thing, TI_il1);

			if (TI_dl)
			{
				oneshot = 1;
				strcpy(final, (char *)tparm(TI_dl, rn, rn));
			}
			else
				strcpy(final, TI_dl1);
		}
	}


	if (!thing[0])
		return;

	/* Do the actual work here */
	if (start[0])
		tputs_x (start);
	term_gotoxy (0, sr);

	if (oneshot)
		tputs_x (thing);
	else
	{
		for (i = 0; i < rn; i++)
			tputs_x(thing);
	}
	term_gotoxy (0, er);
	if (final[0])
		tputs_x(final);
#endif
}

/*
 * term_getsgr(int opt, int fore, int back)
 * Return the string required to set the given mode. OPT defines what
 * we really want it to do. It can have these values:
 * TERM_SGR_BOLD_ON     - turn bold mode on
 * TERM_SGR_BOLD_OFF    - turn bold mode off
 * TERM_SGR_BLINK_ON    - turn blink mode on
 * TERM_SGR_BLINK_OFF   - turn blink mode off
 * TERM_SGR_UNDL_ON     - turn underline mode on
 * TERM_SGR_UNDL_OFF    - turn underline mode off
 * TERM_SGR_REV_ON      - turn reverse video on
 * TERM_SGR_REV_OFF     - turn reverse video off
 * TERM_SGR_NORMAL      - turn all attributes off
 * TERM_SGR_RESET       - all attributes off and back to default colors
 * TERM_SGR_FOREGROUND  - set foreground color
 * TERM_SGR_BACKGROUND  - set background color
 * TERM_SGR_COLORS      - set foreground and background colors
 * TERM_SGR_GCHAR       - print graphics character
 *
 * The colors are defined as:
 * 0    - black
 * 1    - red
 * 2    - green
 * 3    - brown
 * 4    - blue
 * 5    - magenta
 * 6    - cyan
 * 7    - white
 * 8    - grey (foreground only)
 * 9    - bright red (foreground only)
 * 10   - bright green (foreground only)
 * 11   - bright yellow (foreground only)
 * 12   - bright blue (foreground only)
 * 13   - bright magenta (foreground only)
 * 14   - bright cyan (foreground only)
 * 15   - bright white (foreground only)
 */
char *term_getsgr (int opt, int fore, int back)
{
	char *ret = empty_string;

	switch (opt)
	{
		case TERM_SGR_BOLD_ON:
		case TERM_SGR_BOLD_OFF:
		case TERM_SGR_BLINK_OFF:
		case TERM_SGR_BLINK_ON:
		case TERM_SGR_UNDL_ON:
		case TERM_SGR_UNDL_OFF:
		case TERM_SGR_REV_ON:
		case TERM_SGR_REV_OFF:
		case TERM_SGR_NORMAL:
		case TERM_SGR_RESET:
			ret = TI_sgrstrs[opt-1];
			break;
		case TERM_SGR_FOREGROUND:
			ret = TI_forecolors[fore & 0x0f];
			break;
		case TERM_SGR_BACKGROUND:
			ret = TI_backcolors[fore & 0x0f];
			break;
		case TERM_SGR_GCHAR:
			if (TI_dispc)
				ret = (char *)tparm(TI_dispc, fore);
			break;
		default:
			ircpanic ("Unknown option '%d' to term_getsgr", opt);
			break;
	}
	return (ret);
}




int	orig_term_eight_bit (void)
{
#ifdef __EMX__
	return 1;
#else
	if (dumb_mode)
		return 1;
	return (((oldb.c_cflag) & CSIZE) == CS8) ? 1 : 0;
#endif
}

extern	int term_eight_bit (void)
{
#ifdef __EMX__
	return 1;
#else
	if (dumb_mode)
		return 1;
	return (((newb.c_cflag) & CSIZE) == CS8) ? 1 : 0;
#endif
}

extern	void term_beep (void)
{
	if (get_int_var(BEEP_VAR))
	{
#ifdef __EMX__
		DosBeep(1000, 200);
#else
		tputs_x(TI_bel);
		term_flush();
#endif
	}
}

extern	void	set_term_eight_bit (int value)
{
	if (dumb_mode)
		return;
	if (value == ON)
	{
		newb.c_cflag |= CS8;
		newb.c_iflag &= ~ISTRIP;
	}
	else
	{
		newb.c_cflag &= ~CS8;
		newb.c_iflag |= ISTRIP;
	}
	tcsetattr(tty_des, TCSADRAIN, &newb);
}

void	set_meta_8bit (Window *w, char *u, int value)
{
	if (dumb_mode)
		return;

	if (value == 0)
		meta_mode = 0;
	else if (value == 1)
		meta_mode = 1;
	else if (value == 2)
		meta_mode = (TI_km == 0 ? 0 : 1);
}


#endif /* NOT IN WTERM_C */

#if 0
/*
 * term.c -- for windows 95/NT
 *
 * Copyright 1997 Colten Edwards
 *
 */

#include "irc.h"
#include "vars.h"
#include "ircterm.h"
#include "window.h"
#include "screen.h"
#include "output.h"
#ifdef TRANSLATE
#include "translat.h"
#endif

#include <windows.h>

char eolbuf[200];

#ifndef WTERM_C
extern	char	*getenv();

static	int	term_CE_clear_to_eol 	(void);
static	int	term_CS_scroll 		(int, int, int);
static 	int	term_null_function 	(void);

/*
 * Function variables: each returns 1 if the function is not supported on the
 * current term type, otherwise they do their thing and return 0
 */
int	(*term_scroll) (int, int, int);	/* this is set to the best scroll available */
int	(*term_insert) (char);		/* this is set to the best insert available */
int	(*term_delete) (void);		/* this is set to the best delete available */
int	(*term_cursor_left) (void);	/* this is set to the best left available */
int	(*term_cursor_right) (void); 	/* this is set to the best right available */
int	(*term_clear_to_eol) (void); 	/* this is set... figure it out */

int nt_cursor_right(void);
int nt_cursor_left(void);

int	CO = 79,
	LI = 24;
HANDLE ghstdout;
CONSOLE_SCREEN_BUFFER_INFO gscrbuf;
CONSOLE_CURSOR_INFO gcursbuf;
DWORD gdwPlatform;

/*
 * term_reset_flag: set to true whenever the terminal is reset, thus letter
 * the calling program work out what to do
 */
	int	need_redraw = 0;
static	int	term_echo_flag = 1;
static	int	li;
static	int	co;
static	int	def_color = 0x07;
	int	CO, LI;


/*
 * term_echo: if 0, echo is turned off (all characters appear as blanks), if
 * non-zero, all is normal.  The function returns the old value of the
 * term_echo_flag
 */
int	term_echo (int flag)
{
	int	echo;

	echo = term_echo_flag;
	term_echo_flag = flag;
	return (echo);
}

void term_flush (void)
{
	return;
}

/*
 * term_reset: sets terminal attributed back to what they were before the
 * program started
 */
void term_reset (void)
{
	term_move_cursor(0, LI - 1);
}

/*
 * term_cont: sets the terminal back to IRCII stuff when it is restarted
 * after a SIGSTOP.  Somewhere, this must be used in a signal() call
 */

RETSIGTYPE term_cont (int unused)
{
	refresh_screen(0, NULL);
}

/*
 * term_pause: sets terminal back to pre-program days, then SIGSTOPs itself.
 */
extern void term_pause (char unused, char *not_used)
{
char *shell;
DWORD dwmode;
HANDLE hstdin=GetStdHandle(STD_INPUT_HANDLE);
STARTUPINFO si = { 0 };
PROCESS_INFORMATION pi = { 0 };

	si.cb = sizeof(si);

	if (!(shell = get_string_var(SHELL_VAR)))
	{
		if (gdwPlatform == VER_PLATFORM_WIN32_WINDOWS)
			shell = "command.com";
		else
			shell = "cmd.exe";
	}
	if (!(GetConsoleMode(hstdin,&dwmode)))
		return;

	if (!SetConsoleMode(hstdin, dwmode | ((ENABLE_LINE_INPUT|ENABLE_ECHO_INPUT|ENABLE_PROCESSED_INPUT) & ~ENABLE_WINDOW_INPUT)))
		return;

	if(!CreateProcess(NULL, shell, NULL, NULL, TRUE, 0, NULL, NULL, &si, &pi) )
		return;
	CloseHandle(pi.hThread);
	WaitForSingleObject(pi.hProcess,INFINITE);
	CloseHandle(pi.hProcess);

	if (!SetConsoleMode(hstdin, dwmode ) )
		return;
	refresh_screen(0,NULL);
}
#endif /* NOT IN WTERM_C */



/*
 * term_init: does all terminal initialization... reads termcap info, sets
 * the terminal to CBREAK, no ECHO mode.   Chooses the best of the terminal
 * attributes to use ..  for the version of this function that is called for
 * wserv, we set the termial to RAW, no ECHO, so that all the signals are
 * ignored.. fixes quite a few problems...  -phone, jan 1993..
 */
int term_init (void)
{
#ifndef WTERM_C

	CONSOLE_SCREEN_BUFFER_INFO scrbuf;
	HANDLE hinput =GetStdHandle(STD_INPUT_HANDLE);
	DWORD dwmode;
	OSVERSIONINFO osver;

	ghstdout = GetStdHandle(STD_OUTPUT_HANDLE);

	if(!GetConsoleScreenBufferInfo(ghstdout, &gscrbuf) ) 
	{
		fprintf(stderr,"Error from getconsolebufinfo %d\n",GetLastError());
		abort();
	}

	GetConsoleMode(hinput,&dwmode);
	SetConsoleMode(hinput,dwmode & (~(ENABLE_LINE_INPUT |ENABLE_ECHO_INPUT
				| ENABLE_PROCESSED_INPUT)| ENABLE_WINDOW_INPUT/* | ENABLE_WRAP_AT_EOL_OUTPUT*/)
				);

	osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	GetVersionEx(&osver);
	gdwPlatform = osver.dwPlatformId;

	GetConsoleCursorInfo(hinput, &gcursbuf);

	GetConsoleScreenBufferInfo(ghstdout, &scrbuf);
	li = scrbuf.srWindow.Bottom - scrbuf.srWindow.Top + 1;
	co = scrbuf.srWindow.Right - scrbuf.srWindow.Left + 1;
	LI = li;
	CO = co - 1;
	memset(eolbuf,' ',sizeof(eolbuf)-2);
	eolbuf[sizeof(eolbuf)-1] = 0;
	def_color = gscrbuf.wAttributes;

	term_clear_to_eol = term_CE_clear_to_eol;
	term_cursor_right = nt_cursor_right;
	term_cursor_left  = nt_cursor_left;
	term_scroll 	  = term_CS_scroll/* : term_param_ALDL_scroll*/;
	term_delete	  = term_null_function;
	term_insert       = term_null_function;
#endif /* NOT IN WTERM_C */
	return 0;
}


#ifndef WTERM_C
/*
 * term_resize: gets the terminal height and width.  Trys to get the info
 * from the tty driver about size, if it can't... uses the termcap values. If
 * the terminal size has changed since last time term_resize() has been
 * called, 1 is returned.  If it is unchanged, 0 is returned.
 */
int term_resize (void)
{
	static	int	old_li = -1,
			old_co = -1;

	CONSOLE_SCREEN_BUFFER_INFO scrbuf;
	ghstdout = GetStdHandle(STD_OUTPUT_HANDLE);

	GetConsoleScreenBufferInfo(ghstdout, &scrbuf);
	li = scrbuf.srWindow.Bottom - scrbuf.srWindow.Top + 1;
	co = scrbuf.srWindow.Right - scrbuf.srWindow.Left + 1;
	LI = li;
	CO = co - 1;
	if ((old_li != LI) || (old_co != CO))
	{
		old_li = LI;
		old_co = CO;
		return (1);
	}
	return (0);
}


static 	int	term_null_function _((void))
{
	return 1;
}

/* term_CE_clear_to_eol(): the clear to eol function, right? */

static	int term_CE_clear_to_eol _((void))
{
	CONSOLE_SCREEN_BUFFER_INFO scrbuf;
	HANDLE hStdout = ghstdout;
	DWORD numwrote;
	int num=0;
	COORD savepos;

	if(!GetConsoleScreenBufferInfo(hStdout, &scrbuf) )
		return 0 ;

	num = scrbuf.srWindow.Right - scrbuf.dwCursorPosition.X;
	savepos = scrbuf.dwCursorPosition;

	if (!WriteConsole(hStdout,eolbuf,num,&numwrote,NULL))
		return 0;

	SetConsoleCursorPosition(hStdout, savepos);
	return (0);
}

/*
 * term_CS_scroll: should be used if the terminal has the CS capability by
 * setting term_scroll equal to it
 */
static	int	term_CS_scroll (int line1, int line2, int n)
{
	HANDLE hStdout = ghstdout;
	SMALL_RECT src;
	SMALL_RECT clip;
	COORD dest = {0};
	CHAR_INFO chr;

	src.Left = gscrbuf.srWindow.Left;
	src.Right = gscrbuf.srWindow.Right;
	chr.Char.AsciiChar = ' ';
	chr.Attributes = FOREGROUND_BLUE | FOREGROUND_GREEN | FOREGROUND_RED;
	if (n > 0)
	{
		src.Bottom = gscrbuf.srWindow.Top + line2;
		src.Top =  gscrbuf.srWindow.Top + line1 + n;
		dest.Y =  gscrbuf.srWindow.Top + line1;
		ScrollConsoleScreenBuffer(hStdout, &src, NULL, dest, &chr);
	}
	else
	{
		clip = src;
        	src.Bottom = gscrbuf.srWindow.Top + line2 + line1 + n;
		src.Top = gscrbuf.srWindow.Top + line1;
		dest.Y = gscrbuf.srWindow.Top + line1 - n;
		clip.Top = gscrbuf.srWindow.Top + line1;
		clip.Bottom = gscrbuf.srWindow.Top + line1 + line2;
		ScrollConsoleScreenBuffer(hStdout, &src, &clip, dest, &chr);
	}
	return 0;
}

extern	void	copy_window_size (int *lines, int *columns)
{
	*lines = LI;
	*columns = CO;
}

extern	int term_eight_bit (void)
{
	return 1;
}

extern	void term_beep (void)
{
	if (get_int_var(BEEP_VAR))
	{
       		Beep(0x637, 60/8);
	}
}

void ScrollBuf(HANDLE hOut, CONSOLE_SCREEN_BUFFER_INFO *scrbuf,int where)
{
	SMALL_RECT src;
	COORD dest;
	CHAR_INFO chr;

	src.Left = scrbuf->srWindow.Left;
	src.Top = scrbuf->srWindow.Top+where ;
	src.Right = scrbuf->srWindow.Right;
	src.Bottom = scrbuf->srWindow.Bottom;

	dest.X = 0;
	dest.Y = 0;

	chr.Char.AsciiChar = ' ';
	chr.Attributes = 0;
	ScrollConsoleScreenBuffer(hOut, &src, NULL, dest, &chr);

}

void NT_MoveToLineOrChar(void *fd, int where,int line)
{
	CONSOLE_SCREEN_BUFFER_INFO gscrbuf;
	GetConsoleScreenBufferInfo((HANDLE)fd, &gscrbuf);
	if (line)
	{
		if ( ((gscrbuf.dwCursorPosition.Y+where)> (gscrbuf.srWindow.Bottom-1))
			&&( where >0))
			ScrollBuf((HANDLE)fd,&gscrbuf,where);
		else
			gscrbuf.dwCursorPosition.Y += where;
	}
	else
		gscrbuf.dwCursorPosition.X += where;
	SetConsoleCursorPosition((HANDLE)fd, gscrbuf.dwCursorPosition);
}

void NT_MoveToLineOrCharAbs(void *fd, int where,int line)
{

	CONSOLE_SCREEN_BUFFER_INFO scrbuf;
        GetConsoleScreenBufferInfo((HANDLE)fd, &scrbuf);

	if (line)
		scrbuf.dwCursorPosition.Y = where+scrbuf.srWindow.Top;
	else
		scrbuf.dwCursorPosition.X = where;
	SetConsoleCursorPosition((HANDLE)fd, scrbuf.dwCursorPosition);
}

void term_move_cursor(int cursor, int line)
{
	HANDLE hStdout =ghstdout;
	COORD dest;
	dest.X = cursor;
	dest.Y = line + gscrbuf.srWindow.Top;
	SetConsoleCursorPosition(hStdout, dest);
}


void NT_ClearScreen(void)
{
	CONSOLE_SCREEN_BUFFER_INFO scrbuf;
	DWORD numwrote;
	COORD origin = {0, 0};
	HANDLE hStdout = GetStdHandle(STD_OUTPUT_HANDLE);

	GetConsoleScreenBufferInfo(hStdout, &scrbuf);
	FillConsoleOutputCharacter(hStdout,' ',scrbuf.dwSize.X*scrbuf.dwSize.Y,
		origin,&numwrote);
	FillConsoleOutputAttribute(hStdout,scrbuf.wAttributes,
		scrbuf.dwSize.X*scrbuf.dwSize.Y,scrbuf.dwCursorPosition,&numwrote);
	origin.X = scrbuf.srWindow.Left;
	origin.Y = scrbuf.srWindow.Top;
	SetConsoleCursorPosition(hStdout, origin);
	return;
}

int nt_cursor_right(void)
{
	NT_MoveToLineOrChar(GetStdHandle(STD_OUTPUT_HANDLE), 1,0);
	return 0;
}

int nt_cursor_left(void)
{
	NT_MoveToLineOrChar(GetStdHandle(STD_OUTPUT_HANDLE), -1,0);
	return 0;
}

extern	void	set_term_eight_bit (int value)
{
	return;
}

#define NPAR 16

static unsigned int color_table[] = {
        0x00,   0x04,   0x02,   0x06,    0x01,   0x05,   0x03,   0x07,
        0x00|8, 0x04|8, 0x02|8, 0x06|8,  0x01|8, 0x05|8, 0x03|8, 0x07|8
};

enum VC_STATES { ESnormal = 0, ESesc, ESsquare, ESgetpars, ESgotpars, ESfunckey };
enum VC_STATES vc_state;
unsigned long par[NPAR+1] = {0};
unsigned long npar = 0;
int intensity = 1;
int blink = 0;
int underline = 0;
int reverse_ch = 0;


void reset_attrib(void *fd, int *fg, int *bg)
{
	intensity = 0;
	underline = 0;
	reverse_ch = 0;
	blink = 0;
	*fg=FOREGROUND_BLUE | FOREGROUND_GREEN | FOREGROUND_RED;
	*bg=0;
}


void csi_m(void *fd)
{
int i;
static int fg = 7;
static int bg = 0;
	CONSOLE_SCREEN_BUFFER_INFO scrbuf;
	GetConsoleScreenBufferInfo((HANDLE)fd, &scrbuf);
	for (i=0; i <= npar; i++)
	{
        	switch(par[i])
        	{
        		case 0:
                        	/* all off */
				reset_attrib(fd, &fg, &bg);
                        	break;
        		case 1: /* bold */
        			intensity = FOREGROUND_INTENSITY;
        			break;
			case 2: /* all off */
				fg = 0;
				bg = 0;
				intensity = 0;
				break;
        		case 4: /* underline */
		               	fg=FOREGROUND_BLUE | FOREGROUND_RED;
		               	intensity=FOREGROUND_INTENSITY;
		               	bg=0;
                        	break;
        		case 5: /* blink */
				fg=FOREGROUND_BLUE | FOREGROUND_GREEN | FOREGROUND_RED;
				bg=0;
				intensity=FOREGROUND_INTENSITY | BACKGROUND_INTENSITY;
        			break;
        		case 7: /* reverse */
				fg=0;
				bg=BACKGROUND_BLUE | BACKGROUND_GREEN | BACKGROUND_RED;
				intensity=0;
				break;
        		case 49:
        			fg = (def_color & 0xf0) | bg;
        			break;
        		default:
				if (par[i] >=30 && par[i] <= 37)
					fg = color_table[par[i]-30];
				else if (par[i] >=40 && par[i] <= 47)
					bg = color_table[par[i]-40]<<4;
        			break;

        	}
	}
	/* update attribs now */
	SetConsoleTextAttribute(fd, fg | bg | intensity);
}

void set_term_intensity(int flag)
{
	par[0] = flag ? 1 : 0;
	npar = 0;
	csi_m(GetStdHandle(STD_OUTPUT_HANDLE));
}
void set_term_inverse(int flag)
{
	par[0] = flag ? 7 : 0;
	npar = 0;
	csi_m(GetStdHandle(STD_OUTPUT_HANDLE));
}
void set_term_underline(int flag)
{
	par[0] = flag ? 4 : 0;
	npar = 0;
	csi_m(GetStdHandle(STD_OUTPUT_HANDLE));
}

void csi_K(void *fd, int parm)
{
COORD save_curs = {0};
CONSOLE_SCREEN_BUFFER_INFO scrbuf;
DWORD bytes_rtn;
int num = 0;
	switch(parm)
	{
		case 2:
			NT_MoveToLineOrCharAbs(fd, 0, 0);
		case 0:
			term_CE_clear_to_eol();
			break;
		case 1:
			GetConsoleScreenBufferInfo((HANDLE)fd, &scrbuf);
			save_curs.Y = scrbuf.dwCursorPosition.Y;
			SetConsoleCursorPosition((HANDLE)fd, save_curs);
			num = scrbuf.dwCursorPosition.X - 1;
			if (num > 0)
	                        WriteConsole((HANDLE)fd, eolbuf, num, &bytes_rtn, NULL);
			SetConsoleCursorPosition((HANDLE)fd, scrbuf.dwCursorPosition);
			break;
	}
}

int nt_write(void * fd, unsigned char *buf, register int howmany)
{
	DWORD bytes_rtn;
	int num = 0;
	int len = -1;
	register int c;
	int ok;
	register unsigned char *buf1 = buf;
	vc_state = ESnormal;

	while (howmany)
	{
		howmany--;
                num++;
                ok = 0;
		c = *buf1;
		buf1++;
		len++;
		if (!ok && c)
		{
		        switch(c)
			{
		                case 0x1b:
		                case REV_TOG:
		                case UND_TOG:
		                case BOLD_TOG:
		                case ALL_OFF:
		                	ok = 0;
                                	break;
		                default:
					ok = 1;
                	}
		}
		if (vc_state == ESnormal && ok)
		{
                        WriteConsole((HANDLE)fd, buf+len,1, &bytes_rtn, NULL);
			continue;
		}

        	switch(c)
        	{
			case REV_TOG:
				if (reverse_ch) reverse_ch = 0; else reverse_ch = 1;
				set_term_inverse(reverse_ch);
				continue;
			case BOLD_TOG:
				if (intensity) intensity = 0; else intensity = 1;
				set_term_intensity(intensity);
				continue;
			case UND_TOG:
				if (underline) underline = 0; else underline = 1;
                        	set_term_underline(underline);
                        	continue;
			case ALL_OFF:
				set_term_underline(0);
				set_term_intensity(0);
				set_term_inverse(0); /* This turns it all off */
				intensity = 0;
				underline = 0;
				reverse_ch = 0;
				if (!buf1)	/* We're the end of line */
					term_bold_off();
				continue;
                	case 7:
                		Beep(0x637, 60/8);
                        	continue;
                	case 8:
                        	continue;
                	case 9:
                        	continue;
                	case 10: case 11: case 12:
				NT_MoveToLineOrChar(fd, 1, 1);
			case 13:
				NT_MoveToLineOrCharAbs(fd, 0, 0);
                        	continue;
                	case 24: case 26:
                		vc_state = ESnormal;
                        	continue;
			case 27:
				vc_state = ESesc;
                        	continue;
                	case 128+27:
                		vc_state = ESsquare;
                        	continue;
        	}
        	switch(vc_state)
        	{
        		case ESesc:
        			vc_state = ESnormal;
                        	switch(c)
                        	{
                        		case '[':
                        			vc_state = ESsquare;
                        			continue;
                        	}
                        	continue;
                	case ESsquare:
                		vc_state = ESgetpars;
				for(npar = 0 ; npar < NPAR ; npar++)
					par[npar] = 0;
				npar = 0;
				vc_state = ESgetpars;
				if (c == '[') { /* Function key */
					vc_state=ESfunckey;
					continue;
				}
			case ESgetpars:
				if (c==';' && npar<NPAR-1)
				{
					npar++;
					continue;
				} else if (c>='0' && c<='9')
				{
					par[npar] *= 10;
					par[npar] += c-'0';
					continue;
				} else
					vc_state=ESgotpars;
			case ESgotpars:
				vc_state = ESnormal;
				switch(c)
				{
					case 'C': case 'a':
						NT_MoveToLineOrChar(fd, par[0], 0);
						continue;
					case 'K':
						csi_K(fd, par[0]);
						continue;
                                	case 'm':
                                		csi_m(fd);
	                               		continue;
					case 'r':
					{
						int top, bottom;
						if (!par[0]) par[0]++;
						if (!par[1]) par[1] = li;
						if (par[0] < par[1] && par[1] < li)
						{
							top = par[0]-1;
							bottom = par[1];
                                                	term_CS_scroll(top, bottom, 1);
						}
						continue;
					}
				}
			        continue;
			default:
				vc_state = ESnormal;

        	}
	}
	return num;
}

#endif /* NOT IN WTERM_C */
#endif