/* sounding.c */

/* Vis5D version 4.3 */
/*ha*/
/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


/*
 * This plots temperature and dewpoint on a skew-t or
 * plots up to three vertical plots of any variable 
 */



#include <math.h>
#include "../lui5/lui.h"
#include <stdlib.h>
#include <strings.h>
#include "globals.h"
#include "graphics.h"
#include "grid.h"
#include "memory.h"
#include "proj.h"
#include "gui.h"
#include "sounding.h"
#include "soundingGUI.h"
#ifdef OPENGL
#  include <GL/gl.h>
#endif
#if defined(SGI_GL) || defined(DENALI)
#  include <gl/gl.h>
#endif
#include "vis5d.h"

#define PI 3.13159265

#define BORDER 65 

#define HEBGBS 0  

#define TICK_DASH_LENGTH 2 

#define PF_TRUECOLOR 0

#define PF_XALLOC    1

#define PF_8BIT      2




static unsigned long SND_AllocateColorInt( int r, int g, int b );

Status SND_XAllocColor( Display *dpy, Colormap cmap, int cmap_size,
                        XColor *color );

void SND_Initialize(  Context ctx, Display *display,
                     Visual *visual, int depth, Colormap colormap );

GC make_gc(Context ctx, int foregroundR, int foregroundG, int foregroundB,
                  int backgroundR, int backgroundG, int backgroundB, int linewidth);

static void make_soundpixmap( Context ctx);

static void draw_var_stuff( Context ctx, int var);

static void draw_ticks( Context ctx, int var);

static float pres_to_height(float pressure);

static float height_to_pres(float height);

static void draw_box ( Context ctx );

static float svp( float K);

static float mixratio( float K, float pres );

static float thetaE( float pres, float temp);
  
static float get_temp_for_thte(float thte, float pres);

static void draw_wlines( Context ctx);

static void draw_thtelines( Context ctx );

static void draw_thtalines( Context ctx );

static void cut_line_data2( Context ctx, int *x1, int *y1, int *x2, int *y2);

static void precut_line_data (Context ctx, int *x1, int *y1, int x2, int y2);

static void cut_line_data( Context ctx, int x1, int y1, int *x2, int *y2);

static void draw_vert_stuff( Context ctx );

static void draw_millibarlines ( Context ctx );

static void make_a_barb( Context ctx, float spd, float dir, float height);

static void drawbarbflag( Context ctx, XPoint flagpoints[], int up, float dir);

static void drawbarbline( Context ctx, int x1, int y1, int x2, int y2,
                           int up, float dir);

static void convert_xy_to_barb(Context ctx, int inx, int iny, float dir,
                                          int *outx, int *outy);

void vardata_to_xy(Context ctx, float alt, float value, float min, float max, int *x, int *y);

void setvarsteps( Context ctx);

void data_to_xy(Context ctx, float alt, float temp, int *x, int *y);

void data_to_y (Context ctx, float alt, int *y);

float grid_level_to_height( Context ctx, float level );

static int extract_sound( Context ctx, float *grid, int var,
                             int nr, int nc, int nl, int lowlev,
                             float row, float col);

static int extract_wind( Context ctx, float *gridu, float *gridv,
                             int varu, int varv,
                             int nr, int nc, int nl, int lowlev,
                             float row, float col);

static float winterval[34] = { 0.1, 0.2, 0.4, 0.6, 0.8, 1.0, 1.5,
                               2.0, 2.5, 3.0, 4.0, 5.0, 6.0, 7.0, 
                               8.0, 9.0, 10.0, 12.0, 14.0, 16.0, 18.0,
                               20.0, 24.0, 28.0, 32.0, 36.0, 40.0, 44.0,
                               48.0, 52.0, 56.0, 60.0, 68.0, 76.0 };

static int pixelformat;

static unsigned long ctable8[5][9][5];   /* Only for PF_8BIT */

static unsigned long rtable[256], gtable[256], btable[256];  /* PF_TRUECOLOR */




    /**********************************************************************/
    /* This just resizes the window to the given width and height         */
    /**********************************************************************/
    /* Input: ctx- context index                                          */
    /*        width- the width of the resized window                      */
    /*        height- the height of the resized window                    */
    /**********************************************************************/


void resize_snd_window( Context ctx, int width, int height, int x, int y)
{
   ctx->Sound.soundwin_width = width;
   ctx->Sound.soundwin_height = height;
   if (x==0 && y == 0 ){
      x = ctx->Sound.sndx;
      y = ctx->Sound.sndy;
   }
   else {
      ctx->Sound.sndx = x;
      ctx->Sound.sndy = y;
   }
   XMoveResizeWindow( SndDpy, ctx->Sound.soundwin, x, y, width, height);
   if (ctx->Sound.SoundCtrlWindow != 0){
      if (ctx->Sound.otherdpy){
         ctx->Sound.sndheight = height -2*BORDER;
      }
      else {
         ctx->Sound.sndheight = height -95 -2*BORDER;
      }
      ctx->Sound.sndwidth =  width - 2 * BORDER;
      do_pixmap_art ( ctx );
      draw_sounding(ctx, ctx->CurTime);
   }
   else {
      ctx->Sound.sndheight = height - 2 * BORDER;
      ctx->Sound.sndwidth =  width - 2 * BORDER;
      do_pixmap_art ( ctx );
      draw_sounding(ctx, ctx->CurTime);
   }
}   




/*
 * Given an RGB color, return the corresponding pixel value.
 * Input;  r, g, b - red, green, and blue in [0,255]
 * Return:  a pixel value
 *
 ***** This is the same code found in lui.c lines 236 - 263 ****
 */
static unsigned long SND_AllocateColorInt( int r, int g, int b )
{
   XColor xcol;

   switch (pixelformat) {
      case PF_TRUECOLOR:
         return rtable[r] | gtable[g] | btable[b];
      case PF_8BIT:
         return ctable8[r/52][g/31][b/52];
      case PF_XALLOC:
         xcol.red = r << 8;
         xcol.green = g << 8;
         xcol.blue = b << 8;
         SND_XAllocColor( SndDpy, SndColormap, SndVisual->map_entries,
                          &xcol );
         return xcol.pixel;
      default:
         printf("Error in SND_AllocateColorInt %d\n", pixelformat);
         exit(0);
   }
   return 0;
}


/*
 * A replacement for XAllocColor.  This function should never fail
 * to allocate a color.  When XAllocColor fails we return the nearest
 * matching color.
 *
 ***** This is the same code found in lui.c lines 181 - 232 **** 
 */
Status SND_XAllocColor( Display *dpy, Colormap cmap, int cmap_size,
                        XColor *color )
{
   int p, bestmatch;
   double dist, mindist;  /* 3*2^16^2 exceeds long int precision */
   static XColor *allcolors = NULL;
   XColor *acptr;

#define DISTANCE(r1,g1,b1,r2,g2,b2)  ( ((r2)-(r1)) * ((r2)-(r1)) + \
    ((g2)-(g1)) * ((g2)-(g1)) + ((b2)-(b1)) * ((b2)-(b1)) )

   if (!XAllocColor(dpy, cmap, color)) {
      /* query whole colormap if not yet done */
      if (!allcolors) {
         allcolors = (XColor *) malloc( cmap_size * sizeof(XColor) );
         for (p = 0;  p < cmap_size;  p++)
           allcolors[p].pixel = p;
         XQueryColors (dpy, cmap, allcolors, cmap_size);
      }

      /* find best match */
      bestmatch = -1;
      mindist = 0.0;
      p = cmap_size;
      while (p--) {
         acptr = allcolors + p;
         dist = DISTANCE( (double)color->red, (double)color->green,
                          (double)color->blue, (double)acptr->red,
                          (double)acptr->green, (double)acptr->blue);
         if (bestmatch < 0 || dist < mindist)
           mindist = dist, bestmatch = p;
      }
      color->red   = allcolors[bestmatch].red;
      color->green = allcolors[bestmatch].green;
      color->blue  = allcolors[bestmatch].blue;
      if (!XAllocColor( dpy, cmap, color )) {
         /* this is a real hack but should be good enough */
         color->pixel = bestmatch;
      }
   }
#undef DISTANCE

   return 1;
}


/*
 * Input:  program_name - the label for all window title bars
 *         display - the X display (or NULL)
 *         visual - the X visual (or NULL)
 *         depth - the depth of the visual (or 0)
 *         colormap - the X colormap (or 0)
 */
void SND_Initialize(  Context ctx, Display *display,
                     Visual *visual, int depth, Colormap colormap )
{
   static int initialized = 0;
   XSetWindowAttributes attr;
   XVisualInfo visinfo;
   int vertical;
   float vjunk[MAXLEVELS];

   /* Only do this once */
   if (initialized)
      return;
   else
      initialized = 1;

   ctx->Sound.get_vert_data = 1;
   vis5d_get_vertical(ctx->context_index, &vertical, vjunk);
   ctx->Sound.vertsys = vertical;
   if( ctx->TopBound < 1.0 && ctx->BottomBound < -1.0 ) {
      ctx->Sound.oceanonly = 1;
   }
   else {
      ctx->Sound.oceanonly = 0;
   }
 
   /* Open the display if one wasn't passed (already opened) */
   if (display) {
      SndDpy  = display;
   }
   else {
      SndDpy = XOpenDisplay(NULL);
      if (!SndDpy) {
         printf("Can't open sound display");
      }
   }

   /* Set defaults */
   SndRootWindow = DefaultRootWindow (SndDpy);
   SndScr        = DefaultScreen ( SndDpy );
   SndScrWidth   = DisplayWidth (SndDpy, SndScr );
   SndScrHeight  = DisplayHeight( SndDpy, SndScr );


   if (visual) {
      SndVisual = visual;
      SndDepth = depth;
      SndColormap = colormap;
   }
   else {
      if (XMatchVisualInfo( SndDpy,SndScr,24,TrueColor,&visinfo )) {
         SndVisual = visinfo.visual;
         SndDepth = 24;
         SndColormap = XCreateColormap( SndDpy, RootWindow(SndDpy, SndScr),
                                         SndVisual, AllocNone );
      }
      else {
         SndVisual = DefaultVisual( SndDpy, SndScr );
         SndDepth = DefaultDepth( SndDpy, SndScr );
         SndColormap = DefaultColormap( SndDpy, SndScr );
      }
   }

   /* Setup color allocation stuff */
   if (SndVisual->class==TrueColor || SndVisual->class==DirectColor) {
      /* Initialize rtable[], gtable[], and btable[] */
      XColor xcol;
      int i;
      xcol.green = 0;
      xcol.blue = 0;
      for (i=0;i<256;i++) {
         xcol.red = i * 0xffff / 0xff;
         XAllocColor( SndDpy, SndColormap, &xcol );
         rtable[i] = xcol.pixel;
      }
      xcol.red = 0;
      xcol.blue = 0;
      for (i=0;i<256;i++) {
         xcol.green = i * 0xffff / 0xff;
         XAllocColor( SndDpy, SndColormap, &xcol );
         gtable[i] = xcol.pixel;
      }
     xcol.red = 0;
      xcol.green = 0;
      for (i=0;i<256;i++) {
         xcol.blue = i * 0xffff / 0xff;
         XAllocColor( SndDpy, SndColormap, &xcol );
         btable[i] = xcol.pixel;
      }
      pixelformat = PF_TRUECOLOR;
   }
   else if (SndVisual->class==PseudoColor) {
      /* Note: the color allocation scheme must be the same as what's used */
      /* in Mesa to allow colormap sharing! */
      int r, g, b;
      for (r=0;r<5;r++) {
         for (g=0;g<9;g++) {
            for (b=0;b<5;b++) {
               XColor xcol;
               xcol.red   = r * 65535 / 4;
               xcol.green = g * 65535 / 8;
               xcol.blue  = b * 65535 / 4;
               SND_XAllocColor( SndDpy, SndColormap,
                                SndVisual->map_entries, &xcol );
               ctable8[r][g][b] = xcol.pixel;
            }
         }
      }
      pixelformat = PF_8BIT;
   }
   else {
      pixelformat = PF_XALLOC;
   }
   /* now just set some variables for the first time */
   ctx->Sound.mainvarstep = 50;
   ctx->Sound.SndMinTemp = 228.0;
   ctx->Sound.SndMaxTemp = 323.0;
   ctx->Sound.tickstatus = 0;
   ctx->Sound.currentX = .69;
   ctx->Sound.currentY = .69;
   ctx->Sound.currentTime =1069;
   ctx->Sound.soundline = NULL;
   ctx->Sound.uwindline = NULL;
   ctx->Sound.vwindline = NULL;
   ctx->Sound.tgrid = NULL;
   ctx->Sound.dgrid = NULL;
   ctx->Sound.ugrid = NULL;
   ctx->Sound.var1grid = NULL;
   ctx->Sound.var2grid = NULL;
   ctx->Sound.var3grid = NULL;
   ctx->Sound.vertdata = NULL;
   ctx->Sound.PreviousSoundTemp = vis5d_find_var(ctx->context_index,"T");
   ctx->Sound.PreviousSoundDewpt= vis5d_find_var(ctx->context_index,"TD");
   ctx->Sound.PreviousSoundUWind= vis5d_find_var(ctx->context_index,"U");
   ctx->Sound.PreviousSoundVWind= vis5d_find_var(ctx->context_index,"V");
   ctx->Sound.PreviousSoundVar1 = -1;
   ctx->Sound.PreviousSoundVar2 = -1;
   ctx->Sound.PreviousSoundVar3 = -1;
   ctx->Sound.sndx = 15;
   ctx->Sound.sndy = 15; 
   vis5d_set_sound_vars( ctx->context_index, vis5d_find_var(ctx->context_index,"T"),
                                vis5d_find_var(ctx->context_index,"TD"),
                                vis5d_find_var(ctx->context_index,"U"),
                                vis5d_find_var(ctx->context_index,"V"),
                                -1, -1, -1 );
 
/* XSynchronize(SndDpy, True); */
}




    /**********************************************************************/
    /* This makes a graphics context                                      */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        foregroundR - the red value 1..255 of the foreground        */
    /*        foregroundG - the green value 1..255 of the foreground      */
    /*        foregroundB - the blue value 1..255 of the foreground       */
    /*        backgroundR - the red value 1..255 fo the background        */
    /*        backgroundG - the green value 1..255 of the background      */
    /*        backgroundB - the blue value 1..255 of the background       */
    /*        linewidth - the line width value for the GC                 */
    /*                                                                    */
    /* Output: make_gc- the GC with the above specifications              */
    /**********************************************************************/

GC make_gc(Context ctx, int foregroundR, int foregroundG, int foregroundB,
                  int backgroundR, int backgroundG, int backgroundB, int linewidth)
{
   XGCValues vals;
   
   vals.foreground = SND_AllocateColorInt(foregroundR,
                                   foregroundG, foregroundB);
   vals.background = SND_AllocateColorInt(backgroundR,
                                   backgroundG, backgroundB);
   vals.line_width = linewidth;
   return XCreateGC ( SndDpy, ctx->Sound.soundwin, GCLineWidth |
                      GCForeground | GCBackground, &vals );
}




    /**********************************************************************/
    /* This makes the window that all of the sounding graphics is         */
    /* drawn or mapped to.  The sndheight and sndwidth are the dimentions */
    /* inside this window to which the actual data is drawn to.           */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        title - title for window if there is no parent              */
    /*        xpos, ypos - the x and y position of the window if no parent*/
    /*        width, height - of the window                               */
    /* Output: 1 if it works                                              */
    /**********************************************************************/
    
int make_soundGFX_window( Context ctx, char *title, int xpos, int ypos,
                           int width, int height, Window ctrlwindow, char *wdpy_name)
{
   XSetWindowAttributes attr;
   int attr_flags;
   XWindowAttributes winatts;
   
   ctx->Sound.SoundCtrlWindow = ctrlwindow;
   SND_Initialize( ctx, SndDpy, SndVisual, SndDepth, SndColormap);
   attr.event_mask = ExposureMask; 
   attr.colormap = SndColormap;
   attr.background_pixel = BlackPixel( SndDpy, SndScr );
   attr.border_pixel =BlackPixel( SndDpy, SndScr ); 
   attr_flags = CWColormap | CWEventMask | CWBackPixel | CWBorderPixel;
   if (wdpy_name != NULL) {
      ctx->Sound.otherdpy = 1;
   }
   if ((ctx->Sound.SoundCtrlWindow != 0) && (wdpy_name == NULL)) {
      XGetWindowAttributes( SndDpy, ctx->Sound.SoundCtrlWindow, &winatts);

      ctx->Sound.soundwin = XCreateWindow(SndDpy, ctx->Sound.SoundCtrlWindow,
                       0, 95, winatts.width, winatts.height-95+HEBGBS,
                       1, SndDepth, InputOutput,
                       SndVisual, attr_flags, &attr);
      ctx->Sound.soundwin_width = winatts.width;
      ctx->Sound.soundwin_height= winatts.height-95+HEBGBS;
      ctx->Sound.sndheight = winatts.height-95-2*BORDER;
      ctx->Sound.sndwidth = winatts.width-2*BORDER;
   }
   else {
      XSizeHints sizehints;

      ctx->Sound.soundwin = XCreateWindow(SndDpy, RootWindow(SndDpy, SndScr),
                       xpos, ypos, width, height, 1, SndDepth, InputOutput,
                       SndVisual, attr_flags, &attr);
      ctx->Sound.soundwin_width = width;
      ctx->Sound.soundwin_height= height ;
      ctx->Sound.sndheight = height -2*BORDER;
      ctx->Sound.sndwidth  = width - 2*BORDER;
      sizehints.x = 20;
      sizehints.y = 40;
      sizehints.width = 200;
      sizehints.height = 200;
      sizehints.flags = PPosition | PSize; 

      XSetStandardProperties(SndDpy, ctx->Sound.soundwin,
                             "Skew-T and Vertical Plot Display",
                             "Skew-T and Vertical Plot Display",
                             None, (char**)NULL, 0, &sizehints);

   }
   ctx->Sound.vert_gc = make_gc(ctx, 255, 255, 255, 0, 0, 0, 2);
   ctx->Sound.Tempgc = make_gc(ctx, 255, 255, 255, 0, 0, 0, 2);
   ctx->Sound.Dewptgc= make_gc(ctx, 255, 0, 255, 0, 0, 0, 2);
   ctx->Sound.barb_gc = make_gc(ctx, 0, 0, 255, 0, 0, 0, 2);
   ctx->Sound.barb2_gc= make_gc(ctx, 255,255,255, 0, 0, 0, 1);
   ctx->Sound.var1_gc = make_gc(ctx, 0, 255, 0, 0, 0, 0, 2);
   ctx->Sound.var2_gc = make_gc(ctx, 100, 255, 255, 0, 0, 0, 2);
   ctx->Sound.var3_gc = make_gc(ctx, 255, 255, 100, 0, 0, 0, 2);
   ctx->Sound.rect_gc =  make_gc(ctx, 0, 0, 0, 1, 1, 1, 1);
   ctx->Sound.box_gc =  make_gc(ctx, 169, 169, 169, 0, 0, 0, 1);
   if  (XLoadFont(SndDpy, "6x12")){
      XSetFont(SndDpy, ctx->Sound.var1_gc, XLoadFont(SndDpy, "6x12"));
      XSetFont(SndDpy, ctx->Sound.var2_gc, XLoadFont(SndDpy, "6x12"));
      XSetFont(SndDpy, ctx->Sound.var3_gc, XLoadFont(SndDpy, "6x12"));
   }
   do_pixmap_art(ctx);
/*   if (ctx->Sound.SoundCtrlWindow != 0){
      XMapWindow(SndDpy, ctx->Sound.soundwin);
   } */
   do_pixmap_art(ctx);
   return 1;
}

    /**********************************************************************/
    /* This makes the pixmap to which all the extraneous graphics         */
    /* such as the theta lines, thtate lines, millibar lines, outside box */
    /* are drawn to.                                                      */
    /**********************************************************************/
    /* Input - context                                                    */
    /**********************************************************************/

static void make_soundpixmap( Context ctx)
{
   if (ctx->Sound.soundpix){
      XFreePixmap(SndDpy, ctx->Sound.soundpix);
   } 
   if( ctx->Sound.SoundCtrlWindow ) {
      ctx->Sound.soundpix = XCreatePixmap( SndDpy, ctx->Sound.soundwin,
                                          ctx->Sound.sndwidth+2*BORDER,
                                          ctx->Sound.sndheight+95+2*BORDER,
                                          SndDepth);
      XFillRectangle(SndDpy, ctx->Sound.soundpix, ctx->Sound.rect_gc, 0, 0,
                     ctx->Sound.sndwidth+2*BORDER,
                     ctx->Sound.sndheight + 95 +2*BORDER);
   }
   else {
      ctx->Sound.soundpix = XCreatePixmap( SndDpy, ctx->Sound.soundwin,
                                          ctx->Sound.sndwidth+2*BORDER,
                                          ctx->Sound.sndheight+2*BORDER,
                                          SndDepth);
      XFillRectangle(SndDpy, ctx->Sound.soundpix, ctx->Sound.rect_gc, 0, 0,
                     ctx->Sound.sndwidth+2*BORDER,
                     ctx->Sound.sndheight+2*BORDER);
   }
}

    /**********************************************************************/
    /* This draws all the extraneous gphics to a pixmap if they are wanted*/
    /**********************************************************************/
    /* Input - context                                                    */
    /**********************************************************************/

void do_pixmap_art( Context ctx )
{ 


   setvarsteps( ctx);

   make_soundpixmap( ctx);
   if ( ctx->Sound.wstatus ){
      draw_wlines(ctx);
   }
   if ( ctx->Sound.thtestatus ){
      draw_thtelines(ctx);
   }
   if ( ctx->Sound.thtastatus ){
      draw_thtalines(ctx);
   }
   if (ctx->Sound.SoundVar1 >= 0){
      draw_var_stuff(ctx, ctx->Sound.SoundVar1);
      if(ctx->Sound.tickstatus) draw_ticks(ctx, ctx->Sound.SoundVar1);
   }
   if (ctx->Sound.SoundVar2 >= 0){
      draw_var_stuff(ctx, ctx->Sound.SoundVar2);
      if(ctx->Sound.tickstatus) draw_ticks(ctx, ctx->Sound.SoundVar2);
   }
   if (ctx->Sound.SoundVar3 >= 0){
      draw_var_stuff(ctx, ctx->Sound.SoundVar3);
      if(ctx->Sound.tickstatus) draw_ticks(ctx, ctx->Sound.SoundVar3);
   }
   if ((ctx->Sound.vertsys != 0) && (ctx->Sound.oceanonly != 1))  draw_millibarlines(ctx);
   draw_box(ctx);
   if (!ctx->Sound.get_vert_data)  draw_vert_stuff(ctx);
}


    /**********************************************************************/
    /* This draws the numbers and unit for each variable alond the bottom */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        var - the variable whose numbers and unit should be drawn   */
    /**********************************************************************/

static void draw_var_stuff( Context ctx, int var)
{
   int counter= -1;
   float yo, step;
   GC var_gc;
   char num[9];
   int vary = 25;
   int stringnumber;
   char varunit[10] = "";
   int strlength = 0;

   if (var == ctx->Sound.SoundVar1){
      var_gc = ctx->Sound.var1_gc;
      vary += 10;
      step = ctx->Sound.var1step;
      strlength = strlen(ctx->Units[var]);
   }
   if (var == ctx->Sound.SoundVar2){
      var_gc = ctx->Sound.var2_gc;
      vary += 22;
      step = ctx->Sound.var2step;
      strlength = strlen(ctx->Units[var]);
   }
   if (var == ctx->Sound.SoundVar3){
      var_gc = ctx->Sound.var3_gc;
      vary += 34;
      step = ctx->Sound.var3step;
      strlength = strlen(ctx->Units[var]);
   }
   if (ctx->Sound.samestepflag){
      for ( yo = ctx->Sound.samestepmin; yo <= ctx->Sound.samestepmax; yo += step){
         counter ++;
         sprintf(num, "%.1f\n", yo );
         stringnumber = strlen(num)-1;
         if ( (ctx->Sound.mainvarstep * counter)+BORDER < ctx->Sound.sndwidth+BORDER){
            XDrawString( SndDpy, ctx->Sound.soundpix, var_gc,
                         (ctx->Sound.mainvarstep * counter+1)+BORDER-15,
                          ctx->Sound.sndheight+BORDER-HEBGBS+vary,
                          num, stringnumber);
         }
         if(stringnumber > 7){
            yo += step;
            counter ++;
         }
      }
   }
   else{
      for ( yo = ctx->MinVal[var]; yo <= ctx->MaxVal[var]; yo += step){
         counter ++;
         sprintf(num, "%.1f\n", yo );
         stringnumber = strlen(num)-1;
         if ( (ctx->Sound.mainvarstep * counter)+BORDER < ctx->Sound.sndwidth+BORDER){
            XDrawString( SndDpy, ctx->Sound.soundpix, var_gc,
                         (ctx->Sound.mainvarstep * counter+1)+BORDER-15,
                          ctx->Sound.sndheight+BORDER-HEBGBS+vary,
                          num, stringnumber);
         }
         if(stringnumber > 7){
            yo += step;
            counter ++;
         }
      }    
   }
   XDrawString( SndDpy, ctx->Sound.soundpix, var_gc,
                BORDER - 45, ctx->Sound.sndheight + BORDER-HEBGBS+vary,
                ctx->Units[var], strlength);
}

   /*****************************************************************/
   /* This draws the vertical ticks for the different vars          */
   /*****************************************************************/
   /* Input: ctx- context                                           */
   /*        var - the variable for which the vertical ticks should */
   /*              be drawn                                         */
   /*****************************************************************/

 
static void draw_ticks( Context ctx, int var)
{
   float step, yo;
   int dash_offset;
   int counter;
   int dash_list_length = { TICK_DASH_LENGTH };
   static  char dotted[2] = {4,12};
   char *dash_list[] = {dotted};
   GC var_gc;

   XSetLineAttributes(SndDpy, ctx->Sound.var1_gc, 1, LineOnOffDash, CapButt, JoinRound);
   XSetLineAttributes(SndDpy, ctx->Sound.var2_gc, 1, LineOnOffDash, CapButt, JoinRound);
   XSetLineAttributes(SndDpy, ctx->Sound.var3_gc, 1, LineOnOffDash, CapButt, JoinRound);
   XSetDashes(SndDpy, ctx->Sound.var1_gc, 1, dash_list[0], 2);
   XSetDashes(SndDpy, ctx->Sound.var2_gc, 5, dash_list[0], 2);
   XSetDashes(SndDpy, ctx->Sound.var3_gc, 9, dash_list[0], 2);
   counter = -1;
   if (var == ctx->Sound.SoundVar1){
       var_gc = ctx->Sound.var1_gc;
       step = ctx->Sound.var1step;
   }
   if (var == ctx->Sound.SoundVar2){
      var_gc = ctx->Sound.var2_gc;
      step = ctx->Sound.var2step;
   }
   if (var == ctx->Sound.SoundVar3){
      var_gc = ctx->Sound.var3_gc;
      step = ctx->Sound.var3step;
   } 
   if (ctx->Sound.samestepflag){
      for ( yo = ctx->MinVal[var]; yo < ctx->MaxVal[var]; yo += step){
         counter ++;
         if ( (ctx->Sound.mainvarstep * counter)+BORDER < ctx->Sound.sndwidth+BORDER){
            XDrawLine( SndDpy, ctx->Sound.soundpix, var_gc,
                    (ctx->Sound.mainvarstep * counter)+BORDER, ctx->Sound.sndheight+BORDER-HEBGBS,
                    (ctx->Sound.mainvarstep * counter)+BORDER, BORDER-HEBGBS);
         }
      }
   }
   else {
      for ( yo = ctx->MinVal[var]; yo < ctx->MaxVal[var]; yo += step){
         counter ++;
         if ( (ctx->Sound.mainvarstep * counter)+BORDER < ctx->Sound.sndwidth+BORDER){
            XDrawLine( SndDpy, ctx->Sound.soundpix, var_gc, 
                    (ctx->Sound.mainvarstep * counter)+BORDER, ctx->Sound.sndheight+BORDER-HEBGBS,
                    (ctx->Sound.mainvarstep * counter)+BORDER, BORDER-HEBGBS);
         }
      }
   }
   XSetLineAttributes(SndDpy, ctx->Sound.var1_gc, 2, LineSolid, CapButt, JoinRound);
   XSetLineAttributes(SndDpy, ctx->Sound.var2_gc, 2, LineSolid, CapButt, JoinRound);
   XSetLineAttributes(SndDpy, ctx->Sound.var3_gc, 2, LineSolid, CapButt, JoinRound);

}   

    /**********************************************************************/
    /* This just converts pressure to height using DEFAULT_LOG_EXP        */
    /* and DEFAULT_LOG_EXP                                                */
    /**********************************************************************/
    /* Input: ctx -context                                                */
    /*        pressure - the pressure to convert from                     */
    /**********************************************************************/
   
static float pres_to_height(float pressure)
{
   return  (DEFAULT_LOG_EXP * log( pressure / DEFAULT_LOG_SCALE));
}

    /**********************************************************************/
    /* This just converts height to pressure using DEFAULT_LOG_EXP        */
    /* and DEFAULT_LOG_EXP                                                */
    /**********************************************************************/
    /* Input: ctx -context                                                */
    /*        height - the height to convert from                         */
    /**********************************************************************/

static float height_to_pres(float height)
{
   return (DEFAULT_LOG_SCALE * exp( height / DEFAULT_LOG_EXP));
}


    /**********************************************************************/
    /* This draws the main border box surrounding the sounding data       */
    /**********************************************************************/
    /* Input: ctx -context                                                */
    /**********************************************************************/

static void draw_box ( Context ctx )
{ 
   XDrawLine ( SndDpy, ctx->Sound.soundpix, ctx->Sound.box_gc,
               BORDER, BORDER-5-HEBGBS, ctx->Sound.sndwidth+BORDER, BORDER-5-HEBGBS);

   XDrawLine ( SndDpy, ctx->Sound.soundpix, ctx->Sound.box_gc,
               ctx->Sound.sndwidth+BORDER, BORDER-5-HEBGBS, ctx->Sound.sndwidth+BORDER,
               ctx->Sound.sndheight+BORDER-HEBGBS);

   XDrawLine ( SndDpy, ctx->Sound.soundpix, ctx->Sound.box_gc,
               BORDER, BORDER-5-HEBGBS, BORDER, ctx->Sound.sndheight+BORDER-HEBGBS);
   if((ctx->Sound.vertsys == 0) || (ctx->BottomBound < -1.0)){
      XDrawLine ( SndDpy, ctx->Sound.soundpix, ctx->Sound.box_gc,
               BORDER, ctx->Sound.sndheight+BORDER-HEBGBS,
               ctx->Sound.sndwidth+BORDER, ctx->Sound.sndheight+BORDER-HEBGBS);
   }
}

    /**********************************************************************/
    /* This returns the saturation vapor pressure for a specified         */
    /* temperature in Kelvin                                              */
    /**********************************************************************/
    /* Input: K - temperature in Kelvin                                   */
    /* Output: svp - the saturation vapor pressure                        */
    /**********************************************************************/

static float svp( float K)
{
   float hit;

   hit = exp( (17.2693882*K - 4717.306) / (K - 35.7) );
   return  (6.1078 * hit);
}

    /**********************************************************************/
    /* This returns a mixing ration for a given temperature in Kelvin     */
    /* and a given pressure in millibars                                  */
    /**********************************************************************/
    /* Input: K - temperature in Kelvin                                   */
    /*        pres- pressure in millibars                                 */
    /* Output: mixratio - the mixing ratio                                */
    /**********************************************************************/

static float mixratio( float K, float pres )
{
   return (621.97 * svp(K) / ( pres - svp(K)));
}

    /**********************************************************************/
    /* This will give a thta-e at 1012.5 millibars for a given            */
    /* pressure in millibars and temperature in Kelvin                    */
    /**********************************************************************/
    /* Input: pres - pressure in millibars                                */
    /*        temp - temperature in Kelvin                                */
    /* Output: thetaE - theta-E in Kelvin                                 */
    /**********************************************************************/

static float thetaE( float pres, float temp)
{
   float thta, te;

   thta = temp * pow((1012.5/pres),.286);
   te = thta * exp(2.6518986* mixratio(temp, pres) / temp);
   return te;
}

    /**********************************************************************/
    /* This will approximate what the temperature(K)    for a given       */
    /* pressure (Millibars) and a given thetaE (K)                        */
    /**********************************************************************/
    /* Input: thte - Theta-E in Kelvin                                    */
    /*        pres - pressure in millibars                                */
    /* Output: get_temp_for_thte - the returned temperature in Kelving    */
    /**********************************************************************/

   
static float get_temp_for_thte(float thte, float pres)
{
   float tgnu= 293.16;
   float tgnup, tenu, tenup, cor;
   int x;

   for (x=1; x < 100; x++){
      tgnup = tgnu + 1.0;
      tenu = thetaE( pres, tgnu);
      tenup =thetaE( pres, tgnup);
      cor = (thte - tenu) / (tenup - tenu);
      tgnu = tgnu + cor;
      if (( cor < .01) && ((-1 * cor) < .01)){
         return tgnu;
      }
   }
   return 9999.9;
}

    /**********************************************************************/
    /* This draws the contstant mixing ratio lines                        */
    /**********************************************************************/
    /* Input ctx - context                                                */
    /**********************************************************************/

static void draw_wlines( Context ctx)
{
   GC w_gc = make_gc(ctx, 255, 255, 0, 0, 0, 0, 1);
   int wow, step, x, y, oldx, oldy, yo;
   float gotomintemp, p, wi;
   char shortwst[8], longwst[8];

   x = -9999;
   y = -9999;

   if ((ctx->Sound.sndwidth < 300) || (ctx->Sound.sndheight < 300)){
      step= 4;
   }
   else if ((ctx->Sound.sndwidth <550) || (ctx->Sound.sndheight < 550)){
      step= 3;
   }
   else {
      step= 1;
   }
   XSetLineAttributes(SndDpy, w_gc, 1, LineOnOffDash, CapRound, JoinRound);   
   for (yo = 0; yo < 34; yo += step ){
      wow = 1;
      for( gotomintemp = 373.0; gotomintemp > 173.0; gotomintemp -= .1){
         p = ((621.97 * svp(gotomintemp) + (winterval[yo] * svp(gotomintemp)))/winterval[yo]);
         if ((p <= 1012.5) && (wow == 1)){
            wow = 0;
            data_to_xy( ctx, pres_to_height(p), gotomintemp, &x, &y);
            oldx = x;
            oldy = y;
         }
         if ( p <= 200.0) {
            data_to_xy( ctx, pres_to_height(p), gotomintemp, &x, &y);
            gotomintemp = 100;
         }
      }
      cut_line_data2(ctx, &oldx, &oldy, &x, &y); 
      XDrawLine( SndDpy, ctx->Sound.soundpix, w_gc,
                       oldx+BORDER, oldy+BORDER-HEBGBS,
                       x+BORDER, y+BORDER-HEBGBS);
      w_gc = make_gc(ctx, 255, 255, 0, 0, 0, 0, 1);
      XDrawLine( SndDpy, ctx->Sound.soundpix, w_gc,
                       oldx+BORDER, oldy+BORDER-HEBGBS,
                       oldx+BORDER, oldy+BORDER+12-HEBGBS);
      if (XLoadFont(SndDpy, "6x12")){
            w_gc = make_gc(ctx, 255, 255, 0, 0, 0, 0, 2);
            XSetFont(SndDpy, w_gc, XLoadFont(SndDpy, "6x12"));
      }
      if(winterval[yo] < 3.0){
         sprintf(longwst, "%1.1f\n", winterval[yo]);
      XDrawString( SndDpy, ctx->Sound.soundpix, w_gc,
                   oldx+BORDER-7, oldy+BORDER+21-HEBGBS, longwst,3);
      }
      else if(winterval[yo] < 10.0){
         sprintf(longwst, "%d\n", (int) (winterval[yo]));
      XDrawString( SndDpy, ctx->Sound.soundpix, w_gc,
                   oldx+BORDER-2, oldy+BORDER+21-HEBGBS, longwst,1);
      }
      else{
         sprintf(shortwst, "%d\n", (int) (winterval[yo]));
         XDrawString( SndDpy, ctx->Sound.soundpix, w_gc,
                   oldx+BORDER-5, oldy+BORDER+21-HEBGBS, shortwst,2);
      }
      w_gc = make_gc(ctx, 255, 255, 0, 0, 0, 0, 1);
      XSetLineAttributes(SndDpy, w_gc, 1, LineOnOffDash, CapRound, JoinRound);

   }
}          

    /**********************************************************************/
    /* This draws the moist adiabat lines                                 */
    /**********************************************************************/
    /* Input ctx - context                                                */
    /**********************************************************************/

static void draw_thtelines( Context ctx )
{
   GC thte_gc = make_gc(ctx, 10, 140, 55, 0, 0, 0, 2);
   float p, step, step2;
   float yo, gotomaxheight;
   int x, y, oldx, oldy;
   

   step = .5;    
   if ((ctx->Sound.sndwidth < 300) || (ctx->Sound.sndheight < 300)){
      step2= 20;
   }
   else if ((ctx->Sound.sndwidth <550) || (ctx->Sound.sndheight < 550)){
      step2= 10;
   }
   else {
      step2= 5;
   }
   for (yo = ctx->Sound.SndMaxTemp-10; yo > ctx->Sound.SndMinTemp; yo -= step2){
      data_to_xy( ctx, 0, yo, &x, &y);
      for (gotomaxheight = 0.0; 
      gotomaxheight < pres_to_height(200); gotomaxheight += step){
         p = height_to_pres(gotomaxheight);
         oldx = x;
         oldy = y;
         data_to_xy( ctx, gotomaxheight,
                     get_temp_for_thte(thetaE(1012.5, yo), p), &x, &y);
         if ((x <= ctx->Sound.sndwidth) && (y <= ctx->Sound.sndheight)
         && (oldx <= ctx->Sound.sndwidth) && (oldy <= ctx->Sound.sndheight)
         && (x >=0) && (y >= 0) && (oldx >= 0) && (oldy >= 0)){
            XDrawLine( SndDpy, ctx->Sound.soundpix, thte_gc,
                       oldx+BORDER, oldy+BORDER-HEBGBS,
                        x+BORDER, y+BORDER-HEBGBS);
         }
         else if(((oldx >= 0  )&&(oldy >=0 ))&&((x <0) || (y<0))){
            cut_line_data(ctx, oldx, oldy, &x, &y);
            XDrawLine( SndDpy, ctx->Sound.soundpix, thte_gc,
                       oldx+BORDER, oldy+BORDER-HEBGBS,
                        x+BORDER, y+BORDER-HEBGBS);
         }
      }
   }
}

    /**********************************************************************/
    /* This draws the dry adiabat lines                                   */
    /**********************************************************************/
    /* Input ctx - context                                                */
    /**********************************************************************/
 
static void draw_thtalines( Context ctx )
{
   GC thta_gc = make_gc(ctx, 204, 50, 50, 0, 0, 0, 2);
   int yo, yo2, x, y, oldx, oldy, prex, prey;
   float mintemp = 123.0;
   float step = 10.0, step2 = 2.0, step3 = 5.0;
   float p,h;
   float gotomintemp;
   char thtastring [8];
   
   if ((ctx->Sound.sndwidth < 300) || (ctx->Sound.sndheight < 300)){
      step = 20;
      step2= 20;
   }
   else if ((ctx->Sound.sndwidth <550) || (ctx->Sound.sndheight < 550)){
      step = 10;
      step2= 10;
   }
   else {
      step = 10;
      step2= 2;
   }
    
   /* first draw the theta lines along the x axis */
   for (yo = ctx->Sound.SndMaxTemp; yo > ctx->Sound.SndMinTemp; yo -= step2){
      data_to_xy( ctx, 0, yo, &x, &y ); 
      thta_gc = make_gc(ctx, 255, 105, 105, 0, 0, 0, 2);
      if( (((yo - 3) % 10) == 0 ) && yo != ctx->Sound.SndMaxTemp){
         XSetForeground(SndDpy, thta_gc, SND_AllocateColorInt(0,0,0));
         XFillRectangle(SndDpy, ctx->Sound.soundpix, thta_gc,
                        x+BORDER-5, y+BORDER, 22, 12);
         thta_gc = make_gc(ctx, 255, 105, 105, 0, 0, 0, 2);
         if (XLoadFont(SndDpy, "6x12")){
            XSetFont(SndDpy, thta_gc, XLoadFont(SndDpy, "6x12"));
         }
         sprintf(thtastring, "%d\n", yo);
         XDrawString( SndDpy, ctx->Sound.soundpix, thta_gc,
                   x+BORDER-5, y+BORDER+10-HEBGBS, thtastring, 3);
      }
      if ( ((yo - 3) % 10) == 0 ){
         thta_gc = make_gc(ctx, 204, 50, 50, 0, 0, 0, 2);
      }
      else {
         thta_gc = make_gc(ctx, 150, 10, 10, 0, 0, 0, 1);
      }   
      if( (((yo - 3) % 10) == 0 ) && yo == ctx->Sound.SndMaxTemp){
         if (XLoadFont(SndDpy, "6x12")){
            thta_gc = make_gc(ctx, 255, 105, 105, 0, 0, 0, 2);
            XSetFont(SndDpy, thta_gc, XLoadFont(SndDpy, "6x12"));
         }

         sprintf(thtastring, "%d\n", yo);
         XDrawString( SndDpy, ctx->Sound.soundpix, thta_gc,
                   x+BORDER+4, y+BORDER+10-HEBGBS, thtastring, 3);
         thta_gc = make_gc(ctx, 204, 50, 50, 0, 0, 0, 2);
      }
      for (gotomintemp = yo; gotomintemp >  mintemp; gotomintemp -= step3){
         p =  exp((1/.286)*(log((7.23674 * gotomintemp)/yo)));
         h = pres_to_height( p );
         oldx = x;
         oldy = y;
         data_to_xy(ctx, h, gotomintemp,&x,&y);
         if ((x <= ctx->Sound.sndwidth) && (y <= ctx->Sound.sndheight)
         && (oldx <= ctx->Sound.sndwidth) && (oldy <= ctx->Sound.sndheight)
         && (x >=0) && (y >= 0) && (oldx >= 0) && (oldy >= 0)){
            XDrawLine( SndDpy, ctx->Sound.soundpix, thta_gc,
                       oldx+BORDER, oldy+BORDER-HEBGBS,
                       x+BORDER, y+BORDER-HEBGBS);
         }
         else if(((oldx >= 0  )&&(oldy >=0 ))&&((x <0) || (y<0))){
            cut_line_data(ctx, oldx, oldy, &x, &y);
            XDrawLine( SndDpy, ctx->Sound.soundpix, thta_gc,
                       oldx+BORDER, oldy+BORDER-HEBGBS,
                       x+BORDER, y+BORDER-HEBGBS);
            gotomintemp = mintemp-1; 
         } 
      }
   }
   /* now draw the thteta line up the right y axis */
  for (yo = 483; yo > ctx->Sound.SndMaxTemp; yo -= step2){
      data_to_xy( ctx, h, ctx->Sound.SndMaxTemp , &x, &y);
      if ( ((yo - 3) % 10) == 0 ){
         thta_gc = make_gc(ctx, 204, 50, 50, 0, 0, 0, 2);
      }
      else {
         thta_gc = make_gc(ctx, 150, 10, 10, 0, 0, 0, 1); 
      }
      for (gotomintemp = yo; gotomintemp >  mintemp; gotomintemp -= step3){
         p =  exp((1/.286)*(log((7.23674 * gotomintemp)/yo)));
         h = pres_to_height( p );
         oldx = x;
         oldy = y;
         data_to_xy(ctx, h, gotomintemp,&x,&y);
         prex = oldx;
         prey = oldy;
         if(((oldx>ctx->Sound.sndwidth) || ( oldy > ctx->Sound.sndheight)) &&
            (x <= ctx->Sound.sndwidth) && (y <= ctx->Sound.sndheight) &&
            (x >= 0 ) && (y >= 0 )){
            precut_line_data( ctx, &oldx, &oldy, x, y);
            XDrawLine( SndDpy, ctx->Sound.soundpix, thta_gc,
                       oldx+BORDER, oldy+BORDER-HEBGBS,
                       x+BORDER, y+BORDER-HEBGBS);
            if(((yo - 3) % 10) == 0 ){
               thta_gc = make_gc(ctx, 255, 105, 105, 0, 0, 0, 2);
               if (XLoadFont(SndDpy, "6x12")){
                  XSetFont(SndDpy, thta_gc, XLoadFont(SndDpy, "6x12"));
               }
               sprintf(thtastring, "%d", yo);
               XDrawString( SndDpy, ctx->Sound.soundpix, thta_gc,
                         oldx+BORDER+4, oldy+BORDER+7-HEBGBS, thtastring, 3);
               thta_gc = make_gc(ctx, 204, 50, 50, 0, 0, 0, 2);
            }
         }
         if ((x <= ctx->Sound.sndwidth) && (y <= ctx->Sound.sndheight)
         && (oldx <= ctx->Sound.sndwidth) && (oldy <= ctx->Sound.sndheight)
         && (x >=0) && (y >= 0) && (oldx >= 0) && (oldy >= 0)){
            XDrawLine( SndDpy, ctx->Sound.soundpix, thta_gc,
                       oldx+BORDER, oldy+BORDER-HEBGBS,
                       x+BORDER, y+BORDER-HEBGBS);
         }
         else if(yo!= 483 && ((oldx > 0  )&&(oldy >0 ))&&((x <0) || (y<0))){
            cut_line_data(ctx, oldx, oldy, &x, &y);
            XDrawLine( SndDpy, ctx->Sound.soundpix, thta_gc,
                       oldx+BORDER, oldy+BORDER-HEBGBS,
                       x+BORDER, y+BORDER-HEBGBS);
            gotomintemp = mintemp-1;
         }
      }
   }
}

    /**********************************************************************/
    /* This will take two points, check if one of them is outside the data*/
    /* boundry then chop it off, if it is.  It also chops of lines at the */
    /* 200mb level. And if points x1 and y1 are not even in the data      */
    /* boundry then the line will not be drawn                            */
    /**********************************************************************/
    /* Input ctx - context                                                */
    /*       x1, y1, x2, y2 - points making the line to be chopped        */
    /**********************************************************************/

static void cut_line_data2( Context ctx, int *x1, int *y1, int *x2, int *y2)
{
   float m,b;
   float xone, yone, xtwo, ytwo;
   float x_intercept, y_intercept;
   int p_equals_200mb;
   float bottomline;

   xone = (float) (*x1);
   yone = (float) (*y1);
   xtwo = (float) (*x2);
   ytwo = (float) (*y2);
   p_equals_200mb = *y2;

   /* get equation of line for x1, y1, x2, y2 */
   if ((xone - xtwo) == 0 ){
      m = 0.0;
      b = *y1;
   }
   else {
      m = (yone - ytwo)/(xone - xtwo);
      b = yone - (m * xone);
   }
   bottomline = ((ctx->TopBound-0.0001)*((int) (ctx->Sound.sndheight))/
                 (ctx->TopBound-ctx->BottomBound+0.0001));

   /* see where this line intersects the 1012.5 mb line */
   *x1 = (int) ((bottomline - b)/m);
   *y1 = (int) (bottomline);

   /* see where the line intersects the right line */
   *x2 = ctx->Sound.sndwidth;
   *y2 = (int) ((float) ctx->Sound.sndwidth * m + b);

   /* if it intersects the top line, the chop it off */
   if ((*y2) < p_equals_200mb){
      *x2 = (int) ((p_equals_200mb - b) / m );
      *y2 = p_equals_200mb;
   }
   if ((*y2) < 0){
      *x2 = (int) (-b / m);
      *y2 = 0;
   }
}   

    /**********************************************************************/
    /* When figureing out the starting point for the theta lines in the + */
    /* y direction on the right side of the data boundry this will chop   */
    /* off the given line so it starts at the right side given by y =     */
    /* ctx->Sound.sndwidth                                                */
    /**********************************************************************/
    /* Input ctx - context                                                */
    /*       x1, y1, x2, y2 - points making the line to be chopped        */
    /* Output: x1, y1 - these may be altered in order to start at boudry  */
    /**********************************************************************/
 
static void precut_line_data (Context ctx, int *x1, int *y1, int x2, int y2)
{
   float m,b;
   float line_intercept;
   float  xone, yone, xtwo, ytwo;

   /* get equation of line for x1, y1, x2, y2 */
   xone = (float) *x1;
   yone = (float) *y1;
   xtwo = (float) (x2);
   ytwo = (float) (y2);
   if ((xone - xtwo) == 0 ){
      m = 0.0;
      b = yone;
   }
   else {
      m = (yone - ytwo)/(xone - xtwo);
      b = yone - (m * xone);
   }
   
   /* see where on the line x= sndwidth it intersecpts */
   line_intercept = m * ctx->Sound.sndwidth + b;

   *x1 = ctx->Sound.sndwidth;
   *y1 = (int) (line_intercept); 
}



    /**********************************************************************/
    /* This cuts the given line so that it's end point will be on the     */
    /* left or top side of the data boundry.  It is used in drawing the   */
    /* theta lines.                                                       */
    /**********************************************************************/
    /* Input ctx - context                                                */
    /*       x1, y1, x2, y2 - points making the line to be chopped        */
    /* Output x2, y2 - these points may be altered to end at data boudry  */
    /**********************************************************************/

        
static void cut_line_data( Context ctx, int x1, int y1, int *x2, int *y2)
{
   float m,b;
   float x_intercept, y_intercept;
   float  xone, yone, xtwo, ytwo;
 
   /* get equation of line for x1, y1, x2, y2 */
   xone = (float) x1;
   yone = (float) y1;
   xtwo = (float) (*x2);
   ytwo = (float) (*y2);
   if ((x1 - xtwo) == 0 ){
      m = 0.0;
      b = yone;
   }
   else { 
      m = (yone - ytwo)/(xone - xtwo);
      b = yone - (m * xone);
   }

   /* see where this line intersects the axies */
   y_intercept = b;
   x_intercept = (-1.0 * b)/m;

   if (y_intercept == 0 ){
      *x2 = 0;
      *y2 = 0;
   }
   else if ( y_intercept < 0 ){
      *x2 = (int) (x_intercept);
      *y2 = 0;
   }
   else if ( x_intercept < 0){
      *x2 = 0;
      *y2 = (int) (y_intercept);
   }
}

    /**********************************************************************/
    /* This draws the numbers and ticks up the vertical                   */
    /**********************************************************************/
    /* Input ctx - context                                                */
    /**********************************************************************/

static void draw_vert_stuff( Context ctx )
{
   int yo, y;
   int maxlev;
   int dash_offset;
   int dash_list_length = { TICK_DASH_LENGTH };
   static  char dotted[2] = {4, 12};
   char *dash_list[] = {dotted};
   char num[10];
   int stringnumber;
   double average;

   if (!ctx->Sound.vert_gc){
      ctx->Sound.vert_gc = make_gc(ctx, 255, 255, 255, 0, 0, 0, 2);
   }
   maxlev = (int) (ctx->Sound.vertdata[0]);
   {
      double total=0.0;;
      for (yo=1; yo < maxlev + 1; yo++){
         total += ctx->Sound.vertdata[yo];
      }
      average = total / (double)(maxlev);
   }

   for ( yo=1; yo < maxlev + 1; yo++){
      ctx->Sound.vert_gc = make_gc(ctx, 100, 100, 100, 0, 0, 0, 1);
      XSetLineAttributes(SndDpy, ctx->Sound.vert_gc, 0, LineOnOffDash, CapButt, JoinRound);
      XSetDashes(SndDpy, ctx->Sound.vert_gc, 1, dash_list[0], 2);
      data_to_y( ctx, ctx->Sound.vertdata[yo], &y);
      if(ctx->Sound.tickstatus){ 
         XDrawLine( SndDpy, ctx->Sound.soundpix, ctx->Sound.vert_gc, BORDER, y + BORDER - HEBGBS,
                    ctx->Sound.sndwidth+BORDER, y + BORDER - HEBGBS);
      }
      if (average < .1){
         float numb;
         numb = ctx->Sound.vertdata[yo] * 1000.0;
         sprintf(num, "%.2f\n", numb);
      }
      else{
         sprintf(num, "%.1f\n", ctx->Sound.vertdata[yo]);
      }
      stringnumber = strlen(num)-1;
      ctx->Sound.vert_gc = make_gc(ctx, 255, 255, 255, 0, 0, 0, 1);
      if (XLoadFont(SndDpy, "6x12")){
         XSetFont(SndDpy, ctx->Sound.vert_gc, XLoadFont(SndDpy, "6x12"));
      }
      XDrawString( SndDpy, ctx->Sound.soundpix, ctx->Sound.vert_gc,
                   ctx->Sound.sndwidth + BORDER + 25, y + BORDER - HEBGBS + 4,
                   num, stringnumber);
  }
   if (average < .1){
      sprintf(num, "m");
   }
   else{
      sprintf(num, "Km");
   }
  if( ctx->Sound.vertsys != 0 )
     XDrawString( SndDpy, ctx->Sound.soundpix, ctx->Sound.vert_gc,
                ctx->Sound.sndwidth + BORDER + 25+5,  BORDER - HEBGBS -8,
                num, 2);
}     






    /**********************************************************************/
    /* This just draws the millibars.                                     */
    /**********************************************************************/
    /* Input ctx - context                                                */
    /**********************************************************************/
 
static void draw_millibarlines ( Context ctx )
{
   GC millibars_gc = make_gc(ctx, 110, 110, 110, 0, 0, 0, 1);
   float millibars, pressure;
   int yo, x, y, increment, number;
   float temp;
   char nums[8]; 

   increment = 2;
   pressure = 1100.0; 
   for (yo=0; yo < 10; yo++ ){
      pressure -= 50.0 * (float) increment; 
      millibars = pres_to_height( pressure );
      number = pressure;
      sprintf(nums,"%d\n", number );
      data_to_xy( ctx, millibars, 266, &x, &y); 
      if ( y > 10 ) {
         millibars_gc = make_gc(ctx, 140, 140, 140, 0, 0, 0, 1);
         if (pressure == 1000.0){
            XDrawString( SndDpy, ctx->Sound.soundpix, millibars_gc,
                      (ctx->Sound.sndwidth/2-23)+BORDER, y+2+BORDER-HEBGBS, nums, 4);
         }
         else { 
            XDrawString( SndDpy, ctx->Sound.soundpix, millibars_gc,
                      (ctx->Sound.sndwidth/2-19)+BORDER, y+2+BORDER-HEBGBS, nums, 3);
         }
         millibars_gc = make_gc(ctx, 110, 110, 110, 0, 0, 0, 1);
         XDrawLine ( SndDpy, ctx->Sound.soundpix, millibars_gc, BORDER, y+BORDER-HEBGBS,
                     (ctx->Sound.sndwidth/2-24)+BORDER, y+BORDER-HEBGBS);
   
         XDrawLine ( SndDpy, ctx->Sound.soundpix, millibars_gc,
                     (ctx->Sound.sndwidth/2+24)+BORDER , y+BORDER-HEBGBS,
                      ctx->Sound.sndwidth+BORDER, y+BORDER-HEBGBS); 
      }
   }
}
    /**********************************************************************/
    /* This will draw a barb onto the soundwin                            */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        spd - this is the given speed in meters/second              */
    /*        dir - this is the direction in radians                      */
    /*        height - this is the height at which the barb info is given */
    /**********************************************************************/
 
static void make_a_barb( Context ctx, float spd, float dir, float height)
{
   int location;
   int mainstemlength = 48;
   int fiftycount = 0;
   int tencount = 0;
   int fivecount = 0;
   int calmcount = 0;
   int speed;
   int yo;
   int space = 6;
   int bigx  = 10;
   int x,y;
   XPoint  flagpoints[3];

   data_to_xy( ctx, height, 273, &x, &y);
   /* get the count of how many and what kind of flags */
   spd = spd / .51282;
   speed = (int) spd;
   fiftycount = speed / 50;
   speed -= fiftycount * 50;
   tencount   = speed / 10;
   speed -= tencount * 10;
   fivecount  = speed / 5;
   if ( fiftycount == 0 && tencount == 0 && fivecount == 0 ){
      calmcount = 1;
   }

   /* now contruct a barb giving the x and y and draw that line */
   location = mainstemlength;

   /*draw main stem line */
   if (!calmcount) drawbarbline(ctx, 0, 0, 0, location, y, dir);

   /*draw any 50's if there is any*/
   for (yo = 0; yo < fiftycount; yo++){
      drawbarbline(ctx, 0, location, bigx, location, y, dir);
      location -= space;
      drawbarbline(ctx, 0, location, bigx, location + space, y, dir);
      flagpoints[0].x = 0;
      flagpoints[0].y = location + space;
      flagpoints[1].x = bigx;
      flagpoints[1].y = location + space;
      flagpoints[2].x = 0;
      flagpoints[2].y = location;
      drawbarbflag( ctx, flagpoints, y, dir);
}

   /*draw any 10's if there are any*/
   for (yo = 0; yo < tencount; yo ++){
      if ( yo != 0 ) location -= space;
      drawbarbline(ctx, 0, location, bigx, location+space, y, dir);
   }

   /*draw any 5's if there are any */
   for (yo = 0; yo < fivecount; yo++){
      location -= space;
      drawbarbline(ctx, 0, location, bigx/2, location + space/2, y, dir);
   }

   /*draw the calm little x if it is calm out*/
   if (calmcount == 1){
      drawbarbline(ctx,-4, 0, 4, 0, y, 0);
      drawbarbline(ctx, 0, -4, 0, 4,y, 0);
   }
}

    /**********************************************************************/
    /* This will draw and color in any barb flags                         */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        flagpoints - the three points making up the barb flag       */
    /*        up - the lenght from the center of barb to flag             */
    /*        dir - the number or radians from 0 degrees to turn and draw */
    /*              the barb flag                                         */
    /**********************************************************************/

static void drawbarbflag( Context ctx, XPoint flagpoints[], int up, float dir)
{
   int x, y, x1, x2, x3, y1, y2, y3, xx1, yy1, xx2, xx3, yy2, yy3;;

   x = 0;
   y = up;
   x1 = flagpoints[0].x;
   y1 = flagpoints[0].y;
   x2 = flagpoints[1].x;
   y2 = flagpoints[1].y;
   x3 = flagpoints[2].x;
   y3 = flagpoints[2].y;

   convert_xy_to_barb( ctx, x1, y1, dir, &xx1, &yy1);
   convert_xy_to_barb( ctx, x2, y2, dir, &xx2, &yy2);
   convert_xy_to_barb( ctx, x3, y3, dir, &xx3, &yy3);
   flagpoints[0].x = xx1 + x + BORDER;
   flagpoints[0].y = yy1 + y + BORDER-HEBGBS;
   flagpoints[1].x = xx2 + x + BORDER;
   flagpoints[1].y = yy2 + y + BORDER-HEBGBS;
   flagpoints[2].x = xx3 + x + BORDER;
   flagpoints[2].y = yy3 + y + BORDER-HEBGBS;

   XFillPolygon( SndDpy, ctx->Sound.soundwin, ctx->Sound.barb2_gc,
                 flagpoints, 3, Nonconvex, CoordModeOrigin);
}

    /**********************************************************************/
    /* This will take two points,the radius, and direction and draw a line*/
    /**********************************************************************/
    /* Input ctx - context                                                */
    /*       x1, y1, x2, y2 - points for the line to be drawn             */
    /*       up - radius                                                  */
    /*       dir - degrees from 0 degree to turn                          */
    /**********************************************************************/


static void drawbarbline( Context ctx, int x1, int y1, int x2, int y2,
                           int up, float dir)
{
   int x, y, xx1, yy1, xx2, yy2;

   x = 0;
   y = up;
   convert_xy_to_barb( ctx, x1, y1, dir, &xx1, &yy1);
   convert_xy_to_barb( ctx, x2, y2, dir, &xx2, &yy2);
   XDrawLine( SndDpy, ctx->Sound.soundwin, ctx->Sound.barb_gc, xx1 + x + BORDER,
              yy1 + y + BORDER-HEBGBS, xx2 + x + BORDER,
              yy2 + y + BORDER-HEBGBS);
}

    /**********************************************************************/
    /* This will take a point, convert it to polar coordinates, turn it   */
    /* the given direction, then convert it back to catesian coordinates  */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        inx, iny - the x and y fot the point to be transformed      */
    /*        dir - the number of radians to turn the point to the right  */
    /*        outx, outy - the x and y after transformation               */
    /**********************************************************************/


static void convert_xy_to_barb(Context ctx, int inx, int iny, float dir,
                                          int *outx, int *outy)
{
   float x, y, r, angle;

   x = (float) inx;
   y = (float) iny;

   /* first convert to polar coordinates */
   r = sqrt((x * x) + (y * y));
   if ( x > 0 && y < 0 )
      angle = PI + atan(x / y);
   else if ( x < 0 && y < 0)
      angle = PI + atan(x / y);
   else if (y == 0){
      if ( x < 0 )
         angle = 3 * PI / 2;
      else
         angle = PI / 2;
   }
   else if (x == 0){
      if ( y < 0 )
         angle = PI;
      else
         angle = 0;
   }
   else
      angle = atan(x / y);

   /* now just add on the angle */
   angle +=  (float) dir;

   /* now convert back to x and y */
   *outx = (int) (cos(angle) * r);
   *outy = (int) (sin(angle) * r);
}    

    /**********************************************************************/
    /* This reload the grids for the sounding variables in case the       */
    /* variables change or the time changes this can be used to change them*/
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /**********************************************************************/


int reload_sounding_data( Context ctx )
{
   if ( ctx->Sound.SoundTemp >= 0 ) {
      if (ctx->Sound.tgrid == NULL ) {
         ctx->Sound.tgrid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundTemp);
      }
      else {
         release_grid( ctx, ctx->CurTime, ctx->Sound.SoundTemp, ctx->Sound.tgrid );
         ctx->Sound.tgrid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundTemp);
      }
   }

   if ( ctx->Sound.SoundDewpt >= 0){
      if (ctx->Sound.dgrid == NULL ) {
         ctx->Sound.dgrid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundDewpt);
      }
      else {
         release_grid( ctx, ctx->CurTime, ctx->Sound.SoundDewpt, ctx->Sound.dgrid );
         ctx->Sound.dgrid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundDewpt);
      }
   }


   if ( ctx->Sound.SoundUWind >= 0){
      if (ctx->Sound.ugrid == NULL ) {
         ctx->Sound.ugrid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundUWind);
      }
      else {
         release_grid( ctx, ctx->CurTime, ctx->Sound.SoundUWind, ctx->Sound.ugrid );
         ctx->Sound.ugrid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundUWind);
      }
   }


   if ( ctx->Sound.SoundVWind >= 0){
      if (ctx->Sound.vgrid == NULL ) {
         ctx->Sound.vgrid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundVWind);
      }
      else {
         release_grid( ctx, ctx->CurTime, ctx->Sound.SoundVWind, ctx->Sound.vgrid );
         ctx->Sound.vgrid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundVWind);
      }
   }


   if ( ctx->Sound.SoundVar1 >= 0){
      if (ctx->Sound.var1grid == NULL ) {
         ctx->Sound.var1grid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundVar1);
      }
      else {
         release_grid( ctx, ctx->CurTime, ctx->Sound.SoundVar1, ctx->Sound.var1grid );
         ctx->Sound.var1grid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundVar1);
      }
   }


   if ( ctx->Sound.SoundVar2 >= 0){
      if (ctx->Sound.var2grid == NULL ) {
         ctx->Sound.var2grid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundVar2);
      }
      else {
         release_grid( ctx, ctx->CurTime, ctx->Sound.SoundVar2, ctx->Sound.var2grid );
         ctx->Sound.var2grid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundVar2);
      }
   }


   if ( ctx->Sound.SoundVar3 >= 0){
      if (ctx->Sound.var3grid == NULL ) {
         ctx->Sound.var3grid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundVar3);
      }
      else {
         release_grid( ctx, ctx->CurTime, ctx->Sound.SoundVar3, ctx->Sound.var3grid );
         ctx->Sound.var3grid = get_grid( ctx, ctx->CurTime, ctx->Sound.SoundVar3);
      }
   }
   return 1;
}   






    /**********************************************************************/
    /* This function puts everything together and draws the soundings and */
    /* vertical plot lines for a given time, and cursor location which    */
    /* is found in the context                                            */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        time - time step for whcih the sounding line is drawn       */
    /* Output: returns 1 is all goes well                                 */
    /**********************************************************************/



int draw_sounding( Context ctx, int time )
{
   float windspeed, winddir;
   float row, col, lev;
   float alt, temp, value;
   float frow, fcol;
   int oldx = 9999;
   int oldy = 9999;
   int x = 9999;
   int y = 9999;
   int yo, yo2;
   int icol, irow;
   XGCValues values;
   float spd;
   Font varfont;


   /* this will get the heights for the levels*/
   /* and put them in ctx->Sound.vertdata, and the first number in this array */
   /* tells how many levels there are */

   if (ctx->Sound.get_vert_data){
      ctx->Sound.get_vert_data = 0;
      ctx->Sound.vertdata = (float *) allocate_type(ctx, ctx->MaxNl * sizeof(float) + 1 , VERTDATA_TYPE );
      if (!ctx->Sound.vertdata) {
         return NULL;
      }
      ctx->Sound.vertdata[0] = (float) (ctx->MaxNl); 
      for ( yo = 1; yo < ctx->MaxNl + 1; yo++){
         ctx->Sound.vertdata[yo] = grid_level_to_height( ctx, yo-1);
      }
   }
   XSync( SndDpy, 0);
   XCopyArea(SndDpy, ctx->Sound.soundpix, ctx->Sound.soundwin, ctx->Sound.box_gc,
             0, 0, ctx->Sound.sndwidth+2*BORDER,
             ctx->Sound.sndheight+95+2*BORDER+(BORDER/2), 0, 0);

   /*                               */
   /* draw the data into the window */
   /*                               */
   xyz_to_grid( ctx, ctx->CurTime, 1,
                ctx->CursorX, ctx->CursorY, ctx->CursorZ, &row, &col, &lev );
   irow = (int)(row + 0.5);
   icol = (int)(col + 0.5);
   frow = row - (float)irow;
   fcol = col - (float)icol;

   if ( fabs(frow) < 0.001){
      row = (float)(irow);
   }
   if ( fabs(fcol) < 0.001){
      col = (float)(icol);
   }
 

   /* draw the temperature first if it is given*/
   if (ctx->Sound.SoundTemp >= 0){
      if ( (ctx->Sound.SoundTemp != ctx->Sound.PreviousSoundTemp) &&
           ctx->Sound.tgrid != NULL ){
         release_grid( ctx, time, ctx->Sound.PreviousSoundTemp, ctx->Sound.tgrid );
         ctx->Sound.tgrid = get_grid( ctx, time, ctx->Sound.SoundTemp);
      }      
      ctx->Sound.PreviousSoundTemp = ctx->Sound.SoundTemp;
      if (ctx->Sound.tgrid == NULL ) {
         ctx->Sound.tgrid = get_grid( ctx, time, ctx->Sound.SoundTemp);
      }
      if (!ctx->Sound.tgrid){
         return NULL;
      }
      yo =  extract_sound( ctx, ctx->Sound.tgrid, ctx->Sound.SoundTemp, ctx->Nr, ctx->Nc,
                           ctx->Nl[ctx->Sound.SoundTemp],
                           ctx->LowLev[ctx->Sound.SoundTemp], row, col );
      XSetLineAttributes(SndDpy, ctx->Sound.Tempgc, 2, LineSolid, CapRound, JoinRound);
      for (yo=0; yo < ctx->Nl[ctx->Sound.SoundTemp]; yo++){
         alt = grid_level_to_height( ctx, yo);
         temp = ctx->Sound.soundline[yo];
         oldx = x;
         oldy = y;
         data_to_xy(ctx, alt, temp, &x, &y);
         if ( alt < -0.2 ) {
            x = -9999;
         }
         if ((x <= ctx->Sound.sndwidth) && (y <= ctx->Sound.sndheight)
            && (oldx <= ctx->Sound.sndwidth) && (oldy <= ctx->Sound.sndheight)
            && (x >= 0) && (y >= 0) && (oldx >= 0) && (oldy >= 0)){
            XDrawLine(SndDpy, ctx->Sound.soundwin, ctx->Sound.Tempgc, oldx+BORDER,
                      oldy+BORDER-HEBGBS, x +BORDER, y +BORDER-HEBGBS);
         }
      }
   }
   else {
      if (ctx->Sound.tgrid != NULL ){
         release_grid( ctx, time, ctx->Sound.PreviousSoundTemp, ctx->Sound.tgrid );
         ctx->Sound.tgrid = NULL;
      }
   }

   /* now draw the dewpoint line if its there */

   x = -1;
   y = -1;
   oldx = -1;
   oldy = -1;
   if (ctx->Sound.SoundDewpt >= 0){
      if ( (ctx->Sound.SoundDewpt != ctx->Sound.PreviousSoundDewpt) &&
           ctx->Sound.dgrid != NULL ){
         release_grid( ctx, time, ctx->Sound.PreviousSoundDewpt, ctx->Sound.dgrid ); 
         ctx->Sound.dgrid = get_grid( ctx, time, ctx->Sound.SoundDewpt);
      }
      ctx->Sound.PreviousSoundDewpt = ctx->Sound.SoundDewpt;

      if (ctx->Sound.dgrid == NULL ) {
         ctx->Sound.dgrid = get_grid( ctx, time, ctx->Sound.SoundDewpt);
      }
      if (!ctx->Sound.dgrid){
         return NULL;
      }
      yo =  extract_sound( ctx, ctx->Sound.dgrid, ctx->Sound.SoundDewpt, ctx->Nr, ctx->Nc,
                           ctx->Nl[ctx->Sound.SoundDewpt],
                           ctx->LowLev[ctx->Sound.SoundDewpt], row, col );
      XSetLineAttributes(SndDpy, ctx->Sound.Dewptgc, 2, LineSolid, CapRound, JoinRound);
      for (yo=0; yo < ctx->Nl[ctx->Sound.SoundDewpt]; yo++){
         alt = grid_level_to_height( ctx, yo);
         temp = ctx->Sound.soundline[yo];
         oldx = x;
         oldy = y;
         data_to_xy(ctx, alt, temp, &x, &y);
         if ( alt < -0.2 ) {
            x = -9999;
         }
         if ((x <= ctx->Sound.sndwidth) && (y <= ctx->Sound.sndheight)
            && (oldx <= ctx->Sound.sndwidth) && (oldy <= ctx->Sound.sndheight)
            && (x >= 0) && (y >= 0) && (oldx >= 0) && (oldy >= 0)){
            XDrawLine(SndDpy, ctx->Sound.soundwin, ctx->Sound.Dewptgc, oldx+BORDER,
                      oldy+BORDER-HEBGBS, x +BORDER, y +BORDER-HEBGBS);
         }
      }
   }
   else {
      if (ctx->Sound.dgrid != NULL ){
         release_grid( ctx, time, ctx->Sound.PreviousSoundDewpt, ctx->Sound.dgrid );
         ctx->Sound.dgrid = NULL;
      }
   }

   /* now draw all the wind barb stuff */
  
   if ((ctx->Sound.SoundUWind >= 0) && (ctx->Sound.SoundVWind >= 0 )){

      if ( (ctx->Sound.SoundUWind != ctx->Sound.PreviousSoundUWind) &&
           ctx->Sound.ugrid != NULL ){
         release_grid( ctx, time, ctx->Sound.PreviousSoundUWind, ctx->Sound.ugrid );
         ctx->Sound.ugrid = get_grid( ctx, time, ctx->Sound.SoundUWind);
      }

      if ( (ctx->Sound.SoundVWind != ctx->Sound.PreviousSoundVWind) &&
           ctx->Sound.vgrid != NULL ){
         release_grid( ctx, time, ctx->Sound.PreviousSoundVWind, ctx->Sound.vgrid );
         ctx->Sound.vgrid = get_grid( ctx, time, ctx->Sound.SoundVWind);
      }
      ctx->Sound.PreviousSoundVWind = ctx->Sound.SoundVWind;
      ctx->Sound.PreviousSoundUWind = ctx->Sound.SoundUWind;

      if( ctx->Sound.ugrid == NULL ) {
         ctx->Sound.ugrid = get_grid( ctx, time, ctx->Sound.SoundUWind);
      }
      if( ctx->Sound.vgrid == NULL ) {
         ctx->Sound.vgrid = get_grid( ctx, time, ctx->Sound.SoundVWind);
      }
      if ((!ctx->Sound.ugrid) || (!ctx->Sound.vgrid)){
         return NULL;
      }
      yo = extract_wind( ctx, ctx->Sound.ugrid, ctx->Sound.vgrid, ctx->Sound.SoundUWind,
                         ctx->Sound.SoundVWind, ctx->Nr, ctx->Nc,
                         ctx->Nl[ctx->Sound.SoundUWind],
                         ctx->LowLev[ctx->Sound.SoundUWind], row, col );
      for (yo=0; yo < ctx->Nl[ctx->Sound.SoundUWind]; yo++){
         alt = grid_level_to_height( ctx, yo);
         windspeed = sqrt((ctx->Sound.uwindline[yo] * ctx->Sound.uwindline[yo]) +
                          (ctx->Sound.vwindline[yo] * ctx->Sound.vwindline[yo]));
         if (ctx->Sound.uwindline[yo] > 0 && ctx->Sound.vwindline[yo] < 0 )
            winddir = PI + atan(ctx->Sound.uwindline[yo]/ctx->Sound.vwindline[yo]);
         else if (ctx->Sound.uwindline[yo] < 0 && ctx->Sound.vwindline[yo] < 0)
            winddir = PI + atan(ctx->Sound.uwindline[yo]/ctx->Sound.vwindline[yo]);
         else if (ctx->Sound.uwindline[yo] > 0 && ctx->Sound.vwindline[yo] > 0)
            winddir = atan(ctx->Sound.uwindline[yo]/ctx->Sound.vwindline[yo]);
         else if (ctx->Sound.uwindline[yo] < 0 && ctx->Sound.vwindline[yo] > 0)
            winddir = atan(ctx->Sound.uwindline[yo]/ctx->Sound.vwindline[yo]);
         else if ( ctx->Sound.uwindline[yo] == 0 && ctx->Sound.vwindline[yo] > 0)
            winddir = 0;
         else if ( ctx->Sound.uwindline[yo] == 0 && ctx->Sound.vwindline[yo] < 0)
            winddir = PI;
         else if (ctx->Sound.uwindline[yo] > 0 && ctx->Sound.vwindline[yo] == 0)
            winddir = PI / 2;
         else if (ctx->Sound.uwindline[yo] < 0 && ctx->Sound.vwindline[yo] == 0)
            winddir = 3 * PI / 2;
         if(windspeed < 300.0){
            make_a_barb ( ctx, windspeed, winddir+PI/2, alt );
         }
      }
   }
   else {
      if (ctx->Sound.ugrid != NULL ) {
         release_grid( ctx, time, ctx->Sound.PreviousSoundUWind, ctx->Sound.ugrid);
         ctx->Sound.ugrid = NULL;
      }
      if (ctx->Sound.vgrid != NULL ) {
         release_grid( ctx, time, ctx->Sound.PreviousSoundVWind, ctx->Sound.vgrid);
         ctx->Sound.vgrid = NULL;
      }
   }
  
   /* now draw the different vertical variables */
   oldx = -1;
   oldy = -1;
   x    = -1;
   y    = -1;
   if( ctx->Sound.SoundVar1 >= 0){

      if ( (ctx->Sound.SoundVar1 != ctx->Sound.PreviousSoundVar1) &&
           ctx->Sound.var1grid != NULL ){
         release_grid( ctx, time, ctx->Sound.PreviousSoundVar1, ctx->Sound.var1grid );
         ctx->Sound.var1grid = get_grid( ctx, time, ctx->Sound.SoundVar1);
      }
      ctx->Sound.PreviousSoundVar1 = ctx->Sound.SoundVar1;

      if (!ctx->Sound.var1_gc){
         ctx->Sound.var1_gc = make_gc(ctx, 0, 255, 0, 0, 0, 0, 2);
         if  (XLoadFont(SndDpy, "6x12")){
            XSetFont(SndDpy, ctx->Sound.var1_gc, XLoadFont(SndDpy, "6x12"));
         }
      }
      if (ctx->Sound.var1grid == NULL ){
         ctx->Sound.var1grid = get_grid( ctx, time, ctx->Sound.SoundVar1);
      }
      if (!ctx->Sound.var1grid){
         return NULL;
      }
      yo= extract_sound( ctx, ctx->Sound.var1grid, ctx->Sound.SoundVar1, ctx->Nr,
                         ctx->Nc, ctx->Nl[ctx->Sound.SoundVar1],
                         ctx->LowLev[ctx->Sound.SoundVar1], row, col );
      for (yo=0; yo < ctx->Nl[ctx->Sound.SoundVar1]; yo++){
         alt = grid_level_to_height( ctx, yo);
         value = ctx->Sound.soundline[yo];
         oldx = x;
         oldy = y;
         vardata_to_xy(ctx, alt, value, ctx->MinVal[ctx->Sound.SoundVar1],
                       ctx->MaxVal[ctx->Sound.SoundVar1], &x, &y);
         if ((x <= ctx->Sound.sndwidth) && (y <= ctx->Sound.sndheight)
            && (oldx <= ctx->Sound.sndwidth) && (oldy <= ctx->Sound.sndheight)
            && (x >= 0) && (y >= 0) && (oldx >= 0) && (oldy >= 0)){
            XDrawLine(SndDpy, ctx->Sound.soundwin, ctx->Sound.var1_gc, oldx+BORDER,
                      oldy+BORDER-HEBGBS, x +BORDER, y +BORDER-HEBGBS);
         }
      }
   }
   else {
      if (ctx->Sound.var1grid != NULL ){
         release_grid(ctx, time, ctx->Sound.PreviousSoundVar1, ctx->Sound.var1grid);
         ctx->Sound.var1grid = NULL;
      }
   }

   oldx = -1;
   oldy = -1;
   x    = -1;
   y    = -1;
   if( ctx->Sound.SoundVar2 >= 0){

      if ( (ctx->Sound.SoundVar2 != ctx->Sound.PreviousSoundVar2) &&
           ctx->Sound.var2grid != NULL ){
         release_grid( ctx, time, ctx->Sound.PreviousSoundVar2, ctx->Sound.var2grid );
         ctx->Sound.var2grid = get_grid( ctx, time, ctx->Sound.SoundVar2);
      }
      ctx->Sound.PreviousSoundVar2 = ctx->Sound.SoundVar2;

      if (!ctx->Sound.var2_gc){
         ctx->Sound.var2_gc = make_gc(ctx, 0, 255, 0, 0, 0, 0, 2);
         if  (XLoadFont(SndDpy, "6x12")){
            XSetFont(SndDpy, ctx->Sound.var2_gc, XLoadFont(SndDpy, "6x12"));
         }
      }
      if (ctx->Sound.var2grid == NULL ){
         ctx->Sound.var2grid = get_grid( ctx, time, ctx->Sound.SoundVar2);
      }
      if (!ctx->Sound.var2grid){
         return NULL;
      }
      yo= extract_sound( ctx, ctx->Sound.var2grid, ctx->Sound.SoundVar2, ctx->Nr,
                         ctx->Nc, ctx->Nl[ctx->Sound.SoundVar2],
                         ctx->LowLev[ctx->Sound.SoundVar2], row, col );
      for (yo=0; yo < ctx->Nl[ctx->Sound.SoundVar2]; yo++){
         alt = grid_level_to_height( ctx, yo);
         value = ctx->Sound.soundline[yo];
         oldx = x;
         oldy = y;
         vardata_to_xy(ctx, alt, value, ctx->MinVal[ctx->Sound.SoundVar2],
                       ctx->MaxVal[ctx->Sound.SoundVar2], &x, &y);
         if ((x <= ctx->Sound.sndwidth) && (y <= ctx->Sound.sndheight)
            && (oldx <= ctx->Sound.sndwidth) && (oldy <= ctx->Sound.sndheight)
            && (x >= 0) && (y >= 0) && (oldx >= 0) && (oldy >= 0)){
            XDrawLine(SndDpy, ctx->Sound.soundwin, ctx->Sound.var2_gc, oldx+BORDER,
                      oldy+BORDER-HEBGBS, x +BORDER, y +BORDER-HEBGBS);
         }
      }
   }
   else {
      if (ctx->Sound.var2grid != NULL ){
         release_grid(ctx, time, ctx->Sound.PreviousSoundVar2, ctx->Sound.var2grid);
         ctx->Sound.var2grid = NULL;
      }
   }
  
   oldx = -1;
   oldy = -1;
   x    = -1;
   y    = -1;
   if( ctx->Sound.SoundVar3 >= 0){
      if ( (ctx->Sound.SoundVar3 != ctx->Sound.PreviousSoundVar3) &&
           ctx->Sound.var3grid != NULL ){
         release_grid( ctx, time, ctx->Sound.PreviousSoundVar3, ctx->Sound.var3grid );
         ctx->Sound.var3grid = get_grid( ctx, time, ctx->Sound.SoundVar3);
      }
      ctx->Sound.PreviousSoundVar3 = ctx->Sound.SoundVar3;

      if (!ctx->Sound.var3_gc){
         ctx->Sound.var3_gc = make_gc(ctx, 0, 255, 0, 0, 0, 0, 2);
         if  (XLoadFont(SndDpy, "6x12")){
            XSetFont(SndDpy, ctx->Sound.var3_gc, XLoadFont(SndDpy, "6x12"));
         }
      }
      if (ctx->Sound.var3grid == NULL ){
         ctx->Sound.var3grid = get_grid( ctx, time, ctx->Sound.SoundVar3);
      }
      if (!ctx->Sound.var3grid){
         return NULL;
      }
      yo= extract_sound( ctx, ctx->Sound.var3grid, ctx->Sound.SoundVar3, ctx->Nr,
                         ctx->Nc, ctx->Nl[ctx->Sound.SoundVar3],
                         ctx->LowLev[ctx->Sound.SoundVar3], row, col );
      for (yo=0; yo < ctx->Nl[ctx->Sound.SoundVar3]; yo++){
         alt = grid_level_to_height( ctx, yo);
         value = ctx->Sound.soundline[yo];
         oldx = x;
         oldy = y;
         vardata_to_xy(ctx, alt, value, ctx->MinVal[ctx->Sound.SoundVar3],
                       ctx->MaxVal[ctx->Sound.SoundVar3], &x, &y);
         if ((x <= ctx->Sound.sndwidth) && (y <= ctx->Sound.sndheight)
            && (oldx <= ctx->Sound.sndwidth) && (oldy <= ctx->Sound.sndheight)
            && (x >= 0) && (y >= 0) && (oldx >= 0) && (oldy >= 0)){
            XDrawLine(SndDpy, ctx->Sound.soundwin, ctx->Sound.var3_gc, oldx+BORDER,
                      oldy+BORDER-HEBGBS, x +BORDER, y +BORDER-HEBGBS);


         }
      }
   }
   else {
      if (ctx->Sound.var3grid != NULL ){
         release_grid(ctx, time, ctx->Sound.PreviousSoundVar3, ctx->Sound.var3grid);
         ctx->Sound.var3grid = NULL;
      }
   }

   return 1;
}

    /**********************************************************************/
    /* This converts vertical plot data to x and y's matching the soundwin*/
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        alt - altitude of data point in                             */
    /*        value - value of the data point, to be plotted on the y axis*/
    /*        min - minimum value for the data point                      */
    /*        max - maximum value for the data point                      */
    /* Output: x, y - the coordinates to be plotted on soundwin           */
    /**********************************************************************/



void vardata_to_xy(Context ctx, float alt, float value, float min, float max, int *x, int *y)
{

   if (ctx->Sound.samestepflag){
      *x = (int) ((ctx->Sound.sndwidth * (value - ctx->Sound.samestepmin))/(ctx->Sound.samestepmax -
                                           ctx->Sound.samestepmin));
      *y = (int) (ctx->Sound.sndheight - (((alt-ctx->BottomBound)*ctx->Sound.sndheight)/
                                           (ctx->TopBound - ctx->BottomBound))); 
   }
   else {
      *x = (int) ((ctx->Sound.sndwidth * (value - min))/(max - min));
      *y = (int) (ctx->Sound.sndheight - (((alt-ctx->BottomBound)*ctx->Sound.sndheight)/
                                         (ctx->TopBound - ctx->BottomBound)));
   }
}



    /**********************************************************************/
    /* This will figure out what the interval for the value of the        */
    /* variable is compared to ctx->mainvarstep                           */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /**********************************************************************/

void setvarsteps( Context ctx)
{

   if (ctx->Sound.samestepflag){
      int min, max, yo;
  
      if (ctx->Sound.SoundVar1 >= 0){
         min = ctx->MinVal[ctx->Sound.SoundVar1];
         max = ctx->MaxVal[ctx->Sound.SoundVar1];
      }
      if (ctx->Sound.SoundVar2 >= 0){
         if (ctx->MinVal[ctx->Sound.SoundVar2] < min){
            min = ctx->MinVal[ctx->Sound.SoundVar2];
         }
         if (ctx->MaxVal[ctx->Sound.SoundVar2] > max){
            max = ctx->MaxVal[ctx->Sound.SoundVar2];
         }
      }
      if (ctx->Sound.SoundVar3 >= 0){
         if (ctx->MinVal[ctx->Sound.SoundVar3] < min) {
            min = ctx->MinVal[ctx->Sound.SoundVar3];
         }
         if (ctx->MaxVal[ctx->Sound.SoundVar3] > max) {
            max = ctx->MaxVal[ctx->Sound.SoundVar3];
         }
      }
      if (ctx->Sound.SoundVar1 >= 0){
         ctx->Sound.var1step =(( ((float)(ctx->Sound.mainvarstep)) * (max -
                        min ))/ ((float)(ctx->Sound.sndwidth))) ;
      }
      if (ctx->Sound.SoundVar2 >= 0){
         ctx->Sound.var2step =(( ((float)(ctx->Sound.mainvarstep)) * (max -
                        min))/ ((float)(ctx->Sound.sndwidth))) ;
      }
      if (ctx->Sound.SoundVar3 >= 0){
         ctx->Sound.var3step =(( ((float)(ctx->Sound.mainvarstep)) * (max -
                        min))/ ((float)(ctx->Sound.sndwidth))) ;
      }
      ctx->Sound.samestepmax = max;
      ctx->Sound.samestepmin = min;
   }
   else {
      if (ctx->Sound.SoundVar1 >= 0){
         ctx->Sound.var1step =(( ((float)(ctx->Sound.mainvarstep)) * (ctx->MaxVal[ctx->Sound.SoundVar1] -
                        ctx->MinVal[ctx->Sound.SoundVar1]))/ ((float)(ctx->Sound.sndwidth))) ;
      }
      if (ctx->Sound.SoundVar2 >= 0){
         ctx->Sound.var2step =(( ((float)(ctx->Sound.mainvarstep)) * (ctx->MaxVal[ctx->Sound.SoundVar2] -
                        ctx->MinVal[ctx->Sound.SoundVar2]))/ ((float)(ctx->Sound.sndwidth))) ;
      }
      if (ctx->Sound.SoundVar3 >= 0){
         ctx->Sound.var3step =(( ((float)(ctx->Sound.mainvarstep)) * (ctx->MaxVal[ctx->Sound.SoundVar3] -
                        ctx->MinVal[ctx->Sound.SoundVar3]))/ ((float)(ctx->Sound.sndwidth))) ;
      }
   }
}


    /**********************************************************************/
    /* This converts skew-t values to the x and y to be plotted           */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        alt - altitude of data point in kilometers                  */
    /*        temp - temperature in Kelvin of data point to be plotter    */
    /* Output: x, y - the coordinates to be plotted on soundwin           */
    /**********************************************************************/

void data_to_xy(Context ctx, float alt, float temp, int *x, int *y)
{
   alt += .00000069;
   temp += (alt * 4.638);
  *x = (int) ((temp- ctx->Sound.SndMinTemp) * (ctx->Sound.sndwidth/(ctx->Sound.SndMaxTemp - ctx->Sound.SndMinTemp)));
  *y = (int) (ctx->Sound.sndheight - (((alt-ctx->BottomBound)*ctx->Sound.sndheight)/
             (ctx->TopBound - ctx->BottomBound)));

}

    /**********************************************************************/
    /* This just converts vertical values into a y coordinate             */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        alt - altitude of the value                                 */
    /* Output: y - the y coordinate to be plotted in soundwin             */
    /**********************************************************************/


void data_to_y (Context ctx, float alt, int *y)
{
   alt += .00000069;
   *y = (int) (ctx->Sound.sndheight - (((alt-ctx->BottomBound)*
               ctx->Sound.sndheight)/(ctx->TopBound - ctx->BottomBound)));
}


    /**********************************************************************/
    /* This converts a grid level to a height                             */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        level - grid level                                          */
    /* Output: height                                                     */
    /*************************** NOTE!! ***********************************/
    /*******  This same code is used in proj.c lines 614 - 649  ***********/
    /**********************************************************************/


float grid_level_to_height( Context ctx, float level )
{
   int ilevel;
   float rlevel;

   if (level<=0) {
      return ctx->BottomBound;
   }
   else if (level>=ctx->MaxNl-1 || ctx->MaxNl == 1) {
      return ctx->TopBound;
   }
   else {
      switch (ctx->VerticalSystem) {
         case VERT_GENERIC:
         case VERT_EQUAL_KM:
            return ctx->BottomBound + level * ctx->LevInc;
         case VERT_NONEQUAL_MB:
         case VERT_NONEQUAL_KM:
            ilevel = (int) level;
            rlevel = level - ilevel;
            return ctx->Height[ilevel] * (1.0-rlevel) + ctx->Height[ilevel+1] * rlevel;
         default:
            printf("Error in gridlevel_to_height\n");
      }
   }
   return 0.0;
}

    /**********************************************************************/
    /* This interpolates values in between grid points                    */
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        grid - grid array                                           */
    /*        var - variable to extract form grid                         */
    /*        nr, nc, nl - number or rows, columns, and levels for varible*/
    /*        lowlev - the lowest level for the variable                  */
    /*        row, col - row and column where cursor is                   */
    /* Output: return 1 if all goes well                                  */
    /**********************************************************************/

static int extract_sound( Context ctx, float *grid, int var,
                             int nr, int nc, int nl, int lowlev,
                             float row, float col)
{
   float a, b, c, d;
   float I, J, K, L, E, F, Z;
   int  level, lowrow, highrow, leftcol, rightcol;


   /* allocate buffer to put soundline data into */
   if (ctx->Sound.soundline != NULL) deallocate(ctx, ctx->Sound.soundline,
                                          nl * sizeof(float));
   ctx->Sound.soundline = (float *) allocate_type(ctx, nl * sizeof(float), SOUND_TYPE );
   if (!ctx->Sound.soundline) {
     return NULL;
   }

   /*
    *  The best way to see the interpolation here is to draw a box
    *  each of the four coners represents a discrete grid point with a certain
    *  given value and the cursor lies someplace in side this box or on the BORDER
    *  of it.  The box has sides: TOP, BOTTOM, LEFT, and RIGHT.   Rows increase
    *  going down, and Cols increase going right.  So
    *  the upper left  corner is (leftcol, lowrow)  with a data value I
    *  the upper right corner is (rightcol,lowrow)  with a data value J
    *  the lower left  corner is (leftcol, highrow) with a data value K
    *  the lower right corner is (rightcol,highrow) with a data value L
    *  q is the point where the cursor col intersects the BOTTOM line
    *  r is the point where the cursor col intersects the TOP line
    *  s is the point on the line between q and r where the cursor lies
    *  E is the interpolated value for the point q
    *  F is the interpolated value for the point r
    *  Then one simply interpolates for what the value of the point s is
    *
    */

   level = lowlev;
   lowrow = (int) row;
   leftcol = (int) col;
   highrow = lowrow + 1;
   rightcol = leftcol + 1;

   if (highrow > nr-1)
      highrow = nr-1;
   if (rightcol > nc-1)
      rightcol = nc-1;

   a = col - leftcol;
   b = 1 - a;
   c = row - lowrow;
   d = 1 - c;

   if ( (row == ((float)lowrow)) && (col == ((float)leftcol)) ){
      for (level = lowlev; level < nl; level++){
         Z = grid[ lowrow + nr * (leftcol + nc * level)];
         if (IS_MISSING(Z)){
            ctx->Sound.soundline[level] = MISSING;
         }
         else{
            ctx->Sound.soundline[level] = Z;
         }
      }
   }
   else {
      for (level = lowlev; level < nl; level++){
         /* get values for the four coners of the (col, row) box */
         I = grid[ lowrow + nr * (leftcol + nc * level)];
         J = grid[ lowrow + nr * (rightcol+ nc * level)];
         K = grid[ highrow+ nr * (leftcol + nc * level)];
         L = grid[ highrow+ nr * (rightcol+ nc * level)];
         if (IS_MISSING(I) || IS_MISSING(J) || IS_MISSING(K) || IS_MISSING(L)) {
            ctx->Sound.soundline[level] = MISSING;
         }
         else {
            E = a * L +  b * K ;
            F = a * J +  b * I ;
            ctx->Sound.soundline[level] = c * E + d * F;
         }
      }
   }
   return 1;
}
    /**********************************************************************/
    /* This interpolates values in between grid points, for wind variables*/
    /**********************************************************************/
    /* Input: ctx - context                                               */
    /*        ugrid, vgrid  - grid arrays                                 */
    /*        varu, varv - variable to extract form grid                  */
    /*        nr, nc, nl - number or rows, columns, and levels for varible*/
    /*        lowlev - the lowest level for the variables                 */
    /*        row, col - row and column where cursor is                   */
    /* Output: return 1 if all goes well                                  */
    /**********************************************************************/

static int extract_wind( Context ctx, float *gridu, float *gridv,
                             int varu, int varv,
                             int nr, int nc, int nl, int lowlev,
                             float row, float col)
{
   float a, b, c, d;
   float I, J, K, L, E, F, Z;
   int  level, lowrow, highrow, leftcol, rightcol;


   /* allocate buffer to put windline data into */

   if (ctx->Sound.uwindline != NULL) deallocate(ctx, ctx->Sound.uwindline,
                                          nl * sizeof(float));
   if (ctx->Sound.vwindline != NULL) deallocate(ctx, ctx->Sound.vwindline,
                                          nl * sizeof(float));
   ctx->Sound.uwindline = (float *) allocate_type(ctx, nl * sizeof(float), UWIND_TYPE );
   ctx->Sound.vwindline = (float *) allocate_type(ctx, nl * sizeof(float), VWIND_TYPE );
   if ((!ctx->Sound.uwindline) || (!ctx->Sound.vwindline)) {
   return NULL;
   }

   level = lowlev;
   lowrow = (int) row;
   leftcol = (int) col;
   highrow = lowrow + 1;
   rightcol = leftcol + 1;

   if (highrow > nr-1)
      highrow = nr-1;
   if (rightcol > nc-1)
      rightcol = nc-1;

   a = col - leftcol;
   b = 1 - a;
   c = row - lowrow;
   d = 1 - c;

   if ( (row == ((float)lowrow)) && (col == ((float)leftcol)) ){
      for (level = lowlev; level < nl; level++){
         Z = gridu[ lowrow + nr * (leftcol + nc * level)];
         if (IS_MISSING(Z)){
            ctx->Sound.uwindline[level] = MISSING;
         }
         else{
            ctx->Sound.uwindline[level] = Z;
         }
      }
      for (level = lowlev; level < nl; level++){
         Z = gridv[ lowrow + nr * (leftcol + nc * level)];
         if (IS_MISSING(Z)){
            ctx->Sound.vwindline[level] = MISSING;
         }
         else{
            ctx->Sound.vwindline[level] = Z;
         }
      }
   return 1;
   }
   else {
      for (level = lowlev; level < nl; level++){
         /* get values for the four coners of the (col, row) box */
         I = gridu[ lowrow + nr * (leftcol + nc * level)];
         J = gridu[ lowrow + nr * (rightcol+ nc * level)];
         K = gridu[ highrow+ nr * (leftcol + nc * level)];
         L = gridu[ highrow+ nr * (rightcol+ nc * level)];
         if (IS_MISSING(I) || IS_MISSING(J) || IS_MISSING(K) || IS_MISSING(L)) {
            ctx->Sound.uwindline[level] = MISSING;
         }
         else {
            E = a * L +  b * K ;
            F = a * J +  b * I ;
            ctx->Sound.uwindline[level] = c * E + d * F;
         }
      }

      for (level = lowlev; level < nl; level++){
         /* get values for the four coners of the (col, row) box */
         I = gridv[ lowrow + nr * (leftcol + nc * level)];
         J = gridv[ lowrow + nr * (rightcol+ nc * level)];
         K = gridv[ highrow+ nr * (leftcol + nc * level)];
         L = gridv[ highrow+ nr * (rightcol+ nc * level)];
         if (IS_MISSING(I) || IS_MISSING(J) || IS_MISSING(K) || IS_MISSING(L)) {
            ctx->Sound.vwindline[level] = MISSING;
         }
         else {
            E = a * L +  b * K ;
            F = a * J +  b * I ;
            ctx->Sound.vwindline[level] = c * E + d * F;
         }
      }
      return 1;
   }
}
