
/* Vis5D version 4.3 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <math.h>
#include "api.h"
#include "box.h"
#include "graphics.h"
#include "globals.h"
#include "proj.h"
#include "render.h"



#define ADD_VERTEX( X, Y, Z )  if (ctx->NumBoxVerts<MAX_BOX_VERTS) {       \
                                  ctx->BoxVerts[ctx->NumBoxVerts][0] = X;  \
                                  ctx->BoxVerts[ctx->NumBoxVerts][1] = Y;  \
                                  ctx->BoxVerts[ctx->NumBoxVerts][2] = Z;  \
                                  ctx->NumBoxVerts++;                      \
                                }

#define END_OF_LINE    if (ctx->NumBoxVerts<MAX_BOX_VERTS) {            \
                          ctx->BoxVerts[ctx->NumBoxVerts][0] = -999.0;  \
                          ctx->NumBoxVerts++;                           \
                       }


#define VERT(Z) (ctx->VerticalSystem==VERT_NONEQUAL_MB ? height_to_pressure(Z) : (Z))


/*
 * Compute vertices for the 3-D box.
 * This function must be called AFTER the ctx->Nr,ctx->Nc,Nl variables have
 * been initialized.
 * Input:  prop - code for box proportions:
 *                1 = scale box according to lat, lons.
 *                2 = scale according to ax, ay, az.
 *           ax, ay, az - aspect ratio numbers used if prop==3.
 */
static void make_square_box( Context ctx )
{
   ctx->NumBoxVerts = 0;
   ctx->TickMarks = 0;

   /* bottom rectangle */
   ADD_VERTEX( ctx->Xmin, ctx->Ymax, ctx->Zmin );
   ADD_VERTEX( ctx->Xmax, ctx->Ymax, ctx->Zmin );
   ADD_VERTEX( ctx->Xmax, ctx->Ymin, ctx->Zmin );
   ADD_VERTEX( ctx->Xmin, ctx->Ymin, ctx->Zmin );
   ADD_VERTEX( ctx->Xmin, ctx->Ymax, ctx->Zmin );
   END_OF_LINE;

   /* top rectangle */
   ADD_VERTEX( ctx->Xmin, ctx->Ymax, ctx->Zmax );
   ADD_VERTEX( ctx->Xmax, ctx->Ymax, ctx->Zmax );
   ADD_VERTEX( ctx->Xmax, ctx->Ymin, ctx->Zmax );
   ADD_VERTEX( ctx->Xmin, ctx->Ymin, ctx->Zmax );
   ADD_VERTEX( ctx->Xmin, ctx->Ymax, ctx->Zmax );
   END_OF_LINE;

   /* top-to-bottom lines */
   ADD_VERTEX( ctx->Xmax, ctx->Ymax, ctx->Zmax );
   ADD_VERTEX( ctx->Xmax, ctx->Ymax, ctx->Zmin );
   END_OF_LINE;
   ADD_VERTEX( ctx->Xmax, ctx->Ymin, ctx->Zmax );
   ADD_VERTEX( ctx->Xmax, ctx->Ymin, ctx->Zmin );
   END_OF_LINE;
   ADD_VERTEX( ctx->Xmin, ctx->Ymin, ctx->Zmax );
   ADD_VERTEX( ctx->Xmin, ctx->Ymin, ctx->Zmin );
   END_OF_LINE;
   ADD_VERTEX( ctx->Xmin, ctx->Ymax, ctx->Zmax );
   ADD_VERTEX( ctx->Xmin, ctx->Ymax, ctx->Zmin );
   END_OF_LINE;

   /* arrow */
   ADD_VERTEX(  0.025, ctx->Ymax+0.05, ctx->Zmin );
   ADD_VERTEX(  0.0,   ctx->Ymax+0.1,  ctx->Zmin );
   ADD_VERTEX( -0.025, ctx->Ymax+0.05, ctx->Zmin );
   END_OF_LINE;
   ADD_VERTEX(  0.0,   ctx->Ymax,      ctx->Zmin );
   ADD_VERTEX(  0.0,   ctx->Ymax+0.1,  ctx->Zmin );
   END_OF_LINE;

   /* N */
   ADD_VERTEX( -0.025, ctx->Ymax+0.15, ctx->Zmin );
   ADD_VERTEX( -0.025, ctx->Ymax+0.25, ctx->Zmin );
   ADD_VERTEX(  0.025, ctx->Ymax+0.15, ctx->Zmin );
   ADD_VERTEX(  0.025, ctx->Ymax+0.25, ctx->Zmin );
   END_OF_LINE;

   if (ctx->Projection==PROJ_GENERIC || ctx->Projection==PROJ_LINEAR) {
      /* East tick mark */
      ADD_VERTEX( ctx->Xmax, ctx->Ymin, ctx->Zmin );
      ADD_VERTEX( ctx->Xmax, ctx->Ymin-0.05, ctx->Zmin-0.05);
      END_OF_LINE;
      /* West */
      ADD_VERTEX( ctx->Xmin, ctx->Ymin, ctx->Zmin );
      ADD_VERTEX( ctx->Xmin, ctx->Ymin-0.05, ctx->Zmin-0.05 );
      END_OF_LINE;
      /* North */
      ADD_VERTEX( ctx->Xmin, ctx->Ymax, ctx->Zmin );
      ADD_VERTEX( ctx->Xmin-0.05, ctx->Ymax, ctx->Zmin-0.05 );
      END_OF_LINE;
      /* South */
      ADD_VERTEX( ctx->Xmin, ctx->Ymin, ctx->Zmin );
      ADD_VERTEX( ctx->Xmin-0.05, ctx->Ymin, ctx->Zmin-0.05 );
      END_OF_LINE;
      /* Top */
      ADD_VERTEX( ctx->Xmin, ctx->Ymin, ctx->Zmax );
      ADD_VERTEX( ctx->Xmin-0.05, ctx->Ymin-0.05, ctx->Zmax );
      END_OF_LINE;
      /* Bottom */
      ADD_VERTEX( ctx->Xmin, ctx->Ymin, ctx->Zmin );
      ADD_VERTEX( ctx->Xmin-0.05, ctx->Ymin-0.05, ctx->Zmin );
      END_OF_LINE;
      ctx->TickMarks = 1;
   }
   else {
      ctx->TickMarks = 0;
   }
}




/*
 * Make the box used to bound a cylindrical projection.
 */
static void make_cylinder_box( Context ctx )
{
   float row, col, lev;
   int r, c, l;
   float x, y, z;
   int i;

   ctx->NumBoxVerts = 0;
   ctx->TickMarks = 0;

   /* curved edges */
   for (i=0;i<4;i++) {
      switch (i) {
         case 0:
            /* bottom south edge */
            r = ctx->Nr-1;
            l = 0;
            break;
         case 1:
            /* top south edge */
            r = ctx->Nr-1;
            l = ctx->MaxNl-1;
            break;
         case 2:
            /* top north edge */
            r = 0;
            l = ctx->MaxNl-1;
            break;
         case 3:
            /* bottom north edge */
            r = 0;
            l = 0;
            break;
      }

      for (c=0;c<ctx->Nc;c++) {
         row = (float) r;
         col = (float) c;
         lev = (float) l;
         grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
         ADD_VERTEX( x, y, z );
      }
      END_OF_LINE;
   }

   /* top east edge */
   row = 0;
   col = ctx->Nc-1;
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   row = ctx->Nr-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* bottom east edge */
   row = 0;
   col = ctx->Nc-1;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   row = ctx->Nr-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* top west edge */
   row = 0;
   col = 0;
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   row = ctx->Nr-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* bottom west edge */
   row = 0;
   col = 0;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   row = ctx->Nr-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* south-west vertical edge */
   row = ctx->Nr-1;
   col = 0;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* north-west vertical edge */
   row = 0;
   col = 0;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* north-east vertical edge */
   row = 0;
   col = ctx->Nc-1;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* south-east vertical edge */
   row = ctx->Nr-1;
   col = ctx->Nc-1;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

}



static void make_sphere_box( Context ctx )
{
   int i;
   float lat, lon, hgt;
   float x, y, z;
   float row, col, lev;
   int r, c, l;

   ctx->NumBoxVerts = 0;
   ctx->TickMarks = 0;

   /* Axis */
   lat = 90.0;
   lon = 0.0;
   hgt = 0.0;
   geo_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &lat, &lon, &hgt, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   lat = -90.0;
   geo_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &lat, &lon, &hgt, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* Equator */
   lat = 0.0;
   hgt = 0.0;
   for (i=0;i<=360;i+=10) {
      lon = (float) i;
      geo_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &lat, &lon, &hgt, &x, &y, &z );
      ADD_VERTEX( x, y, z );
   }
   END_OF_LINE;

   /* curved edges */
   for (i=0;i<4;i++) {
      switch (i) {
         case 0:
            /* bottom south edge */
            r = ctx->Nr-1;
            l = 0;
            break;
         case 1:
            /* top south edge */
            r = ctx->Nr-1;
            l = ctx->MaxNl-1;
            break;
         case 2:
            /* top north edge */
            r = 0;
            l = ctx->MaxNl-1;
            break;
         case 3:
            /* bottom north edge */
            r = 0;
            l = 0;
            break;
      }

      for (c=0;c<ctx->Nc;c++) {
         row = (float) r;
         col = (float) c;
         lev = (float) l;
         grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
         ADD_VERTEX( x, y, z );
      }
      END_OF_LINE;
   }

   for (i=0;i<4;i++) {
      switch (i) {
         case 0:
            /* bottom west edge */
            c = 0;
            l = 0;
            break;
         case 1:
            /* top west edge */
            c = 0;
            l = ctx->MaxNl-1;
            break;
         case 2:
            /* top east edge */
            c = ctx->Nc-1;
            l = ctx->MaxNl-1;
            break;
         case 3:
            /* bottom east edge */
            c = ctx->Nc-1;
            l = 0;
            break;
      }

      for (r=0;r<ctx->Nr;r++) {
         row = (float) r;
         col = (float) c;
         lev = (float) l;
         grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
         ADD_VERTEX( x, y, z );
      }
      END_OF_LINE;
   }

   /* south-west vertical edge */
   row = ctx->Nr-1;
   col = 0;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* north-west vertical edge */
   row = 0;
   col = 0;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* north-east vertical edge */
   row = 0;
   col = ctx->Nc-1;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;

   /* south-east vertical edge */
   row = ctx->Nr-1;
   col = ctx->Nc-1;
   lev = 0;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   lev = ctx->MaxNl-1;
   grid_to_xyz( ctx, 0, ctx->MaxNlVar, 1, &row, &col, &lev, &x, &y, &z );
   ADD_VERTEX( x, y, z );
   END_OF_LINE;
}





/*
 * Compute the graphics bounds Xmin, Xmax, Ymin then construct the box
 * graphics.
 */
void make_box( Context ctx, float ax, float ay, float az )
{
   /* First compute using user-provided aspect ratios, if given. */
   if (ax!=0.0 || ay!=0.0 || az!=0.0) {
      if (ax>=ay && ax>=az) {
         ay = ay / ax;
         az = az / ax;
         ax = 1.0;
      }
      else if (ay>=ax && ay>=az) {
         ax = ax / ay;
         az = az / ay;
         ay = 1.0;
      }
      else {
         ax = ax / az;
         ay = ay / az;
         az = 1.0;
      }
      ctx->Xmax = ax;    ctx->Xmin = -ctx->Xmax;
      ctx->Ymax = ay;    ctx->Ymin = -ctx->Ymax;
      ctx->Zmax = az;    ctx->Zmin = -ctx->Zmax;
   }

   /* Compute according to the projection. */
   switch (ctx->Projection) {
      case PROJ_GENERIC:
         ctx->CurvedBox = 0;
         if (ax==0.0 && ay==0.0 && az==0.0) {
            float rdim, cdim, ldim;
            rdim = fabs(ctx->Nr * ctx->RowInc);
            cdim = fabs(ctx->Nc * ctx->ColInc);
            ldim = ctx->TopBound - ctx->BottomBound;
            if (ldim==0.0) {
               ldim = 1.0;
            }
            if (rdim>=cdim && rdim>=ldim) {
               ctx->Xmax = cdim / rdim;
               ctx->Ymax = 1.0;
               ctx->Zmax = ldim / rdim;
            }
            else if (cdim>=rdim && cdim>=ldim) {
               ctx->Xmax = 1.0;
               ctx->Ymax = rdim / cdim;
               ctx->Zmax = ldim / cdim;
            }
            else {
               ctx->Xmax = cdim / ldim;
               ctx->Ymax = rdim / ldim;
               ctx->Zmax = 1.0;
            }
            ctx->Xmin = -ctx->Xmax;
            ctx->Ymin = -ctx->Ymax;
            ctx->Zmin = -ctx->Zmax;
         }
         break;
      case PROJ_LINEAR:
         ctx->CurvedBox = 0;
         if (ax==0.0 && ay==0.0 && az==0.0) {

               /* scale box according to lat and lon */
/*
               float lats, lons, hgt, midlat;

               midlat = (ctx->NorthBound+ctx->SouthBound) / 2.0;
               lats = ctx->NorthBound-ctx->SouthBound;
               if (lats<0.0)  lats = -lats;
               lons = (ctx->WestBound-ctx->EastBound) *
                                         cos( midlat * 3.14159/180.0 );
               if (lons<0.0)  lons = -lons;
               hgt = ctx->TopBound-ctx->BottomBound;
*/

            float width, height, midlat;
            midlat = (ctx->NorthBound+ctx->SouthBound) / 2.0;
            width = (ctx->Nc * ctx->ColInc) * cos( midlat * 3.14159/180.0 );

            height = ctx->Nr * ctx->RowInc;
            if (width>height) {
               ctx->Xmax = 1.0;
               ctx->Ymax = height / width;
               ctx->Zmax = ctx->Ymax / 2.0;
            }
            else {
               ctx->Xmax = width / height;
               ctx->Ymax = 1.0;
               ctx->Zmax = ctx->Xmax / 2.0;
            }
            ctx->Xmin = -ctx->Xmax;
            ctx->Ymin = -ctx->Ymax;
            ctx->Zmin = -ctx->Zmax;
         }
         break;
      case PROJ_ROTATED:
         ctx->CurvedBox = 0;
         if (ax==0.0 && ay==0.0 && az==0.0) {

            /* scale box according to lat and lon */

            float width, height, midlat;
            midlat = (ctx->NorthBound+ctx->SouthBound) / 2.0;
            /*midlon = (ctx->WestBound+ctx->EastBound) / 2.0;*/
            width = (ctx->Nc * ctx->ColInc) * cos( midlat * 3.14159/180.0 );
            height = ctx->Nr * ctx->RowInc;
            if (width>height) {
               ctx->Xmax = 1.0;
               ctx->Ymax = height / width;
               ctx->Zmax = ctx->Ymax / 2.0;
            }
            else {
               ctx->Xmax = width / height;
               ctx->Ymax = 1.0;
               ctx->Zmax = ctx->Xmax / 2.0;
            }
            ctx->Xmin = -ctx->Xmax;
            ctx->Ymin = -ctx->Ymax;
            ctx->Zmin = -ctx->Zmax;
         }
         break;
      case PROJ_LAMBERT:
         ctx->CurvedBox = 0;
         if (ax==0.0 && ay==0.0 && az==0.0) {
            float width, height;
            width = ctx->Nc * ctx->ColInc;
            height = ctx->Nr * ctx->ColInc;   /* only thing different from LINEAR */
            if (width>height) {
               ctx->Xmax = 1.0;
               ctx->Ymax = height / width;
               ctx->Zmax = ctx->Ymax / 2.0;
            }
            else {
               ctx->Xmax = width / height;
               ctx->Ymax = 1.0;
               ctx->Zmax = ctx->Xmax / 2.0;
            }
            ctx->Xmin = -ctx->Xmax;
            ctx->Ymin = -ctx->Ymax;
            ctx->Zmin = -ctx->Zmax;
         }
         break;
      case PROJ_STEREO:
         ctx->CurvedBox = 0;
         if (ax==0.0 && ay==0.0 && az==0.0) {
            if (ctx->Nc>ctx->Nr) {
               ctx->Xmax = 1.0;
               ctx->Ymax = (float) ctx->Nr / (float) ctx->Nc;
               ctx->Zmax = ctx->Ymax / 2.0;
            }
            else {
               ctx->Xmax = (float) ctx->Nc / (float) ctx->Nr;
               ctx->Ymax = 1.0;
               ctx->Zmax = ctx->Xmax / 2.0;
            }
            ctx->Xmin = -ctx->Xmax;
            ctx->Ymin = -ctx->Ymax;
            ctx->Zmin = -ctx->Zmax;
         }
         break;
      case PROJ_CYLINDRICAL:
         /* Curved box */
         ctx->CurvedBox = 1;
         if (ax==0.0 && ay==0.0 && az==0.0) {
            ctx->Xmin = -1.0;
            ctx->Xmax =  1.0;
            ctx->Ymin = -1.0;
            ctx->Ymax =  1.0;
            ctx->Zmin = -0.125;   /* TODO: compute better values? */
            ctx->Zmax =  0.125;
         }
         else {
            ctx->Xmin = -1.0;
            ctx->Xmax =  1.0;
            ctx->Ymin = -1.0;
            ctx->Ymax =  1.0;
            ctx->Zmin = -0.125 * az / ax;
            ctx->Zmax =  0.125 * az / ax;
         }
         break;
      case PROJ_SPHERICAL:
         ctx->CurvedBox = 1;
         ctx->Xmin = -1.0;
         ctx->Xmax =  1.0;
         ctx->Ymin = -1.0;
         ctx->Ymax =  1.0;
         ctx->Zmin = -1.0;
         ctx->Zmax =  1.0;
         break;
      default:
         printf("Error in setup_box\n");
   }

   /*
    * Construct the 3-D bounding box.
    */
   switch (ctx->Projection) {
      case PROJ_GENERIC:
      case PROJ_LINEAR:
      case PROJ_LAMBERT:
      case PROJ_STEREO:
      case PROJ_ROTATED:
         make_square_box(ctx);
         break;
      case PROJ_CYLINDRICAL:
         make_cylinder_box(ctx);
         break;
      case PROJ_SPHERICAL:
         make_sphere_box(ctx);
         break;
      default:
         printf("Error in setup_box\n");
   }
}



/*
 * Draw the 3-D box.
 * Input:  it - time step.
 */
void draw_box( Context ctx, int it )
{
   /* base and up vectors for text drawn along x,y,z axes. */
   static float bx[3] = { 0.05, 0.0, 0.0 },      ux[3] = { 0.0, 0.05, 0.05 };
   static float by[3] = { -0.035, 0.0, -0.035 },  uy[3] = { 0.0, 0.07, 0.0 };
   static float bz[3] = { -0.035, -0.035, 0.0 }, uz[3] = { 0.0, 0.0, 0.07 };
   float x1, y1, z1, x2, y2, z2;
   char str[100];

   /* set depth cueing & line color */
   set_color( ctx->BoxColor );
   set_depthcue( ctx->DepthCue );

   draw_multi_lines( ctx->NumBoxVerts, ctx->BoxVerts, ctx->BoxColor );

   if (ctx->TickMarks) {
      /* Draw axis labels. */
      if (ctx->CoordFlag) {
         x1 = 1.0;
         x2 = (float) ctx->Nc;
         y1 = 1.0;
         y2 = (float) ctx->Nr;
         z1 = 1.0;
         z2 = (float) ctx->MaxNl;
      }
      else {
         x1 = ctx->WestBound;
         x2 = ctx->EastBound;
         y1 = ctx->NorthBound;
         y2 = ctx->SouthBound;
#ifdef LEVELTYPES
         z1 = ctx->BottomCoordinate;
         z2 = ctx->TopCoordinate;
#else
         z1 = ctx->BottomBound;
         z2 = ctx->TopBound;
#endif
         z1 = VERT(z1);
         z2 = VERT(z2);
      }

      if (ctx->CursorX-ctx->Xmin > 0.1 || ctx->DisplayCursor==0) {
         float2string( x1, str );
         plot_string( str, ctx->Xmin-0.02, ctx->Ymin-0.1, ctx->Zmin-0.125, bx, ux, 0 );
      }

      if (ctx->Xmax-ctx->CursorX > 0.1 || ctx->DisplayCursor==0) {
         float2string( x2, str );
         plot_string( str, ctx->Xmax-0.05, ctx->Ymin-0.1, ctx->Zmin-0.125, bx, ux, 0 );
      }

      if (ctx->Ymax-ctx->CursorY > 0.1 || ctx->DisplayCursor==0) {
         float2string( y1, str );
         plot_string( str, ctx->Xmin-0.075, ctx->Ymax-0.03, ctx->Zmin-0.075, by, uy, 1 );
      }

      if (ctx->CursorY-ctx->Ymin > 0.1 || ctx->DisplayCursor==0) {
         float2string( y2, str );
         plot_string( str, ctx->Xmin-0.075, ctx->Ymin-0.02, ctx->Zmin-0.075, by, uy, 1 );
      }

      if (ctx->CursorZ-ctx->Zmin > 0.1 || ctx->DisplayCursor==0) {
         float2string( z1, str );
         plot_string( str, ctx->Xmin-0.07, ctx->Ymin-0.07, ctx->Zmin+0.005, bz, uz, 1 );
      }

      if (ctx->Zmax-ctx->CursorZ > 0.1 || ctx->DisplayCursor==0) {
         float2string( z2, str );
         plot_string( str, ctx->Xmin-0.07, ctx->Ymin-0.07, ctx->Zmax+0.005, bz, uz, 1 );
      }
   }

   set_depthcue( 0 );
}





void draw_logo( Context ctx, unsigned int c )
{
   static short vv[7][2] = {{-1,0}, {9, 41}, {11, 41}, {21, 0},
                          {20, 0}, {10, 41}, {0, 0}};
   static short ii[4][2] = { {10,18}, {10,40}, {11,40}, {11,18} };
   static short idot[4][2] = { {10,15}, {10,14}, {11,14}, {12,15} };
   static short ss[24][2] = {{0,37}, {5, 40}, {16,40}, {20, 38},
                           {20, 32}, {16, 30}, {5, 30}, {0,28},
                           {0, 22}, {5, 20}, {15, 20}, {20, 22},
                           {20, 23}, {15, 21}, {5, 21}, {1, 22},
                           {1, 28}, {5, 30}, {16, 30}, {21, 32},
                           {21, 38}, {16, 40}, {5, 40}, {0, 38}};
   static short s5[18][2] = {{0,35}, {5, 40}, {16,40}, {20, 36},
                           {20, 25}, {16, 21}, {0, 21},
                           {0, 0}, {20, 0}, {20,1}, {1, 1},
                           {1, 20}, {16, 20}, {21, 25},
                           {21, 36}, {16, 41}, {5, 41}, {0, 36}};
   static short dd[14][2] = {{0, 0}, {0, 41}, {16, 41}, {21, 36},
                           {21, 5}, {16, 0}, {1, 0}, {1, 1},
                           {16, 1}, {20, 5}, {20, 36}, {16, 40},
                           {1, 40}, {1, 1}};
   short p[30][2];
   int j;

   set_color( c );

   for (j=0; j<7; j++) {
      p[j][0] = ctx->WinWidth - 132 + vv[j][0];
      p[j][1] = ctx->WinHeight - 50 + vv[j][1];
   }
   polyline2d( p, 7 );

   for (j=0; j<4; j++) {
      p[j][0] = ctx->WinWidth - 112 + ii[j][0];
      p[j][1] = ctx->WinHeight - 50 + ii[j][1];
   }
   polyline2d( p, 4 );

   for (j=0; j<4; j++) {
      p[j][0] = ctx->WinWidth - 112 + idot[j][0];
      p[j][1] = ctx->WinHeight - 50 + idot[j][1];
   }
   polyline2d( p, 4 );

   for (j=0; j<24; j++) {
      p[j][0] = ctx->WinWidth - 90 + ss[j][0];
      p[j][1] = ctx->WinHeight - 50 + ss[j][1];
   }
   polyline2d( p, 24 );

   for (j=0; j<18; j++) {
      p[j][0] = ctx->WinWidth - 60 + s5[j][0];
      p[j][1] = ctx->WinHeight - 50 + s5[j][1];
   }
   polyline2d( p, 18 );

   for (j=0; j<14; j++) {
      p[j][0] = ctx->WinWidth - 30 + dd[j][0];
      p[j][1] = ctx->WinHeight - 50 + dd[j][1];
   }
   polyline2d( p, 14 );

}

