/*
  Initialization routines:
  
  	- Read texture images
  	- Initialize planets' data
  	- Create display lists
  	- Other initialization routines
  
  Major code reorganization since 1.02
  
  Satellites' textures (except Moon) are 128x128 so we don't reach the 2MB 
  limit in many Voodoo Graphics cards. May change in the future, since all 
  Voodoo2 cards come with a minimum of 4MB on board.
  
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <malloc.h>
#include <time.h>
#include "jpeglib.h"
#include <setjmp.h>
#include <GL/glut.h>
#include "planets.h"

char * filenames[NUMBODIES]=	{"sun.jpg",
			"mercury.jpg",
			"venus.jpg",
			"earth.jpg",
			"mars.jpg",
			"jupiter.jpg",
			"saturn.jpg",
			"uranus.jpg",
			"neptune.jpg",
			"pluto.jpg",
			"moon.jpg",
			"io.jpg",
			"europa.jpg",
			"ganymede.jpg",
			"callisto.jpg",
			"tethys.jpg",
			"dione.jpg",
			"rhea.jpg",
			"titan.jpg",
			"triton.jpg",
			"charon.jpg",
			"satrings.jpg"};

GLfloat LightPos[4] = {0.0, 0.0, 0.0, 1.0};
GLfloat ambient[4] = {0.6, 0.6, 0.6, 1.0};
GLfloat White[4] = {1.0, 1.0, 1.0, 1.0};
GLfloat Black[4] = {0.0, 0.0, 0.0, 1.0};
GLuint texName[NUMBODIES],texture=1,smodel=GL_SMOOTH,lighting=1,drawstars=1;
GLuint Stars,red,polaris;
int ImgWidth, ImgHeight;
GLenum ImgFormat;
planetdata planets[NUMBODIES];
GLfloat stars[NUMSTARS][4];
double days,timefactor=1/(24.0*60.0); /* iterarion = 1 minute */
int SLICES=12,STACKS=12;
char texturepath[100]=".";
static void InitSun(void),InitMercury(void),InitVenus(void),InitEarth(void),
	InitMars(void),InitJupiter(void),InitSaturn(void),InitUranus(void),
	InitNeptune(void),InitPluto(void),InitStars(void);

GLubyte *read_JPEG_file (char *);

void Init( void )
{
   time_t t;
   
   glEnable(GL_LIGHTING);
   glEnable(GL_LIGHT0);
   glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
   glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambient);

   t=time(NULL);
   srand(t);
   days=t/3600.0/24.0-10093.0;
   glEnable(GL_DEPTH_TEST);
   glEnable(GL_CULL_FACE);
   glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
   glGenTextures(NUMBODIES, texName);
   InitSun();      
   InitMercury();   
   InitVenus();
   InitEarth();
   InitMars();
   InitJupiter(); 
   InitSaturn();   
   InitUranus();   
   InitNeptune();   
   InitPluto();   
   InitStars();
   glEnable(GL_TEXTURE_2D);
   glShadeModel(GL_SMOOTH);
   glBlendFunc(GL_DST_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
}


static void InitSun(void)
{ 
   static int i,j;
   static char *localimage;
   
   planets[SUN].Radius=2.0; /*Real Radius is 109.6*/
   planets[SUN].Degrees=0.0;
   planets[SUN].DeltaRotation=0.0;
   planets[SUN].Rotation=360.0/28.0;
   planets[SUN].Sat=0;
   planets[SUN].posx=planets[SUN].posy=planets[SUN].posz=0.0;
   sprintf(planets[SUN].Name,"Sun");

   planets[SUN].Object = gluNewQuadric();
   gluQuadricTexture( planets[SUN].Object, GL_FALSE );   
   planets[SUN].Sphere= glGenLists(1);
   glNewList( planets[SUN].Sphere, GL_COMPILE );
   gluSphere( planets[SUN].Object,RADIUSSCALE(planets[SUN].Radius), SLICES, STACKS );
   glEndList();

   planets[SUN].Image = read_JPEG_file(filenames[SUN]);
   if (!planets[SUN].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   localimage=malloc(128*128*4);
   for (i=0;i<(128*128);i++) {
	localimage[i*4]=planets[SUN].Image[i*3];
	localimage[i*4+1]=planets[SUN].Image[i*3+1];
	localimage[i*4+2]=planets[SUN].Image[i*3+2];
	j=planets[SUN].Image[i*3]*2;
	localimage[i*4+3]=j>255?255:j;
   }

   glBindTexture(GL_TEXTURE_2D, texName[SUN]);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 4,
                 128, 128,
                 0,
                 GL_RGBA, GL_UNSIGNED_BYTE,
                 localimage );
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);                 
   free(planets[SUN].Image);
   free(localimage);
} 


static void InitMercury(void)
{
   planets[MERCURY].Radius=0.382;
   planets[MERCURY].Degrees=0.0;
   planets[MERCURY].DeltaRotation=0.0;
   planets[MERCURY].Rotation=360.0/58.6462;
   planets[MERCURY].Inclination=DEG2RAD(7.00507);
   planets[MERCURY].AscendingNode=DEG2RAD(48.3339);
   planets[MERCURY].Perihelion=DEG2RAD(77.45399999999999);
   planets[MERCURY].MeanDistance=DISTCORRECTION(0.3870978);
   planets[MERCURY].DailyMotion=DEG2RAD(4.092353);
   planets[MERCURY].Eccentricity=0.2056324;
   planets[MERCURY].MeanLongitude=DEG2RAD(314.42369);
   planets[MERCURY].Sat=0;
   sprintf(planets[MERCURY].Name,"Mercury");

   planets[MERCURY].Image = read_JPEG_file(filenames[MERCURY]);
   if (!planets[MERCURY].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[MERCURY]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 256, 256,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[MERCURY].Image );
   free(planets[MERCURY].Image);
   planets[MERCURY].Object = gluNewQuadric();
   gluQuadricTexture( planets[MERCURY].Object, GL_TRUE );   
   planets[MERCURY].Sphere= glGenLists(1);
   glNewList( planets[MERCURY].Sphere, GL_COMPILE );
   gluSphere( planets[MERCURY].Object,RADIUSSCALE(planets[MERCURY].Radius), SLICES, STACKS );
   glEndList();

} 


static void InitVenus(void)
{
   planets[VENUS].Radius=0.95;
   planets[VENUS].Degrees=177.3;
   planets[VENUS].DeltaRotation=0.0;
   planets[VENUS].Rotation=360.0/243.0187;
   planets[VENUS].Inclination=DEG2RAD(3.39472);
   planets[VENUS].AscendingNode=DEG2RAD(76.6889);
   planets[VENUS].Perihelion=DEG2RAD(131.761);
   planets[VENUS].MeanDistance=DISTCORRECTION(0.7233238);
   planets[VENUS].DailyMotion=DEG2RAD(1.602158);
   planets[VENUS].Eccentricity=0.0067933;
   planets[VENUS].MeanLongitude=DEG2RAD(236.94045);
   planets[VENUS].Sat=0;
   sprintf(planets[VENUS].Name,"Venus");
   

   planets[VENUS].Image = read_JPEG_file(filenames[VENUS]);
   if (!planets[VENUS].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[VENUS]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 128, 128,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[VENUS].Image );
   free(planets[VENUS].Image);
   planets[VENUS].Object = gluNewQuadric();
   gluQuadricTexture( planets[VENUS].Object, GL_TRUE );   
   planets[VENUS].Sphere= glGenLists(1);
   glNewList( planets[VENUS].Sphere, GL_COMPILE );
   gluSphere( planets[VENUS].Object,RADIUSSCALE(planets[VENUS].Radius), SLICES, STACKS );
   glEndList();


} 


static void InitEarth(void)
{
   planets[EARTH].Radius=1.0;
   planets[EARTH].Degrees=23.45;
   planets[EARTH].DeltaRotation=0.0;
   planets[EARTH].Rotation=360.0/(365.0/365.256);
   planets[EARTH].Inclination=DEG2RAD(0.00041);
   planets[EARTH].AscendingNode=DEG2RAD(349.2);
   planets[EARTH].Perihelion=DEG2RAD(102.8517);
   planets[EARTH].MeanDistance=DISTCORRECTION(1.00002);
   planets[EARTH].DailyMotion=DEG2RAD(0.9855796);
   planets[EARTH].Eccentricity=0.0166967;
   planets[EARTH].MeanLongitude=DEG2RAD(328.40353);
   planets[EARTH].Sat=0;
   sprintf(planets[EARTH].Name,"Earth");

   planets[MOON].Radius=0.2724;
   planets[MOON].Degrees=5.1454;
   planets[MOON].DeltaRotation=0.0;
   planets[MOON].Rotation=360.0/27.32166;
   planets[MOON].Inclination=DEG2RAD(5.1454);
   planets[MOON].AscendingNode=DEG2RAD(125.1228);
   planets[MOON].Perihelion=DEG2RAD(83.1862);
   planets[MOON].MeanDistance=RADIUSSCALE(60.2666)/4.0;
   planets[MOON].DailyMotion=DEG2RAD(13.0649929509);
   planets[MOON].Eccentricity=0.054900;
   planets[MOON].MeanLongitude=DEG2RAD(328.40353);
   planets[MOON].Sat=EARTH;
   sprintf(planets[MOON].Name,"Moon");


   planets[EARTH].Image = read_JPEG_file(filenames[EARTH]);
   if (!planets[EARTH].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[EARTH]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 256, 256,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[EARTH].Image );
   free(planets[EARTH].Image);
   planets[EARTH].Object = gluNewQuadric();
   gluQuadricTexture( planets[EARTH].Object, GL_TRUE );   
   planets[EARTH].Sphere= glGenLists(1);
   glNewList( planets[EARTH].Sphere, GL_COMPILE );
   gluSphere( planets[EARTH].Object,RADIUSSCALE(planets[EARTH].Radius), SLICES, STACKS );
   glEndList();


   planets[MOON].Image = read_JPEG_file(filenames[MOON]);
   if (!planets[MOON].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[MOON]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 256, 256,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[MOON].Image );
   free(planets[MOON].Image);
   planets[MOON].Object = gluNewQuadric();
   gluQuadricTexture( planets[MOON].Object, GL_TRUE );   
   planets[MOON].Sphere= glGenLists(1);
   glNewList( planets[MOON].Sphere, GL_COMPILE );
   gluSphere( planets[MOON].Object,RADIUSSCALE(planets[MOON].Radius), SLICES, STACKS );
   glEndList();

} 


static void InitMars(void)
{
   planets[MARS].Radius=0.532;
   planets[MARS].Degrees=25.19;
   planets[MARS].DeltaRotation=0.0;
   planets[MARS].Rotation=360.0/1.025957;
   planets[MARS].Inclination=DEG2RAD(1.84992);
   planets[MARS].AscendingNode=DEG2RAD(49.5664);
   planets[MARS].Perihelion=DEG2RAD(336.0882);
   planets[MARS].MeanDistance=DISTCORRECTION(1.5236365);
   planets[MARS].DailyMotion=DEG2RAD(0.5240613);
   planets[MARS].Eccentricity=0.0934231;
   planets[MARS].MeanLongitude=DEG2RAD(262.42784);
   planets[MARS].Sat=0;
   sprintf(planets[MARS].Name,"Mars");

   planets[MARS].Image = read_JPEG_file(filenames[MARS]);
   if (!planets[MARS].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[MARS]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 256, 256,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[MARS].Image );
   free(planets[MARS].Image);
   planets[MARS].Object = gluNewQuadric();
   gluQuadricTexture( planets[MARS].Object, GL_TRUE );   
   planets[MARS].Sphere= glGenLists(1);
   glNewList( planets[MARS].Sphere, GL_COMPILE );
   gluSphere( planets[MARS].Object,RADIUSSCALE(planets[MARS].Radius), SLICES, STACKS );
   glEndList();

    
           
} 


static void InitJupiter(void)
{
   int i;
   
   planets[JUPITER].Radius=11.27;
   planets[JUPITER].Degrees=3.12;
   planets[JUPITER].DeltaRotation=0.0;
   planets[JUPITER].Rotation=360.0/0.41354;
   planets[JUPITER].Inclination=DEG2RAD(1.30463);
   planets[JUPITER].AscendingNode=DEG2RAD(100.4713);
   planets[JUPITER].Perihelion=DEG2RAD(15.6978);
   planets[JUPITER].MeanDistance=DISTCORRECTION(5.202597);
   planets[JUPITER].DailyMotion=DEG2RAD(8.309618000000001E-02);
   planets[JUPITER].Eccentricity=0.0484646;
   planets[JUPITER].MeanLongitude=DEG2RAD(322.55983);
   planets[JUPITER].Sat=0;
   sprintf(planets[JUPITER].Name,"Jupiter");


   planets[IO].Radius=1820.0/6378.0;
   planets[IO].Degrees=0.0;
   planets[IO].DeltaRotation=360.0/1.769138;
   planets[IO].Rotation=0.0;
   planets[IO].Inclination=0.0;
   planets[IO].Sat=JUPITER;
   sprintf(planets[IO].Name,"Io");

   planets[EUROPA].Radius=1565.0/6378.0;
   planets[EUROPA].Degrees=0.0;
   planets[EUROPA].DeltaRotation=0.0;
   planets[EUROPA].Rotation=360.0/3.551181;
   planets[EUROPA].Inclination=0.0;
   planets[EUROPA].Sat=JUPITER;
   sprintf(planets[EUROPA].Name,"Europa");

   planets[GANYMEDE].Radius=2635.0/6378.0;
   planets[GANYMEDE].Degrees=0.0;
   planets[GANYMEDE].DeltaRotation=0.0;
   planets[GANYMEDE].Rotation=360.0/7.154553;
   planets[GANYMEDE].Inclination=0.0;   
   planets[GANYMEDE].Sat=JUPITER;
   sprintf(planets[GANYMEDE].Name,"Ganymede");
   
   planets[CALLISTO].Radius=2420.0/6378.0;
   planets[CALLISTO].Degrees=0.0;
   planets[CALLISTO].DeltaRotation=0.0;
   planets[CALLISTO].Rotation=360.0/16.68902;
   planets[CALLISTO].Inclination=0.0;   
   planets[CALLISTO].Sat=JUPITER;
   sprintf(planets[CALLISTO].Name,"Callisto");



   planets[JUPITER].Image = read_JPEG_file(filenames[JUPITER]);
   if (!planets[JUPITER].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[JUPITER]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 256,256,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[JUPITER].Image );
   free(planets[JUPITER].Image);
   planets[JUPITER].Object = gluNewQuadric();
   gluQuadricTexture( planets[JUPITER].Object, GL_TRUE );   
   planets[JUPITER].Sphere= glGenLists(1);
   glNewList( planets[JUPITER].Sphere, GL_COMPILE );
   gluSphere( planets[JUPITER].Object,RADIUSSCALE(planets[JUPITER].Radius), SLICES, STACKS );
   glEndList();


   for (i=IO;i<=CALLISTO;i++) {
	   planets[i].Image = read_JPEG_file(filenames[i]);
	   if (!planets[i].Image) {
	      printf("Couldn't read image\n");
	      exit(0);
	   }

	   glBindTexture(GL_TEXTURE_2D, texName[i]);
	   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	   glTexImage2D( GL_TEXTURE_2D,
	                 0,
	                 3,
	                 128, 128,
	                 0,
	                 GL_RGB, GL_UNSIGNED_BYTE,
	                 planets[i].Image );
	   free(planets[i].Image);
	   planets[i].Object = gluNewQuadric();
	   gluQuadricTexture( planets[i].Object, GL_TRUE );   
	   planets[i].Sphere= glGenLists(1);
	   glNewList( planets[i].Sphere, GL_COMPILE );
	   gluSphere( planets[i].Object,RADIUSSCALE(planets[i].Radius), SLICES, STACKS );
	   glEndList();
   }
    
} 


static void InitSaturn(void)
{
   int i;
   unsigned char *localimage;
   
   planets[SATURN].Radius=9.5;
   planets[SATURN].Degrees=26.73;
   planets[SATURN].DeltaRotation=0.0;
   planets[SATURN].Rotation=360.0/0.44401;
   planets[SATURN].Inclination=DEG2RAD(2.48524);
   planets[SATURN].AscendingNode=DEG2RAD(113.6358);
   planets[SATURN].Perihelion=DEG2RAD(88.863);
   planets[SATURN].MeanDistance=DISTCORRECTION(9.571899999999999);
   planets[SATURN].DailyMotion=DEG2RAD(0.03328656);
   planets[SATURN].Eccentricity=0.0531651;
   planets[SATURN].MeanLongitude=DEG2RAD(20.95759);
   planets[SATURN].Sat=0;
   sprintf(planets[SATURN].Name,"Saturn");

   planets[TETHYS].Radius=525.0/6378.0;
   planets[TETHYS].Degrees=0.0;
   planets[TETHYS].DeltaRotation=0.0;
   planets[TETHYS].Rotation=360.0/1.8880926;
   planets[TETHYS].Inclination=0.0;
   planets[TETHYS].Sat=SATURN;
   sprintf(planets[TETHYS].Name,"Tethys");

   planets[DIONE].Radius=560.0/6378.0;
   planets[DIONE].Degrees=0.0;
   planets[DIONE].DeltaRotation=0.0;
   planets[DIONE].Rotation=360.0/2.7375218;
   planets[DIONE].Inclination=0.0;
   planets[DIONE].Sat=SATURN;
   sprintf(planets[DIONE].Name,"Dione");

   planets[RHEA].Radius=765.0/6378.0;
   planets[RHEA].Degrees=0.0;
   planets[RHEA].DeltaRotation=0.0;
   planets[RHEA].Rotation=360.0/4.5191631;
   planets[RHEA].Inclination=0.0;
   planets[RHEA].Sat=SATURN;
   sprintf(planets[RHEA].Name,"Rhea");

   planets[TITAN].Radius=2575.0/6378.0;
   planets[TITAN].Degrees=0.0;
   planets[TITAN].DeltaRotation=0.0;
   planets[TITAN].Rotation=360.0/15.9669028;
   planets[TITAN].Inclination=0.0;
   planets[TITAN].Sat=SATURN;
   sprintf(planets[TITAN].Name,"Titan");

   planets[SATURN].Image = read_JPEG_file(filenames[SATURN]);
   if (!planets[SATURN].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[SATURN]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 128, 128,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[SATURN].Image );
   free(planets[SATURN].Image);

   planets[SATURN].Image = read_JPEG_file(filenames[RINGS]);
   if (!planets[SATURN].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }
   
   localimage=malloc(256*256*4);
   for (i=0;i<(256*256);i++) {
	localimage[i*4]=planets[SATURN].Image[i*3];
	localimage[i*4+1]=planets[SATURN].Image[i*3+1];
	localimage[i*4+2]=planets[SATURN].Image[i*3+2];
	localimage[i*4+3]=planets[SATURN].Image[i*3]>10?255:0;
   }
   glBindTexture(GL_TEXTURE_2D, texName[RINGS]);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 4,
                 256, 256,
                 0,
                 GL_RGBA, GL_UNSIGNED_BYTE,
                 localimage );
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
   free(planets[SATURN].Image);
   free(localimage);



   planets[SATURN].Object = gluNewQuadric();
   gluQuadricTexture( planets[SATURN].Object, GL_TRUE );   
   planets[SATURN].Sphere= glGenLists(1);
   glNewList( planets[SATURN].Sphere, GL_COMPILE );
   gluSphere( planets[SATURN].Object,RADIUSSCALE(planets[SATURN].Radius), SLICES, STACKS );
   glBindTexture(GL_TEXTURE_2D, texName[RINGS]);
   glDisable(GL_CULL_FACE);
   glDisable(GL_LIGHTING);
   glEnable(GL_BLEND);
   gluDisk(planets[SATURN].Object,
           1.5*RADIUSSCALE(planets[SATURN].Radius),
           1.9*RADIUSSCALE(planets[SATURN].Radius),
           SLICES*2,1);
   gluDisk(planets[SATURN].Object,
           2.0*RADIUSSCALE(planets[SATURN].Radius),
           2.2*RADIUSSCALE(planets[SATURN].Radius),
           SLICES*2,1);
   glDisable(GL_BLEND);        
   glEnable(GL_LIGHTING);
   glEnable(GL_CULL_FACE);
   glEndList();

   for (i=TETHYS;i<=TITAN;i++) {
	   planets[i].Image = read_JPEG_file(filenames[i]);
	   if (!planets[i].Image) {
	      printf("Couldn't read image\n");
	      exit(0);
	   }

	   glBindTexture(GL_TEXTURE_2D, texName[i]);
	   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	   glTexImage2D( GL_TEXTURE_2D,
	                 0,
	                 3,
	                 128, 128,
	                 0,
	                 GL_RGB, GL_UNSIGNED_BYTE,
	                 planets[i].Image );
	   free(planets[i].Image);
	   planets[i].Object = gluNewQuadric();
	   gluQuadricTexture( planets[i].Object, GL_TRUE );   
	   planets[i].Sphere= glGenLists(1);
	   glNewList( planets[i].Sphere, GL_COMPILE );
	   gluSphere( planets[i].Object,RADIUSSCALE(planets[i].Radius), SLICES, STACKS );
	   glEndList();
   }
} 


static void InitUranus(void)
{
   planets[URANUS].Radius=3.9;
   planets[URANUS].Degrees=97.86;
   planets[URANUS].DeltaRotation=0.0;
   planets[URANUS].Rotation=360.0/0.71833;
   planets[URANUS].Inclination=DEG2RAD(0.77343);
   planets[URANUS].AscendingNode=DEG2RAD(74.0954);
   planets[URANUS].Perihelion=DEG2RAD(175.6807);
   planets[URANUS].MeanDistance=DISTCORRECTION(19.30181);
   planets[URANUS].DailyMotion=DEG2RAD(0.01162295);
   planets[URANUS].Eccentricity=0.0428959;
   planets[URANUS].MeanLongitude=DEG2RAD(303.18967);
   planets[URANUS].Sat=0;
   sprintf(planets[URANUS].Name,"Uranus");


   planets[URANUS].Image = read_JPEG_file(filenames[URANUS]);
   if (!planets[URANUS].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[URANUS]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 128, 128,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[URANUS].Image );
   free(planets[URANUS].Image);
   planets[URANUS].Object = gluNewQuadric();
   gluQuadricTexture( planets[URANUS].Object, GL_TRUE );   
   planets[URANUS].Sphere= glGenLists(1);
   glNewList( planets[URANUS].Sphere, GL_COMPILE );
   gluSphere( planets[URANUS].Object,RADIUSSCALE(planets[URANUS].Radius), SLICES, STACKS );
   glEndList();

    
} 


static void InitNeptune(void)
{
   planets[NEPTUNE].Radius=4.0;
   planets[NEPTUNE].Degrees=29.0;
   planets[NEPTUNE].DeltaRotation=0.0;
   planets[NEPTUNE].Rotation=360.0/0.67125;
   planets[NEPTUNE].Inclination=DEG2RAD(1.7681);
   planets[NEPTUNE].AscendingNode=DEG2RAD(131.7925);
   planets[NEPTUNE].Perihelion=DEG2RAD(7.206);
   planets[NEPTUNE].MeanDistance=DISTCORRECTION(30.26664);
   planets[NEPTUNE].DailyMotion=DEG2RAD(0.005919282);
   planets[NEPTUNE].Eccentricity=0.0102981;
   planets[NEPTUNE].MeanLongitude=DEG2RAD(299.8641);
   planets[NEPTUNE].Sat=0;
   sprintf(planets[NEPTUNE].Name,"Neptune");

   planets[TRITON].Radius=1353.0/6378.0;
   planets[TRITON].Degrees=0.0;
   planets[TRITON].DeltaRotation=0.0;
   planets[TRITON].Rotation=360.0/5.876854;
   planets[TRITON].Inclination=DEG2RAD(157.0);
   planets[TRITON].Sat=NEPTUNE;
   sprintf(planets[TRITON].Name,"Triton");



   planets[NEPTUNE].Image = read_JPEG_file(filenames[NEPTUNE]);
   if (!planets[NEPTUNE].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[NEPTUNE]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 128, 128,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[NEPTUNE].Image );
   free(planets[NEPTUNE].Image);
   planets[NEPTUNE].Object = gluNewQuadric();
   gluQuadricTexture( planets[NEPTUNE].Object, GL_TRUE );   
   planets[NEPTUNE].Sphere= glGenLists(1);
   glNewList( planets[NEPTUNE].Sphere, GL_COMPILE );
   gluSphere( planets[NEPTUNE].Object,RADIUSSCALE(planets[NEPTUNE].Radius), SLICES, STACKS );
   glEndList();


   planets[TRITON].Image = read_JPEG_file(filenames[TRITON]);
   if (!planets[TRITON].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[TRITON]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 128, 128,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[TRITON].Image );
   free(planets[TRITON].Image);
   planets[TRITON].Object = gluNewQuadric();
   gluQuadricTexture( planets[TRITON].Object, GL_TRUE );   
   planets[TRITON].Sphere= glGenLists(1);
   glNewList( planets[TRITON].Sphere, GL_COMPILE );
   gluSphere( planets[TRITON].Object,RADIUSSCALE(planets[TRITON].Radius), SLICES, STACKS );
   glEndList();


    
} 


static void InitPluto(void)
{
   planets[PLUTO].Radius=0.45;
   planets[PLUTO].Degrees=118.0;
   planets[PLUTO].DeltaRotation=0.0;
   planets[PLUTO].Rotation=360.0/6.3872;
   planets[PLUTO].Inclination=DEG2RAD(17.12137);
   planets[PLUTO].AscendingNode=DEG2RAD(110.3833);
   planets[PLUTO].Perihelion=DEG2RAD(224.8025);
   planets[PLUTO].MeanDistance=DISTCORRECTION(39.5804);
   planets[PLUTO].DailyMotion=DEG2RAD(0.003958072);
   planets[PLUTO].Eccentricity=0.2501272;
   planets[PLUTO].MeanLongitude=DEG2RAD(235.7656);
   planets[PLUTO].Sat=0;
   sprintf(planets[PLUTO].Name,"Pluto");

   planets[CHARON].Radius=593.0/6378.0;
   planets[CHARON].Degrees=0.0;
   planets[CHARON].DeltaRotation=0.0;
   planets[CHARON].Rotation=360.0/6.38725;
   planets[CHARON].Inclination=DEG2RAD(99.0);   
   planets[CHARON].Sat=PLUTO;
   sprintf(planets[CHARON].Name,"Charon");


   planets[PLUTO].Image = read_JPEG_file(filenames[PLUTO]);
   if (!planets[PLUTO].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[PLUTO]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 128, 128,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[PLUTO].Image );
   free(planets[PLUTO].Image);
   planets[PLUTO].Object = gluNewQuadric();
   gluQuadricTexture( planets[PLUTO].Object, GL_TRUE );   
   planets[PLUTO].Sphere= glGenLists(1);
   glNewList( planets[PLUTO].Sphere, GL_COMPILE );
   gluSphere( planets[PLUTO].Object,RADIUSSCALE(planets[PLUTO].Radius), SLICES, STACKS );
   glEndList();


   planets[CHARON].Image = read_JPEG_file(filenames[CHARON]);
   if (!planets[CHARON].Image) {
      printf("Couldn't read image\n");
      exit(0);
   }

   glBindTexture(GL_TEXTURE_2D, texName[CHARON]);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexImage2D( GL_TEXTURE_2D,
                 0,
                 3,
                 128, 128,
                 0,
                 GL_RGB, GL_UNSIGNED_BYTE,
                 planets[CHARON].Image );
   free(planets[CHARON].Image);
   planets[CHARON].Object = gluNewQuadric();
   gluQuadricTexture( planets[CHARON].Object, GL_TRUE );   
   planets[CHARON].Sphere= glGenLists(1);
   glNewList( planets[CHARON].Sphere, GL_COMPILE );
   gluSphere( planets[CHARON].Object,RADIUSSCALE(planets[CHARON].Radius), SLICES, STACKS );
   glEndList();
    
} 



static void InitStars(void)
{ 
   static char line[100],*tmp;
   FILE *f;
   static float ascention,declination,magnitude;
   int i;
   
   f=fopen("Stars.dat","rt");
   for (i=0;i<NUMSTARS;i++) {
	fgets(line,100,f);
	if (feof(f)) break;
	while (line[0]=='#') fgets(line,100,f);
	tmp=line;
	while (tmp[0]!=',') tmp++;
	*tmp='\0';
	tmp++;
	red=0;
	/* Print some stars in red color for testing */
	if (!strcmp(line,"Polaris")) { red=1; polaris=i; }
	if (!strcmp(line,"Dubhe")) red=1;
	if (!strcmp(line,"Merak")) red=1;
	if (!strcmp(line,"Phecda")) red=1;
	if (!strcmp(line,"Megrez")) red=1;
	if (!strcmp(line,"Alioth")) red=1;
	if (!strcmp(line,"Mizar")) red=1;
	sscanf(tmp,"%f,%f,%f\n",&ascention,&declination,&magnitude);
	stars[i][0]=DISTCORRECTION(500.0)*cos(-ascention)*cos(declination);
	stars[i][1]=DISTCORRECTION(500.0)*sin(declination);
	stars[i][2]=DISTCORRECTION(500.0)*sin(-ascention)*cos(declination);
	magnitude=sqrt(1.0/(1.46+magnitude));
	if (red) magnitude=-magnitude;
	stars[i][3]=magnitude;
   }
   fclose(f);

   Stars = glGenLists(1);
   glNewList( Stars, GL_COMPILE );
   glBegin(GL_POINTS);
   magnitude=1.0;
   for (i=0;i<NUMSTARS;i++) {
   	/* Since star catalog is sorted by magnitude and there are many stars
   	   with the same magnitude this will avoid extra calls to glColor */
	if (magnitude!=stars[i][3]) {
		glColor3f(stars[i][3],stars[i][3],stars[i][3]);
		magnitude=stars[i][3];
	}
	if (stars[i][3]<0) glColor3f(1.0,0.0,0.0);
	glVertex3f(stars[i][0],stars[i][1],stars[i][2]);
   }
   glEnd();
   glEndList();
}


/***************************************************************/
/* JPEG DECOMPRESSION CODE TAKEN FROM JPEG LIBRARY SAMPLE CODE */
/***************************************************************/
struct my_error_mgr {
  struct jpeg_error_mgr pub;	/* "public" fields */

  jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

/*
 * Here's the routine that will replace the standard error_exit method:
 */

METHODDEF(void)
my_error_exit (j_common_ptr cinfo)
{
  /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
  my_error_ptr myerr = (my_error_ptr) cinfo->err;

  /* Always display the message. */
  /* We could postpone this until after returning, if we chose. */
  (*cinfo->err->output_message) (cinfo);

  /* Return control to the setjmp point */
  longjmp(myerr->setjmp_buffer, 1);
}


/*
 * Sample routine for JPEG decompression.  We assume that the source file name
 * is passed in.  We want to return 1 on success, 0 on error.
 */


GLubyte *read_JPEG_file (char * fname)
{
  /* This struct contains the JPEG decompression parameters and pointers to
   * working space (which is allocated as needed by the JPEG library).
   */
  struct jpeg_decompress_struct cinfo;
  /* We use our private extension JPEG error handler.
   * Note that this struct must live as long as the main JPEG parameter
   * struct, to avoid dangling-pointer problems.
   */
  struct my_error_mgr jerr;
  /* More stuff */
  FILE * infile;		/* source file */
  JSAMPARRAY buffer;		/* Output row buffer */
  int row_stride;		/* physical row width in output buffer */
  long cont;
  JSAMPLE *image_buffer;
  char filename[128];
  
  /* In this example we want to open the input file before doing anything else,
   * so that the setjmp() error recovery below can assume the file is open.
   * VERY IMPORTANT: use "b" option to fopen() if you are on a machine that
   * requires it in order to read binary files.
   */

  sprintf(filename,"%s/%s",texturepath,fname);
  if ((infile = fopen(filename, "rb")) == NULL) {
    fprintf(stderr, "can't open %s\n", filename);
    return NULL;
  }

  /* Step 1: allocate and initialize JPEG decompression object */

  /* We set up the normal JPEG error routines, then override error_exit. */
  cinfo.err = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit = my_error_exit;
  /* Establish the setjmp return context for my_error_exit to use. */
  if (setjmp(jerr.setjmp_buffer)) {
    /* If we get here, the JPEG code has signaled an error.
     * We need to clean up the JPEG object, close the input file, and return.
     */
    jpeg_destroy_decompress(&cinfo);
    fclose(infile);
    return NULL;
  }
  /* Now we can initialize the JPEG decompression object. */
  jpeg_create_decompress(&cinfo);

  /* Step 2: specify data source (eg, a file) */

  jpeg_stdio_src(&cinfo, infile);

  /* Step 3: read file parameters with jpeg_read_header() */

  (void) jpeg_read_header(&cinfo, TRUE);
  /* We can ignore the return value from jpeg_read_header since
   *   (a) suspension is not possible with the stdio data source, and
   *   (b) we passed TRUE to reject a tables-only JPEG file as an error.
   * See libjpeg.doc for more info.
   */

  /* Step 4: set parameters for decompression */

  /* In this example, we don't need to change any of the defaults set by
   * jpeg_read_header(), so we do nothing here.
   */

  /* Step 5: Start decompressor */

  (void) jpeg_start_decompress(&cinfo);
  /* We can ignore the return value since suspension is not possible
   * with the stdio data source.
   */

  /* We may need to do some setup of our own at this point before reading
   * the data.  After jpeg_start_decompress() we have the correct scaled
   * output image dimensions available, as well as the output colormap
   * if we asked for color quantization.
   * In this example, we need to make an output work buffer of the right size.
   */ 
  /* JSAMPLEs per row in output buffer */
  row_stride = cinfo.output_width * cinfo.output_components;
  /* Make a one-row-high sample array that will go away when done with image */
  buffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

  /* Step 6: while (scan lines remain to be read) */
  /*           jpeg_read_scanlines(...); */

  /* Here we use the library's state variable cinfo.output_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   */

 image_buffer=(JSAMPLE *) malloc(cinfo.image_width*cinfo.image_height*3);


  
  cont=cinfo.output_height-1;
  while (cinfo.output_scanline < cinfo.output_height) {
    /* jpeg_read_scanlines expects an array of pointers to scanlines.
     * Here the array is only one element long, but you could ask for
     * more than one scanline at a time if that's more convenient.
     */
    (void) jpeg_read_scanlines(&cinfo, buffer, 1);
    /* Assume put_scanline_someplace wants a pointer and sample count. */
    /* put_scanline_someplace(buffer[0], row_stride); */
    memcpy(image_buffer+cinfo.image_width*3*cont,buffer[0],row_stride);
    cont--;
  }

  /* Step 7: Finish decompression */

  (void) jpeg_finish_decompress(&cinfo);
  /* We can ignore the return value since suspension is not possible
   * with the stdio data source.
   */

  /* Step 8: Release JPEG decompression object */

  /* This is an important step since it will release a good deal of memory. */
  jpeg_destroy_decompress(&cinfo);

  /* After finish_decompress, we can close the input file.
   * Here we postpone it until after no more JPEG errors are possible,
   * so as to simplify the setjmp error logic above.  (Actually, I don't
   * think that jpeg_destroy can do an error exit, but why assume anything...)
   */
  fclose(infile);

  /* At this point you may want to check to see whether any corrupt-data
   * warnings occurred (test whether jerr.pub.num_warnings is nonzero).
   */

  /* And we're done! */
  return image_buffer;
}


