/*
   Siag, Scheme In A Grid
   Copyright (C) 1996-1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/*
 * railway.c
 */

#include "../common/cmalloc.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>


#include "types.h"
#include "calc.h"
#include "../common/common.h"

int errorflag;
int siag_type;
int a1_refs = 0;

#define MAX_INTERPRETERS 10

int interpreter_count = 0;

static struct {
	char *name;
	cval (*evalfn)(buffer *, char *, int, int);
	void (*execfn)(char *);
	char *(*ref_fn)(char *, int, int, int, int, int, int);
} interpreters[MAX_INTERPRETERS];

char *intnames[MAX_INTERPRETERS];

char *interpreter2name(int interpreter)
{
	if (interpreter < 0 || interpreter >= interpreter_count)
		return "NONE";
	return interpreters[interpreter].name;
}

int name2interpreter(char *name)
{
	int i;
	for (i = 0; i < interpreter_count; i++)
		if (!cstrcasecmp(name, interpreters[i].name)) return i;
	return -1;
}

int register_interpreter(char *name,
			cval (*evalfn)(buffer *, char *, int, int),
			void (*execfn)(char *),
			char *(*ref_fn)(char *, int, int, int, int, int, int))
{
	if (interpreter_count >= MAX_INTERPRETERS) return -1;	/* besetzt */
	interpreters[interpreter_count].evalfn = evalfn;
	interpreters[interpreter_count].execfn = execfn;
	interpreters[interpreter_count].ref_fn = ref_fn;
	interpreters[interpreter_count].name = name;
	intnames[interpreter_count] = cstrdup(name);
	intnames[interpreter_count+1] = NULL;
	return interpreter_count++;
}

void exec_expr(int interpreter, char *expr)
{
	void (*execfn)(char *);

	if (interpreter < 0 || interpreter >= interpreter_count) return;
	execfn = interpreters[interpreter].execfn;
	if (execfn == NULL) {

		return;
	}

	execfn(expr);
}

/* Expressions starting with a '=' are evaluated as C */
/* This same function also does strings */
cval parse_expr(buffer *b, int interpreter,
		char *expr, int row, int col)
{
	cval (*evalfn)(buffer *, char *, int, int);
	cval nothing;

	nothing.number = 0;

	/* the next two lines seem to contradict each other, but it is
	   necessary to set the type in case siod bails out before
	   we make it to siag_print(). */
	errorflag = 0;
	siag_type = ERROR;

	if (interpreter >= interpreter_count) return nothing;
	if ((interpreter == siod_interpreter) && (*expr == '=')) {
		expr++;
		interpreter = C_interpreter;
	}

	evalfn = interpreters[interpreter].evalfn;
	if (evalfn == NULL) return nothing;

	return evalfn(b, expr, row, col);
}

/*
   int calc_matrix(spread **matrix)
   Recalculate all the cells in matrix.
   Returns 0 if successful, otherwise 1.
*/
/* Remarkably enough, this function requires no conversion whatsoever
   to go from long to double. */
int calc_matrix(buffer *b)
{
	register int row, col;
	int lr, lc;

	errorflag = FALSE;
	lr = line_last_used(b);
	for (row = 1; row <= lr; row++) {
		lc = col_last_used(b, row);
		for (col = 1; col <= lc; col++) {
			switch (ret_type(b, row, col)) {
				cval value;
			case STRING:
			case EXPRESSION:
			case ERROR:
				value = parse_expr(b,
						ret_interpreter(b, row, col),
						ret_text(b, row, col),
						row, col);
				ins_data(b,
					ret_interpreter(b, row, col),
					ret_text(b, row, col),
					 value, siag_type, row, col);
				break;
			default:
				;
			}
		}
	}
	return 0;
}				/* calc_matrix */

/* make a rough upper bound on the number of references */
#define MIN(x,y) ((x)<(y)?(x):(y))

int ref_a1_counter(char *expr)
{
	int cd, c;

	cd = 0;

	while ((c = *expr++))
		if (isalpha(c) && *expr && isdigit(*expr)) cd++;
	return cd;
}

int ref_counter(char *expr)
{
	int cd, rd, c;

	cd = 0;
	rd = 0;

	if (a1_refs) {
		return ref_a1_counter(expr);
	}

	while ((c = *expr++)) {
		if (c == 'r' || c == 'R') {
			if (*expr == '$' || isdigit(*expr)) rd++;
		} else if (c == 'c' || c == 'C') {
			if (*expr == '$' || isdigit(*expr)) cd++;
		}
	}

	return MIN(cd, rd);
}

/* Portability note: this code assumes that alphabetic characters are
   contiguous and in alphabetic order. This is not required by C, but by
   common sense and by me.
*/
long a1tol(char *p, char **q)
{
	long n = 0;
	int c;

	while ((c = *p++)) {
		if (c >= 'A' && c <= 'Z') {
			n = 26*n+c-'A'+1;
		} else if (c >= 'a' && c <= 'z') {
			n = 26*n+c-'a'+1;
		} else {
			break;
		}
	}
	*q = p-1;
	return n;
}

/* expand references from src to dest, which must be large enough */
/* brk contains break characters for this interpreter */
/* tmpl contains the expansion template: (get-cell %ld %ld) for Scheme */
/* rng contains the range template, if any: 'RANGE %ld %ld %ld %ld for Scheme */

typedef enum {
	REF_START, REF_STRING, REF_LIMBO
} ref_state;

void ref_a1_expander(char *src, char *dest, char *brk, char *tmpl, char *rng)
{
	ref_state state = REF_START;
	char *exp;
	char *start, *end;
	long row, col, r2, c2;
	int c;

	while ((c = *src++)) {
		*dest++ = c;	/* copy straight through */
		switch (state) {
		case REF_START:
			if (isalpha(c)) {
				start = src;
				exp = dest-1;
				col = a1tol(src-1, &end);
				if (!isdigit(end[0])) {
					state = REF_LIMBO;
					break;
				}
				src = end;	/* start of row number */
				row = strtol(src, &end, 10);
				if (end == src ||
					(end[0] && end[0] != '.' &&
					!strchr(brk, end[0]))) {
					src = start;	/* backpedal */
					state = REF_LIMBO;
					break;
				}
				src = end;
	/* if we get this far, there really is a reference here */
				if (src[0] != '.' || src[1] != '.' ||
					!isalpha(src[2])) {	/* single */
					sprintf(exp, tmpl, row, col);
					dest = exp+strlen(exp);
					continue;
				}
	/* and now, look for the other end of the range */
				src += 2;	/* skip past ".." */
				c2 = a1tol(src, &end);
				if (!isdigit(end[0])) {
					src = start;
					state = REF_LIMBO;
					break;
				}
				src = end;
				r2 = strtol(src, &end, 10);
				if (end == src ||
					(end[0] && !strchr(brk, end[0]))) {
					src = start;
					state = REF_LIMBO;
					break;
				}
				src = end;
	/* found the other end */
				sprintf(exp, rng, row, col, r2, c2);
				dest = exp+strlen(exp);
			} else if (c == '"') {
				state = REF_STRING;
			} else if (!strchr(brk, c)) {
				state = REF_LIMBO;
			}
			break;
		case REF_LIMBO:
			if (strchr(brk, c)) {
				state = REF_START;
			 }else if (c == '"') {
				state = REF_STRING;
			}
			break;
		case REF_STRING:
			if (c == '"') {
				state = REF_LIMBO;
			}
			break;
		default:	/* shouldn't happen */
			break;
		}
	}
	*dest = '\0';
}

void ref_expander(char *src, char *dest, char *brk, char *tmpl, char *rng)
{
	ref_state state = REF_START;
	char *exp;
	char *start, *end;
	long row, col, r2, c2;
	int c;

	if (a1_refs) {
		ref_a1_expander(src, dest, brk, tmpl, rng);
		return;
	}

	while ((c = *src++)) {
		*dest++ = c;	/* copy straight through */
		switch (state) {
		case REF_START:
			if ((c == 'r' || c == 'R') && isdigit(*src)) {
				start = src;
				exp = dest-1;
				row = strtol(src, &end, 10);
				if ((end[0] != 'c' && end[0] != 'C') ||
					(end[1] && !isdigit(end[1]))) {
					state = REF_LIMBO;
					break;
				}
				src = end+1;	/* start of col number */
				col = strtol(src, &end, 10);
				if (end == src ||
					(end[0] && end[0] != '.' &&
					!strchr(brk, end[0]))) {
					src = start;	/* backpedal */
					state = REF_LIMBO;
					break;
				}
				src = end;
	/* if we get this far, there really is a reference here */
				if (src[0] != '.' || src[1] != '.' ||
					(src[2] != 'r' && src[2] != 'R') ||
					!isdigit(src[3])) {	/* single */
					sprintf(exp, tmpl, row, col);
					dest = exp+strlen(exp);
					continue;
				}
	/* and now, look for the other end of the range */
				src += 3;	/* skip past "..r" */
				r2 = strtol(src, &end, 10);
				if ((end[0] != 'c' && end[0] != 'C') ||
					(end[1] && !isdigit(end[1]))) {
					src = start;
					state = REF_LIMBO;
					break;
				}
				src = end+1;
				c2 = strtol(src, &end, 10);
				if (end == src ||
					(end[0] && !strchr(brk, end[0]))) {
					src = start;
					state = REF_LIMBO;
					break;
				}
				src = end;
	/* found the other end */
				sprintf(exp, rng, row, col, r2, c2);
				dest = exp+strlen(exp);
			} else if (c == '"') {
				state = REF_STRING;
			} else if (!strchr(brk, c)) {
				state = REF_LIMBO;
			}
			break;
		case REF_LIMBO:
			if (strchr(brk, c)) {
				state = REF_START;
			 }else if (c == '"') {
				state = REF_STRING;
			}
			break;
		case REF_STRING:
			if (c == '"') {
				state = REF_LIMBO;
			}
			break;
		default:	/* shouldn't happen */
			break;
		}
	}
	*dest = '\0';
}

/* move block between (r1,c1) and (r2,c2), direction (rd,cd) */

static char *a1coord(long n)
{
	static char b[100];
	static char digits[] = " ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	int digit;
	int i = 0, j = 0, c;

	while (n) {
		digit = n % 26;
		if (!digit) digit = 26;
		b[i++] = digits[digit];
		n -= digit;
		n /= 26;
	}
	b[i--] = '\0';
	while (j < i) {
		c = b[i];
		b[i--] = b[j];
		b[j++] = c;
	}
	return b;
}

int ref_a1_updater(char *src, char *dest, char *brk,
		int r1, int c1, int r2, int c2, int rd, int cd)
{
	ref_state state = REF_START;
	char *start, *end, *exp;
	long row, col;
	int c, conv = 0;

	while ((c = *src++)) {
		*dest++ = c;
		switch (state) {
		case REF_START:
			if (isalpha(c)) {
				start = src;
				exp = dest-1;
				col = a1tol(src-1, &end);
				if (!isdigit(end[0])) {
					state = REF_LIMBO;
					break;
				}
				src = end;
				row = strtol(src, &end, 10);
				if (end[0] && end[0] != '.' &&
						!strchr(brk, end[0])) {
					src = start;
					state = REF_LIMBO;
					break;
				}
				src = end;
	/* if we get this far, there really is a reference here */
				if (row >= r1 && row <= r2 &&
					col >= c1 && col <= c2) {
					conv++;
					row += rd;
					col += cd;
				}
				sprintf(exp, "%s%ld", a1coord(col), row);
				dest = exp+strlen(exp);
				if (src[0] == '.' && src[1] == '.' &&
					isalpha(src[2])) {
					strcat(dest, "..");
					dest += 2;
					src += 2;
				}
			} else if (c == '"') {
				state = REF_STRING;
			} else if (!strchr(brk, c)) {
				state = REF_START;
			}
			break;
		case REF_LIMBO:
			if (strchr(brk, c)) {
				state = REF_START;
			} else if (c == '"') {
				state = REF_STRING;
			}
			break;
		case REF_STRING:
			if (c == '"') {
				state = REF_LIMBO;
			}
			break;
		default:	/* shouldn't happen */
			break;
		}
	}
	*dest = '\0';
	return conv;
}

int ref_updater(char *src, char *dest, char *brk,
		int r1, int c1, int r2, int c2, int rd, int cd)
{
	ref_state state = REF_START;
	char *start, *end, *exp;
	long row, col;
	int c, conv = 0;

	if (a1_refs) {
		return ref_a1_updater(src, dest, brk, r1, c1, r2, c2, rd, cd);
	}

	while ((c = *src++)) {
		*dest++ = c;	/* copy straight through */
		switch (state) {
		case REF_START:
			if ((c == 'r' || c == 'R') && isdigit(*src)) {
				start = src;
				exp = dest-1;
				row = strtol(src, &end, 10);
				if ((end[0] != 'c' && end[0] != 'C') ||
					(end[1] && !isdigit(end[1]))) {
					state = REF_LIMBO;
					break;
				}
				src = end+1;
				col = strtol(src, &end, 10);
				if (end[0] && end[0] != '.' &&
					!strchr(brk, end[0])) {
					src = start;
					state = REF_LIMBO;
					break;
				}
				src = end;
	/* if we get this far, there really is a reference here */
				if (row >= r1 && row <= r2 &&
					col >= c1 && col <= c2) {
					conv++;
					row += rd;
					col += cd;
				}
				sprintf(exp, "R%ldC%ld", row, col);
				dest = exp+strlen(exp);
				if (src[0] == '.' && src[1] == '.' &&
					(src[2] == 'r' || src[2] == 'R')) {
					strcat(dest, "..");
					dest += 2;
					src += 2;
				}
			} else if (c == '"') {
				state = REF_STRING;
			} else if (!strchr(brk, c)) {
				state = REF_LIMBO;
			}
			break;
		case REF_LIMBO:
			if (strchr(brk, c)) {
				state = REF_START;
			} else if (c == '"') {
				state = REF_STRING;
			}
			break;
		case REF_STRING:
			if (c == '"') {
				state = REF_LIMBO;
			}
			break;
		default:	/* shouldn't happen */
			break;
		}
	}
	*dest = '\0';
	return conv;
}

/* expand Visicalc references */

char *update_references(int interpreter, char *expr,
		int r1, int c1, int r2, int c2, int rd, int cd)
{
	char *(*ref_fn)(char *, int, int, int, int, int, int);

	if (interpreter < 0 || interpreter >= interpreter_count) return expr;
	ref_fn = interpreters[interpreter].ref_fn;
	if (ref_fn == NULL) {
		return expr;
	}
	return ref_fn(expr, r1, c1, r2, c2, rd, cd);
}

void update_all_references(buffer *b, int r1, int c1, int r2, int c2,
				int rd, int cd)
{
	int row, col;
	int lr, lc;
	char *old, *new;

	lr = line_last_used(b);
	for (row = 1; row <= lr; row++) {
		lc = col_last_used(b, row);
		for (col = 1; col <= lc; col++) {
			switch (ret_type(b, row, col)) {
				cval value;
			case STRING:
			case EXPRESSION:
				old = ret_text(b, row, col);
				new = update_references(
					ret_interpreter(b, row, col),
					old,
					r1, c1, r2, c2, rd, cd);
				if (old != new) {
					value = ret_val(b, row, col);
					ins_data(b,
						ret_interpreter(b, row, col),
						new,
						value, siag_type, row, col);
					cfree(new);
				}
				break;
			default:
				;
			}
		}
	}
}

