/*
   Pathetic Writer
   Copyright (C) 1997, 1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
   fileio_ps.c
 
   Produces data suitable for printing on a Postscript device. I have made
   several assumptions to simplify things:
   - paper size is A4 = 595x842 points
   - margin on all sides = 72 points = 1 inch
   - 1 pixel = 1 point means no scaling is necessary
   - PS font has same geometry as X font
   - no background pattern
   - don't draw the grid
   - The lpr command is used for printing

   This works for me, but will probably break (more or less) for anyone
   not using my printer (a NEC S62P) with A4 paper.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "pw.h"
#include "../common/cmalloc.h"
#include "../common/fonts.h"

int paper_height = 842;
int paper_width = 595;
int left_margin = 72, right_margin = 72, top_margin = 72, bottom_margin = 72;
static int lastfont = 0;

extern int font_size[8];

char *psformat;

static void makefonts(FILE *fp)
{
	int i;
	for (i = 0; i < 128; i += 8) {
		fprintf(fp, "/%s findfont\n", ps_fontname(i));
		fprintf(fp, "dup length dict begin\n");
		fprintf(fp, "  {1 index /FID ne {def} {pop pop} ifelse} forall\n");
		fprintf(fp, "  /Encoding ISOLatin1Encoding def\n");
		fprintf(fp, "  currentdict\n");
		fprintf(fp, "end\n");
		fprintf(fp, "/%s-ISOLatin1 exch definefont pop\n",
			ps_fontname(i));
	}
}

static void set_font(FILE *fp, int newfont)
{
	if (newfont == lastfont)
		return;
	lastfont = newfont;
	fprintf(fp, "/%s-ISOLatin1 findfont\n", ps_fontname(newfont));
	fprintf(fp, "%d scalefont\n", font_size[newfont & SIZE_MASK]);
	fprintf(fp, "setfont\n");
}

/* Sizing the output won't be a problem here, except we don't print
	all the allocated but unused lines */
static int page_height(buffer *buf, int r)
{
	int y_base, i;

	y_base = paper_height - top_margin;
	for (i = r; y_base > bottom_margin; i++)
		y_base -= cell_height(buf, i);
	return (i-r);
}

static int line_start(buffer *buf, int row)
{
	int x = 0;
	int w = paper_width-left_margin-right_margin;
	int adj = buf->text[row].adj;

	if (adj == HADJ_CENTER)
		x = (w-line_width(buf, row, line_length(buf, row)))/2;
	else if (adj == HADJ_RIGHT)
		x = w-line_width(buf, row, line_length(buf, row));

	return x+left_margin;
}

static int print_page(FILE *fp, buffer *buf, int fromr, int tor)
{
	int i, x_base, y_base, y_pos, col;
	rich_char *line;

	y_base = paper_height - top_margin;
	for (i = fromr; i <= buf->used_lines && y_base > bottom_margin; i++) {
		line = buf->text[i].p;
		y_base -= cell_height(buf, i);
		x_base = line_start(buf, i);
		y_pos = 5;
		if (rc_strlen(line) == 0) continue;

		if (buf->text[i].sty == STY_EMBED) {
			char *p = (char *)rc_makeplain(line);
			ps_embed_print(fp, p, x_base, y_base+y_pos);
			cfree(p);
			continue;	/* don't print the text */
		}

		fprintf(fp, "newpath\n");
		fprintf(fp, "%d %d moveto\n", x_base, y_base+y_pos);

		for (col = 0; line[col].c; col++) {
			int c = (unsigned char)(line[col].c);
			int font_index = (line[col].fmt)&127;
			set_font(fp, font_index);

		/* print letters and digits as is, the rest in octal */
			if (isalnum(c)) {
				fprintf(fp, "(%c) show\n", c);
			} else if (c == '\t') {
				fprintf(fp, "%d %d moveto\n",
					x_base+line_width(buf, i, col+1),
					y_base+y_pos);
			} else {
				fprintf(fp, "(\\%03o) show\n", c);
			}
		}
	}

	/* print plugins here! */
	for (i = 0; i < buf->nplugin; i++) {
		long x, y, topx, topy, boty, plx, ply;
		unsigned long width, height;
		buffer_global_coords(buf, fromr, 0, &topx, &topy);
		boty = topy+paper_height-top_margin-bottom_margin;
		buffer_global_coords(buf, buf->plugin[i].row,
                        buf->plugin[i].col, &plx, &ply);
                pw_plugin_size_get(buf->plugin[i].ph, &width, &height);
                if (ply+height < topy || ply > boty) continue;
                x = left_margin+(paper_width-left_margin-right_margin-width)/2;
                y = paper_height-top_margin-(ply-topy)-height;
                /* move to the right place */
                fprintf(fp, "%ld %ld translate\n", x, y);
                /* and let the plugin do its thing */
                pw_plugin_print(buf->plugin[i].ph, fp);
                fprintf(fp, "%ld %ld translate\n", -x, -y);
        }

	fprintf(fp, "showpage\n");
	return 0;
}

static int print_area(char *fn, buffer *buf, int fromr, int tor)
{
	FILE *fp;
	int r;
	int pages;
	time_t t;

	fp = fopen(fn, "w");

	fprintf(fp, "%%!PS-Adobe-3.0 EPSF-3.0\n");
	fprintf(fp, "%%%%Creator: %s\n", VERSION);
	fprintf(fp, "%%%%Title: %s\n", buf->name);
	t = time(NULL);
	fprintf(fp, "%%%%CreationDate: %s\n", ctime(&t));
	fprintf(fp, "%%%%Pages: (atend)\n");
	fprintf(fp, "%%%%PageOrder: Ascend\n");
	fprintf(fp, "%%%%BoundingBox: %d %d %d %d\n",
		0, 0, paper_width, paper_height);
	fprintf(fp, "%%%%Orientation: Portrait\n");
	fprintf(fp, "%%%%EndComments\n");

	/* Use ISO-Latin1 encoding */
	fprintf(fp, "%%%%BeginProlog\n");
	makefonts(fp);
	fprintf(fp, "%%%%EndProlog\n");

	pages = 0;
	for (r = fromr; r <= tor; r += page_height(buf, r)) {
		pages++;
		lastfont = 0;	/* force setfont on top of page */
		fprintf(fp, "%%%%Page: %d %d\n", pages, pages);
		print_page(fp, buf, r, tor);
	}
	fprintf(fp, "%%%%Trailer:\n");
	fprintf(fp, "%%%%Pages: %d\n", pages);
	fprintf(fp, "%%%%EOF\n");
	fclose(fp);
	return 0;
}

/* slight mod: no need to calculate the width */
static int save(char *fn, buffer *buf)
{
	int fromr, tor;

	fromr = 1;
	tor = line_last_used(buf);
	return print_area(fn, buf, fromr, tor);
}

/* Conservative file format guessing: always negative (can't load) */
static int myformat(char *fn)
{
	return 0;
}

void fileio_ps_init()
{
	register_format(NULL, save, myformat, psformat = "Postscript (*.ps)");
}

